/*
Copyright 2014 Temboo, Inc.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Temboosession.js: How we interact with the Temboo server.
*/

var https = require("https");
var querystring = require("querystring");
var texception = require("./exception");

exports.TembooSession = function(accountName, appKeyName, appKeyValue) {
    var accountName = accountName;
    var appKeyName = appKeyName;
    var appKeyValue = appKeyValue;
    var clientIdentifer = "NodeJSSDK_2.24.0";
    var basePath = "/arcturus-web/api-1.0";
    var portNum = 443;
    var reqHeaders = {
        "Accept": "application/json",
        "Content-Type": "application/json",
        "x-temboo-domain": [accountName, "master"].join("/")
    };

    var getHost = function() {
        return [accountName, "temboolive.com"].join(".");
    }

    var getReqPath = function(path, params) {
        var reqPath = path;
        if (params) {
            reqPath = reqPath + "?" + querystring.stringify(params);
        }
        return reqPath;
    }

    var respAggregator = function(response, callback, errorCallback) {
    	response.setEncoding('utf8');
        var currCode = response.statusCode;
        var respBody = "";
        response.on("data", function(chunk) {
            respBody += chunk;
        });
        response.on("end", function() {
            // Response as expected.
            if (currCode >= 200 && currCode < 300) {
                callback(respBody);
            // Bad response codes!
            } else {
                texception.determineException(respBody, currCode, 
                                              errorCallback);
            }
        });
    }

    var makeRequest = function(reqMethod, path, params, body, jsClientVersion, callback, 
                               errorCallback) {
        currCode = null;
        currResp = null;

        // Add client and JS SDK identifiers to parameters
        params["source_id"] = clientIdentifer + (jsClientVersion ? '-' + jsClientVersion : '');

        var options = {
            method: reqMethod,
            hostname: getHost(),
            port: portNum,
            path: getReqPath(path, params),
            headers: reqHeaders,
            auth: [appKeyName, appKeyValue].join(":")
        };
        
        var request = https.request(options, function(response) {
            respAggregator(response, callback, errorCallback);
        });

        request.on("error", function(e) {
            e.message += " Make sure your Temboo account name is spelled correctly."
            throw new Error(e.message);
        });
        
        if (body) {
            request.write(body);
        }
        request.end();
    }

    this.getIdentifier = function() {
         return clientIdentifer;
    }

    this.getBasePath = function() {
        return basePath;
    }

    this.getRequest = function(path, params, callback) {
        return makeRequest("GET", path, params, null, callback);
    }
        
    this.postRequest = function(path, params, body, jsClientVersion, callback, errorCallback) {
        makeRequest("POST", path, params, body, jsClientVersion, callback, errorCallback);
    }
}

