/*
Copyright 2014 Temboo, Inc.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 */
var Choreography = require('./choreography.js');
var Exceptions = require('./exception.js');

exports.TembooProxy = function(){

	var choreos = {};

	var validate = function(name){
		if(typeof choreos[name] == 'undefined')
			throw new Exceptions.TembooNotFoundError('Proxied choreography not found: ' + name);
	}

	this.addChoreo = function(name, choreo, defaultInputs, allowedUserInputs){
		
		choreos[name] = new TembooProxifiedChoreography(choreo);
		
		if(defaultInputs){
			if(!defaultInputs.setInput){
				var i = choreo.newInputSet();
				for(var name in defaultInputs)
					i.setInput(name, defaultInputs[name]);
				defaultInputs = i;
			}

			choreos[name].setDefaultInputs(defaultInputs);
		}

		if(allowedUserInputs){
			if(typeof allowedUserInputs == 'string'){
				// Allow multiple inputs as string args
				allowedUserInputs = Array.prototype.slice.call(arguments, 3);
			}
			choreos[name].allowUserInputs(allowedUserInputs);
		}

		return this;
	};

	this.allowUserInputs = function(name, inputs){
		validate(name);		
		choreos[name].allowUserInputs(inputs);
		return this;
	}

	this.execute = function(request, asJson, successCallback, errorCallback){
		try{
			typeof request == 'string' && (request = JSON.parse(request));

			var inputs = request.inputs ? request.inputs : {},
				outputFilters = request.outputFilters ? request.outputFilters : {};

			if(typeof request.name == undefined)
				throw new Exceptions.TembooNotFoundError('No choreography specified');
			
			validate(request.name);

			if(typeof request.version == undefined)
				throw new Exceptions.TembooNotFoundError('Missing required JavaScript SDK version');

			// Execute, format response
			choreos[request.name].execute(inputs, outputFilters, request.version,
				function(result){
					var response = {success:true, outputs:result.getOutputs()};
					successCallback(asJson ? JSON.stringify(response) : response);
				},
				errorCallback
			);
		}
		catch(error){
			// Notify
			errorCallback(asJson ? JSON.stringify(error) : error);
		}
	};

	this.setDefaultInputs = function(name, inputs){
		validate(name);
		choreos[name].setDefaultInputs(inputs);
		return this;
	}
}

var TembooProxifiedChoreography = function(choreo){
	
	var allowedUserInputs = [],
		defaultInputs = choreo.newInputSet();

	this.allowUserInputs = function(inputs){
		typeof inputs == 'string' && (inputs = [inputs]);
		for(var i in inputs)
			allowedUserInputs.indexOf(inputs[i]) == -1 && allowedUserInputs.push(inputs[i]);
	};

	this.execute = function(inputs, outputFilters, jsClientVersion, successCallback, errorCallback){
		var fullInputs = defaultInputs.clone();

		// Validate/set user inputs
		for(var name in inputs){
			if(-1 == allowedUserInputs.indexOf(name))
				throw new Exceptions.TembooDisallowedInputError('Illegal input argument', name);
			
			fullInputs.setInput(name, inputs[name]);
		}

		// Add output filters
		for(var name in outputFilters){
			fullInputs.addOutputFilter(name, outputFilters[name].path, outputFilters[name].variable);
		}

		// Set required version string
		choreo.setJSClientVersion(jsClientVersion);

		// Run the choreo, specifying success and error callback handlers
		choreo.execute(fullInputs, successCallback, errorCallback);
	}

	/**
	 * @param {InputSet} - The default set of inputs for this proxied choreo
	 */
	this.setDefaultInputs = function(inputs){
		defaultInputs = inputs;
	};
}