/*!
 * @TMBZendesk.m
 *
 * Execute Choreographies from the Temboo Zendesk bundle.
 *
 * iOS version 10.6.8
 *
 * LICENSE: Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @category   Services
 * @author     Reid Simonton, Temboo, Inc.
 * @copyright  2013 Temboo, Inc.
 * @license    http://www.apache.org/licenses/LICENSE-2.0 Apache License 2.0
 * @version    1.7
 */

#import "TMBZendesk.h"
#import "TMBChoreography_Protected.h"
#import "TMBChoreographyExecution.h"
#import "TMBTembooSession.h"
#import "TMBTembooUri.h"


/*!
 * Input object with appropriate setters for specifying arguments to the ListActivity Choreo.
 */
@implementation TMBZendesk_ActivityStream_ListActivity_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) The page number of the results to be returned. Used together with the PerPage parameter to paginate a large set of results.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the PerPage input for this Choreo.
	*(optional, integer) The number of results to return per page. Maximum is 100 and default is 100.
	 */
	-(void)setPerPage:(NSString*)PerPage {
		[super setInput:@"PerPage" toValue:PerPage];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListActivity Choreo.
 */
@implementation TMBZendesk_ActivityStream_ListActivity_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Returns a list of activities for the authenticating user.
 */
@implementation TMBZendesk_ActivityStream_ListActivity

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_ActivityStream_ListActivity Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/ActivityStream/ListActivity"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_ActivityStream_ListActivity_ResultSet *results = [[TMBZendesk_ActivityStream_ListActivity_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListActivity Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_ActivityStream_ListActivity_Inputs*)newInputSet {
		return [[TMBZendesk_ActivityStream_ListActivity_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ShowActivity Choreo.
 */
@implementation TMBZendesk_ActivityStream_ShowActivity_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, string) The ID of the activity to retrieve.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ShowActivity Choreo.
 */
@implementation TMBZendesk_ActivityStream_ShowActivity_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Returns detailed information about a particular activity.
 */
@implementation TMBZendesk_ActivityStream_ShowActivity

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_ActivityStream_ShowActivity Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/ActivityStream/ShowActivity"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_ActivityStream_ShowActivity_ResultSet *results = [[TMBZendesk_ActivityStream_ShowActivity_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ShowActivity Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_ActivityStream_ShowActivity_Inputs*)newInputSet {
		return [[TMBZendesk_ActivityStream_ShowActivity_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UploadAttachment Choreo.
 */
@implementation TMBZendesk_Attachments_UploadAttachment_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the ExistingToken input for this Choreo.
	*(optional, string) Allows you to pass in an existing token when uploading multiple attachments to associate with a ticket.
	 */
	-(void)setExistingToken:(NSString*)ExistingToken {
		[super setInput:@"ExistingToken" toValue:ExistingToken];
	}

	/*!
	 * Set the value of the FileContents input for this Choreo.
	*(required, string) The Base64 encoded file contents of the attachment you want to upload.
	 */
	-(void)setFileContents:(NSString*)FileContents {
		[super setInput:@"FileContents" toValue:FileContents];
	}

	/*!
	 * Set the value of the FileName input for this Choreo.
	*(required, string) The file name of the attachment.
	 */
	-(void)setFileName:(NSString*)FileName {
		[super setInput:@"FileName" toValue:FileName];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the VaultFile input for this Choreo.
	*The path to a vault file to upload. Can be used as an alternative to the FileContents input.
	 */
	-(void)setVaultFile:(NSString*)VaultFile {
		[super setInput:@"VaultFile" toValue:VaultFile];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UploadAttachment Choreo.
 */
@implementation TMBZendesk_Attachments_UploadAttachment_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Token" output from an execution of this Choreo.
	 * @return - NSString* (string) The token returned from Zendesk for the upload. This can be passed to the ExistingToken input when associating  multiple attachments to the same upload token.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getToken {
		return [super getOutputByName:@"Token"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Uploads a file to be used as an attachment to a ticket.
 */
@implementation TMBZendesk_Attachments_UploadAttachment

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Attachments_UploadAttachment Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Attachments/UploadAttachment"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Attachments_UploadAttachment_ResultSet *results = [[TMBZendesk_Attachments_UploadAttachment_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UploadAttachment Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Attachments_UploadAttachment_Inputs*)newInputSet {
		return [[TMBZendesk_Attachments_UploadAttachment_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CreateGroup Choreo.
 */
@implementation TMBZendesk_Groups_CreateGroup_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the GroupName input for this Choreo.
	*(required, string) The name of the group to create.
	 */
	-(void)setGroupName:(NSString*)GroupName {
		[super setInput:@"GroupName" toValue:GroupName];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateGroup Choreo.
 */
@implementation TMBZendesk_Groups_CreateGroup_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Creates a new group.
 */
@implementation TMBZendesk_Groups_CreateGroup

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Groups_CreateGroup Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Groups/CreateGroup"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Groups_CreateGroup_ResultSet *results = [[TMBZendesk_Groups_CreateGroup_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CreateGroup Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Groups_CreateGroup_Inputs*)newInputSet {
		return [[TMBZendesk_Groups_CreateGroup_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteGroup Choreo.
 */
@implementation TMBZendesk_Groups_DeleteGroup_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the GroupID input for this Choreo.
	*(required, string) The ID of group to delete.
	 */
	-(void)setGroupID:(NSString*)GroupID {
		[super setInput:@"GroupID" toValue:GroupID];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteGroup Choreo.
 */
@implementation TMBZendesk_Groups_DeleteGroup_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "ResponseStatusCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) The response status code returned from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponseStatusCode {
		return [super getOutputByName:@"ResponseStatusCode"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Deletes an existing group.
 */
@implementation TMBZendesk_Groups_DeleteGroup

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Groups_DeleteGroup Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Groups/DeleteGroup"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Groups_DeleteGroup_ResultSet *results = [[TMBZendesk_Groups_DeleteGroup_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteGroup Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Groups_DeleteGroup_Inputs*)newInputSet {
		return [[TMBZendesk_Groups_DeleteGroup_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListGroups Choreo.
 */
@implementation TMBZendesk_Groups_ListGroups_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) The page number of the results to be returned. Used together with the PerPage parameter to paginate a large set of results.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the PerPage input for this Choreo.
	*(optional, integer) The number of results to return per page. Maximum is 100 and default is 100.
	 */
	-(void)setPerPage:(NSString*)PerPage {
		[super setInput:@"PerPage" toValue:PerPage];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListGroups Choreo.
 */
@implementation TMBZendesk_Groups_ListGroups_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NextPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the next page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextPage {
		return [super getOutputByName:@"NextPage"];
	}

	/*!
	 * Retrieve the value of the "PreviousPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the previous page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPreviousPage {
		return [super getOutputByName:@"PreviousPage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * List available groups.
 */
@implementation TMBZendesk_Groups_ListGroups

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Groups_ListGroups Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Groups/ListGroups"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Groups_ListGroups_ResultSet *results = [[TMBZendesk_Groups_ListGroups_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListGroups Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Groups_ListGroups_Inputs*)newInputSet {
		return [[TMBZendesk_Groups_ListGroups_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ShowAssignableGroups Choreo.
 */
@implementation TMBZendesk_Groups_ShowAssignableGroups_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) The page number of the results to be returned. Used together with the PerPage parameter to paginate a large set of results.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the PerPage input for this Choreo.
	*(optional, integer) The number of results to return per page. Maximum is 100 and default is 100.
	 */
	-(void)setPerPage:(NSString*)PerPage {
		[super setInput:@"PerPage" toValue:PerPage];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ShowAssignableGroups Choreo.
 */
@implementation TMBZendesk_Groups_ShowAssignableGroups_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NextPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the next page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextPage {
		return [super getOutputByName:@"NextPage"];
	}

	/*!
	 * Retrieve the value of the "PreviousPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the previous page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPreviousPage {
		return [super getOutputByName:@"PreviousPage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Show assignable groups.
 */
@implementation TMBZendesk_Groups_ShowAssignableGroups

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Groups_ShowAssignableGroups Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Groups/ShowAssignableGroups"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Groups_ShowAssignableGroups_ResultSet *results = [[TMBZendesk_Groups_ShowAssignableGroups_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ShowAssignableGroups Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Groups_ShowAssignableGroups_Inputs*)newInputSet {
		return [[TMBZendesk_Groups_ShowAssignableGroups_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ShowGroup Choreo.
 */
@implementation TMBZendesk_Groups_ShowGroup_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the GroupID input for this Choreo.
	*(required, string) The ID of the group to show.
	 */
	-(void)setGroupID:(NSString*)GroupID {
		[super setInput:@"GroupID" toValue:GroupID];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ShowGroup Choreo.
 */
@implementation TMBZendesk_Groups_ShowGroup_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Shows information for an existing group.
 */
@implementation TMBZendesk_Groups_ShowGroup

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Groups_ShowGroup Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Groups/ShowGroup"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Groups_ShowGroup_ResultSet *results = [[TMBZendesk_Groups_ShowGroup_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ShowGroup Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Groups_ShowGroup_Inputs*)newInputSet {
		return [[TMBZendesk_Groups_ShowGroup_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UpdateGroup Choreo.
 */
@implementation TMBZendesk_Groups_UpdateGroup_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the GroupID input for this Choreo.
	*(conditional, string) The ID of the Group to be updated.
	 */
	-(void)setGroupID:(NSString*)GroupID {
		[super setInput:@"GroupID" toValue:GroupID];
	}

	/*!
	 * Set the value of the GroupName input for this Choreo.
	*(required, string) New name to update the Group with.
	 */
	-(void)setGroupName:(NSString*)GroupName {
		[super setInput:@"GroupName" toValue:GroupName];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UpdateGroup Choreo.
 */
@implementation TMBZendesk_Groups_UpdateGroup_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Updates an existing group.
 */
@implementation TMBZendesk_Groups_UpdateGroup

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Groups_UpdateGroup Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Groups/UpdateGroup"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Groups_UpdateGroup_ResultSet *results = [[TMBZendesk_Groups_UpdateGroup_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UpdateGroup Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Groups_UpdateGroup_Inputs*)newInputSet {
		return [[TMBZendesk_Groups_UpdateGroup_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ExportIncrementalTickets Choreo.
 */
@implementation TMBZendesk_IncrementalTickets_ExportIncrementalTickets_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the StartTime input for this Choreo.
	*(required, string) Return a list of tickets created after this timestamp (in seconds since Epoch UTC).  Tickets less than 5 minutes old are not included in the response.
	 */
	-(void)setStartTime:(NSString*)StartTime {
		[super setInput:@"StartTime" toValue:StartTime];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ExportIncrementalTickets Choreo.
 */
@implementation TMBZendesk_IncrementalTickets_ExportIncrementalTickets_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Returns a lightweight representation of what changed in the ticket "since you last asked".
 */
@implementation TMBZendesk_IncrementalTickets_ExportIncrementalTickets

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_IncrementalTickets_ExportIncrementalTickets Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/IncrementalTickets/ExportIncrementalTickets"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_IncrementalTickets_ExportIncrementalTickets_ResultSet *results = [[TMBZendesk_IncrementalTickets_ExportIncrementalTickets_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ExportIncrementalTickets Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_IncrementalTickets_ExportIncrementalTickets_Inputs*)newInputSet {
		return [[TMBZendesk_IncrementalTickets_ExportIncrementalTickets_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ApplyMacroToAllTickets Choreo.
 */
@implementation TMBZendesk_Macros_ApplyMacroToAllTickets_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the MacroID input for this Choreo.
	*(required, integer) The ID of the macro to apply.
	 */
	-(void)setMacroID:(NSString*)MacroID {
		[super setInput:@"MacroID" toValue:MacroID];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ApplyMacroToAllTickets Choreo.
 */
@implementation TMBZendesk_Macros_ApplyMacroToAllTickets_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Applies a given macro to all applicable tickets.
 */
@implementation TMBZendesk_Macros_ApplyMacroToAllTickets

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Macros_ApplyMacroToAllTickets Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Macros/ApplyMacroToAllTickets"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Macros_ApplyMacroToAllTickets_ResultSet *results = [[TMBZendesk_Macros_ApplyMacroToAllTickets_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ApplyMacroToAllTickets Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Macros_ApplyMacroToAllTickets_Inputs*)newInputSet {
		return [[TMBZendesk_Macros_ApplyMacroToAllTickets_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ApplyMacroToTicket Choreo.
 */
@implementation TMBZendesk_Macros_ApplyMacroToTicket_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the MacroID input for this Choreo.
	*(required, integer) The ID of the macro to apply.
	 */
	-(void)setMacroID:(NSString*)MacroID {
		[super setInput:@"MacroID" toValue:MacroID];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the TicketID input for this Choreo.
	*(required, string) The ID of the ticket.
	 */
	-(void)setTicketID:(NSString*)TicketID {
		[super setInput:@"TicketID" toValue:TicketID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ApplyMacroToTicket Choreo.
 */
@implementation TMBZendesk_Macros_ApplyMacroToTicket_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Applies a macro to a given ticket.
 */
@implementation TMBZendesk_Macros_ApplyMacroToTicket

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Macros_ApplyMacroToTicket Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Macros/ApplyMacroToTicket"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Macros_ApplyMacroToTicket_ResultSet *results = [[TMBZendesk_Macros_ApplyMacroToTicket_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ApplyMacroToTicket Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Macros_ApplyMacroToTicket_Inputs*)newInputSet {
		return [[TMBZendesk_Macros_ApplyMacroToTicket_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListMacros Choreo.
 */
@implementation TMBZendesk_Macros_ListMacros_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) The page number of the results to be returned. Used together with the PerPage parameter to paginate a large set of results.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the PerPage input for this Choreo.
	*(optional, integer) The number of results to return per page. Maximum is 100 and default is 100.
	 */
	-(void)setPerPage:(NSString*)PerPage {
		[super setInput:@"PerPage" toValue:PerPage];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListMacros Choreo.
 */
@implementation TMBZendesk_Macros_ListMacros_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NextPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the next page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextPage {
		return [super getOutputByName:@"NextPage"];
	}

	/*!
	 * Retrieve the value of the "PreviousPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the previous page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPreviousPage {
		return [super getOutputByName:@"PreviousPage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a list of all active macros available to the current user.
 */
@implementation TMBZendesk_Macros_ListMacros

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Macros_ListMacros Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Macros/ListMacros"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Macros_ListMacros_ResultSet *results = [[TMBZendesk_Macros_ListMacros_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListMacros Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Macros_ListMacros_Inputs*)newInputSet {
		return [[TMBZendesk_Macros_ListMacros_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetMonitoredTwitterHandle Choreo.
 */
@implementation TMBZendesk_MonitoredTwitterHandles_GetMonitoredTwitterHandle_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, string) ID of the monitored Twitter handle.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetMonitoredTwitterHandle Choreo.
 */
@implementation TMBZendesk_MonitoredTwitterHandles_GetMonitoredTwitterHandle_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 *  Retrieves detailed information on a specified monitored Twitter account. 
 */
@implementation TMBZendesk_MonitoredTwitterHandles_GetMonitoredTwitterHandle

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_MonitoredTwitterHandles_GetMonitoredTwitterHandle Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/MonitoredTwitterHandles/GetMonitoredTwitterHandle"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_MonitoredTwitterHandles_GetMonitoredTwitterHandle_ResultSet *results = [[TMBZendesk_MonitoredTwitterHandles_GetMonitoredTwitterHandle_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetMonitoredTwitterHandle Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_MonitoredTwitterHandles_GetMonitoredTwitterHandle_Inputs*)newInputSet {
		return [[TMBZendesk_MonitoredTwitterHandles_GetMonitoredTwitterHandle_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListMonitoredTwitterHandles Choreo.
 */
@implementation TMBZendesk_MonitoredTwitterHandles_ListMonitoredTwitterHandles_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) The page number of the results to be returned. Used together with the PerPage parameter to paginate a large set of results.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the PerPage input for this Choreo.
	*(optional, integer) The number of results to return per page. Maximum is 100 and default is 100.
	 */
	-(void)setPerPage:(NSString*)PerPage {
		[super setInput:@"PerPage" toValue:PerPage];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListMonitoredTwitterHandles Choreo.
 */
@implementation TMBZendesk_MonitoredTwitterHandles_ListMonitoredTwitterHandles_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NextPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the next page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextPage {
		return [super getOutputByName:@"NextPage"];
	}

	/*!
	 * Retrieve the value of the "PreviousPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the previous page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPreviousPage {
		return [super getOutputByName:@"PreviousPage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a list of monitored Twitter accounts that you have configured in your Zendesk account.  
 */
@implementation TMBZendesk_MonitoredTwitterHandles_ListMonitoredTwitterHandles

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_MonitoredTwitterHandles_ListMonitoredTwitterHandles Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/MonitoredTwitterHandles/ListMonitoredTwitterHandles"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_MonitoredTwitterHandles_ListMonitoredTwitterHandles_ResultSet *results = [[TMBZendesk_MonitoredTwitterHandles_ListMonitoredTwitterHandles_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListMonitoredTwitterHandles Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_MonitoredTwitterHandles_ListMonitoredTwitterHandles_Inputs*)newInputSet {
		return [[TMBZendesk_MonitoredTwitterHandles_ListMonitoredTwitterHandles_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the AutocompleteOrganizationName Choreo.
 */
@implementation TMBZendesk_Organizations_AutocompleteOrganizationName_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Name input for this Choreo.
	*(required, string) Prefix used to generate a list of Organization names. Must be at least 2 characters long.
	 */
	-(void)setName:(NSString*)Name {
		[super setInput:@"Name" toValue:Name];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the AutocompleteOrganizationName Choreo.
 */
@implementation TMBZendesk_Organizations_AutocompleteOrganizationName_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Returns an array of organizations whose name starts with the value specified in the name parameter.
 */
@implementation TMBZendesk_Organizations_AutocompleteOrganizationName

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Organizations_AutocompleteOrganizationName Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Organizations/AutocompleteOrganizationName"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Organizations_AutocompleteOrganizationName_ResultSet *results = [[TMBZendesk_Organizations_AutocompleteOrganizationName_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the AutocompleteOrganizationName Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Organizations_AutocompleteOrganizationName_Inputs*)newInputSet {
		return [[TMBZendesk_Organizations_AutocompleteOrganizationName_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CreateManyOrganizations Choreo.
 */
@implementation TMBZendesk_Organizations_CreateManyOrganizations_Inputs

	/*!
	 * Set the value of the OrganizationData input for this Choreo.
	*(optional, json) A JSON-formatted string containing an array of organization properties you wish to set. This can used when you need to set multiple properties.
	 */
	-(void)setOrganizationData:(NSString*)OrganizationData {
		[super setInput:@"OrganizationData" toValue:OrganizationData];
	}

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Names input for this Choreo.
	*(conditional, string) Comma-seperated list of up to 10  organization names to create.
	 */
	-(void)setNames:(NSString*)Names {
		[super setInput:@"Names" toValue:Names];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateManyOrganizations Choreo.
 */
@implementation TMBZendesk_Organizations_CreateManyOrganizations_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Create multiple organizations with a single request. 
 */
@implementation TMBZendesk_Organizations_CreateManyOrganizations

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Organizations_CreateManyOrganizations Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Organizations/CreateManyOrganizations"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Organizations_CreateManyOrganizations_ResultSet *results = [[TMBZendesk_Organizations_CreateManyOrganizations_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CreateManyOrganizations Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Organizations_CreateManyOrganizations_Inputs*)newInputSet {
		return [[TMBZendesk_Organizations_CreateManyOrganizations_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CreateOrganization Choreo.
 */
@implementation TMBZendesk_Organizations_CreateOrganization_Inputs

	/*!
	 * Set the value of the OrganizationData input for this Choreo.
	*(optional, json) A JSON-formatted string containing the organization properties you wish to set. This can used when you need to set multiple properties.
	 */
	-(void)setOrganizationData:(NSString*)OrganizationData {
		[super setInput:@"OrganizationData" toValue:OrganizationData];
	}

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Name input for this Choreo.
	*(conditional, string) Name of the organization to create.
	 */
	-(void)setName:(NSString*)Name {
		[super setInput:@"Name" toValue:Name];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateOrganization Choreo.
 */
@implementation TMBZendesk_Organizations_CreateOrganization_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Create an organization.
 */
@implementation TMBZendesk_Organizations_CreateOrganization

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Organizations_CreateOrganization Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Organizations/CreateOrganization"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Organizations_CreateOrganization_ResultSet *results = [[TMBZendesk_Organizations_CreateOrganization_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CreateOrganization Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Organizations_CreateOrganization_Inputs*)newInputSet {
		return [[TMBZendesk_Organizations_CreateOrganization_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteOrganization Choreo.
 */
@implementation TMBZendesk_Organizations_DeleteOrganization_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, string) ID of the organization to delete.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteOrganization Choreo.
 */
@implementation TMBZendesk_Organizations_DeleteOrganization_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "ResponseStatusCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) The response status code returned from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponseStatusCode {
		return [super getOutputByName:@"ResponseStatusCode"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Deletes an existing organization.
 */
@implementation TMBZendesk_Organizations_DeleteOrganization

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Organizations_DeleteOrganization Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Organizations/DeleteOrganization"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Organizations_DeleteOrganization_ResultSet *results = [[TMBZendesk_Organizations_DeleteOrganization_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteOrganization Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Organizations_DeleteOrganization_Inputs*)newInputSet {
		return [[TMBZendesk_Organizations_DeleteOrganization_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetOrganization Choreo.
 */
@implementation TMBZendesk_Organizations_GetOrganization_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, string) ID of the organization to get.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetOrganization Choreo.
 */
@implementation TMBZendesk_Organizations_GetOrganization_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Get detailed info for a given organization.
 */
@implementation TMBZendesk_Organizations_GetOrganization

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Organizations_GetOrganization Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Organizations/GetOrganization"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Organizations_GetOrganization_ResultSet *results = [[TMBZendesk_Organizations_GetOrganization_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetOrganization Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Organizations_GetOrganization_Inputs*)newInputSet {
		return [[TMBZendesk_Organizations_GetOrganization_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListOrganizations Choreo.
 */
@implementation TMBZendesk_Organizations_ListOrganizations_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) The page number of the results to be returned. Used together with the PerPage parameter to paginate a large set of results.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the PerPage input for this Choreo.
	*(optional, integer) The number of results to return per page. Maximum is 100 and default is 100.
	 */
	-(void)setPerPage:(NSString*)PerPage {
		[super setInput:@"PerPage" toValue:PerPage];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListOrganizations Choreo.
 */
@implementation TMBZendesk_Organizations_ListOrganizations_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NextPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the next page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextPage {
		return [super getOutputByName:@"NextPage"];
	}

	/*!
	 * Retrieve the value of the "PreviousPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the previous page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPreviousPage {
		return [super getOutputByName:@"PreviousPage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Lists all organizations.
 */
@implementation TMBZendesk_Organizations_ListOrganizations

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Organizations_ListOrganizations Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Organizations/ListOrganizations"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Organizations_ListOrganizations_ResultSet *results = [[TMBZendesk_Organizations_ListOrganizations_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListOrganizations Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Organizations_ListOrganizations_Inputs*)newInputSet {
		return [[TMBZendesk_Organizations_ListOrganizations_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListOrganizationsByUser Choreo.
 */
@implementation TMBZendesk_Organizations_ListOrganizationsByUser_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, string) ID of the user.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) The page number of the results to be returned. Used together with the Number parameter to paginate a large set of results.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the PerPage input for this Choreo.
	*(optional, integer) The number of results to return per page. Maximum is 100 and default is 100.
	 */
	-(void)setPerPage:(NSString*)PerPage {
		[super setInput:@"PerPage" toValue:PerPage];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListOrganizationsByUser Choreo.
 */
@implementation TMBZendesk_Organizations_ListOrganizationsByUser_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NextPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the next page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextPage {
		return [super getOutputByName:@"NextPage"];
	}

	/*!
	 * Retrieve the value of the "PreviousPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the previous page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPreviousPage {
		return [super getOutputByName:@"PreviousPage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Lists all organizations by user.
 */
@implementation TMBZendesk_Organizations_ListOrganizationsByUser

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Organizations_ListOrganizationsByUser Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Organizations/ListOrganizationsByUser"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Organizations_ListOrganizationsByUser_ResultSet *results = [[TMBZendesk_Organizations_ListOrganizationsByUser_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListOrganizationsByUser Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Organizations_ListOrganizationsByUser_Inputs*)newInputSet {
		return [[TMBZendesk_Organizations_ListOrganizationsByUser_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UpdateOrganization Choreo.
 */
@implementation TMBZendesk_Organizations_UpdateOrganization_Inputs

	/*!
	 * Set the value of the OrganizationData input for this Choreo.
	*(optional, json) A JSON-formatted string containing the organization properties you wish to set. This can used when you need to set multiple properties.
	 */
	-(void)setOrganizationData:(NSString*)OrganizationData {
		[super setInput:@"OrganizationData" toValue:OrganizationData];
	}

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, string) ID of the organization to update.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the Notes input for this Choreo.
	*(conditional, string) Notes on the organization.
	 */
	-(void)setNotes:(NSString*)Notes {
		[super setInput:@"Notes" toValue:Notes];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UpdateOrganization Choreo.
 */
@implementation TMBZendesk_Organizations_UpdateOrganization_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Update an existing organization.
 */
@implementation TMBZendesk_Organizations_UpdateOrganization

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Organizations_UpdateOrganization Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Organizations/UpdateOrganization"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Organizations_UpdateOrganization_ResultSet *results = [[TMBZendesk_Organizations_UpdateOrganization_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UpdateOrganization Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Organizations_UpdateOrganization_Inputs*)newInputSet {
		return [[TMBZendesk_Organizations_UpdateOrganization_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CreateRequest Choreo.
 */
@implementation TMBZendesk_Requests_CreateRequest_Inputs

	/*!
	 * Set the value of the RequestData input for this Choreo.
	*(optional, json) A JSON-formatted string containing the request properties you wish to set. This can be used as an alternative to setting individual inputs representing request properties.
	 */
	-(void)setRequestData:(NSString*)RequestData {
		[super setInput:@"RequestData" toValue:RequestData];
	}

	/*!
	 * Set the value of the Comment input for this Choreo.
	*(conditional, string) A comment associated with the request.
	 */
	-(void)setComment:(NSString*)Comment {
		[super setInput:@"Comment" toValue:Comment];
	}

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Priority input for this Choreo.
	*(conditional, string) Priority (e.g. low, normal, high, urgent). Defaults to normal.
	 */
	-(void)setPriority:(NSString*)Priority {
		[super setInput:@"Priority" toValue:Priority];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the Subject input for this Choreo.
	*(conditional, string) The subject of the request.
	 */
	-(void)setSubject:(NSString*)Subject {
		[super setInput:@"Subject" toValue:Subject];
	}

	/*!
	 * Set the value of the Type input for this Choreo.
	*(conditional, string) Type of request (e.g.question, incident, problem, task). Defaults to incident.
	 */
	-(void)setType:(NSString*)Type {
		[super setInput:@"Type" toValue:Type];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateRequest Choreo.
 */
@implementation TMBZendesk_Requests_CreateRequest_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Creates a new end-user request.
 */
@implementation TMBZendesk_Requests_CreateRequest

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Requests_CreateRequest Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Requests/CreateRequest"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Requests_CreateRequest_ResultSet *results = [[TMBZendesk_Requests_CreateRequest_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CreateRequest Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Requests_CreateRequest_Inputs*)newInputSet {
		return [[TMBZendesk_Requests_CreateRequest_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetComment Choreo.
 */
@implementation TMBZendesk_Requests_GetComment_Inputs

	/*!
	 * Set the value of the CommentID input for this Choreo.
	*(required, string) The ID of the comment to retrieve.
	 */
	-(void)setCommentID:(NSString*)CommentID {
		[super setInput:@"CommentID" toValue:CommentID];
	}

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the RequestID input for this Choreo.
	*(required, string) The ID of the request that has been commented on.
	 */
	-(void)setRequestID:(NSString*)RequestID {
		[super setInput:@"RequestID" toValue:RequestID];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetComment Choreo.
 */
@implementation TMBZendesk_Requests_GetComment_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the specified comment from a request.
 */
@implementation TMBZendesk_Requests_GetComment

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Requests_GetComment Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Requests/GetComment"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Requests_GetComment_ResultSet *results = [[TMBZendesk_Requests_GetComment_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetComment Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Requests_GetComment_Inputs*)newInputSet {
		return [[TMBZendesk_Requests_GetComment_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetRequest Choreo.
 */
@implementation TMBZendesk_Requests_GetRequest_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, string) The ID of the request to retrieve.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the Status input for this Choreo.
	*(optional, string) Comma-seperated list of request statuses to match (e.g. hold, open, solved, ccd)
	 */
	-(void)setStatus:(NSString*)Status {
		[super setInput:@"Status" toValue:Status];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetRequest Choreo.
 */
@implementation TMBZendesk_Requests_GetRequest_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the request for the specified ID.
 */
@implementation TMBZendesk_Requests_GetRequest

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Requests_GetRequest Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Requests/GetRequest"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Requests_GetRequest_ResultSet *results = [[TMBZendesk_Requests_GetRequest_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetRequest Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Requests_GetRequest_Inputs*)newInputSet {
		return [[TMBZendesk_Requests_GetRequest_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListAllRequests Choreo.
 */
@implementation TMBZendesk_Requests_ListAllRequests_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) The page number of the results to be returned. Used together with the PerPage parameter to paginate a large set of results.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the PerPage input for this Choreo.
	*(optional, integer) The number of results to return per page. Maximum is 100 and default is 100.
	 */
	-(void)setPerPage:(NSString*)PerPage {
		[super setInput:@"PerPage" toValue:PerPage];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the Status input for this Choreo.
	*(optional, string) Comma-seperated list of request statuses to match (e.g. hold, open, solved, ccd)
	 */
	-(void)setStatus:(NSString*)Status {
		[super setInput:@"Status" toValue:Status];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListAllRequests Choreo.
 */
@implementation TMBZendesk_Requests_ListAllRequests_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NextPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the next page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextPage {
		return [super getOutputByName:@"NextPage"];
	}

	/*!
	 * Retrieve the value of the "PreviousPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the previous page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPreviousPage {
		return [super getOutputByName:@"PreviousPage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * List all existing requests for your account.
 */
@implementation TMBZendesk_Requests_ListAllRequests

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Requests_ListAllRequests Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Requests/ListAllRequests"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Requests_ListAllRequests_ResultSet *results = [[TMBZendesk_Requests_ListAllRequests_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListAllRequests Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Requests_ListAllRequests_Inputs*)newInputSet {
		return [[TMBZendesk_Requests_ListAllRequests_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListComments Choreo.
 */
@implementation TMBZendesk_Requests_ListComments_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, string) The ID of the request to retrieve comments for.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) The page number of the results to be returned. Used together with the PerPage parameter to paginate a large set of results.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the PerPage input for this Choreo.
	*(optional, integer) The number of results to return per page. Maximum is 100 and default is 100.
	 */
	-(void)setPerPage:(NSString*)PerPage {
		[super setInput:@"PerPage" toValue:PerPage];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListComments Choreo.
 */
@implementation TMBZendesk_Requests_ListComments_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NextPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the next page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextPage {
		return [super getOutputByName:@"NextPage"];
	}

	/*!
	 * Retrieve the value of the "PreviousPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the previous page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPreviousPage {
		return [super getOutputByName:@"PreviousPage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Lists all comments for the specifed request.
 */
@implementation TMBZendesk_Requests_ListComments

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Requests_ListComments Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Requests/ListComments"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Requests_ListComments_ResultSet *results = [[TMBZendesk_Requests_ListComments_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListComments Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Requests_ListComments_Inputs*)newInputSet {
		return [[TMBZendesk_Requests_ListComments_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListOrganizationRequests Choreo.
 */
@implementation TMBZendesk_Requests_ListOrganizationRequests_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, string) The ID of the organization to list requests for.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) The page number of the results to be returned. Used together with the PerPage parameter to paginate a large set of results.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the PerPage input for this Choreo.
	*(optional, integer) The number of results to return per page. Maximum is 100 and default is 100.
	 */
	-(void)setPerPage:(NSString*)PerPage {
		[super setInput:@"PerPage" toValue:PerPage];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the Status input for this Choreo.
	*(optional, string) Comma-seperated list of request statuses to match (e.g. hold, open, solved, ccd)
	 */
	-(void)setStatus:(NSString*)Status {
		[super setInput:@"Status" toValue:Status];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListOrganizationRequests Choreo.
 */
@implementation TMBZendesk_Requests_ListOrganizationRequests_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NextPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the next page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextPage {
		return [super getOutputByName:@"NextPage"];
	}

	/*!
	 * Retrieve the value of the "PreviousPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the previous page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPreviousPage {
		return [super getOutputByName:@"PreviousPage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * List all requests for the specified organization.
 */
@implementation TMBZendesk_Requests_ListOrganizationRequests

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Requests_ListOrganizationRequests Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Requests/ListOrganizationRequests"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Requests_ListOrganizationRequests_ResultSet *results = [[TMBZendesk_Requests_ListOrganizationRequests_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListOrganizationRequests Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Requests_ListOrganizationRequests_Inputs*)newInputSet {
		return [[TMBZendesk_Requests_ListOrganizationRequests_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListUserRequests Choreo.
 */
@implementation TMBZendesk_Requests_ListUserRequests_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, string) The ID of the user to list requests for.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) The page number of the results to be returned. Used together with the PerPage parameter to paginate a large set of results.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the PerPage input for this Choreo.
	*(optional, integer) The number of results to return per page. Maximum is 100 and default is 100.
	 */
	-(void)setPerPage:(NSString*)PerPage {
		[super setInput:@"PerPage" toValue:PerPage];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the Status input for this Choreo.
	*(optional, string) Comma-seperated list of request statuses to match (e.g. hold, open, solved, ccd)
	 */
	-(void)setStatus:(NSString*)Status {
		[super setInput:@"Status" toValue:Status];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListUserRequests Choreo.
 */
@implementation TMBZendesk_Requests_ListUserRequests_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NextPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the next page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextPage {
		return [super getOutputByName:@"NextPage"];
	}

	/*!
	 * Retrieve the value of the "PreviousPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the previous page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPreviousPage {
		return [super getOutputByName:@"PreviousPage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * List all requests that are tied to a specified user.
 */
@implementation TMBZendesk_Requests_ListUserRequests

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Requests_ListUserRequests Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Requests/ListUserRequests"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Requests_ListUserRequests_ResultSet *results = [[TMBZendesk_Requests_ListUserRequests_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListUserRequests Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Requests_ListUserRequests_Inputs*)newInputSet {
		return [[TMBZendesk_Requests_ListUserRequests_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UpdateRequest Choreo.
 */
@implementation TMBZendesk_Requests_UpdateRequest_Inputs

	/*!
	 * Set the value of the RequestData input for this Choreo.
	*(optional, json) A JSON-formatted string containing the request properties you wish to set. This can be used as an alternative to setting individual inputs representing request properties.
	 */
	-(void)setRequestData:(NSString*)RequestData {
		[super setInput:@"RequestData" toValue:RequestData];
	}

	/*!
	 * Set the value of the Comment input for this Choreo.
	*(conditional, string) A comment associated with the request.
	 */
	-(void)setComment:(NSString*)Comment {
		[super setInput:@"Comment" toValue:Comment];
	}

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(conditional, string) The ID of the request to update.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Priority input for this Choreo.
	*(conditional, string) Priority (e.g. low, normal, high, urgent). Defaults to normal.
	 */
	-(void)setPriority:(NSString*)Priority {
		[super setInput:@"Priority" toValue:Priority];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the Subject input for this Choreo.
	*(conditional, string) The subject of the request.
	 */
	-(void)setSubject:(NSString*)Subject {
		[super setInput:@"Subject" toValue:Subject];
	}

	/*!
	 * Set the value of the Type input for this Choreo.
	*(conditional, string) Type of request (e.g.question, incident, problem, task). Defaults to incident.
	 */
	-(void)setType:(NSString*)Type {
		[super setInput:@"Type" toValue:Type];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UpdateRequest Choreo.
 */
@implementation TMBZendesk_Requests_UpdateRequest_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Updates an existing request.
 */
@implementation TMBZendesk_Requests_UpdateRequest

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Requests_UpdateRequest Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Requests/UpdateRequest"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Requests_UpdateRequest_ResultSet *results = [[TMBZendesk_Requests_UpdateRequest_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UpdateRequest Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Requests_UpdateRequest_Inputs*)newInputSet {
		return [[TMBZendesk_Requests_UpdateRequest_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SearchAll Choreo.
 */
@implementation TMBZendesk_Search_SearchAll_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) The page number of the results to be returned. Used together with the PerPage parameter to paginate a large set of results.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the PerPage input for this Choreo.
	*(optional, integer) The number of results to return per page. Maximum is 100 and default is 100.
	 */
	-(void)setPerPage:(NSString*)PerPage {
		[super setInput:@"PerPage" toValue:PerPage];
	}

	/*!
	 * Set the value of the Query input for this Choreo.
	*(required, string) The search text to be matched.
	 */
	-(void)setQuery:(NSString*)Query {
		[super setInput:@"Query" toValue:Query];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the SortBy input for this Choreo.
	*(optional, string) Acceptable values: updated_at, created_at, priority, status, ticket_type.
	 */
	-(void)setSortBy:(NSString*)SortBy {
		[super setInput:@"SortBy" toValue:SortBy];
	}

	/*!
	 * Set the value of the SortOrder input for this Choreo.
	*(optional, string) Indicate either: relevance, asc (for ascending), desc (for descending). Defaults to relevance.
	 */
	-(void)setSortOrder:(NSString*)SortOrder {
		[super setInput:@"SortOrder" toValue:SortOrder];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SearchAll Choreo.
 */
@implementation TMBZendesk_Search_SearchAll_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NextPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the next page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextPage {
		return [super getOutputByName:@"NextPage"];
	}

	/*!
	 * Retrieve the value of the "PreviousPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the previous page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPreviousPage {
		return [super getOutputByName:@"PreviousPage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Returns search results across all ticket fields.
 */
@implementation TMBZendesk_Search_SearchAll

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Search_SearchAll Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Search/SearchAll"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Search_SearchAll_ResultSet *results = [[TMBZendesk_Search_SearchAll_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SearchAll Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Search_SearchAll_Inputs*)newInputSet {
		return [[TMBZendesk_Search_SearchAll_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SearchAnonymous Choreo.
 */
@implementation TMBZendesk_Search_SearchAnonymous_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) The page number of the results to be returned. Used together with the PerPage parameter to paginate a large set of results.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the PerPage input for this Choreo.
	*(optional, integer) The number of results to return per page. Maximum is 100 and default is 100.
	 */
	-(void)setPerPage:(NSString*)PerPage {
		[super setInput:@"PerPage" toValue:PerPage];
	}

	/*!
	 * Set the value of the Query input for this Choreo.
	*(required, string) The search text to be matched.
	 */
	-(void)setQuery:(NSString*)Query {
		[super setInput:@"Query" toValue:Query];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the SortBy input for this Choreo.
	*(optional, string) Acceptable values: updated_at, created_at, priority, status, ticket_type.
	 */
	-(void)setSortBy:(NSString*)SortBy {
		[super setInput:@"SortBy" toValue:SortBy];
	}

	/*!
	 * Set the value of the SortOrder input for this Choreo.
	*(optional, string) Indicate either: relevance, asc (for ascending), desc (for descending). Defaults to relevance.
	 */
	-(void)setSortOrder:(NSString*)SortOrder {
		[super setInput:@"SortOrder" toValue:SortOrder];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SearchAnonymous Choreo.
 */
@implementation TMBZendesk_Search_SearchAnonymous_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NextPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the next page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextPage {
		return [super getOutputByName:@"NextPage"];
	}

	/*!
	 * Retrieve the value of the "PreviousPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the previous page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPreviousPage {
		return [super getOutputByName:@"PreviousPage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Allows anonymous users to search public forums.
 */
@implementation TMBZendesk_Search_SearchAnonymous

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Search_SearchAnonymous Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Search/SearchAnonymous"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Search_SearchAnonymous_ResultSet *results = [[TMBZendesk_Search_SearchAnonymous_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SearchAnonymous Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Search_SearchAnonymous_Inputs*)newInputSet {
		return [[TMBZendesk_Search_SearchAnonymous_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListTags Choreo.
 */
@implementation TMBZendesk_Tags_ListTags_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) The page number of the results to be returned. Used together with the PerPage parameter to paginate a large set of results.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the PerPage input for this Choreo.
	*(optional, integer) The number of results to return per page. Maximum is 100 and default is 100.
	 */
	-(void)setPerPage:(NSString*)PerPage {
		[super setInput:@"PerPage" toValue:PerPage];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListTags Choreo.
 */
@implementation TMBZendesk_Tags_ListTags_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NextPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the next page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextPage {
		return [super getOutputByName:@"NextPage"];
	}

	/*!
	 * Retrieve the value of the "PreviousPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the previous page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPreviousPage {
		return [super getOutputByName:@"PreviousPage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the most popular recent tags in decreasing popularity.
 */
@implementation TMBZendesk_Tags_ListTags

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Tags_ListTags Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Tags/ListTags"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Tags_ListTags_ResultSet *results = [[TMBZendesk_Tags_ListTags_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListTags Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Tags_ListTags_Inputs*)newInputSet {
		return [[TMBZendesk_Tags_ListTags_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ChangeCommentFromPublicToPrivate Choreo.
 */
@implementation TMBZendesk_TicketAudits_ChangeCommentFromPublicToPrivate_Inputs

	/*!
	 * Set the value of the AuditID input for this Choreo.
	*(required, string) The ID of the audit to make private.
	 */
	-(void)setAuditID:(NSString*)AuditID {
		[super setInput:@"AuditID" toValue:AuditID];
	}

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the TicketID input for this Choreo.
	*(required, string) The ID of the ticket associated with the audit.
	 */
	-(void)setTicketID:(NSString*)TicketID {
		[super setInput:@"TicketID" toValue:TicketID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ChangeCommentFromPublicToPrivate Choreo.
 */
@implementation TMBZendesk_TicketAudits_ChangeCommentFromPublicToPrivate_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "ResponseStatusCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) The response status code returned from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponseStatusCode {
		return [super getOutputByName:@"ResponseStatusCode"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Changes a comment from public to private.
 */
@implementation TMBZendesk_TicketAudits_ChangeCommentFromPublicToPrivate

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_TicketAudits_ChangeCommentFromPublicToPrivate Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/TicketAudits/ChangeCommentFromPublicToPrivate"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_TicketAudits_ChangeCommentFromPublicToPrivate_ResultSet *results = [[TMBZendesk_TicketAudits_ChangeCommentFromPublicToPrivate_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ChangeCommentFromPublicToPrivate Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_TicketAudits_ChangeCommentFromPublicToPrivate_Inputs*)newInputSet {
		return [[TMBZendesk_TicketAudits_ChangeCommentFromPublicToPrivate_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListAudits Choreo.
 */
@implementation TMBZendesk_TicketAudits_ListAudits_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, string) The ID of the ticket you wish to show audits for.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) The page number of the results to be returned. Used together with the PerPage parameter to paginate a large set of results.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the PerPage input for this Choreo.
	*(optional, integer) The number of results to return per page. Maximum is 100 and default is 100.
	 */
	-(void)setPerPage:(NSString*)PerPage {
		[super setInput:@"PerPage" toValue:PerPage];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListAudits Choreo.
 */
@implementation TMBZendesk_TicketAudits_ListAudits_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NextPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the next page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextPage {
		return [super getOutputByName:@"NextPage"];
	}

	/*!
	 * Retrieve the value of the "PreviousPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the previous page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPreviousPage {
		return [super getOutputByName:@"PreviousPage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Lists all the audits made for the specifed ticket.
 */
@implementation TMBZendesk_TicketAudits_ListAudits

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_TicketAudits_ListAudits Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/TicketAudits/ListAudits"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_TicketAudits_ListAudits_ResultSet *results = [[TMBZendesk_TicketAudits_ListAudits_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListAudits Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_TicketAudits_ListAudits_Inputs*)newInputSet {
		return [[TMBZendesk_TicketAudits_ListAudits_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the MarkAuditAsTrusted Choreo.
 */
@implementation TMBZendesk_TicketAudits_MarkAuditAsTrusted_Inputs

	/*!
	 * Set the value of the AuditID input for this Choreo.
	*(required, string) The ID of the audit that you want to mark as trusted.
	 */
	-(void)setAuditID:(NSString*)AuditID {
		[super setInput:@"AuditID" toValue:AuditID];
	}

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the TicketID input for this Choreo.
	*(required, string) The ID of the ticket associated with the audit.
	 */
	-(void)setTicketID:(NSString*)TicketID {
		[super setInput:@"TicketID" toValue:TicketID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the MarkAuditAsTrusted Choreo.
 */
@implementation TMBZendesk_TicketAudits_MarkAuditAsTrusted_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "ResponseStatusCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) The response status code returned from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponseStatusCode {
		return [super getOutputByName:@"ResponseStatusCode"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Marks the specified audit as trusted. 
 */
@implementation TMBZendesk_TicketAudits_MarkAuditAsTrusted

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_TicketAudits_MarkAuditAsTrusted Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/TicketAudits/MarkAuditAsTrusted"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_TicketAudits_MarkAuditAsTrusted_ResultSet *results = [[TMBZendesk_TicketAudits_MarkAuditAsTrusted_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the MarkAuditAsTrusted Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_TicketAudits_MarkAuditAsTrusted_Inputs*)newInputSet {
		return [[TMBZendesk_TicketAudits_MarkAuditAsTrusted_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ShowAudit Choreo.
 */
@implementation TMBZendesk_TicketAudits_ShowAudit_Inputs

	/*!
	 * Set the value of the AuditID input for this Choreo.
	*(required, string) The ID of the audit to show.
	 */
	-(void)setAuditID:(NSString*)AuditID {
		[super setInput:@"AuditID" toValue:AuditID];
	}

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the TicketID input for this Choreo.
	*(required, string) The ID of the ticket associated with the audit.
	 */
	-(void)setTicketID:(NSString*)TicketID {
		[super setInput:@"TicketID" toValue:TicketID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ShowAudit Choreo.
 */
@implementation TMBZendesk_TicketAudits_ShowAudit_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Returns detailed information on the specified audit.
 */
@implementation TMBZendesk_TicketAudits_ShowAudit

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_TicketAudits_ShowAudit Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/TicketAudits/ShowAudit"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_TicketAudits_ShowAudit_ResultSet *results = [[TMBZendesk_TicketAudits_ShowAudit_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ShowAudit Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_TicketAudits_ShowAudit_Inputs*)newInputSet {
		return [[TMBZendesk_TicketAudits_ShowAudit_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetTicketMetrics Choreo.
 */
@implementation TMBZendesk_TicketMetrics_GetTicketMetrics_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(conditional, string) The ID of the ticket metric to retrieve.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetTicketMetrics Choreo.
 */
@implementation TMBZendesk_TicketMetrics_GetTicketMetrics_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a specific ticket metric.
 */
@implementation TMBZendesk_TicketMetrics_GetTicketMetrics

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_TicketMetrics_GetTicketMetrics Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/TicketMetrics/GetTicketMetrics"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_TicketMetrics_GetTicketMetrics_ResultSet *results = [[TMBZendesk_TicketMetrics_GetTicketMetrics_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetTicketMetrics Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_TicketMetrics_GetTicketMetrics_Inputs*)newInputSet {
		return [[TMBZendesk_TicketMetrics_GetTicketMetrics_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListTicketMetrics Choreo.
 */
@implementation TMBZendesk_TicketMetrics_ListTicketMetrics_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) The page number of the results to be returned. Used together with the PerPage parameter to paginate a large set of results.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the PerPage input for this Choreo.
	*(optional, integer) The number of results to return per page. Maximum is 100 and default is 100.
	 */
	-(void)setPerPage:(NSString*)PerPage {
		[super setInput:@"PerPage" toValue:PerPage];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListTicketMetrics Choreo.
 */
@implementation TMBZendesk_TicketMetrics_ListTicketMetrics_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NextPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the next page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextPage {
		return [super getOutputByName:@"NextPage"];
	}

	/*!
	 * Retrieve the value of the "PreviousPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the previous page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPreviousPage {
		return [super getOutputByName:@"PreviousPage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a list of metrics for all available tickets.
 */
@implementation TMBZendesk_TicketMetrics_ListTicketMetrics

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_TicketMetrics_ListTicketMetrics Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/TicketMetrics/ListTicketMetrics"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_TicketMetrics_ListTicketMetrics_ResultSet *results = [[TMBZendesk_TicketMetrics_ListTicketMetrics_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListTicketMetrics Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_TicketMetrics_ListTicketMetrics_Inputs*)newInputSet {
		return [[TMBZendesk_TicketMetrics_ListTicketMetrics_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CreateTicket Choreo.
 */
@implementation TMBZendesk_Tickets_CreateTicket_Inputs

	/*!
	 * Set the value of the Comment input for this Choreo.
	*(conditional, string) The comment for the ticket that is being created.
	 */
	-(void)setComment:(NSString*)Comment {
		[super setInput:@"Comment" toValue:Comment];
	}

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (i.e. temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the Subject input for this Choreo.
	*(conditional, string) The subject for the ticket that is being created.
	 */
	-(void)setSubject:(NSString*)Subject {
		[super setInput:@"Subject" toValue:Subject];
	}

	/*!
	 * Set the value of the TicketData input for this Choreo.
	*(optional, json) A JSON-formatted string containing the ticket properties you wish to set. This can be used as an alternative to setting individual inputs representing ticket properties.
	 */
	-(void)setTicketData:(NSString*)TicketData {
		[super setInput:@"TicketData" toValue:TicketData];
	}

	/*!
	 * Set the value of the Token input for this Choreo.
	*(optional, string) The token associated with an upload to attach to this ticket. Note that tokens can be retrieved by running the UploadFile Choreo.
	 */
	-(void)setToken:(NSString*)Token {
		[super setInput:@"Token" toValue:Token];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateTicket Choreo.
 */
@implementation TMBZendesk_Tickets_CreateTicket_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Creates a new ticket.
 */
@implementation TMBZendesk_Tickets_CreateTicket

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Tickets_CreateTicket Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Tickets/CreateTicket"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Tickets_CreateTicket_ResultSet *results = [[TMBZendesk_Tickets_CreateTicket_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CreateTicket Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Tickets_CreateTicket_Inputs*)newInputSet {
		return [[TMBZendesk_Tickets_CreateTicket_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CreateTicketNewRequester Choreo.
 */
@implementation TMBZendesk_Tickets_CreateTicketNewRequester_Inputs

	/*!
	 * Set the value of the Comment input for this Choreo.
	*(required, string) The comment for the ticket that is being created.
	 */
	-(void)setComment:(NSString*)Comment {
		[super setInput:@"Comment" toValue:Comment];
	}

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the LocaleID input for this Choreo.
	*(required, integer) LocaleID for the new requester. Indicate 1 for English, 8 for Deutsch, etc.
	 */
	-(void)setLocaleID:(NSString*)LocaleID {
		[super setInput:@"LocaleID" toValue:LocaleID];
	}

	/*!
	 * Set the value of the Name input for this Choreo.
	*(required, string) Name of new requester.
	 */
	-(void)setName:(NSString*)Name {
		[super setInput:@"Name" toValue:Name];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the RequesterEmail input for this Choreo.
	*(required, string) Email of new requester.
	 */
	-(void)setRequesterEmail:(NSString*)RequesterEmail {
		[super setInput:@"RequesterEmail" toValue:RequesterEmail];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the Subject input for this Choreo.
	*(required, string) The subject for the ticket that is being created.
	 */
	-(void)setSubject:(NSString*)Subject {
		[super setInput:@"Subject" toValue:Subject];
	}

	/*!
	 * Set the value of the Token input for this Choreo.
	*(optional, string) The token associated with an upload to attach to this ticket. Note that tokens can be retrieved by running the UploadFile Choreo.
	 */
	-(void)setToken:(NSString*)Token {
		[super setInput:@"Token" toValue:Token];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateTicketNewRequester Choreo.
 */
@implementation TMBZendesk_Tickets_CreateTicketNewRequester_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Creates a new ticket as well as a new requester account.
 */
@implementation TMBZendesk_Tickets_CreateTicketNewRequester

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Tickets_CreateTicketNewRequester Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Tickets/CreateTicketNewRequester"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Tickets_CreateTicketNewRequester_ResultSet *results = [[TMBZendesk_Tickets_CreateTicketNewRequester_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CreateTicketNewRequester Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Tickets_CreateTicketNewRequester_Inputs*)newInputSet {
		return [[TMBZendesk_Tickets_CreateTicketNewRequester_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteTicket Choreo.
 */
@implementation TMBZendesk_Tickets_DeleteTicket_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, integer) The ID number of a ticket.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteTicket Choreo.
 */
@implementation TMBZendesk_Tickets_DeleteTicket_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "ResponseStatusCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) The response status code returned from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponseStatusCode {
		return [super getOutputByName:@"ResponseStatusCode"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Deletes an existing ticket.
 */
@implementation TMBZendesk_Tickets_DeleteTicket

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Tickets_DeleteTicket Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Tickets/DeleteTicket"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Tickets_DeleteTicket_ResultSet *results = [[TMBZendesk_Tickets_DeleteTicket_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteTicket Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Tickets_DeleteTicket_Inputs*)newInputSet {
		return [[TMBZendesk_Tickets_DeleteTicket_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetTicket Choreo.
 */
@implementation TMBZendesk_Tickets_GetTicket_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, integer) The ID number of a ticket.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetTicket Choreo.
 */
@implementation TMBZendesk_Tickets_GetTicket_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Returns information about a ticket using its ID.
 */
@implementation TMBZendesk_Tickets_GetTicket

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Tickets_GetTicket Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Tickets/GetTicket"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Tickets_GetTicket_ResultSet *results = [[TMBZendesk_Tickets_GetTicket_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetTicket Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Tickets_GetTicket_Inputs*)newInputSet {
		return [[TMBZendesk_Tickets_GetTicket_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListAllTickets Choreo.
 */
@implementation TMBZendesk_Tickets_ListAllTickets_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) The page number of the results to be returned. Used together with the PerPage parameter to paginate a large set of results.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the PerPage input for this Choreo.
	*(optional, integer) The number of results to return per page. Maximum is 100 and default is 100.
	 */
	-(void)setPerPage:(NSString*)PerPage {
		[super setInput:@"PerPage" toValue:PerPage];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListAllTickets Choreo.
 */
@implementation TMBZendesk_Tickets_ListAllTickets_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NextPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the next page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextPage {
		return [super getOutputByName:@"NextPage"];
	}

	/*!
	 * Retrieve the value of the "PreviousPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the previous page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPreviousPage {
		return [super getOutputByName:@"PreviousPage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a list of all existing tickets.
 */
@implementation TMBZendesk_Tickets_ListAllTickets

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Tickets_ListAllTickets Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Tickets/ListAllTickets"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Tickets_ListAllTickets_ResultSet *results = [[TMBZendesk_Tickets_ListAllTickets_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListAllTickets Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Tickets_ListAllTickets_Inputs*)newInputSet {
		return [[TMBZendesk_Tickets_ListAllTickets_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListRecent Choreo.
 */
@implementation TMBZendesk_Tickets_ListRecent_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) The page number of the results to be returned. Used together with the PerPage parameter to paginate a large set of results.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the PerPage input for this Choreo.
	*(optional, integer) The number of results to return per page. Maximum is 100 and default is 100.
	 */
	-(void)setPerPage:(NSString*)PerPage {
		[super setInput:@"PerPage" toValue:PerPage];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListRecent Choreo.
 */
@implementation TMBZendesk_Tickets_ListRecent_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NextPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the next page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextPage {
		return [super getOutputByName:@"NextPage"];
	}

	/*!
	 * Retrieve the value of the "PreviousPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the previous page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPreviousPage {
		return [super getOutputByName:@"PreviousPage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a list of all recent tickets.
 */
@implementation TMBZendesk_Tickets_ListRecent

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Tickets_ListRecent Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Tickets/ListRecent"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Tickets_ListRecent_ResultSet *results = [[TMBZendesk_Tickets_ListRecent_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListRecent Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Tickets_ListRecent_Inputs*)newInputSet {
		return [[TMBZendesk_Tickets_ListRecent_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListTicketCollaborators Choreo.
 */
@implementation TMBZendesk_Tickets_ListTicketCollaborators_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, integer) The ID number of a ticket.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) The page number of the results to be returned. Used together with the PerPage parameter to paginate a large set of results.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the PerPage input for this Choreo.
	*(optional, integer) The number of results to return per page. Maximum is 100 and default is 100.
	 */
	-(void)setPerPage:(NSString*)PerPage {
		[super setInput:@"PerPage" toValue:PerPage];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListTicketCollaborators Choreo.
 */
@implementation TMBZendesk_Tickets_ListTicketCollaborators_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NextPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the next page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextPage {
		return [super getOutputByName:@"NextPage"];
	}

	/*!
	 * Retrieve the value of the "PreviousPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the previous page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPreviousPage {
		return [super getOutputByName:@"PreviousPage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves all the collaborators associated with a given ticket.
 */
@implementation TMBZendesk_Tickets_ListTicketCollaborators

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Tickets_ListTicketCollaborators Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Tickets/ListTicketCollaborators"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Tickets_ListTicketCollaborators_ResultSet *results = [[TMBZendesk_Tickets_ListTicketCollaborators_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListTicketCollaborators Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Tickets_ListTicketCollaborators_Inputs*)newInputSet {
		return [[TMBZendesk_Tickets_ListTicketCollaborators_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListTicketIncidents Choreo.
 */
@implementation TMBZendesk_Tickets_ListTicketIncidents_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, integer) The ID number of a ticket.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) The page number of the results to be returned. Used together with the PerPage parameter to paginate a large set of results.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the PerPage input for this Choreo.
	*(conditional, integer) The number of results to return per page. Maximum is 100 and default is 100.
	 */
	-(void)setPerPage:(NSString*)PerPage {
		[super setInput:@"PerPage" toValue:PerPage];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListTicketIncidents Choreo.
 */
@implementation TMBZendesk_Tickets_ListTicketIncidents_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NextPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the next page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextPage {
		return [super getOutputByName:@"NextPage"];
	}

	/*!
	 * Retrieve the value of the "PreviousPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the previous page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPreviousPage {
		return [super getOutputByName:@"PreviousPage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves all the incidents associated with a given ticket.
 */
@implementation TMBZendesk_Tickets_ListTicketIncidents

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Tickets_ListTicketIncidents Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Tickets/ListTicketIncidents"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Tickets_ListTicketIncidents_ResultSet *results = [[TMBZendesk_Tickets_ListTicketIncidents_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListTicketIncidents Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Tickets_ListTicketIncidents_Inputs*)newInputSet {
		return [[TMBZendesk_Tickets_ListTicketIncidents_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListTicketsByOrganization Choreo.
 */
@implementation TMBZendesk_Tickets_ListTicketsByOrganization_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the OrganizationID input for this Choreo.
	*(required, integer) The ID number of the organization.
	 */
	-(void)setOrganizationID:(NSString*)OrganizationID {
		[super setInput:@"OrganizationID" toValue:OrganizationID];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) The page number of the results to be returned. Used together with the PerPage parameter to paginate a large set of results.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the PerPage input for this Choreo.
	*(optional, integer) The number of results to return per page. Maximum is 100 and default is 100.
	 */
	-(void)setPerPage:(NSString*)PerPage {
		[super setInput:@"PerPage" toValue:PerPage];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListTicketsByOrganization Choreo.
 */
@implementation TMBZendesk_Tickets_ListTicketsByOrganization_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NextPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the next page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextPage {
		return [super getOutputByName:@"NextPage"];
	}

	/*!
	 * Retrieve the value of the "PreviousPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the previous page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPreviousPage {
		return [super getOutputByName:@"PreviousPage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a list of all tickets involving a specified organization.
 */
@implementation TMBZendesk_Tickets_ListTicketsByOrganization

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Tickets_ListTicketsByOrganization Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Tickets/ListTicketsByOrganization"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Tickets_ListTicketsByOrganization_ResultSet *results = [[TMBZendesk_Tickets_ListTicketsByOrganization_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListTicketsByOrganization Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Tickets_ListTicketsByOrganization_Inputs*)newInputSet {
		return [[TMBZendesk_Tickets_ListTicketsByOrganization_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListTicketsByUser Choreo.
 */
@implementation TMBZendesk_Tickets_ListTicketsByUser_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) The page number of the results to be returned. Used together with the PerPage parameter to paginate a large set of results.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the PerPage input for this Choreo.
	*(optional, integer) The number of results to return per page. Maximum is 100 and default is 100.
	 */
	-(void)setPerPage:(NSString*)PerPage {
		[super setInput:@"PerPage" toValue:PerPage];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(required, integer) The ID number of the user.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}

	/*!
	 * Set the value of the UserType input for this Choreo.
	*(optional, string) Specify "requested" to find tickets that the given user requested and "ccd" to find tickets on which a given user was CC'd. Defaults to searching for tickets that the user requested.
	 */
	-(void)setUserType:(NSString*)UserType {
		[super setInput:@"UserType" toValue:UserType];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListTicketsByUser Choreo.
 */
@implementation TMBZendesk_Tickets_ListTicketsByUser_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NextPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the next page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextPage {
		return [super getOutputByName:@"NextPage"];
	}

	/*!
	 * Retrieve the value of the "PreviousPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the previous page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPreviousPage {
		return [super getOutputByName:@"PreviousPage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a list of all tickets involving a specified user.
 */
@implementation TMBZendesk_Tickets_ListTicketsByUser

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Tickets_ListTicketsByUser Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Tickets/ListTicketsByUser"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Tickets_ListTicketsByUser_ResultSet *results = [[TMBZendesk_Tickets_ListTicketsByUser_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListTicketsByUser Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Tickets_ListTicketsByUser_Inputs*)newInputSet {
		return [[TMBZendesk_Tickets_ListTicketsByUser_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UpdateTicket Choreo.
 */
@implementation TMBZendesk_Tickets_UpdateTicket_Inputs

	/*!
	 * Set the value of the Comment input for this Choreo.
	*(conditional, string) The text for a ticket comment.
	 */
	-(void)setComment:(NSString*)Comment {
		[super setInput:@"Comment" toValue:Comment];
	}

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Metadata input for this Choreo.
	*(optional, json) Ticket metadata formatted in JSON. See below for more information on the input format.
	 */
	-(void)setMetadata:(NSString*)Metadata {
		[super setInput:@"Metadata" toValue:Metadata];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Public input for this Choreo.
	*(optional, boolean) A flag indicating if this update is public or not. Defaults to "true"
	 */
	-(void)setPublic:(NSString*)Public {
		[super setInput:@"Public" toValue:Public];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the Status input for this Choreo.
	*(conditional, string) The status of the ticket (i.e. solved, pending, open).
	 */
	-(void)setStatus:(NSString*)Status {
		[super setInput:@"Status" toValue:Status];
	}

	/*!
	 * Set the value of the TicketData input for this Choreo.
	*(optional, json) A JSON-formatted string containing the ticket properties you wish to set. This can be used as an alternative to setting individual inputs representing ticket properties.
	 */
	-(void)setTicketData:(NSString*)TicketData {
		[super setInput:@"TicketData" toValue:TicketData];
	}

	/*!
	 * Set the value of the TicketID input for this Choreo.
	*(required, integer) The id of the ticket being updated.
	 */
	-(void)setTicketID:(NSString*)TicketID {
		[super setInput:@"TicketID" toValue:TicketID];
	}

	/*!
	 * Set the value of the Token input for this Choreo.
	*(optional, string) The token associated with an upload to attach to this ticket. Note that tokens can be retrieved by running the UploadFile Choreo.
	 */
	-(void)setToken:(NSString*)Token {
		[super setInput:@"Token" toValue:Token];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UpdateTicket Choreo.
 */
@implementation TMBZendesk_Tickets_UpdateTicket_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Updates an existing ticket.
 */
@implementation TMBZendesk_Tickets_UpdateTicket

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Tickets_UpdateTicket Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Tickets/UpdateTicket"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Tickets_UpdateTicket_ResultSet *results = [[TMBZendesk_Tickets_UpdateTicket_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UpdateTicket Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Tickets_UpdateTicket_Inputs*)newInputSet {
		return [[TMBZendesk_Tickets_UpdateTicket_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the AddUserIdentity Choreo.
 */
@implementation TMBZendesk_UserIdentities_AddUserIdentity_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(conditional, string) The ID of the user.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the Identity input for this Choreo.
	*(required, string) The unique idenity (e.g.  test@test.com, test@gmail.com, screen_name)
	 */
	-(void)setIdentity:(NSString*)Identity {
		[super setInput:@"Identity" toValue:Identity];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the Type input for this Choreo.
	*(conditional, string) Type of identity to add (e.g. email, facebook, twitter, google)
	 */
	-(void)setType:(NSString*)Type {
		[super setInput:@"Type" toValue:Type];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the AddUserIdentity Choreo.
 */
@implementation TMBZendesk_UserIdentities_AddUserIdentity_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Allows an agent to add new identities for a given user ID. 
 */
@implementation TMBZendesk_UserIdentities_AddUserIdentity

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_UserIdentities_AddUserIdentity Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/UserIdentities/AddUserIdentity"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_UserIdentities_AddUserIdentity_ResultSet *results = [[TMBZendesk_UserIdentities_AddUserIdentity_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the AddUserIdentity Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_UserIdentities_AddUserIdentity_Inputs*)newInputSet {
		return [[TMBZendesk_UserIdentities_AddUserIdentity_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the AddUserIdentityByEndUser Choreo.
 */
@implementation TMBZendesk_UserIdentities_AddUserIdentityByEndUser_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(conditional, string) The ID of the user.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the Identity input for this Choreo.
	*(required, string) The unique idenity (e.g.  test@test.com, test@gmail.com, screen_name)
	 */
	-(void)setIdentity:(NSString*)Identity {
		[super setInput:@"Identity" toValue:Identity];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the Type input for this Choreo.
	*(conditional, string) Type of identity to add (e.g. email, facebook, twitter, google)
	 */
	-(void)setType:(NSString*)Type {
		[super setInput:@"Type" toValue:Type];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the AddUserIdentityByEndUser Choreo.
 */
@implementation TMBZendesk_UserIdentities_AddUserIdentityByEndUser_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 *  Allows an end-user  to add new identities for a given user ID.
 */
@implementation TMBZendesk_UserIdentities_AddUserIdentityByEndUser

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_UserIdentities_AddUserIdentityByEndUser Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/UserIdentities/AddUserIdentityByEndUser"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_UserIdentities_AddUserIdentityByEndUser_ResultSet *results = [[TMBZendesk_UserIdentities_AddUserIdentityByEndUser_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the AddUserIdentityByEndUser Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_UserIdentities_AddUserIdentityByEndUser_Inputs*)newInputSet {
		return [[TMBZendesk_UserIdentities_AddUserIdentityByEndUser_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteUserIdentity Choreo.
 */
@implementation TMBZendesk_UserIdentities_DeleteUserIdentity_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the IdentityID input for this Choreo.
	*(required, string) The ID of the Identity to delete.
	 */
	-(void)setIdentityID:(NSString*)IdentityID {
		[super setInput:@"IdentityID" toValue:IdentityID];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(conditional, string) The ID of the user.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteUserIdentity Choreo.
 */
@implementation TMBZendesk_UserIdentities_DeleteUserIdentity_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 *  Deletes the user identity for a user/
 */
@implementation TMBZendesk_UserIdentities_DeleteUserIdentity

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_UserIdentities_DeleteUserIdentity Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/UserIdentities/DeleteUserIdentity"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_UserIdentities_DeleteUserIdentity_ResultSet *results = [[TMBZendesk_UserIdentities_DeleteUserIdentity_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteUserIdentity Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_UserIdentities_DeleteUserIdentity_Inputs*)newInputSet {
		return [[TMBZendesk_UserIdentities_DeleteUserIdentity_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListUserIdentities Choreo.
 */
@implementation TMBZendesk_UserIdentities_ListUserIdentities_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(conditional, string) The ID of the user.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) The page number of the results to be returned. Used together with the PerPage parameter to paginate a large set of results.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the PerPage input for this Choreo.
	*(optional, integer) The number of results to return per page. Maximum is 100 and default is 100.
	 */
	-(void)setPerPage:(NSString*)PerPage {
		[super setInput:@"PerPage" toValue:PerPage];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListUserIdentities Choreo.
 */
@implementation TMBZendesk_UserIdentities_ListUserIdentities_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NextPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the next page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextPage {
		return [super getOutputByName:@"NextPage"];
	}

	/*!
	 * Retrieve the value of the "PreviousPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the previous page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPreviousPage {
		return [super getOutputByName:@"PreviousPage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Lists all User Identities for a given User
 */
@implementation TMBZendesk_UserIdentities_ListUserIdentities

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_UserIdentities_ListUserIdentities Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/UserIdentities/ListUserIdentities"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_UserIdentities_ListUserIdentities_ResultSet *results = [[TMBZendesk_UserIdentities_ListUserIdentities_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListUserIdentities Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_UserIdentities_ListUserIdentities_Inputs*)newInputSet {
		return [[TMBZendesk_UserIdentities_ListUserIdentities_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the MakeUserIdentityThePrimary Choreo.
 */
@implementation TMBZendesk_UserIdentities_MakeUserIdentityThePrimary_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the IdentityID input for this Choreo.
	*(conditional, string) ID of the Identity to be updated.
	 */
	-(void)setIdentityID:(NSString*)IdentityID {
		[super setInput:@"IdentityID" toValue:IdentityID];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(conditional, string) The ID of the user.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the MakeUserIdentityThePrimary Choreo.
 */
@implementation TMBZendesk_UserIdentities_MakeUserIdentityThePrimary_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Sets the given user identity to primary.
 */
@implementation TMBZendesk_UserIdentities_MakeUserIdentityThePrimary

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_UserIdentities_MakeUserIdentityThePrimary Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/UserIdentities/MakeUserIdentityThePrimary"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_UserIdentities_MakeUserIdentityThePrimary_ResultSet *results = [[TMBZendesk_UserIdentities_MakeUserIdentityThePrimary_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the MakeUserIdentityThePrimary Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_UserIdentities_MakeUserIdentityThePrimary_Inputs*)newInputSet {
		return [[TMBZendesk_UserIdentities_MakeUserIdentityThePrimary_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the RequestUserVerification Choreo.
 */
@implementation TMBZendesk_UserIdentities_RequestUserVerification_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the IdentityID input for this Choreo.
	*(conditional, string) The ID of the Identity to be verified.
	 */
	-(void)setIdentityID:(NSString*)IdentityID {
		[super setInput:@"IdentityID" toValue:IdentityID];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(conditional, string) The ID of the user.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the RequestUserVerification Choreo.
 */
@implementation TMBZendesk_UserIdentities_RequestUserVerification_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * This sends a verification email to the user, asking him to click a link in order to verify ownership of the email address.
 */
@implementation TMBZendesk_UserIdentities_RequestUserVerification

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_UserIdentities_RequestUserVerification Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/UserIdentities/RequestUserVerification"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_UserIdentities_RequestUserVerification_ResultSet *results = [[TMBZendesk_UserIdentities_RequestUserVerification_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the RequestUserVerification Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_UserIdentities_RequestUserVerification_Inputs*)newInputSet {
		return [[TMBZendesk_UserIdentities_RequestUserVerification_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ShowUserIdentity Choreo.
 */
@implementation TMBZendesk_UserIdentities_ShowUserIdentity_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the IdentityID input for this Choreo.
	*(required, string) The ID of the Identity.
	 */
	-(void)setIdentityID:(NSString*)IdentityID {
		[super setInput:@"IdentityID" toValue:IdentityID];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(conditional, string) The ID of the user.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ShowUserIdentity Choreo.
 */
@implementation TMBZendesk_UserIdentities_ShowUserIdentity_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Shows the identity with the given ID.
 */
@implementation TMBZendesk_UserIdentities_ShowUserIdentity

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_UserIdentities_ShowUserIdentity Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/UserIdentities/ShowUserIdentity"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_UserIdentities_ShowUserIdentity_ResultSet *results = [[TMBZendesk_UserIdentities_ShowUserIdentity_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ShowUserIdentity Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_UserIdentities_ShowUserIdentity_Inputs*)newInputSet {
		return [[TMBZendesk_UserIdentities_ShowUserIdentity_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UpdateUserIdentity Choreo.
 */
@implementation TMBZendesk_UserIdentities_UpdateUserIdentity_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the IdentityID input for this Choreo.
	*(conditional, string) The ID of the Identity to be updated.
	 */
	-(void)setIdentityID:(NSString*)IdentityID {
		[super setInput:@"IdentityID" toValue:IdentityID];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(conditional, string) The ID of the user.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}

	/*!
	 * Set the value of the Verified input for this Choreo.
	*(required, string) Specifies whether the identity has been verified (true or false).
	 */
	-(void)setVerified:(NSString*)Verified {
		[super setInput:@"Verified" toValue:Verified];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UpdateUserIdentity Choreo.
 */
@implementation TMBZendesk_UserIdentities_UpdateUserIdentity_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Updates the User Identity.    This API method only allows you to set an identity as verified.
 */
@implementation TMBZendesk_UserIdentities_UpdateUserIdentity

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_UserIdentities_UpdateUserIdentity Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/UserIdentities/UpdateUserIdentity"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_UserIdentities_UpdateUserIdentity_ResultSet *results = [[TMBZendesk_UserIdentities_UpdateUserIdentity_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UpdateUserIdentity Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_UserIdentities_UpdateUserIdentity_Inputs*)newInputSet {
		return [[TMBZendesk_UserIdentities_UpdateUserIdentity_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the VerifyUserIdentity Choreo.
 */
@implementation TMBZendesk_UserIdentities_VerifyUserIdentity_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the IdentityID input for this Choreo.
	*(conditional, string) The ID of the Identity to verify.
	 */
	-(void)setIdentityID:(NSString*)IdentityID {
		[super setInput:@"IdentityID" toValue:IdentityID];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(conditional, string) The ID of the user.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the VerifyUserIdentity Choreo.
 */
@implementation TMBZendesk_UserIdentities_VerifyUserIdentity_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Verifies the User Identity.   
 */
@implementation TMBZendesk_UserIdentities_VerifyUserIdentity

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_UserIdentities_VerifyUserIdentity Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/UserIdentities/VerifyUserIdentity"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_UserIdentities_VerifyUserIdentity_ResultSet *results = [[TMBZendesk_UserIdentities_VerifyUserIdentity_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the VerifyUserIdentity Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_UserIdentities_VerifyUserIdentity_Inputs*)newInputSet {
		return [[TMBZendesk_UserIdentities_VerifyUserIdentity_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CreateManyUsers Choreo.
 */
@implementation TMBZendesk_Users_CreateManyUsers_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the Users input for this Choreo.
	*(required, json) A JSON-formatted string containing an array of user properties you wish to set.
	 */
	-(void)setUsers:(NSString*)Users {
		[super setInput:@"Users" toValue:Users];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateManyUsers Choreo.
 */
@implementation TMBZendesk_Users_CreateManyUsers_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) 
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Creates many new users at one time.
 */
@implementation TMBZendesk_Users_CreateManyUsers

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Users_CreateManyUsers Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Users/CreateManyUsers"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Users_CreateManyUsers_ResultSet *results = [[TMBZendesk_Users_CreateManyUsers_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CreateManyUsers Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Users_CreateManyUsers_Inputs*)newInputSet {
		return [[TMBZendesk_Users_CreateManyUsers_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CreateUser Choreo.
 */
@implementation TMBZendesk_Users_CreateUser_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the UserData input for this Choreo.
	*(required, json) A JSON-formatted string containing the user properties you wish to set.
	 */
	-(void)setUserData:(NSString*)UserData {
		[super setInput:@"UserData" toValue:UserData];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateUser Choreo.
 */
@implementation TMBZendesk_Users_CreateUser_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) 
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Creates a new user.
 */
@implementation TMBZendesk_Users_CreateUser

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Users_CreateUser Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Users/CreateUser"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Users_CreateUser_ResultSet *results = [[TMBZendesk_Users_CreateUser_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CreateUser Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Users_CreateUser_Inputs*)newInputSet {
		return [[TMBZendesk_Users_CreateUser_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteUser Choreo.
 */
@implementation TMBZendesk_Users_DeleteUser_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, integer) The ID of the user to delete.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteUser Choreo.
 */
@implementation TMBZendesk_Users_DeleteUser_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "ResponseStatusCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) The response status code returned from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponseStatusCode {
		return [super getOutputByName:@"ResponseStatusCode"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Deletes a given user.
 */
@implementation TMBZendesk_Users_DeleteUser

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Users_DeleteUser Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Users/DeleteUser"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Users_DeleteUser_ResultSet *results = [[TMBZendesk_Users_DeleteUser_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteUser Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Users_DeleteUser_Inputs*)newInputSet {
		return [[TMBZendesk_Users_DeleteUser_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListAllUsers Choreo.
 */
@implementation TMBZendesk_Users_ListAllUsers_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) The page number of the results to be returned. Used together with the PerPage parameter to paginate a large set of results.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the PerPage input for this Choreo.
	*(optional, integer) The number of results to return per page. Maximum is 100 and default is 100.
	 */
	-(void)setPerPage:(NSString*)PerPage {
		[super setInput:@"PerPage" toValue:PerPage];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListAllUsers Choreo.
 */
@implementation TMBZendesk_Users_ListAllUsers_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NextPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the next page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextPage {
		return [super getOutputByName:@"NextPage"];
	}

	/*!
	 * Retrieve the value of the "PreviousPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the previous page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPreviousPage {
		return [super getOutputByName:@"PreviousPage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a list of all existing users.
 */
@implementation TMBZendesk_Users_ListAllUsers

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Users_ListAllUsers Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Users/ListAllUsers"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Users_ListAllUsers_ResultSet *results = [[TMBZendesk_Users_ListAllUsers_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListAllUsers Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Users_ListAllUsers_Inputs*)newInputSet {
		return [[TMBZendesk_Users_ListAllUsers_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListUsersByGroup Choreo.
 */
@implementation TMBZendesk_Users_ListUsersByGroup_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the GroupID input for this Choreo.
	*(required, integer) The ID number of the group.
	 */
	-(void)setGroupID:(NSString*)GroupID {
		[super setInput:@"GroupID" toValue:GroupID];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) The page number of the results to be returned. Used together with the PerPage parameter to paginate a large set of results.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the PerPage input for this Choreo.
	*(optional, integer) The number of results to return per page. Maximum is 100 and default is 100.
	 */
	-(void)setPerPage:(NSString*)PerPage {
		[super setInput:@"PerPage" toValue:PerPage];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListUsersByGroup Choreo.
 */
@implementation TMBZendesk_Users_ListUsersByGroup_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NextPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the next page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextPage {
		return [super getOutputByName:@"NextPage"];
	}

	/*!
	 * Retrieve the value of the "PreviousPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the previous page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPreviousPage {
		return [super getOutputByName:@"PreviousPage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a list of all users in a specified group.
 */
@implementation TMBZendesk_Users_ListUsersByGroup

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Users_ListUsersByGroup Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Users/ListUsersByGroup"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Users_ListUsersByGroup_ResultSet *results = [[TMBZendesk_Users_ListUsersByGroup_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListUsersByGroup Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Users_ListUsersByGroup_Inputs*)newInputSet {
		return [[TMBZendesk_Users_ListUsersByGroup_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListUsersByOrganization Choreo.
 */
@implementation TMBZendesk_Users_ListUsersByOrganization_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the OrganizationID input for this Choreo.
	*(required, integer) The ID number of the organization.
	 */
	-(void)setOrganizationID:(NSString*)OrganizationID {
		[super setInput:@"OrganizationID" toValue:OrganizationID];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) The page number of the results to be returned. Used together with the PerPage parameter to paginate a large set of results.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the PerPage input for this Choreo.
	*(optional, integer) The number of results to return per page. Maximum is 100 and default is 100.
	 */
	-(void)setPerPage:(NSString*)PerPage {
		[super setInput:@"PerPage" toValue:PerPage];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListUsersByOrganization Choreo.
 */
@implementation TMBZendesk_Users_ListUsersByOrganization_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NextPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the next page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextPage {
		return [super getOutputByName:@"NextPage"];
	}

	/*!
	 * Retrieve the value of the "PreviousPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the previous page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPreviousPage {
		return [super getOutputByName:@"PreviousPage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a list of users involving a specified organization.
 */
@implementation TMBZendesk_Users_ListUsersByOrganization

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Users_ListUsersByOrganization Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Users/ListUsersByOrganization"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Users_ListUsersByOrganization_ResultSet *results = [[TMBZendesk_Users_ListUsersByOrganization_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListUsersByOrganization Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Users_ListUsersByOrganization_Inputs*)newInputSet {
		return [[TMBZendesk_Users_ListUsersByOrganization_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SearchUsers Choreo.
 */
@implementation TMBZendesk_Users_SearchUsers_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) The page number of the results to be returned. Used together with the PerPage parameter to paginate a large set of results.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the PerPage input for this Choreo.
	*(optional, integer) The number of results to return per page. Maximum is 100 and default is 100.
	 */
	-(void)setPerPage:(NSString*)PerPage {
		[super setInput:@"PerPage" toValue:PerPage];
	}

	/*!
	 * Set the value of the Query input for this Choreo.
	*(required, string) Perform a search across usernames or email addresses.
	 */
	-(void)setQuery:(NSString*)Query {
		[super setInput:@"Query" toValue:Query];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SearchUsers Choreo.
 */
@implementation TMBZendesk_Users_SearchUsers_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NextPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the next page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextPage {
		return [super getOutputByName:@"NextPage"];
	}

	/*!
	 * Retrieve the value of the "PreviousPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the previous page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPreviousPage {
		return [super getOutputByName:@"PreviousPage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Returns a list of users who match the search parameters.
 */
@implementation TMBZendesk_Users_SearchUsers

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Users_SearchUsers Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Users/SearchUsers"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Users_SearchUsers_ResultSet *results = [[TMBZendesk_Users_SearchUsers_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SearchUsers Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Users_SearchUsers_Inputs*)newInputSet {
		return [[TMBZendesk_Users_SearchUsers_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ShowCurrentUser Choreo.
 */
@implementation TMBZendesk_Users_ShowCurrentUser_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ShowCurrentUser Choreo.
 */
@implementation TMBZendesk_Users_ShowCurrentUser_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Returns information about the currently authenticated user.
 */
@implementation TMBZendesk_Users_ShowCurrentUser

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Users_ShowCurrentUser Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Users/ShowCurrentUser"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Users_ShowCurrentUser_ResultSet *results = [[TMBZendesk_Users_ShowCurrentUser_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ShowCurrentUser Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Users_ShowCurrentUser_Inputs*)newInputSet {
		return [[TMBZendesk_Users_ShowCurrentUser_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ShowUser Choreo.
 */
@implementation TMBZendesk_Users_ShowUser_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, integer) The ID number of a user.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ShowUser Choreo.
 */
@implementation TMBZendesk_Users_ShowUser_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Returns information about given user.
 */
@implementation TMBZendesk_Users_ShowUser

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Users_ShowUser Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Users/ShowUser"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Users_ShowUser_ResultSet *results = [[TMBZendesk_Users_ShowUser_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ShowUser Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Users_ShowUser_Inputs*)newInputSet {
		return [[TMBZendesk_Users_ShowUser_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SuspendUser Choreo.
 */
@implementation TMBZendesk_Users_SuspendUser_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(required, integer) The id of the user being updated.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SuspendUser Choreo.
 */
@implementation TMBZendesk_Users_SuspendUser_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Suspends an existing user.
 */
@implementation TMBZendesk_Users_SuspendUser

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Users_SuspendUser Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Users/SuspendUser"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Users_SuspendUser_ResultSet *results = [[TMBZendesk_Users_SuspendUser_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SuspendUser Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Users_SuspendUser_Inputs*)newInputSet {
		return [[TMBZendesk_Users_SuspendUser_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UpdateUser Choreo.
 */
@implementation TMBZendesk_Users_UpdateUser_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the UserData input for this Choreo.
	*(required, json) A JSON-formatted string containing the user properties you wish to set.
	 */
	-(void)setUserData:(NSString*)UserData {
		[super setInput:@"UserData" toValue:UserData];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(required, integer) The ID of the user being updated.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UpdateUser Choreo.
 */
@implementation TMBZendesk_Users_UpdateUser_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Updates an existing user.
 */
@implementation TMBZendesk_Users_UpdateUser

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Users_UpdateUser Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Users/UpdateUser"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Users_UpdateUser_ResultSet *results = [[TMBZendesk_Users_UpdateUser_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UpdateUser Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Users_UpdateUser_Inputs*)newInputSet {
		return [[TMBZendesk_Users_UpdateUser_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UpdateUserImage Choreo.
 */
@implementation TMBZendesk_Users_UpdateUserImage_Inputs

	/*!
	 * Set the value of the Response input for this Choreo.
	*(required, json) The response from Zendesk.
	 */
	-(void)setResponse:(NSString*)Response {
		[super setInput:@"Response" toValue:Response];
	}

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the ImageURL input for this Choreo.
	*(required, string) The URL of the profile image.
	 */
	-(void)setImageURL:(NSString*)ImageURL {
		[super setInput:@"ImageURL" toValue:ImageURL];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(required, integer) The ID of the user being updated.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UpdateUserImage Choreo.
 */
@implementation TMBZendesk_Users_UpdateUserImage_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (required, json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Updates a user's profile image.
 */
@implementation TMBZendesk_Users_UpdateUserImage

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Users_UpdateUserImage Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Users/UpdateUserImage"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Users_UpdateUserImage_ResultSet *results = [[TMBZendesk_Users_UpdateUserImage_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UpdateUserImage Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Users_UpdateUserImage_Inputs*)newInputSet {
		return [[TMBZendesk_Users_UpdateUserImage_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CreateView Choreo.
 */
@implementation TMBZendesk_Views_CreateView_Inputs

	/*!
	 * Set the value of the AllConditions input for this Choreo.
	*(conditional, json) JSON Array of conditions.  Tickets must fulfill all of the conditions to be considered matching.
	 */
	-(void)setAllConditions:(NSString*)AllConditions {
		[super setInput:@"AllConditions" toValue:AllConditions];
	}

	/*!
	 * Set the value of the AnyConditions input for this Choreo.
	*(conditional, json) JSON Array of conditions.  Tickets may fulfill any of the conditions to be considered matching.
	 */
	-(void)setAnyConditions:(NSString*)AnyConditions {
		[super setInput:@"AnyConditions" toValue:AnyConditions];
	}

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the Title input for this Choreo.
	*(required, string) Title of the view to be created.
	 */
	-(void)setTitle:(NSString*)Title {
		[super setInput:@"Title" toValue:Title];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateView Choreo.
 */
@implementation TMBZendesk_Views_CreateView_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Creates a new view.
 */
@implementation TMBZendesk_Views_CreateView

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Views_CreateView Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Views/CreateView"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Views_CreateView_ResultSet *results = [[TMBZendesk_Views_CreateView_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CreateView Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Views_CreateView_Inputs*)newInputSet {
		return [[TMBZendesk_Views_CreateView_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteView Choreo.
 */
@implementation TMBZendesk_Views_DeleteView_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(conditional, string) ID of the view to delete.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteView Choreo.
 */
@implementation TMBZendesk_Views_DeleteView_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "ResponseStatusCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) The response status code returned from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponseStatusCode {
		return [super getOutputByName:@"ResponseStatusCode"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Deletes the specified view.
 */
@implementation TMBZendesk_Views_DeleteView

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Views_DeleteView Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Views/DeleteView"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Views_DeleteView_ResultSet *results = [[TMBZendesk_Views_DeleteView_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteView Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Views_DeleteView_Inputs*)newInputSet {
		return [[TMBZendesk_Views_DeleteView_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ExecuteViews Choreo.
 */
@implementation TMBZendesk_Views_ExecuteViews_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(conditional, string) ID of the view to execute.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ExecuteViews Choreo.
 */
@implementation TMBZendesk_Views_ExecuteViews_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Execute a view and retrieve tickets that fulfill the conditions of the view.
 */
@implementation TMBZendesk_Views_ExecuteViews

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Views_ExecuteViews Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Views/ExecuteViews"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Views_ExecuteViews_ResultSet *results = [[TMBZendesk_Views_ExecuteViews_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ExecuteViews Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Views_ExecuteViews_Inputs*)newInputSet {
		return [[TMBZendesk_Views_ExecuteViews_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ExportViews Choreo.
 */
@implementation TMBZendesk_Views_ExportViews_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(conditional, string) ID of the view to export.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ExportViews Choreo.
 */
@implementation TMBZendesk_Views_ExportViews_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Returns the CSV attachment of the current view if possible.
 */
@implementation TMBZendesk_Views_ExportViews

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Views_ExportViews Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Views/ExportViews"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Views_ExportViews_ResultSet *results = [[TMBZendesk_Views_ExportViews_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ExportViews Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Views_ExportViews_Inputs*)newInputSet {
		return [[TMBZendesk_Views_ExportViews_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetTicketsFromAView Choreo.
 */
@implementation TMBZendesk_Views_GetTicketsFromAView_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(conditional, string) ID of the view to get tickets from.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetTicketsFromAView Choreo.
 */
@implementation TMBZendesk_Views_GetTicketsFromAView_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieve tickets from a view.
 */
@implementation TMBZendesk_Views_GetTicketsFromAView

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Views_GetTicketsFromAView Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Views/GetTicketsFromAView"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Views_GetTicketsFromAView_ResultSet *results = [[TMBZendesk_Views_GetTicketsFromAView_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetTicketsFromAView Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Views_GetTicketsFromAView_Inputs*)newInputSet {
		return [[TMBZendesk_Views_GetTicketsFromAView_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetViewCount Choreo.
 */
@implementation TMBZendesk_Views_GetViewCount_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(conditional, string) Retrieve a view count for the ID of the specified view.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetViewCount Choreo.
 */
@implementation TMBZendesk_Views_GetViewCount_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Returns the ticket count for a single view.
 */
@implementation TMBZendesk_Views_GetViewCount

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Views_GetViewCount Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Views/GetViewCount"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Views_GetViewCount_ResultSet *results = [[TMBZendesk_Views_GetViewCount_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetViewCount Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Views_GetViewCount_Inputs*)newInputSet {
		return [[TMBZendesk_Views_GetViewCount_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetViewCounts Choreo.
 */
@implementation TMBZendesk_Views_GetViewCounts_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the IDs input for this Choreo.
	*(conditional, string) Comma-seperated list of view IDs to retrieve counts for.
	 */
	-(void)setIDs:(NSString*)IDs {
		[super setInput:@"IDs" toValue:IDs];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetViewCounts Choreo.
 */
@implementation TMBZendesk_Views_GetViewCounts_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Calculates the size of the view in terms of number of tickets the view will return.
 */
@implementation TMBZendesk_Views_GetViewCounts

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Views_GetViewCounts Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Views/GetViewCounts"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Views_GetViewCounts_ResultSet *results = [[TMBZendesk_Views_GetViewCounts_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetViewCounts Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Views_GetViewCounts_Inputs*)newInputSet {
		return [[TMBZendesk_Views_GetViewCounts_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListActiveViews Choreo.
 */
@implementation TMBZendesk_Views_ListActiveViews_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) The page number of the results to be returned. Used together with the PerPage parameter to paginate a large set of results.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the PerPage input for this Choreo.
	*(optional, integer) The number of results to return per page. Maximum is 100 and default is 100.
	 */
	-(void)setPerPage:(NSString*)PerPage {
		[super setInput:@"PerPage" toValue:PerPage];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListActiveViews Choreo.
 */
@implementation TMBZendesk_Views_ListActiveViews_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NextPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the next page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextPage {
		return [super getOutputByName:@"NextPage"];
	}

	/*!
	 * Retrieve the value of the "PreviousPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the previous page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPreviousPage {
		return [super getOutputByName:@"PreviousPage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Lists active shared and personal views available to the current user.
 */
@implementation TMBZendesk_Views_ListActiveViews

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Views_ListActiveViews Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Views/ListActiveViews"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Views_ListActiveViews_ResultSet *results = [[TMBZendesk_Views_ListActiveViews_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListActiveViews Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Views_ListActiveViews_Inputs*)newInputSet {
		return [[TMBZendesk_Views_ListActiveViews_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListCompactViews Choreo.
 */
@implementation TMBZendesk_Views_ListCompactViews_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) The page number of the results to be returned. Used together with the PerPage parameter to paginate a large set of results.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the PerPage input for this Choreo.
	*(optional, integer) The number of results to return per page. Maximum is 100 and default is 100.
	 */
	-(void)setPerPage:(NSString*)PerPage {
		[super setInput:@"PerPage" toValue:PerPage];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListCompactViews Choreo.
 */
@implementation TMBZendesk_Views_ListCompactViews_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NextPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the next page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextPage {
		return [super getOutputByName:@"NextPage"];
	}

	/*!
	 * Retrieve the value of the "PreviousPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the previous page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPreviousPage {
		return [super getOutputByName:@"PreviousPage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a compacted list of shared and personal views available to the current user.
 */
@implementation TMBZendesk_Views_ListCompactViews

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Views_ListCompactViews Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Views/ListCompactViews"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Views_ListCompactViews_ResultSet *results = [[TMBZendesk_Views_ListCompactViews_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListCompactViews Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Views_ListCompactViews_Inputs*)newInputSet {
		return [[TMBZendesk_Views_ListCompactViews_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListViews Choreo.
 */
@implementation TMBZendesk_Views_ListViews_Inputs

	/*!
	 * Set the value of the Email input for this Choreo.
	*(required, string) The email address you use to login to your Zendesk account.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) The page number of the results to be returned. Used together with the PerPage parameter to paginate a large set of results.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Your Zendesk password.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the PerPage input for this Choreo.
	*(optional, integer) The number of results to return per page. Maximum is 100 and default is 100.
	 */
	-(void)setPerPage:(NSString*)PerPage {
		[super setInput:@"PerPage" toValue:PerPage];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) Your Zendesk domain and subdomain (e.g., temboocare.zendesk.com).
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListViews Choreo.
 */
@implementation TMBZendesk_Views_ListViews_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NextPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the next page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextPage {
		return [super getOutputByName:@"NextPage"];
	}

	/*!
	 * Retrieve the value of the "PreviousPage" output from an execution of this Choreo.
	 * @return - NSString* (integer) The index for the previous page of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPreviousPage {
		return [super getOutputByName:@"PreviousPage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Zendesk.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Lists shared and personal views available to the current user.
 */
@implementation TMBZendesk_Views_ListViews

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBZendesk_Views_ListViews Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Zendesk/Views/ListViews"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBZendesk_Views_ListViews_ResultSet *results = [[TMBZendesk_Views_ListViews_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListViews Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBZendesk_Views_ListViews_Inputs*)newInputSet {
		return [[TMBZendesk_Views_ListViews_Inputs alloc] init];
	}
@end
	