/*!
 * @TMBTwitter.m
 *
 * Execute Choreographies from the Temboo Twitter bundle.
 *
 * iOS version 10.6.8
 *
 * LICENSE: Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @category   Services
 * @author     Reid Simonton, Temboo, Inc.
 * @copyright  2013 Temboo, Inc.
 * @license    http://www.apache.org/licenses/LICENSE-2.0 Apache License 2.0
 * @version    1.7
 */

#import "TMBTwitter.h"
#import "TMBChoreography_Protected.h"
#import "TMBChoreographyExecution.h"
#import "TMBTembooSession.h"
#import "TMBTembooUri.h"


/*!
 * Input object with appropriate setters for specifying arguments to the DestroyDirectMessage Choreo.
 */
@implementation TMBTwitter_DirectMessages_DestroyDirectMessage_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, string) The ID of the direct message.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the IncludeEntities input for this Choreo.
	*(optional, boolean) The "entities" node containing extra metadata will not be included when set to false.
	 */
	-(void)setIncludeEntities:(NSString*)IncludeEntities {
		[super setInput:@"IncludeEntities" toValue:IncludeEntities];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DestroyDirectMessage Choreo.
 */
@implementation TMBTwitter_DirectMessages_DestroyDirectMessage_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Deletes a specified Direct Message.
 */
@implementation TMBTwitter_DirectMessages_DestroyDirectMessage

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_DirectMessages_DestroyDirectMessage Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/DirectMessages/DestroyDirectMessage"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_DirectMessages_DestroyDirectMessage_ResultSet *results = [[TMBTwitter_DirectMessages_DestroyDirectMessage_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DestroyDirectMessage Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_DirectMessages_DestroyDirectMessage_Inputs*)newInputSet {
		return [[TMBTwitter_DirectMessages_DestroyDirectMessage_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DirectMessagesSent Choreo.
 */
@implementation TMBTwitter_DirectMessages_DirectMessagesSent_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Count input for this Choreo.
	*(optional, integer) Specifies the number of records to retrieve up to a maximum of 200.
	 */
	-(void)setCount:(NSString*)Count {
		[super setInput:@"Count" toValue:Count];
	}

	/*!
	 * Set the value of the IncludeEntities input for this Choreo.
	*(optional, boolean) The "entities" node containing extra metadata will not be included when set to false.
	 */
	-(void)setIncludeEntities:(NSString*)IncludeEntities {
		[super setInput:@"IncludeEntities" toValue:IncludeEntities];
	}

	/*!
	 * Set the value of the MaxID input for this Choreo.
	*(optional, string) Returns results with an ID less than (older than) or equal to the specified ID.
	 */
	-(void)setMaxID:(NSString*)MaxID {
		[super setInput:@"MaxID" toValue:MaxID];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) Specifies the page of results to retrieve.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the SinceID input for this Choreo.
	*(optional, string) Returns results with an ID greater than (more recent than) the specified ID.
	 */
	-(void)setSinceID:(NSString*)SinceID {
		[super setInput:@"SinceID" toValue:SinceID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DirectMessagesSent Choreo.
 */
@implementation TMBTwitter_DirectMessages_DirectMessagesSent_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the 20 most recent direct messages sent by the authenticating user.
 */
@implementation TMBTwitter_DirectMessages_DirectMessagesSent

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_DirectMessages_DirectMessagesSent Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/DirectMessages/DirectMessagesSent"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_DirectMessages_DirectMessagesSent_ResultSet *results = [[TMBTwitter_DirectMessages_DirectMessagesSent_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DirectMessagesSent Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_DirectMessages_DirectMessagesSent_Inputs*)newInputSet {
		return [[TMBTwitter_DirectMessages_DirectMessagesSent_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetDirectMessages Choreo.
 */
@implementation TMBTwitter_DirectMessages_GetDirectMessages_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Count input for this Choreo.
	*(optional, integer) Specifies the number of records to retrieve up to a maximum of 200.
	 */
	-(void)setCount:(NSString*)Count {
		[super setInput:@"Count" toValue:Count];
	}

	/*!
	 * Set the value of the IncludeEntities input for this Choreo.
	*(optional, boolean) The "entities" node containing extra metadata will not be included when set to false.
	 */
	-(void)setIncludeEntities:(NSString*)IncludeEntities {
		[super setInput:@"IncludeEntities" toValue:IncludeEntities];
	}

	/*!
	 * Set the value of the MaxID input for this Choreo.
	*(optional, string) Returns results with an ID less than (older than) or equal to the specified ID.
	 */
	-(void)setMaxID:(NSString*)MaxID {
		[super setInput:@"MaxID" toValue:MaxID];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) Specifies the page of results to retrieve.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the SinceID input for this Choreo.
	*(optional, string) Returns results with an ID greater than (more recent than) the specified ID.
	 */
	-(void)setSinceID:(NSString*)SinceID {
		[super setInput:@"SinceID" toValue:SinceID];
	}

	/*!
	 * Set the value of the SkipStatus input for this Choreo.
	*(optional, boolean) When set to true, statuses will not be included in the returned user objects.
	 */
	-(void)setSkipStatus:(NSString*)SkipStatus {
		[super setInput:@"SkipStatus" toValue:SkipStatus];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetDirectMessages Choreo.
 */
@implementation TMBTwitter_DirectMessages_GetDirectMessages_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the 20 most recent direct messages sent to the authenticating user.
 */
@implementation TMBTwitter_DirectMessages_GetDirectMessages

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_DirectMessages_GetDirectMessages Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/DirectMessages/GetDirectMessages"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_DirectMessages_GetDirectMessages_ResultSet *results = [[TMBTwitter_DirectMessages_GetDirectMessages_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetDirectMessages Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_DirectMessages_GetDirectMessages_Inputs*)newInputSet {
		return [[TMBTwitter_DirectMessages_GetDirectMessages_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetLatestDirectMessage Choreo.
 */
@implementation TMBTwitter_DirectMessages_GetLatestDirectMessage_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the IncludeEntities input for this Choreo.
	*(optional, boolean) The "entities" node containing extra metadata will not be included when set to false.
	 */
	-(void)setIncludeEntities:(NSString*)IncludeEntities {
		[super setInput:@"IncludeEntities" toValue:IncludeEntities];
	}

	/*!
	 * Set the value of the MaxID input for this Choreo.
	*(optional, string) Returns results with an ID less than (older than) or equal to the specified ID.
	 */
	-(void)setMaxID:(NSString*)MaxID {
		[super setInput:@"MaxID" toValue:MaxID];
	}

	/*!
	 * Set the value of the SinceID input for this Choreo.
	*(optional, string) Returns results with an ID greater than (more recent than) the specified ID.
	 */
	-(void)setSinceID:(NSString*)SinceID {
		[super setInput:@"SinceID" toValue:SinceID];
	}

	/*!
	 * Set the value of the SkipStatus input for this Choreo.
	*(optional, boolean) When set to true, statuses will not be included in the returned user objects.
	 */
	-(void)setSkipStatus:(NSString*)SkipStatus {
		[super setInput:@"SkipStatus" toValue:SkipStatus];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetLatestDirectMessage Choreo.
 */
@implementation TMBTwitter_DirectMessages_GetLatestDirectMessage_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "ID" output from an execution of this Choreo.
	 * @return - NSString* (string) The message ID.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getID {
		return [super getOutputByName:@"ID"];
	}

	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "RecipientScreenName" output from an execution of this Choreo.
	 * @return - NSString* (string) The recipient's screen name.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRecipientScreenName {
		return [super getOutputByName:@"RecipientScreenName"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "SenderScreenName" output from an execution of this Choreo.
	 * @return - NSString* (string) The sender's screen name.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getSenderScreenName {
		return [super getOutputByName:@"SenderScreenName"];
	}

	/*!
	 * Retrieve the value of the "Text" output from an execution of this Choreo.
	 * @return - NSString* (string) The message text.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getText {
		return [super getOutputByName:@"Text"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the latest direct message sent to the authenticating user.
 */
@implementation TMBTwitter_DirectMessages_GetLatestDirectMessage

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_DirectMessages_GetLatestDirectMessage Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/DirectMessages/GetLatestDirectMessage"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_DirectMessages_GetLatestDirectMessage_ResultSet *results = [[TMBTwitter_DirectMessages_GetLatestDirectMessage_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetLatestDirectMessage Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_DirectMessages_GetLatestDirectMessage_Inputs*)newInputSet {
		return [[TMBTwitter_DirectMessages_GetLatestDirectMessage_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetMessageByID Choreo.
 */
@implementation TMBTwitter_DirectMessages_GetMessageByID_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, string) The ID of the direct message.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetMessageByID Choreo.
 */
@implementation TMBTwitter_DirectMessages_GetMessageByID_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a single direct message, specified by an id.
 */
@implementation TMBTwitter_DirectMessages_GetMessageByID

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_DirectMessages_GetMessageByID Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/DirectMessages/GetMessageByID"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_DirectMessages_GetMessageByID_ResultSet *results = [[TMBTwitter_DirectMessages_GetMessageByID_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetMessageByID Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_DirectMessages_GetMessageByID_Inputs*)newInputSet {
		return [[TMBTwitter_DirectMessages_GetMessageByID_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SendDirectMessage Choreo.
 */
@implementation TMBTwitter_DirectMessages_SendDirectMessage_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the ScreenName input for this Choreo.
	*(conditional, string) The screen name of the user who should receive the direct message. Required unless specifying the UserId.
	 */
	-(void)setScreenName:(NSString*)ScreenName {
		[super setInput:@"ScreenName" toValue:ScreenName];
	}

	/*!
	 * Set the value of the Text input for this Choreo.
	*(required, string) The text for the direct message. Max characters is 140.
	 */
	-(void)setText:(NSString*)Text {
		[super setInput:@"Text" toValue:Text];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(conditional, string) The ID of the user who should receive the direct message. Required unless specifying the ScreenName.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SendDirectMessage Choreo.
 */
@implementation TMBTwitter_DirectMessages_SendDirectMessage_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Sends a new direct message to the specified user from the authenticating user.
 */
@implementation TMBTwitter_DirectMessages_SendDirectMessage

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_DirectMessages_SendDirectMessage Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/DirectMessages/SendDirectMessage"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_DirectMessages_SendDirectMessage_ResultSet *results = [[TMBTwitter_DirectMessages_SendDirectMessage_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SendDirectMessage Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_DirectMessages_SendDirectMessage_Inputs*)newInputSet {
		return [[TMBTwitter_DirectMessages_SendDirectMessage_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CreateFavorite Choreo.
 */
@implementation TMBTwitter_Favorites_CreateFavorite_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, string) The ID of the status to favorite.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the IncludeEntities input for this Choreo.
	*(optional, boolean) The "entities" node containing extra metadata will not be included when set to false.
	 */
	-(void)setIncludeEntities:(NSString*)IncludeEntities {
		[super setInput:@"IncludeEntities" toValue:IncludeEntities];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateFavorite Choreo.
 */
@implementation TMBTwitter_Favorites_CreateFavorite_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Marks a specified status as a favorite.
 */
@implementation TMBTwitter_Favorites_CreateFavorite

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Favorites_CreateFavorite Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Favorites/CreateFavorite"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Favorites_CreateFavorite_ResultSet *results = [[TMBTwitter_Favorites_CreateFavorite_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CreateFavorite Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Favorites_CreateFavorite_Inputs*)newInputSet {
		return [[TMBTwitter_Favorites_CreateFavorite_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DestroyFavorite Choreo.
 */
@implementation TMBTwitter_Favorites_DestroyFavorite_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, string) The ID of the status to remove from your favorites.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the IncludeEntities input for this Choreo.
	*(optional, boolean) The "entities" node containing extra metadata will not be included when set to false.
	 */
	-(void)setIncludeEntities:(NSString*)IncludeEntities {
		[super setInput:@"IncludeEntities" toValue:IncludeEntities];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DestroyFavorite Choreo.
 */
@implementation TMBTwitter_Favorites_DestroyFavorite_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Removes the specified status from a favorites list.
 */
@implementation TMBTwitter_Favorites_DestroyFavorite

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Favorites_DestroyFavorite Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Favorites/DestroyFavorite"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Favorites_DestroyFavorite_ResultSet *results = [[TMBTwitter_Favorites_DestroyFavorite_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DestroyFavorite Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Favorites_DestroyFavorite_Inputs*)newInputSet {
		return [[TMBTwitter_Favorites_DestroyFavorite_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListFavorites Choreo.
 */
@implementation TMBTwitter_Favorites_ListFavorites_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Count input for this Choreo.
	*(optional, integer) Specifies the number of records to retrieve. Must be less than or equal to 200. Defaults to 20.
	 */
	-(void)setCount:(NSString*)Count {
		[super setInput:@"Count" toValue:Count];
	}

	/*!
	 * Set the value of the IncludeEntities input for this Choreo.
	*(optional, boolean) The "entities" node containing extra metadata will not be included when set to false.
	 */
	-(void)setIncludeEntities:(NSString*)IncludeEntities {
		[super setInput:@"IncludeEntities" toValue:IncludeEntities];
	}

	/*!
	 * Set the value of the MaxId input for this Choreo.
	*(optional, string) Returns results with an ID less than (older than) or equal to the specified ID.
	 */
	-(void)setMaxId:(NSString*)MaxId {
		[super setInput:@"MaxId" toValue:MaxId];
	}

	/*!
	 * Set the value of the ScreenName input for this Choreo.
	*(optional, string) Screen name of the user to return results for. If ScreenName or UserId are not provided, the authenticating user is assumed.
	 */
	-(void)setScreenName:(NSString*)ScreenName {
		[super setInput:@"ScreenName" toValue:ScreenName];
	}

	/*!
	 * Set the value of the SinceId input for this Choreo.
	*(optional, string) Returns results with an ID greater than (more recent than) the specified ID.
	 */
	-(void)setSinceId:(NSString*)SinceId {
		[super setInput:@"SinceId" toValue:SinceId];
	}

	/*!
	 * Set the value of the UserId input for this Choreo.
	*(optional, string) ID of the user to return results for. If ScreenName or UserId are not provided, the authenticating user is assumed.
	 */
	-(void)setUserId:(NSString*)UserId {
		[super setInput:@"UserId" toValue:UserId];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListFavorites Choreo.
 */
@implementation TMBTwitter_Favorites_ListFavorites_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the 20 most recent Tweets favorited by the authenticating or specified user.
 */
@implementation TMBTwitter_Favorites_ListFavorites

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Favorites_ListFavorites Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Favorites/ListFavorites"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Favorites_ListFavorites_ResultSet *results = [[TMBTwitter_Favorites_ListFavorites_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListFavorites Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Favorites_ListFavorites_Inputs*)newInputSet {
		return [[TMBTwitter_Favorites_ListFavorites_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CreateFriendship Choreo.
 */
@implementation TMBTwitter_FriendsAndFollowers_CreateFriendship_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Follow input for this Choreo.
	*(optional, boolean) A boolean flag that enables notifications for the target user when set to true.
	 */
	-(void)setFollow:(NSString*)Follow {
		[super setInput:@"Follow" toValue:Follow];
	}

	/*!
	 * Set the value of the ScreenName input for this Choreo.
	*(conditional, string) The screen name for the friend you want to create a friendship with. Required if UserId isn't specified.
	 */
	-(void)setScreenName:(NSString*)ScreenName {
		[super setInput:@"ScreenName" toValue:ScreenName];
	}

	/*!
	 * Set the value of the UserId input for this Choreo.
	*(conditional, string) The user id for the friend you want to create a friendship with. Required if ScreenName isn't specified.
	 */
	-(void)setUserId:(NSString*)UserId {
		[super setInput:@"UserId" toValue:UserId];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateFriendship Choreo.
 */
@implementation TMBTwitter_FriendsAndFollowers_CreateFriendship_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Allows you to follow another Twitter user by specifying a Twitter user id or screen name.
 */
@implementation TMBTwitter_FriendsAndFollowers_CreateFriendship

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_FriendsAndFollowers_CreateFriendship Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/FriendsAndFollowers/CreateFriendship"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_FriendsAndFollowers_CreateFriendship_ResultSet *results = [[TMBTwitter_FriendsAndFollowers_CreateFriendship_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CreateFriendship Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_FriendsAndFollowers_CreateFriendship_Inputs*)newInputSet {
		return [[TMBTwitter_FriendsAndFollowers_CreateFriendship_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteFriendship Choreo.
 */
@implementation TMBTwitter_FriendsAndFollowers_DeleteFriendship_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the ScreenName input for this Choreo.
	*(conditional, string) The screen name for the friend you want to unfollow. Required if UserId isn't specified.
	 */
	-(void)setScreenName:(NSString*)ScreenName {
		[super setInput:@"ScreenName" toValue:ScreenName];
	}

	/*!
	 * Set the value of the UserId input for this Choreo.
	*(conditional, string) The user ID for the friend you want to unfollow. Required if ScreenName isn't specified.
	 */
	-(void)setUserId:(NSString*)UserId {
		[super setInput:@"UserId" toValue:UserId];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteFriendship Choreo.
 */
@implementation TMBTwitter_FriendsAndFollowers_DeleteFriendship_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Allows you to unfollow another Twitter user when you specify a Twitter user ID or screen name.
 */
@implementation TMBTwitter_FriendsAndFollowers_DeleteFriendship

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_FriendsAndFollowers_DeleteFriendship Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/FriendsAndFollowers/DeleteFriendship"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_FriendsAndFollowers_DeleteFriendship_ResultSet *results = [[TMBTwitter_FriendsAndFollowers_DeleteFriendship_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteFriendship Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_FriendsAndFollowers_DeleteFriendship_Inputs*)newInputSet {
		return [[TMBTwitter_FriendsAndFollowers_DeleteFriendship_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the FriendshipsLookup Choreo.
 */
@implementation TMBTwitter_FriendsAndFollowers_FriendshipsLookup_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the ScreenName input for this Choreo.
	*(conditional, string) A comma separated list of screen names. Up to 100 are allowed. Required unless UserID is specified.
	 */
	-(void)setScreenName:(NSString*)ScreenName {
		[super setInput:@"ScreenName" toValue:ScreenName];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(conditional, string) A comma separated list of user IDs. Up to 100 are allowed. Required unless ScreenName is specified.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the FriendshipsLookup Choreo.
 */
@implementation TMBTwitter_FriendsAndFollowers_FriendshipsLookup_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the relationship of the authenticating user to the comma-separated list of up to 100 screen names or user IDs provided.
 */
@implementation TMBTwitter_FriendsAndFollowers_FriendshipsLookup

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_FriendsAndFollowers_FriendshipsLookup Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/FriendsAndFollowers/FriendshipsLookup"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_FriendsAndFollowers_FriendshipsLookup_ResultSet *results = [[TMBTwitter_FriendsAndFollowers_FriendshipsLookup_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the FriendshipsLookup Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_FriendsAndFollowers_FriendshipsLookup_Inputs*)newInputSet {
		return [[TMBTwitter_FriendsAndFollowers_FriendshipsLookup_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the FriendshipsShow Choreo.
 */
@implementation TMBTwitter_FriendsAndFollowers_FriendshipsShow_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the SourceScreenName input for this Choreo.
	*(conditional, string) The screen_name of the subject user. Required unless specifying the SourceUserID instead.
	 */
	-(void)setSourceScreenName:(NSString*)SourceScreenName {
		[super setInput:@"SourceScreenName" toValue:SourceScreenName];
	}

	/*!
	 * Set the value of the SourceUserID input for this Choreo.
	*(conditional, string) The ID of the subject user. Required unless specifying the SourceScreenName instead.
	 */
	-(void)setSourceUserID:(NSString*)SourceUserID {
		[super setInput:@"SourceUserID" toValue:SourceUserID];
	}

	/*!
	 * Set the value of the TargetScreenName input for this Choreo.
	*(conditional, string) The screen_name of the target user. Required unless specifying the TargetUserID instead.
	 */
	-(void)setTargetScreenName:(NSString*)TargetScreenName {
		[super setInput:@"TargetScreenName" toValue:TargetScreenName];
	}

	/*!
	 * Set the value of the TargetUserID input for this Choreo.
	*(conditional, string) The ID of the target user. Required unless specifying the TargetScreenName instead.
	 */
	-(void)setTargetUserID:(NSString*)TargetUserID {
		[super setInput:@"TargetUserID" toValue:TargetUserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the FriendshipsShow Choreo.
 */
@implementation TMBTwitter_FriendsAndFollowers_FriendshipsShow_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Returns detailed information about the relationship between two users.
 */
@implementation TMBTwitter_FriendsAndFollowers_FriendshipsShow

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_FriendsAndFollowers_FriendshipsShow Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/FriendsAndFollowers/FriendshipsShow"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_FriendsAndFollowers_FriendshipsShow_ResultSet *results = [[TMBTwitter_FriendsAndFollowers_FriendshipsShow_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the FriendshipsShow Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_FriendsAndFollowers_FriendshipsShow_Inputs*)newInputSet {
		return [[TMBTwitter_FriendsAndFollowers_FriendshipsShow_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetFollowersByID Choreo.
 */
@implementation TMBTwitter_FriendsAndFollowers_GetFollowersByID_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Cursor input for this Choreo.
	*(optional, string) Allows you to pass in the previous_cursor or next_cursor in order to page through results.
	 */
	-(void)setCursor:(NSString*)Cursor {
		[super setInput:@"Cursor" toValue:Cursor];
	}

	/*!
	 * Set the value of the ScreenName input for this Choreo.
	*(conditional, string) The screen name of the user for whom to return results for. Required if UserID isn't specified.
	 */
	-(void)setScreenName:(NSString*)ScreenName {
		[super setInput:@"ScreenName" toValue:ScreenName];
	}

	/*!
	 * Set the value of the StringifyIDs input for this Choreo.
	*(optional, boolean) A boolean flag indicating that Tweet IDs should be returned as strings.
	 */
	-(void)setStringifyIDs:(NSString*)StringifyIDs {
		[super setInput:@"StringifyIDs" toValue:StringifyIDs];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(conditional, string) The ID of the user for whom to return results for. Required if ScreenName isn't specified.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetFollowersByID Choreo.
 */
@implementation TMBTwitter_FriendsAndFollowers_GetFollowersByID_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a list of numeric IDs for every user following the specified user.
 */
@implementation TMBTwitter_FriendsAndFollowers_GetFollowersByID

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_FriendsAndFollowers_GetFollowersByID Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/FriendsAndFollowers/GetFollowersByID"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_FriendsAndFollowers_GetFollowersByID_ResultSet *results = [[TMBTwitter_FriendsAndFollowers_GetFollowersByID_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetFollowersByID Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_FriendsAndFollowers_GetFollowersByID_Inputs*)newInputSet {
		return [[TMBTwitter_FriendsAndFollowers_GetFollowersByID_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetFriendsByID Choreo.
 */
@implementation TMBTwitter_FriendsAndFollowers_GetFriendsByID_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Cursor input for this Choreo.
	*(optional, string) Allows you to pass in the previous_cursor or next_cursor in order to page through results.
	 */
	-(void)setCursor:(NSString*)Cursor {
		[super setInput:@"Cursor" toValue:Cursor];
	}

	/*!
	 * Set the value of the ScreenName input for this Choreo.
	*(conditional, string) The screen name of the user for whom to return results for. Required if UserId isn't specified.
	 */
	-(void)setScreenName:(NSString*)ScreenName {
		[super setInput:@"ScreenName" toValue:ScreenName];
	}

	/*!
	 * Set the value of the StringifyIDs input for this Choreo.
	*(optional, boolean) A boolean flag indicating that Tweet IDs should be returned as strings.
	 */
	-(void)setStringifyIDs:(NSString*)StringifyIDs {
		[super setInput:@"StringifyIDs" toValue:StringifyIDs];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(conditional, string) The ID of the user for whom to return results for. Required if ScreenName isn't specified.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetFriendsByID Choreo.
 */
@implementation TMBTwitter_FriendsAndFollowers_GetFriendsByID_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a list of numeric IDs for every user the specified user is following (also known as their "friends").
 */
@implementation TMBTwitter_FriendsAndFollowers_GetFriendsByID

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_FriendsAndFollowers_GetFriendsByID Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/FriendsAndFollowers/GetFriendsByID"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_FriendsAndFollowers_GetFriendsByID_ResultSet *results = [[TMBTwitter_FriendsAndFollowers_GetFriendsByID_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetFriendsByID Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_FriendsAndFollowers_GetFriendsByID_Inputs*)newInputSet {
		return [[TMBTwitter_FriendsAndFollowers_GetFriendsByID_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the IncomingFriendships Choreo.
 */
@implementation TMBTwitter_FriendsAndFollowers_IncomingFriendships_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Cursor input for this Choreo.
	*(optional, integer) Allows you to pass in the previous_cursor or next_cursor in order to page through results.
	 */
	-(void)setCursor:(NSString*)Cursor {
		[super setInput:@"Cursor" toValue:Cursor];
	}

	/*!
	 * Set the value of the StringifyIDs input for this Choreo.
	*(optional, boolean) A boolean flag indicating that Tweet IDs should be returned as strings.
	 */
	-(void)setStringifyIDs:(NSString*)StringifyIDs {
		[super setInput:@"StringifyIDs" toValue:StringifyIDs];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the IncomingFriendships Choreo.
 */
@implementation TMBTwitter_FriendsAndFollowers_IncomingFriendships_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a list of numeric IDs for every user who has a pending request to follow the authenticating user.
 */
@implementation TMBTwitter_FriendsAndFollowers_IncomingFriendships

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_FriendsAndFollowers_IncomingFriendships Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/FriendsAndFollowers/IncomingFriendships"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_FriendsAndFollowers_IncomingFriendships_ResultSet *results = [[TMBTwitter_FriendsAndFollowers_IncomingFriendships_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the IncomingFriendships Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_FriendsAndFollowers_IncomingFriendships_Inputs*)newInputSet {
		return [[TMBTwitter_FriendsAndFollowers_IncomingFriendships_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListFollowers Choreo.
 */
@implementation TMBTwitter_FriendsAndFollowers_ListFollowers_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Cursor input for this Choreo.
	*(optional, string) Allows you to pass in the previous_cursor or next_cursor in order to page through results.
	 */
	-(void)setCursor:(NSString*)Cursor {
		[super setInput:@"Cursor" toValue:Cursor];
	}

	/*!
	 * Set the value of the IncludeUserEntities input for this Choreo.
	*(optional, boolean) The user "entities" node containing extra metadata will not be included when set to false.
	 */
	-(void)setIncludeUserEntities:(NSString*)IncludeUserEntities {
		[super setInput:@"IncludeUserEntities" toValue:IncludeUserEntities];
	}

	/*!
	 * Set the value of the ScreenName input for this Choreo.
	*(conditional, string) The screen name of the user for whom to return results for. Required if UserID isn't specified.
	 */
	-(void)setScreenName:(NSString*)ScreenName {
		[super setInput:@"ScreenName" toValue:ScreenName];
	}

	/*!
	 * Set the value of the SkipStatus input for this Choreo.
	*(optional, boolean) When set to true, statuses will not be included in the returned user objects.
	 */
	-(void)setSkipStatus:(NSString*)SkipStatus {
		[super setInput:@"SkipStatus" toValue:SkipStatus];
	}

	/*!
	 * Set the value of the StringifyIDs input for this Choreo.
	*(optional, boolean) A boolean flag indicating that Tweet IDs should be returned as strings.
	 */
	-(void)setStringifyIDs:(NSString*)StringifyIDs {
		[super setInput:@"StringifyIDs" toValue:StringifyIDs];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(conditional, string) The ID of the user for whom to return results for. Required if ScreenName isn't specified.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListFollowers Choreo.
 */
@implementation TMBTwitter_FriendsAndFollowers_ListFollowers_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a collection of user objects for users following the specified user.
 */
@implementation TMBTwitter_FriendsAndFollowers_ListFollowers

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_FriendsAndFollowers_ListFollowers Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/FriendsAndFollowers/ListFollowers"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_FriendsAndFollowers_ListFollowers_ResultSet *results = [[TMBTwitter_FriendsAndFollowers_ListFollowers_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListFollowers Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_FriendsAndFollowers_ListFollowers_Inputs*)newInputSet {
		return [[TMBTwitter_FriendsAndFollowers_ListFollowers_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListFriends Choreo.
 */
@implementation TMBTwitter_FriendsAndFollowers_ListFriends_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Cursor input for this Choreo.
	*(optional, string) Allows you to pass in the previous_cursor or next_cursor in order to page through results.
	 */
	-(void)setCursor:(NSString*)Cursor {
		[super setInput:@"Cursor" toValue:Cursor];
	}

	/*!
	 * Set the value of the IncludeUserEntities input for this Choreo.
	*(optional, boolean) The user "entities" node containing extra metadata will not be included when set to false.
	 */
	-(void)setIncludeUserEntities:(NSString*)IncludeUserEntities {
		[super setInput:@"IncludeUserEntities" toValue:IncludeUserEntities];
	}

	/*!
	 * Set the value of the ScreenName input for this Choreo.
	*(conditional, string) The screen name of the user for whom to return results for. Required if UserID isn't specified.
	 */
	-(void)setScreenName:(NSString*)ScreenName {
		[super setInput:@"ScreenName" toValue:ScreenName];
	}

	/*!
	 * Set the value of the SkipStatus input for this Choreo.
	*(optional, boolean) When set to true, statuses will not be included in the returned user objects.
	 */
	-(void)setSkipStatus:(NSString*)SkipStatus {
		[super setInput:@"SkipStatus" toValue:SkipStatus];
	}

	/*!
	 * Set the value of the StringifyIDs input for this Choreo.
	*(optional, boolean) A boolean flag indicating that Tweet IDs should be returned as strings.
	 */
	-(void)setStringifyIDs:(NSString*)StringifyIDs {
		[super setInput:@"StringifyIDs" toValue:StringifyIDs];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(conditional, string) The ID of the user for whom to return results for. Required if ScreenName isn't specified.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListFriends Choreo.
 */
@implementation TMBTwitter_FriendsAndFollowers_ListFriends_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a collection of user objects for every user the specified user is following.
 */
@implementation TMBTwitter_FriendsAndFollowers_ListFriends

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_FriendsAndFollowers_ListFriends Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/FriendsAndFollowers/ListFriends"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_FriendsAndFollowers_ListFriends_ResultSet *results = [[TMBTwitter_FriendsAndFollowers_ListFriends_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListFriends Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_FriendsAndFollowers_ListFriends_Inputs*)newInputSet {
		return [[TMBTwitter_FriendsAndFollowers_ListFriends_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the OutgoingFriendships Choreo.
 */
@implementation TMBTwitter_FriendsAndFollowers_OutgoingFriendships_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Cursor input for this Choreo.
	*(optional, integer) Allows you to pass in the previous_cursor or next_cursor in order to page through results.
	 */
	-(void)setCursor:(NSString*)Cursor {
		[super setInput:@"Cursor" toValue:Cursor];
	}

	/*!
	 * Set the value of the StringifyIDs input for this Choreo.
	*(optional, boolean) A boolean flag indicating that Tweet IDs should be returned as strings.
	 */
	-(void)setStringifyIDs:(NSString*)StringifyIDs {
		[super setInput:@"StringifyIDs" toValue:StringifyIDs];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the OutgoingFriendships Choreo.
 */
@implementation TMBTwitter_FriendsAndFollowers_OutgoingFriendships_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a list of numeric IDs for every protected user for whom the authenticating user has a pending follow request.
 */
@implementation TMBTwitter_FriendsAndFollowers_OutgoingFriendships

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_FriendsAndFollowers_OutgoingFriendships Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/FriendsAndFollowers/OutgoingFriendships"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_FriendsAndFollowers_OutgoingFriendships_ResultSet *results = [[TMBTwitter_FriendsAndFollowers_OutgoingFriendships_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the OutgoingFriendships Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_FriendsAndFollowers_OutgoingFriendships_Inputs*)newInputSet {
		return [[TMBTwitter_FriendsAndFollowers_OutgoingFriendships_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetRateLimitStatus Choreo.
 */
@implementation TMBTwitter_Help_GetRateLimitStatus_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Resources input for this Choreo.
	*(optional, string) A comma-separated list of resources you want to know the current rate limit disposition for (e.g., statuses,friends,trends).
	 */
	-(void)setResources:(NSString*)Resources {
		[super setInput:@"Resources" toValue:Resources];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetRateLimitStatus Choreo.
 */
@implementation TMBTwitter_Help_GetRateLimitStatus_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Allows you to predict the rate limits available to your application by returning the limits for specified families of methods.
 */
@implementation TMBTwitter_Help_GetRateLimitStatus

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Help_GetRateLimitStatus Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Help/GetRateLimitStatus"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Help_GetRateLimitStatus_ResultSet *results = [[TMBTwitter_Help_GetRateLimitStatus_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetRateLimitStatus Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Help_GetRateLimitStatus_Inputs*)newInputSet {
		return [[TMBTwitter_Help_GetRateLimitStatus_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CreateList Choreo.
 */
@implementation TMBTwitter_Lists_CreateList_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Description input for this Choreo.
	*(optional, string) A description of the list.
	 */
	-(void)setDescription:(NSString*)Description {
		[super setInput:@"Description" toValue:Description];
	}

	/*!
	 * Set the value of the Mode input for this Choreo.
	*(optional, string) Indicates if this list is "public" (the default) or "private".
	 */
	-(void)setMode:(NSString*)Mode {
		[super setInput:@"Mode" toValue:Mode];
	}

	/*!
	 * Set the value of the Name input for this Choreo.
	*(required, string) The name of the new list.
	 */
	-(void)setName:(NSString*)Name {
		[super setInput:@"Name" toValue:Name];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateList Choreo.
 */
@implementation TMBTwitter_Lists_CreateList_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Creates a new list.
 */
@implementation TMBTwitter_Lists_CreateList

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Lists_CreateList Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Lists/CreateList"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Lists_CreateList_ResultSet *results = [[TMBTwitter_Lists_CreateList_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CreateList Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Lists_CreateList_Inputs*)newInputSet {
		return [[TMBTwitter_Lists_CreateList_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CreateMember Choreo.
 */
@implementation TMBTwitter_Lists_CreateMember_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the ListId input for this Choreo.
	*(conditional, string) The numerical ID of the list. Required unless Slug is provided.
	 */
	-(void)setListId:(NSString*)ListId {
		[super setInput:@"ListId" toValue:ListId];
	}

	/*!
	 * Set the value of the OwnerId input for this Choreo.
	*(optional, string) The user ID of the user who owns the list being requested by a slug.
	 */
	-(void)setOwnerId:(NSString*)OwnerId {
		[super setInput:@"OwnerId" toValue:OwnerId];
	}

	/*!
	 * Set the value of the OwnerScreenName input for this Choreo.
	*(optional, string) The screen name of the user who owns the list being requested by a slug.
	 */
	-(void)setOwnerScreenName:(NSString*)OwnerScreenName {
		[super setInput:@"OwnerScreenName" toValue:OwnerScreenName];
	}

	/*!
	 * Set the value of the ScreenName input for this Choreo.
	*(conditional, string) The screen name of the user to add to the list. Required unless providing the UserId. Multiple screen names can be provided in a comma-separated list.
	 */
	-(void)setScreenName:(NSString*)ScreenName {
		[super setInput:@"ScreenName" toValue:ScreenName];
	}

	/*!
	 * Set the value of the Slug input for this Choreo.
	*(optional, string) When identifying a list by a slug, either OwnerScreenName or OwnerId must be provided.
	 */
	-(void)setSlug:(NSString*)Slug {
		[super setInput:@"Slug" toValue:Slug];
	}

	/*!
	 * Set the value of the UserId input for this Choreo.
	*(conditional, string) The ID of the user to add to the list. Required unless providing the ScreenName. Multiple user IDs can be provided in a comma-separated list.
	 */
	-(void)setUserId:(NSString*)UserId {
		[super setInput:@"UserId" toValue:UserId];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateMember Choreo.
 */
@implementation TMBTwitter_Lists_CreateMember_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Adds one or more members to a list.
 */
@implementation TMBTwitter_Lists_CreateMember

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Lists_CreateMember Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Lists/CreateMember"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Lists_CreateMember_ResultSet *results = [[TMBTwitter_Lists_CreateMember_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CreateMember Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Lists_CreateMember_Inputs*)newInputSet {
		return [[TMBTwitter_Lists_CreateMember_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CreateSubscriber Choreo.
 */
@implementation TMBTwitter_Lists_CreateSubscriber_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the ListId input for this Choreo.
	*(conditional, string) The numerical ID of the list. Required unless Slug is provided.
	 */
	-(void)setListId:(NSString*)ListId {
		[super setInput:@"ListId" toValue:ListId];
	}

	/*!
	 * Set the value of the OwnerId input for this Choreo.
	*(optional, string) The user ID of the user who owns the list being requested by a slug.
	 */
	-(void)setOwnerId:(NSString*)OwnerId {
		[super setInput:@"OwnerId" toValue:OwnerId];
	}

	/*!
	 * Set the value of the OwnerScreenName input for this Choreo.
	*(optional, string) The screen name of the user who owns the list being requested by a slug.
	 */
	-(void)setOwnerScreenName:(NSString*)OwnerScreenName {
		[super setInput:@"OwnerScreenName" toValue:OwnerScreenName];
	}

	/*!
	 * Set the value of the Slug input for this Choreo.
	*(optional, string) When identifying a list by a slug, either OwnerScreenName or OwnerId must be provided.
	 */
	-(void)setSlug:(NSString*)Slug {
		[super setInput:@"Slug" toValue:Slug];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateSubscriber Choreo.
 */
@implementation TMBTwitter_Lists_CreateSubscriber_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Subscribes the authenticated user to the specified list.
 */
@implementation TMBTwitter_Lists_CreateSubscriber

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Lists_CreateSubscriber Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Lists/CreateSubscriber"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Lists_CreateSubscriber_ResultSet *results = [[TMBTwitter_Lists_CreateSubscriber_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CreateSubscriber Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Lists_CreateSubscriber_Inputs*)newInputSet {
		return [[TMBTwitter_Lists_CreateSubscriber_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DestroyList Choreo.
 */
@implementation TMBTwitter_Lists_DestroyList_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the ListId input for this Choreo.
	*(conditional, string) The numerical ID of the list. Required unless Slug is provided.
	 */
	-(void)setListId:(NSString*)ListId {
		[super setInput:@"ListId" toValue:ListId];
	}

	/*!
	 * Set the value of the OwnerId input for this Choreo.
	*(optional, string) The user ID of the user who owns the list being requested by a slug.
	 */
	-(void)setOwnerId:(NSString*)OwnerId {
		[super setInput:@"OwnerId" toValue:OwnerId];
	}

	/*!
	 * Set the value of the OwnerScreenName input for this Choreo.
	*(optional, string) The screen name of the user who owns the list being requested by a slug.
	 */
	-(void)setOwnerScreenName:(NSString*)OwnerScreenName {
		[super setInput:@"OwnerScreenName" toValue:OwnerScreenName];
	}

	/*!
	 * Set the value of the Slug input for this Choreo.
	*(optional, string) When identifying a list by a slug, either OwnerScreenName or OwnerId must be provided.
	 */
	-(void)setSlug:(NSString*)Slug {
		[super setInput:@"Slug" toValue:Slug];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DestroyList Choreo.
 */
@implementation TMBTwitter_Lists_DestroyList_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Deletes the specified list.
 */
@implementation TMBTwitter_Lists_DestroyList

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Lists_DestroyList Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Lists/DestroyList"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Lists_DestroyList_ResultSet *results = [[TMBTwitter_Lists_DestroyList_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DestroyList Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Lists_DestroyList_Inputs*)newInputSet {
		return [[TMBTwitter_Lists_DestroyList_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DestroyMember Choreo.
 */
@implementation TMBTwitter_Lists_DestroyMember_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the ListId input for this Choreo.
	*(conditional, string) The numerical ID of the list. Required unless Slug is provided.
	 */
	-(void)setListId:(NSString*)ListId {
		[super setInput:@"ListId" toValue:ListId];
	}

	/*!
	 * Set the value of the OwnerId input for this Choreo.
	*(optional, string) The user ID of the user who owns the list being requested by a slug.
	 */
	-(void)setOwnerId:(NSString*)OwnerId {
		[super setInput:@"OwnerId" toValue:OwnerId];
	}

	/*!
	 * Set the value of the OwnerScreenName input for this Choreo.
	*(optional, string) The screen name of the user who owns the list being requested by a slug.
	 */
	-(void)setOwnerScreenName:(NSString*)OwnerScreenName {
		[super setInput:@"OwnerScreenName" toValue:OwnerScreenName];
	}

	/*!
	 * Set the value of the ScreenName input for this Choreo.
	*(conditional, string) The screen name of the user to remove from the list. Required unless providing the UserId. Multiple screen names can be provided in a comma-separated list.
	 */
	-(void)setScreenName:(NSString*)ScreenName {
		[super setInput:@"ScreenName" toValue:ScreenName];
	}

	/*!
	 * Set the value of the Slug input for this Choreo.
	*(optional, string) When identifying a list by a slug, either OwnerScreenName or OwnerId must be provided.
	 */
	-(void)setSlug:(NSString*)Slug {
		[super setInput:@"Slug" toValue:Slug];
	}

	/*!
	 * Set the value of the UserId input for this Choreo.
	*(conditional, string) The ID of the user to remove from the list. Required unless providing the ScreenName. Multiple user IDs can be provided in a comma-separated list.
	 */
	-(void)setUserId:(NSString*)UserId {
		[super setInput:@"UserId" toValue:UserId];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DestroyMember Choreo.
 */
@implementation TMBTwitter_Lists_DestroyMember_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Removes one or more members from a list.
 */
@implementation TMBTwitter_Lists_DestroyMember

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Lists_DestroyMember Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Lists/DestroyMember"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Lists_DestroyMember_ResultSet *results = [[TMBTwitter_Lists_DestroyMember_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DestroyMember Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Lists_DestroyMember_Inputs*)newInputSet {
		return [[TMBTwitter_Lists_DestroyMember_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DestroySubscriber Choreo.
 */
@implementation TMBTwitter_Lists_DestroySubscriber_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the ListId input for this Choreo.
	*(conditional, string) The numerical ID of the list. Required unless Slug is provided.
	 */
	-(void)setListId:(NSString*)ListId {
		[super setInput:@"ListId" toValue:ListId];
	}

	/*!
	 * Set the value of the OwnerId input for this Choreo.
	*(optional, string) The user ID of the user who owns the list being requested by a slug.
	 */
	-(void)setOwnerId:(NSString*)OwnerId {
		[super setInput:@"OwnerId" toValue:OwnerId];
	}

	/*!
	 * Set the value of the OwnerScreenName input for this Choreo.
	*(optional, string) The screen name of the user who owns the list being requested by a slug.
	 */
	-(void)setOwnerScreenName:(NSString*)OwnerScreenName {
		[super setInput:@"OwnerScreenName" toValue:OwnerScreenName];
	}

	/*!
	 * Set the value of the Slug input for this Choreo.
	*(optional, string) When identifying a list by a slug, either OwnerScreenName or OwnerId must be provided.
	 */
	-(void)setSlug:(NSString*)Slug {
		[super setInput:@"Slug" toValue:Slug];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DestroySubscriber Choreo.
 */
@implementation TMBTwitter_Lists_DestroySubscriber_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Unsubscribes the authenticated user from the specified list.
 */
@implementation TMBTwitter_Lists_DestroySubscriber

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Lists_DestroySubscriber Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Lists/DestroySubscriber"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Lists_DestroySubscriber_ResultSet *results = [[TMBTwitter_Lists_DestroySubscriber_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DestroySubscriber Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Lists_DestroySubscriber_Inputs*)newInputSet {
		return [[TMBTwitter_Lists_DestroySubscriber_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetLists Choreo.
 */
@implementation TMBTwitter_Lists_GetLists_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Reverse input for this Choreo.
	*(optional, boolean) When set to true, owned lists will be returned first in the response.
	 */
	-(void)setReverse:(NSString*)Reverse {
		[super setInput:@"Reverse" toValue:Reverse];
	}

	/*!
	 * Set the value of the ScreenName input for this Choreo.
	*(optional, string) The screen name of the user for whom to return results for. If not provided, the subscriptions for the authenticating user are returned.
	 */
	-(void)setScreenName:(NSString*)ScreenName {
		[super setInput:@"ScreenName" toValue:ScreenName];
	}

	/*!
	 * Set the value of the UserId input for this Choreo.
	*(optional, string) The ID of the user for whom to return results for. If not provided, the subscriptions for the authenticating user are returned.
	 */
	-(void)setUserId:(NSString*)UserId {
		[super setInput:@"UserId" toValue:UserId];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetLists Choreo.
 */
@implementation TMBTwitter_Lists_GetLists_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves all lists the authenticating or specified user subscribes to, including lists that they own.
 */
@implementation TMBTwitter_Lists_GetLists

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Lists_GetLists Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Lists/GetLists"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Lists_GetLists_ResultSet *results = [[TMBTwitter_Lists_GetLists_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetLists Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Lists_GetLists_Inputs*)newInputSet {
		return [[TMBTwitter_Lists_GetLists_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetMembers Choreo.
 */
@implementation TMBTwitter_Lists_GetMembers_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Cursor input for this Choreo.
	*(optional, string) Allows you to pass in the previous_cursor or next_cursor in order to page through results.
	 */
	-(void)setCursor:(NSString*)Cursor {
		[super setInput:@"Cursor" toValue:Cursor];
	}

	/*!
	 * Set the value of the IncludeEntities input for this Choreo.
	*(optional, boolean) The "entities" node containing extra metadata will not be included when set to false.
	 */
	-(void)setIncludeEntities:(NSString*)IncludeEntities {
		[super setInput:@"IncludeEntities" toValue:IncludeEntities];
	}

	/*!
	 * Set the value of the ListId input for this Choreo.
	*(conditional, string) The numerical ID of the list. Required unless Slug is provided.
	 */
	-(void)setListId:(NSString*)ListId {
		[super setInput:@"ListId" toValue:ListId];
	}

	/*!
	 * Set the value of the OwnerId input for this Choreo.
	*(optional, string) The user ID of the user who owns the list being requested by a slug.
	 */
	-(void)setOwnerId:(NSString*)OwnerId {
		[super setInput:@"OwnerId" toValue:OwnerId];
	}

	/*!
	 * Set the value of the OwnerScreenName input for this Choreo.
	*(optional, string) The screen name of the user who owns the list being requested by a slug.
	 */
	-(void)setOwnerScreenName:(NSString*)OwnerScreenName {
		[super setInput:@"OwnerScreenName" toValue:OwnerScreenName];
	}

	/*!
	 * Set the value of the SkipStatus input for this Choreo.
	*(optional, boolean) When set to either true, statuses will not be included in the returned user objects.
	 */
	-(void)setSkipStatus:(NSString*)SkipStatus {
		[super setInput:@"SkipStatus" toValue:SkipStatus];
	}

	/*!
	 * Set the value of the Slug input for this Choreo.
	*(optional, string) When identifying a list by a slug, either OwnerScreenName or OwnerId must be provided.
	 */
	-(void)setSlug:(NSString*)Slug {
		[super setInput:@"Slug" toValue:Slug];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetMembers Choreo.
 */
@implementation TMBTwitter_Lists_GetMembers_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the members of a specified list.
 */
@implementation TMBTwitter_Lists_GetMembers

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Lists_GetMembers Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Lists/GetMembers"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Lists_GetMembers_ResultSet *results = [[TMBTwitter_Lists_GetMembers_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetMembers Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Lists_GetMembers_Inputs*)newInputSet {
		return [[TMBTwitter_Lists_GetMembers_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetMemberships Choreo.
 */
@implementation TMBTwitter_Lists_GetMemberships_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Cursor input for this Choreo.
	*(optional, string) Allows you to pass in the previous_cursor or next_cursor in order to page through results.
	 */
	-(void)setCursor:(NSString*)Cursor {
		[super setInput:@"Cursor" toValue:Cursor];
	}

	/*!
	 * Set the value of the FilterToOwnedLists input for this Choreo.
	*(optional, boolean) When set to true, the response includes only lists that the authenticating user owns and lists that the specified user is a member of.
	 */
	-(void)setFilterToOwnedLists:(NSString*)FilterToOwnedLists {
		[super setInput:@"FilterToOwnedLists" toValue:FilterToOwnedLists];
	}

	/*!
	 * Set the value of the ScreenName input for this Choreo.
	*(conditional, string) The screen name of the user for whom to return results for. If not provided, the memberships for the authenticating user are returned.
	 */
	-(void)setScreenName:(NSString*)ScreenName {
		[super setInput:@"ScreenName" toValue:ScreenName];
	}

	/*!
	 * Set the value of the UserId input for this Choreo.
	*(conditional, string) The ID of the user for whom to return results for. If not provided, the memberships for the authenticating user are returned.
	 */
	-(void)setUserId:(NSString*)UserId {
		[super setInput:@"UserId" toValue:UserId];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetMemberships Choreo.
 */
@implementation TMBTwitter_Lists_GetMemberships_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the lists that the specified user has been added to.
 */
@implementation TMBTwitter_Lists_GetMemberships

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Lists_GetMemberships Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Lists/GetMemberships"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Lists_GetMemberships_ResultSet *results = [[TMBTwitter_Lists_GetMemberships_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetMemberships Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Lists_GetMemberships_Inputs*)newInputSet {
		return [[TMBTwitter_Lists_GetMemberships_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetOwnedLists Choreo.
 */
@implementation TMBTwitter_Lists_GetOwnedLists_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Count input for this Choreo.
	*(optional, integer) The number of results to return per page.
	 */
	-(void)setCount:(NSString*)Count {
		[super setInput:@"Count" toValue:Count];
	}

	/*!
	 * Set the value of the Cursor input for this Choreo.
	*(optional, string) Allows you to pass in the previous_cursor or next_cursor in order to page through results.
	 */
	-(void)setCursor:(NSString*)Cursor {
		[super setInput:@"Cursor" toValue:Cursor];
	}

	/*!
	 * Set the value of the ScreenName input for this Choreo.
	*(optional, string) The screen name of the user for whom to return results for. If not provided, the authenticating user is assumed.
	 */
	-(void)setScreenName:(NSString*)ScreenName {
		[super setInput:@"ScreenName" toValue:ScreenName];
	}

	/*!
	 * Set the value of the UserId input for this Choreo.
	*(optional, string) The ID of the user for whom to return results for. If not provided, the authenticating user is assumed.
	 */
	-(void)setUserId:(NSString*)UserId {
		[super setInput:@"UserId" toValue:UserId];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetOwnedLists Choreo.
 */
@implementation TMBTwitter_Lists_GetOwnedLists_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves lists owned by the specified user.
 */
@implementation TMBTwitter_Lists_GetOwnedLists

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Lists_GetOwnedLists Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Lists/GetOwnedLists"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Lists_GetOwnedLists_ResultSet *results = [[TMBTwitter_Lists_GetOwnedLists_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetOwnedLists Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Lists_GetOwnedLists_Inputs*)newInputSet {
		return [[TMBTwitter_Lists_GetOwnedLists_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetStatuses Choreo.
 */
@implementation TMBTwitter_Lists_GetStatuses_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Count input for this Choreo.
	*(optional, integer) Specifies the number of records to retrieve. Must be less than or equal to 200. Defaults to 20.
	 */
	-(void)setCount:(NSString*)Count {
		[super setInput:@"Count" toValue:Count];
	}

	/*!
	 * Set the value of the IncludeEntitities input for this Choreo.
	*(optional, boolean) The "entities" node containing extra metadata will not be included when set to false.
	 */
	-(void)setIncludeEntitities:(NSString*)IncludeEntitities {
		[super setInput:@"IncludeEntitities" toValue:IncludeEntitities];
	}

	/*!
	 * Set the value of the IncludeRetweets input for this Choreo.
	*(optional, boolean) When set to true, the response will include retweets if they exist.
	 */
	-(void)setIncludeRetweets:(NSString*)IncludeRetweets {
		[super setInput:@"IncludeRetweets" toValue:IncludeRetweets];
	}

	/*!
	 * Set the value of the ListId input for this Choreo.
	*(conditional, string) The numerical ID of the list. Required unless Slug is provided.
	 */
	-(void)setListId:(NSString*)ListId {
		[super setInput:@"ListId" toValue:ListId];
	}

	/*!
	 * Set the value of the MaxId input for this Choreo.
	*(optional, string) Returns results with an ID less than (older than) or equal to the specified ID.
	 */
	-(void)setMaxId:(NSString*)MaxId {
		[super setInput:@"MaxId" toValue:MaxId];
	}

	/*!
	 * Set the value of the OwnerId input for this Choreo.
	*(optional, string) The user ID of the user who owns the list being requested by a slug.
	 */
	-(void)setOwnerId:(NSString*)OwnerId {
		[super setInput:@"OwnerId" toValue:OwnerId];
	}

	/*!
	 * Set the value of the OwnerScreenName input for this Choreo.
	*(optional, string) The screen name of the user who owns the list being requested by a slug.
	 */
	-(void)setOwnerScreenName:(NSString*)OwnerScreenName {
		[super setInput:@"OwnerScreenName" toValue:OwnerScreenName];
	}

	/*!
	 * Set the value of the SinceId input for this Choreo.
	*(optional, string) Returns results with an ID greater than (more recent than) the specified ID.
	 */
	-(void)setSinceId:(NSString*)SinceId {
		[super setInput:@"SinceId" toValue:SinceId];
	}

	/*!
	 * Set the value of the Slug input for this Choreo.
	*(optional, string) When identifying a list by a slug, either OwnerScreenName or OwnerId must be provided.
	 */
	-(void)setSlug:(NSString*)Slug {
		[super setInput:@"Slug" toValue:Slug];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetStatuses Choreo.
 */
@implementation TMBTwitter_Lists_GetStatuses_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a timeline of tweets posted by members of the specified list.
 */
@implementation TMBTwitter_Lists_GetStatuses

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Lists_GetStatuses Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Lists/GetStatuses"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Lists_GetStatuses_ResultSet *results = [[TMBTwitter_Lists_GetStatuses_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetStatuses Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Lists_GetStatuses_Inputs*)newInputSet {
		return [[TMBTwitter_Lists_GetStatuses_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetSubscribers Choreo.
 */
@implementation TMBTwitter_Lists_GetSubscribers_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Cursor input for this Choreo.
	*(optional, string) Allows you to pass in the previous_cursor or next_cursor in order to page through results.
	 */
	-(void)setCursor:(NSString*)Cursor {
		[super setInput:@"Cursor" toValue:Cursor];
	}

	/*!
	 * Set the value of the IncludeEntities input for this Choreo.
	*(optional, boolean) The "entities" node containing extra metadata will not be included when set to false.
	 */
	-(void)setIncludeEntities:(NSString*)IncludeEntities {
		[super setInput:@"IncludeEntities" toValue:IncludeEntities];
	}

	/*!
	 * Set the value of the ListId input for this Choreo.
	*(conditional, string) The numerical ID of the list. Required unless Slug is provided.
	 */
	-(void)setListId:(NSString*)ListId {
		[super setInput:@"ListId" toValue:ListId];
	}

	/*!
	 * Set the value of the OwnerId input for this Choreo.
	*(optional, string) The user ID of the user who owns the list being requested by a slug.
	 */
	-(void)setOwnerId:(NSString*)OwnerId {
		[super setInput:@"OwnerId" toValue:OwnerId];
	}

	/*!
	 * Set the value of the OwnerScreenName input for this Choreo.
	*(optional, string) The screen name of the user who owns the list being requested by a slug.
	 */
	-(void)setOwnerScreenName:(NSString*)OwnerScreenName {
		[super setInput:@"OwnerScreenName" toValue:OwnerScreenName];
	}

	/*!
	 * Set the value of the SkipStatus input for this Choreo.
	*(optional, boolean) When set to either true, statuses will not be included in the returned user objects.
	 */
	-(void)setSkipStatus:(NSString*)SkipStatus {
		[super setInput:@"SkipStatus" toValue:SkipStatus];
	}

	/*!
	 * Set the value of the Slug input for this Choreo.
	*(optional, string) When identifying a list by a slug, either OwnerScreenName or OwnerId must be provided.
	 */
	-(void)setSlug:(NSString*)Slug {
		[super setInput:@"Slug" toValue:Slug];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetSubscribers Choreo.
 */
@implementation TMBTwitter_Lists_GetSubscribers_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the subscribers for a specified list.
 */
@implementation TMBTwitter_Lists_GetSubscribers

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Lists_GetSubscribers Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Lists/GetSubscribers"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Lists_GetSubscribers_ResultSet *results = [[TMBTwitter_Lists_GetSubscribers_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetSubscribers Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Lists_GetSubscribers_Inputs*)newInputSet {
		return [[TMBTwitter_Lists_GetSubscribers_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetSubscriptions Choreo.
 */
@implementation TMBTwitter_Lists_GetSubscriptions_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Count input for this Choreo.
	*(optional, integer) The number of results to return per page.
	 */
	-(void)setCount:(NSString*)Count {
		[super setInput:@"Count" toValue:Count];
	}

	/*!
	 * Set the value of the Cursor input for this Choreo.
	*(optional, string) Allows you to pass in the previous_cursor or next_cursor in order to page through results.
	 */
	-(void)setCursor:(NSString*)Cursor {
		[super setInput:@"Cursor" toValue:Cursor];
	}

	/*!
	 * Set the value of the ScreenName input for this Choreo.
	*(conditional, string) The screen name of the user for whom to return results for. If not provided, the memberships for the authenticating user are returned.
	 */
	-(void)setScreenName:(NSString*)ScreenName {
		[super setInput:@"ScreenName" toValue:ScreenName];
	}

	/*!
	 * Set the value of the UserId input for this Choreo.
	*(conditional, string) The ID of the user for whom to return results for. If not provided, the memberships for the authenticating user are returned.
	 */
	-(void)setUserId:(NSString*)UserId {
		[super setInput:@"UserId" toValue:UserId];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetSubscriptions Choreo.
 */
@implementation TMBTwitter_Lists_GetSubscriptions_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a collection of the lists the specified user is subscribed to.
 */
@implementation TMBTwitter_Lists_GetSubscriptions

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Lists_GetSubscriptions Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Lists/GetSubscriptions"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Lists_GetSubscriptions_ResultSet *results = [[TMBTwitter_Lists_GetSubscriptions_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetSubscriptions Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Lists_GetSubscriptions_Inputs*)newInputSet {
		return [[TMBTwitter_Lists_GetSubscriptions_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ShowList Choreo.
 */
@implementation TMBTwitter_Lists_ShowList_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the ListId input for this Choreo.
	*(conditional, string) The numerical ID of the list. Required unless Slug is provided.
	 */
	-(void)setListId:(NSString*)ListId {
		[super setInput:@"ListId" toValue:ListId];
	}

	/*!
	 * Set the value of the OwnerId input for this Choreo.
	*(optional, string) The user ID of the user who owns the list being requested by a slug.
	 */
	-(void)setOwnerId:(NSString*)OwnerId {
		[super setInput:@"OwnerId" toValue:OwnerId];
	}

	/*!
	 * Set the value of the OwnerScreenName input for this Choreo.
	*(optional, string) The screen name of the user who owns the list being requested by a slug.
	 */
	-(void)setOwnerScreenName:(NSString*)OwnerScreenName {
		[super setInput:@"OwnerScreenName" toValue:OwnerScreenName];
	}

	/*!
	 * Set the value of the Slug input for this Choreo.
	*(optional, string) When identifying a list by a slug, either OwnerScreenName or OwnerId must be provided.
	 */
	-(void)setSlug:(NSString*)Slug {
		[super setInput:@"Slug" toValue:Slug];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ShowList Choreo.
 */
@implementation TMBTwitter_Lists_ShowList_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the specified list.
 */
@implementation TMBTwitter_Lists_ShowList

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Lists_ShowList Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Lists/ShowList"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Lists_ShowList_ResultSet *results = [[TMBTwitter_Lists_ShowList_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ShowList Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Lists_ShowList_Inputs*)newInputSet {
		return [[TMBTwitter_Lists_ShowList_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ShowMember Choreo.
 */
@implementation TMBTwitter_Lists_ShowMember_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the IncludeEntities input for this Choreo.
	*(optional, boolean) The "entities" node containing extra metadata will not be included when set to false.
	 */
	-(void)setIncludeEntities:(NSString*)IncludeEntities {
		[super setInput:@"IncludeEntities" toValue:IncludeEntities];
	}

	/*!
	 * Set the value of the ListId input for this Choreo.
	*(conditional, string) The numerical ID of the list. Required unless Slug is provided.
	 */
	-(void)setListId:(NSString*)ListId {
		[super setInput:@"ListId" toValue:ListId];
	}

	/*!
	 * Set the value of the OwnerId input for this Choreo.
	*(optional, string) The user ID of the user who owns the list being requested by a slug.
	 */
	-(void)setOwnerId:(NSString*)OwnerId {
		[super setInput:@"OwnerId" toValue:OwnerId];
	}

	/*!
	 * Set the value of the OwnerScreenName input for this Choreo.
	*(optional, string) The screen name of the user who owns the list being requested by a slug.
	 */
	-(void)setOwnerScreenName:(NSString*)OwnerScreenName {
		[super setInput:@"OwnerScreenName" toValue:OwnerScreenName];
	}

	/*!
	 * Set the value of the ScreenName input for this Choreo.
	*(conditional, string) The screen name of the user to retrieve from the list. Required unless providing the UserId.
	 */
	-(void)setScreenName:(NSString*)ScreenName {
		[super setInput:@"ScreenName" toValue:ScreenName];
	}

	/*!
	 * Set the value of the SkipStatus input for this Choreo.
	*(optional, boolean) When set to either true, statuses will not be included in the returned user objects.
	 */
	-(void)setSkipStatus:(NSString*)SkipStatus {
		[super setInput:@"SkipStatus" toValue:SkipStatus];
	}

	/*!
	 * Set the value of the Slug input for this Choreo.
	*(optional, string) When identifying a list by a slug, either OwnerScreenName or OwnerId must be provided.
	 */
	-(void)setSlug:(NSString*)Slug {
		[super setInput:@"Slug" toValue:Slug];
	}

	/*!
	 * Set the value of the UserId input for this Choreo.
	*(conditional, string) The ID of the user to retrieve from the list. Required unless providing the ScreenName.
	 */
	-(void)setUserId:(NSString*)UserId {
		[super setInput:@"UserId" toValue:UserId];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ShowMember Choreo.
 */
@implementation TMBTwitter_Lists_ShowMember_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Verifies that the specified user is a member of the specified list.
 */
@implementation TMBTwitter_Lists_ShowMember

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Lists_ShowMember Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Lists/ShowMember"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Lists_ShowMember_ResultSet *results = [[TMBTwitter_Lists_ShowMember_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ShowMember Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Lists_ShowMember_Inputs*)newInputSet {
		return [[TMBTwitter_Lists_ShowMember_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ShowSubscriber Choreo.
 */
@implementation TMBTwitter_Lists_ShowSubscriber_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the IncludeEntities input for this Choreo.
	*(optional, boolean) The "entities" node containing extra metadata will not be included when set to false.
	 */
	-(void)setIncludeEntities:(NSString*)IncludeEntities {
		[super setInput:@"IncludeEntities" toValue:IncludeEntities];
	}

	/*!
	 * Set the value of the ListId input for this Choreo.
	*(conditional, string) The numerical ID of the list. Required unless Slug is provided.
	 */
	-(void)setListId:(NSString*)ListId {
		[super setInput:@"ListId" toValue:ListId];
	}

	/*!
	 * Set the value of the OwnerId input for this Choreo.
	*(optional, string) The user ID of the user who owns the list being requested by a slug.
	 */
	-(void)setOwnerId:(NSString*)OwnerId {
		[super setInput:@"OwnerId" toValue:OwnerId];
	}

	/*!
	 * Set the value of the OwnerScreenName input for this Choreo.
	*(optional, string) The screen name of the user who owns the list being requested by a slug.
	 */
	-(void)setOwnerScreenName:(NSString*)OwnerScreenName {
		[super setInput:@"OwnerScreenName" toValue:OwnerScreenName];
	}

	/*!
	 * Set the value of the ScreenName input for this Choreo.
	*(conditional, string) The screen name of the user to retrieve from the list. Required unless providing the UserId.
	 */
	-(void)setScreenName:(NSString*)ScreenName {
		[super setInput:@"ScreenName" toValue:ScreenName];
	}

	/*!
	 * Set the value of the SkipStatus input for this Choreo.
	*(optional, boolean) When set to either true, statuses will not be included in the returned user objects.
	 */
	-(void)setSkipStatus:(NSString*)SkipStatus {
		[super setInput:@"SkipStatus" toValue:SkipStatus];
	}

	/*!
	 * Set the value of the Slug input for this Choreo.
	*(optional, string) When identifying a list by a slug, either OwnerScreenName or OwnerId must be provided.
	 */
	-(void)setSlug:(NSString*)Slug {
		[super setInput:@"Slug" toValue:Slug];
	}

	/*!
	 * Set the value of the UserId input for this Choreo.
	*(conditional, string) The ID of the user to retrieve from the list. Required unless providing the ScreenName.
	 */
	-(void)setUserId:(NSString*)UserId {
		[super setInput:@"UserId" toValue:UserId];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ShowSubscriber Choreo.
 */
@implementation TMBTwitter_Lists_ShowSubscriber_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Verifies that the specified user is a subscriber of the specified list.
 */
@implementation TMBTwitter_Lists_ShowSubscriber

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Lists_ShowSubscriber Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Lists/ShowSubscriber"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Lists_ShowSubscriber_ResultSet *results = [[TMBTwitter_Lists_ShowSubscriber_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ShowSubscriber Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Lists_ShowSubscriber_Inputs*)newInputSet {
		return [[TMBTwitter_Lists_ShowSubscriber_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UpdateList Choreo.
 */
@implementation TMBTwitter_Lists_UpdateList_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Description input for this Choreo.
	*(optional, string) A description of the list.
	 */
	-(void)setDescription:(NSString*)Description {
		[super setInput:@"Description" toValue:Description];
	}

	/*!
	 * Set the value of the ListId input for this Choreo.
	*(conditional, string) The numerical ID of the list. Required unless Slug is provided.
	 */
	-(void)setListId:(NSString*)ListId {
		[super setInput:@"ListId" toValue:ListId];
	}

	/*!
	 * Set the value of the Mode input for this Choreo.
	*(optional, string) Indicates if this list is "public" (the default) or "private".
	 */
	-(void)setMode:(NSString*)Mode {
		[super setInput:@"Mode" toValue:Mode];
	}

	/*!
	 * Set the value of the Name input for this Choreo.
	*(optional, string) The name of the new list.
	 */
	-(void)setName:(NSString*)Name {
		[super setInput:@"Name" toValue:Name];
	}

	/*!
	 * Set the value of the OwnerId input for this Choreo.
	*(optional, string) The user ID of the user who owns the list being requested by a slug.
	 */
	-(void)setOwnerId:(NSString*)OwnerId {
		[super setInput:@"OwnerId" toValue:OwnerId];
	}

	/*!
	 * Set the value of the OwnerScreenName input for this Choreo.
	*(optional, string) The screen name of the user who owns the list being requested by a slug.
	 */
	-(void)setOwnerScreenName:(NSString*)OwnerScreenName {
		[super setInput:@"OwnerScreenName" toValue:OwnerScreenName];
	}

	/*!
	 * Set the value of the Slug input for this Choreo.
	*(optional, string) When identifying a list by a slug, either OwnerScreenName or OwnerId must be provided.
	 */
	-(void)setSlug:(NSString*)Slug {
		[super setInput:@"Slug" toValue:Slug];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UpdateList Choreo.
 */
@implementation TMBTwitter_Lists_UpdateList_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Updates an existing list.
 */
@implementation TMBTwitter_Lists_UpdateList

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Lists_UpdateList Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Lists/UpdateList"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Lists_UpdateList_ResultSet *results = [[TMBTwitter_Lists_UpdateList_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UpdateList Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Lists_UpdateList_Inputs*)newInputSet {
		return [[TMBTwitter_Lists_UpdateList_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the FinalizeOAuth Choreo.
 */
@implementation TMBTwitter_OAuth_FinalizeOAuth_Inputs

	/*!
	 * Set the value of the AccountName input for this Choreo.
	*(optional, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setAccountName:(NSString*)AccountName {
		[super setInput:@"AccountName" toValue:AccountName];
	}

	/*!
	 * Set the value of the AppKeyName input for this Choreo.
	*(optional, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setAppKeyName:(NSString*)AppKeyName {
		[super setInput:@"AppKeyName" toValue:AppKeyName];
	}

	/*!
	 * Set the value of the AppKeyValue input for this Choreo.
	*(optional, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setAppKeyValue:(NSString*)AppKeyValue {
		[super setInput:@"AppKeyValue" toValue:AppKeyValue];
	}

	/*!
	 * Set the value of the CallbackID input for this Choreo.
	*(required, string) The callback token returned by the InitializeOAuth Choreo. Used to retrieve the callback data after the user authorizes.
	 */
	-(void)setCallbackID:(NSString*)CallbackID {
		[super setInput:@"CallbackID" toValue:CallbackID];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the OAuthTokenSecret input for this Choreo.
	*(required, string) The oauth_token_secret retrieved during the OAuth process. This is returned by the InitializeOAuth Choreo.
	 */
	-(void)setOAuthTokenSecret:(NSString*)OAuthTokenSecret {
		[super setInput:@"OAuthTokenSecret" toValue:OAuthTokenSecret];
	}

	/*!
	 * Set the value of the SuppressErrors input for this Choreo.
	*(optional, boolean) When set to true, errors received during the OAuth redirect process will be suppressed and returned in the ErrorMessage output.
	 */
	-(void)setSuppressErrors:(NSString*)SuppressErrors {
		[super setInput:@"SuppressErrors" toValue:SuppressErrors];
	}

	/*!
	 * Set the value of the Timeout input for this Choreo.
	*(optional, integer) The amount of time (in seconds) to poll your Temboo callback URL to see if your app's user has allowed or denied the request for access. Defaults to 20. Max is 60.
	 */
	-(void)setTimeout:(NSString*)Timeout {
		[super setInput:@"Timeout" toValue:Timeout];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the FinalizeOAuth Choreo.
 */
@implementation TMBTwitter_OAuth_FinalizeOAuth_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "AccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) The Access Token retrieved during the OAuth process.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getAccessToken {
		return [super getOutputByName:@"AccessToken"];
	}

	/*!
	 * Retrieve the value of the "AccessTokenSecret" output from an execution of this Choreo.
	 * @return - NSString* (string) The Access Token Secret retrieved during the OAuth process.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getAccessTokenSecret {
		return [super getOutputByName:@"AccessTokenSecret"];
	}

	/*!
	 * Retrieve the value of the "ErrorMessage" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains an error message if an error occurs during the OAuth redirect process and if SuppressErrors is set to true.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getErrorMessage {
		return [super getOutputByName:@"ErrorMessage"];
	}

	/*!
	 * Retrieve the value of the "ScreenName" output from an execution of this Choreo.
	 * @return - NSString* (string) The Twitter screen name associated with the access token that is being retrieved.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getScreenName {
		return [super getOutputByName:@"ScreenName"];
	}

	/*!
	 * Retrieve the value of the "UserID" output from an execution of this Choreo.
	 * @return - NSString* (integer) The Twitter user id associated with the AccessToken that is being retrieved.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getUserID {
		return [super getOutputByName:@"UserID"];
	}
	
@end

/*!
 * Completes the OAuth process by retrieving a Twitter access token and access token secret for a user, after they have visited the authorization URL returned by the InitializeOAuth Choreo and clicked "allow."
 */
@implementation TMBTwitter_OAuth_FinalizeOAuth

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_OAuth_FinalizeOAuth Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/OAuth/FinalizeOAuth"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_OAuth_FinalizeOAuth_ResultSet *results = [[TMBTwitter_OAuth_FinalizeOAuth_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the FinalizeOAuth Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_OAuth_FinalizeOAuth_Inputs*)newInputSet {
		return [[TMBTwitter_OAuth_FinalizeOAuth_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the InitializeOAuth Choreo.
 */
@implementation TMBTwitter_OAuth_InitializeOAuth_Inputs

	/*!
	 * Set the value of the AccountName input for this Choreo.
	*(optional, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setAccountName:(NSString*)AccountName {
		[super setInput:@"AccountName" toValue:AccountName];
	}

	/*!
	 * Set the value of the AppKeyName input for this Choreo.
	*(optional, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setAppKeyName:(NSString*)AppKeyName {
		[super setInput:@"AppKeyName" toValue:AppKeyName];
	}

	/*!
	 * Set the value of the AppKeyValue input for this Choreo.
	*(optional, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setAppKeyValue:(NSString*)AppKeyValue {
		[super setInput:@"AppKeyValue" toValue:AppKeyValue];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the ForwardingURL input for this Choreo.
	*(optional, string) The URL that Temboo will redirect your users to after they grant access to your application. This should include the "https://" or "http://" prefix and be a fully qualified URL.
	 */
	-(void)setForwardingURL:(NSString*)ForwardingURL {
		[super setInput:@"ForwardingURL" toValue:ForwardingURL];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the InitializeOAuth Choreo.
 */
@implementation TMBTwitter_OAuth_InitializeOAuth_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "AuthorizationURL" output from an execution of this Choreo.
	 * @return - NSString* (string) The authorization URL that the application's user needs to go to in order to grant access to your application.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getAuthorizationURL {
		return [super getOutputByName:@"AuthorizationURL"];
	}

	/*!
	 * Retrieve the value of the "CallbackID" output from an execution of this Choreo.
	 * @return - NSString* (string) An ID used to retrieve the callback data that Temboo stores once your application's user authorizes.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getCallbackID {
		return [super getOutputByName:@"CallbackID"];
	}

	/*!
	 * Retrieve the value of the "OAuthTokenSecret" output from an execution of this Choreo.
	 * @return - NSString* (string) The temporary OAuth Token Secret that can be exchanged for permanent tokens using the FinalizeOAuth Choreo.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getOAuthTokenSecret {
		return [super getOutputByName:@"OAuthTokenSecret"];
	}
	
@end

/*!
 * Generates an authorization URL that an application can use to complete the first step in the OAuth process.
 */
@implementation TMBTwitter_OAuth_InitializeOAuth

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_OAuth_InitializeOAuth Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/OAuth/InitializeOAuth"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_OAuth_InitializeOAuth_ResultSet *results = [[TMBTwitter_OAuth_InitializeOAuth_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the InitializeOAuth Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_OAuth_InitializeOAuth_Inputs*)newInputSet {
		return [[TMBTwitter_OAuth_InitializeOAuth_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GeoSearch Choreo.
 */
@implementation TMBTwitter_PlacesAndGeo_GeoSearch_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the Accuracy input for this Choreo.
	*(optional, string) A hint on the "region" in which to search. If a number, then this is a radius in meters. You can also specify feet by using the ft suffix (i.e. 5ft).
	 */
	-(void)setAccuracy:(NSString*)Accuracy {
		[super setInput:@"Accuracy" toValue:Accuracy];
	}

	/*!
	 * Set the value of the Callback input for this Choreo.
	*(optional, string) If supplied, the response will use the JSONP format with a callback of the given name.
	 */
	-(void)setCallback:(NSString*)Callback {
		[super setInput:@"Callback" toValue:Callback];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the ContainedWithin input for this Choreo.
	*(optional, string) This is the place_id which you would like to restrict the search results to. This id can be retrieved with the GetPlaceInformation Choreo.
	 */
	-(void)setContainedWithin:(NSString*)ContainedWithin {
		[super setInput:@"ContainedWithin" toValue:ContainedWithin];
	}

	/*!
	 * Set the value of the Granularity input for this Choreo.
	*(optional, string) This is the minimal granularity of place types to return and must be one of: poi, neighborhood, city, admin or country. Defaults to neighborhood.
	 */
	-(void)setGranularity:(NSString*)Granularity {
		[super setInput:@"Granularity" toValue:Granularity];
	}

	/*!
	 * Set the value of the IP input for this Choreo.
	*(conditional, string) An IP address. Used when attempting to fix geolocation based off of the user's IP address. You must provide Latitude and Longitude, IP, or Query.
	 */
	-(void)setIP:(NSString*)IP {
		[super setInput:@"IP" toValue:IP];
	}

	/*!
	 * Set the value of the Latitude input for this Choreo.
	*(conditional, decimal) The latitude to search around. You must provide Latitude and Longitude, IP, or Query.
	 */
	-(void)setLatitude:(NSString*)Latitude {
		[super setInput:@"Latitude" toValue:Latitude];
	}

	/*!
	 * Set the value of the Longitude input for this Choreo.
	*(conditional, decimal) The longitude to search around. You must provide Latitude and Longitude, IP, or Query.
	 */
	-(void)setLongitude:(NSString*)Longitude {
		[super setInput:@"Longitude" toValue:Longitude];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of results to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the Query input for this Choreo.
	*(conditional, string) Free-form text to match against while executing a geo-based query. You must provide Latitude and Longitude, IP, or Query.
	 */
	-(void)setQuery:(NSString*)Query {
		[super setInput:@"Query" toValue:Query];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GeoSearch Choreo.
 */
@implementation TMBTwitter_PlacesAndGeo_GeoSearch_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Searches for places that can be attached to a status update using a latitude and a longitude pair, an IP address, or a name.
 */
@implementation TMBTwitter_PlacesAndGeo_GeoSearch

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_PlacesAndGeo_GeoSearch Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/PlacesAndGeo/GeoSearch"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_PlacesAndGeo_GeoSearch_ResultSet *results = [[TMBTwitter_PlacesAndGeo_GeoSearch_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GeoSearch Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_PlacesAndGeo_GeoSearch_Inputs*)newInputSet {
		return [[TMBTwitter_PlacesAndGeo_GeoSearch_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetPlaceInformation Choreo.
 */
@implementation TMBTwitter_PlacesAndGeo_GetPlaceInformation_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the PlaceId input for this Choreo.
	*(required, string) The id for a place in the world. These IDs can be retrieved from the ReverseGeocode Choreo.
	 */
	-(void)setPlaceId:(NSString*)PlaceId {
		[super setInput:@"PlaceId" toValue:PlaceId];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetPlaceInformation Choreo.
 */
@implementation TMBTwitter_PlacesAndGeo_GetPlaceInformation_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Searches for places that can be attached to a statuses/update using a latitude and a longitude pair, an IP address, or a name.
 */
@implementation TMBTwitter_PlacesAndGeo_GetPlaceInformation

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_PlacesAndGeo_GetPlaceInformation Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/PlacesAndGeo/GetPlaceInformation"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_PlacesAndGeo_GetPlaceInformation_ResultSet *results = [[TMBTwitter_PlacesAndGeo_GetPlaceInformation_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetPlaceInformation Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_PlacesAndGeo_GetPlaceInformation_Inputs*)newInputSet {
		return [[TMBTwitter_PlacesAndGeo_GetPlaceInformation_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ReverseGeocode Choreo.
 */
@implementation TMBTwitter_PlacesAndGeo_ReverseGeocode_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the Accuracy input for this Choreo.
	*(optional, string) A hint on the "region" in which to search. If a number, then this is a radius in meters. You can also specify feet by using the ft suffix (i.e. 5ft).
	 */
	-(void)setAccuracy:(NSString*)Accuracy {
		[super setInput:@"Accuracy" toValue:Accuracy];
	}

	/*!
	 * Set the value of the Callback input for this Choreo.
	*(optional, string) If supplied, the response will use the JSONP format with a callback of the given name.
	 */
	-(void)setCallback:(NSString*)Callback {
		[super setInput:@"Callback" toValue:Callback];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Granularity input for this Choreo.
	*(optional, string) This is the minimal granularity of place types to return and must be one of: poi, neighborhood, city, admin or country. Defaults to neighborhood.
	 */
	-(void)setGranularity:(NSString*)Granularity {
		[super setInput:@"Granularity" toValue:Granularity];
	}

	/*!
	 * Set the value of the Latitude input for this Choreo.
	*(required, decimal) The latitude to search around.
	 */
	-(void)setLatitude:(NSString*)Latitude {
		[super setInput:@"Latitude" toValue:Latitude];
	}

	/*!
	 * Set the value of the Longitude input for this Choreo.
	*(required, decimal) The longitude to search around.
	 */
	-(void)setLongitude:(NSString*)Longitude {
		[super setInput:@"Longitude" toValue:Longitude];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of results to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ReverseGeocode Choreo.
 */
@implementation TMBTwitter_PlacesAndGeo_ReverseGeocode_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Given a latitude and a longitude coordinates, returns up to 20 places that can be used as a Place ID when updating a status.
 */
@implementation TMBTwitter_PlacesAndGeo_ReverseGeocode

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_PlacesAndGeo_ReverseGeocode Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/PlacesAndGeo/ReverseGeocode"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_PlacesAndGeo_ReverseGeocode_ResultSet *results = [[TMBTwitter_PlacesAndGeo_ReverseGeocode_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ReverseGeocode Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_PlacesAndGeo_ReverseGeocode_Inputs*)newInputSet {
		return [[TMBTwitter_PlacesAndGeo_ReverseGeocode_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SimilarPlaces Choreo.
 */
@implementation TMBTwitter_PlacesAndGeo_SimilarPlaces_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the Callback input for this Choreo.
	*(optional, string) If supplied, the response will use the JSONP format with a callback of the given name.
	 */
	-(void)setCallback:(NSString*)Callback {
		[super setInput:@"Callback" toValue:Callback];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the ContainedWithin input for this Choreo.
	*(optional, string) This is the place_id which you would like to restrict the search results to. This id can be retrieved with the GetPlaceInformation Choreo.
	 */
	-(void)setContainedWithin:(NSString*)ContainedWithin {
		[super setInput:@"ContainedWithin" toValue:ContainedWithin];
	}

	/*!
	 * Set the value of the Latitude input for this Choreo.
	*(required, decimal) The latitude to search around.
	 */
	-(void)setLatitude:(NSString*)Latitude {
		[super setInput:@"Latitude" toValue:Latitude];
	}

	/*!
	 * Set the value of the Longitude input for this Choreo.
	*(required, decimal) The longitude to search around.
	 */
	-(void)setLongitude:(NSString*)Longitude {
		[super setInput:@"Longitude" toValue:Longitude];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of results to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the Name input for this Choreo.
	*(required, string) The name of the place.
	 */
	-(void)setName:(NSString*)Name {
		[super setInput:@"Name" toValue:Name];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SimilarPlaces Choreo.
 */
@implementation TMBTwitter_PlacesAndGeo_SimilarPlaces_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Locates places near the given coordinates which have a similar name.
 */
@implementation TMBTwitter_PlacesAndGeo_SimilarPlaces

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_PlacesAndGeo_SimilarPlaces Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/PlacesAndGeo/SimilarPlaces"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_PlacesAndGeo_SimilarPlaces_ResultSet *results = [[TMBTwitter_PlacesAndGeo_SimilarPlaces_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SimilarPlaces Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_PlacesAndGeo_SimilarPlaces_Inputs*)newInputSet {
		return [[TMBTwitter_PlacesAndGeo_SimilarPlaces_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the LatestTweet Choreo.
 */
@implementation TMBTwitter_Search_LatestTweet_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Geocode input for this Choreo.
	*(optional, string) Returns tweets by users located within a given radius of the given latitude/longitude. Formatted like: 37.781157,-122.398720,1mi.
	 */
	-(void)setGeocode:(NSString*)Geocode {
		[super setInput:@"Geocode" toValue:Geocode];
	}

	/*!
	 * Set the value of the IncludeEntities input for this Choreo.
	*(optional, boolean) The "entities" node containing extra metadata will not be included when set to false.
	 */
	-(void)setIncludeEntities:(NSString*)IncludeEntities {
		[super setInput:@"IncludeEntities" toValue:IncludeEntities];
	}

	/*!
	 * Set the value of the Language input for this Choreo.
	*(optional, string) Restricts tweets to the given language, given by an ISO 639-1 code.
	 */
	-(void)setLanguage:(NSString*)Language {
		[super setInput:@"Language" toValue:Language];
	}

	/*!
	 * Set the value of the MaxId input for this Choreo.
	*(optional, string) Returns results with an ID less than (older than) or equal to the specified ID.
	 */
	-(void)setMaxId:(NSString*)MaxId {
		[super setInput:@"MaxId" toValue:MaxId];
	}

	/*!
	 * Set the value of the Query input for this Choreo.
	*(required, string) A search query of up to 1,000 characters.
	 */
	-(void)setQuery:(NSString*)Query {
		[super setInput:@"Query" toValue:Query];
	}

	/*!
	 * Set the value of the ResultType input for this Choreo.
	*(optional, string) Specifies what type of search results you want to receive. The default is "mixed." Valid values are: mixed, recent, and popular.
	 */
	-(void)setResultType:(NSString*)ResultType {
		[super setInput:@"ResultType" toValue:ResultType];
	}

	/*!
	 * Set the value of the SinceId input for this Choreo.
	*(optional, string) Returns results with an ID greater than (more recent than) the specified ID.
	 */
	-(void)setSinceId:(NSString*)SinceId {
		[super setInput:@"SinceId" toValue:SinceId];
	}

	/*!
	 * Set the value of the Until input for this Choreo.
	*(optional, date) Returns tweets generated before the given date. Date should be formatted as YYYY-MM-DD.
	 */
	-(void)setUntil:(NSString*)Until {
		[super setInput:@"Until" toValue:Until];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the LatestTweet Choreo.
 */
@implementation TMBTwitter_Search_LatestTweet_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "ID" output from an execution of this Choreo.
	 * @return - NSString* (string) The Tweet ID.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getID {
		return [super getOutputByName:@"ID"];
	}

	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "ScreenName" output from an execution of this Choreo.
	 * @return - NSString* (string) The screen name of the user who posted this Tweet.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getScreenName {
		return [super getOutputByName:@"ScreenName"];
	}

	/*!
	 * Retrieve the value of the "Text" output from an execution of this Choreo.
	 * @return - NSString* (string) The Tweet text.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getText {
		return [super getOutputByName:@"Text"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the latest Tweet matching a specified query.
 */
@implementation TMBTwitter_Search_LatestTweet

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Search_LatestTweet Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Search/LatestTweet"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Search_LatestTweet_ResultSet *results = [[TMBTwitter_Search_LatestTweet_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the LatestTweet Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Search_LatestTweet_Inputs*)newInputSet {
		return [[TMBTwitter_Search_LatestTweet_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Tweets Choreo.
 */
@implementation TMBTwitter_Search_Tweets_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Count input for this Choreo.
	*(optional, integer) Specifies the number of records to retrieve. Must be less than or equal to 200. Defaults to 20.
	 */
	-(void)setCount:(NSString*)Count {
		[super setInput:@"Count" toValue:Count];
	}

	/*!
	 * Set the value of the Geocode input for this Choreo.
	*(optional, string) Returns tweets by users located within a given radius of the given latitude/longitude. Formatted like: 37.781157,-122.398720,1mi.
	 */
	-(void)setGeocode:(NSString*)Geocode {
		[super setInput:@"Geocode" toValue:Geocode];
	}

	/*!
	 * Set the value of the IncludeEntities input for this Choreo.
	*(optional, boolean) The "entities" node containing extra metadata will not be included when set to false.
	 */
	-(void)setIncludeEntities:(NSString*)IncludeEntities {
		[super setInput:@"IncludeEntities" toValue:IncludeEntities];
	}

	/*!
	 * Set the value of the Language input for this Choreo.
	*(optional, string) Restricts tweets to the given language, given by an ISO 639-1 code.
	 */
	-(void)setLanguage:(NSString*)Language {
		[super setInput:@"Language" toValue:Language];
	}

	/*!
	 * Set the value of the MaxId input for this Choreo.
	*(optional, string) Returns results with an ID less than (older than) or equal to the specified ID.
	 */
	-(void)setMaxId:(NSString*)MaxId {
		[super setInput:@"MaxId" toValue:MaxId];
	}

	/*!
	 * Set the value of the Query input for this Choreo.
	*(required, string) A search query of up to 1,000 characters.
	 */
	-(void)setQuery:(NSString*)Query {
		[super setInput:@"Query" toValue:Query];
	}

	/*!
	 * Set the value of the ResultType input for this Choreo.
	*(optional, string) Specifies what type of search results you want to receive. The default is "mixed." Valid values are: mixed, recent, and popular.
	 */
	-(void)setResultType:(NSString*)ResultType {
		[super setInput:@"ResultType" toValue:ResultType];
	}

	/*!
	 * Set the value of the SinceId input for this Choreo.
	*(optional, string) Returns results with an ID greater than (more recent than) the specified ID.
	 */
	-(void)setSinceId:(NSString*)SinceId {
		[super setInput:@"SinceId" toValue:SinceId];
	}

	/*!
	 * Set the value of the Until input for this Choreo.
	*(optional, date) Returns tweets generated before the given date. Date should be formatted as YYYY-MM-DD.
	 */
	-(void)setUntil:(NSString*)Until {
		[super setInput:@"Until" toValue:Until];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Tweets Choreo.
 */
@implementation TMBTwitter_Search_Tweets_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a collection of relevant Tweets matching a specified query.
 */
@implementation TMBTwitter_Search_Tweets

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Search_Tweets Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Search/Tweets"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Search_Tweets_ResultSet *results = [[TMBTwitter_Search_Tweets_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Tweets Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Search_Tweets_Inputs*)newInputSet {
		return [[TMBTwitter_Search_Tweets_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetSuggestedUsers Choreo.
 */
@implementation TMBTwitter_SuggestedUsers_GetSuggestedUsers_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Language input for this Choreo.
	*(optional, string) Restricts the suggested categories to the requested language. The language must be specified by the appropriate two letter ISO 639-1 code (e.g., en).
	 */
	-(void)setLanguage:(NSString*)Language {
		[super setInput:@"Language" toValue:Language];
	}

	/*!
	 * Set the value of the Members input for this Choreo.
	*(optional, boolean) When set to true, makes a request to users/suggestions/:slug/members and retrieves the most recent statuses for users that are not protected.
	 */
	-(void)setMembers:(NSString*)Members {
		[super setInput:@"Members" toValue:Members];
	}

	/*!
	 * Set the value of the Slug input for this Choreo.
	*(required, string) The short name of  the category (e.g., news, technology, government). These are returned in the response of the GetSuggestedCategories Choreo.
	 */
	-(void)setSlug:(NSString*)Slug {
		[super setInput:@"Slug" toValue:Slug];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetSuggestedUsers Choreo.
 */
@implementation TMBTwitter_SuggestedUsers_GetSuggestedUsers_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves users in a given category of the Twitter suggested user list.
 */
@implementation TMBTwitter_SuggestedUsers_GetSuggestedUsers

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_SuggestedUsers_GetSuggestedUsers Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/SuggestedUsers/GetSuggestedUsers"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_SuggestedUsers_GetSuggestedUsers_ResultSet *results = [[TMBTwitter_SuggestedUsers_GetSuggestedUsers_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetSuggestedUsers Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_SuggestedUsers_GetSuggestedUsers_Inputs*)newInputSet {
		return [[TMBTwitter_SuggestedUsers_GetSuggestedUsers_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetUserCategories Choreo.
 */
@implementation TMBTwitter_SuggestedUsers_GetUserCategories_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Language input for this Choreo.
	*(optional, string) Restricts the suggested categories to the requested language. The language must be specified by the appropriate two letter ISO 639-1 code (e.g., en). 
	 */
	-(void)setLanguage:(NSString*)Language {
		[super setInput:@"Language" toValue:Language];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetUserCategories Choreo.
 */
@implementation TMBTwitter_SuggestedUsers_GetUserCategories_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the list of suggested user categories.
 */
@implementation TMBTwitter_SuggestedUsers_GetUserCategories

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_SuggestedUsers_GetUserCategories Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/SuggestedUsers/GetUserCategories"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_SuggestedUsers_GetUserCategories_ResultSet *results = [[TMBTwitter_SuggestedUsers_GetUserCategories_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetUserCategories Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_SuggestedUsers_GetUserCategories_Inputs*)newInputSet {
		return [[TMBTwitter_SuggestedUsers_GetUserCategories_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the HomeTimeline Choreo.
 */
@implementation TMBTwitter_Timelines_HomeTimeline_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the ContributorDetails input for this Choreo.
	*(optional, boolean) Set to true to include the screen_name of the contributor. By default only the user_id of the contributor is included.
	 */
	-(void)setContributorDetails:(NSString*)ContributorDetails {
		[super setInput:@"ContributorDetails" toValue:ContributorDetails];
	}

	/*!
	 * Set the value of the Count input for this Choreo.
	*(optional, integer) Specifies the number of records to retrieve. Must be less than or equal to 200. Defaults to 20.
	 */
	-(void)setCount:(NSString*)Count {
		[super setInput:@"Count" toValue:Count];
	}

	/*!
	 * Set the value of the ExcludeReplies input for this Choreo.
	*(optional, boolean) Set to true to prevent replies from appearing in the returned timeline.
	 */
	-(void)setExcludeReplies:(NSString*)ExcludeReplies {
		[super setInput:@"ExcludeReplies" toValue:ExcludeReplies];
	}

	/*!
	 * Set the value of the IncludeEntities input for this Choreo.
	*(optional, boolean) The "entities" node containing extra metadata will not be included when set to false.
	 */
	-(void)setIncludeEntities:(NSString*)IncludeEntities {
		[super setInput:@"IncludeEntities" toValue:IncludeEntities];
	}

	/*!
	 * Set the value of the MaxId input for this Choreo.
	*(optional, string) Returns results with an ID less than (older than) or equal to the specified ID.
	 */
	-(void)setMaxId:(NSString*)MaxId {
		[super setInput:@"MaxId" toValue:MaxId];
	}

	/*!
	 * Set the value of the SinceId input for this Choreo.
	*(optional, string) Returns results with an ID greater than (more recent than) the specified ID.
	 */
	-(void)setSinceId:(NSString*)SinceId {
		[super setInput:@"SinceId" toValue:SinceId];
	}

	/*!
	 * Set the value of the TrimUser input for this Choreo.
	*(optional, boolean) When set to true, each tweet returned in a timeline will include a user object including only the status authors numerical ID. Defaults to false.
	 */
	-(void)setTrimUser:(NSString*)TrimUser {
		[super setInput:@"TrimUser" toValue:TrimUser];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the HomeTimeline Choreo.
 */
@implementation TMBTwitter_Timelines_HomeTimeline_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a collection of the most recent Tweets and retweets posted by the authenticating user and the users they follow.
 */
@implementation TMBTwitter_Timelines_HomeTimeline

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Timelines_HomeTimeline Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Timelines/HomeTimeline"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Timelines_HomeTimeline_ResultSet *results = [[TMBTwitter_Timelines_HomeTimeline_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the HomeTimeline Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Timelines_HomeTimeline_Inputs*)newInputSet {
		return [[TMBTwitter_Timelines_HomeTimeline_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the HomeTimelineLatestTweet Choreo.
 */
@implementation TMBTwitter_Timelines_HomeTimelineLatestTweet_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the ContributorDetails input for this Choreo.
	*(optional, boolean) Set to true to include the screen_name of the contributor. By default only the user_id of the contributor is included.
	 */
	-(void)setContributorDetails:(NSString*)ContributorDetails {
		[super setInput:@"ContributorDetails" toValue:ContributorDetails];
	}

	/*!
	 * Set the value of the ExcludeReplies input for this Choreo.
	*(optional, boolean) Set to true to prevent replies from appearing in the returned timeline.
	 */
	-(void)setExcludeReplies:(NSString*)ExcludeReplies {
		[super setInput:@"ExcludeReplies" toValue:ExcludeReplies];
	}

	/*!
	 * Set the value of the IncludeEntities input for this Choreo.
	*(optional, boolean) The "entities" node containing extra metadata will not be included when set to false.
	 */
	-(void)setIncludeEntities:(NSString*)IncludeEntities {
		[super setInput:@"IncludeEntities" toValue:IncludeEntities];
	}

	/*!
	 * Set the value of the MaxId input for this Choreo.
	*(optional, string) Returns results with an ID less than (older than) or equal to the specified ID.
	 */
	-(void)setMaxId:(NSString*)MaxId {
		[super setInput:@"MaxId" toValue:MaxId];
	}

	/*!
	 * Set the value of the SinceId input for this Choreo.
	*(optional, string) Returns results with an ID greater than (more recent than) the specified ID.
	 */
	-(void)setSinceId:(NSString*)SinceId {
		[super setInput:@"SinceId" toValue:SinceId];
	}

	/*!
	 * Set the value of the TrimUser input for this Choreo.
	*(optional, boolean) When set to true, each tweet returned in a timeline will include a user object including only the status authors numerical ID. Defaults to false.
	 */
	-(void)setTrimUser:(NSString*)TrimUser {
		[super setInput:@"TrimUser" toValue:TrimUser];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the HomeTimelineLatestTweet Choreo.
 */
@implementation TMBTwitter_Timelines_HomeTimelineLatestTweet_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "ID" output from an execution of this Choreo.
	 * @return - NSString* (string) The Tweet ID.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getID {
		return [super getOutputByName:@"ID"];
	}

	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "ScreenName" output from an execution of this Choreo.
	 * @return - NSString* (string) The screen name of the user who posted this Tweet.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getScreenName {
		return [super getOutputByName:@"ScreenName"];
	}

	/*!
	 * Retrieve the value of the "Text" output from an execution of this Choreo.
	 * @return - NSString* (string) The Tweet text.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getText {
		return [super getOutputByName:@"Text"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the latest Tweet posted by the authenticating user or the users they follow.
 */
@implementation TMBTwitter_Timelines_HomeTimelineLatestTweet

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Timelines_HomeTimelineLatestTweet Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Timelines/HomeTimelineLatestTweet"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Timelines_HomeTimelineLatestTweet_ResultSet *results = [[TMBTwitter_Timelines_HomeTimelineLatestTweet_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the HomeTimelineLatestTweet Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Timelines_HomeTimelineLatestTweet_Inputs*)newInputSet {
		return [[TMBTwitter_Timelines_HomeTimelineLatestTweet_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the LatestMention Choreo.
 */
@implementation TMBTwitter_Timelines_LatestMention_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the ContributorDetails input for this Choreo.
	*(optional, boolean) Set to true to include the screen_name of the contributor. By default only the user_id of the contributor is included.
	 */
	-(void)setContributorDetails:(NSString*)ContributorDetails {
		[super setInput:@"ContributorDetails" toValue:ContributorDetails];
	}

	/*!
	 * Set the value of the IncludeEntities input for this Choreo.
	*(optional, boolean) The "entities" node containing extra metadata will not be included when set to false.
	 */
	-(void)setIncludeEntities:(NSString*)IncludeEntities {
		[super setInput:@"IncludeEntities" toValue:IncludeEntities];
	}

	/*!
	 * Set the value of the MaxId input for this Choreo.
	*(optional, string) Returns results with an ID less than (older than) or equal to the specified ID.
	 */
	-(void)setMaxId:(NSString*)MaxId {
		[super setInput:@"MaxId" toValue:MaxId];
	}

	/*!
	 * Set the value of the SinceId input for this Choreo.
	*(optional, string) Returns results with an ID greater than (more recent than) the specified ID.
	 */
	-(void)setSinceId:(NSString*)SinceId {
		[super setInput:@"SinceId" toValue:SinceId];
	}

	/*!
	 * Set the value of the TrimUser input for this Choreo.
	*(optional, boolean) When set to true, each tweet returned in a timeline will include a user object including only the status authors numerical ID. Defaults to false, which returns the complete user object.
	 */
	-(void)setTrimUser:(NSString*)TrimUser {
		[super setInput:@"TrimUser" toValue:TrimUser];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the LatestMention Choreo.
 */
@implementation TMBTwitter_Timelines_LatestMention_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "ID" output from an execution of this Choreo.
	 * @return - NSString* (string) The Tweet ID.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getID {
		return [super getOutputByName:@"ID"];
	}

	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "ScreenName" output from an execution of this Choreo.
	 * @return - NSString* (string) The screen name of the user who has mentioned the authenticated user.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getScreenName {
		return [super getOutputByName:@"ScreenName"];
	}

	/*!
	 * Retrieve the value of the "Text" output from an execution of this Choreo.
	 * @return - NSString* (string) The Tweet text.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getText {
		return [super getOutputByName:@"Text"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the latest mention for the authenticating user.
 */
@implementation TMBTwitter_Timelines_LatestMention

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Timelines_LatestMention Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Timelines/LatestMention"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Timelines_LatestMention_ResultSet *results = [[TMBTwitter_Timelines_LatestMention_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the LatestMention Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Timelines_LatestMention_Inputs*)newInputSet {
		return [[TMBTwitter_Timelines_LatestMention_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Mentions Choreo.
 */
@implementation TMBTwitter_Timelines_Mentions_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the ContributorDetails input for this Choreo.
	*(optional, boolean) Set to true to include the screen_name of the contributor. By default only the user_id of the contributor is included.
	 */
	-(void)setContributorDetails:(NSString*)ContributorDetails {
		[super setInput:@"ContributorDetails" toValue:ContributorDetails];
	}

	/*!
	 * Set the value of the Count input for this Choreo.
	*(optional, integer) Specifies the number of tweets to retrieve. Must be less than or equal to 200. Defaults to 20.
	 */
	-(void)setCount:(NSString*)Count {
		[super setInput:@"Count" toValue:Count];
	}

	/*!
	 * Set the value of the IncludeEntities input for this Choreo.
	*(optional, boolean) The "entities" node containing extra metadata will not be included when set to false.
	 */
	-(void)setIncludeEntities:(NSString*)IncludeEntities {
		[super setInput:@"IncludeEntities" toValue:IncludeEntities];
	}

	/*!
	 * Set the value of the MaxId input for this Choreo.
	*(optional, string) Returns results with an ID less than (older than) or equal to the specified ID.
	 */
	-(void)setMaxId:(NSString*)MaxId {
		[super setInput:@"MaxId" toValue:MaxId];
	}

	/*!
	 * Set the value of the SinceId input for this Choreo.
	*(optional, string) Returns results with an ID greater than (more recent than) the specified ID.
	 */
	-(void)setSinceId:(NSString*)SinceId {
		[super setInput:@"SinceId" toValue:SinceId];
	}

	/*!
	 * Set the value of the TrimUser input for this Choreo.
	*(optional, boolean) When set to true, each tweet returned in a timeline will include a user object including only the status authors numerical ID. Defaults to false, which returns the complete user object.
	 */
	-(void)setTrimUser:(NSString*)TrimUser {
		[super setInput:@"TrimUser" toValue:TrimUser];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Mentions Choreo.
 */
@implementation TMBTwitter_Timelines_Mentions_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the 20 most recent mentions for the authenticating user.
 */
@implementation TMBTwitter_Timelines_Mentions

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Timelines_Mentions Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Timelines/Mentions"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Timelines_Mentions_ResultSet *results = [[TMBTwitter_Timelines_Mentions_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Mentions Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Timelines_Mentions_Inputs*)newInputSet {
		return [[TMBTwitter_Timelines_Mentions_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the RetweetsOfMe Choreo.
 */
@implementation TMBTwitter_Timelines_RetweetsOfMe_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Count input for this Choreo.
	*(optional, integer) Specifies the number of records to retrieve. Must be less than or equal to 200. Defaults to 20.
	 */
	-(void)setCount:(NSString*)Count {
		[super setInput:@"Count" toValue:Count];
	}

	/*!
	 * Set the value of the IncludeEntities input for this Choreo.
	*(optional, boolean) The tweets "entities" node containing extra metadata will not be included when set to false.
	 */
	-(void)setIncludeEntities:(NSString*)IncludeEntities {
		[super setInput:@"IncludeEntities" toValue:IncludeEntities];
	}

	/*!
	 * Set the value of the IncludeUserEntities input for this Choreo.
	*(optional, boolean) The user "entities" node containing extra metadata will not be included when set to false.
	 */
	-(void)setIncludeUserEntities:(NSString*)IncludeUserEntities {
		[super setInput:@"IncludeUserEntities" toValue:IncludeUserEntities];
	}

	/*!
	 * Set the value of the MaxId input for this Choreo.
	*(optional, string) Returns results with an ID less than (older than) or equal to the specified ID.
	 */
	-(void)setMaxId:(NSString*)MaxId {
		[super setInput:@"MaxId" toValue:MaxId];
	}

	/*!
	 * Set the value of the SinceId input for this Choreo.
	*(optional, string) Returns results with an ID greater than (more recent than) the specified ID.
	 */
	-(void)setSinceId:(NSString*)SinceId {
		[super setInput:@"SinceId" toValue:SinceId];
	}

	/*!
	 * Set the value of the TrimUser input for this Choreo.
	*(optional, boolean) When set to true, each tweet returned in a timeline will include a user object including only the status authors numerical ID. Defaults to false.
	 */
	-(void)setTrimUser:(NSString*)TrimUser {
		[super setInput:@"TrimUser" toValue:TrimUser];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the RetweetsOfMe Choreo.
 */
@implementation TMBTwitter_Timelines_RetweetsOfMe_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the most recent Tweets posted by the authenticating user that have recently been retweeted by others.
 */
@implementation TMBTwitter_Timelines_RetweetsOfMe

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Timelines_RetweetsOfMe Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Timelines/RetweetsOfMe"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Timelines_RetweetsOfMe_ResultSet *results = [[TMBTwitter_Timelines_RetweetsOfMe_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the RetweetsOfMe Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Timelines_RetweetsOfMe_Inputs*)newInputSet {
		return [[TMBTwitter_Timelines_RetweetsOfMe_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UserTimeline Choreo.
 */
@implementation TMBTwitter_Timelines_UserTimeline_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(conditional, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(conditional, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(conditional, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(conditional, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the ContributorDetails input for this Choreo.
	*(optional, boolean) Set to true to include the screen_name of the contributor. By default only the user_id of the contributor is included.
	 */
	-(void)setContributorDetails:(NSString*)ContributorDetails {
		[super setInput:@"ContributorDetails" toValue:ContributorDetails];
	}

	/*!
	 * Set the value of the Count input for this Choreo.
	*(optional, integer) Specifies the number of records to retrieve. Must be less than or equal to 200. Defaults to 20.
	 */
	-(void)setCount:(NSString*)Count {
		[super setInput:@"Count" toValue:Count];
	}

	/*!
	 * Set the value of the ExcludeReplies input for this Choreo.
	*(optional, boolean) Set to true to prevent replies from appearing in the returned timeline.
	 */
	-(void)setExcludeReplies:(NSString*)ExcludeReplies {
		[super setInput:@"ExcludeReplies" toValue:ExcludeReplies];
	}

	/*!
	 * Set the value of the IncludeRetweets input for this Choreo.
	*(optional, boolean) When set to true, the response will include the "entities" node.
	 */
	-(void)setIncludeRetweets:(NSString*)IncludeRetweets {
		[super setInput:@"IncludeRetweets" toValue:IncludeRetweets];
	}

	/*!
	 * Set the value of the MaxId input for this Choreo.
	*(optional, string) Returns results with an ID less than (older than) or equal to the specified ID.
	 */
	-(void)setMaxId:(NSString*)MaxId {
		[super setInput:@"MaxId" toValue:MaxId];
	}

	/*!
	 * Set the value of the ScreenName input for this Choreo.
	*(conditional, string) Screen name of the user to return results for. Required unless providing the UserId.
	 */
	-(void)setScreenName:(NSString*)ScreenName {
		[super setInput:@"ScreenName" toValue:ScreenName];
	}

	/*!
	 * Set the value of the SinceId input for this Choreo.
	*(optional, string) Returns results with an ID greater than (more recent than) the specified ID.
	 */
	-(void)setSinceId:(NSString*)SinceId {
		[super setInput:@"SinceId" toValue:SinceId];
	}

	/*!
	 * Set the value of the TrimUser input for this Choreo.
	*(optional, boolean) When set to true, each tweet returned in a timeline will include a user object including only the status authors numerical ID. Defaults to false.
	 */
	-(void)setTrimUser:(NSString*)TrimUser {
		[super setInput:@"TrimUser" toValue:TrimUser];
	}

	/*!
	 * Set the value of the UserId input for this Choreo.
	*(conditional, string) ID of the user to return results for. Required unless providing the ScreenName.
	 */
	-(void)setUserId:(NSString*)UserId {
		[super setInput:@"UserId" toValue:UserId];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UserTimeline Choreo.
 */
@implementation TMBTwitter_Timelines_UserTimeline_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a collection of the most recent Tweets posted by the user whose screen_name or user_id is indicated.
 */
@implementation TMBTwitter_Timelines_UserTimeline

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Timelines_UserTimeline Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Timelines/UserTimeline"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Timelines_UserTimeline_ResultSet *results = [[TMBTwitter_Timelines_UserTimeline_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UserTimeline Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Timelines_UserTimeline_Inputs*)newInputSet {
		return [[TMBTwitter_Timelines_UserTimeline_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UserTimelineLatestTweet Choreo.
 */
@implementation TMBTwitter_Timelines_UserTimelineLatestTweet_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(conditional, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(conditional, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(conditional, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(conditional, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the ContributorDetails input for this Choreo.
	*(optional, boolean) Set to true to include the screen_name of the contributor. By default only the user_id of the contributor is included.
	 */
	-(void)setContributorDetails:(NSString*)ContributorDetails {
		[super setInput:@"ContributorDetails" toValue:ContributorDetails];
	}

	/*!
	 * Set the value of the ExcludeReplies input for this Choreo.
	*(optional, boolean) Set to true to prevent replies from appearing in the returned timeline.
	 */
	-(void)setExcludeReplies:(NSString*)ExcludeReplies {
		[super setInput:@"ExcludeReplies" toValue:ExcludeReplies];
	}

	/*!
	 * Set the value of the IncludeRetweets input for this Choreo.
	*(optional, boolean) When set to true, the response will include the "entities" node.
	 */
	-(void)setIncludeRetweets:(NSString*)IncludeRetweets {
		[super setInput:@"IncludeRetweets" toValue:IncludeRetweets];
	}

	/*!
	 * Set the value of the MaxId input for this Choreo.
	*(optional, string) Returns results with an ID less than (older than) or equal to the specified ID.
	 */
	-(void)setMaxId:(NSString*)MaxId {
		[super setInput:@"MaxId" toValue:MaxId];
	}

	/*!
	 * Set the value of the ScreenName input for this Choreo.
	*(conditional, string) Screen name of the user to return results for. Required unless providing the UserId.
	 */
	-(void)setScreenName:(NSString*)ScreenName {
		[super setInput:@"ScreenName" toValue:ScreenName];
	}

	/*!
	 * Set the value of the SinceId input for this Choreo.
	*(optional, string) Returns results with an ID greater than (more recent than) the specified ID.
	 */
	-(void)setSinceId:(NSString*)SinceId {
		[super setInput:@"SinceId" toValue:SinceId];
	}

	/*!
	 * Set the value of the TrimUser input for this Choreo.
	*(optional, boolean) When set to true, each tweet returned in a timeline will include a user object including only the status authors numerical ID. Defaults to false.
	 */
	-(void)setTrimUser:(NSString*)TrimUser {
		[super setInput:@"TrimUser" toValue:TrimUser];
	}

	/*!
	 * Set the value of the UserId input for this Choreo.
	*(conditional, string) ID of the user to return results for. Required unless providing the ScreenName.
	 */
	-(void)setUserId:(NSString*)UserId {
		[super setInput:@"UserId" toValue:UserId];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UserTimelineLatestTweet Choreo.
 */
@implementation TMBTwitter_Timelines_UserTimelineLatestTweet_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "ID" output from an execution of this Choreo.
	 * @return - NSString* (string) The Tweet ID.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getID {
		return [super getOutputByName:@"ID"];
	}

	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Text" output from an execution of this Choreo.
	 * @return - NSString* (string) The Tweet text.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getText {
		return [super getOutputByName:@"Text"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the latest Tweet posted by the user whose screen_name or user_id is indicated.
 */
@implementation TMBTwitter_Timelines_UserTimelineLatestTweet

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Timelines_UserTimelineLatestTweet Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Timelines/UserTimelineLatestTweet"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Timelines_UserTimelineLatestTweet_ResultSet *results = [[TMBTwitter_Timelines_UserTimelineLatestTweet_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UserTimelineLatestTweet Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Timelines_UserTimelineLatestTweet_Inputs*)newInputSet {
		return [[TMBTwitter_Timelines_UserTimelineLatestTweet_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Available Choreo.
 */
@implementation TMBTwitter_Trends_Available_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Available Choreo.
 */
@implementation TMBTwitter_Trends_Available_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the locations that Twitter has trending topic information for.
 */
@implementation TMBTwitter_Trends_Available

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Trends_Available Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Trends/Available"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Trends_Available_ResultSet *results = [[TMBTwitter_Trends_Available_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Available Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Trends_Available_Inputs*)newInputSet {
		return [[TMBTwitter_Trends_Available_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Closest Choreo.
 */
@implementation TMBTwitter_Trends_Closest_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Latitude input for this Choreo.
	*(required, decimal) The latitude coordinate of the location.
	 */
	-(void)setLatitude:(NSString*)Latitude {
		[super setInput:@"Latitude" toValue:Latitude];
	}

	/*!
	 * Set the value of the Longitude input for this Choreo.
	*(required, decimal) The longitude coordinate of the location.
	 */
	-(void)setLongitude:(NSString*)Longitude {
		[super setInput:@"Longitude" toValue:Longitude];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Closest Choreo.
 */
@implementation TMBTwitter_Trends_Closest_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves locations closest to a specified location that Twitter has trending topic info for.
 */
@implementation TMBTwitter_Trends_Closest

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Trends_Closest Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Trends/Closest"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Trends_Closest_ResultSet *results = [[TMBTwitter_Trends_Closest_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Closest Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Trends_Closest_Inputs*)newInputSet {
		return [[TMBTwitter_Trends_Closest_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Place Choreo.
 */
@implementation TMBTwitter_Trends_Place_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Exclude input for this Choreo.
	*(optional, boolean) If set to "hashtags", all hashtags from the trends list will be removed.
	 */
	-(void)setExclude:(NSString*)Exclude {
		[super setInput:@"Exclude" toValue:Exclude];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, string) The (WOEID) Yahoo! Where On Earth ID of the location to return trending information for. Global information is available by setting this parameter to 1.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Place Choreo.
 */
@implementation TMBTwitter_Trends_Place_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the top 10 trending topics for a specific WOEID.
 */
@implementation TMBTwitter_Trends_Place

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Trends_Place Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Trends/Place"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Trends_Place_ResultSet *results = [[TMBTwitter_Trends_Place_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Place Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Trends_Place_Inputs*)newInputSet {
		return [[TMBTwitter_Trends_Place_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetRetweeters Choreo.
 */
@implementation TMBTwitter_Tweets_GetRetweeters_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Cursor input for this Choreo.
	*(optional, integer) Allows you to pass in the previous_cursor or next_cursor in order to page through results.
	 */
	-(void)setCursor:(NSString*)Cursor {
		[super setInput:@"Cursor" toValue:Cursor];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, string) The numerical ID of the desired status.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the StringifyIDs input for this Choreo.
	*(optional, boolean) A boolean flag indicating that IDs should be returned as strings.
	 */
	-(void)setStringifyIDs:(NSString*)StringifyIDs {
		[super setInput:@"StringifyIDs" toValue:StringifyIDs];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetRetweeters Choreo.
 */
@implementation TMBTwitter_Tweets_GetRetweeters_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a collection of up to 100 user IDs belonging to users who have retweeted the specified tweet.
 */
@implementation TMBTwitter_Tweets_GetRetweeters

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Tweets_GetRetweeters Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Tweets/GetRetweeters"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Tweets_GetRetweeters_ResultSet *results = [[TMBTwitter_Tweets_GetRetweeters_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetRetweeters Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Tweets_GetRetweeters_Inputs*)newInputSet {
		return [[TMBTwitter_Tweets_GetRetweeters_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetRetweets Choreo.
 */
@implementation TMBTwitter_Tweets_GetRetweets_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Count input for this Choreo.
	*(optional, integer) Specifies the number of records to, up to a maximum of 100.
	 */
	-(void)setCount:(NSString*)Count {
		[super setInput:@"Count" toValue:Count];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, string) The numerical ID of the tweet to retrieve retweets for.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the TrimUser input for this Choreo.
	*(optional, boolean) When set to true, each tweet returned in a timeline will include a user object including only the status authors numerical ID.
	 */
	-(void)setTrimUser:(NSString*)TrimUser {
		[super setInput:@"TrimUser" toValue:TrimUser];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetRetweets Choreo.
 */
@implementation TMBTwitter_Tweets_GetRetweets_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves up to 100 of the first retweets of a given tweet.
 */
@implementation TMBTwitter_Tweets_GetRetweets

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Tweets_GetRetweets Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Tweets/GetRetweets"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Tweets_GetRetweets_ResultSet *results = [[TMBTwitter_Tweets_GetRetweets_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetRetweets Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Tweets_GetRetweets_Inputs*)newInputSet {
		return [[TMBTwitter_Tweets_GetRetweets_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the StatusesDestroy Choreo.
 */
@implementation TMBTwitter_Tweets_StatusesDestroy_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, string) The numerical ID of the status to delete.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the TrimUser input for this Choreo.
	*(optional, boolean) When set to true, each tweet returned in a timeline will include a user object including only the status authors numerical ID.
	 */
	-(void)setTrimUser:(NSString*)TrimUser {
		[super setInput:@"TrimUser" toValue:TrimUser];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the StatusesDestroy Choreo.
 */
@implementation TMBTwitter_Tweets_StatusesDestroy_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Deletes a specified status.
 */
@implementation TMBTwitter_Tweets_StatusesDestroy

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Tweets_StatusesDestroy Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Tweets/StatusesDestroy"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Tweets_StatusesDestroy_ResultSet *results = [[TMBTwitter_Tweets_StatusesDestroy_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the StatusesDestroy Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Tweets_StatusesDestroy_Inputs*)newInputSet {
		return [[TMBTwitter_Tweets_StatusesDestroy_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the StatusesShow Choreo.
 */
@implementation TMBTwitter_Tweets_StatusesShow_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, string) The numerical ID of the desired Tweet.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the IncludeEntities input for this Choreo.
	*(optional, boolean) The "entities" node containing extra metadata will not be included when set to false.
	 */
	-(void)setIncludeEntities:(NSString*)IncludeEntities {
		[super setInput:@"IncludeEntities" toValue:IncludeEntities];
	}

	/*!
	 * Set the value of the IncludeMyRetweet input for this Choreo.
	*(optional, boolean) When set to true, any Tweets returned that have been retweeted by the authenticating user will include an additional current_user_retweet node, containing the ID of the source status for the retweet.
	 */
	-(void)setIncludeMyRetweet:(NSString*)IncludeMyRetweet {
		[super setInput:@"IncludeMyRetweet" toValue:IncludeMyRetweet];
	}

	/*!
	 * Set the value of the TrimUser input for this Choreo.
	*(optional, boolean) When set to true, each tweet returned in a timeline will include a user object including only the status authors numerical ID. Defaults to false.
	 */
	-(void)setTrimUser:(NSString*)TrimUser {
		[super setInput:@"TrimUser" toValue:TrimUser];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the StatusesShow Choreo.
 */
@implementation TMBTwitter_Tweets_StatusesShow_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a single Tweet with a given ID.
 */
@implementation TMBTwitter_Tweets_StatusesShow

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Tweets_StatusesShow Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Tweets/StatusesShow"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Tweets_StatusesShow_ResultSet *results = [[TMBTwitter_Tweets_StatusesShow_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the StatusesShow Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Tweets_StatusesShow_Inputs*)newInputSet {
		return [[TMBTwitter_Tweets_StatusesShow_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the StatusesUpdate Choreo.
 */
@implementation TMBTwitter_Tweets_StatusesUpdate_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the DisplayCoordinates input for this Choreo.
	*(optional, boolean) Whether or not to put a pin on the exact coordinates a tweet has been sent from.
	 */
	-(void)setDisplayCoordinates:(NSString*)DisplayCoordinates {
		[super setInput:@"DisplayCoordinates" toValue:DisplayCoordinates];
	}

	/*!
	 * Set the value of the InReplyTo input for this Choreo.
	*(optional, string) The ID of an existing status that the update is in reply to.
	 */
	-(void)setInReplyTo:(NSString*)InReplyTo {
		[super setInput:@"InReplyTo" toValue:InReplyTo];
	}

	/*!
	 * Set the value of the Latitude input for this Choreo.
	*(optional, decimal) The latitude of the location this tweet refers to e.g., 40.71863.
	 */
	-(void)setLatitude:(NSString*)Latitude {
		[super setInput:@"Latitude" toValue:Latitude];
	}

	/*!
	 * Set the value of the Longitude input for this Choreo.
	*(optional, decimal) The longitude of the location this tweet refers to e.g., -74.005584.
	 */
	-(void)setLongitude:(NSString*)Longitude {
		[super setInput:@"Longitude" toValue:Longitude];
	}

	/*!
	 * Set the value of the PlaceID input for this Choreo.
	*(optional, string) The ID associated with a place in the world. These IDs can be retrieved from the PlacesAndGeo.ReverseGeocode Choreo.
	 */
	-(void)setPlaceID:(NSString*)PlaceID {
		[super setInput:@"PlaceID" toValue:PlaceID];
	}

	/*!
	 * Set the value of the PossiblySensitive input for this Choreo.
	*(optional, boolean) Set to true for content which may not be suitable for every audience.
	 */
	-(void)setPossiblySensitive:(NSString*)PossiblySensitive {
		[super setInput:@"PossiblySensitive" toValue:PossiblySensitive];
	}

	/*!
	 * Set the value of the StatusUpdate input for this Choreo.
	*(required, string) The text for your status update. 140-character limit.
	 */
	-(void)setStatusUpdate:(NSString*)StatusUpdate {
		[super setInput:@"StatusUpdate" toValue:StatusUpdate];
	}

	/*!
	 * Set the value of the TrimUser input for this Choreo.
	*(optional, boolean) When set to either true, each tweet returned in a timeline will include a user object including only the status authors numerical ID.
	 */
	-(void)setTrimUser:(NSString*)TrimUser {
		[super setInput:@"TrimUser" toValue:TrimUser];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the StatusesUpdate Choreo.
 */
@implementation TMBTwitter_Tweets_StatusesUpdate_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Allows you to update your Twitter status (aka Tweet).
 */
@implementation TMBTwitter_Tweets_StatusesUpdate

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Tweets_StatusesUpdate Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Tweets/StatusesUpdate"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Tweets_StatusesUpdate_ResultSet *results = [[TMBTwitter_Tweets_StatusesUpdate_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the StatusesUpdate Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Tweets_StatusesUpdate_Inputs*)newInputSet {
		return [[TMBTwitter_Tweets_StatusesUpdate_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UpdateWithMedia Choreo.
 */
@implementation TMBTwitter_Tweets_UpdateWithMedia_Inputs

	/*!
	 * Set the value of the MediaContent input for this Choreo.
	*(required, string) The Base64 encoded image content to post to Twitter.
	 */
	-(void)setMediaContent:(NSString*)MediaContent {
		[super setInput:@"MediaContent" toValue:MediaContent];
	}

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the DisplayCoordinates input for this Choreo.
	*(optional, boolean) Whether or not to put a pin on the exact coordinates a tweet has been sent from.
	 */
	-(void)setDisplayCoordinates:(NSString*)DisplayCoordinates {
		[super setInput:@"DisplayCoordinates" toValue:DisplayCoordinates];
	}

	/*!
	 * Set the value of the InReplyTo input for this Choreo.
	*(optional, string) The ID of an existing status that the update is in reply to.
	 */
	-(void)setInReplyTo:(NSString*)InReplyTo {
		[super setInput:@"InReplyTo" toValue:InReplyTo];
	}

	/*!
	 * Set the value of the Latitude input for this Choreo.
	*(optional, decimal) The latitude of the location this tweet refers to e.g., 40.71863.
	 */
	-(void)setLatitude:(NSString*)Latitude {
		[super setInput:@"Latitude" toValue:Latitude];
	}

	/*!
	 * Set the value of the Longitude input for this Choreo.
	*(optional, decimal) The longitude of the location this tweet refers to e.g., -74.005584.
	 */
	-(void)setLongitude:(NSString*)Longitude {
		[super setInput:@"Longitude" toValue:Longitude];
	}

	/*!
	 * Set the value of the PlaceID input for this Choreo.
	*(optional, string) The ID associated with a place in the world. These IDs can be retrieved from the PlacesAndGeo.ReverseGeocode Choreo.
	 */
	-(void)setPlaceID:(NSString*)PlaceID {
		[super setInput:@"PlaceID" toValue:PlaceID];
	}

	/*!
	 * Set the value of the PossiblySensitive input for this Choreo.
	*(optional, boolean) Set to true for content which may not be suitable for every audience.
	 */
	-(void)setPossiblySensitive:(NSString*)PossiblySensitive {
		[super setInput:@"PossiblySensitive" toValue:PossiblySensitive];
	}

	/*!
	 * Set the value of the StatusUpdate input for this Choreo.
	*(required, string) The text for your status update. 140-character limit.
	 */
	-(void)setStatusUpdate:(NSString*)StatusUpdate {
		[super setInput:@"StatusUpdate" toValue:StatusUpdate];
	}

	/*!
	 * Set the value of the VaultFile input for this Choreo.
	*
	 */
	-(void)setVaultFile:(NSString*)VaultFile {
		[super setInput:@"VaultFile" toValue:VaultFile];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UpdateWithMedia Choreo.
 */
@implementation TMBTwitter_Tweets_UpdateWithMedia_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Allows you to update your Twitter status and attach an image.
 */
@implementation TMBTwitter_Tweets_UpdateWithMedia

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Tweets_UpdateWithMedia Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Tweets/UpdateWithMedia"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Tweets_UpdateWithMedia_ResultSet *results = [[TMBTwitter_Tweets_UpdateWithMedia_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UpdateWithMedia Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Tweets_UpdateWithMedia_Inputs*)newInputSet {
		return [[TMBTwitter_Tweets_UpdateWithMedia_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetAccountSettings Choreo.
 */
@implementation TMBTwitter_Users_GetAccountSettings_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetAccountSettings Choreo.
 */
@implementation TMBTwitter_Users_GetAccountSettings_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves settings for the authenticating user including trend, geo, and sleep time information.
 */
@implementation TMBTwitter_Users_GetAccountSettings

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Users_GetAccountSettings Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Users/GetAccountSettings"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Users_GetAccountSettings_ResultSet *results = [[TMBTwitter_Users_GetAccountSettings_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetAccountSettings Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Users_GetAccountSettings_Inputs*)newInputSet {
		return [[TMBTwitter_Users_GetAccountSettings_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Lookup Choreo.
 */
@implementation TMBTwitter_Users_Lookup_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the IncludeEntities input for this Choreo.
	*(optional, boolean) The "entities" node containing extra metadata will not be included when set to false.
	 */
	-(void)setIncludeEntities:(NSString*)IncludeEntities {
		[super setInput:@"IncludeEntities" toValue:IncludeEntities];
	}

	/*!
	 * Set the value of the ScreenName input for this Choreo.
	*(conditional, string) A comma-separated list of up to 100 screen names. Required if UserId isn't specified.
	 */
	-(void)setScreenName:(NSString*)ScreenName {
		[super setInput:@"ScreenName" toValue:ScreenName];
	}

	/*!
	 * Set the value of the UserId input for this Choreo.
	*(conditional, string) A comma-separated list of up to 100 user IDs. Required if ScreenName isn't specified.
	 */
	-(void)setUserId:(NSString*)UserId {
		[super setInput:@"UserId" toValue:UserId];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Lookup Choreo.
 */
@implementation TMBTwitter_Users_Lookup_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Returns user objects for up to 100 Twitter users.
 */
@implementation TMBTwitter_Users_Lookup

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Users_Lookup Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Users/Lookup"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Users_Lookup_ResultSet *results = [[TMBTwitter_Users_Lookup_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Lookup Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Users_Lookup_Inputs*)newInputSet {
		return [[TMBTwitter_Users_Lookup_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Search Choreo.
 */
@implementation TMBTwitter_Users_Search_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Count input for this Choreo.
	*(optional, integer) The number of potential user results to retrieve per page. This value has a maximum of 20.
	 */
	-(void)setCount:(NSString*)Count {
		[super setInput:@"Count" toValue:Count];
	}

	/*!
	 * Set the value of the IncludeEntities input for this Choreo.
	*(optional, boolean) The "entities" node containing extra metadata will not be included when set to false.
	 */
	-(void)setIncludeEntities:(NSString*)IncludeEntities {
		[super setInput:@"IncludeEntities" toValue:IncludeEntities];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) Specifies the page of results to retrieve.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the SearchString input for this Choreo.
	*(required, string) The string used to search for users.
	 */
	-(void)setSearchString:(NSString*)SearchString {
		[super setInput:@"SearchString" toValue:SearchString];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Search Choreo.
 */
@implementation TMBTwitter_Users_Search_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Allows you to search public user accounts on Twitter.
 */
@implementation TMBTwitter_Users_Search

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Users_Search Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Users/Search"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Users_Search_ResultSet *results = [[TMBTwitter_Users_Search_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Search Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Users_Search_Inputs*)newInputSet {
		return [[TMBTwitter_Users_Search_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Show Choreo.
 */
@implementation TMBTwitter_Users_Show_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the IncludeEntities input for this Choreo.
	*(optional, boolean) The "entities" node containing extra metadata will not be included when set to false.
	 */
	-(void)setIncludeEntities:(NSString*)IncludeEntities {
		[super setInput:@"IncludeEntities" toValue:IncludeEntities];
	}

	/*!
	 * Set the value of the ScreenName input for this Choreo.
	*(conditional, string) The screen name of the user for whom to return results for. Required if UserId isn't specified.
	 */
	-(void)setScreenName:(NSString*)ScreenName {
		[super setInput:@"ScreenName" toValue:ScreenName];
	}

	/*!
	 * Set the value of the UserId input for this Choreo.
	*(conditional, string) The ID of the user for whom to return results for. Required if ScreenName isn't specified.
	 */
	-(void)setUserId:(NSString*)UserId {
		[super setInput:@"UserId" toValue:UserId];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Show Choreo.
 */
@implementation TMBTwitter_Users_Show_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves information about a specific Twitter user.
 */
@implementation TMBTwitter_Users_Show

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Users_Show Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Users/Show"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Users_Show_ResultSet *results = [[TMBTwitter_Users_Show_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Show Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Users_Show_Inputs*)newInputSet {
		return [[TMBTwitter_Users_Show_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UpdateAccountSettings Choreo.
 */
@implementation TMBTwitter_Users_UpdateAccountSettings_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the EndSleepTime input for this Choreo.
	*(optional, string) The hour that sleep time should end if it is enabled. The value should be provided in ISO8601 format (e.g., 00-23).
	 */
	-(void)setEndSleepTime:(NSString*)EndSleepTime {
		[super setInput:@"EndSleepTime" toValue:EndSleepTime];
	}

	/*!
	 * Set the value of the Language input for this Choreo.
	*(optional, string) The language which Twitter should render in for this user. The language must be specified by the appropriate two letter ISO 639-1 representation.
	 */
	-(void)setLanguage:(NSString*)Language {
		[super setInput:@"Language" toValue:Language];
	}

	/*!
	 * Set the value of the SleepTimeEnabled input for this Choreo.
	*(optional, boolean) When set to true, enables sleep time for the user. Sleep time is when push or SMS notifications should not be sent to the user.
	 */
	-(void)setSleepTimeEnabled:(NSString*)SleepTimeEnabled {
		[super setInput:@"SleepTimeEnabled" toValue:SleepTimeEnabled];
	}

	/*!
	 * Set the value of the StartSleepTime input for this Choreo.
	*(optional, string) The hour that sleep time should begin if it is enabled. The value should be provided in ISO8601 format (e.g., 00-23).
	 */
	-(void)setStartSleepTime:(NSString*)StartSleepTime {
		[super setInput:@"StartSleepTime" toValue:StartSleepTime];
	}

	/*!
	 * Set the value of the Timezone input for this Choreo.
	*(optional, string) The timezone dates and times that should be displayed for the user (e.g., Europe/Copenhagen, Pacific/Tongatapu)
	 */
	-(void)setTimezone:(NSString*)Timezone {
		[super setInput:@"Timezone" toValue:Timezone];
	}

	/*!
	 * Set the value of the TrendLocationWOEID input for this Choreo.
	*(optional, string) The Yahoo! Where On Earth ID to use as the user's default trend location.
	 */
	-(void)setTrendLocationWOEID:(NSString*)TrendLocationWOEID {
		[super setInput:@"TrendLocationWOEID" toValue:TrendLocationWOEID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UpdateAccountSettings Choreo.
 */
@implementation TMBTwitter_Users_UpdateAccountSettings_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Updates the authenticating user's settings such as trend location and sleep time settings.
 */
@implementation TMBTwitter_Users_UpdateAccountSettings

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Users_UpdateAccountSettings Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Users/UpdateAccountSettings"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Users_UpdateAccountSettings_ResultSet *results = [[TMBTwitter_Users_UpdateAccountSettings_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UpdateAccountSettings Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Users_UpdateAccountSettings_Inputs*)newInputSet {
		return [[TMBTwitter_Users_UpdateAccountSettings_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UpdateProfile Choreo.
 */
@implementation TMBTwitter_Users_UpdateProfile_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the Description input for this Choreo.
	*(optional, string) A description of the user owning the account. Maximum of 160 characters.
	 */
	-(void)setDescription:(NSString*)Description {
		[super setInput:@"Description" toValue:Description];
	}

	/*!
	 * Set the value of the IncludeUserEntities input for this Choreo.
	*(optional, boolean) The user "entities" node containing extra metadata will not be included when set to false.
	 */
	-(void)setIncludeUserEntities:(NSString*)IncludeUserEntities {
		[super setInput:@"IncludeUserEntities" toValue:IncludeUserEntities];
	}

	/*!
	 * Set the value of the Location input for this Choreo.
	*(optional, string) The city or country describing where the user of the account is located.
	 */
	-(void)setLocation:(NSString*)Location {
		[super setInput:@"Location" toValue:Location];
	}

	/*!
	 * Set the value of the Name input for this Choreo.
	*(optional, string) The full name associated with the profile. Maximum of 20 characters.
	 */
	-(void)setName:(NSString*)Name {
		[super setInput:@"Name" toValue:Name];
	}

	/*!
	 * Set the value of the SkipStatus input for this Choreo.
	*(optional, boolean) When set to true, statuses will not be included in the returned user objects.
	 */
	-(void)setSkipStatus:(NSString*)SkipStatus {
		[super setInput:@"SkipStatus" toValue:SkipStatus];
	}

	/*!
	 * Set the value of the URL input for this Choreo.
	*(optional, string) URL associated with the profile.
	 */
	-(void)setURL:(NSString*)URL {
		[super setInput:@"URL" toValue:URL];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UpdateProfile Choreo.
 */
@implementation TMBTwitter_Users_UpdateProfile_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Updates values that users are able to set under "Account" tab of their settings page.
 */
@implementation TMBTwitter_Users_UpdateProfile

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Users_UpdateProfile Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Users/UpdateProfile"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Users_UpdateProfile_ResultSet *results = [[TMBTwitter_Users_UpdateProfile_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UpdateProfile Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Users_UpdateProfile_Inputs*)newInputSet {
		return [[TMBTwitter_Users_UpdateProfile_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the VerifyCredentials Choreo.
 */
@implementation TMBTwitter_Users_VerifyCredentials_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret provided by Twitter or retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The API Key (or Consumer Key) provided by Twitter.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The API Secret (or Consumer Secret) provided by Twitter.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the IncludeUserEntities input for this Choreo.
	*(optional, boolean) The user "entities" node containing extra metadata will not be included when set to false.
	 */
	-(void)setIncludeUserEntities:(NSString*)IncludeUserEntities {
		[super setInput:@"IncludeUserEntities" toValue:IncludeUserEntities];
	}

	/*!
	 * Set the value of the SkipStatus input for this Choreo.
	*(optional, boolean) When set to true, statuses will not be included in the returned user objects.
	 */
	-(void)setSkipStatus:(NSString*)SkipStatus {
		[super setInput:@"SkipStatus" toValue:SkipStatus];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the VerifyCredentials Choreo.
 */
@implementation TMBTwitter_Users_VerifyCredentials_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Limit" output from an execution of this Choreo.
	 * @return - NSString* (integer) The rate limit ceiling for this particular request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLimit {
		return [super getOutputByName:@"Limit"];
	}

	/*!
	 * Retrieve the value of the "Remaining" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of requests left for the 15 minute window.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRemaining {
		return [super getOutputByName:@"Remaining"];
	}

	/*!
	 * Retrieve the value of the "Reset" output from an execution of this Choreo.
	 * @return - NSString* (date) The remaining window before the rate limit resets in UTC epoch seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getReset {
		return [super getOutputByName:@"Reset"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Twitter.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Allows you to test if supplied user credentials are valid.
 */
@implementation TMBTwitter_Users_VerifyCredentials

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBTwitter_Users_VerifyCredentials Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Twitter/Users/VerifyCredentials"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBTwitter_Users_VerifyCredentials_ResultSet *results = [[TMBTwitter_Users_VerifyCredentials_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the VerifyCredentials Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBTwitter_Users_VerifyCredentials_Inputs*)newInputSet {
		return [[TMBTwitter_Users_VerifyCredentials_Inputs alloc] init];
	}
@end
	