/*!
 * @TMBNexmo.m
 *
 * Execute Choreographies from the Temboo Nexmo bundle.
 *
 * iOS version 10.6.8
 *
 * LICENSE: Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @category   Services
 * @author     Reid Simonton, Temboo, Inc.
 * @copyright  2013 Temboo, Inc.
 * @license    http://www.apache.org/licenses/LICENSE-2.0 Apache License 2.0
 * @version    1.7
 */

#import "TMBNexmo.h"
#import "TMBChoreography_Protected.h"
#import "TMBChoreographyExecution.h"
#import "TMBTembooSession.h"
#import "TMBTembooUri.h"


/*!
 * Input object with appropriate setters for specifying arguments to the BuyNumber Choreo.
 */
@implementation TMBNexmo_Account_BuyNumber_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) Your API Key provided to you by Nexmo.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) Your API Secret provided to you by Nexmo.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the Country input for this Choreo.
	*(required, string) Country code. (e.g.: ES)
	 */
	-(void)setCountry:(NSString*)Country {
		[super setInput:@"Country" toValue:Country];
	}

	/*!
	 * Set the value of the Number input for this Choreo.
	*(required, string) An available inbound (MSISDN) number  (e.g. 34911067000).
	 */
	-(void)setNumber:(NSString*)Number {
		[super setInput:@"Number" toValue:Number];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the BuyNumber Choreo.
 */
@implementation TMBNexmo_Account_BuyNumber_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "ResponseStatusCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) The response status code returned from Nexmo. A 200 is returned for a successful request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponseStatusCode {
		return [super getOutputByName:@"ResponseStatusCode"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Nexmo. For a successful request, an empty response body is returned.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Purchase the specified inbound number.
 */
@implementation TMBNexmo_Account_BuyNumber

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBNexmo_Account_BuyNumber Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Nexmo/Account/BuyNumber"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBNexmo_Account_BuyNumber_ResultSet *results = [[TMBNexmo_Account_BuyNumber_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the BuyNumber Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBNexmo_Account_BuyNumber_Inputs*)newInputSet {
		return [[TMBNexmo_Account_BuyNumber_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CancelNumber Choreo.
 */
@implementation TMBNexmo_Account_CancelNumber_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) Your API Key provided to you by Nexmo.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) Your API Secret provided to you by Nexmo.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the Country input for this Choreo.
	*(required, string) 2-digit country code. (e.g.: CA)
	 */
	-(void)setCountry:(NSString*)Country {
		[super setInput:@"Country" toValue:Country];
	}

	/*!
	 * Set the value of the Number input for this Choreo.
	*(required, string) Your inbound (MSISDN) number (e.g. 34911067000).
	 */
	-(void)setNumber:(NSString*)Number {
		[super setInput:@"Number" toValue:Number];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CancelNumber Choreo.
 */
@implementation TMBNexmo_Account_CancelNumber_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "ResponseStatusCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) The response status code returned from Nexmo. A 200 is returned for a successful request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponseStatusCode {
		return [super getOutputByName:@"ResponseStatusCode"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Nexmo. For a successful request, an empty response body is returned.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Cancels the specified inbound number subscription.
 */
@implementation TMBNexmo_Account_CancelNumber

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBNexmo_Account_CancelNumber Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Nexmo/Account/CancelNumber"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBNexmo_Account_CancelNumber_ResultSet *results = [[TMBNexmo_Account_CancelNumber_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CancelNumber Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBNexmo_Account_CancelNumber_Inputs*)newInputSet {
		return [[TMBNexmo_Account_CancelNumber_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetBalance Choreo.
 */
@implementation TMBNexmo_Account_GetBalance_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) Your API Key provided to you by Nexmo.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) Your API Secret provided to you by Nexmo.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are "json" (the default) and "xml".
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetBalance Choreo.
 */
@implementation TMBNexmo_Account_GetBalance_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Nexmo. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieve your current account balance.
 */
@implementation TMBNexmo_Account_GetBalance

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBNexmo_Account_GetBalance Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Nexmo/Account/GetBalance"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBNexmo_Account_GetBalance_ResultSet *results = [[TMBNexmo_Account_GetBalance_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetBalance Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBNexmo_Account_GetBalance_Inputs*)newInputSet {
		return [[TMBNexmo_Account_GetBalance_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetNumbers Choreo.
 */
@implementation TMBNexmo_Account_GetNumbers_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) Your API Key provided to you by Nexmo.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) Your API Secret provided to you by Nexmo.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the Index input for this Choreo.
	*(optional, integer) Page index.
	 */
	-(void)setIndex:(NSString*)Index {
		[super setInput:@"Index" toValue:Index];
	}

	/*!
	 * Set the value of the Pattern input for this Choreo.
	*(optional, string) Pattern to match.
	 */
	-(void)setPattern:(NSString*)Pattern {
		[super setInput:@"Pattern" toValue:Pattern];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are "json" (the default) and "xml".
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Size input for this Choreo.
	*(optional, integer) Page size.
	 */
	-(void)setSize:(NSString*)Size {
		[super setInput:@"Size" toValue:Size];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetNumbers Choreo.
 */
@implementation TMBNexmo_Account_GetNumbers_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Nexmo. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Get all inbound numbers associated with your Nexmo account.
 */
@implementation TMBNexmo_Account_GetNumbers

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBNexmo_Account_GetNumbers Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Nexmo/Account/GetNumbers"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBNexmo_Account_GetNumbers_ResultSet *results = [[TMBNexmo_Account_GetNumbers_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetNumbers Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBNexmo_Account_GetNumbers_Inputs*)newInputSet {
		return [[TMBNexmo_Account_GetNumbers_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetPricingByCountry Choreo.
 */
@implementation TMBNexmo_Account_GetPricingByCountry_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) Your API Key provided to you by Nexmo.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) Your API Secret provided to you by Nexmo.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the Country input for this Choreo.
	*(required, string) A 2 letter country code.  (e.g. CA)
	 */
	-(void)setCountry:(NSString*)Country {
		[super setInput:@"Country" toValue:Country];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are "json" (the default) and "xml".
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetPricingByCountry Choreo.
 */
@implementation TMBNexmo_Account_GetPricingByCountry_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Nexmo. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieve Nexmo's outbound pricing for the specified country.
 */
@implementation TMBNexmo_Account_GetPricingByCountry

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBNexmo_Account_GetPricingByCountry Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Nexmo/Account/GetPricingByCountry"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBNexmo_Account_GetPricingByCountry_ResultSet *results = [[TMBNexmo_Account_GetPricingByCountry_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetPricingByCountry Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBNexmo_Account_GetPricingByCountry_Inputs*)newInputSet {
		return [[TMBNexmo_Account_GetPricingByCountry_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetPricingByPrefix Choreo.
 */
@implementation TMBNexmo_Account_GetPricingByPrefix_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) Your API Key provided to you by Nexmo.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) Your API Secret provided to you by Nexmo.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the Prefix input for this Choreo.
	*(required, integer) International dialing code. (e.g. 44)
	 */
	-(void)setPrefix:(NSString*)Prefix {
		[super setInput:@"Prefix" toValue:Prefix];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are "json" (the default) and "xml".
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetPricingByPrefix Choreo.
 */
@implementation TMBNexmo_Account_GetPricingByPrefix_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Nexmo. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieve Nexmo's outbound pricing for the specified international prefix.
 */
@implementation TMBNexmo_Account_GetPricingByPrefix

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBNexmo_Account_GetPricingByPrefix Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Nexmo/Account/GetPricingByPrefix"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBNexmo_Account_GetPricingByPrefix_ResultSet *results = [[TMBNexmo_Account_GetPricingByPrefix_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetPricingByPrefix Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBNexmo_Account_GetPricingByPrefix_Inputs*)newInputSet {
		return [[TMBNexmo_Account_GetPricingByPrefix_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the TopUp Choreo.
 */
@implementation TMBNexmo_Account_TopUp_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) Your API Key provided to you by Nexmo.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) Your API Secret provided to you by Nexmo.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are "json" (the default) and "xml".
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the TransactionID input for this Choreo.
	*(required, string) The transaction id associated with your **first** 'auto reload' top-up.
	 */
	-(void)setTransactionID:(NSString*)TransactionID {
		[super setInput:@"TransactionID" toValue:TransactionID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the TopUp Choreo.
 */
@implementation TMBNexmo_Account_TopUp_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "ResponseStatusCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) The response status code returned from Nexmo. A 200 is returned for a successful request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponseStatusCode {
		return [super getOutputByName:@"ResponseStatusCode"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Nexmo. For a successful request, an empty response body is returned.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Allows you to top-up your account provided that you've set up "auto-reload" in your Account Settings.
 */
@implementation TMBNexmo_Account_TopUp

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBNexmo_Account_TopUp Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Nexmo/Account/TopUp"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBNexmo_Account_TopUp_ResultSet *results = [[TMBNexmo_Account_TopUp_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the TopUp Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBNexmo_Account_TopUp_Inputs*)newInputSet {
		return [[TMBNexmo_Account_TopUp_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UpdateAccountSettings Choreo.
 */
@implementation TMBNexmo_Account_UpdateAccountSettings_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) Your API Key provided to you by Nexmo.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) Your API Secret provided to you by Nexmo.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the DeliveryReceiptCallbackURL input for this Choreo.
	*(conditional, string) Your new Delivery Receipt Callback URL.
	 */
	-(void)setDeliveryReceiptCallbackURL:(NSString*)DeliveryReceiptCallbackURL {
		[super setInput:@"DeliveryReceiptCallbackURL" toValue:DeliveryReceiptCallbackURL];
	}

	/*!
	 * Set the value of the InboundCallbackURL input for this Choreo.
	*(conditional, string) Your new Inbound Callback URL.
	 */
	-(void)setInboundCallbackURL:(NSString*)InboundCallbackURL {
		[super setInput:@"InboundCallbackURL" toValue:InboundCallbackURL];
	}

	/*!
	 * Set the value of the NewSecret input for this Choreo.
	*(optional, string) Your new API secret. (8 characters max)
	 */
	-(void)setNewSecret:(NSString*)NewSecret {
		[super setInput:@"NewSecret" toValue:NewSecret];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are "json" (the default) and "xml".
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UpdateAccountSettings Choreo.
 */
@implementation TMBNexmo_Account_UpdateAccountSettings_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "ResponseStatusCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) The response status code returned from Nexmo.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponseStatusCode {
		return [super getOutputByName:@"ResponseStatusCode"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Nexmo. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Update your account settings.
 */
@implementation TMBNexmo_Account_UpdateAccountSettings

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBNexmo_Account_UpdateAccountSettings Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Nexmo/Account/UpdateAccountSettings"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBNexmo_Account_UpdateAccountSettings_ResultSet *results = [[TMBNexmo_Account_UpdateAccountSettings_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UpdateAccountSettings Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBNexmo_Account_UpdateAccountSettings_Inputs*)newInputSet {
		return [[TMBNexmo_Account_UpdateAccountSettings_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UpdateNumber Choreo.
 */
@implementation TMBNexmo_Account_UpdateNumber_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) Your API Key provided to you by Nexmo.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) Your API Secret provided to you by Nexmo.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the CallbackURL input for this Choreo.
	*(optional, string) Mobile originated Callback URL.
	 */
	-(void)setCallbackURL:(NSString*)CallbackURL {
		[super setInput:@"CallbackURL" toValue:CallbackURL];
	}

	/*!
	 * Set the value of the Country input for this Choreo.
	*(required, string) 2-digit country code. (e.g. CA)
	 */
	-(void)setCountry:(NSString*)Country {
		[super setInput:@"Country" toValue:Country];
	}

	/*!
	 * Set the value of the Number input for this Choreo.
	*(required, string) Your inbound (MSISDN) number (e.g. 34911067000).
	 */
	-(void)setNumber:(NSString*)Number {
		[super setInput:@"Number" toValue:Number];
	}

	/*!
	 * Set the value of the SMPPSystemType input for this Choreo.
	*(optional, string) The Mobile Orignated associated system type for SMPP client only. (e.g.: inbound)
	 */
	-(void)setSMPPSystemType:(NSString*)SMPPSystemType {
		[super setInput:@"SMPPSystemType" toValue:SMPPSystemType];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UpdateNumber Choreo.
 */
@implementation TMBNexmo_Account_UpdateNumber_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "ResponseStatusCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) The response status code returned from Nexmo. A 200 is returned for a successful request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponseStatusCode {
		return [super getOutputByName:@"ResponseStatusCode"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Nexmo. For a successful request, an empty response body is returned.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Updates the callback details for the specified number.
 */
@implementation TMBNexmo_Account_UpdateNumber

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBNexmo_Account_UpdateNumber Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Nexmo/Account/UpdateNumber"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBNexmo_Account_UpdateNumber_ResultSet *results = [[TMBNexmo_Account_UpdateNumber_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UpdateNumber Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBNexmo_Account_UpdateNumber_Inputs*)newInputSet {
		return [[TMBNexmo_Account_UpdateNumber_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SendMessage Choreo.
 */
@implementation TMBNexmo_SMS_SendMessage_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) Your API Key provided to you by Nexmo.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) Your API Secret provided to you by Nexmo.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the Body input for this Choreo.
	*(optional, string) Hex encoded binary data. (e.g. 0011223344556677).  Required when Type is binary.
	 */
	-(void)setBody:(NSString*)Body {
		[super setInput:@"Body" toValue:Body];
	}

	/*!
	 * Set the value of the CallbackID input for this Choreo.
	*(conditional, string) A unique identifier that is part of your Temboo callback URL registered at Nexmo. Required in order to listen for a reply. See Choreo description for details.
	 */
	-(void)setCallbackID:(NSString*)CallbackID {
		[super setInput:@"CallbackID" toValue:CallbackID];
	}

	/*!
	 * Set the value of the ClientReference input for this Choreo.
	*(optional, string) Include a note for your reference. (40 characters max).
	 */
	-(void)setClientReference:(NSString*)ClientReference {
		[super setInput:@"ClientReference" toValue:ClientReference];
	}

	/*!
	 * Set the value of the DeliveryReceipt input for this Choreo.
	*(optional, integer) Set to 1 to receive a Delivery Receipt for this message. Make sure to configure your "Callback URL" in your "API Settings".
	 */
	-(void)setDeliveryReceipt:(NSString*)DeliveryReceipt {
		[super setInput:@"DeliveryReceipt" toValue:DeliveryReceipt];
	}

	/*!
	 * Set the value of the From input for this Choreo.
	*(required, string) The phone number associated with your Nexmo account e.g. 17185551234.
	 */
	-(void)setFrom:(NSString*)From {
		[super setInput:@"From" toValue:From];
	}

	/*!
	 * Set the value of the MessageClass input for this Choreo.
	*(optional, integer) Set to 0 for Flash SMS.
	 */
	-(void)setMessageClass:(NSString*)MessageClass {
		[super setInput:@"MessageClass" toValue:MessageClass];
	}

	/*!
	 * Set the value of the NetworkCode input for this Choreo.
	*(optional, string) Sends this message to the specifed network operator (MCCMNC).
	 */
	-(void)setNetworkCode:(NSString*)NetworkCode {
		[super setInput:@"NetworkCode" toValue:NetworkCode];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are "json" (the default) and "xml".
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the TTL input for this Choreo.
	*(optional, integer) Message life span (Time to  live) in milliseconds.
	 */
	-(void)setTTL:(NSString*)TTL {
		[super setInput:@"TTL" toValue:TTL];
	}

	/*!
	 * Set the value of the Text input for this Choreo.
	*(conditional, string) Required when Type is "text".  Body of the text message (with a maximum length of 3200 characters).
	 */
	-(void)setText:(NSString*)Text {
		[super setInput:@"Text" toValue:Text];
	}

	/*!
	 * Set the value of the Timeout input for this Choreo.
	*(conditional, integer) The amount of time (in minutes) to wait for a reply when a CallbackID is provided. Defaults to 10. See Choreo description for details.
	 */
	-(void)setTimeout:(NSString*)Timeout {
		[super setInput:@"Timeout" toValue:Timeout];
	}

	/*!
	 * Set the value of the To input for this Choreo.
	*(required, string) The mobile number in international format (e.g. 447525856424 or 00447525856424 when sending to UK).
	 */
	-(void)setTo:(NSString*)To {
		[super setInput:@"To" toValue:To];
	}

	/*!
	 * Set the value of the Type input for this Choreo.
	*(optional, string) This can be omitted for text (default), but is required when sending a Binary (binary), WAP Push (wappush), Unicode message (unicode), VCal (vcal) or VCard (vcard).
	 */
	-(void)setType:(NSString*)Type {
		[super setInput:@"Type" toValue:Type];
	}

	/*!
	 * Set the value of the UDH input for this Choreo.
	*(optional, string) Hex encoded User data header. (e.g. 06050415811581) (Required when Type is binary).
	 */
	-(void)setUDH:(NSString*)UDH {
		[super setInput:@"UDH" toValue:UDH];
	}

	/*!
	 * Set the value of the VCal input for this Choreo.
	*(optional, string) Correctly formatted VCal text body.
	 */
	-(void)setVCal:(NSString*)VCal {
		[super setInput:@"VCal" toValue:VCal];
	}

	/*!
	 * Set the value of the VCard input for this Choreo.
	*(optional, string) Correctly formatted VCard text body.
	 */
	-(void)setVCard:(NSString*)VCard {
		[super setInput:@"VCard" toValue:VCard];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SendMessage Choreo.
 */
@implementation TMBNexmo_SMS_SendMessage_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "CallbackData" output from an execution of this Choreo.
	 * @return - NSString* The Nexmo callback data retrieved after a user has replied to the SMS message. This is only returned if you've setup your Temboo callback URL at Nexmo, and  have provided the CallbackID input.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getCallbackData {
		return [super getOutputByName:@"CallbackData"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Nexmo. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Send a text message to any global number.
 */
@implementation TMBNexmo_SMS_SendMessage

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBNexmo_SMS_SendMessage Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Nexmo/SMS/SendMessage"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBNexmo_SMS_SendMessage_ResultSet *results = [[TMBNexmo_SMS_SendMessage_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SendMessage Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBNexmo_SMS_SendMessage_Inputs*)newInputSet {
		return [[TMBNexmo_SMS_SendMessage_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SearchMessage Choreo.
 */
@implementation TMBNexmo_Search_SearchMessage_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) Your API Key provided to you by Nexmo.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) Your API Secret provided to you by Nexmo.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the MessageID input for this Choreo.
	*(required, string) Your Message ID.
	 */
	-(void)setMessageID:(NSString*)MessageID {
		[super setInput:@"MessageID" toValue:MessageID];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are "json" (the default) and "xml".
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SearchMessage Choreo.
 */
@implementation TMBNexmo_Search_SearchMessage_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Nexmo. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Search for a previously sent message by Message ID.  Note that a sent message won't be immediately available for search.
 */
@implementation TMBNexmo_Search_SearchMessage

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBNexmo_Search_SearchMessage Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Nexmo/Search/SearchMessage"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBNexmo_Search_SearchMessage_ResultSet *results = [[TMBNexmo_Search_SearchMessage_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SearchMessage Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBNexmo_Search_SearchMessage_Inputs*)newInputSet {
		return [[TMBNexmo_Search_SearchMessage_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SearchMessages Choreo.
 */
@implementation TMBNexmo_Search_SearchMessages_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) Your API Key provided to you by Nexmo.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) Your API Secret provided to you by Nexmo.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the MessageIDs input for this Choreo.
	*(required, string) Comma-separated list of up to 10 MessageIDs
	 */
	-(void)setMessageIDs:(NSString*)MessageIDs {
		[super setInput:@"MessageIDs" toValue:MessageIDs];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are "json" (the default) and "xml".
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SearchMessages Choreo.
 */
@implementation TMBNexmo_Search_SearchMessages_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Nexmo. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Search for previously sent messages by Message IDs.  Note that  sent messages won't be immediately available for search.
 */
@implementation TMBNexmo_Search_SearchMessages

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBNexmo_Search_SearchMessages Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Nexmo/Search/SearchMessages"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBNexmo_Search_SearchMessages_ResultSet *results = [[TMBNexmo_Search_SearchMessages_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SearchMessages Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBNexmo_Search_SearchMessages_Inputs*)newInputSet {
		return [[TMBNexmo_Search_SearchMessages_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SearchMessagesByRecipientAndDate Choreo.
 */
@implementation TMBNexmo_Search_SearchMessagesByRecipientAndDate_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) Your API Key provided to you by Nexmo.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) Your API Secret provided to you by Nexmo.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the Date input for this Choreo.
	*(required, string) Date message was sent in the form of YYYY-MM-DD (UTC). (e.g. 2013-07-01)
	 */
	-(void)setDate:(NSString*)Date {
		[super setInput:@"Date" toValue:Date];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are "json" (the default) and "xml".
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the To input for this Choreo.
	*(required, string) The recipient's phone number.  (e.g. 123456780)
	 */
	-(void)setTo:(NSString*)To {
		[super setInput:@"To" toValue:To];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SearchMessagesByRecipientAndDate Choreo.
 */
@implementation TMBNexmo_Search_SearchMessagesByRecipientAndDate_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Nexmo. Corresponds to the ResponseFormat input. Defaults to json.  Note that if no matches are found for the specified Date and To, the Response will contain an error-code "-2".
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Searches sent messages by Date and Recipient. Note that a sent message won't be immediately available for searching.
 */
@implementation TMBNexmo_Search_SearchMessagesByRecipientAndDate

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBNexmo_Search_SearchMessagesByRecipientAndDate Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Nexmo/Search/SearchMessagesByRecipientAndDate"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBNexmo_Search_SearchMessagesByRecipientAndDate_ResultSet *results = [[TMBNexmo_Search_SearchMessagesByRecipientAndDate_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SearchMessagesByRecipientAndDate Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBNexmo_Search_SearchMessagesByRecipientAndDate_Inputs*)newInputSet {
		return [[TMBNexmo_Search_SearchMessagesByRecipientAndDate_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SearchNumbers Choreo.
 */
@implementation TMBNexmo_Search_SearchNumbers_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) Your API Key provided to you by Nexmo.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) Your API Secret provided to you by Nexmo.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the Country input for this Choreo.
	*(required, string) 2-digit country code. (e.g. CA)
	 */
	-(void)setCountry:(NSString*)Country {
		[super setInput:@"Country" toValue:Country];
	}

	/*!
	 * Set the value of the Index input for this Choreo.
	*(optional, integer) Page index
	 */
	-(void)setIndex:(NSString*)Index {
		[super setInput:@"Index" toValue:Index];
	}

	/*!
	 * Set the value of the Pattern input for this Choreo.
	*(optional, string) Pattern to match.
	 */
	-(void)setPattern:(NSString*)Pattern {
		[super setInput:@"Pattern" toValue:Pattern];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are "json" (the default) and "xml".
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Size input for this Choreo.
	*(optional, integer) Page size.
	 */
	-(void)setSize:(NSString*)Size {
		[super setInput:@"Size" toValue:Size];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SearchNumbers Choreo.
 */
@implementation TMBNexmo_Search_SearchNumbers_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Nexmo. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Get available inbound numbers for a given country.
 */
@implementation TMBNexmo_Search_SearchNumbers

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBNexmo_Search_SearchNumbers Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Nexmo/Search/SearchNumbers"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBNexmo_Search_SearchNumbers_ResultSet *results = [[TMBNexmo_Search_SearchNumbers_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SearchNumbers Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBNexmo_Search_SearchNumbers_Inputs*)newInputSet {
		return [[TMBNexmo_Search_SearchNumbers_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SearchRejectedMessages Choreo.
 */
@implementation TMBNexmo_Search_SearchRejectedMessages_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) Your API Key provided to you by Nexmo.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) Your API Secret provided to you by Nexmo.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the Date input for this Choreo.
	*(required, string) Date message was sent in the form of YYYY-MM-DD. (e.g. 2013-07-01)
	 */
	-(void)setDate:(NSString*)Date {
		[super setInput:@"Date" toValue:Date];
	}

	/*!
	 * Set the value of the MessageID input for this Choreo.
	*(required, string) Your Message ID.
	 */
	-(void)setMessageID:(NSString*)MessageID {
		[super setInput:@"MessageID" toValue:MessageID];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are "json" (the default) and "xml".
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the To input for this Choreo.
	*(required, string) The recipient's phone number.  (e.g. 123456780)
	 */
	-(void)setTo:(NSString*)To {
		[super setInput:@"To" toValue:To];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SearchRejectedMessages Choreo.
 */
@implementation TMBNexmo_Search_SearchRejectedMessages_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Nexmo. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Search for a previously sent message by Message ID.  Note that a sent message won't be immediately available for search.
 */
@implementation TMBNexmo_Search_SearchRejectedMessages

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBNexmo_Search_SearchRejectedMessages Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Nexmo/Search/SearchRejectedMessages"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBNexmo_Search_SearchRejectedMessages_ResultSet *results = [[TMBNexmo_Search_SearchRejectedMessages_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SearchRejectedMessages Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBNexmo_Search_SearchRejectedMessages_Inputs*)newInputSet {
		return [[TMBNexmo_Search_SearchRejectedMessages_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CaptureTextToSpeechPrompt Choreo.
 */
@implementation TMBNexmo_Voice_CaptureTextToSpeechPrompt_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) Your API Key provided to you by Nexmo.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) Your API Secret provided to you by Nexmo.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the ByeText input for this Choreo.
	*(required, string) Body of the text message played after digits are entered.  (with a maximum length of 500 characters).
	 */
	-(void)setByeText:(NSString*)ByeText {
		[super setInput:@"ByeText" toValue:ByeText];
	}

	/*!
	 * Set the value of the CallbackMethod input for this Choreo.
	*(optional, string) The HTTP method for your callback. Valid values are: GET (default) or POST.
	 */
	-(void)setCallbackMethod:(NSString*)CallbackMethod {
		[super setInput:@"CallbackMethod" toValue:CallbackMethod];
	}

	/*!
	 * Set the value of the CallbackURL input for this Choreo.
	*(optional, string) A CallbackURL that Nexmo will request when the call ends to notify your application.  If left empty, the Choreo will handle the callback for you and return the results in CallbackData.
	 */
	-(void)setCallbackURL:(NSString*)CallbackURL {
		[super setInput:@"CallbackURL" toValue:CallbackURL];
	}

	/*!
	 * Set the value of the DropIfMachine input for this Choreo.
	*(optional, integer) Deprecated (retained for backward compatibility only).
	 */
	-(void)setDropIfMachine:(NSString*)DropIfMachine {
		[super setInput:@"DropIfMachine" toValue:DropIfMachine];
	}

	/*!
	 * Set the value of the From input for this Choreo.
	*(optional, string) A voice enabled inbound number associated with your Nexmo account.
	 */
	-(void)setFrom:(NSString*)From {
		[super setInput:@"From" toValue:From];
	}

	/*!
	 * Set the value of the Language input for this Choreo.
	*(optional, string) The language used to play back your message.  The default is "en-us" which corresponds to United States english.
	 */
	-(void)setLanguage:(NSString*)Language {
		[super setInput:@"Language" toValue:Language];
	}

	/*!
	 * Set the value of the MachineDetection input for this Choreo.
	*(optional, string) If set to "hangup", the call will hang up immediately if a machine is detected, and the status in the CallbackData output will be set to "machine".
	 */
	-(void)setMachineDetection:(NSString*)MachineDetection {
		[super setInput:@"MachineDetection" toValue:MachineDetection];
	}

	/*!
	 * Set the value of the MachineTimeout input for this Choreo.
	*(optional, integer) Time allocated to analyze if the call has been answered by a machine. The default value is 15000 (milliseconds).
	 */
	-(void)setMachineTimeout:(NSString*)MachineTimeout {
		[super setInput:@"MachineTimeout" toValue:MachineTimeout];
	}

	/*!
	 * Set the value of the MaxDigits input for this Choreo.
	*(conditional, integer) Number of digits entered by the end user. Valid values are 1-9. Defaults to 1.
	 */
	-(void)setMaxDigits:(NSString*)MaxDigits {
		[super setInput:@"MaxDigits" toValue:MaxDigits];
	}

	/*!
	 * Set the value of the Repeat input for this Choreo.
	*(optional, integer) Define how many times you want to repeat the text message (default is 1 , maximum is 10).
	 */
	-(void)setRepeat:(NSString*)Repeat {
		[super setInput:@"Repeat" toValue:Repeat];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are "json" (the default) and "xml".
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Text input for this Choreo.
	*(conditional, string) Body of the text message. (with a maximum length of 1000 characters).
	 */
	-(void)setText:(NSString*)Text {
		[super setInput:@"Text" toValue:Text];
	}

	/*!
	 * Set the value of the To input for this Choreo.
	*(required, string) Phone number in international format and one recipient per request. (e.g. 447525856424 when sending to UK)
	 */
	-(void)setTo:(NSString*)To {
		[super setInput:@"To" toValue:To];
	}

	/*!
	 * Set the value of the Voice input for this Choreo.
	*(optional, string) The voice to be used female (default) or male
	 */
	-(void)setVoice:(NSString*)Voice {
		[super setInput:@"Voice" toValue:Voice];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CaptureTextToSpeechPrompt Choreo.
 */
@implementation TMBNexmo_Voice_CaptureTextToSpeechPrompt_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "CallbackData" output from an execution of this Choreo.
	 * @return - NSString* The Nexmo callback data retrieved after a user has entered a pin code. Note that this will only be returned when the CallbackURL input  is not provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getCallbackData {
		return [super getOutputByName:@"CallbackData"];
	}

	/*!
	 * Retrieve the value of the "Digits" output from an execution of this Choreo.
	 * @return - NSString* (integer) The digits that the call recipient entered into the keypad after receiving the call from Nexmo.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getDigits {
		return [super getOutputByName:@"Digits"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Nexmo. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Sends a Text-to-Speech message to the specifed Number and captures keypad entries from the receiver.
 */
@implementation TMBNexmo_Voice_CaptureTextToSpeechPrompt

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBNexmo_Voice_CaptureTextToSpeechPrompt Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Nexmo/Voice/CaptureTextToSpeechPrompt"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBNexmo_Voice_CaptureTextToSpeechPrompt_ResultSet *results = [[TMBNexmo_Voice_CaptureTextToSpeechPrompt_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CaptureTextToSpeechPrompt Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBNexmo_Voice_CaptureTextToSpeechPrompt_Inputs*)newInputSet {
		return [[TMBNexmo_Voice_CaptureTextToSpeechPrompt_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ConfirmTextToSpeechPrompt Choreo.
 */
@implementation TMBNexmo_Voice_ConfirmTextToSpeechPrompt_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) Your API Key provided to you by Nexmo.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) Your API Secret provided to you by Nexmo.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the ByeText input for this Choreo.
	*(required, string) Body of the text message played after digits are entered. (with a maximum length of 500 characters),
	 */
	-(void)setByeText:(NSString*)ByeText {
		[super setInput:@"ByeText" toValue:ByeText];
	}

	/*!
	 * Set the value of the CallbackMethod input for this Choreo.
	*(optional, string) The HTTP method for your callback. Valid values are: GET (default) or POST.
	 */
	-(void)setCallbackMethod:(NSString*)CallbackMethod {
		[super setInput:@"CallbackMethod" toValue:CallbackMethod];
	}

	/*!
	 * Set the value of the CallbackURL input for this Choreo.
	*(required, string) A URL that Nexmo will request when the call ends to notify your application.
	 */
	-(void)setCallbackURL:(NSString*)CallbackURL {
		[super setInput:@"CallbackURL" toValue:CallbackURL];
	}

	/*!
	 * Set the value of the DropIfMachine input for this Choreo.
	*(optional, integer) Deprecated (retained for backward compatibility only).
	 */
	-(void)setDropIfMachine:(NSString*)DropIfMachine {
		[super setInput:@"DropIfMachine" toValue:DropIfMachine];
	}

	/*!
	 * Set the value of the FailedText input for this Choreo.
	*(required, string) Body of the text message played after 3 failed attempts. (with a maximum length of 500 characters),
	 */
	-(void)setFailedText:(NSString*)FailedText {
		[super setInput:@"FailedText" toValue:FailedText];
	}

	/*!
	 * Set the value of the From input for this Choreo.
	*(optional, string) A voice enabled inbound number associated with your Nexmo account.
	 */
	-(void)setFrom:(NSString*)From {
		[super setInput:@"From" toValue:From];
	}

	/*!
	 * Set the value of the Language input for this Choreo.
	*(optional, string) The language used to play back your message.  The default is "en-us" which corresponds to United States english.
	 */
	-(void)setLanguage:(NSString*)Language {
		[super setInput:@"Language" toValue:Language];
	}

	/*!
	 * Set the value of the MachineDetection input for this Choreo.
	*(optional, string) If set to "hangup", the call will hang up immediately if a machine is detected, and the status in the CallbackData output will be set to "machine".
	 */
	-(void)setMachineDetection:(NSString*)MachineDetection {
		[super setInput:@"MachineDetection" toValue:MachineDetection];
	}

	/*!
	 * Set the value of the MachineTimeout input for this Choreo.
	*(optional, integer) Time allocated to analyze if the call has been answered by a machine. The default value is 15000 (milliseconds).
	 */
	-(void)setMachineTimeout:(NSString*)MachineTimeout {
		[super setInput:@"MachineTimeout" toValue:MachineTimeout];
	}

	/*!
	 * Set the value of the MaxDigits input for this Choreo.
	*(required, integer) Number of digits entered by the end user.
	 */
	-(void)setMaxDigits:(NSString*)MaxDigits {
		[super setInput:@"MaxDigits" toValue:MaxDigits];
	}

	/*!
	 * Set the value of the PinCode input for this Choreo.
	*(required, string) Pin-code to be entered by end user (Pin-code length should be equals to MaxDigits).
	 */
	-(void)setPinCode:(NSString*)PinCode {
		[super setInput:@"PinCode" toValue:PinCode];
	}

	/*!
	 * Set the value of the Repeat input for this Choreo.
	*(optional, integer) Define how many times you want to repeat the text message (default is 1 , maximum is 10).
	 */
	-(void)setRepeat:(NSString*)Repeat {
		[super setInput:@"Repeat" toValue:Repeat];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are "json" (the default) and "xml".
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Text input for this Choreo.
	*(conditional, string) Body of the text message (with a maximum length of 1000 characters).
	 */
	-(void)setText:(NSString*)Text {
		[super setInput:@"Text" toValue:Text];
	}

	/*!
	 * Set the value of the To input for this Choreo.
	*(required, string) Phone number in international format and one recipient per request. (e.g. 447525856424 when sending to UK)
	 */
	-(void)setTo:(NSString*)To {
		[super setInput:@"To" toValue:To];
	}

	/*!
	 * Set the value of the Voice input for this Choreo.
	*(optional, string) The voice to be used female (default) or male.
	 */
	-(void)setVoice:(NSString*)Voice {
		[super setInput:@"Voice" toValue:Voice];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ConfirmTextToSpeechPrompt Choreo.
 */
@implementation TMBNexmo_Voice_ConfirmTextToSpeechPrompt_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Nexmo. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Sends a Text-to-Speech message to the specifed Number and confirms the specifed pin-code.
 */
@implementation TMBNexmo_Voice_ConfirmTextToSpeechPrompt

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBNexmo_Voice_ConfirmTextToSpeechPrompt Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Nexmo/Voice/ConfirmTextToSpeechPrompt"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBNexmo_Voice_ConfirmTextToSpeechPrompt_ResultSet *results = [[TMBNexmo_Voice_ConfirmTextToSpeechPrompt_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ConfirmTextToSpeechPrompt Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBNexmo_Voice_ConfirmTextToSpeechPrompt_Inputs*)newInputSet {
		return [[TMBNexmo_Voice_ConfirmTextToSpeechPrompt_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the TextToSpeech Choreo.
 */
@implementation TMBNexmo_Voice_TextToSpeech_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) Your API Key provided to you by Nexmo.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) Your API Secret provided to you by Nexmo.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the CallbackMethod input for this Choreo.
	*(optional, string) The HTTP method for your callback. Valid values are: GET (default) or POST.
	 */
	-(void)setCallbackMethod:(NSString*)CallbackMethod {
		[super setInput:@"CallbackMethod" toValue:CallbackMethod];
	}

	/*!
	 * Set the value of the CallbackURL input for this Choreo.
	*(optional, string) A URL that Nexmo will request when the call ends to notify your application.
	 */
	-(void)setCallbackURL:(NSString*)CallbackURL {
		[super setInput:@"CallbackURL" toValue:CallbackURL];
	}

	/*!
	 * Set the value of the DropIfMachine input for this Choreo.
	*(optional, integer) Deprecated (retained for backward compatibility only).
	 */
	-(void)setDropIfMachine:(NSString*)DropIfMachine {
		[super setInput:@"DropIfMachine" toValue:DropIfMachine];
	}

	/*!
	 * Set the value of the From input for this Choreo.
	*(optional, string) A voice enabled inbound number associated with your Nexmo account.
	 */
	-(void)setFrom:(NSString*)From {
		[super setInput:@"From" toValue:From];
	}

	/*!
	 * Set the value of the Language input for this Choreo.
	*(optional, string) The language used to play back your message.  The default is "en-us" which corresponds to United States english.
	 */
	-(void)setLanguage:(NSString*)Language {
		[super setInput:@"Language" toValue:Language];
	}

	/*!
	 * Set the value of the MachineDetection input for this Choreo.
	*(optional, string) If set to "hangup", the call will hang up immediately if a machine is detected, and the status in the CallbackData output will be set to "machine".
	 */
	-(void)setMachineDetection:(NSString*)MachineDetection {
		[super setInput:@"MachineDetection" toValue:MachineDetection];
	}

	/*!
	 * Set the value of the MachineTimeout input for this Choreo.
	*(optional, integer) Time allocated to analyze if the call has been answered by a machine. The default value is 15000 (milliseconds).
	 */
	-(void)setMachineTimeout:(NSString*)MachineTimeout {
		[super setInput:@"MachineTimeout" toValue:MachineTimeout];
	}

	/*!
	 * Set the value of the Repeat input for this Choreo.
	*(optional, integer) Define how many times you want to repeat the text message (default is 1 , maximum is 10).
	 */
	-(void)setRepeat:(NSString*)Repeat {
		[super setInput:@"Repeat" toValue:Repeat];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are "json" (the default) and "xml".
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Text input for this Choreo.
	*(conditional, string) Body of the text message (with a maximum length of 1000 characters).
	 */
	-(void)setText:(NSString*)Text {
		[super setInput:@"Text" toValue:Text];
	}

	/*!
	 * Set the value of the To input for this Choreo.
	*(required, string) Phone number in international format and one recipient per request. (e.g.: 447525856424 when sending to UK)
	 */
	-(void)setTo:(NSString*)To {
		[super setInput:@"To" toValue:To];
	}

	/*!
	 * Set the value of the Voice input for this Choreo.
	*(optional, string) The voice to be used female (default) or male
	 */
	-(void)setVoice:(NSString*)Voice {
		[super setInput:@"Voice" toValue:Voice];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the TextToSpeech Choreo.
 */
@implementation TMBNexmo_Voice_TextToSpeech_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Nexmo. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Sends a Text-to-Speech message to the specified number.
 */
@implementation TMBNexmo_Voice_TextToSpeech

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBNexmo_Voice_TextToSpeech Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Nexmo/Voice/TextToSpeech"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBNexmo_Voice_TextToSpeech_ResultSet *results = [[TMBNexmo_Voice_TextToSpeech_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the TextToSpeech Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBNexmo_Voice_TextToSpeech_Inputs*)newInputSet {
		return [[TMBNexmo_Voice_TextToSpeech_Inputs alloc] init];
	}
@end
	