/*!
 * @TMBGoogle.m
 *
 * Execute Choreographies from the Temboo Google bundle.
 *
 * iOS version 10.6.8
 *
 * LICENSE: Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @category   Services
 * @author     Reid Simonton, Temboo, Inc.
 * @copyright  2013 Temboo, Inc.
 * @license    http://www.apache.org/licenses/LICENSE-2.0 Apache License 2.0
 * @version    1.7
 */

#import "TMBGoogle.h"
#import "TMBChoreography_Protected.h"
#import "TMBChoreographyExecution.h"
#import "TMBTembooSession.h"
#import "TMBTembooUri.h"


/*!
 * Input object with appropriate setters for specifying arguments to the GetMetrics Choreo.
 */
@implementation TMBGoogle_Analytics_GetMetrics_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Dimensions input for this Choreo.
	*(optional, string) Defines the primary data keys for your Analytics report. Use dimensions to segment your web property metrics (e.g.  ga:browser or ga:city).
	 */
	-(void)setDimensions:(NSString*)Dimensions {
		[super setInput:@"Dimensions" toValue:Dimensions];
	}

	/*!
	 * Set the value of the EndDate input for this Choreo.
	*(required, date) The end date for the range of data you want to retrieve. Epoch timestamp in milliseconds or formatted as yyyy-MM-dd.
	 */
	-(void)setEndDate:(NSString*)EndDate {
		[super setInput:@"EndDate" toValue:EndDate];
	}

	/*!
	 * Set the value of the Filters input for this Choreo.
	*(optional, string) Restricts the data returned by a dimension or metric you want to filter by using an expression (i.e. ga:timeOnPage==10).
	 */
	-(void)setFilters:(NSString*)Filters {
		[super setInput:@"Filters" toValue:Filters];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The max results to be returned in the feed. Defaults to 50.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the Metrics input for this Choreo.
	*(optional, string) This is a comma separated list of metrics you want to retrieve. Defaults to: ga:visits,ga:bounces,ga:pageviews.
	 */
	-(void)setMetrics:(NSString*)Metrics {
		[super setInput:@"Metrics" toValue:Metrics];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) Deprecated (retained for backward compatibility only).
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the ProfileId input for this Choreo.
	*(required, integer) The Google Analytics Profile ID to access. This is also known as the View ID. It can be found in the Admin > View Settings section of a particular profile.
	 */
	-(void)setProfileId:(NSString*)ProfileId {
		[super setInput:@"ProfileId" toValue:ProfileId];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: XML (the default) and JSON.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Segment input for this Choreo.
	*(optional, string) Used to segment your data by dimensions and/or metrics. You can use expressions for segments just as you would for the Filters parameter.
	 */
	-(void)setSegment:(NSString*)Segment {
		[super setInput:@"Segment" toValue:Segment];
	}

	/*!
	 * Set the value of the Sort input for this Choreo.
	*(optional, string) Indicates the sorting order and direction for the returned data. Values can be separated by commas (i.e. ga:browser,ga:pageviews).
	 */
	-(void)setSort:(NSString*)Sort {
		[super setInput:@"Sort" toValue:Sort];
	}

	/*!
	 * Set the value of the StartDate input for this Choreo.
	*(required, date) The start date for the range of data to retrieve. Use epoch timestamp in milliseconds or formatted as yyyy-MM-dd.
	 */
	-(void)setStartDate:(NSString*)StartDate {
		[super setInput:@"StartDate" toValue:StartDate];
	}

	/*!
	 * Set the value of the StartIndex input for this Choreo.
	*(optional, integer) The starting entry for the feed. Defaults to 1.
	 */
	-(void)setStartIndex:(NSString*)StartIndex {
		[super setInput:@"StartIndex" toValue:StartIndex];
	}

	/*!
	 * Set the value of the Username input for this Choreo.
	*(required, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setUsername:(NSString*)Username {
		[super setInput:@"Username" toValue:Username];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetMetrics Choreo.
 */
@implementation TMBGoogle_Analytics_GetMetrics_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Bounces" output from an execution of this Choreo.
	 * @return - NSString* (integer) The bounces metrics parsed from the Google Analytics response
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getBounces {
		return [super getOutputByName:@"Bounces"];
	}

	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "PageViews" output from an execution of this Choreo.
	 * @return - NSString* (integer) The page views parsed from the Google Analytics response
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPageViews {
		return [super getOutputByName:@"PageViews"];
	}

	/*!
	 * Retrieve the value of the "Visits" output from an execution of this Choreo.
	 * @return - NSString* (integer) The visits metrics parsed from the Google Analytics response.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getVisits {
		return [super getOutputByName:@"Visits"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves metrics such as visits, page views, bounces within a specified time frame.
 */
@implementation TMBGoogle_Analytics_GetMetrics

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Analytics_GetMetrics Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Analytics/GetMetrics"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Analytics_GetMetrics_ResultSet *results = [[TMBGoogle_Analytics_GetMetrics_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetMetrics Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Analytics_GetMetrics_Inputs*)newInputSet {
		return [[TMBGoogle_Analytics_GetMetrics_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetQueryResults Choreo.
 */
@implementation TMBGoogle_BigQuery_Jobs_GetQueryResults_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying which fields to include in a partial response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the JobID input for this Choreo.
	*(required, string) Job ID of the query job.
	 */
	-(void)setJobID:(NSString*)JobID {
		[super setInput:@"JobID" toValue:JobID];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) Maximum number of results to return. Defaults to 100. Max is 1000.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) Page token, returned by a previous call, identifying the result set.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the ProjectID input for this Choreo.
	*(required, string) The ID of your Google API project.
	 */
	-(void)setProjectID:(NSString*)ProjectID {
		[super setInput:@"ProjectID" toValue:ProjectID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the StartIndex input for this Choreo.
	*(optional, integer) Zero-based index of the starting row to read.
	 */
	-(void)setStartIndex:(NSString*)StartIndex {
		[super setInput:@"StartIndex" toValue:StartIndex];
	}

	/*!
	 * Set the value of the TimeoutMs input for this Choreo.
	*(optional, integer) How long to wait for the query to complete, in milliseconds, before returning. Default is 10000. If the timeout passes before the job completes, the 'jobComplete' field in the response will be false.
	 */
	-(void)setTimeoutMs:(NSString*)TimeoutMs {
		[super setInput:@"TimeoutMs" toValue:TimeoutMs];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetQueryResults Choreo.
 */
@implementation TMBGoogle_BigQuery_Jobs_GetQueryResults_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the results of a query with a given JobID when a query request times out and returns before the results are available.
 */
@implementation TMBGoogle_BigQuery_Jobs_GetQueryResults

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_BigQuery_Jobs_GetQueryResults Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/BigQuery/Jobs/GetQueryResults"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_BigQuery_Jobs_GetQueryResults_ResultSet *results = [[TMBGoogle_BigQuery_Jobs_GetQueryResults_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetQueryResults Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_BigQuery_Jobs_GetQueryResults_Inputs*)newInputSet {
		return [[TMBGoogle_BigQuery_Jobs_GetQueryResults_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Query Choreo.
 */
@implementation TMBGoogle_BigQuery_Jobs_Query_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the DatasetID input for this Choreo.
	*(optional, string) The ID of the dataset that your table belongs to.
	 */
	-(void)setDatasetID:(NSString*)DatasetID {
		[super setInput:@"DatasetID" toValue:DatasetID];
	}

	/*!
	 * Set the value of the DryRun input for this Choreo.
	*(optional, boolean) If set to true, BigQuery doesn't run the job. Instead, if the query is valid, BigQuery returns statistics about the job such as how many bytes would be processed. Defaults to false.
	 */
	-(void)setDryRun:(NSString*)DryRun {
		[super setInput:@"DryRun" toValue:DryRun];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying which fields to include in a partial response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) Maximum number of results to return. Defaults to 100. Max is 1000.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) Page token, returned by a previous call, identifying the result set.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the ProjectID input for this Choreo.
	*(required, string) The ID of your Google API project.
	 */
	-(void)setProjectID:(NSString*)ProjectID {
		[super setInput:@"ProjectID" toValue:ProjectID];
	}

	/*!
	 * Set the value of the Query input for this Choreo.
	*(required, string) A query to execute. Example: SELECT count(f1) FROM [myProjectId:myDatasetId.myTableId].
	 */
	-(void)setQuery:(NSString*)Query {
		[super setInput:@"Query" toValue:Query];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the TimeoutMs input for this Choreo.
	*(optional, integer) How long to wait for the query to complete, in milliseconds, before the request times out and returns. Defaults to 10000.
	 */
	-(void)setTimeoutMs:(NSString*)TimeoutMs {
		[super setInput:@"TimeoutMs" toValue:TimeoutMs];
	}

	/*!
	 * Set the value of the UseQueryCache input for this Choreo.
	*(optional, boolean) Whether to look for the result in the query cache. Defaults to true.
	 */
	-(void)setUseQueryCache:(NSString*)UseQueryCache {
		[super setInput:@"UseQueryCache" toValue:UseQueryCache];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Query Choreo.
 */
@implementation TMBGoogle_BigQuery_Jobs_Query_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Runs a BigQuery SQL query and returns results if the query completes within a specified timeout. When a query timeout occurs, results can be retrieved using the Job ID returned in the response.
 */
@implementation TMBGoogle_BigQuery_Jobs_Query

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_BigQuery_Jobs_Query Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/BigQuery/Jobs/Query"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_BigQuery_Jobs_Query_ResultSet *results = [[TMBGoogle_BigQuery_Jobs_Query_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Query Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_BigQuery_Jobs_Query_Inputs*)newInputSet {
		return [[TMBGoogle_BigQuery_Jobs_Query_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the InsertAll Choreo.
 */
@implementation TMBGoogle_BigQuery_TableData_InsertAll_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the DatasetID input for this Choreo.
	*(required, string) The ID of the dataset that your table belongs to.
	 */
	-(void)setDatasetID:(NSString*)DatasetID {
		[super setInput:@"DatasetID" toValue:DatasetID];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying which fields to include in a partial response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the IgnoreUnknownValues input for this Choreo.
	*(optional, boolean) Accept rows that contain values that do not match the schema. The unknown values are ignored. Default is false, which treats unknown values as errors.
	 */
	-(void)setIgnoreUnknownValues:(NSString*)IgnoreUnknownValues {
		[super setInput:@"IgnoreUnknownValues" toValue:IgnoreUnknownValues];
	}

	/*!
	 * Set the value of the ProjectID input for this Choreo.
	*(required, string) The ID of your Google API project.
	 */
	-(void)setProjectID:(NSString*)ProjectID {
		[super setInput:@"ProjectID" toValue:ProjectID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Rows input for this Choreo.
	*(conditional, json) The rows to insert. This should be a JSON array containing at least one object representing a row. See Choreo notes for formatting details.
	 */
	-(void)setRows:(NSString*)Rows {
		[super setInput:@"Rows" toValue:Rows];
	}

	/*!
	 * Set the value of the TableID input for this Choreo.
	*(required, string) The ID of the BigQuery table to insert a row into.
	 */
	-(void)setTableID:(NSString*)TableID {
		[super setInput:@"TableID" toValue:TableID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the InsertAll Choreo.
 */
@implementation TMBGoogle_BigQuery_TableData_InsertAll_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Streams data into BigQuery one record at a time.
 */
@implementation TMBGoogle_BigQuery_TableData_InsertAll

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_BigQuery_TableData_InsertAll Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/BigQuery/TableData/InsertAll"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_BigQuery_TableData_InsertAll_ResultSet *results = [[TMBGoogle_BigQuery_TableData_InsertAll_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the InsertAll Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_BigQuery_TableData_InsertAll_Inputs*)newInputSet {
		return [[TMBGoogle_BigQuery_TableData_InsertAll_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the List Choreo.
 */
@implementation TMBGoogle_BigQuery_TableData_List_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the DatasetID input for this Choreo.
	*(required, string) The ID of the dataset that your table belongs to.
	 */
	-(void)setDatasetID:(NSString*)DatasetID {
		[super setInput:@"DatasetID" toValue:DatasetID];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying which fields to include in a partial response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) Maximum number of results to return. Defaults to 100. Max is 1000.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) Page token, returned by a previous call, identifying the result set.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the ProjectID input for this Choreo.
	*(required, string) The ID of your Google API project.
	 */
	-(void)setProjectID:(NSString*)ProjectID {
		[super setInput:@"ProjectID" toValue:ProjectID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the StartIndex input for this Choreo.
	*(optional, integer) Zero-based index of the starting row to read.
	 */
	-(void)setStartIndex:(NSString*)StartIndex {
		[super setInput:@"StartIndex" toValue:StartIndex];
	}

	/*!
	 * Set the value of the TableID input for this Choreo.
	*(required, string) The ID of the BigQuery table to insert a row into.
	 */
	-(void)setTableID:(NSString*)TableID {
		[super setInput:@"TableID" toValue:TableID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the List Choreo.
 */
@implementation TMBGoogle_BigQuery_TableData_List_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves table data from a specified set of rows.
 */
@implementation TMBGoogle_BigQuery_TableData_List

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_BigQuery_TableData_List Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/BigQuery/TableData/List"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_BigQuery_TableData_List_ResultSet *results = [[TMBGoogle_BigQuery_TableData_List_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the List Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_BigQuery_TableData_List_Inputs*)newInputSet {
		return [[TMBGoogle_BigQuery_TableData_List_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CreateCalendar Choreo.
 */
@implementation TMBGoogle_Calendar_CreateCalendar_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Description input for this Choreo.
	*(optional, string) A description of the calendar.
	 */
	-(void)setDescription:(NSString*)Description {
		[super setInput:@"Description" toValue:Description];
	}

	/*!
	 * Set the value of the Location input for this Choreo.
	*(optional, string) Geographic location of the calendar such as "Los Angeles" or "New York".
	 */
	-(void)setLocation:(NSString*)Location {
		[super setInput:@"Location" toValue:Location];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Timezone input for this Choreo.
	*(optional, string) The timezone for the new calendar, such as "America/Los_Angeles". Defaults to UTC if left blank.
	 */
	-(void)setTimezone:(NSString*)Timezone {
		[super setInput:@"Timezone" toValue:Timezone];
	}

	/*!
	 * Set the value of the Title input for this Choreo.
	*(required, string) The name for the new calendar.
	 */
	-(void)setTitle:(NSString*)Title {
		[super setInput:@"Title" toValue:Title];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateCalendar Choreo.
 */
@implementation TMBGoogle_Calendar_CreateCalendar_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Google. Corresponds to the ResponseFormat input. Defaults to JSON.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Create a new secondary calendar.
 */
@implementation TMBGoogle_Calendar_CreateCalendar

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Calendar_CreateCalendar Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Calendar/CreateCalendar"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Calendar_CreateCalendar_ResultSet *results = [[TMBGoogle_Calendar_CreateCalendar_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CreateCalendar Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Calendar_CreateCalendar_Inputs*)newInputSet {
		return [[TMBGoogle_Calendar_CreateCalendar_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CreateEvent Choreo.
 */
@implementation TMBGoogle_Calendar_CreateEvent_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the CalendarID input for this Choreo.
	*(required, string) The unique ID for the calendar in which to add the event.  Note that calendar IDs can be retrieved by running GetAllCalendars or SearchCalendarsByName.
	 */
	-(void)setCalendarID:(NSString*)CalendarID {
		[super setInput:@"CalendarID" toValue:CalendarID];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the EndDate input for this Choreo.
	*(required, string) The end date of the event, in the format "2012-04-10".
	 */
	-(void)setEndDate:(NSString*)EndDate {
		[super setInput:@"EndDate" toValue:EndDate];
	}

	/*!
	 * Set the value of the EndTime input for this Choreo.
	*(required, string) The end time for the event, in the format "10:30:00".
	 */
	-(void)setEndTime:(NSString*)EndTime {
		[super setInput:@"EndTime" toValue:EndTime];
	}

	/*!
	 * Set the value of the EventDescription input for this Choreo.
	*(optional, string) A short description of the event.
	 */
	-(void)setEventDescription:(NSString*)EventDescription {
		[super setInput:@"EventDescription" toValue:EventDescription];
	}

	/*!
	 * Set the value of the EventLocation input for this Choreo.
	*(optional, string) The location for the new event.
	 */
	-(void)setEventLocation:(NSString*)EventLocation {
		[super setInput:@"EventLocation" toValue:EventLocation];
	}

	/*!
	 * Set the value of the EventTitle input for this Choreo.
	*(required, string) The title for the new event.
	 */
	-(void)setEventTitle:(NSString*)EventTitle {
		[super setInput:@"EventTitle" toValue:EventTitle];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the StartDate input for this Choreo.
	*(required, string) The start date of the event, in the format "2012-11-03".
	 */
	-(void)setStartDate:(NSString*)StartDate {
		[super setInput:@"StartDate" toValue:StartDate];
	}

	/*!
	 * Set the value of the StartTime input for this Choreo.
	*(required, string) The start time for the event, in the format "10:00:00".
	 */
	-(void)setStartTime:(NSString*)StartTime {
		[super setInput:@"StartTime" toValue:StartTime];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateEvent Choreo.
 */
@implementation TMBGoogle_Calendar_CreateEvent_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "TimezoneSetting" output from an execution of this Choreo.
	 * @return - NSString* (string) The timezone setting retrieved from the specified calendar.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getTimezoneSetting {
		return [super getOutputByName:@"TimezoneSetting"];
	}

	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Google. Corresponds to the ResponseFormat input. Defaults to JSON.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Create a new event in a specified calendar.
 */
@implementation TMBGoogle_Calendar_CreateEvent

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Calendar_CreateEvent Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Calendar/CreateEvent"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Calendar_CreateEvent_ResultSet *results = [[TMBGoogle_Calendar_CreateEvent_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CreateEvent Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Calendar_CreateEvent_Inputs*)newInputSet {
		return [[TMBGoogle_Calendar_CreateEvent_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteAllEvents Choreo.
 */
@implementation TMBGoogle_Calendar_DeleteAllEvents_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteAllEvents Choreo.
 */
@implementation TMBGoogle_Calendar_DeleteAllEvents_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* No content is returned for delete calendar operations.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Delete all events from a primary calendar of an account. Note that this operation can't be undone.
 */
@implementation TMBGoogle_Calendar_DeleteAllEvents

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Calendar_DeleteAllEvents Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Calendar/DeleteAllEvents"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Calendar_DeleteAllEvents_ResultSet *results = [[TMBGoogle_Calendar_DeleteAllEvents_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteAllEvents Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Calendar_DeleteAllEvents_Inputs*)newInputSet {
		return [[TMBGoogle_Calendar_DeleteAllEvents_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteCalendar Choreo.
 */
@implementation TMBGoogle_Calendar_DeleteCalendar_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the CalendarID input for this Choreo.
	*(required, string) The unique ID for the calendar to delete. Note that calendar IDs can be retrieved by running GetAllCalendars or SearchCalendarsByName.
	 */
	-(void)setCalendarID:(NSString*)CalendarID {
		[super setInput:@"CalendarID" toValue:CalendarID];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(required, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(required, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(required, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteCalendar Choreo.
 */
@implementation TMBGoogle_Calendar_DeleteCalendar_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* No content is returned for delete calendar operations.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Delete a secondary calendar.
 */
@implementation TMBGoogle_Calendar_DeleteCalendar

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Calendar_DeleteCalendar Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Calendar/DeleteCalendar"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Calendar_DeleteCalendar_ResultSet *results = [[TMBGoogle_Calendar_DeleteCalendar_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteCalendar Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Calendar_DeleteCalendar_Inputs*)newInputSet {
		return [[TMBGoogle_Calendar_DeleteCalendar_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteEvent Choreo.
 */
@implementation TMBGoogle_Calendar_DeleteEvent_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the CalendarID input for this Choreo.
	*(required, string) The unique ID for the calendar to delete. Note that calendar IDs can be retrieved by running GetAllCalendars or SearchCalendarsByName.
	 */
	-(void)setCalendarID:(NSString*)CalendarID {
		[super setInput:@"CalendarID" toValue:CalendarID];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The name of the calendar that you want to retrieve information for. Note that if there are multiple calendars with the same name, only the first one will be returned.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the EventID input for this Choreo.
	*(required, string) The unique ID for the event to delete.
	 */
	-(void)setEventID:(NSString*)EventID {
		[super setInput:@"EventID" toValue:EventID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteEvent Choreo.
 */
@implementation TMBGoogle_Calendar_DeleteEvent_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "AccessToken" output from an execution of this Choreo.
	 * @return - NSString* (optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getAccessToken {
		return [super getOutputByName:@"AccessToken"];
	}

	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* No content is returned for delete calendar operations.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Delete a specific event from a specified calendar.
 */
@implementation TMBGoogle_Calendar_DeleteEvent

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Calendar_DeleteEvent Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Calendar/DeleteEvent"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Calendar_DeleteEvent_ResultSet *results = [[TMBGoogle_Calendar_DeleteEvent_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteEvent Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Calendar_DeleteEvent_Inputs*)newInputSet {
		return [[TMBGoogle_Calendar_DeleteEvent_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetAllCalendars Choreo.
 */
@implementation TMBGoogle_Calendar_GetAllCalendars_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Count input for this Choreo.
	*(optional, integer) The maximum number of calendars to return. The default is 15.
	 */
	-(void)setCount:(NSString*)Count {
		[super setInput:@"Count" toValue:Count];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) The calendar fields to return. Accepted values are "etag", "items", "kind", and "nextPageToken". All calendar items are returned by default.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Hidden input for this Choreo.
	*(optional, boolean) Enter "1" to return hidden calendars in the results, or "0" (the default) to not return them.
	 */
	-(void)setHidden:(NSString*)Hidden {
		[super setInput:@"Hidden" toValue:Hidden];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetAllCalendars Choreo.
 */
@implementation TMBGoogle_Calendar_GetAllCalendars_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Google. Corresponds to the ResponseFormat input. Defaults to JSON.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieve data for all calendars.
 */
@implementation TMBGoogle_Calendar_GetAllCalendars

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Calendar_GetAllCalendars Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Calendar/GetAllCalendars"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Calendar_GetAllCalendars_ResultSet *results = [[TMBGoogle_Calendar_GetAllCalendars_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetAllCalendars Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Calendar_GetAllCalendars_Inputs*)newInputSet {
		return [[TMBGoogle_Calendar_GetAllCalendars_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetAllEvents Choreo.
 */
@implementation TMBGoogle_Calendar_GetAllEvents_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the CalendarID input for this Choreo.
	*(required, string) The unique ID for the calendar with the events to retrieve. Note that calendar IDs can be retrieved by running GetAllCalendars or SearchCalendarsByName.
	 */
	-(void)setCalendarID:(NSString*)CalendarID {
		[super setInput:@"CalendarID" toValue:CalendarID];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetAllEvents Choreo.
 */
@implementation TMBGoogle_Calendar_GetAllEvents_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Google. Corresponds to the ResponseFormat input. Defaults to JSON.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieve data for all events in a specified calendar.
 */
@implementation TMBGoogle_Calendar_GetAllEvents

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Calendar_GetAllEvents Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Calendar/GetAllEvents"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Calendar_GetAllEvents_ResultSet *results = [[TMBGoogle_Calendar_GetAllEvents_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetAllEvents Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Calendar_GetAllEvents_Inputs*)newInputSet {
		return [[TMBGoogle_Calendar_GetAllEvents_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetAllSettings Choreo.
 */
@implementation TMBGoogle_Calendar_GetAllSettings_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) The settings fields to return. Returns etag, item IDs, and item values by default.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetAllSettings Choreo.
 */
@implementation TMBGoogle_Calendar_GetAllSettings_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Google. Corresponds to the ResponseFormat input. Defaults to JSON.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves all settings for a Google Calendar account.
 */
@implementation TMBGoogle_Calendar_GetAllSettings

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Calendar_GetAllSettings Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Calendar/GetAllSettings"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Calendar_GetAllSettings_ResultSet *results = [[TMBGoogle_Calendar_GetAllSettings_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetAllSettings Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Calendar_GetAllSettings_Inputs*)newInputSet {
		return [[TMBGoogle_Calendar_GetAllSettings_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetCalendar Choreo.
 */
@implementation TMBGoogle_Calendar_GetCalendar_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the CalendarID input for this Choreo.
	*(required, string) The unique ID for the calendar  to retrieve. Note that calendar IDs can be retrieved by running GetAllCalendars or SearchCalendarsByName.
	 */
	-(void)setCalendarID:(NSString*)CalendarID {
		[super setInput:@"CalendarID" toValue:CalendarID];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetCalendar Choreo.
 */
@implementation TMBGoogle_Calendar_GetCalendar_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Google. Corresponds to the ResponseFormat input. Defaults to JSON.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieve data for a specific calendar in a Google account.
 */
@implementation TMBGoogle_Calendar_GetCalendar

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Calendar_GetCalendar Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Calendar/GetCalendar"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Calendar_GetCalendar_ResultSet *results = [[TMBGoogle_Calendar_GetCalendar_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetCalendar Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Calendar_GetCalendar_Inputs*)newInputSet {
		return [[TMBGoogle_Calendar_GetCalendar_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetEvent Choreo.
 */
@implementation TMBGoogle_Calendar_GetEvent_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the CalendarID input for this Choreo.
	*(required, string) The unique ID for the calendar with the event to retrieve. Note that calendar IDs can be retrieved by running GetAllCalendars or SearchCalendarsByName.
	 */
	-(void)setCalendarID:(NSString*)CalendarID {
		[super setInput:@"CalendarID" toValue:CalendarID];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the EventID input for this Choreo.
	*(required, string) The unique ID for the event to retrieve.
	 */
	-(void)setEventID:(NSString*)EventID {
		[super setInput:@"EventID" toValue:EventID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetEvent Choreo.
 */
@implementation TMBGoogle_Calendar_GetEvent_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Google. Corresponds to the ResponseFormat input. Defaults to JSON.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieve data for all calendars in an account.
 */
@implementation TMBGoogle_Calendar_GetEvent

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Calendar_GetEvent Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Calendar/GetEvent"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Calendar_GetEvent_ResultSet *results = [[TMBGoogle_Calendar_GetEvent_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetEvent Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Calendar_GetEvent_Inputs*)newInputSet {
		return [[TMBGoogle_Calendar_GetEvent_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetNextEvent Choreo.
 */
@implementation TMBGoogle_Calendar_GetNextEvent_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the CalendarID input for this Choreo.
	*(required, string) The unique ID for the calendar with the next event to retrieve. Note that calendar IDs can be retrieved by running GetAllCalendars or SearchCalendarsByName.
	 */
	-(void)setCalendarID:(NSString*)CalendarID {
		[super setInput:@"CalendarID" toValue:CalendarID];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Timezone input for this Choreo.
	*(optional, string) The time zone used in the response (i.e. America/Los_Angeles). The default is the time zone of the calendar.
	 */
	-(void)setTimezone:(NSString*)Timezone {
		[super setInput:@"Timezone" toValue:Timezone];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetNextEvent Choreo.
 */
@implementation TMBGoogle_Calendar_GetNextEvent_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "BeginTime" output from an execution of this Choreo.
	 * @return - NSString* (date) The start time of the event.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getBeginTime {
		return [super getOutputByName:@"BeginTime"];
	}

	/*!
	 * Retrieve the value of the "EndTime" output from an execution of this Choreo.
	 * @return - NSString* (date) The end time of the event.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getEndTime {
		return [super getOutputByName:@"EndTime"];
	}

	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Summary" output from an execution of this Choreo.
	 * @return - NSString* (string) The event summary.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getSummary {
		return [super getOutputByName:@"Summary"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Google. Corresponds to the ResponseFormat input. Defaults to JSON.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the next upcoming event in a Google calendar based on the current timestamp and the specified calendar's timezone setting.
 */
@implementation TMBGoogle_Calendar_GetNextEvent

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Calendar_GetNextEvent Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Calendar/GetNextEvent"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Calendar_GetNextEvent_ResultSet *results = [[TMBGoogle_Calendar_GetNextEvent_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetNextEvent Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Calendar_GetNextEvent_Inputs*)newInputSet {
		return [[TMBGoogle_Calendar_GetNextEvent_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetSetting Choreo.
 */
@implementation TMBGoogle_Calendar_GetSetting_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the SettingID input for this Choreo.
	*(required, string) The unique ID for the setting to retrieve. Use the GetAllSettings Choreo to retrieve settings IDs.
	 */
	-(void)setSettingID:(NSString*)SettingID {
		[super setInput:@"SettingID" toValue:SettingID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetSetting Choreo.
 */
@implementation TMBGoogle_Calendar_GetSetting_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Google. Corresponds to the ResponseFormat input. Defaults to JSON.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a specifed setting for a Google Calendar account.
 */
@implementation TMBGoogle_Calendar_GetSetting

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Calendar_GetSetting Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Calendar/GetSetting"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Calendar_GetSetting_ResultSet *results = [[TMBGoogle_Calendar_GetSetting_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetSetting Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Calendar_GetSetting_Inputs*)newInputSet {
		return [[TMBGoogle_Calendar_GetSetting_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SearchCalendarsByName Choreo.
 */
@implementation TMBGoogle_Calendar_SearchCalendarsByName_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the CalendarName input for this Choreo.
	*(required, string) The name of the calendar that you want to retrieve information for. Note that if there are multiple calendars with the same name, only the first one will be returned.
	 */
	-(void)setCalendarName:(NSString*)CalendarName {
		[super setInput:@"CalendarName" toValue:CalendarName];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Count input for this Choreo.
	*(optional, integer) The maximum number of calendars to search by name. The default is 15.
	 */
	-(void)setCount:(NSString*)Count {
		[super setInput:@"Count" toValue:Count];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SearchCalendarsByName Choreo.
 */
@implementation TMBGoogle_Calendar_SearchCalendarsByName_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "CalendarDescription" output from an execution of this Choreo.
	 * @return - NSString* (string) The calendar description parsed from the Google response.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getCalendarDescription {
		return [super getOutputByName:@"CalendarDescription"];
	}

	/*!
	 * Retrieve the value of the "CalendarId" output from an execution of this Choreo.
	 * @return - NSString* (string) The calendar id parsed from the Google response.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getCalendarId {
		return [super getOutputByName:@"CalendarId"];
	}

	/*!
	 * Retrieve the value of the "CalendarSummary" output from an execution of this Choreo.
	 * @return - NSString* (string) The summary or calendar name parsed from the Google response.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getCalendarSummary {
		return [super getOutputByName:@"CalendarSummary"];
	}

	/*!
	 * Retrieve the value of the "CalendarTimezone" output from an execution of this Choreo.
	 * @return - NSString* (string) The calendar timezone parsed from the Google response.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getCalendarTimezone {
		return [super getOutputByName:@"CalendarTimezone"];
	}

	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}
	
@end

/*!
 * Retrieves information about a calendar including the id with a given calendar name.
 */
@implementation TMBGoogle_Calendar_SearchCalendarsByName

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Calendar_SearchCalendarsByName Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Calendar/SearchCalendarsByName"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Calendar_SearchCalendarsByName_ResultSet *results = [[TMBGoogle_Calendar_SearchCalendarsByName_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SearchCalendarsByName Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Calendar_SearchCalendarsByName_Inputs*)newInputSet {
		return [[TMBGoogle_Calendar_SearchCalendarsByName_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SearchEvents Choreo.
 */
@implementation TMBGoogle_Calendar_SearchEvents_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the CalendarID input for this Choreo.
	*(required, string) The unique ID for the calendar with the events to search. Note that calendar IDs can be retrieved by running GetAllCalendars or SearchCalendarsByName.
	 */
	-(void)setCalendarID:(NSString*)CalendarID {
		[super setInput:@"CalendarID" toValue:CalendarID];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the LastModified input for this Choreo.
	*(optional, date) An event's last modification time (as a RFC 3339 timestamp) to filter by.
	 */
	-(void)setLastModified:(NSString*)LastModified {
		[super setInput:@"LastModified" toValue:LastModified];
	}

	/*!
	 * Set the value of the MaxAttendees input for this Choreo.
	*(optional, integer) The maximum number of attendees to include in the response. If there are more than the specified number of attendees, only the participant is returned.
	 */
	-(void)setMaxAttendees:(NSString*)MaxAttendees {
		[super setInput:@"MaxAttendees" toValue:MaxAttendees];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of events to return on one result page.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the MaxTime input for this Choreo.
	*(optional, date) The max start time to filter by (formatted like 2012-05-22T00:47:43.000Z).
	 */
	-(void)setMaxTime:(NSString*)MaxTime {
		[super setInput:@"MaxTime" toValue:MaxTime];
	}

	/*!
	 * Set the value of the MinTime input for this Choreo.
	*(optional, date) The minimum start time to filter by (formatted like 2012-05-22T00:47:43.000Z).
	 */
	-(void)setMinTime:(NSString*)MinTime {
		[super setInput:@"MinTime" toValue:MinTime];
	}

	/*!
	 * Set the value of the OrderBy input for this Choreo.
	*(optional, string) The order of the events returned in the result. Accepted values are: "startTime" (ordered by start date/time. Must set SingleEvents to 1 to use this) or "updated" (ordered by modification date/time).
	 */
	-(void)setOrderBy:(NSString*)OrderBy {
		[super setInput:@"OrderBy" toValue:OrderBy];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, integer) Indicates which result page to return. Used for paging through results.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the Query input for this Choreo.
	*(optional, string) A keyword search to find events.
	 */
	-(void)setQuery:(NSString*)Query {
		[super setInput:@"Query" toValue:Query];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the ShowDeleted input for this Choreo.
	*(optional, boolean) Whether to include deleted events. Set to 1 (true) to include deleted events. Defaults to 0 (false).
	 */
	-(void)setShowDeleted:(NSString*)ShowDeleted {
		[super setInput:@"ShowDeleted" toValue:ShowDeleted];
	}

	/*!
	 * Set the value of the ShowHiddenInvitations input for this Choreo.
	*(optional, boolean) Whether to include hidden invitations in the result. Set to 1 (true) to enable. The default is 0 (false).
	 */
	-(void)setShowHiddenInvitations:(NSString*)ShowHiddenInvitations {
		[super setInput:@"ShowHiddenInvitations" toValue:ShowHiddenInvitations];
	}

	/*!
	 * Set the value of the SingleEvent input for this Choreo.
	*(optional, boolean) Whether to expand recurring events into instances and only return single one-off events and instances of recurring events. Defaults to 0 (false).
	 */
	-(void)setSingleEvent:(NSString*)SingleEvent {
		[super setInput:@"SingleEvent" toValue:SingleEvent];
	}

	/*!
	 * Set the value of the Timezone input for this Choreo.
	*(optional, string) The time zone used in the response (i.e. America/Los_Angeles). The default is the time zone of the calendar.
	 */
	-(void)setTimezone:(NSString*)Timezone {
		[super setInput:@"Timezone" toValue:Timezone];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SearchEvents Choreo.
 */
@implementation TMBGoogle_Calendar_SearchEvents_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Google. Corresponds to the ResponseFormat input. Defaults to JSON.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Allows you to search for events using a variety of search parameters.
 */
@implementation TMBGoogle_Calendar_SearchEvents

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Calendar_SearchEvents Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Calendar/SearchEvents"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Calendar_SearchEvents_ResultSet *results = [[TMBGoogle_Calendar_SearchEvents_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SearchEvents Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Calendar_SearchEvents_Inputs*)newInputSet {
		return [[TMBGoogle_Calendar_SearchEvents_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UpdateCalendar Choreo.
 */
@implementation TMBGoogle_Calendar_UpdateCalendar_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the CalendarID input for this Choreo.
	*(required, string) The unique ID for the calendar to update. Note that calendar IDs can be retrieved by running GetAllCalendars or SearchCalendarsByName.
	 */
	-(void)setCalendarID:(NSString*)CalendarID {
		[super setInput:@"CalendarID" toValue:CalendarID];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the NewDescription input for this Choreo.
	*(optional, string) The new description for the calendar to update.
	 */
	-(void)setNewDescription:(NSString*)NewDescription {
		[super setInput:@"NewDescription" toValue:NewDescription];
	}

	/*!
	 * Set the value of the NewLocation input for this Choreo.
	*(optional, string) The new location for the calendar to update.
	 */
	-(void)setNewLocation:(NSString*)NewLocation {
		[super setInput:@"NewLocation" toValue:NewLocation];
	}

	/*!
	 * Set the value of the NewSummary input for this Choreo.
	*(required, string) The new summary for the calendar to update.
	 */
	-(void)setNewSummary:(NSString*)NewSummary {
		[super setInput:@"NewSummary" toValue:NewSummary];
	}

	/*!
	 * Set the value of the NewTimezone input for this Choreo.
	*(optional, string) The new timezone for the calendar to update.
	 */
	-(void)setNewTimezone:(NSString*)NewTimezone {
		[super setInput:@"NewTimezone" toValue:NewTimezone];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UpdateCalendar Choreo.
 */
@implementation TMBGoogle_Calendar_UpdateCalendar_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Google. Corresponds to the ResponseFormat input. Defaults to JSON.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Updates the metadata for a calendar.
 */
@implementation TMBGoogle_Calendar_UpdateCalendar

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Calendar_UpdateCalendar Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Calendar/UpdateCalendar"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Calendar_UpdateCalendar_ResultSet *results = [[TMBGoogle_Calendar_UpdateCalendar_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UpdateCalendar Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Calendar_UpdateCalendar_Inputs*)newInputSet {
		return [[TMBGoogle_Calendar_UpdateCalendar_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the AggregatedList Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Addresses_AggregatedList_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Filter input for this Choreo.
	*(optional, string) A filter expression for narrowing results in the form: {field_name} {comparison_string} {literal_string} (e.g. name eq your-address-name). Comparison strings can be eq (equals) or ne (not equals).
	 */
	-(void)setFilter:(NSString*)Filter {
		[super setInput:@"Filter" toValue:Filter];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of results to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) The "nextPageToken" found in the response which is used to page through results.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the AggregatedList Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Addresses_AggregatedList_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a list of Address resources across all available regions.
 */
@implementation TMBGoogle_ComputeEngine_Addresses_AggregatedList

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Addresses_AggregatedList Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Addresses/AggregatedList"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Addresses_AggregatedList_ResultSet *results = [[TMBGoogle_ComputeEngine_Addresses_AggregatedList_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the AggregatedList Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Addresses_AggregatedList_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Addresses_AggregatedList_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteAddress Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Addresses_DeleteAddress_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the Address input for this Choreo.
	*(required, string) The name of the address to delete.
	 */
	-(void)setAddress:(NSString*)Address {
		[super setInput:@"Address" toValue:Address];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Region input for this Choreo.
	*(required, string) The name of the region associated with this request.
	 */
	-(void)setRegion:(NSString*)Region {
		[super setInput:@"Region" toValue:Region];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteAddress Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Addresses_DeleteAddress_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Deletes a specified address resource.
 */
@implementation TMBGoogle_ComputeEngine_Addresses_DeleteAddress

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Addresses_DeleteAddress Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Addresses/DeleteAddress"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Addresses_DeleteAddress_ResultSet *results = [[TMBGoogle_ComputeEngine_Addresses_DeleteAddress_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteAddress Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Addresses_DeleteAddress_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Addresses_DeleteAddress_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetAddress Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Addresses_GetAddress_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the Address input for this Choreo.
	*(required, string) The name of the address to retrieve.
	 */
	-(void)setAddress:(NSString*)Address {
		[super setInput:@"Address" toValue:Address];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Region input for this Choreo.
	*(required, string) The name of the region associated with this request.
	 */
	-(void)setRegion:(NSString*)Region {
		[super setInput:@"Region" toValue:Region];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetAddress Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Addresses_GetAddress_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves information about the specified Address.
 */
@implementation TMBGoogle_ComputeEngine_Addresses_GetAddress

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Addresses_GetAddress Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Addresses/GetAddress"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Addresses_GetAddress_ResultSet *results = [[TMBGoogle_ComputeEngine_Addresses_GetAddress_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetAddress Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Addresses_GetAddress_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Addresses_GetAddress_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the InsertAddress Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Addresses_InsertAddress_Inputs

	/*!
	 * Set the value of the AddressResource input for this Choreo.
	*(optional, json) A JSON string containing the address resource properties you wish to set. This can be used as an alternative to individual inputs that represent address resource properties.
	 */
	-(void)setAddressResource:(NSString*)AddressResource {
		[super setInput:@"AddressResource" toValue:AddressResource];
	}

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the Address input for this Choreo.
	*(optional, string) The IP address represented by this resource.
	 */
	-(void)setAddress:(NSString*)Address {
		[super setInput:@"Address" toValue:Address];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Description input for this Choreo.
	*(optional, string) A description of the address.
	 */
	-(void)setDescription:(NSString*)Description {
		[super setInput:@"Description" toValue:Description];
	}

	/*!
	 * Set the value of the Name input for this Choreo.
	*(conditional, string) The name of the address resource.
	 */
	-(void)setName:(NSString*)Name {
		[super setInput:@"Name" toValue:Name];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Region input for this Choreo.
	*(required, string) Name of the region associated with the request.
	 */
	-(void)setRegion:(NSString*)Region {
		[super setInput:@"Region" toValue:Region];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the InsertAddress Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Addresses_InsertAddress_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Creates an Address resource in the specified project.
 */
@implementation TMBGoogle_ComputeEngine_Addresses_InsertAddress

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Addresses_InsertAddress Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Addresses/InsertAddress"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Addresses_InsertAddress_ResultSet *results = [[TMBGoogle_ComputeEngine_Addresses_InsertAddress_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the InsertAddress Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Addresses_InsertAddress_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Addresses_InsertAddress_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListAddresses Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Addresses_ListAddresses_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Filter input for this Choreo.
	*(optional, string) A filter expression for narrowing results in the form: {field_name} {comparison_string} {literal_string} (e.g. name eq your-address-name). Comparison strings can be eq (equals) or ne (not equals).
	 */
	-(void)setFilter:(NSString*)Filter {
		[super setInput:@"Filter" toValue:Filter];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of results to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) The "nextPageToken" found in the response which is used to page through results.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Region input for this Choreo.
	*(required, string) The name of the region associated with this request.
	 */
	-(void)setRegion:(NSString*)Region {
		[super setInput:@"Region" toValue:Region];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListAddresses Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Addresses_ListAddresses_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a list of Address resources contained within the specified region
 */
@implementation TMBGoogle_ComputeEngine_Addresses_ListAddresses

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Addresses_ListAddresses Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Addresses/ListAddresses"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Addresses_ListAddresses_ResultSet *results = [[TMBGoogle_ComputeEngine_Addresses_ListAddresses_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListAddresses Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Addresses_ListAddresses_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Addresses_ListAddresses_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the AggregatedList Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Disks_AggregatedList_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Filter input for this Choreo.
	*(optional, string) A filter expression for narrowing results in the form: {field_name} {comparison_string} {literal_string} (e.g. name eq your-disk-name). Comparison strings can be eq (equals) or ne (not equals).
	 */
	-(void)setFilter:(NSString*)Filter {
		[super setInput:@"Filter" toValue:Filter];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of results to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) The "nextPageToken" found in the response which is used to page through results.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the AggregatedList Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Disks_AggregatedList_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves an aggregated list of Disk resources across all zones.
 */
@implementation TMBGoogle_ComputeEngine_Disks_AggregatedList

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Disks_AggregatedList Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Disks/AggregatedList"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Disks_AggregatedList_ResultSet *results = [[TMBGoogle_ComputeEngine_Disks_AggregatedList_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the AggregatedList Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Disks_AggregatedList_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Disks_AggregatedList_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CreateSnapshot Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Disks_CreateSnapshot_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Description input for this Choreo.
	*(optional, string) A description for the snapshot resource.
	 */
	-(void)setDescription:(NSString*)Description {
		[super setInput:@"Description" toValue:Description];
	}

	/*!
	 * Set the value of the Disk input for this Choreo.
	*(required, string) The name of the persistent disk resource to use to create this snapshot.
	 */
	-(void)setDisk:(NSString*)Disk {
		[super setInput:@"Disk" toValue:Disk];
	}

	/*!
	 * Set the value of the Name input for this Choreo.
	*(required, string) The name of the snapshot resource being created.
	 */
	-(void)setName:(NSString*)Name {
		[super setInput:@"Name" toValue:Name];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the SourceDiskID input for this Choreo.
	*(required, string) The ID of the disk being used to create the snapshot.
	 */
	-(void)setSourceDiskID:(NSString*)SourceDiskID {
		[super setInput:@"SourceDiskID" toValue:SourceDiskID];
	}

	/*!
	 * Set the value of the StorageBytes input for this Choreo.
	*(optional, integer) The size of the storage used by the snapshot.
	 */
	-(void)setStorageBytes:(NSString*)StorageBytes {
		[super setInput:@"StorageBytes" toValue:StorageBytes];
	}

	/*!
	 * Set the value of the StorageBytesStatus input for this Choreo.
	*(optional, string) Indicates whether storageBytes is in a stable state, or it is being adjusted as a result of shared storage reallocation. Valid values: are "UPDATING" AND "UP_TO_DATE".
	 */
	-(void)setStorageBytesStatus:(NSString*)StorageBytesStatus {
		[super setInput:@"StorageBytesStatus" toValue:StorageBytesStatus];
	}

	/*!
	 * Set the value of the Zone input for this Choreo.
	*(required, string) The name of the zone associated with this request.
	 */
	-(void)setZone:(NSString*)Zone {
		[super setInput:@"Zone" toValue:Zone];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateSnapshot Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Disks_CreateSnapshot_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Creates a snapshot of a specified disk.
 */
@implementation TMBGoogle_ComputeEngine_Disks_CreateSnapshot

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Disks_CreateSnapshot Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Disks/CreateSnapshot"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Disks_CreateSnapshot_ResultSet *results = [[TMBGoogle_ComputeEngine_Disks_CreateSnapshot_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CreateSnapshot Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Disks_CreateSnapshot_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Disks_CreateSnapshot_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteDisk Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Disks_DeleteDisk_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Disk input for this Choreo.
	*(required, string) The name of the persistent disk resource to delete.
	 */
	-(void)setDisk:(NSString*)Disk {
		[super setInput:@"Disk" toValue:Disk];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Zone input for this Choreo.
	*(required, string) The name of the zone associated with this request.
	 */
	-(void)setZone:(NSString*)Zone {
		[super setInput:@"Zone" toValue:Zone];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteDisk Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Disks_DeleteDisk_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Deletes the specified Persistent Disk resource.
 */
@implementation TMBGoogle_ComputeEngine_Disks_DeleteDisk

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Disks_DeleteDisk Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Disks/DeleteDisk"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Disks_DeleteDisk_ResultSet *results = [[TMBGoogle_ComputeEngine_Disks_DeleteDisk_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteDisk Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Disks_DeleteDisk_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Disks_DeleteDisk_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetDisk Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Disks_GetDisk_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Disk input for this Choreo.
	*(required, string) The name of the persistent disk resource to retrieve.
	 */
	-(void)setDisk:(NSString*)Disk {
		[super setInput:@"Disk" toValue:Disk];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Zone input for this Choreo.
	*(required, string) The name of the zone associated with this request.
	 */
	-(void)setZone:(NSString*)Zone {
		[super setInput:@"Zone" toValue:Zone];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetDisk Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Disks_GetDisk_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves information about the specified Persistent Disk.
 */
@implementation TMBGoogle_ComputeEngine_Disks_GetDisk

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Disks_GetDisk Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Disks/GetDisk"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Disks_GetDisk_ResultSet *results = [[TMBGoogle_ComputeEngine_Disks_GetDisk_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetDisk Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Disks_GetDisk_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Disks_GetDisk_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the InsertDisk Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Disks_InsertDisk_Inputs

	/*!
	 * Set the value of the DiskResource input for this Choreo.
	*(optional, json) A JSON string containing the disk resource properties you wish to set. This can be used as an alternative to individual inputs that represent disk properties.
	 */
	-(void)setDiskResource:(NSString*)DiskResource {
		[super setInput:@"DiskResource" toValue:DiskResource];
	}

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Name input for this Choreo.
	*(required, string) The name of the persistent disk resource being created.
	 */
	-(void)setName:(NSString*)Name {
		[super setInput:@"Name" toValue:Name];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the SizeGB input for this Choreo.
	*(conditional, string) Size of the persistent disk, specified in GB. This is optional when using a SourceSnapshot or SourceImage, otherwise it is required.
	 */
	-(void)setSizeGB:(NSString*)SizeGB {
		[super setInput:@"SizeGB" toValue:SizeGB];
	}

	/*!
	 * Set the value of the SourceImage input for this Choreo.
	*(conditional, string) The URL for the source image to apply to the disk. This is required if SizeGB or SourceSnapshot is not provided.
	 */
	-(void)setSourceImage:(NSString*)SourceImage {
		[super setInput:@"SourceImage" toValue:SourceImage];
	}

	/*!
	 * Set the value of the SourceSnapshot input for this Choreo.
	*(conditional, string) The source snapshot used to create this disk. This is required if SizeGB and SourceImage are not specified.
	 */
	-(void)setSourceSnapshot:(NSString*)SourceSnapshot {
		[super setInput:@"SourceSnapshot" toValue:SourceSnapshot];
	}

	/*!
	 * Set the value of the Zone input for this Choreo.
	*(required, string) The name of the zone associated with this request.
	 */
	-(void)setZone:(NSString*)Zone {
		[super setInput:@"Zone" toValue:Zone];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the InsertDisk Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Disks_InsertDisk_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Creates a Persistent Disk resource in the specified project.
 */
@implementation TMBGoogle_ComputeEngine_Disks_InsertDisk

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Disks_InsertDisk Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Disks/InsertDisk"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Disks_InsertDisk_ResultSet *results = [[TMBGoogle_ComputeEngine_Disks_InsertDisk_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the InsertDisk Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Disks_InsertDisk_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Disks_InsertDisk_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListDisks Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Disks_ListDisks_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Filter input for this Choreo.
	*(optional, string) A filter expression for narrowing results in the form: {field_name} {comparison_string} {literal_string} (e.g. name eq europe-west1). Comparison strings can be eq (equals) or ne (not equals).
	 */
	-(void)setFilter:(NSString*)Filter {
		[super setInput:@"Filter" toValue:Filter];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of results to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) The "nextPageToken" found in the response which is used to page through results.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Zone input for this Choreo.
	*(required, string) The name of the zone associated with this request.
	 */
	-(void)setZone:(NSString*)Zone {
		[super setInput:@"Zone" toValue:Zone];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListDisks Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Disks_ListDisks_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the list of Disk resources available to the specified project.
 */
@implementation TMBGoogle_ComputeEngine_Disks_ListDisks

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Disks_ListDisks Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Disks/ListDisks"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Disks_ListDisks_ResultSet *results = [[TMBGoogle_ComputeEngine_Disks_ListDisks_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListDisks Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Disks_ListDisks_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Disks_ListDisks_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteFirewall Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Firewalls_DeleteFirewall_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Firewall input for this Choreo.
	*(required, string) The name of the firewall to delete.
	 */
	-(void)setFirewall:(NSString*)Firewall {
		[super setInput:@"Firewall" toValue:Firewall];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteFirewall Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Firewalls_DeleteFirewall_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Deletes the specified Firewall resource.
 */
@implementation TMBGoogle_ComputeEngine_Firewalls_DeleteFirewall

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Firewalls_DeleteFirewall Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Firewalls/DeleteFirewall"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Firewalls_DeleteFirewall_ResultSet *results = [[TMBGoogle_ComputeEngine_Firewalls_DeleteFirewall_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteFirewall Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Firewalls_DeleteFirewall_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Firewalls_DeleteFirewall_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetFirewall Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Firewalls_GetFirewall_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Firewall input for this Choreo.
	*(required, string) The name of the firewall to retrieve.
	 */
	-(void)setFirewall:(NSString*)Firewall {
		[super setInput:@"Firewall" toValue:Firewall];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetFirewall Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Firewalls_GetFirewall_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves information about the specified Firewall.
 */
@implementation TMBGoogle_ComputeEngine_Firewalls_GetFirewall

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Firewalls_GetFirewall Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Firewalls/GetFirewall"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Firewalls_GetFirewall_ResultSet *results = [[TMBGoogle_ComputeEngine_Firewalls_GetFirewall_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetFirewall Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Firewalls_GetFirewall_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Firewalls_GetFirewall_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the InsertFirewall Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Firewalls_InsertFirewall_Inputs

	/*!
	 * Set the value of the FirewallResource input for this Choreo.
	*(optional, json) A JSON string containing the firewall resource properties to set. This can be used as an alternative to individual inputs representing firewall properties.
	 */
	-(void)setFirewallResource:(NSString*)FirewallResource {
		[super setInput:@"FirewallResource" toValue:FirewallResource];
	}

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AllowedIPProtocol input for this Choreo.
	*(conditional, json) The IP protocol that is allowed for this rule. This is an array and can have the following properties: IPProtocol (valid values are: tcp, udp, or icmp) and ports[].
	 */
	-(void)setAllowedIPProtocol:(NSString*)AllowedIPProtocol {
		[super setInput:@"AllowedIPProtocol" toValue:AllowedIPProtocol];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Description input for this Choreo.
	*(optional, string) A description of the firewall.
	 */
	-(void)setDescription:(NSString*)Description {
		[super setInput:@"Description" toValue:Description];
	}

	/*!
	 * Set the value of the Name input for this Choreo.
	*(conditional, string) The name of the firewall resource being created.
	 */
	-(void)setName:(NSString*)Name {
		[super setInput:@"Name" toValue:Name];
	}

	/*!
	 * Set the value of the Network input for this Choreo.
	*(conditional, string) The fully-qualified URL of the network to which this firewall is applied.
	 */
	-(void)setNetwork:(NSString*)Network {
		[super setInput:@"Network" toValue:Network];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the SourceRanges input for this Choreo.
	*(conditional, json) An array of address blocks that this rule applies to. This is required if the SourceTags input is not provided.
	 */
	-(void)setSourceRanges:(NSString*)SourceRanges {
		[super setInput:@"SourceRanges" toValue:SourceRanges];
	}

	/*!
	 * Set the value of the SourceTags input for this Choreo.
	*(conditional, json) An array of instance tags which this rule applies to. This is required unless the SourceRanges input is provided.
	 */
	-(void)setSourceTags:(NSString*)SourceTags {
		[super setInput:@"SourceTags" toValue:SourceTags];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the InsertFirewall Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Firewalls_InsertFirewall_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Creates a Firewall resource in the specified project.
 */
@implementation TMBGoogle_ComputeEngine_Firewalls_InsertFirewall

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Firewalls_InsertFirewall Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Firewalls/InsertFirewall"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Firewalls_InsertFirewall_ResultSet *results = [[TMBGoogle_ComputeEngine_Firewalls_InsertFirewall_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the InsertFirewall Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Firewalls_InsertFirewall_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Firewalls_InsertFirewall_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListFirewalls Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Firewalls_ListFirewalls_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Filter input for this Choreo.
	*(optional, string) A filter expression for narrowing results in the form: {field_name} {comparison_string} {literal_string} (e.g. name eq default-ssh). Comparison strings can be eq (equals) or ne (not equals).
	 */
	-(void)setFilter:(NSString*)Filter {
		[super setInput:@"Filter" toValue:Filter];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of results to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) The "nextPageToken" found in the response which is used to page through results.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListFirewalls Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Firewalls_ListFirewalls_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the list of Firewall resources for the specified project.
 */
@implementation TMBGoogle_ComputeEngine_Firewalls_ListFirewalls

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Firewalls_ListFirewalls Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Firewalls/ListFirewalls"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Firewalls_ListFirewalls_ResultSet *results = [[TMBGoogle_ComputeEngine_Firewalls_ListFirewalls_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListFirewalls Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Firewalls_ListFirewalls_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Firewalls_ListFirewalls_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UpdateFirewall Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Firewalls_UpdateFirewall_Inputs

	/*!
	 * Set the value of the FirewallResource input for this Choreo.
	*(required, json) A JSON string containing the firewall resource fields to set. Required properties include: allowed, name, network, and either sourceRanges or sourceTags.
	 */
	-(void)setFirewallResource:(NSString*)FirewallResource {
		[super setInput:@"FirewallResource" toValue:FirewallResource];
	}

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Firewall input for this Choreo.
	*(required, string) The name of the firewall to update.
	 */
	-(void)setFirewall:(NSString*)Firewall {
		[super setInput:@"Firewall" toValue:Firewall];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UpdateFirewall Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Firewalls_UpdateFirewall_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Updates an existing Firewall resource.
 */
@implementation TMBGoogle_ComputeEngine_Firewalls_UpdateFirewall

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Firewalls_UpdateFirewall Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Firewalls/UpdateFirewall"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Firewalls_UpdateFirewall_ResultSet *results = [[TMBGoogle_ComputeEngine_Firewalls_UpdateFirewall_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UpdateFirewall Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Firewalls_UpdateFirewall_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Firewalls_UpdateFirewall_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the AggregatedList Choreo.
 */
@implementation TMBGoogle_ComputeEngine_GlobalOperations_AggregatedList_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Filter input for this Choreo.
	*(optional, string) A filter expression for narrowing results in the form: {field_name} {comparison_string} {literal_string} (e.g. name eq operation-1234). Comparison strings can be eq (equals) or ne (not equals).
	 */
	-(void)setFilter:(NSString*)Filter {
		[super setInput:@"Filter" toValue:Filter];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of results to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) The "nextPageToken" found in the response which is used to page through results.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the AggregatedList Choreo.
 */
@implementation TMBGoogle_ComputeEngine_GlobalOperations_AggregatedList_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a list of all Global Operation resources grouped by scope.
 */
@implementation TMBGoogle_ComputeEngine_GlobalOperations_AggregatedList

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_GlobalOperations_AggregatedList Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/GlobalOperations/AggregatedList"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_GlobalOperations_AggregatedList_ResultSet *results = [[TMBGoogle_ComputeEngine_GlobalOperations_AggregatedList_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the AggregatedList Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_GlobalOperations_AggregatedList_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_GlobalOperations_AggregatedList_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteGlobalOperation Choreo.
 */
@implementation TMBGoogle_ComputeEngine_GlobalOperations_DeleteGlobalOperation_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Operation input for this Choreo.
	*(required, string) The name of the operation to delete.
	 */
	-(void)setOperation:(NSString*)Operation {
		[super setInput:@"Operation" toValue:Operation];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteGlobalOperation Choreo.
 */
@implementation TMBGoogle_ComputeEngine_GlobalOperations_DeleteGlobalOperation_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "ResponseStatusCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) The response status code returned from Google. A 204 is expected for a successful delete operation.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponseStatusCode {
		return [super getOutputByName:@"ResponseStatusCode"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Deletes the specified Global Operation.
 */
@implementation TMBGoogle_ComputeEngine_GlobalOperations_DeleteGlobalOperation

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_GlobalOperations_DeleteGlobalOperation Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/GlobalOperations/DeleteGlobalOperation"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_GlobalOperations_DeleteGlobalOperation_ResultSet *results = [[TMBGoogle_ComputeEngine_GlobalOperations_DeleteGlobalOperation_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteGlobalOperation Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_GlobalOperations_DeleteGlobalOperation_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_GlobalOperations_DeleteGlobalOperation_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetGlobalOperation Choreo.
 */
@implementation TMBGoogle_ComputeEngine_GlobalOperations_GetGlobalOperation_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Operation input for this Choreo.
	*(required, string) The name of the operation to retrieve.
	 */
	-(void)setOperation:(NSString*)Operation {
		[super setInput:@"Operation" toValue:Operation];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetGlobalOperation Choreo.
 */
@implementation TMBGoogle_ComputeEngine_GlobalOperations_GetGlobalOperation_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves information about the specified Global Operation.
 */
@implementation TMBGoogle_ComputeEngine_GlobalOperations_GetGlobalOperation

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_GlobalOperations_GetGlobalOperation Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/GlobalOperations/GetGlobalOperation"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_GlobalOperations_GetGlobalOperation_ResultSet *results = [[TMBGoogle_ComputeEngine_GlobalOperations_GetGlobalOperation_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetGlobalOperation Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_GlobalOperations_GetGlobalOperation_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_GlobalOperations_GetGlobalOperation_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListGlobalOperations Choreo.
 */
@implementation TMBGoogle_ComputeEngine_GlobalOperations_ListGlobalOperations_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Filter input for this Choreo.
	*(optional, string) A filter expression for narrowing results in the form: {field_name} {comparison_string} {literal_string} (e.g. name eq operation-1234). Comparison strings can be eq (equals) or ne (not equals).
	 */
	-(void)setFilter:(NSString*)Filter {
		[super setInput:@"Filter" toValue:Filter];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of results to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) The "nextPageToken" found in the response which is used to page through results.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListGlobalOperations Choreo.
 */
@implementation TMBGoogle_ComputeEngine_GlobalOperations_ListGlobalOperations_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a list of GlobalOperations resources contained within the specified project.
 */
@implementation TMBGoogle_ComputeEngine_GlobalOperations_ListGlobalOperations

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_GlobalOperations_ListGlobalOperations Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/GlobalOperations/ListGlobalOperations"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_GlobalOperations_ListGlobalOperations_ResultSet *results = [[TMBGoogle_ComputeEngine_GlobalOperations_ListGlobalOperations_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListGlobalOperations Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_GlobalOperations_ListGlobalOperations_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_GlobalOperations_ListGlobalOperations_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteImage Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Images_DeleteImage_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Image input for this Choreo.
	*(required, string) The name of the Image to delete.
	 */
	-(void)setImage:(NSString*)Image {
		[super setInput:@"Image" toValue:Image];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteImage Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Images_DeleteImage_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Deletes the specified Image resource.
 */
@implementation TMBGoogle_ComputeEngine_Images_DeleteImage

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Images_DeleteImage Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Images/DeleteImage"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Images_DeleteImage_ResultSet *results = [[TMBGoogle_ComputeEngine_Images_DeleteImage_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteImage Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Images_DeleteImage_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Images_DeleteImage_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetImage Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Images_GetImage_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Image input for this Choreo.
	*(required, string) The name of the Image to retrieve.
	 */
	-(void)setImage:(NSString*)Image {
		[super setInput:@"Image" toValue:Image];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetImage Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Images_GetImage_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves information about the specified Image.
 */
@implementation TMBGoogle_ComputeEngine_Images_GetImage

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Images_GetImage Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Images/GetImage"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Images_GetImage_ResultSet *results = [[TMBGoogle_ComputeEngine_Images_GetImage_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetImage Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Images_GetImage_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Images_GetImage_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the InsertImage Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Images_InsertImage_Inputs

	/*!
	 * Set the value of the ImageResource input for this Choreo.
	*(optional, json) A JSON string containing the image resource properties to set. This can used as an alternative to individual inputs representing image properties.
	 */
	-(void)setImageResource:(NSString*)ImageResource {
		[super setInput:@"ImageResource" toValue:ImageResource];
	}

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Description input for this Choreo.
	*(optional, string) A description of the image.
	 */
	-(void)setDescription:(NSString*)Description {
		[super setInput:@"Description" toValue:Description];
	}

	/*!
	 * Set the value of the Name input for this Choreo.
	*(conditional, string) The name of the image.
	 */
	-(void)setName:(NSString*)Name {
		[super setInput:@"Name" toValue:Name];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RawDiskSource input for this Choreo.
	*(conditional, string) The full Google Cloud Storage URL where the disk image is stored.
	 */
	-(void)setRawDiskSource:(NSString*)RawDiskSource {
		[super setInput:@"RawDiskSource" toValue:RawDiskSource];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the InsertImage Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Images_InsertImage_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Creates a new custom Image resource in the specified project.
 */
@implementation TMBGoogle_ComputeEngine_Images_InsertImage

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Images_InsertImage Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Images/InsertImage"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Images_InsertImage_ResultSet *results = [[TMBGoogle_ComputeEngine_Images_InsertImage_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the InsertImage Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Images_InsertImage_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Images_InsertImage_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListImages Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Images_ListImages_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Filter input for this Choreo.
	*(optional, string) A filter expression for narrowing results in the form: {field_name} {comparison_string} {literal_string} (e.g. name eq testimage). Comparison strings can be eq (equals) or ne (not equals).
	 */
	-(void)setFilter:(NSString*)Filter {
		[super setInput:@"Filter" toValue:Filter];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of results to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) The "nextPageToken" found in the response which is used to page through results.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListImages Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Images_ListImages_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a list of Image resources for the specified project.
 */
@implementation TMBGoogle_ComputeEngine_Images_ListImages

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Images_ListImages Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Images/ListImages"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Images_ListImages_ResultSet *results = [[TMBGoogle_ComputeEngine_Images_ListImages_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListImages Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Images_ListImages_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Images_ListImages_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the AddAccessConfig Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Instances_AddAccessConfig_Inputs

	/*!
	 * Set the value of the AccessConfiguration input for this Choreo.
	*(optional, json) A JSON string containing the access configuration properties you wish to set. This can be used as an alternative to individual inputs that represent access configuration properties.
	 */
	-(void)setAccessConfiguration:(NSString*)AccessConfiguration {
		[super setInput:@"AccessConfiguration" toValue:AccessConfiguration];
	}

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Instance input for this Choreo.
	*(required, string) Name of the instance for which to add an access configuration.
	 */
	-(void)setInstance:(NSString*)Instance {
		[super setInput:@"Instance" toValue:Instance];
	}

	/*!
	 * Set the value of the Name input for this Choreo.
	*(optional, string) The name of this access configuration. Defaults to "External NAT" if not specified.
	 */
	-(void)setName:(NSString*)Name {
		[super setInput:@"Name" toValue:Name];
	}

	/*!
	 * Set the value of the NatIP input for this Choreo.
	*(optional, string) An external IP address associated with this instance. Specify an unused static IP address available to the project. An external IP will be drawn from a shared ephemeral pool when not specified.
	 */
	-(void)setNatIP:(NSString*)NatIP {
		[super setInput:@"NatIP" toValue:NatIP];
	}

	/*!
	 * Set the value of the NetworkInterface input for this Choreo.
	*(required, string) The name of the network interface to add the access config (e.g. nic0, nic1, etc).
	 */
	-(void)setNetworkInterface:(NSString*)NetworkInterface {
		[super setInput:@"NetworkInterface" toValue:NetworkInterface];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Type input for this Choreo.
	*(required, string) Type of configuration. Must be set to ONE_TO_ONE_NAT.
	 */
	-(void)setType:(NSString*)Type {
		[super setInput:@"Type" toValue:Type];
	}

	/*!
	 * Set the value of the Zone input for this Choreo.
	*(required, string) The name of the zone associated with this request.
	 */
	-(void)setZone:(NSString*)Zone {
		[super setInput:@"Zone" toValue:Zone];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the AddAccessConfig Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Instances_AddAccessConfig_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Adds an access config to an instance's network interface.
 */
@implementation TMBGoogle_ComputeEngine_Instances_AddAccessConfig

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Instances_AddAccessConfig Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Instances/AddAccessConfig"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Instances_AddAccessConfig_ResultSet *results = [[TMBGoogle_ComputeEngine_Instances_AddAccessConfig_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the AddAccessConfig Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Instances_AddAccessConfig_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Instances_AddAccessConfig_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the AggregatedList Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Instances_AggregatedList_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Filter input for this Choreo.
	*(optional, string) A filter expression for narrowing results in the form: {field_name} {comparison_string} {literal_string} (e.g. name eq my_instance). Comparison strings can be eq (equals) or ne (not equals).
	 */
	-(void)setFilter:(NSString*)Filter {
		[super setInput:@"Filter" toValue:Filter];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of results to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) The "nextPageToken" found in the response which is used to page through results.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the AggregatedList Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Instances_AggregatedList_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a list of Instance resources across all available zones.
 */
@implementation TMBGoogle_ComputeEngine_Instances_AggregatedList

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Instances_AggregatedList Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Instances/AggregatedList"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Instances_AggregatedList_ResultSet *results = [[TMBGoogle_ComputeEngine_Instances_AggregatedList_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the AggregatedList Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Instances_AggregatedList_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Instances_AggregatedList_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the AttachDisk Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Instances_AttachDisk_Inputs

	/*!
	 * Set the value of the AttachedDisk input for this Choreo.
	*(optional, json) A JSON string containing the attached disk properties to set. This can be used as an alternative to the individual inputs representing the attached disk properties.
	 */
	-(void)setAttachedDisk:(NSString*)AttachedDisk {
		[super setInput:@"AttachedDisk" toValue:AttachedDisk];
	}

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the Boot input for this Choreo.
	*(conditional, boolean) Whether or not this is a boot disk. Valid values are: true or false.
	 */
	-(void)setBoot:(NSString*)Boot {
		[super setInput:@"Boot" toValue:Boot];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the DeviceName input for this Choreo.
	*(conditional, string) The name of the disk to attach.
	 */
	-(void)setDeviceName:(NSString*)DeviceName {
		[super setInput:@"DeviceName" toValue:DeviceName];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Instance input for this Choreo.
	*(required, string) The name of the instance to attach a disk resource to.
	 */
	-(void)setInstance:(NSString*)Instance {
		[super setInput:@"Instance" toValue:Instance];
	}

	/*!
	 * Set the value of the Mode input for this Choreo.
	*(conditional, string) The mode in which to attach the disk. Valid values are: READ_WRITE or READ_ONLY.
	 */
	-(void)setMode:(NSString*)Mode {
		[super setInput:@"Mode" toValue:Mode];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Source input for this Choreo.
	*(conditional, string) The URL to the Disk resource.
	 */
	-(void)setSource:(NSString*)Source {
		[super setInput:@"Source" toValue:Source];
	}

	/*!
	 * Set the value of the Type input for this Choreo.
	*(conditional, string) The type of disk. Valid values are: SCRATCH or PERSISTENT. Persistent disks must already exist before you can attach them.
	 */
	-(void)setType:(NSString*)Type {
		[super setInput:@"Type" toValue:Type];
	}

	/*!
	 * Set the value of the Zone input for this Choreo.
	*(required, string) The name of the zone associated with this request.
	 */
	-(void)setZone:(NSString*)Zone {
		[super setInput:@"Zone" toValue:Zone];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the AttachDisk Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Instances_AttachDisk_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Attaches a Disk resource to an instance.
 */
@implementation TMBGoogle_ComputeEngine_Instances_AttachDisk

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Instances_AttachDisk Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Instances/AttachDisk"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Instances_AttachDisk_ResultSet *results = [[TMBGoogle_ComputeEngine_Instances_AttachDisk_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the AttachDisk Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Instances_AttachDisk_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Instances_AttachDisk_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteAccessConfig Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Instances_DeleteAccessConfig_Inputs

	/*!
	 * Set the value of the AccessConfig input for this Choreo.
	*(required, string) The name of the access configuration to delete.
	 */
	-(void)setAccessConfig:(NSString*)AccessConfig {
		[super setInput:@"AccessConfig" toValue:AccessConfig];
	}

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Instance input for this Choreo.
	*(required, string) The name of the instance that this access configuration applies to.
	 */
	-(void)setInstance:(NSString*)Instance {
		[super setInput:@"Instance" toValue:Instance];
	}

	/*!
	 * Set the value of the NetworkInterface input for this Choreo.
	*(required, string) The name of the network interface associated with the access config that is being deleted.
	 */
	-(void)setNetworkInterface:(NSString*)NetworkInterface {
		[super setInput:@"NetworkInterface" toValue:NetworkInterface];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Zone input for this Choreo.
	*(required, string) The name of the zone associated with this request.
	 */
	-(void)setZone:(NSString*)Zone {
		[super setInput:@"Zone" toValue:Zone];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteAccessConfig Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Instances_DeleteAccessConfig_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Deletes an access configuation from an instance's network interface.
 */
@implementation TMBGoogle_ComputeEngine_Instances_DeleteAccessConfig

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Instances_DeleteAccessConfig Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Instances/DeleteAccessConfig"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Instances_DeleteAccessConfig_ResultSet *results = [[TMBGoogle_ComputeEngine_Instances_DeleteAccessConfig_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteAccessConfig Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Instances_DeleteAccessConfig_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Instances_DeleteAccessConfig_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteInstance Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Instances_DeleteInstance_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Instance input for this Choreo.
	*(required, string) The name of the instance to delete.
	 */
	-(void)setInstance:(NSString*)Instance {
		[super setInput:@"Instance" toValue:Instance];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Zone input for this Choreo.
	*(required, string) The name of the zone associated with this request. 
	 */
	-(void)setZone:(NSString*)Zone {
		[super setInput:@"Zone" toValue:Zone];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteInstance Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Instances_DeleteInstance_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Deletes the specified Instance resource.
 */
@implementation TMBGoogle_ComputeEngine_Instances_DeleteInstance

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Instances_DeleteInstance Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Instances/DeleteInstance"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Instances_DeleteInstance_ResultSet *results = [[TMBGoogle_ComputeEngine_Instances_DeleteInstance_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteInstance Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Instances_DeleteInstance_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Instances_DeleteInstance_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DetachDisk Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Instances_DetachDisk_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the DeviceName input for this Choreo.
	*(required, string) The name of the disk to detach.
	 */
	-(void)setDeviceName:(NSString*)DeviceName {
		[super setInput:@"DeviceName" toValue:DeviceName];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Instance input for this Choreo.
	*(required, string) The name of the instance that is associated with the disk to detach.
	 */
	-(void)setInstance:(NSString*)Instance {
		[super setInput:@"Instance" toValue:Instance];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Zone input for this Choreo.
	*(required, string) The name of the zone associated with this request.
	 */
	-(void)setZone:(NSString*)Zone {
		[super setInput:@"Zone" toValue:Zone];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DetachDisk Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Instances_DetachDisk_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Detaches a Disk resource from an instance.
 */
@implementation TMBGoogle_ComputeEngine_Instances_DetachDisk

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Instances_DetachDisk Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Instances/DetachDisk"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Instances_DetachDisk_ResultSet *results = [[TMBGoogle_ComputeEngine_Instances_DetachDisk_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DetachDisk Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Instances_DetachDisk_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Instances_DetachDisk_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetInstance Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Instances_GetInstance_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Instance input for this Choreo.
	*(required, string) The name of the instance to retrieve.
	 */
	-(void)setInstance:(NSString*)Instance {
		[super setInput:@"Instance" toValue:Instance];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Zone input for this Choreo.
	*(required, string) The name of the zone associated with this request.
	 */
	-(void)setZone:(NSString*)Zone {
		[super setInput:@"Zone" toValue:Zone];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetInstance Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Instances_GetInstance_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves information about the specified Instance.
 */
@implementation TMBGoogle_ComputeEngine_Instances_GetInstance

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Instances_GetInstance Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Instances/GetInstance"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Instances_GetInstance_ResultSet *results = [[TMBGoogle_ComputeEngine_Instances_GetInstance_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetInstance Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Instances_GetInstance_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Instances_GetInstance_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetSerialPortOutput Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Instances_GetSerialPortOutput_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Instance input for this Choreo.
	*(required, string) The name of the instance associated with the serial port output to retrieve.
	 */
	-(void)setInstance:(NSString*)Instance {
		[super setInput:@"Instance" toValue:Instance];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Zone input for this Choreo.
	*(required, string) The name of the zone associated with this request.
	 */
	-(void)setZone:(NSString*)Zone {
		[super setInput:@"Zone" toValue:Zone];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetSerialPortOutput Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Instances_GetSerialPortOutput_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Returns the specified instance's serial port output.
 */
@implementation TMBGoogle_ComputeEngine_Instances_GetSerialPortOutput

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Instances_GetSerialPortOutput Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Instances/GetSerialPortOutput"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Instances_GetSerialPortOutput_ResultSet *results = [[TMBGoogle_ComputeEngine_Instances_GetSerialPortOutput_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetSerialPortOutput Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Instances_GetSerialPortOutput_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Instances_GetSerialPortOutput_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the InsertInstance Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Instances_InsertInstance_Inputs

	/*!
	 * Set the value of the InstanceResource input for this Choreo.
	*(optional, json) A JSON string containing the instance resource properties to set. This an be used as an alternative to individual inputs representing instance properties.
	 */
	-(void)setInstanceResource:(NSString*)InstanceResource {
		[super setInput:@"InstanceResource" toValue:InstanceResource];
	}

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Description input for this Choreo.
	*(optional, string) The description of the instance.
	 */
	-(void)setDescription:(NSString*)Description {
		[super setInput:@"Description" toValue:Description];
	}

	/*!
	 * Set the value of the Disks input for this Choreo.
	*(conditional, json) An array of persistent disks. This array contains the following properties: source, type, and boot.
	 */
	-(void)setDisks:(NSString*)Disks {
		[super setInput:@"Disks" toValue:Disks];
	}

	/*!
	 * Set the value of the MachineType input for this Choreo.
	*(conditional, string) The fully-qualified URL of the machine type resource to use for this instance.
	 */
	-(void)setMachineType:(NSString*)MachineType {
		[super setInput:@"MachineType" toValue:MachineType];
	}

	/*!
	 * Set the value of the Name input for this Choreo.
	*(conditional, string) The name of the instance being created.
	 */
	-(void)setName:(NSString*)Name {
		[super setInput:@"Name" toValue:Name];
	}

	/*!
	 * Set the value of the NetworkInterfaces input for this Choreo.
	*(conditional, json) An array of network configurations for this instance. This array contains the following properties: network, accessConfigs[], accessConfigs[].name, and accessConfigs[].type.
	 */
	-(void)setNetworkInterfaces:(NSString*)NetworkInterfaces {
		[super setInput:@"NetworkInterfaces" toValue:NetworkInterfaces];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Zone input for this Choreo.
	*(required, string) The name of the zone associated with this request.
	 */
	-(void)setZone:(NSString*)Zone {
		[super setInput:@"Zone" toValue:Zone];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the InsertInstance Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Instances_InsertInstance_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Creates an Instance resource in the specified project.
 */
@implementation TMBGoogle_ComputeEngine_Instances_InsertInstance

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Instances_InsertInstance Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Instances/InsertInstance"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Instances_InsertInstance_ResultSet *results = [[TMBGoogle_ComputeEngine_Instances_InsertInstance_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the InsertInstance Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Instances_InsertInstance_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Instances_InsertInstance_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListInstances Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Instances_ListInstances_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Filter input for this Choreo.
	*(optional, string) A filter expression for narrowing results in the form: {field_name} {comparison_string} {literal_string} (e.g. name eq testinstance). Comparison strings can be eq (equals) or ne (not equals).
	 */
	-(void)setFilter:(NSString*)Filter {
		[super setInput:@"Filter" toValue:Filter];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of results to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) The "nextPageToken" found in the response which is used to page through results.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Zone input for this Choreo.
	*(required, string) The name of the zone associated with this request.
	 */
	-(void)setZone:(NSString*)Zone {
		[super setInput:@"Zone" toValue:Zone];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListInstances Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Instances_ListInstances_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the list of Instance resources contained within the specified Zone.
 */
@implementation TMBGoogle_ComputeEngine_Instances_ListInstances

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Instances_ListInstances Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Instances/ListInstances"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Instances_ListInstances_ResultSet *results = [[TMBGoogle_ComputeEngine_Instances_ListInstances_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListInstances Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Instances_ListInstances_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Instances_ListInstances_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SetMetadata Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Instances_SetMetadata_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fingerprint input for this Choreo.
	*(required, string) The current fingerprint of this resource. You must provide the current version of the fingerprint to successfully update any metadata.
	 */
	-(void)setFingerprint:(NSString*)Fingerprint {
		[super setInput:@"Fingerprint" toValue:Fingerprint];
	}

	/*!
	 * Set the value of the Instance input for this Choreo.
	*(required, string) The name of the instance that you're setting metadata for.
	 */
	-(void)setInstance:(NSString*)Instance {
		[super setInput:@"Instance" toValue:Instance];
	}

	/*!
	 * Set the value of the Items input for this Choreo.
	*(required, json) An array of key/value pairs. The total size of the keys and values should not exceed 512 KB.
	 */
	-(void)setItems:(NSString*)Items {
		[super setInput:@"Items" toValue:Items];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Zone input for this Choreo.
	*(required, string) The name of the zone associated with this request.
	 */
	-(void)setZone:(NSString*)Zone {
		[super setInput:@"Zone" toValue:Zone];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SetMetadata Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Instances_SetMetadata_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Sets the metadata for the specified instance.
 */
@implementation TMBGoogle_ComputeEngine_Instances_SetMetadata

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Instances_SetMetadata Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Instances/SetMetadata"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Instances_SetMetadata_ResultSet *results = [[TMBGoogle_ComputeEngine_Instances_SetMetadata_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SetMetadata Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Instances_SetMetadata_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Instances_SetMetadata_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SetTags Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Instances_SetTags_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fingerprint input for this Choreo.
	*(required, string) The current fingerprint of this resource. You must provide the current version of the fingerprint to successfully update tags.
	 */
	-(void)setFingerprint:(NSString*)Fingerprint {
		[super setInput:@"Fingerprint" toValue:Fingerprint];
	}

	/*!
	 * Set the value of the Instance input for this Choreo.
	*(required, string) The name of the instance that you're setting tags for.
	 */
	-(void)setInstance:(NSString*)Instance {
		[super setInput:@"Instance" toValue:Instance];
	}

	/*!
	 * Set the value of the Items input for this Choreo.
	*(required, json) An array of tags. The max length of a tag is 63 characters.
	 */
	-(void)setItems:(NSString*)Items {
		[super setInput:@"Items" toValue:Items];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Zone input for this Choreo.
	*(required, string) The name of the zone associated with this request.
	 */
	-(void)setZone:(NSString*)Zone {
		[super setInput:@"Zone" toValue:Zone];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SetTags Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Instances_SetTags_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Sets tags for the specified instance.
 */
@implementation TMBGoogle_ComputeEngine_Instances_SetTags

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Instances_SetTags Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Instances/SetTags"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Instances_SetTags_ResultSet *results = [[TMBGoogle_ComputeEngine_Instances_SetTags_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SetTags Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Instances_SetTags_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Instances_SetTags_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the AggregatedList Choreo.
 */
@implementation TMBGoogle_ComputeEngine_MachineTypes_AggregatedList_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Filter input for this Choreo.
	*(optional, string) A filter expression for narrowing results in the form: {field_name} {comparison_string} {literal_string} (e.g. name eq eq f1-micro). Comparison strings can be eq (equals) or ne (not equals).
	 */
	-(void)setFilter:(NSString*)Filter {
		[super setInput:@"Filter" toValue:Filter];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of results to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) The "nextPageToken" found in the response which is used to page through results.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the AggregatedList Choreo.
 */
@implementation TMBGoogle_ComputeEngine_MachineTypes_AggregatedList_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a list of Machine Type resources across all available zones.
 */
@implementation TMBGoogle_ComputeEngine_MachineTypes_AggregatedList

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_MachineTypes_AggregatedList Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/MachineTypes/AggregatedList"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_MachineTypes_AggregatedList_ResultSet *results = [[TMBGoogle_ComputeEngine_MachineTypes_AggregatedList_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the AggregatedList Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_MachineTypes_AggregatedList_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_MachineTypes_AggregatedList_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetMachineType Choreo.
 */
@implementation TMBGoogle_ComputeEngine_MachineTypes_GetMachineType_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the MachineType input for this Choreo.
	*(required, string) The name of the machine type to retrieve.
	 */
	-(void)setMachineType:(NSString*)MachineType {
		[super setInput:@"MachineType" toValue:MachineType];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Zone input for this Choreo.
	*(required, string) The name of the zone that contains the machine type to retrieve.
	 */
	-(void)setZone:(NSString*)Zone {
		[super setInput:@"Zone" toValue:Zone];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetMachineType Choreo.
 */
@implementation TMBGoogle_ComputeEngine_MachineTypes_GetMachineType_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves information about the specified Machine Type.
 */
@implementation TMBGoogle_ComputeEngine_MachineTypes_GetMachineType

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_MachineTypes_GetMachineType Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/MachineTypes/GetMachineType"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_MachineTypes_GetMachineType_ResultSet *results = [[TMBGoogle_ComputeEngine_MachineTypes_GetMachineType_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetMachineType Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_MachineTypes_GetMachineType_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_MachineTypes_GetMachineType_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListMachineTypes Choreo.
 */
@implementation TMBGoogle_ComputeEngine_MachineTypes_ListMachineTypes_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Filter input for this Choreo.
	*(optional, string) A filter expression for narrowing results in the form: {field_name} {comparison_string} {literal_string} (e.g. name eq f1-micro). Comparison strings can be eq (equals) or ne (not equals).
	 */
	-(void)setFilter:(NSString*)Filter {
		[super setInput:@"Filter" toValue:Filter];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of results to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) The "nextPageToken" found in the response which is used to page through results.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Zone input for this Choreo.
	*(required, string) The name of the zone that contains the machine type resources to retrieve.
	 */
	-(void)setZone:(NSString*)Zone {
		[super setInput:@"Zone" toValue:Zone];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListMachineTypes Choreo.
 */
@implementation TMBGoogle_ComputeEngine_MachineTypes_ListMachineTypes_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the list of Machine Type resources for the specified project.
 */
@implementation TMBGoogle_ComputeEngine_MachineTypes_ListMachineTypes

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_MachineTypes_ListMachineTypes Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/MachineTypes/ListMachineTypes"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_MachineTypes_ListMachineTypes_ResultSet *results = [[TMBGoogle_ComputeEngine_MachineTypes_ListMachineTypes_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListMachineTypes Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_MachineTypes_ListMachineTypes_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_MachineTypes_ListMachineTypes_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteNetwork Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Networks_DeleteNetwork_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Network input for this Choreo.
	*(required, string) The name of the network to delete.
	 */
	-(void)setNetwork:(NSString*)Network {
		[super setInput:@"Network" toValue:Network];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteNetwork Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Networks_DeleteNetwork_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Deletes the specified Network resource.
 */
@implementation TMBGoogle_ComputeEngine_Networks_DeleteNetwork

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Networks_DeleteNetwork Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Networks/DeleteNetwork"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Networks_DeleteNetwork_ResultSet *results = [[TMBGoogle_ComputeEngine_Networks_DeleteNetwork_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteNetwork Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Networks_DeleteNetwork_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Networks_DeleteNetwork_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetNetwork Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Networks_GetNetwork_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Network input for this Choreo.
	*(required, string) The name of the network to retrieve.
	 */
	-(void)setNetwork:(NSString*)Network {
		[super setInput:@"Network" toValue:Network];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetNetwork Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Networks_GetNetwork_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves information about the specified Network.
 */
@implementation TMBGoogle_ComputeEngine_Networks_GetNetwork

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Networks_GetNetwork Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Networks/GetNetwork"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Networks_GetNetwork_ResultSet *results = [[TMBGoogle_ComputeEngine_Networks_GetNetwork_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetNetwork Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Networks_GetNetwork_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Networks_GetNetwork_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the InsertNetwork Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Networks_InsertNetwork_Inputs

	/*!
	 * Set the value of the NetworkResource input for this Choreo.
	*(optional, json) A JSON string containing the network resource properties you wish to set. This can be used as an alternative to individual inputs that represent network resource properties.
	 */
	-(void)setNetworkResource:(NSString*)NetworkResource {
		[super setInput:@"NetworkResource" toValue:NetworkResource];
	}

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Description input for this Choreo.
	*(optional, string) A description of the network.
	 */
	-(void)setDescription:(NSString*)Description {
		[super setInput:@"Description" toValue:Description];
	}

	/*!
	 * Set the value of the GatewayIP input for this Choreo.
	*(optional, string) An optional address used for default routing to other networks. Must be within the range specified by IPRange.
	 */
	-(void)setGatewayIP:(NSString*)GatewayIP {
		[super setInput:@"GatewayIP" toValue:GatewayIP];
	}

	/*!
	 * Set the value of the IPRange input for this Choreo.
	*(conditional, string) The range of internal addresses that are allowed on the network.
	 */
	-(void)setIPRange:(NSString*)IPRange {
		[super setInput:@"IPRange" toValue:IPRange];
	}

	/*!
	 * Set the value of the Name input for this Choreo.
	*(conditional, string) The name of the network.
	 */
	-(void)setName:(NSString*)Name {
		[super setInput:@"Name" toValue:Name];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the InsertNetwork Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Networks_InsertNetwork_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Creates a new Network resource in the specified project.
 */
@implementation TMBGoogle_ComputeEngine_Networks_InsertNetwork

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Networks_InsertNetwork Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Networks/InsertNetwork"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Networks_InsertNetwork_ResultSet *results = [[TMBGoogle_ComputeEngine_Networks_InsertNetwork_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the InsertNetwork Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Networks_InsertNetwork_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Networks_InsertNetwork_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListNetworks Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Networks_ListNetworks_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Filter input for this Choreo.
	*(optional, string) A filter expression for narrowing results in the form: {field_name} {comparison_string} {literal_string} (e.g. name eq default). Comparison strings can be eq (equals) or ne (not equals).
	 */
	-(void)setFilter:(NSString*)Filter {
		[super setInput:@"Filter" toValue:Filter];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of results to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) The "nextPageToken" found in the response which is used to page through results.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListNetworks Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Networks_ListNetworks_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the list of Network resources for the specified project.
 */
@implementation TMBGoogle_ComputeEngine_Networks_ListNetworks

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Networks_ListNetworks Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Networks/ListNetworks"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Networks_ListNetworks_ResultSet *results = [[TMBGoogle_ComputeEngine_Networks_ListNetworks_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListNetworks Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Networks_ListNetworks_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Networks_ListNetworks_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetProject Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Projects_GetProject_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetProject Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Projects_GetProject_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the specified Project resource.
 */
@implementation TMBGoogle_ComputeEngine_Projects_GetProject

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Projects_GetProject Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Projects/GetProject"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Projects_GetProject_ResultSet *results = [[TMBGoogle_ComputeEngine_Projects_GetProject_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetProject Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Projects_GetProject_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Projects_GetProject_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SetCommonInstanceMetadata Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Projects_SetCommonInstanceMetadata_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fingerprint input for this Choreo.
	*(required, string) The fingerprint of this resource, which is a hash of the metadata's contents. This field is used for optimistic locking. Providing an up-to-date metadata fingerprint is required to modify metadata.
	 */
	-(void)setFingerprint:(NSString*)Fingerprint {
		[super setInput:@"Fingerprint" toValue:Fingerprint];
	}

	/*!
	 * Set the value of the Items input for this Choreo.
	*(conditional, json) An array of key/value pairs. The max size of all keys and values is 512 KB.
	 */
	-(void)setItems:(NSString*)Items {
		[super setInput:@"Items" toValue:Items];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SetCommonInstanceMetadata Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Projects_SetCommonInstanceMetadata_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Sets metadata common to all instances within the specified Project.
 */
@implementation TMBGoogle_ComputeEngine_Projects_SetCommonInstanceMetadata

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Projects_SetCommonInstanceMetadata Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Projects/SetCommonInstanceMetadata"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Projects_SetCommonInstanceMetadata_ResultSet *results = [[TMBGoogle_ComputeEngine_Projects_SetCommonInstanceMetadata_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SetCommonInstanceMetadata Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Projects_SetCommonInstanceMetadata_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Projects_SetCommonInstanceMetadata_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteRegionOperation Choreo.
 */
@implementation TMBGoogle_ComputeEngine_RegionOperations_DeleteRegionOperation_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Operation input for this Choreo.
	*(required, string) The name of the operation to delete.
	 */
	-(void)setOperation:(NSString*)Operation {
		[super setInput:@"Operation" toValue:Operation];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Region input for this Choreo.
	*(required, string) The name of the region associated with the request.
	 */
	-(void)setRegion:(NSString*)Region {
		[super setInput:@"Region" toValue:Region];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteRegionOperation Choreo.
 */
@implementation TMBGoogle_ComputeEngine_RegionOperations_DeleteRegionOperation_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "ResponseStatusCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) The response status code returned from Google. A 204 is expected for a successful delete operation.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponseStatusCode {
		return [super getOutputByName:@"ResponseStatusCode"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Deletes the specified operation within a region.
 */
@implementation TMBGoogle_ComputeEngine_RegionOperations_DeleteRegionOperation

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_RegionOperations_DeleteRegionOperation Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/RegionOperations/DeleteRegionOperation"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_RegionOperations_DeleteRegionOperation_ResultSet *results = [[TMBGoogle_ComputeEngine_RegionOperations_DeleteRegionOperation_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteRegionOperation Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_RegionOperations_DeleteRegionOperation_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_RegionOperations_DeleteRegionOperation_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetRegionOperation Choreo.
 */
@implementation TMBGoogle_ComputeEngine_RegionOperations_GetRegionOperation_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Operation input for this Choreo.
	*(required, string) The name of the operation to retrieve.
	 */
	-(void)setOperation:(NSString*)Operation {
		[super setInput:@"Operation" toValue:Operation];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Region input for this Choreo.
	*(required, string) The name of the region associated with the request.
	 */
	-(void)setRegion:(NSString*)Region {
		[super setInput:@"Region" toValue:Region];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetRegionOperation Choreo.
 */
@implementation TMBGoogle_ComputeEngine_RegionOperations_GetRegionOperation_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves information about the specified operation within a region.
 */
@implementation TMBGoogle_ComputeEngine_RegionOperations_GetRegionOperation

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_RegionOperations_GetRegionOperation Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/RegionOperations/GetRegionOperation"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_RegionOperations_GetRegionOperation_ResultSet *results = [[TMBGoogle_ComputeEngine_RegionOperations_GetRegionOperation_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetRegionOperation Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_RegionOperations_GetRegionOperation_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_RegionOperations_GetRegionOperation_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListRegionOperations Choreo.
 */
@implementation TMBGoogle_ComputeEngine_RegionOperations_ListRegionOperations_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Filter input for this Choreo.
	*(optional, string) A filter expression for narrowing results in the form: {field_name} {comparison_string} {literal_string} (e.g. name eq operation-1234). Comparison strings can be eq (equals) or ne (not equals).
	 */
	-(void)setFilter:(NSString*)Filter {
		[super setInput:@"Filter" toValue:Filter];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of results to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) The "nextPageToken" found in the response which is used to page through results.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Region input for this Choreo.
	*(required, string) The name of the region associated with this request.
	 */
	-(void)setRegion:(NSString*)Region {
		[super setInput:@"Region" toValue:Region];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListRegionOperations Choreo.
 */
@implementation TMBGoogle_ComputeEngine_RegionOperations_ListRegionOperations_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a list of operation resources contained within the specified region.
 */
@implementation TMBGoogle_ComputeEngine_RegionOperations_ListRegionOperations

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_RegionOperations_ListRegionOperations Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/RegionOperations/ListRegionOperations"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_RegionOperations_ListRegionOperations_ResultSet *results = [[TMBGoogle_ComputeEngine_RegionOperations_ListRegionOperations_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListRegionOperations Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_RegionOperations_ListRegionOperations_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_RegionOperations_ListRegionOperations_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetRegion Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Regions_GetRegion_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Region input for this Choreo.
	*(required, string) The name of the region to retrieve.
	 */
	-(void)setRegion:(NSString*)Region {
		[super setInput:@"Region" toValue:Region];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetRegion Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Regions_GetRegion_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves information about the specified Region.
 */
@implementation TMBGoogle_ComputeEngine_Regions_GetRegion

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Regions_GetRegion Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Regions/GetRegion"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Regions_GetRegion_ResultSet *results = [[TMBGoogle_ComputeEngine_Regions_GetRegion_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetRegion Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Regions_GetRegion_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Regions_GetRegion_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListRegions Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Regions_ListRegions_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Filter input for this Choreo.
	*(optional, string) A filter expression for narrowing results in the form: {field_name} {comparison_string} {literal_string} (e.g. name eq europe-west1). Comparison strings can be eq (equals) or ne (not equals).
	 */
	-(void)setFilter:(NSString*)Filter {
		[super setInput:@"Filter" toValue:Filter];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of results to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) The "nextPageToken" found in the response which is used to page through results.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListRegions Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Regions_ListRegions_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the list of Regions available to the specified project.
 */
@implementation TMBGoogle_ComputeEngine_Regions_ListRegions

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Regions_ListRegions Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Regions/ListRegions"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Regions_ListRegions_ResultSet *results = [[TMBGoogle_ComputeEngine_Regions_ListRegions_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListRegions Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Regions_ListRegions_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Regions_ListRegions_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteRoute Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Routes_DeleteRoute_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Route input for this Choreo.
	*(required, string) The name of the route to delete.
	 */
	-(void)setRoute:(NSString*)Route {
		[super setInput:@"Route" toValue:Route];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteRoute Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Routes_DeleteRoute_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Deletes the specified Route.
 */
@implementation TMBGoogle_ComputeEngine_Routes_DeleteRoute

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Routes_DeleteRoute Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Routes/DeleteRoute"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Routes_DeleteRoute_ResultSet *results = [[TMBGoogle_ComputeEngine_Routes_DeleteRoute_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteRoute Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Routes_DeleteRoute_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Routes_DeleteRoute_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetRoute Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Routes_GetRoute_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response..
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Route input for this Choreo.
	*(required, string) The name of the route to retrieve.
	 */
	-(void)setRoute:(NSString*)Route {
		[super setInput:@"Route" toValue:Route];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetRoute Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Routes_GetRoute_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves information about the specified Route.
 */
@implementation TMBGoogle_ComputeEngine_Routes_GetRoute

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Routes_GetRoute Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Routes/GetRoute"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Routes_GetRoute_ResultSet *results = [[TMBGoogle_ComputeEngine_Routes_GetRoute_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetRoute Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Routes_GetRoute_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Routes_GetRoute_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the InsertRoute Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Routes_InsertRoute_Inputs

	/*!
	 * Set the value of the RouteResource input for this Choreo.
	*(required, json) A JSON string containing the route resource fields to set. Required properties include: destRange, name, network, priority, and at at least one nextHop property.
	 */
	-(void)setRouteResource:(NSString*)RouteResource {
		[super setInput:@"RouteResource" toValue:RouteResource];
	}

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the InsertRoute Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Routes_InsertRoute_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Creates a Route resource in the specified project.
 */
@implementation TMBGoogle_ComputeEngine_Routes_InsertRoute

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Routes_InsertRoute Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Routes/InsertRoute"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Routes_InsertRoute_ResultSet *results = [[TMBGoogle_ComputeEngine_Routes_InsertRoute_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the InsertRoute Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Routes_InsertRoute_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Routes_InsertRoute_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListRoutes Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Routes_ListRoutes_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Filter input for this Choreo.
	*(optional, string) A filter expression for narrowing results in the form: {field_name} {comparison_string} {literal_string} (e.g. name eq default-route-1234). Comparison strings can be eq (equals) or ne (not equals).
	 */
	-(void)setFilter:(NSString*)Filter {
		[super setInput:@"Filter" toValue:Filter];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of results to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) The "nextPageToken" found in the response which is used to page through results.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListRoutes Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Routes_ListRoutes_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a list of route resources available to the specified project.
 */
@implementation TMBGoogle_ComputeEngine_Routes_ListRoutes

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Routes_ListRoutes Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Routes/ListRoutes"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Routes_ListRoutes_ResultSet *results = [[TMBGoogle_ComputeEngine_Routes_ListRoutes_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListRoutes Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Routes_ListRoutes_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Routes_ListRoutes_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteSnapshot Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Snapshots_DeleteSnapshot_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Snapshot input for this Choreo.
	*(required, string) The name of the snapshot to delete.
	 */
	-(void)setSnapshot:(NSString*)Snapshot {
		[super setInput:@"Snapshot" toValue:Snapshot];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteSnapshot Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Snapshots_DeleteSnapshot_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Deletes a specified Snapshot resource.
 */
@implementation TMBGoogle_ComputeEngine_Snapshots_DeleteSnapshot

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Snapshots_DeleteSnapshot Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Snapshots/DeleteSnapshot"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Snapshots_DeleteSnapshot_ResultSet *results = [[TMBGoogle_ComputeEngine_Snapshots_DeleteSnapshot_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteSnapshot Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Snapshots_DeleteSnapshot_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Snapshots_DeleteSnapshot_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetSnapshot Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Snapshots_GetSnapshot_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Snapshot input for this Choreo.
	*(required, string) The name of the snapshot to retrieve.
	 */
	-(void)setSnapshot:(NSString*)Snapshot {
		[super setInput:@"Snapshot" toValue:Snapshot];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetSnapshot Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Snapshots_GetSnapshot_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves information about the specified Snapshot.
 */
@implementation TMBGoogle_ComputeEngine_Snapshots_GetSnapshot

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Snapshots_GetSnapshot Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Snapshots/GetSnapshot"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Snapshots_GetSnapshot_ResultSet *results = [[TMBGoogle_ComputeEngine_Snapshots_GetSnapshot_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetSnapshot Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Snapshots_GetSnapshot_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Snapshots_GetSnapshot_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListSnapshots Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Snapshots_ListSnapshots_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Filter input for this Choreo.
	*(optional, string) A filter expression for narrowing results in the form: {field_name} {comparison_string} {literal_string} (e.g. name eq testsnapshot). Comparison strings can be eq (equals) or ne (not equals).
	 */
	-(void)setFilter:(NSString*)Filter {
		[super setInput:@"Filter" toValue:Filter];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of results to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) The "nextPageToken" found in the response which is used to page through results.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListSnapshots Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Snapshots_ListSnapshots_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a list of persistent disk snapshot resources for the specified project.
 */
@implementation TMBGoogle_ComputeEngine_Snapshots_ListSnapshots

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Snapshots_ListSnapshots Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Snapshots/ListSnapshots"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Snapshots_ListSnapshots_ResultSet *results = [[TMBGoogle_ComputeEngine_Snapshots_ListSnapshots_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListSnapshots Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Snapshots_ListSnapshots_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Snapshots_ListSnapshots_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteZoneOperation Choreo.
 */
@implementation TMBGoogle_ComputeEngine_ZoneOperations_DeleteZoneOperation_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Operation input for this Choreo.
	*(required, string) The name of the operation to delete.
	 */
	-(void)setOperation:(NSString*)Operation {
		[super setInput:@"Operation" toValue:Operation];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Zone input for this Choreo.
	*(required, string) The name of the zone associated with the operation being deleted.
	 */
	-(void)setZone:(NSString*)Zone {
		[super setInput:@"Zone" toValue:Zone];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteZoneOperation Choreo.
 */
@implementation TMBGoogle_ComputeEngine_ZoneOperations_DeleteZoneOperation_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "ResponseStatusCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) The response status code returned from Google. A 204 is expected for a successful delete operation.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponseStatusCode {
		return [super getOutputByName:@"ResponseStatusCode"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Deletes the specified Zone Operation resource.
 */
@implementation TMBGoogle_ComputeEngine_ZoneOperations_DeleteZoneOperation

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_ZoneOperations_DeleteZoneOperation Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/ZoneOperations/DeleteZoneOperation"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_ZoneOperations_DeleteZoneOperation_ResultSet *results = [[TMBGoogle_ComputeEngine_ZoneOperations_DeleteZoneOperation_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteZoneOperation Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_ZoneOperations_DeleteZoneOperation_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_ZoneOperations_DeleteZoneOperation_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetZoneOperation Choreo.
 */
@implementation TMBGoogle_ComputeEngine_ZoneOperations_GetZoneOperation_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Operation input for this Choreo.
	*(required, string) The name of the operation to retrieve.
	 */
	-(void)setOperation:(NSString*)Operation {
		[super setInput:@"Operation" toValue:Operation];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Zone input for this Choreo.
	*(required, string) The name of the zone associated with the operation being retrieved.
	 */
	-(void)setZone:(NSString*)Zone {
		[super setInput:@"Zone" toValue:Zone];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetZoneOperation Choreo.
 */
@implementation TMBGoogle_ComputeEngine_ZoneOperations_GetZoneOperation_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves information about the specified Zone Operation.
 */
@implementation TMBGoogle_ComputeEngine_ZoneOperations_GetZoneOperation

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_ZoneOperations_GetZoneOperation Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/ZoneOperations/GetZoneOperation"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_ZoneOperations_GetZoneOperation_ResultSet *results = [[TMBGoogle_ComputeEngine_ZoneOperations_GetZoneOperation_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetZoneOperation Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_ZoneOperations_GetZoneOperation_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_ZoneOperations_GetZoneOperation_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListZoneOperations Choreo.
 */
@implementation TMBGoogle_ComputeEngine_ZoneOperations_ListZoneOperations_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Filter input for this Choreo.
	*(optional, string) A filter expression for narrowing results in the form: {field_name} {comparison_string} {literal_string} (e.g. name eq europe-west1-a). Comparison strings can be eq (equals) or ne (not equals).
	 */
	-(void)setFilter:(NSString*)Filter {
		[super setInput:@"Filter" toValue:Filter];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of results to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) The "nextPageToken" found in the response which is used to page through results.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Zone input for this Choreo.
	*(required, string) The name of the zone that contains the operation resources to retrieve.
	 */
	-(void)setZone:(NSString*)Zone {
		[super setInput:@"Zone" toValue:Zone];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListZoneOperations Choreo.
 */
@implementation TMBGoogle_ComputeEngine_ZoneOperations_ListZoneOperations_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the list of Zone Operation resources contained within the specified Zone.
 */
@implementation TMBGoogle_ComputeEngine_ZoneOperations_ListZoneOperations

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_ZoneOperations_ListZoneOperations Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/ZoneOperations/ListZoneOperations"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_ZoneOperations_ListZoneOperations_ResultSet *results = [[TMBGoogle_ComputeEngine_ZoneOperations_ListZoneOperations_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListZoneOperations Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_ZoneOperations_ListZoneOperations_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_ZoneOperations_ListZoneOperations_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetZone Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Zones_GetZone_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Zone input for this Choreo.
	*(required, string) The name of the zone to retrieve.
	 */
	-(void)setZone:(NSString*)Zone {
		[super setInput:@"Zone" toValue:Zone];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetZone Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Zones_GetZone_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves information about the specified Zone.
 */
@implementation TMBGoogle_ComputeEngine_Zones_GetZone

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Zones_GetZone Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Zones/GetZone"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Zones_GetZone_ResultSet *results = [[TMBGoogle_ComputeEngine_Zones_GetZone_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetZone Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Zones_GetZone_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Zones_GetZone_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListZones Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Zones_ListZones_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-seperated list of fields you want to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Filter input for this Choreo.
	*(optional, string) A filter expression for narrowing results in the form: {field_name} {comparison_string} {literal_string} (e.g. name eq europe-west1-a). Comparison strings can be eq (equals) or ne (not equals).
	 */
	-(void)setFilter:(NSString*)Filter {
		[super setInput:@"Filter" toValue:Filter];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of results to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) The "nextPageToken" found in the response which is used to page through results.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the Project input for this Choreo.
	*(required, string) The ID of a Google Compute project.
	 */
	-(void)setProject:(NSString*)Project {
		[super setInput:@"Project" toValue:Project];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListZones Choreo.
 */
@implementation TMBGoogle_ComputeEngine_Zones_ListZones_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the list of Zone resources for the specified project.
 */
@implementation TMBGoogle_ComputeEngine_Zones_ListZones

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_ComputeEngine_Zones_ListZones Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/ComputeEngine/Zones/ListZones"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_ComputeEngine_Zones_ListZones_ResultSet *results = [[TMBGoogle_ComputeEngine_Zones_ListZones_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListZones Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_ComputeEngine_Zones_ListZones_Inputs*)newInputSet {
		return [[TMBGoogle_ComputeEngine_Zones_ListZones_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CreateContact Choreo.
 */
@implementation TMBGoogle_Contacts_CreateContact_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) The access token retrieved in the last step of the OAuth process. Access tokens that are expired will be refreshed and returned in the Choreo output.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The OAuth client ID provided by Google when you register your application.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The OAuth client secret provided by Google when you registered your application.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Email input for this Choreo.
	*(conditional, string) The new contact's email address.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the FirstName input for this Choreo.
	*(conditional, string) The new contact's first name.
	 */
	-(void)setFirstName:(NSString*)FirstName {
		[super setInput:@"FirstName" toValue:FirstName];
	}

	/*!
	 * Set the value of the LastName input for this Choreo.
	*(conditional, string) The new contact's last name.
	 */
	-(void)setLastName:(NSString*)LastName {
		[super setInput:@"LastName" toValue:LastName];
	}

	/*!
	 * Set the value of the Phone input for this Choreo.
	*(optional, string) The phone number for the new contact. It's best to use the "(555) 123-4567" format.
	 */
	-(void)setPhone:(NSString*)Phone {
		[super setInput:@"Phone" toValue:Phone];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) The refresh token retrieved in the last step of the OAuth process. This is used when an access token is expired or not provided.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateContact Choreo.
 */
@implementation TMBGoogle_Contacts_CreateContact_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "AccessToken" output from an execution of this Choreo.
	 * @return - NSString* (optional, string) The access token retrieved in the last step of the OAuth process. Access tokens that are expired will be refreshed and returned in the Choreo output.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getAccessToken {
		return [super getOutputByName:@"AccessToken"];
	}

	/*!
	 * Retrieve the value of the "ContactID" output from an execution of this Choreo.
	 * @return - NSString* (string) The unique ID supplied by Google for the new user.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getContactID {
		return [super getOutputByName:@"ContactID"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (xml) The response from the API.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Create a new contact.
 */
@implementation TMBGoogle_Contacts_CreateContact

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Contacts_CreateContact Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Contacts/CreateContact"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Contacts_CreateContact_ResultSet *results = [[TMBGoogle_Contacts_CreateContact_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CreateContact Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Contacts_CreateContact_Inputs*)newInputSet {
		return [[TMBGoogle_Contacts_CreateContact_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteContact Choreo.
 */
@implementation TMBGoogle_Contacts_DeleteContact_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) The access token retrieved in the last step of the OAuth process. Access tokens that are expired will be refreshed and returned in the Choreo output.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(required, string) The OAuth client ID provided by Google when you register your application.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(required, string) The OAuth client secret provided by Google when you registered your application.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the ContactID input for this Choreo.
	*(required, string) The unique ID string for the contact you want to delete.
	 */
	-(void)setContactID:(NSString*)ContactID {
		[super setInput:@"ContactID" toValue:ContactID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(required, string) The refresh token retrieved in the last step of the OAuth process. This is used when an access token is expired or not provided.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteContact Choreo.
 */
@implementation TMBGoogle_Contacts_DeleteContact_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "AccessToken" output from an execution of this Choreo.
	 * @return - NSString* (optional, string) The access token retrieved in the last step of the OAuth process. Access tokens that are expired will be refreshed and returned in the Choreo output.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getAccessToken {
		return [super getOutputByName:@"AccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Google. No content is returned for a successful delete request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Deletes a specified contact.
 */
@implementation TMBGoogle_Contacts_DeleteContact

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Contacts_DeleteContact Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Contacts/DeleteContact"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Contacts_DeleteContact_ResultSet *results = [[TMBGoogle_Contacts_DeleteContact_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteContact Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Contacts_DeleteContact_Inputs*)newInputSet {
		return [[TMBGoogle_Contacts_DeleteContact_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetAllContacts Choreo.
 */
@implementation TMBGoogle_Contacts_GetAllContacts_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) The access token retrieved in the last step of the OAuth process. Access tokens that are expired will be refreshed and returned in the Choreo output.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(required, string) The client ID provided by Google when you register your application.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(required, string) The client secret provided by Google when you registered your application.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Group input for this Choreo.
	*(optional, string) Constrains the results to only the contacts belonging to the group specified. The value of this parameter specifies group ID.
	 */
	-(void)setGroup:(NSString*)Group {
		[super setInput:@"Group" toValue:Group];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of entries to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(required, string) The refresh token retrieved in the last step of the OAuth process. This is used when an access token is expired or not provided.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the SortOrder input for this Choreo.
	*(optional, string) Sorting order direction. Can be either ascending or descending.
	 */
	-(void)setSortOrder:(NSString*)SortOrder {
		[super setInput:@"SortOrder" toValue:SortOrder];
	}

	/*!
	 * Set the value of the StartIndex input for this Choreo.
	*(optional, integer) The index of the first result to be retrieved (for paging).
	 */
	-(void)setStartIndex:(NSString*)StartIndex {
		[super setInput:@"StartIndex" toValue:StartIndex];
	}

	/*!
	 * Set the value of the UpdatedMin input for this Choreo.
	*(optional, date) The lower bound on entry update dates to filter by (e.g., 2015-01-16T00:00:00).
	 */
	-(void)setUpdatedMin:(NSString*)UpdatedMin {
		[super setInput:@"UpdatedMin" toValue:UpdatedMin];
	}

	/*!
	 * Set the value of the UserEmail input for this Choreo.
	*(optional, string) The email address of the user whose contacts you want to retrieve. Defaults to "default," or the user whose OAuth access token is passed.
	 */
	-(void)setUserEmail:(NSString*)UserEmail {
		[super setInput:@"UserEmail" toValue:UserEmail];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetAllContacts Choreo.
 */
@implementation TMBGoogle_Contacts_GetAllContacts_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "AccessToken" output from an execution of this Choreo.
	 * @return - NSString* (optional, string) The access token retrieved in the last step of the OAuth process. Access tokens that are expired will be refreshed and returned in the Choreo output.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getAccessToken {
		return [super getOutputByName:@"AccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (xml) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieve data for all contacts in an account.
 */
@implementation TMBGoogle_Contacts_GetAllContacts

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Contacts_GetAllContacts Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Contacts/GetAllContacts"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Contacts_GetAllContacts_ResultSet *results = [[TMBGoogle_Contacts_GetAllContacts_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetAllContacts Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Contacts_GetAllContacts_Inputs*)newInputSet {
		return [[TMBGoogle_Contacts_GetAllContacts_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetAllGroups Choreo.
 */
@implementation TMBGoogle_Contacts_GetAllGroups_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) The access token retrieved in the last step of the OAuth process. Access tokens that are expired will be refreshed and returned in the Choreo output.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(required, string) The OAuth client ID provided by Google when you register your application.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(required, string) The OAuth client secret provided by Google when you registered your application.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of entries to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(required, string) The refresh token retrieved in the last step of the OAuth process. This is used when an access token is expired or not provided.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the StartIndex input for this Choreo.
	*(optional, integer) The index of the first result to be retrieved (for paging).
	 */
	-(void)setStartIndex:(NSString*)StartIndex {
		[super setInput:@"StartIndex" toValue:StartIndex];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetAllGroups Choreo.
 */
@implementation TMBGoogle_Contacts_GetAllGroups_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "AccessToken" output from an execution of this Choreo.
	 * @return - NSString* (optional, string) The access token retrieved in the last step of the OAuth process. Access tokens that are expired will be refreshed and returned in the Choreo output.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getAccessToken {
		return [super getOutputByName:@"AccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (xml) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieve data for all groups in an account.
 */
@implementation TMBGoogle_Contacts_GetAllGroups

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Contacts_GetAllGroups Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Contacts/GetAllGroups"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Contacts_GetAllGroups_ResultSet *results = [[TMBGoogle_Contacts_GetAllGroups_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetAllGroups Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Contacts_GetAllGroups_Inputs*)newInputSet {
		return [[TMBGoogle_Contacts_GetAllGroups_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetContactById Choreo.
 */
@implementation TMBGoogle_Contacts_GetContactById_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) The access token retrieved in the last step of the OAuth process. Access tokens that are expired will be refreshed and returned in the Choreo output.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(required, string) The client ID provided by Google when you register your application.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(required, string) The client secret provided by Google when you registered your application.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the ContactId input for this Choreo.
	*(required, string) The id associated with the contact to return. This can either be the individual id of the contact, or the full 'edit' link returned in the entry nodes of the GetAllContacts output.
	 */
	-(void)setContactId:(NSString*)ContactId {
		[super setInput:@"ContactId" toValue:ContactId];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(required, string) The refresh token retrieved in the last step of the OAuth process. This is used when an access token is expired or not provided.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the UserEmail input for this Choreo.
	*(required, string) The email address of the user whose contacts you want to retrieve. Defaults to "default," or the user whose OAuth access token is passed.
	 */
	-(void)setUserEmail:(NSString*)UserEmail {
		[super setInput:@"UserEmail" toValue:UserEmail];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetContactById Choreo.
 */
@implementation TMBGoogle_Contacts_GetContactById_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "AccessToken" output from an execution of this Choreo.
	 * @return - NSString* (optional, string) The access token retrieved in the last step of the OAuth process. Access tokens that are expired will be refreshed and returned in the Choreo output.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getAccessToken {
		return [super getOutputByName:@"AccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (xml) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a specific contact with a given id.
 */
@implementation TMBGoogle_Contacts_GetContactById

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Contacts_GetContactById Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Contacts/GetContactById"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Contacts_GetContactById_ResultSet *results = [[TMBGoogle_Contacts_GetContactById_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetContactById Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Contacts_GetContactById_Inputs*)newInputSet {
		return [[TMBGoogle_Contacts_GetContactById_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetContactsWithQuery Choreo.
 */
@implementation TMBGoogle_Contacts_GetContactsWithQuery_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) The access token retrieved in the last step of the OAuth process. Access tokens that are expired will be refreshed and returned in the Choreo output.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(required, string) The OAuth client ID provided by Google when you register your application.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(required, string) The OAuth client secret provided by Google when you registered your application.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Group input for this Choreo.
	*(optional, string) Constrains the results to only the contacts belonging to the group specified. The value of this parameter specifies group ID.
	 */
	-(void)setGroup:(NSString*)Group {
		[super setInput:@"Group" toValue:Group];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of entries to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the Query input for this Choreo.
	*(required, string) The contact criteria to search for, such as name or email address.
	 */
	-(void)setQuery:(NSString*)Query {
		[super setInput:@"Query" toValue:Query];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(required, string) The refresh token retrieved in the last step of the OAuth process. This is used when an access token is expired or not provided.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the SortOrder input for this Choreo.
	*(optional, string) Sorting order direction. Can be either ascending or descending.
	 */
	-(void)setSortOrder:(NSString*)SortOrder {
		[super setInput:@"SortOrder" toValue:SortOrder];
	}

	/*!
	 * Set the value of the StartIndex input for this Choreo.
	*(optional, integer) The index of the first result to be retrieved (for paging).
	 */
	-(void)setStartIndex:(NSString*)StartIndex {
		[super setInput:@"StartIndex" toValue:StartIndex];
	}

	/*!
	 * Set the value of the UpdatedMin input for this Choreo.
	*(optional, date) The lower bound on entry update dates to filter by (e.g., 2015-01-16T00:00:00).
	 */
	-(void)setUpdatedMin:(NSString*)UpdatedMin {
		[super setInput:@"UpdatedMin" toValue:UpdatedMin];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetContactsWithQuery Choreo.
 */
@implementation TMBGoogle_Contacts_GetContactsWithQuery_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "AccessToken" output from an execution of this Choreo.
	 * @return - NSString* (optional, string) The access token retrieved in the last step of the OAuth process. Access tokens that are expired will be refreshed and returned in the Choreo output.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getAccessToken {
		return [super getOutputByName:@"AccessToken"];
	}

	/*!
	 * Retrieve the value of the "ContactID" output from an execution of this Choreo.
	 * @return - NSString* (string) The unique ID string for the retrieved contact. If more than one contact is retrieved by the request, only the first contact's ID is output.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getContactID {
		return [super getOutputByName:@"ContactID"];
	}

	/*!
	 * Retrieve the value of the "Link" output from an execution of this Choreo.
	 * @return - NSString* (string) The unique edit link for the retrieved contact. If more than one contact is retrieved by the request, only the first contact's edit link is output.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLink {
		return [super getOutputByName:@"Link"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (xml) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the contact or contacts in that account that match a specified query term.
 */
@implementation TMBGoogle_Contacts_GetContactsWithQuery

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Contacts_GetContactsWithQuery Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Contacts/GetContactsWithQuery"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Contacts_GetContactsWithQuery_ResultSet *results = [[TMBGoogle_Contacts_GetContactsWithQuery_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetContactsWithQuery Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Contacts_GetContactsWithQuery_Inputs*)newInputSet {
		return [[TMBGoogle_Contacts_GetContactsWithQuery_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UpdateContact Choreo.
 */
@implementation TMBGoogle_Contacts_UpdateContact_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) The access token retrieved in the last step of the OAuth process. Access tokens that are expired will be refreshed and returned in the Choreo output.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The OAuth client ID provided by Google when you register your application.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The OAuth client secret provided by Google when you registered your application.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(conditional, string) The id of the contact to update. Required unless providing a value for the Query input.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the NewEmail input for this Choreo.
	*(conditional, string) The contact's new email address.
	 */
	-(void)setNewEmail:(NSString*)NewEmail {
		[super setInput:@"NewEmail" toValue:NewEmail];
	}

	/*!
	 * Set the value of the NewFirstName input for this Choreo.
	*(conditional, string) The contact's new first name.
	 */
	-(void)setNewFirstName:(NSString*)NewFirstName {
		[super setInput:@"NewFirstName" toValue:NewFirstName];
	}

	/*!
	 * Set the value of the NewLastName input for this Choreo.
	*(conditional, string) The contact's new last name.
	 */
	-(void)setNewLastName:(NSString*)NewLastName {
		[super setInput:@"NewLastName" toValue:NewLastName];
	}

	/*!
	 * Set the value of the NewPhone input for this Choreo.
	*(optional, string) The contact's new telephone number.
	 */
	-(void)setNewPhone:(NSString*)NewPhone {
		[super setInput:@"NewPhone" toValue:NewPhone];
	}

	/*!
	 * Set the value of the Query input for this Choreo.
	*(conditional, string) A search term to retrieve the contact to update, such as an email address, last name, or address. Required unless providing a value for the ID input.
	 */
	-(void)setQuery:(NSString*)Query {
		[super setInput:@"Query" toValue:Query];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) The refresh token retrieved in the last step of the OAuth process. This is used when an access token is expired or not provided.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UpdateContact Choreo.
 */
@implementation TMBGoogle_Contacts_UpdateContact_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "AccessToken" output from an execution of this Choreo.
	 * @return - NSString* (optional, string) The access token retrieved in the last step of the OAuth process. Access tokens that are expired will be refreshed and returned in the Choreo output.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getAccessToken {
		return [super getOutputByName:@"AccessToken"];
	}

	/*!
	 * Retrieve the value of the "ContactID" output from an execution of this Choreo.
	 * @return - NSString* (string) The unique ID for the contact returned by Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getContactID {
		return [super getOutputByName:@"ContactID"];
	}

	/*!
	 * Retrieve the value of the "Email" output from an execution of this Choreo.
	 * @return - NSString* (string) The contact's current email address.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getEmail {
		return [super getOutputByName:@"Email"];
	}

	/*!
	 * Retrieve the value of the "FirstName" output from an execution of this Choreo.
	 * @return - NSString* (string) The contact's current given name.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getFirstName {
		return [super getOutputByName:@"FirstName"];
	}

	/*!
	 * Retrieve the value of the "LastName" output from an execution of this Choreo.
	 * @return - NSString* (string) The contact's current family name.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLastName {
		return [super getOutputByName:@"LastName"];
	}

	/*!
	 * Retrieve the value of the "Phone" output from an execution of this Choreo.
	 * @return - NSString* (string) The contact's current telephone number.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPhone {
		return [super getOutputByName:@"Phone"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (xml) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Update an existing contact's information.
 */
@implementation TMBGoogle_Contacts_UpdateContact

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Contacts_UpdateContact Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Contacts/UpdateContact"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Contacts_UpdateContact_ResultSet *results = [[TMBGoogle_Contacts_UpdateContact_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UpdateContact Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Contacts_UpdateContact_Inputs*)newInputSet {
		return [[TMBGoogle_Contacts_UpdateContact_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetBicyclingDirections Choreo.
 */
@implementation TMBGoogle_Directions_GetBicyclingDirections_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) The API Key provided by Google.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the Alternatives input for this Choreo.
	*(optional, string) If set to true, additional routes will be returned.
	 */
	-(void)setAlternatives:(NSString*)Alternatives {
		[super setInput:@"Alternatives" toValue:Alternatives];
	}

	/*!
	 * Set the value of the Destination input for this Choreo.
	*(required, string) Enter the address or latitude/longitude coordinates from which directions will be generated (i.e."104 Franklin St, New York, NY" or "40.7160,-74.0037").
	 */
	-(void)setDestination:(NSString*)Destination {
		[super setInput:@"Destination" toValue:Destination];
	}

	/*!
	 * Set the value of the Origin input for this Choreo.
	*(required, string) Enter the address or latitude/longitude coordinates from which directions will be computed (i.e."104 Franklin St, New York, NY" or "40.7160,-74.0037").
	 */
	-(void)setOrigin:(NSString*)Origin {
		[super setInput:@"Origin" toValue:Origin];
	}

	/*!
	 * Set the value of the Region input for this Choreo.
	*(optional, string) Enter the region code for the directions, specified as a ccTLD two-character value.
	 */
	-(void)setRegion:(NSString*)Region {
		[super setInput:@"Region" toValue:Region];
	}

	/*!
	 * Set the value of the Sensor input for this Choreo.
	*(optional, boolean) Indicates whether or not the directions request is from a device with a location sensor. Value must be either 1 or 0. Defaults to 0 (false).
	 */
	-(void)setSensor:(NSString*)Sensor {
		[super setInput:@"Sensor" toValue:Sensor];
	}

	/*!
	 * Set the value of the Units input for this Choreo.
	*(optional, string) Specify the units to be used when displaying results.  Options include, metric, or imperial.
	 */
	-(void)setUnits:(NSString*)Units {
		[super setInput:@"Units" toValue:Units];
	}

	/*!
	 * Set the value of the Waypoints input for this Choreo.
	*(optional, string) Specify route waypoints, either by address, or latitude/longitude coordinates.
	 */
	-(void)setWaypoints:(NSString*)Waypoints {
		[super setInput:@"Waypoints" toValue:Waypoints];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetBicyclingDirections Choreo.
 */
@implementation TMBGoogle_Directions_GetBicyclingDirections_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 *  Generate biking directions between two locations, denoted by address or latitude/longitude coordinates.
 */
@implementation TMBGoogle_Directions_GetBicyclingDirections

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Directions_GetBicyclingDirections Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Directions/GetBicyclingDirections"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Directions_GetBicyclingDirections_ResultSet *results = [[TMBGoogle_Directions_GetBicyclingDirections_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetBicyclingDirections Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Directions_GetBicyclingDirections_Inputs*)newInputSet {
		return [[TMBGoogle_Directions_GetBicyclingDirections_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetDrivingDirections Choreo.
 */
@implementation TMBGoogle_Directions_GetDrivingDirections_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) The API Key provided by Google.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the Alternatives input for this Choreo.
	*(optional, string) If set to true, additional routes will be returned.
	 */
	-(void)setAlternatives:(NSString*)Alternatives {
		[super setInput:@"Alternatives" toValue:Alternatives];
	}

	/*!
	 * Set the value of the Avoid input for this Choreo.
	*(optional, string) Specify whether the calculated directions should avoid the following features: tolls, or highways.
	 */
	-(void)setAvoid:(NSString*)Avoid {
		[super setInput:@"Avoid" toValue:Avoid];
	}

	/*!
	 * Set the value of the Destination input for this Choreo.
	*(required, string) Enter the address or latitude/longitude coordinates from which directions will be generated (i.e."104 Franklin St, New York, NY" or "40.7160,-74.0037").
	 */
	-(void)setDestination:(NSString*)Destination {
		[super setInput:@"Destination" toValue:Destination];
	}

	/*!
	 * Set the value of the Origin input for this Choreo.
	*(required, string) Enter the address or latitude/longitude coordinates from which directions will be computed (i.e."104 Franklin St, New York, NY" or "40.7160,-74.0037").
	 */
	-(void)setOrigin:(NSString*)Origin {
		[super setInput:@"Origin" toValue:Origin];
	}

	/*!
	 * Set the value of the Region input for this Choreo.
	*(optional, string) Enter the region code for the directions, specified as a ccTLD two-character value.
	 */
	-(void)setRegion:(NSString*)Region {
		[super setInput:@"Region" toValue:Region];
	}

	/*!
	 * Set the value of the Sensor input for this Choreo.
	*(optional, boolean) Indicates whether or not the directions request is from a device with a location sensor. Value must be either 1 or 0. Defaults to 0 (false).
	 */
	-(void)setSensor:(NSString*)Sensor {
		[super setInput:@"Sensor" toValue:Sensor];
	}

	/*!
	 * Set the value of the Units input for this Choreo.
	*(optional, string) Specify the units to be used when displaying results.  Options include, metric, or imperial.
	 */
	-(void)setUnits:(NSString*)Units {
		[super setInput:@"Units" toValue:Units];
	}

	/*!
	 * Set the value of the Waypoints input for this Choreo.
	*(optional, string) Specify route waypoints, either by address, or latitude/longitude coordinates.
	 */
	-(void)setWaypoints:(NSString*)Waypoints {
		[super setInput:@"Waypoints" toValue:Waypoints];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetDrivingDirections Choreo.
 */
@implementation TMBGoogle_Directions_GetDrivingDirections_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 *  Generates directions between two locations, denoted by address, or latitude/longitude coordinates.
 */
@implementation TMBGoogle_Directions_GetDrivingDirections

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Directions_GetDrivingDirections Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Directions/GetDrivingDirections"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Directions_GetDrivingDirections_ResultSet *results = [[TMBGoogle_Directions_GetDrivingDirections_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetDrivingDirections Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Directions_GetDrivingDirections_Inputs*)newInputSet {
		return [[TMBGoogle_Directions_GetDrivingDirections_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetWalkingDirections Choreo.
 */
@implementation TMBGoogle_Directions_GetWalkingDirections_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) The API Key provided by Google.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the Alternatives input for this Choreo.
	*(optional, string) If set to true, additional routes will be returned.
	 */
	-(void)setAlternatives:(NSString*)Alternatives {
		[super setInput:@"Alternatives" toValue:Alternatives];
	}

	/*!
	 * Set the value of the Destination input for this Choreo.
	*(required, string) Enter the address or latitude/longitude coordinates from which directions will be generated (i.e."104 Franklin St, New York, NY" or "40.7160,-74.0037").
	 */
	-(void)setDestination:(NSString*)Destination {
		[super setInput:@"Destination" toValue:Destination];
	}

	/*!
	 * Set the value of the Origin input for this Choreo.
	*(required, string) Enter the address or latitude/longitude coordinates from which directions will be computed (i.e."104 Franklin St, New York, NY" or "40.7160,-74.0037").
	 */
	-(void)setOrigin:(NSString*)Origin {
		[super setInput:@"Origin" toValue:Origin];
	}

	/*!
	 * Set the value of the Region input for this Choreo.
	*(optional, string) Enter the region code for the directions, specified as a ccTLD two-character value.
	 */
	-(void)setRegion:(NSString*)Region {
		[super setInput:@"Region" toValue:Region];
	}

	/*!
	 * Set the value of the Sensor input for this Choreo.
	*(optional, boolean) Indicates whether or not the directions request is from a device with a location sensor. Value must be either 1 or 0. Defaults to 0 (false).
	 */
	-(void)setSensor:(NSString*)Sensor {
		[super setInput:@"Sensor" toValue:Sensor];
	}

	/*!
	 * Set the value of the Units input for this Choreo.
	*(optional, string) Specify the units to be used when displaying results.  Options include, metric, or imperial.
	 */
	-(void)setUnits:(NSString*)Units {
		[super setInput:@"Units" toValue:Units];
	}

	/*!
	 * Set the value of the Waypoints input for this Choreo.
	*(optional, string) Specify route waypoints, either by address, or latitude/longitude coordinates.
	 */
	-(void)setWaypoints:(NSString*)Waypoints {
		[super setInput:@"Waypoints" toValue:Waypoints];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetWalkingDirections Choreo.
 */
@implementation TMBGoogle_Directions_GetWalkingDirections_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 *  Generate walking directions between two locations, denoted by address or latitude/longitude coordinates.
 */
@implementation TMBGoogle_Directions_GetWalkingDirections

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Directions_GetWalkingDirections Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Directions/GetWalkingDirections"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Directions_GetWalkingDirections_ResultSet *results = [[TMBGoogle_Directions_GetWalkingDirections_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetWalkingDirections Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Directions_GetWalkingDirections_Inputs*)newInputSet {
		return [[TMBGoogle_Directions_GetWalkingDirections_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the BicyclingDistanceMatrix Choreo.
 */
@implementation TMBGoogle_DistanceMatrix_BicyclingDistanceMatrix_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) The API Key provided by Google.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the Alternatives input for this Choreo.
	*(optional, string) If set to true, additional routes will be returned.
	 */
	-(void)setAlternatives:(NSString*)Alternatives {
		[super setInput:@"Alternatives" toValue:Alternatives];
	}

	/*!
	 * Set the value of the Destinations input for this Choreo.
	*(required, string) Enter the address or latitude/longitude coordinates to which directions will be generated. Multiple destinations can be separated by pipes (|).  For example: Boston, MA|New Haven|40.7160,-74.0037.
	 */
	-(void)setDestinations:(NSString*)Destinations {
		[super setInput:@"Destinations" toValue:Destinations];
	}

	/*!
	 * Set the value of the Language input for this Choreo.
	*(optional, string) Set the language in which to return restults.
	 */
	-(void)setLanguage:(NSString*)Language {
		[super setInput:@"Language" toValue:Language];
	}

	/*!
	 * Set the value of the Origins input for this Choreo.
	*(required, string) Enter the address(es) or geo-coordinates from which distance and time will be computed. Multiple destinations can be separated by pipes (|).  For example: Boston, MA|New Haven|40.7160,-74.0037.
	 */
	-(void)setOrigins:(NSString*)Origins {
		[super setInput:@"Origins" toValue:Origins];
	}

	/*!
	 * Set the value of the Region input for this Choreo.
	*(optional, string) Enter the region code for the directions, specified as a ccTLD two-character value.
	 */
	-(void)setRegion:(NSString*)Region {
		[super setInput:@"Region" toValue:Region];
	}

	/*!
	 * Set the value of the Sensor input for this Choreo.
	*(optional, boolean) Indicates whether or not the directions request is from a device with a location sensor. Value must be either 1 or 0. Defaults to 0 (false).
	 */
	-(void)setSensor:(NSString*)Sensor {
		[super setInput:@"Sensor" toValue:Sensor];
	}

	/*!
	 * Set the value of the Units input for this Choreo.
	*(optional, string) Specify the units to be used when displaying results.  Options include, metric, or imperial.
	 */
	-(void)setUnits:(NSString*)Units {
		[super setInput:@"Units" toValue:Units];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the BicyclingDistanceMatrix Choreo.
 */
@implementation TMBGoogle_DistanceMatrix_BicyclingDistanceMatrix_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Distance" output from an execution of this Choreo.
	 * @return - NSString* (integer) The distance of this route, expressed in meters.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getDistance {
		return [super getOutputByName:@"Distance"];
	}

	/*!
	 * Retrieve the value of the "Duration" output from an execution of this Choreo.
	 * @return - NSString* (integer) The duration of this route, expressed in seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getDuration {
		return [super getOutputByName:@"Duration"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Obtain bicycling distances and times for a matrix of addresses and/or latitude/longitude coordinates.
 */
@implementation TMBGoogle_DistanceMatrix_BicyclingDistanceMatrix

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_DistanceMatrix_BicyclingDistanceMatrix Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/DistanceMatrix/BicyclingDistanceMatrix"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_DistanceMatrix_BicyclingDistanceMatrix_ResultSet *results = [[TMBGoogle_DistanceMatrix_BicyclingDistanceMatrix_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the BicyclingDistanceMatrix Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_DistanceMatrix_BicyclingDistanceMatrix_Inputs*)newInputSet {
		return [[TMBGoogle_DistanceMatrix_BicyclingDistanceMatrix_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DrivingDistanceMatrix Choreo.
 */
@implementation TMBGoogle_DistanceMatrix_DrivingDistanceMatrix_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) The API Key provided by Google.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the Alternatives input for this Choreo.
	*(optional, string) If set to true, additional routes will be returned.
	 */
	-(void)setAlternatives:(NSString*)Alternatives {
		[super setInput:@"Alternatives" toValue:Alternatives];
	}

	/*!
	 * Set the value of the Avoid input for this Choreo.
	*(optional, string) Specify whether the calculated directions should avoid the following features: tolls, or highways.
	 */
	-(void)setAvoid:(NSString*)Avoid {
		[super setInput:@"Avoid" toValue:Avoid];
	}

	/*!
	 * Set the value of the Destinations input for this Choreo.
	*(required, string) Enter the address or latitude/longitude coordinates to which directions will be generated. Multiple destinations can be separated by pipes (|).  For example: Boston, MA|New Haven|40.7160,-74.0037.
	 */
	-(void)setDestinations:(NSString*)Destinations {
		[super setInput:@"Destinations" toValue:Destinations];
	}

	/*!
	 * Set the value of the Language input for this Choreo.
	*(optional, string) Set the language in which to return restults.  A list of supported languages is available here: https://spreadsheets.google.com/pub?key=p9pdwsai2hDMsLkXsoM05KQ&gid=1
	 */
	-(void)setLanguage:(NSString*)Language {
		[super setInput:@"Language" toValue:Language];
	}

	/*!
	 * Set the value of the Origins input for this Choreo.
	*(required, string) Enter the address(es) or geo-coordinates from which distance and time will be computed. Multiple destinations can be separated by pipes (|).  For example: Boston, MA|New Haven|40.7160,-74.0037.
	 */
	-(void)setOrigins:(NSString*)Origins {
		[super setInput:@"Origins" toValue:Origins];
	}

	/*!
	 * Set the value of the Region input for this Choreo.
	*(optional, string) Enter the region code for the directions, specified as a ccTLD two-character value.
	 */
	-(void)setRegion:(NSString*)Region {
		[super setInput:@"Region" toValue:Region];
	}

	/*!
	 * Set the value of the Sensor input for this Choreo.
	*(optional, boolean) Indicates whether or not the directions request is from a device with a location sensor. Value must be either 1 or 0. Defaults to 0 (false).
	 */
	-(void)setSensor:(NSString*)Sensor {
		[super setInput:@"Sensor" toValue:Sensor];
	}

	/*!
	 * Set the value of the Units input for this Choreo.
	*(optional, string) Specify the units to be used when displaying results.  Options include, metric, or imperial.
	 */
	-(void)setUnits:(NSString*)Units {
		[super setInput:@"Units" toValue:Units];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DrivingDistanceMatrix Choreo.
 */
@implementation TMBGoogle_DistanceMatrix_DrivingDistanceMatrix_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Distance" output from an execution of this Choreo.
	 * @return - NSString* (integer) The distance of this route, expressed in meters.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getDistance {
		return [super getOutputByName:@"Distance"];
	}

	/*!
	 * Retrieve the value of the "Duration" output from an execution of this Choreo.
	 * @return - NSString* (integer) The duration of this route, expressed in seconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getDuration {
		return [super getOutputByName:@"Duration"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Obtain travel distances and times for a matrix of addresses and/or latitude/longitude coordinates.
 */
@implementation TMBGoogle_DistanceMatrix_DrivingDistanceMatrix

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_DistanceMatrix_DrivingDistanceMatrix Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/DistanceMatrix/DrivingDistanceMatrix"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_DistanceMatrix_DrivingDistanceMatrix_ResultSet *results = [[TMBGoogle_DistanceMatrix_DrivingDistanceMatrix_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DrivingDistanceMatrix Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_DistanceMatrix_DrivingDistanceMatrix_Inputs*)newInputSet {
		return [[TMBGoogle_DistanceMatrix_DrivingDistanceMatrix_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the WalkingDistanceMatrix Choreo.
 */
@implementation TMBGoogle_DistanceMatrix_WalkingDistanceMatrix_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) The API Key provided by Google.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the Alternatives input for this Choreo.
	*(optional, string) If set to true, additional routes will be returned.
	 */
	-(void)setAlternatives:(NSString*)Alternatives {
		[super setInput:@"Alternatives" toValue:Alternatives];
	}

	/*!
	 * Set the value of the Destinations input for this Choreo.
	*(required, string) Enter the address or latitude/longitude coordinates to which directions will be generated. Multiple destinations can be separated by pipes (|).  For example: Boston, MA|New Haven|40.7160,-74.0037.
	 */
	-(void)setDestinations:(NSString*)Destinations {
		[super setInput:@"Destinations" toValue:Destinations];
	}

	/*!
	 * Set the value of the Language input for this Choreo.
	*(optional, string) Set the language in which to return restults.  A list of supported languages is available here: https://spreadsheets.google.com/pub?key=p9pdwsai2hDMsLkXsoM05KQ&gid=1
	 */
	-(void)setLanguage:(NSString*)Language {
		[super setInput:@"Language" toValue:Language];
	}

	/*!
	 * Set the value of the Origins input for this Choreo.
	*(required, string) Enter the address(es) or geo-coordinates from which distance and time will be computed. Multiple destinations can be separated by pipes (|).  For example: Boston, MA|New Haven|40.7160,-74.0037.
	 */
	-(void)setOrigins:(NSString*)Origins {
		[super setInput:@"Origins" toValue:Origins];
	}

	/*!
	 * Set the value of the Region input for this Choreo.
	*(optional, string) Enter the region code for the directions, specified as a ccTLD two-character value.
	 */
	-(void)setRegion:(NSString*)Region {
		[super setInput:@"Region" toValue:Region];
	}

	/*!
	 * Set the value of the Sensor input for this Choreo.
	*(optional, boolean) Indicates whether or not the directions request is from a device with a location sensor. Value must be either 1 or 0. Defaults to 0 (false).
	 */
	-(void)setSensor:(NSString*)Sensor {
		[super setInput:@"Sensor" toValue:Sensor];
	}

	/*!
	 * Set the value of the Units input for this Choreo.
	*(optional, string) Specify the units to be used when displaying results.  Options include, metric, or imperial.
	 */
	-(void)setUnits:(NSString*)Units {
		[super setInput:@"Units" toValue:Units];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the WalkingDistanceMatrix Choreo.
 */
@implementation TMBGoogle_DistanceMatrix_WalkingDistanceMatrix_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Distance" output from an execution of this Choreo.
	 * @return - NSString* (integer) The distance of this route, expressed in meters.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getDistance {
		return [super getOutputByName:@"Distance"];
	}

	/*!
	 * Retrieve the value of the "Duration" output from an execution of this Choreo.
	 * @return - NSString* (integer) The duration of this route, expressed in meters.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getDuration {
		return [super getOutputByName:@"Duration"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Obtain walking distances and times for a matrix of addresses and/or latitude/longitude coordinates.
 */
@implementation TMBGoogle_DistanceMatrix_WalkingDistanceMatrix

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_DistanceMatrix_WalkingDistanceMatrix Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/DistanceMatrix/WalkingDistanceMatrix"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_DistanceMatrix_WalkingDistanceMatrix_ResultSet *results = [[TMBGoogle_DistanceMatrix_WalkingDistanceMatrix_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the WalkingDistanceMatrix Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_DistanceMatrix_WalkingDistanceMatrix_Inputs*)newInputSet {
		return [[TMBGoogle_DistanceMatrix_WalkingDistanceMatrix_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Get Choreo.
 */
@implementation TMBGoogle_Drive_Changes_Get_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ChangeID input for this Choreo.
	*(required, string) The ID of the change.
	 */
	-(void)setChangeID:(NSString*)ChangeID {
		[super setInput:@"ChangeID" toValue:ChangeID];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying a subset of fields to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Get Choreo.
 */
@implementation TMBGoogle_Drive_Changes_Get_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Gets a specific change.
 */
@implementation TMBGoogle_Drive_Changes_Get

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Changes_Get Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Changes/Get"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Changes_Get_ResultSet *results = [[TMBGoogle_Drive_Changes_Get_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Get Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Changes_Get_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Changes_Get_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the List Choreo.
 */
@implementation TMBGoogle_Drive_Changes_List_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying a subset of fields to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the IncludeDeleted input for this Choreo.
	*(optional, boolean) Whether to include deleted items. (Default: true).
	 */
	-(void)setIncludeDeleted:(NSString*)IncludeDeleted {
		[super setInput:@"IncludeDeleted" toValue:IncludeDeleted];
	}

	/*!
	 * Set the value of the IncludeSubscribed input for this Choreo.
	*(optional, boolean) Whether to include shared files and public files the user opened. If false, the list includes owned files plus any shared or public files the user explictly added to a folder in Drive. (Default: true)
	 */
	-(void)setIncludeSubscribed:(NSString*)IncludeSubscribed {
		[super setInput:@"IncludeSubscribed" toValue:IncludeSubscribed];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, boolean) Maximum number of changes to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) Page token for changes.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the StartChangeID input for this Choreo.
	*(optional, multiline) Change ID to start listing changes from.
	 */
	-(void)setStartChangeID:(NSString*)StartChangeID {
		[super setInput:@"StartChangeID" toValue:StartChangeID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the List Choreo.
 */
@implementation TMBGoogle_Drive_Changes_List_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Lists the changes for a user.
 */
@implementation TMBGoogle_Drive_Changes_List

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Changes_List Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Changes/List"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Changes_List_ResultSet *results = [[TMBGoogle_Drive_Changes_List_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the List Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Changes_List_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Changes_List_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Delete Choreo.
 */
@implementation TMBGoogle_Drive_Children_Delete_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ChildID input for this Choreo.
	*(required, string) The ID of the child.
	 */
	-(void)setChildID:(NSString*)ChildID {
		[super setInput:@"ChildID" toValue:ChildID];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying a subset of fields to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the FolderID input for this Choreo.
	*(required, string) The ID of the folder.
	 */
	-(void)setFolderID:(NSString*)FolderID {
		[super setInput:@"FolderID" toValue:FolderID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Delete Choreo.
 */
@implementation TMBGoogle_Drive_Children_Delete_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Removes a child for a folder.
 */
@implementation TMBGoogle_Drive_Children_Delete

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Children_Delete Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Children/Delete"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Children_Delete_ResultSet *results = [[TMBGoogle_Drive_Children_Delete_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Delete Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Children_Delete_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Children_Delete_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Get Choreo.
 */
@implementation TMBGoogle_Drive_Children_Get_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ChildID input for this Choreo.
	*(required, string) The ID of the child.
	 */
	-(void)setChildID:(NSString*)ChildID {
		[super setInput:@"ChildID" toValue:ChildID];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying a subset of fields to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the FolderID input for this Choreo.
	*(required, string) The ID of the folder.
	 */
	-(void)setFolderID:(NSString*)FolderID {
		[super setInput:@"FolderID" toValue:FolderID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Get Choreo.
 */
@implementation TMBGoogle_Drive_Children_Get_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Gets a specific child reference.
 */
@implementation TMBGoogle_Drive_Children_Get

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Children_Get Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Children/Get"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Children_Get_ResultSet *results = [[TMBGoogle_Drive_Children_Get_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Get Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Children_Get_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Children_Get_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Insert Choreo.
 */
@implementation TMBGoogle_Drive_Children_Insert_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying a subset of fields to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the FolderID input for this Choreo.
	*(required, string) The ID of the folder.
	 */
	-(void)setFolderID:(NSString*)FolderID {
		[super setInput:@"FolderID" toValue:FolderID];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, string) The ID of the child.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Insert Choreo.
 */
@implementation TMBGoogle_Drive_Children_Insert_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Inserts a file into a folder.
 */
@implementation TMBGoogle_Drive_Children_Insert

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Children_Insert Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Children/Insert"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Children_Insert_ResultSet *results = [[TMBGoogle_Drive_Children_Insert_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Insert Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Children_Insert_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Children_Insert_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the List Choreo.
 */
@implementation TMBGoogle_Drive_Children_List_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying a subset of fields to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the FolderID input for this Choreo.
	*(required, string) The ID of the folder. To list all files in the root folder, specify the value "root".
	 */
	-(void)setFolderID:(NSString*)FolderID {
		[super setInput:@"FolderID" toValue:FolderID];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) Maximum number of children to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) Page token for children.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the Query input for this Choreo.
	*(optional, string) Query string for searching children.
	 */
	-(void)setQuery:(NSString*)Query {
		[super setInput:@"Query" toValue:Query];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the List Choreo.
 */
@implementation TMBGoogle_Drive_Children_List_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Lists the children of a specified folder.
 */
@implementation TMBGoogle_Drive_Children_List

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Children_List Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Children/List"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Children_List_ResultSet *results = [[TMBGoogle_Drive_Children_List_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the List Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Children_List_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Children_List_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Delete Choreo.
 */
@implementation TMBGoogle_Drive_Comments_Delete_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the CommentID input for this Choreo.
	*(required, string) The ID of the comment.
	 */
	-(void)setCommentID:(NSString*)CommentID {
		[super setInput:@"CommentID" toValue:CommentID];
	}

	/*!
	 * Set the value of the FileID input for this Choreo.
	*(required, string) The ID of the file.
	 */
	-(void)setFileID:(NSString*)FileID {
		[super setInput:@"FileID" toValue:FileID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Delete Choreo.
 */
@implementation TMBGoogle_Drive_Comments_Delete_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Deletes a comment.
 */
@implementation TMBGoogle_Drive_Comments_Delete

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Comments_Delete Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Comments/Delete"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Comments_Delete_ResultSet *results = [[TMBGoogle_Drive_Comments_Delete_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Delete Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Comments_Delete_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Comments_Delete_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Get Choreo.
 */
@implementation TMBGoogle_Drive_Comments_Get_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the CommentID input for this Choreo.
	*(required, string) The ID of the comment.
	 */
	-(void)setCommentID:(NSString*)CommentID {
		[super setInput:@"CommentID" toValue:CommentID];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying a subset of fields to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the FileID input for this Choreo.
	*(required, string) The ID of the file.
	 */
	-(void)setFileID:(NSString*)FileID {
		[super setInput:@"FileID" toValue:FileID];
	}

	/*!
	 * Set the value of the IncludeDeleted input for this Choreo.
	*(optional, boolean) If set, this will succeed when retrieving a deleted comment, and will include any deleted replies. (Default: false)
	 */
	-(void)setIncludeDeleted:(NSString*)IncludeDeleted {
		[super setInput:@"IncludeDeleted" toValue:IncludeDeleted];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Get Choreo.
 */
@implementation TMBGoogle_Drive_Comments_Get_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Gets a comment by ID.
 */
@implementation TMBGoogle_Drive_Comments_Get

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Comments_Get Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Comments/Get"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Comments_Get_ResultSet *results = [[TMBGoogle_Drive_Comments_Get_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Get Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Comments_Get_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Comments_Get_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Insert Choreo.
 */
@implementation TMBGoogle_Drive_Comments_Insert_Inputs

	/*!
	 * Set the value of the RequestBody input for this Choreo.
	*(required, json) A JSON representation of fields in a comment resource which shoud contain at least one key for content. See documentation for formatting examples.
	 */
	-(void)setRequestBody:(NSString*)RequestBody {
		[super setInput:@"RequestBody" toValue:RequestBody];
	}

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying a subset of fields to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the FileID input for this Choreo.
	*(required, string) The ID of the file.
	 */
	-(void)setFileID:(NSString*)FileID {
		[super setInput:@"FileID" toValue:FileID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Insert Choreo.
 */
@implementation TMBGoogle_Drive_Comments_Insert_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Creates a new comment on the given file.
 */
@implementation TMBGoogle_Drive_Comments_Insert

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Comments_Insert Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Comments/Insert"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Comments_Insert_ResultSet *results = [[TMBGoogle_Drive_Comments_Insert_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Insert Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Comments_Insert_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Comments_Insert_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the List Choreo.
 */
@implementation TMBGoogle_Drive_Comments_List_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying a subset of fields to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the FileID input for this Choreo.
	*(required, string) The ID of the file.
	 */
	-(void)setFileID:(NSString*)FileID {
		[super setInput:@"FileID" toValue:FileID];
	}

	/*!
	 * Set the value of the IncludeDeleted input for this Choreo.
	*(optional, boolean) If set, this will succeed when retrieving a deleted comment, and will include any deleted replies. (Default: false)
	 */
	-(void)setIncludeDeleted:(NSString*)IncludeDeleted {
		[super setInput:@"IncludeDeleted" toValue:IncludeDeleted];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, string) The maximum number of discussions to include in the response, used for paging. Acceptable values are 0 to 100, inclusive. (Default: 20)
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) The continuation token, used to page through large result sets. To get the next page of results, set this parameter to the value of "nextPageToken" from the previous response.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the UpdatedMIn input for this Choreo.
	*(optional, string) Only discussions that were updated after this timestamp will be returned. Formatted as an RFC 3339 timestamp.
	 */
	-(void)setUpdatedMIn:(NSString*)UpdatedMIn {
		[super setInput:@"UpdatedMIn" toValue:UpdatedMIn];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the List Choreo.
 */
@implementation TMBGoogle_Drive_Comments_List_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Lists a file's comments.
 */
@implementation TMBGoogle_Drive_Comments_List

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Comments_List Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Comments/List"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Comments_List_ResultSet *results = [[TMBGoogle_Drive_Comments_List_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the List Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Comments_List_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Comments_List_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Update Choreo.
 */
@implementation TMBGoogle_Drive_Comments_Update_Inputs

	/*!
	 * Set the value of the RequestBody input for this Choreo.
	*(required, json) A JSON representation of fields in a comment resource which shoud contain at least one key for content. See documentation for formatting examples.
	 */
	-(void)setRequestBody:(NSString*)RequestBody {
		[super setInput:@"RequestBody" toValue:RequestBody];
	}

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the CommentID input for this Choreo.
	*(required, string) The ID of the comment.
	 */
	-(void)setCommentID:(NSString*)CommentID {
		[super setInput:@"CommentID" toValue:CommentID];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying a subset of fields to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the FileID input for this Choreo.
	*(required, string) The ID of the file.
	 */
	-(void)setFileID:(NSString*)FileID {
		[super setInput:@"FileID" toValue:FileID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Update Choreo.
 */
@implementation TMBGoogle_Drive_Comments_Update_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Updates an existing comment.
 */
@implementation TMBGoogle_Drive_Comments_Update

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Comments_Update Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Comments/Update"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Comments_Update_ResultSet *results = [[TMBGoogle_Drive_Comments_Update_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Update Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Comments_Update_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Comments_Update_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Copy Choreo.
 */
@implementation TMBGoogle_Drive_Files_Copy_Inputs

	/*!
	 * Set the value of the RequestBody input for this Choreo.
	*(optional, json) A JSON representation of fields in a file resource. File metadata information (such as the title) can be set using this input. See documentation for formatting examples.
	 */
	-(void)setRequestBody:(NSString*)RequestBody {
		[super setInput:@"RequestBody" toValue:RequestBody];
	}

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Convert input for this Choreo.
	*(optional, boolean) Whether to convert this file to the corresponding Google Docs format. (Default: false).
	 */
	-(void)setConvert:(NSString*)Convert {
		[super setInput:@"Convert" toValue:Convert];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying a subset of fields to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the FileID input for this Choreo.
	*(required, string) The ID of the file to copy.
	 */
	-(void)setFileID:(NSString*)FileID {
		[super setInput:@"FileID" toValue:FileID];
	}

	/*!
	 * Set the value of the OCR input for this Choreo.
	*(optional, boolean) Whether to attempt OCR on .jpg, .png, .gif, or .pdf uploads. (Default: false)
	 */
	-(void)setOCR:(NSString*)OCR {
		[super setInput:@"OCR" toValue:OCR];
	}

	/*!
	 * Set the value of the OcrLanguage input for this Choreo.
	*(optional, string) If ocr is true, hints at the language to use. Valid values are ISO 639-1 codes.
	 */
	-(void)setOcrLanguage:(NSString*)OcrLanguage {
		[super setInput:@"OcrLanguage" toValue:OcrLanguage];
	}

	/*!
	 * Set the value of the Pinned input for this Choreo.
	*(optional, boolean) Whether to pin the head revision of the uploaded file. (Default: false).
	 */
	-(void)setPinned:(NSString*)Pinned {
		[super setInput:@"Pinned" toValue:Pinned];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the SourceLanguage input for this Choreo.
	*(optional, string) The language of the original file to be translated.
	 */
	-(void)setSourceLanguage:(NSString*)SourceLanguage {
		[super setInput:@"SourceLanguage" toValue:SourceLanguage];
	}

	/*!
	 * Set the value of the TargetLanguage input for this Choreo.
	*(optional, string) Target language to translate the file to. If no sourceLanguage is provided, the API will attempt to detect the language.
	 */
	-(void)setTargetLanguage:(NSString*)TargetLanguage {
		[super setInput:@"TargetLanguage" toValue:TargetLanguage];
	}

	/*!
	 * Set the value of the TimedTextLanguage input for this Choreo.
	*(optional, string) The language of the timed text.
	 */
	-(void)setTimedTextLanguage:(NSString*)TimedTextLanguage {
		[super setInput:@"TimedTextLanguage" toValue:TimedTextLanguage];
	}

	/*!
	 * Set the value of the TimedTextTrackName input for this Choreo.
	*(optional, string) The timed text track name.
	 */
	-(void)setTimedTextTrackName:(NSString*)TimedTextTrackName {
		[super setInput:@"TimedTextTrackName" toValue:TimedTextTrackName];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Copy Choreo.
 */
@implementation TMBGoogle_Drive_Files_Copy_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Creates a copy of the specified file.
 */
@implementation TMBGoogle_Drive_Files_Copy

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Files_Copy Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Files/Copy"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Files_Copy_ResultSet *results = [[TMBGoogle_Drive_Files_Copy_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Copy Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Files_Copy_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Files_Copy_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Delete Choreo.
 */
@implementation TMBGoogle_Drive_Files_Delete_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the FileID input for this Choreo.
	*(required, string) The ID of the file to delete.
	 */
	-(void)setFileID:(NSString*)FileID {
		[super setInput:@"FileID" toValue:FileID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Delete Choreo.
 */
@implementation TMBGoogle_Drive_Files_Delete_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Permanently deletes a file by ID.
 */
@implementation TMBGoogle_Drive_Files_Delete

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Files_Delete Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Files/Delete"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Files_Delete_ResultSet *results = [[TMBGoogle_Drive_Files_Delete_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Delete Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Files_Delete_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Files_Delete_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Get Choreo.
 */
@implementation TMBGoogle_Drive_Files_Get_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the ExportFormat input for this Choreo.
	*(optional, string) Indicates the download format (i.e. pdf, doc, txt, rtf, odt, etc). When specified, the FileContent output will contain the file's base64 encoded value.
	 */
	-(void)setExportFormat:(NSString*)ExportFormat {
		[super setInput:@"ExportFormat" toValue:ExportFormat];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying a subset of fields to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the FileID input for this Choreo.
	*(required, string) The ID of the file to retrieve.
	 */
	-(void)setFileID:(NSString*)FileID {
		[super setInput:@"FileID" toValue:FileID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the UpdateViewDate input for this Choreo.
	*(optional, boolean) Whether to update the view date after successfully retrieving the file. Default value is false.
	 */
	-(void)setUpdateViewDate:(NSString*)UpdateViewDate {
		[super setInput:@"UpdateViewDate" toValue:UpdateViewDate];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Get Choreo.
 */
@implementation TMBGoogle_Drive_Files_Get_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "FileContent" output from an execution of this Choreo.
	 * @return - NSString* (string) The Base64 encoded file content. Only returned when the optional "Format" parameter is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getFileContent {
		return [super getOutputByName:@"FileContent"];
	}

	/*!
	 * Retrieve the value of the "FileMetadata" output from an execution of this Choreo.
	 * @return - NSString* (json) The file metadata returned in the response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getFileMetadata {
		return [super getOutputByName:@"FileMetadata"];
	}

	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}
	
@end

/*!
 * Gets a file's metadata and content by ID.
 */
@implementation TMBGoogle_Drive_Files_Get

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Files_Get Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Files/Get"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Files_Get_ResultSet *results = [[TMBGoogle_Drive_Files_Get_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Get Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Files_Get_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Files_Get_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Insert Choreo.
 */
@implementation TMBGoogle_Drive_Files_Insert_Inputs

	/*!
	 * Set the value of the RequestBody input for this Choreo.
	*(conditional, json) A JSON representation of fields in a file resource. File metadata information (such as the title) can be inserted using this input. See documentation for formatting examples.
	 */
	-(void)setRequestBody:(NSString*)RequestBody {
		[super setInput:@"RequestBody" toValue:RequestBody];
	}

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the ContentType input for this Choreo.
	*(conditional, string) The Content-Type of the file that is being uploaded (i.e. image/jpg). Required when specifying the FileContent input.
	 */
	-(void)setContentType:(NSString*)ContentType {
		[super setInput:@"ContentType" toValue:ContentType];
	}

	/*!
	 * Set the value of the Convert input for this Choreo.
	*(optional, boolean) Whether to convert this file to the corresponding Google Docs format. (Default: false).
	 */
	-(void)setConvert:(NSString*)Convert {
		[super setInput:@"Convert" toValue:Convert];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying a subset of fields to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the FileContent input for this Choreo.
	*(conditional, string) The Base64 encoded contents of the file to upload.
	 */
	-(void)setFileContent:(NSString*)FileContent {
		[super setInput:@"FileContent" toValue:FileContent];
	}

	/*!
	 * Set the value of the OCR input for this Choreo.
	*(optional, boolean) Whether to attempt OCR on .jpg, .png, .gif, or .pdf uploads. (Default: false)
	 */
	-(void)setOCR:(NSString*)OCR {
		[super setInput:@"OCR" toValue:OCR];
	}

	/*!
	 * Set the value of the OcrLanguage input for this Choreo.
	*(optional, string) If ocr is true, hints at the language to use. Valid values are ISO 639-1 codes.
	 */
	-(void)setOcrLanguage:(NSString*)OcrLanguage {
		[super setInput:@"OcrLanguage" toValue:OcrLanguage];
	}

	/*!
	 * Set the value of the Pinned input for this Choreo.
	*(optional, boolean) Whether to pin the head revision of the uploaded file. (Default: false).
	 */
	-(void)setPinned:(NSString*)Pinned {
		[super setInput:@"Pinned" toValue:Pinned];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the SourceLanguage input for this Choreo.
	*(optional, string) The language of the original file to be translated.
	 */
	-(void)setSourceLanguage:(NSString*)SourceLanguage {
		[super setInput:@"SourceLanguage" toValue:SourceLanguage];
	}

	/*!
	 * Set the value of the TargetLanguage input for this Choreo.
	*(optional, string) Target language to translate the file to. If no sourceLanguage is provided, the API will attempt to detect the language.
	 */
	-(void)setTargetLanguage:(NSString*)TargetLanguage {
		[super setInput:@"TargetLanguage" toValue:TargetLanguage];
	}

	/*!
	 * Set the value of the TimedTextLanguage input for this Choreo.
	*(optional, string) The language of the timed text.
	 */
	-(void)setTimedTextLanguage:(NSString*)TimedTextLanguage {
		[super setInput:@"TimedTextLanguage" toValue:TimedTextLanguage];
	}

	/*!
	 * Set the value of the TimedTextTrackName input for this Choreo.
	*(optional, string) The timed text track name.
	 */
	-(void)setTimedTextTrackName:(NSString*)TimedTextTrackName {
		[super setInput:@"TimedTextTrackName" toValue:TimedTextTrackName];
	}

	/*!
	 * Set the value of the VaultFile input for this Choreo.
	*A path to a vault file to upload. This can be used as an alternative to the FileContent input.
	 */
	-(void)setVaultFile:(NSString*)VaultFile {
		[super setInput:@"VaultFile" toValue:VaultFile];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Insert Choreo.
 */
@implementation TMBGoogle_Drive_Files_Insert_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Inserts a new file.
 */
@implementation TMBGoogle_Drive_Files_Insert

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Files_Insert Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Files/Insert"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Files_Insert_ResultSet *results = [[TMBGoogle_Drive_Files_Insert_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Insert Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Files_Insert_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Files_Insert_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the List Choreo.
 */
@implementation TMBGoogle_Drive_Files_List_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying a subset of fields to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of results to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) The "nextPageToken" found in the response which is used to page through results.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the Query input for this Choreo.
	*(optional, string) A search query combining one or more search clauses (e.g. title='myFile.txt').
	 */
	-(void)setQuery:(NSString*)Query {
		[super setInput:@"Query" toValue:Query];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the List Choreo.
 */
@implementation TMBGoogle_Drive_Files_List_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "FileID" output from an execution of this Choreo.
	 * @return - NSString* (string) The id of the file returned. This is returned only when using the Query input to search for specific files.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getFileID {
		return [super getOutputByName:@"FileID"];
	}

	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Lists the user's files.
 */
@implementation TMBGoogle_Drive_Files_List

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Files_List Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Files/List"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Files_List_ResultSet *results = [[TMBGoogle_Drive_Files_List_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the List Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Files_List_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Files_List_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Touch Choreo.
 */
@implementation TMBGoogle_Drive_Files_Touch_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying a subset of fields to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the FileID input for this Choreo.
	*(required, string) The ID of the file to update.
	 */
	-(void)setFileID:(NSString*)FileID {
		[super setInput:@"FileID" toValue:FileID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Touch Choreo.
 */
@implementation TMBGoogle_Drive_Files_Touch_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Sets the file's updated time to the current server time.
 */
@implementation TMBGoogle_Drive_Files_Touch

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Files_Touch Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Files/Touch"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Files_Touch_ResultSet *results = [[TMBGoogle_Drive_Files_Touch_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Touch Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Files_Touch_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Files_Touch_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Trash Choreo.
 */
@implementation TMBGoogle_Drive_Files_Trash_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying a subset of fields to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the FileID input for this Choreo.
	*(required, string) The ID of the file to trash.
	 */
	-(void)setFileID:(NSString*)FileID {
		[super setInput:@"FileID" toValue:FileID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Trash Choreo.
 */
@implementation TMBGoogle_Drive_Files_Trash_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Moves a file to the trash.
 */
@implementation TMBGoogle_Drive_Files_Trash

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Files_Trash Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Files/Trash"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Files_Trash_ResultSet *results = [[TMBGoogle_Drive_Files_Trash_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Trash Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Files_Trash_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Files_Trash_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Untrash Choreo.
 */
@implementation TMBGoogle_Drive_Files_Untrash_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying a subset of fields to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the FileID input for this Choreo.
	*(required, string) The ID of the file to untrash.
	 */
	-(void)setFileID:(NSString*)FileID {
		[super setInput:@"FileID" toValue:FileID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Untrash Choreo.
 */
@implementation TMBGoogle_Drive_Files_Untrash_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Restores a file from the trash.
 */
@implementation TMBGoogle_Drive_Files_Untrash

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Files_Untrash Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Files/Untrash"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Files_Untrash_ResultSet *results = [[TMBGoogle_Drive_Files_Untrash_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Untrash Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Files_Untrash_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Files_Untrash_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Update Choreo.
 */
@implementation TMBGoogle_Drive_Files_Update_Inputs

	/*!
	 * Set the value of the RequestBody input for this Choreo.
	*(conditional, json) A JSON representation of fields in a file resource. File metadata information (such as the title) can be updated using this input. See documentation for formatting examples.
	 */
	-(void)setRequestBody:(NSString*)RequestBody {
		[super setInput:@"RequestBody" toValue:RequestBody];
	}

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the ContentType input for this Choreo.
	*(conditional, string) The Content-Type of the file that is being updated (i.e. image/jpeg). Required if modifying the file content.
	 */
	-(void)setContentType:(NSString*)ContentType {
		[super setInput:@"ContentType" toValue:ContentType];
	}

	/*!
	 * Set the value of the Convert input for this Choreo.
	*(optional, boolean) Whether to convert this file to the corresponding Google Docs format. (Default: false).
	 */
	-(void)setConvert:(NSString*)Convert {
		[super setInput:@"Convert" toValue:Convert];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying which fields to include in a partial response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the FileContent input for this Choreo.
	*(conditional, string) The new Base64 encoded contents of the file that is being updated.
	 */
	-(void)setFileContent:(NSString*)FileContent {
		[super setInput:@"FileContent" toValue:FileContent];
	}

	/*!
	 * Set the value of the FileID input for this Choreo.
	*(required, string) The id of the file to update.
	 */
	-(void)setFileID:(NSString*)FileID {
		[super setInput:@"FileID" toValue:FileID];
	}

	/*!
	 * Set the value of the OCR input for this Choreo.
	*(optional, boolean) Whether to attempt OCR on .jpg, .png, .gif, or .pdf uploads. (Default: false)
	 */
	-(void)setOCR:(NSString*)OCR {
		[super setInput:@"OCR" toValue:OCR];
	}

	/*!
	 * Set the value of the OcrLanguage input for this Choreo.
	*(optional, string) If ocr is true, hints at the language to use. Valid values are ISO 639-1 codes.
	 */
	-(void)setOcrLanguage:(NSString*)OcrLanguage {
		[super setInput:@"OcrLanguage" toValue:OcrLanguage];
	}

	/*!
	 * Set the value of the Pinned input for this Choreo.
	*(optional, boolean) Whether to pin the new revision. (Default: false).
	 */
	-(void)setPinned:(NSString*)Pinned {
		[super setInput:@"Pinned" toValue:Pinned];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the SetModifiedDate input for this Choreo.
	*(optional, boolean) Whether to set the modified date with the supplied modified date.
	 */
	-(void)setSetModifiedDate:(NSString*)SetModifiedDate {
		[super setInput:@"SetModifiedDate" toValue:SetModifiedDate];
	}

	/*!
	 * Set the value of the SourceLanguage input for this Choreo.
	*(optional, string) The language of the original file to be translated.
	 */
	-(void)setSourceLanguage:(NSString*)SourceLanguage {
		[super setInput:@"SourceLanguage" toValue:SourceLanguage];
	}

	/*!
	 * Set the value of the TargetLanguage input for this Choreo.
	*(optional, string) Target language to translate the file to. If no sourceLanguage is provided, the API will attempt to detect the language.
	 */
	-(void)setTargetLanguage:(NSString*)TargetLanguage {
		[super setInput:@"TargetLanguage" toValue:TargetLanguage];
	}

	/*!
	 * Set the value of the TimedTextLanguage input for this Choreo.
	*(optional, string) The language of the timed text.
	 */
	-(void)setTimedTextLanguage:(NSString*)TimedTextLanguage {
		[super setInput:@"TimedTextLanguage" toValue:TimedTextLanguage];
	}

	/*!
	 * Set the value of the TimedTextTrackName input for this Choreo.
	*(optional, string) The timed text track name.
	 */
	-(void)setTimedTextTrackName:(NSString*)TimedTextTrackName {
		[super setInput:@"TimedTextTrackName" toValue:TimedTextTrackName];
	}

	/*!
	 * Set the value of the UpdateViewedDate input for this Choreo.
	*(optional, boolean) Whether to update the view date after successfully updating the file.
	 */
	-(void)setUpdateViewedDate:(NSString*)UpdateViewedDate {
		[super setInput:@"UpdateViewedDate" toValue:UpdateViewedDate];
	}

	/*!
	 * Set the value of the VaultFile input for this Choreo.
	*A path to a vault file to upload. This can be used as an alternative to the FileContent input.
	 */
	-(void)setVaultFile:(NSString*)VaultFile {
		[super setInput:@"VaultFile" toValue:VaultFile];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Update Choreo.
 */
@implementation TMBGoogle_Drive_Files_Update_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Updates the metadata or content of an existing file.
 */
@implementation TMBGoogle_Drive_Files_Update

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Files_Update Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Files/Update"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Files_Update_ResultSet *results = [[TMBGoogle_Drive_Files_Update_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Update Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Files_Update_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Files_Update_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Delete Choreo.
 */
@implementation TMBGoogle_Drive_Parents_Delete_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the FileID input for this Choreo.
	*(required, string) The ID of the file.
	 */
	-(void)setFileID:(NSString*)FileID {
		[super setInput:@"FileID" toValue:FileID];
	}

	/*!
	 * Set the value of the ParentID input for this Choreo.
	*(required, string) The ID of the parent.
	 */
	-(void)setParentID:(NSString*)ParentID {
		[super setInput:@"ParentID" toValue:ParentID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Delete Choreo.
 */
@implementation TMBGoogle_Drive_Parents_Delete_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Removes a parent from a file.
 */
@implementation TMBGoogle_Drive_Parents_Delete

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Parents_Delete Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Parents/Delete"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Parents_Delete_ResultSet *results = [[TMBGoogle_Drive_Parents_Delete_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Delete Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Parents_Delete_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Parents_Delete_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Get Choreo.
 */
@implementation TMBGoogle_Drive_Parents_Get_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the FileID input for this Choreo.
	*(required, string) The ID of the file.
	 */
	-(void)setFileID:(NSString*)FileID {
		[super setInput:@"FileID" toValue:FileID];
	}

	/*!
	 * Set the value of the ParentID input for this Choreo.
	*(required, string) The ID of the parent.
	 */
	-(void)setParentID:(NSString*)ParentID {
		[super setInput:@"ParentID" toValue:ParentID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Get Choreo.
 */
@implementation TMBGoogle_Drive_Parents_Get_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Gets a specific parent reference.
 */
@implementation TMBGoogle_Drive_Parents_Get

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Parents_Get Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Parents/Get"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Parents_Get_ResultSet *results = [[TMBGoogle_Drive_Parents_Get_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Get Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Parents_Get_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Parents_Get_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Insert Choreo.
 */
@implementation TMBGoogle_Drive_Parents_Insert_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying a subset of fields to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the FileID input for this Choreo.
	*(required, string) The ID of the file.
	 */
	-(void)setFileID:(NSString*)FileID {
		[super setInput:@"FileID" toValue:FileID];
	}

	/*!
	 * Set the value of the ParentID input for this Choreo.
	*(required, string) The ID of the parent.
	 */
	-(void)setParentID:(NSString*)ParentID {
		[super setInput:@"ParentID" toValue:ParentID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Insert Choreo.
 */
@implementation TMBGoogle_Drive_Parents_Insert_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Adds a parent folder for a file.
 */
@implementation TMBGoogle_Drive_Parents_Insert

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Parents_Insert Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Parents/Insert"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Parents_Insert_ResultSet *results = [[TMBGoogle_Drive_Parents_Insert_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Insert Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Parents_Insert_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Parents_Insert_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the List Choreo.
 */
@implementation TMBGoogle_Drive_Parents_List_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the FileID input for this Choreo.
	*(required, string) The ID of the file.
	 */
	-(void)setFileID:(NSString*)FileID {
		[super setInput:@"FileID" toValue:FileID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the List Choreo.
 */
@implementation TMBGoogle_Drive_Parents_List_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Lists a file's parents.
 */
@implementation TMBGoogle_Drive_Parents_List

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Parents_List Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Parents/List"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Parents_List_ResultSet *results = [[TMBGoogle_Drive_Parents_List_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the List Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Parents_List_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Parents_List_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Delete Choreo.
 */
@implementation TMBGoogle_Drive_Permissions_Delete_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the FileID input for this Choreo.
	*(required, string) The ID of the file.
	 */
	-(void)setFileID:(NSString*)FileID {
		[super setInput:@"FileID" toValue:FileID];
	}

	/*!
	 * Set the value of the PermissionID input for this Choreo.
	*(required, string) The ID for the permission.
	 */
	-(void)setPermissionID:(NSString*)PermissionID {
		[super setInput:@"PermissionID" toValue:PermissionID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Delete Choreo.
 */
@implementation TMBGoogle_Drive_Permissions_Delete_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Deletes a permission from a file.
 */
@implementation TMBGoogle_Drive_Permissions_Delete

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Permissions_Delete Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Permissions/Delete"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Permissions_Delete_ResultSet *results = [[TMBGoogle_Drive_Permissions_Delete_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Delete Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Permissions_Delete_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Permissions_Delete_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Get Choreo.
 */
@implementation TMBGoogle_Drive_Permissions_Get_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying a subset of fields to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the FileID input for this Choreo.
	*(required, string) The ID of the file.
	 */
	-(void)setFileID:(NSString*)FileID {
		[super setInput:@"FileID" toValue:FileID];
	}

	/*!
	 * Set the value of the PermissionID input for this Choreo.
	*(required, string) The ID for the permission.
	 */
	-(void)setPermissionID:(NSString*)PermissionID {
		[super setInput:@"PermissionID" toValue:PermissionID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Get Choreo.
 */
@implementation TMBGoogle_Drive_Permissions_Get_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Gets a permission by ID.
 */
@implementation TMBGoogle_Drive_Permissions_Get

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Permissions_Get Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Permissions/Get"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Permissions_Get_ResultSet *results = [[TMBGoogle_Drive_Permissions_Get_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Get Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Permissions_Get_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Permissions_Get_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Insert Choreo.
 */
@implementation TMBGoogle_Drive_Permissions_Insert_Inputs

	/*!
	 * Set the value of the RequestBody input for this Choreo.
	*(required, json) A JSON representation of fields in a permissions resource. The JSON string must contain keys for role, type, and value. See documentation for formatting examples.
	 */
	-(void)setRequestBody:(NSString*)RequestBody {
		[super setInput:@"RequestBody" toValue:RequestBody];
	}

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying a subset of fields to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the FileID input for this Choreo.
	*(required, string) The ID of the file.
	 */
	-(void)setFileID:(NSString*)FileID {
		[super setInput:@"FileID" toValue:FileID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the SendNotificationEmails input for this Choreo.
	*(optional, boolean) Whether to send notification emails. (Default: true).
	 */
	-(void)setSendNotificationEmails:(NSString*)SendNotificationEmails {
		[super setInput:@"SendNotificationEmails" toValue:SendNotificationEmails];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Insert Choreo.
 */
@implementation TMBGoogle_Drive_Permissions_Insert_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Inserts a permission for a file.
 */
@implementation TMBGoogle_Drive_Permissions_Insert

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Permissions_Insert Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Permissions/Insert"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Permissions_Insert_ResultSet *results = [[TMBGoogle_Drive_Permissions_Insert_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Insert Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Permissions_Insert_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Permissions_Insert_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the List Choreo.
 */
@implementation TMBGoogle_Drive_Permissions_List_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying a subset of fields to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the FileID input for this Choreo.
	*(required, string) The ID of the file.
	 */
	-(void)setFileID:(NSString*)FileID {
		[super setInput:@"FileID" toValue:FileID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the List Choreo.
 */
@implementation TMBGoogle_Drive_Permissions_List_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Lists a file's permissions.
 */
@implementation TMBGoogle_Drive_Permissions_List

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Permissions_List Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Permissions/List"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Permissions_List_ResultSet *results = [[TMBGoogle_Drive_Permissions_List_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the List Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Permissions_List_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Permissions_List_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Update Choreo.
 */
@implementation TMBGoogle_Drive_Permissions_Update_Inputs

	/*!
	 * Set the value of the RequestBody input for this Choreo.
	*(required, json) A JSON representation of fields in a permissions resource. The JSON string can contain keys for role and additionalRoles. See documentation for formatting examples.
	 */
	-(void)setRequestBody:(NSString*)RequestBody {
		[super setInput:@"RequestBody" toValue:RequestBody];
	}

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying a subset of fields to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the FileID input for this Choreo.
	*(required, string) The ID of the file.
	 */
	-(void)setFileID:(NSString*)FileID {
		[super setInput:@"FileID" toValue:FileID];
	}

	/*!
	 * Set the value of the PermissionID input for this Choreo.
	*(required, string) The ID for the permission.
	 */
	-(void)setPermissionID:(NSString*)PermissionID {
		[super setInput:@"PermissionID" toValue:PermissionID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Update Choreo.
 */
@implementation TMBGoogle_Drive_Permissions_Update_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Updates an existing permission.
 */
@implementation TMBGoogle_Drive_Permissions_Update

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Permissions_Update Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Permissions/Update"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Permissions_Update_ResultSet *results = [[TMBGoogle_Drive_Permissions_Update_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Update Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Permissions_Update_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Permissions_Update_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Delete Choreo.
 */
@implementation TMBGoogle_Drive_Replies_Delete_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the CommentID input for this Choreo.
	*(required, string) The ID of the comment.
	 */
	-(void)setCommentID:(NSString*)CommentID {
		[super setInput:@"CommentID" toValue:CommentID];
	}

	/*!
	 * Set the value of the FileID input for this Choreo.
	*(required, string) The ID of the file.
	 */
	-(void)setFileID:(NSString*)FileID {
		[super setInput:@"FileID" toValue:FileID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ReplyID input for this Choreo.
	*(required, string) The ID of the reply.
	 */
	-(void)setReplyID:(NSString*)ReplyID {
		[super setInput:@"ReplyID" toValue:ReplyID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Delete Choreo.
 */
@implementation TMBGoogle_Drive_Replies_Delete_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Deletes a reply.
 */
@implementation TMBGoogle_Drive_Replies_Delete

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Replies_Delete Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Replies/Delete"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Replies_Delete_ResultSet *results = [[TMBGoogle_Drive_Replies_Delete_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Delete Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Replies_Delete_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Replies_Delete_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Get Choreo.
 */
@implementation TMBGoogle_Drive_Replies_Get_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the CommentID input for this Choreo.
	*(required, string) The ID of the comment.
	 */
	-(void)setCommentID:(NSString*)CommentID {
		[super setInput:@"CommentID" toValue:CommentID];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying a subset of fields to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the FileID input for this Choreo.
	*(required, string) The ID of the file.
	 */
	-(void)setFileID:(NSString*)FileID {
		[super setInput:@"FileID" toValue:FileID];
	}

	/*!
	 * Set the value of the IncludeDeleted input for this Choreo.
	*(optional, boolean) If set, this will succeed when retrieving a deleted reply. (Default: false)
	 */
	-(void)setIncludeDeleted:(NSString*)IncludeDeleted {
		[super setInput:@"IncludeDeleted" toValue:IncludeDeleted];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ReplyID input for this Choreo.
	*(required, string) The ID of the reply.
	 */
	-(void)setReplyID:(NSString*)ReplyID {
		[super setInput:@"ReplyID" toValue:ReplyID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Get Choreo.
 */
@implementation TMBGoogle_Drive_Replies_Get_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Gets a reply
 */
@implementation TMBGoogle_Drive_Replies_Get

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Replies_Get Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Replies/Get"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Replies_Get_ResultSet *results = [[TMBGoogle_Drive_Replies_Get_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Get Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Replies_Get_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Replies_Get_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Insert Choreo.
 */
@implementation TMBGoogle_Drive_Replies_Insert_Inputs

	/*!
	 * Set the value of the RequestBody input for this Choreo.
	*(required, json) A JSON representation of fields in a reply resource. The JSON string must contain a key for content or verb. See documentation for formatting examples.
	 */
	-(void)setRequestBody:(NSString*)RequestBody {
		[super setInput:@"RequestBody" toValue:RequestBody];
	}

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the CommentID input for this Choreo.
	*(required, string) The ID of the comment.
	 */
	-(void)setCommentID:(NSString*)CommentID {
		[super setInput:@"CommentID" toValue:CommentID];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying a subset of fields to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the FileID input for this Choreo.
	*(required, string) The ID of the file.
	 */
	-(void)setFileID:(NSString*)FileID {
		[super setInput:@"FileID" toValue:FileID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Insert Choreo.
 */
@implementation TMBGoogle_Drive_Replies_Insert_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Creates a new reply to the given comment.
 */
@implementation TMBGoogle_Drive_Replies_Insert

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Replies_Insert Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Replies/Insert"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Replies_Insert_ResultSet *results = [[TMBGoogle_Drive_Replies_Insert_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Insert Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Replies_Insert_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Replies_Insert_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the List Choreo.
 */
@implementation TMBGoogle_Drive_Replies_List_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the CommentID input for this Choreo.
	*(required, string) The ID of the comment.
	 */
	-(void)setCommentID:(NSString*)CommentID {
		[super setInput:@"CommentID" toValue:CommentID];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying a subset of fields to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the FileID input for this Choreo.
	*(required, string) The ID of the file.
	 */
	-(void)setFileID:(NSString*)FileID {
		[super setInput:@"FileID" toValue:FileID];
	}

	/*!
	 * Set the value of the IncludeDeleted input for this Choreo.
	*(optional, boolean) If set, this will succeed when retrieving a deleted reply. (Default: false)
	 */
	-(void)setIncludeDeleted:(NSString*)IncludeDeleted {
		[super setInput:@"IncludeDeleted" toValue:IncludeDeleted];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of replies to include in the response, used for paging. Acceptable values are 0 to 100, inclusive. (Default: 20).
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) The continuation token, used to page through large result sets. To get the next page of results, set this parameter to the value of "nextPageToken" from the previous response.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the List Choreo.
 */
@implementation TMBGoogle_Drive_Replies_List_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Lists all of the replies to a comment.
 */
@implementation TMBGoogle_Drive_Replies_List

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Replies_List Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Replies/List"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Replies_List_ResultSet *results = [[TMBGoogle_Drive_Replies_List_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the List Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Replies_List_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Replies_List_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Update Choreo.
 */
@implementation TMBGoogle_Drive_Replies_Update_Inputs

	/*!
	 * Set the value of the RequestBody input for this Choreo.
	*(required, json) A JSON representation of fields in a reply resource. The JSON string must contain a key for content. See documentation for formatting examples.
	 */
	-(void)setRequestBody:(NSString*)RequestBody {
		[super setInput:@"RequestBody" toValue:RequestBody];
	}

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the CommentID input for this Choreo.
	*(required, string) The ID of the comment.
	 */
	-(void)setCommentID:(NSString*)CommentID {
		[super setInput:@"CommentID" toValue:CommentID];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying a subset of fields to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the FileID input for this Choreo.
	*(required, string) The ID of the file.
	 */
	-(void)setFileID:(NSString*)FileID {
		[super setInput:@"FileID" toValue:FileID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ReplyID input for this Choreo.
	*(required, string) The ID of the reply.
	 */
	-(void)setReplyID:(NSString*)ReplyID {
		[super setInput:@"ReplyID" toValue:ReplyID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Update Choreo.
 */
@implementation TMBGoogle_Drive_Replies_Update_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Updates an existing reply.
 */
@implementation TMBGoogle_Drive_Replies_Update

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Replies_Update Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Replies/Update"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Replies_Update_ResultSet *results = [[TMBGoogle_Drive_Replies_Update_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Update Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Replies_Update_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Replies_Update_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Delete Choreo.
 */
@implementation TMBGoogle_Drive_Revisions_Delete_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the FileID input for this Choreo.
	*(required, string) The ID of the file.
	 */
	-(void)setFileID:(NSString*)FileID {
		[super setInput:@"FileID" toValue:FileID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the RevisionID input for this Choreo.
	*(required, string) The ID of the revision.
	 */
	-(void)setRevisionID:(NSString*)RevisionID {
		[super setInput:@"RevisionID" toValue:RevisionID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Delete Choreo.
 */
@implementation TMBGoogle_Drive_Revisions_Delete_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Removes a revision.
 */
@implementation TMBGoogle_Drive_Revisions_Delete

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Revisions_Delete Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Revisions/Delete"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Revisions_Delete_ResultSet *results = [[TMBGoogle_Drive_Revisions_Delete_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Delete Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Revisions_Delete_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Revisions_Delete_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Get Choreo.
 */
@implementation TMBGoogle_Drive_Revisions_Get_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying a subset of fields to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the FileID input for this Choreo.
	*(required, string) The ID of the file.
	 */
	-(void)setFileID:(NSString*)FileID {
		[super setInput:@"FileID" toValue:FileID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the RevisionID input for this Choreo.
	*(required, string) The ID of the revision.
	 */
	-(void)setRevisionID:(NSString*)RevisionID {
		[super setInput:@"RevisionID" toValue:RevisionID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Get Choreo.
 */
@implementation TMBGoogle_Drive_Revisions_Get_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Gets a specific revision.
 */
@implementation TMBGoogle_Drive_Revisions_Get

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Revisions_Get Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Revisions/Get"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Revisions_Get_ResultSet *results = [[TMBGoogle_Drive_Revisions_Get_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Get Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Revisions_Get_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Revisions_Get_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the List Choreo.
 */
@implementation TMBGoogle_Drive_Revisions_List_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying a subset of fields to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the FileID input for this Choreo.
	*(required, string) The ID of the file.
	 */
	-(void)setFileID:(NSString*)FileID {
		[super setInput:@"FileID" toValue:FileID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the List Choreo.
 */
@implementation TMBGoogle_Drive_Revisions_List_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Lists a file's revisions.
 */
@implementation TMBGoogle_Drive_Revisions_List

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Revisions_List Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Revisions/List"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Revisions_List_ResultSet *results = [[TMBGoogle_Drive_Revisions_List_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the List Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Revisions_List_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Revisions_List_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Update Choreo.
 */
@implementation TMBGoogle_Drive_Revisions_Update_Inputs

	/*!
	 * Set the value of the RequestBody input for this Choreo.
	*(required, json) A JSON representation of fields in a reply resource which shoud contain at least one key for pinned, publishAuto, published, or publishedOutsideDomain. See documentation for formatting examples.
	 */
	-(void)setRequestBody:(NSString*)RequestBody {
		[super setInput:@"RequestBody" toValue:RequestBody];
	}

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying a subset of fields to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the FileID input for this Choreo.
	*(required, string) The ID of the file.
	 */
	-(void)setFileID:(NSString*)FileID {
		[super setInput:@"FileID" toValue:FileID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the RevisionID input for this Choreo.
	*(required, string) The ID of the revision.
	 */
	-(void)setRevisionID:(NSString*)RevisionID {
		[super setInput:@"RevisionID" toValue:RevisionID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Update Choreo.
 */
@implementation TMBGoogle_Drive_Revisions_Update_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Updates a revision.
 */
@implementation TMBGoogle_Drive_Revisions_Update

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Drive_Revisions_Update Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Drive/Revisions/Update"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Drive_Revisions_Update_ResultSet *results = [[TMBGoogle_Drive_Revisions_Update_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Update Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Drive_Revisions_Update_Inputs*)newInputSet {
		return [[TMBGoogle_Drive_Revisions_Update_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetLocationElevation Choreo.
 */
@implementation TMBGoogle_Elevation_GetLocationElevation_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) The API Key provided by Google.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the Locations input for this Choreo.
	*(required, string) Enter the location(s) for which elevation data will be obtained.  Input formats: a single latitude/longitude coordinate pair; an array of coordinates separated by a |. A set of encoded coordinates.
	 */
	-(void)setLocations:(NSString*)Locations {
		[super setInput:@"Locations" toValue:Locations];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Sensor input for this Choreo.
	*(optional, boolean) Indicates whether or not the directions request is from a device with a location sensor. Value must be either 1 or 0. Defaults to 0 (false).
	 */
	-(void)setSensor:(NSString*)Sensor {
		[super setInput:@"Sensor" toValue:Sensor];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetLocationElevation Choreo.
 */
@implementation TMBGoogle_Elevation_GetLocationElevation_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (xml) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Obtain elevation information for a path generated by a set of geo-coordinates.
 */
@implementation TMBGoogle_Elevation_GetLocationElevation

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Elevation_GetLocationElevation Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Elevation/GetLocationElevation"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Elevation_GetLocationElevation_ResultSet *results = [[TMBGoogle_Elevation_GetLocationElevation_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetLocationElevation Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Elevation_GetLocationElevation_Inputs*)newInputSet {
		return [[TMBGoogle_Elevation_GetLocationElevation_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetPathElevation Choreo.
 */
@implementation TMBGoogle_Elevation_GetPathElevation_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) The API Key provided by Google.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the Path input for this Choreo.
	*(required, string) Specify the path for which elevation data will be obtained.  Input formats: an array of two or more lat/longitude coordinate pairs; A set of encoded coordinates using the Encoded Polyline Algorithm.
	 */
	-(void)setPath:(NSString*)Path {
		[super setInput:@"Path" toValue:Path];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Samples input for this Choreo.
	*(required, integer) Enter the number of sample points.  See API docs for additional information.
	 */
	-(void)setSamples:(NSString*)Samples {
		[super setInput:@"Samples" toValue:Samples];
	}

	/*!
	 * Set the value of the Sensor input for this Choreo.
	*(optional, boolean) Indicates whether or not the directions request is from a device with a location sensor. Value must be either 1 or 0. Defaults to 0 (false).
	 */
	-(void)setSensor:(NSString*)Sensor {
		[super setInput:@"Sensor" toValue:Sensor];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetPathElevation Choreo.
 */
@implementation TMBGoogle_Elevation_GetPathElevation_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (xml) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Obtain elevation information for a path specified by a set of  geo-coordinates.
 */
@implementation TMBGoogle_Elevation_GetPathElevation

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Elevation_GetPathElevation Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Elevation/GetPathElevation"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Elevation_GetPathElevation_ResultSet *results = [[TMBGoogle_Elevation_GetPathElevation_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetPathElevation Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Elevation_GetPathElevation_Inputs*)newInputSet {
		return [[TMBGoogle_Elevation_GetPathElevation_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GeocodeByAddress Choreo.
 */
@implementation TMBGoogle_Geocoding_GeocodeByAddress_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) The API Key provided by Google.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the Address input for this Choreo.
	*(required, string) The address that you want to geocode.
	 */
	-(void)setAddress:(NSString*)Address {
		[super setInput:@"Address" toValue:Address];
	}

	/*!
	 * Set the value of the Bounds input for this Choreo.
	*(optional, string) The bounding box of the viewport within which to bias geocode results more prominently.
	 */
	-(void)setBounds:(NSString*)Bounds {
		[super setInput:@"Bounds" toValue:Bounds];
	}

	/*!
	 * Set the value of the Language input for this Choreo.
	*(optional, string) The language in which to return results e.g.: es, fr, it, en (the default). See Choreo notes for a link to a full list of supported languages.
	 */
	-(void)setLanguage:(NSString*)Language {
		[super setInput:@"Language" toValue:Language];
	}

	/*!
	 * Set the value of the Region input for this Choreo.
	*(optional, string) The region code, specified as a ccTLD ("top-level domain") two-character value. Defaults to 'us' (United States).
	 */
	-(void)setRegion:(NSString*)Region {
		[super setInput:@"Region" toValue:Region];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are "xml" (the default) and "json".
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Sensor input for this Choreo.
	*(optional, boolean) Indicates whether or not the geocoding request is from a device with a location sensor. Value must be either 1 or 0. Defaults to 0 (false).
	 */
	-(void)setSensor:(NSString*)Sensor {
		[super setInput:@"Sensor" toValue:Sensor];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GeocodeByAddress Choreo.
 */
@implementation TMBGoogle_Geocoding_GeocodeByAddress_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Latitude" output from an execution of this Choreo.
	 * @return - NSString* (decimal) The latitude coordinate associated with the address provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLatitude {
		return [super getOutputByName:@"Latitude"];
	}

	/*!
	 * Retrieve the value of the "Longitude" output from an execution of this Choreo.
	 * @return - NSString* (decimal) The longitude coordinate associated with the address provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLongitude {
		return [super getOutputByName:@"Longitude"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Converts a human-readable address into geographic coordinates.
 */
@implementation TMBGoogle_Geocoding_GeocodeByAddress

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Geocoding_GeocodeByAddress Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Geocoding/GeocodeByAddress"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Geocoding_GeocodeByAddress_ResultSet *results = [[TMBGoogle_Geocoding_GeocodeByAddress_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GeocodeByAddress Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Geocoding_GeocodeByAddress_Inputs*)newInputSet {
		return [[TMBGoogle_Geocoding_GeocodeByAddress_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GeocodeByCoordinates Choreo.
 */
@implementation TMBGoogle_Geocoding_GeocodeByCoordinates_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) The API Key provided by Google.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the Bounds input for this Choreo.
	*(optional, string) The bounding box of the viewport within which to bias geocode results more prominently.
	 */
	-(void)setBounds:(NSString*)Bounds {
		[super setInput:@"Bounds" toValue:Bounds];
	}

	/*!
	 * Set the value of the Language input for this Choreo.
	*(optional, string) The language in which to return results. Defaults to 'en' (English).
	 */
	-(void)setLanguage:(NSString*)Language {
		[super setInput:@"Language" toValue:Language];
	}

	/*!
	 * Set the value of the Latitude input for this Choreo.
	*(required, decimal) The latitude value for which you wish to obtain the closest, human-readable address.
	 */
	-(void)setLatitude:(NSString*)Latitude {
		[super setInput:@"Latitude" toValue:Latitude];
	}

	/*!
	 * Set the value of the Longitude input for this Choreo.
	*(required, decimal) The longitude value for which you wish to obtain the closest, human-readable address.
	 */
	-(void)setLongitude:(NSString*)Longitude {
		[super setInput:@"Longitude" toValue:Longitude];
	}

	/*!
	 * Set the value of the Region input for this Choreo.
	*(optional, string) The region code, specified as a ccTLD ("top-level domain") two-character value. Defaults to 'us' (United States).
	 */
	-(void)setRegion:(NSString*)Region {
		[super setInput:@"Region" toValue:Region];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are "xml" (the default) and "json".
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Sensor input for this Choreo.
	*(optional, boolean) Indicates whether or not the geocoding request is from a device with a location sensor. Value must be either 1 or 0. Defaults to 0 (false).
	 */
	-(void)setSensor:(NSString*)Sensor {
		[super setInput:@"Sensor" toValue:Sensor];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GeocodeByCoordinates Choreo.
 */
@implementation TMBGoogle_Geocoding_GeocodeByCoordinates_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "FormattedAddress" output from an execution of this Choreo.
	 * @return - NSString* (string) The formatted address associated with the specified coordinates.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getFormattedAddress {
		return [super getOutputByName:@"FormattedAddress"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Converts latitude and longitude coordinates into a human-readable address.
 */
@implementation TMBGoogle_Geocoding_GeocodeByCoordinates

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Geocoding_GeocodeByCoordinates Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Geocoding/GeocodeByCoordinates"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Geocoding_GeocodeByCoordinates_ResultSet *results = [[TMBGoogle_Geocoding_GeocodeByCoordinates_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GeocodeByCoordinates Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Geocoding_GeocodeByCoordinates_Inputs*)newInputSet {
		return [[TMBGoogle_Geocoding_GeocodeByCoordinates_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetLatestUnreadEmail Choreo.
 */
@implementation TMBGoogle_Gmail_GetLatestUnreadEmail_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Label input for this Choreo.
	*(optional, string) The name of a Gmail Label to retrieve messages from (e.g., important, starred, sent, junk-e-mail, all).
	 */
	-(void)setLabel:(NSString*)Label {
		[super setInput:@"Label" toValue:Label];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(optional, password) A Google App-specific password that you've generated after enabling 2-Step Verification (Note: authenticating with OAuth credentials is the preferred authentication method).
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format for the response. Valid values are JSON and XML. This will be ignored when providng an XPath query because results are returned as a string or JSON depending on the Mode specified.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Username input for this Choreo.
	*(optional, string) Your full Google email address e.g., martha.temboo@gmail.com (Note: authenticating with OAuth credentials is the preferred authentication method).
	 */
	-(void)setUsername:(NSString*)Username {
		[super setInput:@"Username" toValue:Username];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetLatestUnreadEmail Choreo.
 */
@implementation TMBGoogle_Gmail_GetLatestUnreadEmail_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "AuthorEmail" output from an execution of this Choreo.
	 * @return - NSString* (string) The author's email address.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getAuthorEmail {
		return [super getOutputByName:@"AuthorEmail"];
	}

	/*!
	 * Retrieve the value of the "AuthorName" output from an execution of this Choreo.
	 * @return - NSString* (string) The author's name.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getAuthorName {
		return [super getOutputByName:@"AuthorName"];
	}

	/*!
	 * Retrieve the value of the "MessageBody" output from an execution of this Choreo.
	 * @return - NSString* (string) The email body. Note that this corresponds to the "summary" element in the feed.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getMessageBody {
		return [super getOutputByName:@"MessageBody"];
	}

	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Subject" output from an execution of this Choreo.
	 * @return - NSString* (string) The subject line of the email. Note that this corresponds to the "title" element in the feed.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getSubject {
		return [super getOutputByName:@"Subject"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Returns the latest unread email from a user's Gmail feed.
 */
@implementation TMBGoogle_Gmail_GetLatestUnreadEmail

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmail_GetLatestUnreadEmail Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmail/GetLatestUnreadEmail"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmail_GetLatestUnreadEmail_ResultSet *results = [[TMBGoogle_Gmail_GetLatestUnreadEmail_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetLatestUnreadEmail Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmail_GetLatestUnreadEmail_Inputs*)newInputSet {
		return [[TMBGoogle_Gmail_GetLatestUnreadEmail_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetUnreadImportantEmail Choreo.
 */
@implementation TMBGoogle_Gmail_GetUnreadImportantEmail_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(optional, password) A Google App-specific password that you've generated after enabling 2-Step Verification (Note: authenticating with OAuth credentials is the preferred authentication method).
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseMode input for this Choreo.
	*(optional, string) Used to simplify the response. Valid values are: simple and verbose. When set to simple, only the message string is returned. Verbose mode returns the full object. Defaults to "simple".
	 */
	-(void)setResponseMode:(NSString*)ResponseMode {
		[super setInput:@"ResponseMode" toValue:ResponseMode];
	}

	/*!
	 * Set the value of the Username input for this Choreo.
	*(optional, string) Your full Google email address e.g., martha.temboo@gmail.com (Note: authenticating with OAuth credentials is the preferred authentication method).
	 */
	-(void)setUsername:(NSString*)Username {
		[super setInput:@"Username" toValue:Username];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetUnreadImportantEmail Choreo.
 */
@implementation TMBGoogle_Gmail_GetUnreadImportantEmail_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Allows you to access a read-only Gmail feed that contains a list of unread emails that are marked important.
 */
@implementation TMBGoogle_Gmail_GetUnreadImportantEmail

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmail_GetUnreadImportantEmail Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmail/GetUnreadImportantEmail"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmail_GetUnreadImportantEmail_ResultSet *results = [[TMBGoogle_Gmail_GetUnreadImportantEmail_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetUnreadImportantEmail Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmail_GetUnreadImportantEmail_Inputs*)newInputSet {
		return [[TMBGoogle_Gmail_GetUnreadImportantEmail_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetUnreadMail Choreo.
 */
@implementation TMBGoogle_Gmail_GetUnreadMail_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Label input for this Choreo.
	*(optional, string) The name of a Gmail Label to retrieve messages from (e.g., important, starred, sent, junk-e-mail, all).
	 */
	-(void)setLabel:(NSString*)Label {
		[super setInput:@"Label" toValue:Label];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(optional, password) A Google App-specific password that you've generated after enabling 2-Step Verification (Note: authenticating with OAuth credentials is the preferred authentication method).
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseMode input for this Choreo.
	*(optional, string) Used to simplify the response. Valid values are: simple and verbose. When set to simple, only the message string is returned. Verbose mode returns the full object. Defaults to "simple".
	 */
	-(void)setResponseMode:(NSString*)ResponseMode {
		[super setInput:@"ResponseMode" toValue:ResponseMode];
	}

	/*!
	 * Set the value of the Username input for this Choreo.
	*(optional, string) Your full Google email address e.g., martha.temboo@gmail.com (Note: authenticating with OAuth credentials is the preferred authentication method).
	 */
	-(void)setUsername:(NSString*)Username {
		[super setInput:@"Username" toValue:Username];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetUnreadMail Choreo.
 */
@implementation TMBGoogle_Gmail_GetUnreadMail_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "FullCount" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of unread messages. This is parsed from the Google XML response. Note that when using the Label input to retrieve messages from a particular Gmail label, the full count element may be 0.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getFullCount {
		return [super getOutputByName:@"FullCount"];
	}

	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Allows you to access a read-only Gmail feed that contains a list of unread emails.
 */
@implementation TMBGoogle_Gmail_GetUnreadMail

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmail_GetUnreadMail Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmail/GetUnreadMail"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmail_GetUnreadMail_ResultSet *results = [[TMBGoogle_Gmail_GetUnreadMail_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetUnreadMail Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmail_GetUnreadMail_Inputs*)newInputSet {
		return [[TMBGoogle_Gmail_GetUnreadMail_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetUnreadMailFromSender Choreo.
 */
@implementation TMBGoogle_Gmail_GetUnreadMailFromSender_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Label input for this Choreo.
	*(optional, string) The name of a Gmail Label to retrieve messages from (e.g., important, starred, sent, junk-e-mail, all).
	 */
	-(void)setLabel:(NSString*)Label {
		[super setInput:@"Label" toValue:Label];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(optional, password) A Google App-specific password that you've generated after enabling 2-Step Verification (Note: authenticating with OAuth credentials is the preferred authentication method).
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseMode input for this Choreo.
	*(optional, string) Used to simplify the response. Valid values are: simple and verbose. When set to simple, only the message string is returned. Verbose mode returns the full object. Defaults to "simple".
	 */
	-(void)setResponseMode:(NSString*)ResponseMode {
		[super setInput:@"ResponseMode" toValue:ResponseMode];
	}

	/*!
	 * Set the value of the SenderEmail input for this Choreo.
	*(required, string) Email address of the sender you wish to filter by.
	 */
	-(void)setSenderEmail:(NSString*)SenderEmail {
		[super setInput:@"SenderEmail" toValue:SenderEmail];
	}

	/*!
	 * Set the value of the Username input for this Choreo.
	*(optional, string) Your full Google email address e.g., martha.temboo@gmail.com (Note: authenticating with OAuth credentials is the preferred authentication method).
	 */
	-(void)setUsername:(NSString*)Username {
		[super setInput:@"Username" toValue:Username];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetUnreadMailFromSender Choreo.
 */
@implementation TMBGoogle_Gmail_GetUnreadMailFromSender_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "FullCount" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of unread messages. This is parsed from the Google XML response. Note that when using the Label input to retrieve messages from a particular Gmail label, the full count element may be 0.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getFullCount {
		return [super getOutputByName:@"FullCount"];
	}

	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Allows you to access a read-only Gmail feed that contains a list of unread emails for the specified sender.
 */
@implementation TMBGoogle_Gmail_GetUnreadMailFromSender

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmail_GetUnreadMailFromSender Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmail/GetUnreadMailFromSender"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmail_GetUnreadMailFromSender_ResultSet *results = [[TMBGoogle_Gmail_GetUnreadMailFromSender_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetUnreadMailFromSender Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmail_GetUnreadMailFromSender_Inputs*)newInputSet {
		return [[TMBGoogle_Gmail_GetUnreadMailFromSender_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetUnreadMailWithLabel Choreo.
 */
@implementation TMBGoogle_Gmail_GetUnreadMailWithLabel_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Label input for this Choreo.
	*(required, string) The name of a Gmail Label to retrieve messages from (e.g., important, starred, sent, junk-e-mail, all).
	 */
	-(void)setLabel:(NSString*)Label {
		[super setInput:@"Label" toValue:Label];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(optional, password) A Google App-specific password that you've generated after enabling 2-Step Verification (Note: authenticating with OAuth credentials is the preferred authentication method).
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseMode input for this Choreo.
	*(optional, string) Used to simplify the response. Valid values are: simple and verbose. When set to simple, only the message string is returned. Verbose mode returns the full object. Defaults to "simple".
	 */
	-(void)setResponseMode:(NSString*)ResponseMode {
		[super setInput:@"ResponseMode" toValue:ResponseMode];
	}

	/*!
	 * Set the value of the Username input for this Choreo.
	*(optional, string) Your full Google email address e.g., martha.temboo@gmail.com (Note: authenticating with OAuth credentials is the preferred authentication method).
	 */
	-(void)setUsername:(NSString*)Username {
		[super setInput:@"Username" toValue:Username];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetUnreadMailWithLabel Choreo.
 */
@implementation TMBGoogle_Gmail_GetUnreadMailWithLabel_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "FullCount" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of unread messages. This is parsed from the Google XML response. Note the full count element may be 0 because this Choreography retrieves Gmail messages by a particular Label.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getFullCount {
		return [super getOutputByName:@"FullCount"];
	}

	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Allows you to access a read-only Gmail feed that contains a list of unread emails with the specified label.
 */
@implementation TMBGoogle_Gmail_GetUnreadMailWithLabel

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmail_GetUnreadMailWithLabel Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmail/GetUnreadMailWithLabel"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmail_GetUnreadMailWithLabel_ResultSet *results = [[TMBGoogle_Gmail_GetUnreadMailWithLabel_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetUnreadMailWithLabel Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmail_GetUnreadMailWithLabel_Inputs*)newInputSet {
		return [[TMBGoogle_Gmail_GetUnreadMailWithLabel_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the InboxFeed Choreo.
 */
@implementation TMBGoogle_Gmail_InboxFeed_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Label input for this Choreo.
	*(optional, string) The name of a Gmail Label to retrieve messages from (e.g., important, starred, sent, junk-e-mail, all).
	 */
	-(void)setLabel:(NSString*)Label {
		[super setInput:@"Label" toValue:Label];
	}

	/*!
	 * Set the value of the Mode input for this Choreo.
	*(optional, string) Used when an XPath query is provided. Valid values are "select" or "recursive". Select mode will return the first match of the query. In recursive mode, the XPath query will be applied within a loop.
	 */
	-(void)setMode:(NSString*)Mode {
		[super setInput:@"Mode" toValue:Mode];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(optional, password) A Google App-specific password that you've generated after enabling 2-Step Verification (Note: authenticating with OAuth credentials is the preferred authentication method).
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format for the response. Valid values are JSON and XML. This will be ignored when providng an XPath query because results are returned as a string or JSON depending on the Mode specified.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Username input for this Choreo.
	*(optional, string) Your full Google email address e.g., martha.temboo@gmail.com (Note: authenticating with OAuth credentials is the preferred authentication method).
	 */
	-(void)setUsername:(NSString*)Username {
		[super setInput:@"Username" toValue:Username];
	}

	/*!
	 * Set the value of the XPath input for this Choreo.
	*(optional, string) An XPATH query to run.
	 */
	-(void)setXPath:(NSString*)XPath {
		[super setInput:@"XPath" toValue:XPath];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the InboxFeed Choreo.
 */
@implementation TMBGoogle_Gmail_InboxFeed_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "FullCount" output from an execution of this Choreo.
	 * @return - NSString* (integer) The number of unread messages. This is parsed from the Google XML response. Note that when using the Label input to retrieve messages from a particular Gmail label, the full count element may be 0.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getFullCount {
		return [super getOutputByName:@"FullCount"];
	}

	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Google. This will contain the data from the Gmail feed, or if the XPath input is provided, it will contain the result of the XPath query.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Allows you to access a read-only Gmail feed that contains a list of unread emails.
 */
@implementation TMBGoogle_Gmail_InboxFeed

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmail_InboxFeed Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmail/InboxFeed"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmail_InboxFeed_ResultSet *results = [[TMBGoogle_Gmail_InboxFeed_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the InboxFeed Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmail_InboxFeed_Inputs*)newInputSet {
		return [[TMBGoogle_Gmail_InboxFeed_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SendEmail Choreo.
 */
@implementation TMBGoogle_Gmail_SendEmail_Inputs

	/*!
	 * Set the value of the Attachment input for this Choreo.
	*(optional, string) The Base64 encoded contents of the file to attach to the email.  Use this instead of AttachmentURL.
	 */
	-(void)setAttachment:(NSString*)Attachment {
		[super setInput:@"Attachment" toValue:Attachment];
	}

	/*!
	 * Set the value of the AttachmentName input for this Choreo.
	*(optional, string) The name of the file to attach to the email.
	 */
	-(void)setAttachmentName:(NSString*)AttachmentName {
		[super setInput:@"AttachmentName" toValue:AttachmentName];
	}

	/*!
	 * Set the value of the AttachmentURL input for this Choreo.
	*(optional, string) URL of a hosted file that you wish to add as an attachment.  Use this instead of a normal Attachment.
	 */
	-(void)setAttachmentURL:(NSString*)AttachmentURL {
		[super setInput:@"AttachmentURL" toValue:AttachmentURL];
	}

	/*!
	 * Set the value of the BCC input for this Choreo.
	*(optional, string) An email address to BCC on the email you're sending. Can be a comma separated list of email addresses.
	 */
	-(void)setBCC:(NSString*)BCC {
		[super setInput:@"BCC" toValue:BCC];
	}

	/*!
	 * Set the value of the CC input for this Choreo.
	*(optional, string) An email address to CC on the email you're sending. Can be a comma separated list of email addresses.
	 */
	-(void)setCC:(NSString*)CC {
		[super setInput:@"CC" toValue:CC];
	}

	/*!
	 * Set the value of the FromAddress input for this Choreo.
	*(conditional, string) The address and name (optional) that the email is being sent from e.g., "Dan" <dan@temboo.com>
	 */
	-(void)setFromAddress:(NSString*)FromAddress {
		[super setInput:@"FromAddress" toValue:FromAddress];
	}

	/*!
	 * Set the value of the MessageBody input for this Choreo.
	*(required, string) The message body for the email.
	 */
	-(void)setMessageBody:(NSString*)MessageBody {
		[super setInput:@"MessageBody" toValue:MessageBody];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) A Google App-specific password that you've generated after enabling 2-Step Verification. See the Gmailv2 bundle for OAuth.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Subject input for this Choreo.
	*(required, string) The subject line of the email.
	 */
	-(void)setSubject:(NSString*)Subject {
		[super setInput:@"Subject" toValue:Subject];
	}

	/*!
	 * Set the value of the ToAddress input for this Choreo.
	*(required, string) The email address that you want to send an email to. Can be a comma separated list of email addresses.
	 */
	-(void)setToAddress:(NSString*)ToAddress {
		[super setInput:@"ToAddress" toValue:ToAddress];
	}

	/*!
	 * Set the value of the Username input for this Choreo.
	*(required, string) Your full Google email address e.g., martha.temboo@gmail.com. See the Gmailv2 bundle for OAuth.
	 */
	-(void)setUsername:(NSString*)Username {
		[super setInput:@"Username" toValue:Username];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SendEmail Choreo.
 */
@implementation TMBGoogle_Gmail_SendEmail_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Success" output from an execution of this Choreo.
	 * @return - NSString* (boolean) Indicates the result of the SMTP operation. The value will be "true" for a successful request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getSuccess {
		return [super getOutputByName:@"Success"];
	}
	
@end

/*!
 * Sends an email using a specified Gmail account.
 */
@implementation TMBGoogle_Gmail_SendEmail

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmail_SendEmail Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmail/SendEmail"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmail_SendEmail_ResultSet *results = [[TMBGoogle_Gmail_SendEmail_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SendEmail Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmail_SendEmail_Inputs*)newInputSet {
		return [[TMBGoogle_Gmail_SendEmail_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetAttachment Choreo.
 */
@implementation TMBGoogle_Gmailv2_Attachments_GetAttachment_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AttachmentID input for this Choreo.
	*(required, string) The ID of the attachment to retrieve.
	 */
	-(void)setAttachmentID:(NSString*)AttachmentID {
		[super setInput:@"AttachmentID" toValue:AttachmentID];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying a subset of fields to include in the response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Format input for this Choreo.
	*(optional, boolean) Specifies the format of the message returned. Valid values are: full (the default), minimal, and raw. See Choreo notes for more details about these formats.
	 */
	-(void)setFormat:(NSString*)Format {
		[super setInput:@"Format" toValue:Format];
	}

	/*!
	 * Set the value of the MessageID input for this Choreo.
	*(required, string) The ID of the message to retrieve.
	 */
	-(void)setMessageID:(NSString*)MessageID {
		[super setInput:@"MessageID" toValue:MessageID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The id of the acting user. Defaults to "me" indicating the user associated with the access token or refresh token provided.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetAttachment Choreo.
 */
@implementation TMBGoogle_Gmailv2_Attachments_GetAttachment_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Gets the specified message attachment.
 */
@implementation TMBGoogle_Gmailv2_Attachments_GetAttachment

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmailv2_Attachments_GetAttachment Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmailv2/Attachments/GetAttachment"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmailv2_Attachments_GetAttachment_ResultSet *results = [[TMBGoogle_Gmailv2_Attachments_GetAttachment_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetAttachment Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmailv2_Attachments_GetAttachment_Inputs*)newInputSet {
		return [[TMBGoogle_Gmailv2_Attachments_GetAttachment_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CreateDraft Choreo.
 */
@implementation TMBGoogle_Gmailv2_Drafts_CreateDraft_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AttachmentContentType input for this Choreo.
	*(optional, string) The Content-Type of the attachment. This is required when providing AttachmentFileContent (e.g., image/jpeg, text/plain, etc).
	 */
	-(void)setAttachmentContentType:(NSString*)AttachmentContentType {
		[super setInput:@"AttachmentContentType" toValue:AttachmentContentType];
	}

	/*!
	 * Set the value of the AttachmentFileContent input for this Choreo.
	*(optional, string) The Base64 encoded file content for the attachment. You must specify the AttachmentFileContentType when including an attachment.
	 */
	-(void)setAttachmentFileContent:(NSString*)AttachmentFileContent {
		[super setInput:@"AttachmentFileContent" toValue:AttachmentFileContent];
	}

	/*!
	 * Set the value of the AttachmentFileName input for this Choreo.
	*(optional, string) The file name of the attachment.
	 */
	-(void)setAttachmentFileName:(NSString*)AttachmentFileName {
		[super setInput:@"AttachmentFileName" toValue:AttachmentFileName];
	}

	/*!
	 * Set the value of the BCC input for this Choreo.
	*(optional, string) The address and name (optional) that should be bcc'd e.g., Dan <dan@temboo.com>.
	 */
	-(void)setBCC:(NSString*)BCC {
		[super setInput:@"BCC" toValue:BCC];
	}

	/*!
	 * Set the value of the CC input for this Choreo.
	*(optional, string) The address and name (optional) that should be cc'd e.g., Dan <dan@temboo.com>.
	 */
	-(void)setCC:(NSString*)CC {
		[super setInput:@"CC" toValue:CC];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Used to specify fields to include in a partial response. This can be used to reduce the amount of data returned. See Choreo notes for syntax rules.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the From input for this Choreo.
	*(conditional, string) The address and name (optional) that the email is being sent from e.g., Dan <dan@temboo.com>.
	 */
	-(void)setFrom:(NSString*)From {
		[super setInput:@"From" toValue:From];
	}

	/*!
	 * Set the value of the MessageBody input for this Choreo.
	*(conditional, string) The text for the message body of the email.
	 */
	-(void)setMessageBody:(NSString*)MessageBody {
		[super setInput:@"MessageBody" toValue:MessageBody];
	}

	/*!
	 * Set the value of the MessageBodyContentType input for this Choreo.
	*(optional, string) The Content-Type of the message body. Defaults to "text/plain; charset=UTF-8".
	 */
	-(void)setMessageBodyContentType:(NSString*)MessageBodyContentType {
		[super setInput:@"MessageBodyContentType" toValue:MessageBodyContentType];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ReplyTo input for this Choreo.
	*(optional, string) An email address to set as the Reply-To address.
	 */
	-(void)setReplyTo:(NSString*)ReplyTo {
		[super setInput:@"ReplyTo" toValue:ReplyTo];
	}

	/*!
	 * Set the value of the Subject input for this Choreo.
	*(conditional, string) The email subject.
	 */
	-(void)setSubject:(NSString*)Subject {
		[super setInput:@"Subject" toValue:Subject];
	}

	/*!
	 * Set the value of the ThreadID input for this Choreo.
	*(optional, string) The ID of the thread the message belongs to.
	 */
	-(void)setThreadID:(NSString*)ThreadID {
		[super setInput:@"ThreadID" toValue:ThreadID];
	}

	/*!
	 * Set the value of the To input for this Choreo.
	*(conditional, string) The address and name (optional) that the email is being sent to e.g., Dan <dan@temboo.com>.
	 */
	-(void)setTo:(NSString*)To {
		[super setInput:@"To" toValue:To];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The ID of the acting user. Defaults to "me" indicating the user associated with the Access Token or Refresh Token provided.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateDraft Choreo.
 */
@implementation TMBGoogle_Gmailv2_Drafts_CreateDraft_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Creates a new draft with the DRAFT label.
 */
@implementation TMBGoogle_Gmailv2_Drafts_CreateDraft

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmailv2_Drafts_CreateDraft Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmailv2/Drafts/CreateDraft"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmailv2_Drafts_CreateDraft_ResultSet *results = [[TMBGoogle_Gmailv2_Drafts_CreateDraft_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CreateDraft Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmailv2_Drafts_CreateDraft_Inputs*)newInputSet {
		return [[TMBGoogle_Gmailv2_Drafts_CreateDraft_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteDraft Choreo.
 */
@implementation TMBGoogle_Gmailv2_Drafts_DeleteDraft_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the DraftID input for this Choreo.
	*(required, string) The ID of the draft to delete.
	 */
	-(void)setDraftID:(NSString*)DraftID {
		[super setInput:@"DraftID" toValue:DraftID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The id of the acting user. Defaults to "me" indicating the user associated with the access token or refresh token provided.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteDraft Choreo.
 */
@implementation TMBGoogle_Gmailv2_Drafts_DeleteDraft_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "ResponseStatusCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) The response status code returned from Google. A 204 is expected for a successful delete operation.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponseStatusCode {
		return [super getOutputByName:@"ResponseStatusCode"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google. An empty response is expected for a successful delete operation.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Permanently deletes the specified draft.
 */
@implementation TMBGoogle_Gmailv2_Drafts_DeleteDraft

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmailv2_Drafts_DeleteDraft Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmailv2/Drafts/DeleteDraft"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmailv2_Drafts_DeleteDraft_ResultSet *results = [[TMBGoogle_Gmailv2_Drafts_DeleteDraft_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteDraft Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmailv2_Drafts_DeleteDraft_Inputs*)newInputSet {
		return [[TMBGoogle_Gmailv2_Drafts_DeleteDraft_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetDraft Choreo.
 */
@implementation TMBGoogle_Gmailv2_Drafts_GetDraft_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the DraftID input for this Choreo.
	*(required, string) The ID of the draft to retrieve.
	 */
	-(void)setDraftID:(NSString*)DraftID {
		[super setInput:@"DraftID" toValue:DraftID];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Used to specify fields to include in a partial response. This can be used to reduce the amount of data returned. See Choreo notes for syntax rules.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Format input for this Choreo.
	*(optional, string) Specifies the format of the draft returned. Valid values are: full (the default), minimal, and raw. See Choreo notes for more details about these formats.
	 */
	-(void)setFormat:(NSString*)Format {
		[super setInput:@"Format" toValue:Format];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The ID of the acting user. Defaults to "me" indicating the user associated with the Access Token or Refresh Token provided.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetDraft Choreo.
 */
@implementation TMBGoogle_Gmailv2_Drafts_GetDraft_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a specific draft from a user's mailbox.
 */
@implementation TMBGoogle_Gmailv2_Drafts_GetDraft

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmailv2_Drafts_GetDraft Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmailv2/Drafts/GetDraft"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmailv2_Drafts_GetDraft_ResultSet *results = [[TMBGoogle_Gmailv2_Drafts_GetDraft_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetDraft Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmailv2_Drafts_GetDraft_Inputs*)newInputSet {
		return [[TMBGoogle_Gmailv2_Drafts_GetDraft_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListDrafts Choreo.
 */
@implementation TMBGoogle_Gmailv2_Drafts_ListDrafts_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Used to specify fields to include in a partial response. This can be used to reduce the amount of data returned. See Choreo notes for syntax rules.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of results to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) The "nextPageToken" found in the response which is used to page through results.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The ID of the acting user. Defaults to "me" indicating the user associated with the Access Token or Refresh Token provided.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListDrafts Choreo.
 */
@implementation TMBGoogle_Gmailv2_Drafts_ListDrafts_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Lists the drafts in the user's mailbox.
 */
@implementation TMBGoogle_Gmailv2_Drafts_ListDrafts

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmailv2_Drafts_ListDrafts Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmailv2/Drafts/ListDrafts"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmailv2_Drafts_ListDrafts_ResultSet *results = [[TMBGoogle_Gmailv2_Drafts_ListDrafts_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListDrafts Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmailv2_Drafts_ListDrafts_Inputs*)newInputSet {
		return [[TMBGoogle_Gmailv2_Drafts_ListDrafts_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SendDraft Choreo.
 */
@implementation TMBGoogle_Gmailv2_Drafts_SendDraft_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the DraftID input for this Choreo.
	*(required, string) The ID of the draft to send.
	 */
	-(void)setDraftID:(NSString*)DraftID {
		[super setInput:@"DraftID" toValue:DraftID];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Used to specify fields to include in a partial response. This can be used to reduce the amount of data returned. See Choreo notes for syntax rules.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The ID of the acting user. Defaults to "me" indicating the user associated with the Access Token or Refresh Token provided.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SendDraft Choreo.
 */
@implementation TMBGoogle_Gmailv2_Drafts_SendDraft_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Sends the specified, existing draft.
 */
@implementation TMBGoogle_Gmailv2_Drafts_SendDraft

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmailv2_Drafts_SendDraft Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmailv2/Drafts/SendDraft"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmailv2_Drafts_SendDraft_ResultSet *results = [[TMBGoogle_Gmailv2_Drafts_SendDraft_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SendDraft Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmailv2_Drafts_SendDraft_Inputs*)newInputSet {
		return [[TMBGoogle_Gmailv2_Drafts_SendDraft_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UpdateDraft Choreo.
 */
@implementation TMBGoogle_Gmailv2_Drafts_UpdateDraft_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AttachmentContentType input for this Choreo.
	*(optional, string) The Content-Type of the attachment. This is required when providing AttachmentFileContent (e.g., image/jpeg, text/plain, etc).
	 */
	-(void)setAttachmentContentType:(NSString*)AttachmentContentType {
		[super setInput:@"AttachmentContentType" toValue:AttachmentContentType];
	}

	/*!
	 * Set the value of the AttachmentFileContent input for this Choreo.
	*(optional, string) The Base64 encoded file content for the attachment. You must specify the AttachmentFileContentType when including an attachment.
	 */
	-(void)setAttachmentFileContent:(NSString*)AttachmentFileContent {
		[super setInput:@"AttachmentFileContent" toValue:AttachmentFileContent];
	}

	/*!
	 * Set the value of the AttachmentFileName input for this Choreo.
	*(optional, string) The file name of the attachment.
	 */
	-(void)setAttachmentFileName:(NSString*)AttachmentFileName {
		[super setInput:@"AttachmentFileName" toValue:AttachmentFileName];
	}

	/*!
	 * Set the value of the BCC input for this Choreo.
	*(optional, string) The address and name (optional) that should be bcc'd e.g., Dan <dan@temboo.com>.
	 */
	-(void)setBCC:(NSString*)BCC {
		[super setInput:@"BCC" toValue:BCC];
	}

	/*!
	 * Set the value of the CC input for this Choreo.
	*(optional, string) The address and name (optional) that should be cc'd e.g., Dan <dan@temboo.com>.
	 */
	-(void)setCC:(NSString*)CC {
		[super setInput:@"CC" toValue:CC];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the DraftID input for this Choreo.
	*(required, string) The ID of the draft to update.
	 */
	-(void)setDraftID:(NSString*)DraftID {
		[super setInput:@"DraftID" toValue:DraftID];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Used to specify fields to include in a partial response. This can be used to reduce the amount of data returned. See Choreo notes for syntax rules.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the From input for this Choreo.
	*(conditional, string) The address and name (optional) that the email is being sent from e.g., Dan <dan@temboo.com>.
	 */
	-(void)setFrom:(NSString*)From {
		[super setInput:@"From" toValue:From];
	}

	/*!
	 * Set the value of the MessageBody input for this Choreo.
	*(conditional, string) The text for the message body of the email.
	 */
	-(void)setMessageBody:(NSString*)MessageBody {
		[super setInput:@"MessageBody" toValue:MessageBody];
	}

	/*!
	 * Set the value of the MessageBodyContentType input for this Choreo.
	*(optional, string) The Content-Type of the message body. Defaults to "text/plain; charset=UTF-8".
	 */
	-(void)setMessageBodyContentType:(NSString*)MessageBodyContentType {
		[super setInput:@"MessageBodyContentType" toValue:MessageBodyContentType];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ReplyTo input for this Choreo.
	*(optional, string) An email address to set as the Reply-To address.
	 */
	-(void)setReplyTo:(NSString*)ReplyTo {
		[super setInput:@"ReplyTo" toValue:ReplyTo];
	}

	/*!
	 * Set the value of the Subject input for this Choreo.
	*(conditional, string) The email subject.
	 */
	-(void)setSubject:(NSString*)Subject {
		[super setInput:@"Subject" toValue:Subject];
	}

	/*!
	 * Set the value of the ThreadID input for this Choreo.
	*(optional, string) The ID of the thread the message belongs to.
	 */
	-(void)setThreadID:(NSString*)ThreadID {
		[super setInput:@"ThreadID" toValue:ThreadID];
	}

	/*!
	 * Set the value of the To input for this Choreo.
	*(conditional, string) The address and name (optional) that the email is being sent to e.g., Dan <dan@temboo.com>.
	 */
	-(void)setTo:(NSString*)To {
		[super setInput:@"To" toValue:To];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The ID of the acting user. Defaults to "me" indicating the user associated with the Access Token or Refresh Token provided.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UpdateDraft Choreo.
 */
@implementation TMBGoogle_Gmailv2_Drafts_UpdateDraft_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Updates the content of an existing draft.
 */
@implementation TMBGoogle_Gmailv2_Drafts_UpdateDraft

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmailv2_Drafts_UpdateDraft Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmailv2/Drafts/UpdateDraft"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmailv2_Drafts_UpdateDraft_ResultSet *results = [[TMBGoogle_Gmailv2_Drafts_UpdateDraft_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UpdateDraft Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmailv2_Drafts_UpdateDraft_Inputs*)newInputSet {
		return [[TMBGoogle_Gmailv2_Drafts_UpdateDraft_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListHistory Choreo.
 */
@implementation TMBGoogle_Gmailv2_History_ListHistory_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Used to specify fields to include in a partial response. This can be used to reduce the amount of data returned. See Choreo notes for syntax rules.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the LabelID input for this Choreo.
	*(optional, string) Returns messages with a label matching this ID.
	 */
	-(void)setLabelID:(NSString*)LabelID {
		[super setInput:@"LabelID" toValue:LabelID];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of results to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) The "nextPageToken" found in the response which is used to page through results.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the StartHistoryID input for this Choreo.
	*(required, string) Returns history records after the specified marker. The history ID is returned by the Messages.GetMessage Choreo.
	 */
	-(void)setStartHistoryID:(NSString*)StartHistoryID {
		[super setInput:@"StartHistoryID" toValue:StartHistoryID];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The ID of the acting user. Defaults to "me" indicating the user associated with the Access Token or Refresh Token provided.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListHistory Choreo.
 */
@implementation TMBGoogle_Gmailv2_History_ListHistory_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Lists the history of all changes to the given mailbox.
 */
@implementation TMBGoogle_Gmailv2_History_ListHistory

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmailv2_History_ListHistory Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmailv2/History/ListHistory"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmailv2_History_ListHistory_ResultSet *results = [[TMBGoogle_Gmailv2_History_ListHistory_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListHistory Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmailv2_History_ListHistory_Inputs*)newInputSet {
		return [[TMBGoogle_Gmailv2_History_ListHistory_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CreateLabel Choreo.
 */
@implementation TMBGoogle_Gmailv2_Labels_CreateLabel_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Used to specify fields to include in a partial response. This can be used to reduce the amount of data returned. See Choreo notes for syntax rules.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the LabelListVisibility input for this Choreo.
	*(required, string) The visibility of the label in the label list in the Gmail web interface. Valid values are: "labelHide", "labelShow", and "labelShowIfUnread.
	 */
	-(void)setLabelListVisibility:(NSString*)LabelListVisibility {
		[super setInput:@"LabelListVisibility" toValue:LabelListVisibility];
	}

	/*!
	 * Set the value of the MessageListVisibility input for this Choreo.
	*(required, string) The visibility of messages with this label in the message list in the Gmail web interface. Valid values are: "hide" and "show".
	 */
	-(void)setMessageListVisibility:(NSString*)MessageListVisibility {
		[super setInput:@"MessageListVisibility" toValue:MessageListVisibility];
	}

	/*!
	 * Set the value of the Name input for this Choreo.
	*(required, string) The display name of the label.
	 */
	-(void)setName:(NSString*)Name {
		[super setInput:@"Name" toValue:Name];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The ID of the acting user. Defaults to "me" indicating the user associated with the Access Token or Refresh Token provided.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateLabel Choreo.
 */
@implementation TMBGoogle_Gmailv2_Labels_CreateLabel_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Creates a new label.
 */
@implementation TMBGoogle_Gmailv2_Labels_CreateLabel

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmailv2_Labels_CreateLabel Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmailv2/Labels/CreateLabel"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmailv2_Labels_CreateLabel_ResultSet *results = [[TMBGoogle_Gmailv2_Labels_CreateLabel_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CreateLabel Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmailv2_Labels_CreateLabel_Inputs*)newInputSet {
		return [[TMBGoogle_Gmailv2_Labels_CreateLabel_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteLabel Choreo.
 */
@implementation TMBGoogle_Gmailv2_Labels_DeleteLabel_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the LabelID input for this Choreo.
	*(required, string) The ID of the label to delete.
	 */
	-(void)setLabelID:(NSString*)LabelID {
		[super setInput:@"LabelID" toValue:LabelID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The ID of the acting user. Defaults to "me" indicating the user associated with the Access Token or Refresh Token provided.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteLabel Choreo.
 */
@implementation TMBGoogle_Gmailv2_Labels_DeleteLabel_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "ResponseStatusCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) The response status code returned from Google. A 204 is expected for a successful delete operation.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponseStatusCode {
		return [super getOutputByName:@"ResponseStatusCode"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google. An empty response is expected for a successful delete operation.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Deletes a specific label.
 */
@implementation TMBGoogle_Gmailv2_Labels_DeleteLabel

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmailv2_Labels_DeleteLabel Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmailv2/Labels/DeleteLabel"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmailv2_Labels_DeleteLabel_ResultSet *results = [[TMBGoogle_Gmailv2_Labels_DeleteLabel_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteLabel Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmailv2_Labels_DeleteLabel_Inputs*)newInputSet {
		return [[TMBGoogle_Gmailv2_Labels_DeleteLabel_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetLabel Choreo.
 */
@implementation TMBGoogle_Gmailv2_Labels_GetLabel_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Used to specify fields to include in a partial response. This can be used to reduce the amount of data returned. See Choreo notes for syntax rules.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the LabelID input for this Choreo.
	*(required, string) The ID of the label to retrieve.
	 */
	-(void)setLabelID:(NSString*)LabelID {
		[super setInput:@"LabelID" toValue:LabelID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The ID of the acting user. Defaults to "me" indicating the user associated with the Access Token or Refresh Token provided.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetLabel Choreo.
 */
@implementation TMBGoogle_Gmailv2_Labels_GetLabel_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a specific label.
 */
@implementation TMBGoogle_Gmailv2_Labels_GetLabel

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmailv2_Labels_GetLabel Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmailv2/Labels/GetLabel"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmailv2_Labels_GetLabel_ResultSet *results = [[TMBGoogle_Gmailv2_Labels_GetLabel_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetLabel Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmailv2_Labels_GetLabel_Inputs*)newInputSet {
		return [[TMBGoogle_Gmailv2_Labels_GetLabel_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListLabels Choreo.
 */
@implementation TMBGoogle_Gmailv2_Labels_ListLabels_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Used to specify fields to include in a partial response. This can be used to reduce the amount of data returned. See Choreo notes for syntax rules.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The ID of the acting user. Defaults to "me" indicating the user associated with the Access Token or Refresh Token provided.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListLabels Choreo.
 */
@implementation TMBGoogle_Gmailv2_Labels_ListLabels_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Lists all labels in the user's mailbox.
 */
@implementation TMBGoogle_Gmailv2_Labels_ListLabels

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmailv2_Labels_ListLabels Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmailv2/Labels/ListLabels"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmailv2_Labels_ListLabels_ResultSet *results = [[TMBGoogle_Gmailv2_Labels_ListLabels_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListLabels Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmailv2_Labels_ListLabels_Inputs*)newInputSet {
		return [[TMBGoogle_Gmailv2_Labels_ListLabels_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UpdateLabel Choreo.
 */
@implementation TMBGoogle_Gmailv2_Labels_UpdateLabel_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Used to specify fields to include in a partial response. This can be used to reduce the amount of data returned. See Choreo notes for syntax rules.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the LabelID input for this Choreo.
	*(required, string) The ID of the label to update.
	 */
	-(void)setLabelID:(NSString*)LabelID {
		[super setInput:@"LabelID" toValue:LabelID];
	}

	/*!
	 * Set the value of the LabelListVisibility input for this Choreo.
	*(required, string) The visibility of the label in the label list in the Gmail web interface. Valid values are: "labelHide", "labelShow", and "labelShowIfUnread.
	 */
	-(void)setLabelListVisibility:(NSString*)LabelListVisibility {
		[super setInput:@"LabelListVisibility" toValue:LabelListVisibility];
	}

	/*!
	 * Set the value of the MessageListVisibility input for this Choreo.
	*(required, string) The visibility of messages with this label in the message list in the Gmail web interface. Valid values are: "hide" and "show".
	 */
	-(void)setMessageListVisibility:(NSString*)MessageListVisibility {
		[super setInput:@"MessageListVisibility" toValue:MessageListVisibility];
	}

	/*!
	 * Set the value of the Name input for this Choreo.
	*(required, string) The display name of the label.
	 */
	-(void)setName:(NSString*)Name {
		[super setInput:@"Name" toValue:Name];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The ID of the acting user. Defaults to "me" indicating the user associated with the Access Token or Refresh Token provided.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UpdateLabel Choreo.
 */
@implementation TMBGoogle_Gmailv2_Labels_UpdateLabel_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Updates a specific label.
 */
@implementation TMBGoogle_Gmailv2_Labels_UpdateLabel

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmailv2_Labels_UpdateLabel Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmailv2/Labels/UpdateLabel"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmailv2_Labels_UpdateLabel_ResultSet *results = [[TMBGoogle_Gmailv2_Labels_UpdateLabel_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UpdateLabel Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmailv2_Labels_UpdateLabel_Inputs*)newInputSet {
		return [[TMBGoogle_Gmailv2_Labels_UpdateLabel_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ClearStoredHistory Choreo.
 */
@implementation TMBGoogle_Gmailv2_Messages_ClearStoredHistory_Inputs

	/*!
	 * Set the value of the LabelID input for this Choreo.
	*(optional, string) The Label ID associated with the stored history ID that should be deleted.
	 */
	-(void)setLabelID:(NSString*)LabelID {
		[super setInput:@"LabelID" toValue:LabelID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ClearStoredHistory Choreo.
 */
@implementation TMBGoogle_Gmailv2_Messages_ClearStoredHistory_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Deleted" output from an execution of this Choreo.
	 * @return - NSString* (boolean) Returns true when the history ID is successfully deleted.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getDeleted {
		return [super getOutputByName:@"Deleted"];
	}
	
@end

/*!
 * Clears the history ID stored in your Temboo account when executing the GetNextMessage Choreo.
 */
@implementation TMBGoogle_Gmailv2_Messages_ClearStoredHistory

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmailv2_Messages_ClearStoredHistory Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmailv2/Messages/ClearStoredHistory"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmailv2_Messages_ClearStoredHistory_ResultSet *results = [[TMBGoogle_Gmailv2_Messages_ClearStoredHistory_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ClearStoredHistory Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmailv2_Messages_ClearStoredHistory_Inputs*)newInputSet {
		return [[TMBGoogle_Gmailv2_Messages_ClearStoredHistory_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteMessage Choreo.
 */
@implementation TMBGoogle_Gmailv2_Messages_DeleteMessage_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the MessageID input for this Choreo.
	*(required, string) The ID of the message to delete.
	 */
	-(void)setMessageID:(NSString*)MessageID {
		[super setInput:@"MessageID" toValue:MessageID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The ID of the acting user. Defaults to "me" indicating the user associated with the Access Token or Refresh Token provided.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteMessage Choreo.
 */
@implementation TMBGoogle_Gmailv2_Messages_DeleteMessage_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "ResponseStatusCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) The response status code returned from Google. A 204 is expected for a successful delete operation.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponseStatusCode {
		return [super getOutputByName:@"ResponseStatusCode"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google. An empty response is expected for a successful delete operation.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Permanently deletes a specified message.
 */
@implementation TMBGoogle_Gmailv2_Messages_DeleteMessage

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmailv2_Messages_DeleteMessage Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmailv2/Messages/DeleteMessage"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmailv2_Messages_DeleteMessage_ResultSet *results = [[TMBGoogle_Gmailv2_Messages_DeleteMessage_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteMessage Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmailv2_Messages_DeleteMessage_Inputs*)newInputSet {
		return [[TMBGoogle_Gmailv2_Messages_DeleteMessage_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetLatestMessage Choreo.
 */
@implementation TMBGoogle_Gmailv2_Messages_GetLatestMessage_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the EncodeMessage input for this Choreo.
	*(optional, boolean) When set to "true" (the default), the message Body will be Base64 encoded.
	 */
	-(void)setEncodeMessage:(NSString*)EncodeMessage {
		[super setInput:@"EncodeMessage" toValue:EncodeMessage];
	}

	/*!
	 * Set the value of the IncludeSpamTrash input for this Choreo.
	*(optional, boolean) Set to "true" to include messages from SPAM and TRASH in the results. Defaults to "false".
	 */
	-(void)setIncludeSpamTrash:(NSString*)IncludeSpamTrash {
		[super setInput:@"IncludeSpamTrash" toValue:IncludeSpamTrash];
	}

	/*!
	 * Set the value of the LabelID input for this Choreo.
	*(optional, string) Returns messages with a label matching this ID.
	 */
	-(void)setLabelID:(NSString*)LabelID {
		[super setInput:@"LabelID" toValue:LabelID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the StartHistoryID input for this Choreo.
	*(optional, string) Returns history records after the specified marker. The history ID is returned by this Choreo after retrieving a message.
	 */
	-(void)setStartHistoryID:(NSString*)StartHistoryID {
		[super setInput:@"StartHistoryID" toValue:StartHistoryID];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The ID of the acting user. Defaults to "me" indicating the user associated with the Access Token or Refresh Token provided.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetLatestMessage Choreo.
 */
@implementation TMBGoogle_Gmailv2_Messages_GetLatestMessage_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Body" output from an execution of this Choreo.
	 * @return - NSString* (string) The message body.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getBody {
		return [super getOutputByName:@"Body"];
	}

	/*!
	 * Retrieve the value of the "From" output from an execution of this Choreo.
	 * @return - NSString* (string) The sender address.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getFrom {
		return [super getOutputByName:@"From"];
	}

	/*!
	 * Retrieve the value of the "HistoryID" output from an execution of this Choreo.
	 * @return - NSString* (string) The history ID. This can be passed to the StartHistoryID input to retrieve only mail received after this marker.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getHistoryID {
		return [super getOutputByName:@"HistoryID"];
	}

	/*!
	 * Retrieve the value of the "MessageID" output from an execution of this Choreo.
	 * @return - NSString* (string) The ID of the message.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getMessageID {
		return [super getOutputByName:@"MessageID"];
	}

	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Snippet" output from an execution of this Choreo.
	 * @return - NSString* (string) The email body snippet.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getSnippet {
		return [super getOutputByName:@"Snippet"];
	}

	/*!
	 * Retrieve the value of the "Subject" output from an execution of this Choreo.
	 * @return - NSString* (string) The message subject.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getSubject {
		return [super getOutputByName:@"Subject"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the latest email from a user's inbox.
 */
@implementation TMBGoogle_Gmailv2_Messages_GetLatestMessage

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmailv2_Messages_GetLatestMessage Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmailv2/Messages/GetLatestMessage"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmailv2_Messages_GetLatestMessage_ResultSet *results = [[TMBGoogle_Gmailv2_Messages_GetLatestMessage_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetLatestMessage Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmailv2_Messages_GetLatestMessage_Inputs*)newInputSet {
		return [[TMBGoogle_Gmailv2_Messages_GetLatestMessage_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetMessage Choreo.
 */
@implementation TMBGoogle_Gmailv2_Messages_GetMessage_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Used to specify fields to include in a partial response. This can be used to reduce the amount of data returned. See Choreo notes for syntax rules.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Format input for this Choreo.
	*(optional, boolean) Specifies the format of the message returned. Valid values are: full (the default), minimal, and raw. See Choreo notes for more details about these formats.
	 */
	-(void)setFormat:(NSString*)Format {
		[super setInput:@"Format" toValue:Format];
	}

	/*!
	 * Set the value of the MessageID input for this Choreo.
	*(required, string) The ID of the message to retrieve.
	 */
	-(void)setMessageID:(NSString*)MessageID {
		[super setInput:@"MessageID" toValue:MessageID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The ID of the acting user. Defaults to "me" indicating the user associated with the Access Token or Refresh Token provided.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetMessage Choreo.
 */
@implementation TMBGoogle_Gmailv2_Messages_GetMessage_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves an individual message from a user's mailbox.
 */
@implementation TMBGoogle_Gmailv2_Messages_GetMessage

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmailv2_Messages_GetMessage Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmailv2/Messages/GetMessage"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmailv2_Messages_GetMessage_ResultSet *results = [[TMBGoogle_Gmailv2_Messages_GetMessage_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetMessage Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmailv2_Messages_GetMessage_Inputs*)newInputSet {
		return [[TMBGoogle_Gmailv2_Messages_GetMessage_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetNextMessage Choreo.
 */
@implementation TMBGoogle_Gmailv2_Messages_GetNextMessage_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the EncodeMessage input for this Choreo.
	*(optional, boolean) When set to "true" (the default), the message Body will be Base64 encoded.
	 */
	-(void)setEncodeMessage:(NSString*)EncodeMessage {
		[super setInput:@"EncodeMessage" toValue:EncodeMessage];
	}

	/*!
	 * Set the value of the LabelID input for this Choreo.
	*(conditional, string) Returns messages with a label matching this ID (e.g., INBOX, UNREAD, Label_1, etc.)
	 */
	-(void)setLabelID:(NSString*)LabelID {
		[super setInput:@"LabelID" toValue:LabelID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The ID of the acting user. Defaults to "me" indicating the user associated with the Access Token or Refresh Token provided.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetNextMessage Choreo.
 */
@implementation TMBGoogle_Gmailv2_Messages_GetNextMessage_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Body" output from an execution of this Choreo.
	 * @return - NSString* (string) The message body.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getBody {
		return [super getOutputByName:@"Body"];
	}

	/*!
	 * Retrieve the value of the "From" output from an execution of this Choreo.
	 * @return - NSString* (string) The sender address.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getFrom {
		return [super getOutputByName:@"From"];
	}

	/*!
	 * Retrieve the value of the "MessageID" output from an execution of this Choreo.
	 * @return - NSString* (string) The ID of the message.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getMessageID {
		return [super getOutputByName:@"MessageID"];
	}

	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "ResponseStatusCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) The response code returned from Google. A 200 is expected for a successful request. A 404 indicates that the stored history id has become invalid or expired. See Choreo notes for more details.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponseStatusCode {
		return [super getOutputByName:@"ResponseStatusCode"];
	}

	/*!
	 * Retrieve the value of the "Snippet" output from an execution of this Choreo.
	 * @return - NSString* (string) The email body snippet.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getSnippet {
		return [super getOutputByName:@"Snippet"];
	}

	/*!
	 * Retrieve the value of the "Subject" output from an execution of this Choreo.
	 * @return - NSString* (string) The message subject.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getSubject {
		return [super getOutputByName:@"Subject"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the next email from a user's inbox.
 */
@implementation TMBGoogle_Gmailv2_Messages_GetNextMessage

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmailv2_Messages_GetNextMessage Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmailv2/Messages/GetNextMessage"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmailv2_Messages_GetNextMessage_ResultSet *results = [[TMBGoogle_Gmailv2_Messages_GetNextMessage_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetNextMessage Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmailv2_Messages_GetNextMessage_Inputs*)newInputSet {
		return [[TMBGoogle_Gmailv2_Messages_GetNextMessage_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the InsertMessage Choreo.
 */
@implementation TMBGoogle_Gmailv2_Messages_InsertMessage_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AttachmentContentType input for this Choreo.
	*(optional, string) The Content-Type of the attachment. This is required when providing AttachmentFileContent (e.g., image/jpeg, text/plain, etc).
	 */
	-(void)setAttachmentContentType:(NSString*)AttachmentContentType {
		[super setInput:@"AttachmentContentType" toValue:AttachmentContentType];
	}

	/*!
	 * Set the value of the AttachmentFileContent input for this Choreo.
	*(optional, string) The Base64 encoded file content for the attachment. You must specify the AttachmentFileContentType when including an attachment.
	 */
	-(void)setAttachmentFileContent:(NSString*)AttachmentFileContent {
		[super setInput:@"AttachmentFileContent" toValue:AttachmentFileContent];
	}

	/*!
	 * Set the value of the AttachmentFileName input for this Choreo.
	*(optional, string) The file name of the attachment.
	 */
	-(void)setAttachmentFileName:(NSString*)AttachmentFileName {
		[super setInput:@"AttachmentFileName" toValue:AttachmentFileName];
	}

	/*!
	 * Set the value of the BCC input for this Choreo.
	*(optional, string) The address and name (optional) that should be bcc'd e.g., Dan <dan@temboo.com>.
	 */
	-(void)setBCC:(NSString*)BCC {
		[super setInput:@"BCC" toValue:BCC];
	}

	/*!
	 * Set the value of the CC input for this Choreo.
	*(optional, string) The address and name (optional) that should be cc'd e.g., Dan <dan@temboo.com>.
	 */
	-(void)setCC:(NSString*)CC {
		[super setInput:@"CC" toValue:CC];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Used to specify fields to include in a partial response. This can be used to reduce the amount of data returned. See Choreo notes for syntax rules.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the From input for this Choreo.
	*(required, string) The address and name (optional) that the email is being sent from e.g., Dan <dan@temboo.com>.
	 */
	-(void)setFrom:(NSString*)From {
		[super setInput:@"From" toValue:From];
	}

	/*!
	 * Set the value of the LabelIDs input for this Choreo.
	*(optional, json) An array of Gmail labels to apply to this message (e.g., ["SENT", "INBOX", "UNREAD"].
	 */
	-(void)setLabelIDs:(NSString*)LabelIDs {
		[super setInput:@"LabelIDs" toValue:LabelIDs];
	}

	/*!
	 * Set the value of the MessageBody input for this Choreo.
	*(required, string) The text for the message body of the email.
	 */
	-(void)setMessageBody:(NSString*)MessageBody {
		[super setInput:@"MessageBody" toValue:MessageBody];
	}

	/*!
	 * Set the value of the MessageBodyContentType input for this Choreo.
	*(optional, string) The Content-Type of the message body. Defaults to "text/plain; charset=UTF-8".
	 */
	-(void)setMessageBodyContentType:(NSString*)MessageBodyContentType {
		[super setInput:@"MessageBodyContentType" toValue:MessageBodyContentType];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ReplyTo input for this Choreo.
	*(optional, string) An email address to set as the Reply-To address.
	 */
	-(void)setReplyTo:(NSString*)ReplyTo {
		[super setInput:@"ReplyTo" toValue:ReplyTo];
	}

	/*!
	 * Set the value of the Subject input for this Choreo.
	*(required, string) The email subject.
	 */
	-(void)setSubject:(NSString*)Subject {
		[super setInput:@"Subject" toValue:Subject];
	}

	/*!
	 * Set the value of the ThreadID input for this Choreo.
	*(optional, string) The ID of the thread the message belongs to.
	 */
	-(void)setThreadID:(NSString*)ThreadID {
		[super setInput:@"ThreadID" toValue:ThreadID];
	}

	/*!
	 * Set the value of the To input for this Choreo.
	*(required, string) The address and name (optional) that the email is being sent to e.g., Dan <dan@temboo.com>.
	 */
	-(void)setTo:(NSString*)To {
		[super setInput:@"To" toValue:To];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The ID of the acting user. Defaults to "me" indicating the user associated with the Access Token or Refresh Token provided.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the InsertMessage Choreo.
 */
@implementation TMBGoogle_Gmailv2_Messages_InsertMessage_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Directly inserts a message into a user's mailbox. Does not send a message.
 */
@implementation TMBGoogle_Gmailv2_Messages_InsertMessage

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmailv2_Messages_InsertMessage Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmailv2/Messages/InsertMessage"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmailv2_Messages_InsertMessage_ResultSet *results = [[TMBGoogle_Gmailv2_Messages_InsertMessage_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the InsertMessage Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmailv2_Messages_InsertMessage_Inputs*)newInputSet {
		return [[TMBGoogle_Gmailv2_Messages_InsertMessage_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListMessages Choreo.
 */
@implementation TMBGoogle_Gmailv2_Messages_ListMessages_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Used to specify fields to include in a partial response. This can be used to reduce the amount of data returned. See Choreo notes for syntax rules.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the IncludeSpamTrash input for this Choreo.
	*(optional, boolean) Set to "true" to include messages from SPAM and TRASH in the results. Defaults to "false".
	 */
	-(void)setIncludeSpamTrash:(NSString*)IncludeSpamTrash {
		[super setInput:@"IncludeSpamTrash" toValue:IncludeSpamTrash];
	}

	/*!
	 * Set the value of the LabelIDs input for this Choreo.
	*(optional, json) A JSON array containing labels to filter by. When specified, only messages with labels that match are returned.
	 */
	-(void)setLabelIDs:(NSString*)LabelIDs {
		[super setInput:@"LabelIDs" toValue:LabelIDs];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of results to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) The "nextPageToken" found in the response which is used to page through results.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the Query input for this Choreo.
	*(optional, string) Filters messages that match the specified query. Supports the same query format as the Gmail search box. For example, "from:someuser@example.com rfc822msgid: is:unread".
	 */
	-(void)setQuery:(NSString*)Query {
		[super setInput:@"Query" toValue:Query];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The ID of the acting user. Defaults to "me" indicating the user associated with the Access Token or Refresh Token provided.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListMessages Choreo.
 */
@implementation TMBGoogle_Gmailv2_Messages_ListMessages_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Lists the messages in the user's mailbox.
 */
@implementation TMBGoogle_Gmailv2_Messages_ListMessages

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmailv2_Messages_ListMessages Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmailv2/Messages/ListMessages"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmailv2_Messages_ListMessages_ResultSet *results = [[TMBGoogle_Gmailv2_Messages_ListMessages_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListMessages Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmailv2_Messages_ListMessages_Inputs*)newInputSet {
		return [[TMBGoogle_Gmailv2_Messages_ListMessages_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ModifyMessage Choreo.
 */
@implementation TMBGoogle_Gmailv2_Messages_ModifyMessage_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AddLabelIDs input for this Choreo.
	*(conditional, json) An array of label IDs to apply to the message.
	 */
	-(void)setAddLabelIDs:(NSString*)AddLabelIDs {
		[super setInput:@"AddLabelIDs" toValue:AddLabelIDs];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Used to specify fields to include in a partial response. This can be used to reduce the amount of data returned. See Choreo notes for syntax rules.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the MessageID input for this Choreo.
	*(required, string) The ID of the message to modify.
	 */
	-(void)setMessageID:(NSString*)MessageID {
		[super setInput:@"MessageID" toValue:MessageID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the RemoveLabelIDs input for this Choreo.
	*(conditional, json) An array of label IDs to remove from the message.
	 */
	-(void)setRemoveLabelIDs:(NSString*)RemoveLabelIDs {
		[super setInput:@"RemoveLabelIDs" toValue:RemoveLabelIDs];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The ID of the acting user. Defaults to "me" indicating the user associated with the Access Token or Refresh Token provided.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ModifyMessage Choreo.
 */
@implementation TMBGoogle_Gmailv2_Messages_ModifyMessage_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Modifies the labels for a specific message.
 */
@implementation TMBGoogle_Gmailv2_Messages_ModifyMessage

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmailv2_Messages_ModifyMessage Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmailv2/Messages/ModifyMessage"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmailv2_Messages_ModifyMessage_ResultSet *results = [[TMBGoogle_Gmailv2_Messages_ModifyMessage_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ModifyMessage Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmailv2_Messages_ModifyMessage_Inputs*)newInputSet {
		return [[TMBGoogle_Gmailv2_Messages_ModifyMessage_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SendMessage Choreo.
 */
@implementation TMBGoogle_Gmailv2_Messages_SendMessage_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AttachmentContentType input for this Choreo.
	*(optional, string) The Content-Type of the attachment. This is required when providing AttachmentFileContent (e.g., image/jpeg, text/plain, etc).
	 */
	-(void)setAttachmentContentType:(NSString*)AttachmentContentType {
		[super setInput:@"AttachmentContentType" toValue:AttachmentContentType];
	}

	/*!
	 * Set the value of the AttachmentFileContent input for this Choreo.
	*(optional, string) The Base64 encoded file content for the attachment. You must specify the AttachmentFileContentType when including an attachment.
	 */
	-(void)setAttachmentFileContent:(NSString*)AttachmentFileContent {
		[super setInput:@"AttachmentFileContent" toValue:AttachmentFileContent];
	}

	/*!
	 * Set the value of the AttachmentFileName input for this Choreo.
	*(optional, string) The file name of the attachment.
	 */
	-(void)setAttachmentFileName:(NSString*)AttachmentFileName {
		[super setInput:@"AttachmentFileName" toValue:AttachmentFileName];
	}

	/*!
	 * Set the value of the BCC input for this Choreo.
	*(optional, string) The address and name (optional) that should be bcc'd e.g., Dan <dan@temboo.com>.
	 */
	-(void)setBCC:(NSString*)BCC {
		[super setInput:@"BCC" toValue:BCC];
	}

	/*!
	 * Set the value of the CC input for this Choreo.
	*(optional, string) The address and name (optional) that should be cc'd e.g., Dan <dan@temboo.com>.
	 */
	-(void)setCC:(NSString*)CC {
		[super setInput:@"CC" toValue:CC];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Used to specify fields to include in a partial response. This can be used to reduce the amount of data returned. See Choreo notes for syntax rules.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the From input for this Choreo.
	*(required, string) The address and name (optional) that the email is being sent from e.g., Dan <dan@temboo.com>.
	 */
	-(void)setFrom:(NSString*)From {
		[super setInput:@"From" toValue:From];
	}

	/*!
	 * Set the value of the MessageBody input for this Choreo.
	*(required, string) The text for the message body of the email.
	 */
	-(void)setMessageBody:(NSString*)MessageBody {
		[super setInput:@"MessageBody" toValue:MessageBody];
	}

	/*!
	 * Set the value of the MessageBodyContentType input for this Choreo.
	*(optional, string) The Content-Type of the message body. Defaults to "text/plain; charset=UTF-8".
	 */
	-(void)setMessageBodyContentType:(NSString*)MessageBodyContentType {
		[super setInput:@"MessageBodyContentType" toValue:MessageBodyContentType];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ReplyTo input for this Choreo.
	*(optional, string) An email address to set as the Reply-To address.
	 */
	-(void)setReplyTo:(NSString*)ReplyTo {
		[super setInput:@"ReplyTo" toValue:ReplyTo];
	}

	/*!
	 * Set the value of the Subject input for this Choreo.
	*(required, string) The email subject.
	 */
	-(void)setSubject:(NSString*)Subject {
		[super setInput:@"Subject" toValue:Subject];
	}

	/*!
	 * Set the value of the ThreadID input for this Choreo.
	*(optional, string) The ID of the thread the message belongs to.
	 */
	-(void)setThreadID:(NSString*)ThreadID {
		[super setInput:@"ThreadID" toValue:ThreadID];
	}

	/*!
	 * Set the value of the To input for this Choreo.
	*(required, string) The address and name (optional) that the email is being sent to e.g., Dan <dan@temboo.com>.
	 */
	-(void)setTo:(NSString*)To {
		[super setInput:@"To" toValue:To];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The ID of the acting user. Defaults to "me" indicating the user associated with the Access Token or Refresh Token provided.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SendMessage Choreo.
 */
@implementation TMBGoogle_Gmailv2_Messages_SendMessage_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Sends an email to the recipients specified in the To, CC, and BCC parameters.
 */
@implementation TMBGoogle_Gmailv2_Messages_SendMessage

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmailv2_Messages_SendMessage Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmailv2/Messages/SendMessage"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmailv2_Messages_SendMessage_ResultSet *results = [[TMBGoogle_Gmailv2_Messages_SendMessage_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SendMessage Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmailv2_Messages_SendMessage_Inputs*)newInputSet {
		return [[TMBGoogle_Gmailv2_Messages_SendMessage_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the TrashMessage Choreo.
 */
@implementation TMBGoogle_Gmailv2_Messages_TrashMessage_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Used to specify fields to include in a partial response. This can be used to reduce the amount of data returned. See Choreo notes for syntax rules.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the MessageID input for this Choreo.
	*(required, string) The ID of the message to move to the trash.
	 */
	-(void)setMessageID:(NSString*)MessageID {
		[super setInput:@"MessageID" toValue:MessageID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The ID of the acting user. Defaults to "me" indicating the user associated with the Access Token or Refresh Token provided.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the TrashMessage Choreo.
 */
@implementation TMBGoogle_Gmailv2_Messages_TrashMessage_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Moves a specific message to the trash.
 */
@implementation TMBGoogle_Gmailv2_Messages_TrashMessage

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmailv2_Messages_TrashMessage Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmailv2/Messages/TrashMessage"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmailv2_Messages_TrashMessage_ResultSet *results = [[TMBGoogle_Gmailv2_Messages_TrashMessage_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the TrashMessage Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmailv2_Messages_TrashMessage_Inputs*)newInputSet {
		return [[TMBGoogle_Gmailv2_Messages_TrashMessage_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UnTrashMessage Choreo.
 */
@implementation TMBGoogle_Gmailv2_Messages_UnTrashMessage_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Used to specify fields to include in a partial response. This can be used to reduce the amount of data returned. See Choreo notes for syntax rules.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the MessageID input for this Choreo.
	*(required, string) The ID of the message to restore.
	 */
	-(void)setMessageID:(NSString*)MessageID {
		[super setInput:@"MessageID" toValue:MessageID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The ID of the acting user. Defaults to "me" indicating the user associated with the Access Token or Refresh Token provided.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UnTrashMessage Choreo.
 */
@implementation TMBGoogle_Gmailv2_Messages_UnTrashMessage_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Moves a specific message out of the trash.
 */
@implementation TMBGoogle_Gmailv2_Messages_UnTrashMessage

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmailv2_Messages_UnTrashMessage Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmailv2/Messages/UnTrashMessage"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmailv2_Messages_UnTrashMessage_ResultSet *results = [[TMBGoogle_Gmailv2_Messages_UnTrashMessage_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UnTrashMessage Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmailv2_Messages_UnTrashMessage_Inputs*)newInputSet {
		return [[TMBGoogle_Gmailv2_Messages_UnTrashMessage_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteThread Choreo.
 */
@implementation TMBGoogle_Gmailv2_Threads_DeleteThread_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ThreadID input for this Choreo.
	*(required, string) The ID of the thread to delete.
	 */
	-(void)setThreadID:(NSString*)ThreadID {
		[super setInput:@"ThreadID" toValue:ThreadID];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The ID of the acting user. Defaults to "me" indicating the user associated with the Access Token or Refresh Token provided.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteThread Choreo.
 */
@implementation TMBGoogle_Gmailv2_Threads_DeleteThread_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google. An empty response is expected for a successful delete operation.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Permanently deletes a specific thread.
 */
@implementation TMBGoogle_Gmailv2_Threads_DeleteThread

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmailv2_Threads_DeleteThread Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmailv2/Threads/DeleteThread"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmailv2_Threads_DeleteThread_ResultSet *results = [[TMBGoogle_Gmailv2_Threads_DeleteThread_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteThread Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmailv2_Threads_DeleteThread_Inputs*)newInputSet {
		return [[TMBGoogle_Gmailv2_Threads_DeleteThread_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetThread Choreo.
 */
@implementation TMBGoogle_Gmailv2_Threads_GetThread_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Used to specify fields to include in a partial response. This can be used to reduce the amount of data returned. See Choreo notes for syntax rules.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ThreadID input for this Choreo.
	*(required, string) The ID of the thread to retrieve.
	 */
	-(void)setThreadID:(NSString*)ThreadID {
		[super setInput:@"ThreadID" toValue:ThreadID];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The ID of the acting user. Defaults to "me" indicating the user associated with the Access Token or Refresh Token provided.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetThread Choreo.
 */
@implementation TMBGoogle_Gmailv2_Threads_GetThread_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves an individual thread from a user's mailbox.
 */
@implementation TMBGoogle_Gmailv2_Threads_GetThread

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmailv2_Threads_GetThread Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmailv2/Threads/GetThread"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmailv2_Threads_GetThread_ResultSet *results = [[TMBGoogle_Gmailv2_Threads_GetThread_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetThread Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmailv2_Threads_GetThread_Inputs*)newInputSet {
		return [[TMBGoogle_Gmailv2_Threads_GetThread_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListThreads Choreo.
 */
@implementation TMBGoogle_Gmailv2_Threads_ListThreads_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Used to specify fields to include in a partial response. This can be used to reduce the amount of data returned. See Choreo notes for syntax rules.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the IncludeSpamTrash input for this Choreo.
	*(optional, boolean) Set to "true" to include threads from SPAM and TRASH in the results. Defaults to "false".
	 */
	-(void)setIncludeSpamTrash:(NSString*)IncludeSpamTrash {
		[super setInput:@"IncludeSpamTrash" toValue:IncludeSpamTrash];
	}

	/*!
	 * Set the value of the LabelIDs input for this Choreo.
	*(optional, json) A JSON array containing labels to filter by. When specified, only threads with labels that match are returned.
	 */
	-(void)setLabelIDs:(NSString*)LabelIDs {
		[super setInput:@"LabelIDs" toValue:LabelIDs];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The maximum number of results to return.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) The "nextPageToken" found in the response which is used to page through results.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the Query input for this Choreo.
	*(optional, string) Filters threads that match the specified query. Supports the same query format as the Gmail search box. For example, "from:someuser@example.com rfc822msgid: is:unread".
	 */
	-(void)setQuery:(NSString*)Query {
		[super setInput:@"Query" toValue:Query];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The ID of the acting user. Defaults to "me" indicating the user associated with the Access Token or Refresh Token provided.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListThreads Choreo.
 */
@implementation TMBGoogle_Gmailv2_Threads_ListThreads_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Lists the threads in the user's mailbox.
 */
@implementation TMBGoogle_Gmailv2_Threads_ListThreads

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmailv2_Threads_ListThreads Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmailv2/Threads/ListThreads"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmailv2_Threads_ListThreads_ResultSet *results = [[TMBGoogle_Gmailv2_Threads_ListThreads_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListThreads Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmailv2_Threads_ListThreads_Inputs*)newInputSet {
		return [[TMBGoogle_Gmailv2_Threads_ListThreads_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ModifyThread Choreo.
 */
@implementation TMBGoogle_Gmailv2_Threads_ModifyThread_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AddLabelIDs input for this Choreo.
	*(conditional, json) An array of label IDs to apply to the thread.
	 */
	-(void)setAddLabelIDs:(NSString*)AddLabelIDs {
		[super setInput:@"AddLabelIDs" toValue:AddLabelIDs];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Used to specify fields to include in a partial response. This can be used to reduce the amount of data returned. See Choreo notes for syntax rules.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the RemoveLabelIDs input for this Choreo.
	*(conditional, json) An array of label IDs to remove from the thread.
	 */
	-(void)setRemoveLabelIDs:(NSString*)RemoveLabelIDs {
		[super setInput:@"RemoveLabelIDs" toValue:RemoveLabelIDs];
	}

	/*!
	 * Set the value of the ThreadID input for this Choreo.
	*(required, string) The ID of the thread to modify.
	 */
	-(void)setThreadID:(NSString*)ThreadID {
		[super setInput:@"ThreadID" toValue:ThreadID];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The ID of the acting user. Defaults to "me" indicating the user associated with the Access Token or Refresh Token provided.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ModifyThread Choreo.
 */
@implementation TMBGoogle_Gmailv2_Threads_ModifyThread_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Modifies the labels for a specific thread.
 */
@implementation TMBGoogle_Gmailv2_Threads_ModifyThread

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmailv2_Threads_ModifyThread Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmailv2/Threads/ModifyThread"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmailv2_Threads_ModifyThread_ResultSet *results = [[TMBGoogle_Gmailv2_Threads_ModifyThread_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ModifyThread Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmailv2_Threads_ModifyThread_Inputs*)newInputSet {
		return [[TMBGoogle_Gmailv2_Threads_ModifyThread_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the TrashThread Choreo.
 */
@implementation TMBGoogle_Gmailv2_Threads_TrashThread_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Used to specify fields to include in a partial response. This can be used to reduce the amount of data returned. See Choreo notes for syntax rules.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ThreadID input for this Choreo.
	*(required, string) The ID of the thread to move to the trash.
	 */
	-(void)setThreadID:(NSString*)ThreadID {
		[super setInput:@"ThreadID" toValue:ThreadID];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The ID of the acting user. Defaults to "me" indicating the user associated with the Access Token or Refresh Token provided.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the TrashThread Choreo.
 */
@implementation TMBGoogle_Gmailv2_Threads_TrashThread_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Moves a specific thread to the trash.
 */
@implementation TMBGoogle_Gmailv2_Threads_TrashThread

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmailv2_Threads_TrashThread Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmailv2/Threads/TrashThread"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmailv2_Threads_TrashThread_ResultSet *results = [[TMBGoogle_Gmailv2_Threads_TrashThread_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the TrashThread Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmailv2_Threads_TrashThread_Inputs*)newInputSet {
		return [[TMBGoogle_Gmailv2_Threads_TrashThread_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UnTrashThread Choreo.
 */
@implementation TMBGoogle_Gmailv2_Threads_UnTrashThread_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid Access Token retrieved during the OAuth2 process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new Access Token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Used to specify fields to include in a partial response. This can be used to reduce the amount of data returned. See Choreo notes for syntax rules.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new Access Token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ThreadID input for this Choreo.
	*(required, string) The ID of the thread to restore.
	 */
	-(void)setThreadID:(NSString*)ThreadID {
		[super setInput:@"ThreadID" toValue:ThreadID];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The ID of the acting user. Defaults to "me" indicating the user associated with the Access Token or Refresh Token provided.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UnTrashThread Choreo.
 */
@implementation TMBGoogle_Gmailv2_Threads_UnTrashThread_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Moves a specific thread out of the trash.
 */
@implementation TMBGoogle_Gmailv2_Threads_UnTrashThread

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Gmailv2_Threads_UnTrashThread Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Gmailv2/Threads/UnTrashThread"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Gmailv2_Threads_UnTrashThread_ResultSet *results = [[TMBGoogle_Gmailv2_Threads_UnTrashThread_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UnTrashThread Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Gmailv2_Threads_UnTrashThread_Inputs*)newInputSet {
		return [[TMBGoogle_Gmailv2_Threads_UnTrashThread_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the AnalyzeEntities Choreo.
 */
@implementation TMBGoogle_NaturalLanguage_Documents_AnalyzeEntities_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(conditional, string) The API Key provided by Google.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the Content input for this Choreo.
	*(conditional, string) The content to analyze.
	 */
	-(void)setContent:(NSString*)Content {
		[super setInput:@"Content" toValue:Content];
	}

	/*!
	 * Set the value of the EncodingType input for this Choreo.
	*(optional, string) The encoding type used by the API to calculate sentence offsets. Defaults to UTF-8.
	 */
	-(void)setEncodingType:(NSString*)EncodingType {
		[super setInput:@"EncodingType" toValue:EncodingType];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying which fields to include in a partial response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the GCSContentUri input for this Choreo.
	*(optional, string) The Google Cloud Storage URI where the file content is located. This can be used instead of the Content input.
	 */
	-(void)setGCSContentUri:(NSString*)GCSContentUri {
		[super setInput:@"GCSContentUri" toValue:GCSContentUri];
	}

	/*!
	 * Set the value of the Language input for this Choreo.
	*(optional, string) The language of the document. This is specified with the ISO-639-1 Code (e.g. "en"). If not specified, the language is detected if possible.
	 */
	-(void)setLanguage:(NSString*)Language {
		[super setInput:@"Language" toValue:Language];
	}

	/*!
	 * Set the value of the Type input for this Choreo.
	*(optional, string) The type of content being sent. Valid values are PLAIN_TEXT (the default) and HTML.
	 */
	-(void)setType:(NSString*)Type {
		[super setInput:@"Type" toValue:Type];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the AnalyzeEntities Choreo.
 */
@implementation TMBGoogle_NaturalLanguage_Documents_AnalyzeEntities_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Finds named entities (currently proper names and common nouns) in the text along with entity types, salience, mentions for each entity, and other properties.
 */
@implementation TMBGoogle_NaturalLanguage_Documents_AnalyzeEntities

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_NaturalLanguage_Documents_AnalyzeEntities Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/NaturalLanguage/Documents/AnalyzeEntities"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_NaturalLanguage_Documents_AnalyzeEntities_ResultSet *results = [[TMBGoogle_NaturalLanguage_Documents_AnalyzeEntities_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the AnalyzeEntities Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_NaturalLanguage_Documents_AnalyzeEntities_Inputs*)newInputSet {
		return [[TMBGoogle_NaturalLanguage_Documents_AnalyzeEntities_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the AnalyzeSentiment Choreo.
 */
@implementation TMBGoogle_NaturalLanguage_Documents_AnalyzeSentiment_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(conditional, string) The API Key provided by Google.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the Content input for this Choreo.
	*(conditional, string) The content to analyze.
	 */
	-(void)setContent:(NSString*)Content {
		[super setInput:@"Content" toValue:Content];
	}

	/*!
	 * Set the value of the EncodingType input for this Choreo.
	*(optional, string) The encoding type used by the API to calculate sentence offsets. Defaults to UTF-8.
	 */
	-(void)setEncodingType:(NSString*)EncodingType {
		[super setInput:@"EncodingType" toValue:EncodingType];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying which fields to include in a partial response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the GCSContentUri input for this Choreo.
	*(optional, string) The Google Cloud Storage URI where the file content is located. This can be used instead of the Content input.
	 */
	-(void)setGCSContentUri:(NSString*)GCSContentUri {
		[super setInput:@"GCSContentUri" toValue:GCSContentUri];
	}

	/*!
	 * Set the value of the Language input for this Choreo.
	*(optional, string) The language of the document. This is specified with the ISO-639-1 Code (e.g. "en"). If not specified, the language is detected if possible.
	 */
	-(void)setLanguage:(NSString*)Language {
		[super setInput:@"Language" toValue:Language];
	}

	/*!
	 * Set the value of the Type input for this Choreo.
	*(optional, string) The type of content being sent. Valid values are PLAIN_TEXT (the default) and HTML.
	 */
	-(void)setType:(NSString*)Type {
		[super setInput:@"Type" toValue:Type];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the AnalyzeSentiment Choreo.
 */
@implementation TMBGoogle_NaturalLanguage_Documents_AnalyzeSentiment_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Score" output from an execution of this Choreo.
	 * @return - NSString* (decimal) The overall sentiment score of the input document.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getScore {
		return [super getOutputByName:@"Score"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Analyzes the sentiment of the provided text.
 */
@implementation TMBGoogle_NaturalLanguage_Documents_AnalyzeSentiment

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_NaturalLanguage_Documents_AnalyzeSentiment Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/NaturalLanguage/Documents/AnalyzeSentiment"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_NaturalLanguage_Documents_AnalyzeSentiment_ResultSet *results = [[TMBGoogle_NaturalLanguage_Documents_AnalyzeSentiment_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the AnalyzeSentiment Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_NaturalLanguage_Documents_AnalyzeSentiment_Inputs*)newInputSet {
		return [[TMBGoogle_NaturalLanguage_Documents_AnalyzeSentiment_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the AnalyzeSyntax Choreo.
 */
@implementation TMBGoogle_NaturalLanguage_Documents_AnalyzeSyntax_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(conditional, string) The API Key provided by Google.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the Content input for this Choreo.
	*(conditional, string) The content to analyze.
	 */
	-(void)setContent:(NSString*)Content {
		[super setInput:@"Content" toValue:Content];
	}

	/*!
	 * Set the value of the EncodingType input for this Choreo.
	*(optional, string) The encoding type used by the API to calculate sentence offsets. Defaults to UTF-8.
	 */
	-(void)setEncodingType:(NSString*)EncodingType {
		[super setInput:@"EncodingType" toValue:EncodingType];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying which fields to include in a partial response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the GCSContentUri input for this Choreo.
	*(optional, string) The Google Cloud Storage URI where the file content is located. This can be used instead of the Content input.
	 */
	-(void)setGCSContentUri:(NSString*)GCSContentUri {
		[super setInput:@"GCSContentUri" toValue:GCSContentUri];
	}

	/*!
	 * Set the value of the Language input for this Choreo.
	*(optional, string) The language of the document. This is specified with the ISO-639-1 Code (e.g. "en"). If not specified, the language is detected if possible.
	 */
	-(void)setLanguage:(NSString*)Language {
		[super setInput:@"Language" toValue:Language];
	}

	/*!
	 * Set the value of the Type input for this Choreo.
	*(optional, string) The type of content being sent. Valid values are PLAIN_TEXT (the default) and HTML.
	 */
	-(void)setType:(NSString*)Type {
		[super setInput:@"Type" toValue:Type];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the AnalyzeSyntax Choreo.
 */
@implementation TMBGoogle_NaturalLanguage_Documents_AnalyzeSyntax_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Analyzes the syntax of the text and provides sentence boundaries and tokenization along with part of speech tags, dependency trees, and other properties.
 */
@implementation TMBGoogle_NaturalLanguage_Documents_AnalyzeSyntax

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_NaturalLanguage_Documents_AnalyzeSyntax Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/NaturalLanguage/Documents/AnalyzeSyntax"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_NaturalLanguage_Documents_AnalyzeSyntax_ResultSet *results = [[TMBGoogle_NaturalLanguage_Documents_AnalyzeSyntax_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the AnalyzeSyntax Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_NaturalLanguage_Documents_AnalyzeSyntax_Inputs*)newInputSet {
		return [[TMBGoogle_NaturalLanguage_Documents_AnalyzeSyntax_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the AnnotateText Choreo.
 */
@implementation TMBGoogle_NaturalLanguage_Documents_AnnotateText_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(conditional, string) The API Key provided by Google.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the Content input for this Choreo.
	*(conditional, string) The content to analyze.
	 */
	-(void)setContent:(NSString*)Content {
		[super setInput:@"Content" toValue:Content];
	}

	/*!
	 * Set the value of the EncodingType input for this Choreo.
	*(optional, string) The encoding type used by the API to calculate sentence offsets. Defaults to UTF-8.
	 */
	-(void)setEncodingType:(NSString*)EncodingType {
		[super setInput:@"EncodingType" toValue:EncodingType];
	}

	/*!
	 * Set the value of the ExtractDocumentSentiment input for this Choreo.
	*(optional, boolean) Extract document-level sentiment. Defaults to true.
	 */
	-(void)setExtractDocumentSentiment:(NSString*)ExtractDocumentSentiment {
		[super setInput:@"ExtractDocumentSentiment" toValue:ExtractDocumentSentiment];
	}

	/*!
	 * Set the value of the ExtractEntities input for this Choreo.
	*(optional, boolean) Extract entities. Defaults to true.
	 */
	-(void)setExtractEntities:(NSString*)ExtractEntities {
		[super setInput:@"ExtractEntities" toValue:ExtractEntities];
	}

	/*!
	 * Set the value of the ExtractSyntax input for this Choreo.
	*(optional, boolean) Extract syntax information. Defaults to true.
	 */
	-(void)setExtractSyntax:(NSString*)ExtractSyntax {
		[super setInput:@"ExtractSyntax" toValue:ExtractSyntax];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Selector specifying which fields to include in a partial response.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the GCSContentUri input for this Choreo.
	*(optional, string) The Google Cloud Storage URI where the file content is located. This can be used instead of the Content input.
	 */
	-(void)setGCSContentUri:(NSString*)GCSContentUri {
		[super setInput:@"GCSContentUri" toValue:GCSContentUri];
	}

	/*!
	 * Set the value of the Language input for this Choreo.
	*(optional, string) The language of the document. This is specified with the ISO-639-1 Code (e.g. "en"). If not specified, the language is detected if possible.
	 */
	-(void)setLanguage:(NSString*)Language {
		[super setInput:@"Language" toValue:Language];
	}

	/*!
	 * Set the value of the Type input for this Choreo.
	*(optional, string) The type of content being sent. Valid values are PLAIN_TEXT (the default) and HTML.
	 */
	-(void)setType:(NSString*)Type {
		[super setInput:@"Type" toValue:Type];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the AnnotateText Choreo.
 */
@implementation TMBGoogle_NaturalLanguage_Documents_AnnotateText_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * A convenience method that provides all the features that analyzeSentiment, analyzeEntities, and analyzeSyntax provide in one call.
 */
@implementation TMBGoogle_NaturalLanguage_Documents_AnnotateText

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_NaturalLanguage_Documents_AnnotateText Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/NaturalLanguage/Documents/AnnotateText"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_NaturalLanguage_Documents_AnnotateText_ResultSet *results = [[TMBGoogle_NaturalLanguage_Documents_AnnotateText_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the AnnotateText Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_NaturalLanguage_Documents_AnnotateText_Inputs*)newInputSet {
		return [[TMBGoogle_NaturalLanguage_Documents_AnnotateText_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the FinalizeOAuth Choreo.
 */
@implementation TMBGoogle_OAuth_FinalizeOAuth_Inputs

	/*!
	 * Set the value of the AccountName input for this Choreo.
	*(optional, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setAccountName:(NSString*)AccountName {
		[super setInput:@"AccountName" toValue:AccountName];
	}

	/*!
	 * Set the value of the AppKeyName input for this Choreo.
	*(optional, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setAppKeyName:(NSString*)AppKeyName {
		[super setInput:@"AppKeyName" toValue:AppKeyName];
	}

	/*!
	 * Set the value of the AppKeyValue input for this Choreo.
	*(optional, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setAppKeyValue:(NSString*)AppKeyValue {
		[super setInput:@"AppKeyValue" toValue:AppKeyValue];
	}

	/*!
	 * Set the value of the CallbackID input for this Choreo.
	*(required, string) The callback token returned by the InitializeOAuth Choreo. Used to retrieve the authorization code after the user authorizes.
	 */
	-(void)setCallbackID:(NSString*)CallbackID {
		[super setInput:@"CallbackID" toValue:CallbackID];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(required, string) The Client ID provided by Google after registering your application.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(required, string) The Client Secret provided by Google after registering your application.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the SuppressErrors input for this Choreo.
	*(optional, boolean) When set to true, errors received during the OAuth redirect process will be suppressed and returned in the ErrorMessage output.
	 */
	-(void)setSuppressErrors:(NSString*)SuppressErrors {
		[super setInput:@"SuppressErrors" toValue:SuppressErrors];
	}

	/*!
	 * Set the value of the Timeout input for this Choreo.
	*(optional, integer) The amount of time (in seconds) to poll your Temboo callback URL to see if your app's user has allowed or denied the request for access. Defaults to 20. Max is 60.
	 */
	-(void)setTimeout:(NSString*)Timeout {
		[super setInput:@"Timeout" toValue:Timeout];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the FinalizeOAuth Choreo.
 */
@implementation TMBGoogle_OAuth_FinalizeOAuth_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "AccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) The access token for the user that has granted access to your application.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getAccessToken {
		return [super getOutputByName:@"AccessToken"];
	}

	/*!
	 * Retrieve the value of the "ErrorMessage" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains an error message if an error occurs during the OAuth redirect process and if SuppressErrors is set to true.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getErrorMessage {
		return [super getOutputByName:@"ErrorMessage"];
	}

	/*!
	 * Retrieve the value of the "Expires" output from an execution of this Choreo.
	 * @return - NSString* (integer) The remaining lifetime of the short-lived access token.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getExpires {
		return [super getOutputByName:@"Expires"];
	}

	/*!
	 * Retrieve the value of the "RefreshToken" output from an execution of this Choreo.
	 * @return - NSString* (string) A token that may be used to obtain a new access token when the short-lived access token expires.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRefreshToken {
		return [super getOutputByName:@"RefreshToken"];
	}
	
@end

/*!
 * Completes the OAuth process by retrieving a Google access token for a user, after they have visited the authorization URL returned by the InitializeOAuth Choreo and clicked "allow."
 */
@implementation TMBGoogle_OAuth_FinalizeOAuth

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_OAuth_FinalizeOAuth Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/OAuth/FinalizeOAuth"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_OAuth_FinalizeOAuth_ResultSet *results = [[TMBGoogle_OAuth_FinalizeOAuth_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the FinalizeOAuth Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_OAuth_FinalizeOAuth_Inputs*)newInputSet {
		return [[TMBGoogle_OAuth_FinalizeOAuth_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the InitializeOAuth Choreo.
 */
@implementation TMBGoogle_OAuth_InitializeOAuth_Inputs

	/*!
	 * Set the value of the AccountName input for this Choreo.
	*(optional, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setAccountName:(NSString*)AccountName {
		[super setInput:@"AccountName" toValue:AccountName];
	}

	/*!
	 * Set the value of the AppKeyName input for this Choreo.
	*(optional, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setAppKeyName:(NSString*)AppKeyName {
		[super setInput:@"AppKeyName" toValue:AppKeyName];
	}

	/*!
	 * Set the value of the AppKeyValue input for this Choreo.
	*(optional, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setAppKeyValue:(NSString*)AppKeyValue {
		[super setInput:@"AppKeyValue" toValue:AppKeyValue];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(required, string) The Client ID provided by Google after registering your application.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the CustomCallbackID input for this Choreo.
	*(optional, string) A unique identifier that you can pass to eliminate the need to wait for a Temboo generated CallbackID. Callback identifiers may only contain numbers, letters, periods, and hyphens.
	 */
	-(void)setCustomCallbackID:(NSString*)CustomCallbackID {
		[super setInput:@"CustomCallbackID" toValue:CustomCallbackID];
	}

	/*!
	 * Set the value of the ForwardingURL input for this Choreo.
	*(optional, string) The URL that Temboo will redirect your users to after they grant access to your application. This should include the "https://" or "http://" prefix and be a fully qualified URL.
	 */
	-(void)setForwardingURL:(NSString*)ForwardingURL {
		[super setInput:@"ForwardingURL" toValue:ForwardingURL];
	}

	/*!
	 * Set the value of the Scope input for this Choreo.
	*(required, string) A space-delimited list of scopes to requests access for. See Choreo notes for a list of valid scopes.
	 */
	-(void)setScope:(NSString*)Scope {
		[super setInput:@"Scope" toValue:Scope];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the InitializeOAuth Choreo.
 */
@implementation TMBGoogle_OAuth_InitializeOAuth_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "AuthorizationURL" output from an execution of this Choreo.
	 * @return - NSString* (string) The authorization URL that the application's user needs to go to in order to grant access to your application.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getAuthorizationURL {
		return [super getOutputByName:@"AuthorizationURL"];
	}

	/*!
	 * Retrieve the value of the "CallbackID" output from an execution of this Choreo.
	 * @return - NSString* (string) An ID used to retrieve the callback data that Temboo stores once your application's user authorizes.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getCallbackID {
		return [super getOutputByName:@"CallbackID"];
	}
	
@end

/*!
 * Generates an authorization URL that an application can use to complete the first step in the OAuth process.
 */
@implementation TMBGoogle_OAuth_InitializeOAuth

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_OAuth_InitializeOAuth Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/OAuth/InitializeOAuth"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_OAuth_InitializeOAuth_ResultSet *results = [[TMBGoogle_OAuth_InitializeOAuth_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the InitializeOAuth Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_OAuth_InitializeOAuth_Inputs*)newInputSet {
		return [[TMBGoogle_OAuth_InitializeOAuth_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the AddCommentToPhoto Choreo.
 */
@implementation TMBGoogle_Picasa_AddCommentToPhoto_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AlbumID input for this Choreo.
	*(required, integer) The id of the album which contains the photo you want to add a comment to.
	 */
	-(void)setAlbumID:(NSString*)AlbumID {
		[super setInput:@"AlbumID" toValue:AlbumID];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Comment input for this Choreo.
	*(required, string) The comment that you want to add to a photo.
	 */
	-(void)setComment:(NSString*)Comment {
		[super setInput:@"Comment" toValue:Comment];
	}

	/*!
	 * Set the value of the PhotoID input for this Choreo.
	*(required, integer) The id of the photo you want to add a comment to.
	 */
	-(void)setPhotoID:(NSString*)PhotoID {
		[super setInput:@"PhotoID" toValue:PhotoID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) Google Picasa username. Defaults to 'default' which means the server will use the UserID of the user whose access token was specified.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the AddCommentToPhoto Choreo.
 */
@implementation TMBGoogle_Picasa_AddCommentToPhoto_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (xml) The response from Google Picasa.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Adds a comment to a specified photo in Google Picasa.
 */
@implementation TMBGoogle_Picasa_AddCommentToPhoto

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Picasa_AddCommentToPhoto Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Picasa/AddCommentToPhoto"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Picasa_AddCommentToPhoto_ResultSet *results = [[TMBGoogle_Picasa_AddCommentToPhoto_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the AddCommentToPhoto Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Picasa_AddCommentToPhoto_Inputs*)newInputSet {
		return [[TMBGoogle_Picasa_AddCommentToPhoto_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CreateAlbum Choreo.
 */
@implementation TMBGoogle_Picasa_CreateAlbum_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Keywords input for this Choreo.
	*(optional, string) Keywords to associate with the album you are creating separated by commas.
	 */
	-(void)setKeywords:(NSString*)Keywords {
		[super setInput:@"Keywords" toValue:Keywords];
	}

	/*!
	 * Set the value of the PhotoAccess input for this Choreo.
	*(optional, string) The perssion level to specify for photo access. Defaults to 'public'.
	 */
	-(void)setPhotoAccess:(NSString*)PhotoAccess {
		[super setInput:@"PhotoAccess" toValue:PhotoAccess];
	}

	/*!
	 * Set the value of the PhotoLocation input for this Choreo.
	*(optional, string) The location to associate with the photo (i.e. Italy).
	 */
	-(void)setPhotoLocation:(NSString*)PhotoLocation {
		[super setInput:@"PhotoLocation" toValue:PhotoLocation];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Summary input for this Choreo.
	*(optional, string) The album summary.
	 */
	-(void)setSummary:(NSString*)Summary {
		[super setInput:@"Summary" toValue:Summary];
	}

	/*!
	 * Set the value of the Timestamp input for this Choreo.
	*(optional, date) The timestamp to associate with the photo.  Defaults to the current timestamp. Should be an epoch timestamp in milliseconds.
	 */
	-(void)setTimestamp:(NSString*)Timestamp {
		[super setInput:@"Timestamp" toValue:Timestamp];
	}

	/*!
	 * Set the value of the Title input for this Choreo.
	*(required, string) The title of the album.
	 */
	-(void)setTitle:(NSString*)Title {
		[super setInput:@"Title" toValue:Title];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) Google Picasa username. Defaults to 'default' which means the server will use the UserID of the user whose access token was specified.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateAlbum Choreo.
 */
@implementation TMBGoogle_Picasa_CreateAlbum_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (xml) The response from Google Picasa.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Creates a photo album in a Google Picasa account.
 */
@implementation TMBGoogle_Picasa_CreateAlbum

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Picasa_CreateAlbum Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Picasa/CreateAlbum"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Picasa_CreateAlbum_ResultSet *results = [[TMBGoogle_Picasa_CreateAlbum_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CreateAlbum Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Picasa_CreateAlbum_Inputs*)newInputSet {
		return [[TMBGoogle_Picasa_CreateAlbum_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteAlbum Choreo.
 */
@implementation TMBGoogle_Picasa_DeleteAlbum_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AlbumID input for this Choreo.
	*(required, integer) The album id. Note that this can be retrieved by running the ListAlbums Choreo.
	 */
	-(void)setAlbumID:(NSString*)AlbumID {
		[super setInput:@"AlbumID" toValue:AlbumID];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) Google Picasa username. Defaults to 'default' which means the server will use the UserID of the user whose access token was specified.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteAlbum Choreo.
 */
@implementation TMBGoogle_Picasa_DeleteAlbum_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (xml) The response from Google Picasa.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Deletes an album in Google Picasa with a specified album Id.
 */
@implementation TMBGoogle_Picasa_DeleteAlbum

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Picasa_DeleteAlbum Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Picasa/DeleteAlbum"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Picasa_DeleteAlbum_ResultSet *results = [[TMBGoogle_Picasa_DeleteAlbum_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteAlbum Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Picasa_DeleteAlbum_Inputs*)newInputSet {
		return [[TMBGoogle_Picasa_DeleteAlbum_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteTag Choreo.
 */
@implementation TMBGoogle_Picasa_DeleteTag_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AlbumId input for this Choreo.
	*(required, integer) The id for the album which has the tagged photo. Note that this can be retrieved by running the ListAlbums Choreo.
	 */
	-(void)setAlbumId:(NSString*)AlbumId {
		[super setInput:@"AlbumId" toValue:AlbumId];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the PhotoID input for this Choreo.
	*(required, integer) The id for the photo that has a tag to delete.
	 */
	-(void)setPhotoID:(NSString*)PhotoID {
		[super setInput:@"PhotoID" toValue:PhotoID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the TagID input for this Choreo.
	*(required, string) The Id (or tag name) for the tag that you want to delete.
	 */
	-(void)setTagID:(NSString*)TagID {
		[super setInput:@"TagID" toValue:TagID];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) Google Picasa username. Defaults to 'default' which means the server will use the UserID of the user whose access token was specified.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteTag Choreo.
 */
@implementation TMBGoogle_Picasa_DeleteTag_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (xml) The response from Google Picasa.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Removes a tag from a specified photo in Google Picasa.
 */
@implementation TMBGoogle_Picasa_DeleteTag

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Picasa_DeleteTag Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Picasa/DeleteTag"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Picasa_DeleteTag_ResultSet *results = [[TMBGoogle_Picasa_DeleteTag_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteTag Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Picasa_DeleteTag_Inputs*)newInputSet {
		return [[TMBGoogle_Picasa_DeleteTag_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListAlbums Choreo.
 */
@implementation TMBGoogle_Picasa_ListAlbums_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) Google Picasa username. Defaults to 'default' which means the server will use the UserID of the user whose access token was specified.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListAlbums Choreo.
 */
@implementation TMBGoogle_Picasa_ListAlbums_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (xml) The response from Google Picasa.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * List all albums created in a Google Picasa account.
 */
@implementation TMBGoogle_Picasa_ListAlbums

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Picasa_ListAlbums Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Picasa/ListAlbums"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Picasa_ListAlbums_ResultSet *results = [[TMBGoogle_Picasa_ListAlbums_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListAlbums Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Picasa_ListAlbums_Inputs*)newInputSet {
		return [[TMBGoogle_Picasa_ListAlbums_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListPhotosByAlbum Choreo.
 */
@implementation TMBGoogle_Picasa_ListPhotosByAlbum_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AlbumID input for this Choreo.
	*(required, integer) The album id. Note that this can be retrieved by running the ListAlbums Choreo.
	 */
	-(void)setAlbumID:(NSString*)AlbumID {
		[super setInput:@"AlbumID" toValue:AlbumID];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) Google Picasa username. Defaults to 'default' which means the server will use the UserID of the user whose access token was specified.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListPhotosByAlbum Choreo.
 */
@implementation TMBGoogle_Picasa_ListPhotosByAlbum_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (xml) The response from Google Picasa.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Lists all photos in a specified Google Picasa album.
 */
@implementation TMBGoogle_Picasa_ListPhotosByAlbum

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Picasa_ListPhotosByAlbum Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Picasa/ListPhotosByAlbum"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Picasa_ListPhotosByAlbum_ResultSet *results = [[TMBGoogle_Picasa_ListPhotosByAlbum_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListPhotosByAlbum Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Picasa_ListPhotosByAlbum_Inputs*)newInputSet {
		return [[TMBGoogle_Picasa_ListPhotosByAlbum_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the PostPhoto Choreo.
 */
@implementation TMBGoogle_Picasa_PostPhoto_Inputs

	/*!
	 * Set the value of the FileContents input for this Choreo.
	*(conditional, string) The base64 encoded file contents of the photo you want to upload.
	 */
	-(void)setFileContents:(NSString*)FileContents {
		[super setInput:@"FileContents" toValue:FileContents];
	}

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AlbumID input for this Choreo.
	*(optional, integer) The id of the album that you want to upload the photo to. Defaults to 'default' which means the photo will be uploaded to the dropbox album.
	 */
	-(void)setAlbumID:(NSString*)AlbumID {
		[super setInput:@"AlbumID" toValue:AlbumID];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the ImageName input for this Choreo.
	*(required, string) The name of the image you want to upload.
	 */
	-(void)setImageName:(NSString*)ImageName {
		[super setInput:@"ImageName" toValue:ImageName];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) Google Picasa username. Defaults to "default" which means the server will use the UserID of the user whose access token was specified.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}

	/*!
	 * Set the value of the VaultFile input for this Choreo.
	*(vault file) The path to the vault file you want to upload. Required unless using the FileContents input.
	 */
	-(void)setVaultFile:(NSString*)VaultFile {
		[super setInput:@"VaultFile" toValue:VaultFile];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the PostPhoto Choreo.
 */
@implementation TMBGoogle_Picasa_PostPhoto_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (xml) The response from Google Picasa.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Uploads a photo to Google Picasa.
 */
@implementation TMBGoogle_Picasa_PostPhoto

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Picasa_PostPhoto Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Picasa/PostPhoto"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Picasa_PostPhoto_ResultSet *results = [[TMBGoogle_Picasa_PostPhoto_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the PostPhoto Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Picasa_PostPhoto_Inputs*)newInputSet {
		return [[TMBGoogle_Picasa_PostPhoto_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the RetrieveComments Choreo.
 */
@implementation TMBGoogle_Picasa_RetrieveComments_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, integer) The max number of comments to return. Defaults to 10.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) Google Picasa username. Defaults to 'default' which means the server will use the UserID of the user whose access token was specified.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the RetrieveComments Choreo.
 */
@implementation TMBGoogle_Picasa_RetrieveComments_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (xml) The response from Google Picasa.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Returns a feed of the photo comments for a specified user.
 */
@implementation TMBGoogle_Picasa_RetrieveComments

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Picasa_RetrieveComments Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Picasa/RetrieveComments"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Picasa_RetrieveComments_ResultSet *results = [[TMBGoogle_Picasa_RetrieveComments_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the RetrieveComments Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Picasa_RetrieveComments_Inputs*)newInputSet {
		return [[TMBGoogle_Picasa_RetrieveComments_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the RetrieveTagsByUser Choreo.
 */
@implementation TMBGoogle_Picasa_RetrieveTagsByUser_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) Google Picasa username. Defaults to 'default' which means the server will use the UserID of the user whose access token was specified.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the RetrieveTagsByUser Choreo.
 */
@implementation TMBGoogle_Picasa_RetrieveTagsByUser_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (xml) The response from Google Picasa.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Returns a feed of the photo tags for a specified user.
 */
@implementation TMBGoogle_Picasa_RetrieveTagsByUser

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Picasa_RetrieveTagsByUser Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Picasa/RetrieveTagsByUser"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Picasa_RetrieveTagsByUser_ResultSet *results = [[TMBGoogle_Picasa_RetrieveTagsByUser_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the RetrieveTagsByUser Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Picasa_RetrieveTagsByUser_Inputs*)newInputSet {
		return [[TMBGoogle_Picasa_RetrieveTagsByUser_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SearchPhotosByTag Choreo.
 */
@implementation TMBGoogle_Picasa_SearchPhotosByTag_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Tag input for this Choreo.
	*(required, string) A photo tag to search by. Multiple tags can be separated by commas.
	 */
	-(void)setTag:(NSString*)Tag {
		[super setInput:@"Tag" toValue:Tag];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) Google Picasa username. Defaults to 'default' which means the server will use the UserID of the user whose access token was specified.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SearchPhotosByTag Choreo.
 */
@implementation TMBGoogle_Picasa_SearchPhotosByTag_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (xml) The response from Google Picasa.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Searches photos using tags that you specify.
 */
@implementation TMBGoogle_Picasa_SearchPhotosByTag

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Picasa_SearchPhotosByTag Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Picasa/SearchPhotosByTag"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Picasa_SearchPhotosByTag_ResultSet *results = [[TMBGoogle_Picasa_SearchPhotosByTag_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SearchPhotosByTag Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Picasa_SearchPhotosByTag_Inputs*)newInputSet {
		return [[TMBGoogle_Picasa_SearchPhotosByTag_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the TagPhoto Choreo.
 */
@implementation TMBGoogle_Picasa_TagPhoto_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AlbumID input for this Choreo.
	*(required, integer) The id of the album which contains the photo you want to tag.
	 */
	-(void)setAlbumID:(NSString*)AlbumID {
		[super setInput:@"AlbumID" toValue:AlbumID];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Google. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the PhotoID input for this Choreo.
	*(required, integer) The id of the photo you want to tag.
	 */
	-(void)setPhotoID:(NSString*)PhotoID {
		[super setInput:@"PhotoID" toValue:PhotoID];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the Tag input for this Choreo.
	*(required, string) The text for the photo tag.
	 */
	-(void)setTag:(NSString*)Tag {
		[super setInput:@"Tag" toValue:Tag];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) Google Picasa username. Defaults to "default" which means the server will use the UserID of the user whose access token was specified.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the TagPhoto Choreo.
 */
@implementation TMBGoogle_Picasa_TagPhoto_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "AccessToken" output from an execution of this Choreo.
	 * @return - NSString* (optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getAccessToken {
		return [super getOutputByName:@"AccessToken"];
	}

	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (xml) The response from Google Picasa.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Creates a tag for a specified photo in Google Picasa.
 */
@implementation TMBGoogle_Picasa_TagPhoto

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Picasa_TagPhoto Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Picasa/TagPhoto"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Picasa_TagPhoto_ResultSet *results = [[TMBGoogle_Picasa_TagPhoto_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the TagPhoto Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Picasa_TagPhoto_Inputs*)newInputSet {
		return [[TMBGoogle_Picasa_TagPhoto_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeletePlace Choreo.
 */
@implementation TMBGoogle_Places_DeletePlace_Inputs

	/*!
	 * Set the value of the Key input for this Choreo.
	*(required, string) The API Key provided by Google.
	 */
	-(void)setKey:(NSString*)Key {
		[super setInput:@"Key" toValue:Key];
	}

	/*!
	 * Set the value of the PlaceID input for this Choreo.
	*(conditional, string) A textual identifier that uniquely identifies a place.
	 */
	-(void)setPlaceID:(NSString*)PlaceID {
		[super setInput:@"PlaceID" toValue:PlaceID];
	}

	/*!
	 * Set the value of the PlaceReference input for this Choreo.
	*(optional, string) A textual identifier that uniquely identifies a place. Note, this parameter is deprecated as of June 24, 2014. Use PlaceID instead.
	 */
	-(void)setPlaceReference:(NSString*)PlaceReference {
		[super setInput:@"PlaceReference" toValue:PlaceReference];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Sensor input for this Choreo.
	*(optional, boolean) Indicates whether or not the directions request is from a device with a location sensor. Value must be either 1 or 0. Defaults to 0 (false).
	 */
	-(void)setSensor:(NSString*)Sensor {
		[super setInput:@"Sensor" toValue:Sensor];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeletePlace Choreo.
 */
@implementation TMBGoogle_Places_DeletePlace_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Delete a new Place from Google Places.
 */
@implementation TMBGoogle_Places_DeletePlace

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Places_DeletePlace Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Places/DeletePlace"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Places_DeletePlace_ResultSet *results = [[TMBGoogle_Places_DeletePlace_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeletePlace Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Places_DeletePlace_Inputs*)newInputSet {
		return [[TMBGoogle_Places_DeletePlace_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the PlaceDetails Choreo.
 */
@implementation TMBGoogle_Places_PlaceDetails_Inputs

	/*!
	 * Set the value of the Key input for this Choreo.
	*(required, string) The API Key provided by Google.
	 */
	-(void)setKey:(NSString*)Key {
		[super setInput:@"Key" toValue:Key];
	}

	/*!
	 * Set the value of the Language input for this Choreo.
	*(optional, string) The language code, indicating in which language the results should be returned, if possible. See Choreo notes for a list of supported languages and their codes.
	 */
	-(void)setLanguage:(NSString*)Language {
		[super setInput:@"Language" toValue:Language];
	}

	/*!
	 * Set the value of the PlaceID input for this Choreo.
	*(conditional, string) A textual identifier that uniquely identifies a place. PlaceIDs are returned by the PlaceSearch Choreo.
	 */
	-(void)setPlaceID:(NSString*)PlaceID {
		[super setInput:@"PlaceID" toValue:PlaceID];
	}

	/*!
	 * Set the value of the Reference input for this Choreo.
	*(optional, string) A textual identifier that uniquely identifies a place. Note, this parameter is deprecated as of June 24, 2014. Use PlaceID instead.
	 */
	-(void)setReference:(NSString*)Reference {
		[super setInput:@"Reference" toValue:Reference];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Sensor input for this Choreo.
	*(optional, boolean) Indicates whether or not the directions request is from a device with a location sensor. Value must be either 1 or 0. Defaults to 0 (false).
	 */
	-(void)setSensor:(NSString*)Sensor {
		[super setInput:@"Sensor" toValue:Sensor];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the PlaceDetails Choreo.
 */
@implementation TMBGoogle_Places_PlaceDetails_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieve detailed information about places retrieved by the PlaceSearch Choreo.
 */
@implementation TMBGoogle_Places_PlaceDetails

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Places_PlaceDetails Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Places/PlaceDetails"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Places_PlaceDetails_ResultSet *results = [[TMBGoogle_Places_PlaceDetails_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the PlaceDetails Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Places_PlaceDetails_Inputs*)newInputSet {
		return [[TMBGoogle_Places_PlaceDetails_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the PlaceSearch Choreo.
 */
@implementation TMBGoogle_Places_PlaceSearch_Inputs

	/*!
	 * Set the value of the Key input for this Choreo.
	*(required, string) The API Key provided by Google.
	 */
	-(void)setKey:(NSString*)Key {
		[super setInput:@"Key" toValue:Key];
	}

	/*!
	 * Set the value of the Keyword input for this Choreo.
	*(optional, string) Enter a keyword (term, address, type, customer review, etc.) to be matched against all results retrieved for this Place.
	 */
	-(void)setKeyword:(NSString*)Keyword {
		[super setInput:@"Keyword" toValue:Keyword];
	}

	/*!
	 * Set the value of the Language input for this Choreo.
	*(optional, string) The language code, indicating in which language the results should be returned, if possible. See Choreo notes for a list of supported languages and their codes.
	 */
	-(void)setLanguage:(NSString*)Language {
		[super setInput:@"Language" toValue:Language];
	}

	/*!
	 * Set the value of the Latitude input for this Choreo.
	*(required, string) Specify a latitude point around which Places results will be retrieved (e.g., 38.898717).
	 */
	-(void)setLatitude:(NSString*)Latitude {
		[super setInput:@"Latitude" toValue:Latitude];
	}

	/*!
	 * Set the value of the Longitude input for this Choreo.
	*(required, string) Specify a longitude point around which Places results will be retrieved (e.g., -77.035974).
	 */
	-(void)setLongitude:(NSString*)Longitude {
		[super setInput:@"Longitude" toValue:Longitude];
	}

	/*!
	 * Set the value of the MaxPrice input for this Choreo.
	*(optional, integer) Restricts results to only those places within the specified range. Valid values range between 0 (most affordable) to 4 (most expensive), inclusive. The exact amount will vary from region to region.
	 */
	-(void)setMaxPrice:(NSString*)MaxPrice {
		[super setInput:@"MaxPrice" toValue:MaxPrice];
	}

	/*!
	 * Set the value of the MinPrice input for this Choreo.
	*(optional, integer) Restricts results to only those places within the specified range. Valid values range between 0 (most affordable) to 4 (most expensive), inclusive. The exact amount will vary from region to region.
	 */
	-(void)setMinPrice:(NSString*)MinPrice {
		[super setInput:@"MinPrice" toValue:MinPrice];
	}

	/*!
	 * Set the value of the Name input for this Choreo.
	*(optional, string) Enter a name to be matched when results are retrieved for this specified Place.
	 */
	-(void)setName:(NSString*)Name {
		[super setInput:@"Name" toValue:Name];
	}

	/*!
	 * Set the value of the OpenNow input for this Choreo.
	*(optional, boolean) Returns only those Places that are open for business at the time the query is sent. Places that do not specify opening hours in the Google Places database will not be returned.
	 */
	-(void)setOpenNow:(NSString*)OpenNow {
		[super setInput:@"OpenNow" toValue:OpenNow];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) The "NextPageToken" returned in the choreo output from a previous run. Used to page through large result sets. When the PageToken is specified, all other inputs are ignored.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the Radius input for this Choreo.
	*(required, integer) Specify the radius in meters for which Places results will be returned. Maximum radius is limited to 50,000 meters. If rankby=distance, then radius must not be specified.
	 */
	-(void)setRadius:(NSString*)Radius {
		[super setInput:@"Radius" toValue:Radius];
	}

	/*!
	 * Set the value of the RankBy input for this Choreo.
	*(optional, string) Specify how results are listed. Values include: prominence (default); distance - sorts results by distance from specified location. Radius must not be used, and Keyword, Name, or Types are required).
	 */
	-(void)setRankBy:(NSString*)RankBy {
		[super setInput:@"RankBy" toValue:RankBy];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Sensor input for this Choreo.
	*(optional, boolean) Indicates whether or not the directions request is from a device with a location sensor. Value must be either 1 or 0. Defaults to 0 (false).
	 */
	-(void)setSensor:(NSString*)Sensor {
		[super setInput:@"Sensor" toValue:Sensor];
	}

	/*!
	 * Set the value of the Types input for this Choreo.
	*(optional, string) Filter results by types, such as: bar, dentist.  Multiple types must be separated by the pipe ("|") symbol: bar|dentist|airport.
	 */
	-(void)setTypes:(NSString*)Types {
		[super setInput:@"Types" toValue:Types];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the PlaceSearch Choreo.
 */
@implementation TMBGoogle_Places_PlaceSearch_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Google.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Search for places based on latitude/longitude coordinates, keywords, and distance.
 */
@implementation TMBGoogle_Places_PlaceSearch

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGoogle_Places_PlaceSearch Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Google/Places/PlaceSearch"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGoogle_Places_PlaceSearch_ResultSet *results = [[TMBGoogle_Places_PlaceSearch_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the PlaceSearch Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGoogle_Places_PlaceSearch_Inputs*)newInputSet {
		return [[TMBGoogle_Places_PlaceSearch_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the RadarSearch Choreo.
 */
@implementation TMBGoogle_Places_RadarSearch_Inputs

	/*!
	 * Set the value of the Key input for this Choreo.
	*(required, string) The API Key provided by Google.
	 */
	-(void)setKey:(NSString*)Key {
		[super setInput:@"Key" toValue:Key];
	}

	/*!
	 * Set the value of the Keyword input for this Choreo.
	*(conditional, string) Enter a keyword (term, address, type, customer review, etc.) to be matched against all results retrieved for this Place.  At least one of Keyword, Name or Types must be specified.
	 */
	-(void)setKeyword:(NSString*)Keyword {
		[super setInput:@"Keyword" toValue:Keyword];
	}

	/*!
	 * Set the value of the Language input for this Choreo.
	*(optional, string) The language code, indicating in which language the results should be returned, if possible. See Choreo notes for a list of supported languages and their codes.
	 */
	-(void)setLanguage:(NSString*)Language {
		[super setInput:@"Language" toValue:Language];
	}

	/*!
	 * Set the value of the Latitude input for this Choreo.
	*(required, string) Specify a latitude point around which Places results will be retrieved (e.g., 38.898717).
	 */
	-(void)setLatitude:(NSString*)Latitude {
		[super setInput:@"Latitude" toValue:Latitude];
	}

	/*!
	 * Set the value of the Longitude input for this Choreo.
	*(required, string) Specify a longitude point around which Places results will be retrieved (e.g., -77.035974).
	 */
	-(void)setLongitude:(NSString*)Longitude {
		[super setInput:@"Longitude" toValue:Longitude];
	}

	/*!
	 * Set the value of the MaxPrice input for this Choreo.
	*(optional, integer) Restricts results to only those places within the specified range. Valid values range between 0 (most affordable) to 4 (most expensive), inclusive. The exact amount will vary from region to region.
	 */
	-(void)setMaxPrice:(NSString*)MaxPrice {
		[super setInput:@"MaxPrice" toValue:MaxPrice];
	}

	/*!
	 * Set the value of the MinPrice input for this Choreo.
	*(optional, integer) Restricts results to only those places within the specified range. Valid values range between 0 (most affordable) to 4 (most expensive), inclusive. The exact amount will vary from region to region.
	 */
	-(void)setMinPrice:(NSString*)MinPrice {
		[super setInput:@"MinPrice" toValue:MinPrice];
	}

	/*!
	 * Set the value of the Name input for this Choreo.
	*(conditional, string) Enter a name to be matched when results are retrieved for this specified Place.  At least one of Keyword, Name or Types must be specified.
	 */
	-(void)setName:(NSString*)Name {
		[super setInput:@"Name" toValue:Name];
	}

	/*!
	 * Set the value of the OpenNow input for this Choreo.
	*(optional, boolean) Returns only those Places that are open for business at the time the query is sent. Places that do not specify opening hours in the Google Places database will not be returned.
	 */
	-(void)setOpenNow:(NSString*)OpenNow {
		[super setInput:@"OpenNow" toValue:OpenNow];
	}

	/*!
	 * Set the value of the PageToken input for this Choreo.
	*(optional, string) The "NextPageToken" returned in the choreo output from a previous run. Used to page through large result sets. When the PageToken is specified, all other inputs are ignored.
	 */
	-(void)setPageToken:(NSString*)PageToken {
		[super setInput:@"PageToken" toValue:PageToken];
	}

	/*!
	 * Set the value of the Radius input for this Choreo.
	*(req