/*!
 * @TMBPusher.m
 *
 * Execute Choreographies from the Temboo Pusher bundle.
 *
 * iOS version 10.6.8
 *
 * LICENSE: Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @category   Services
 * @author     Reid Simonton, Temboo, Inc.
 * @copyright  2013 Temboo, Inc.
 * @license    http://www.apache.org/licenses/LICENSE-2.0 Apache License 2.0
 * @version    1.7
 */

#import "TMBPusher.h"
#import "TMBChoreography_Protected.h"
#import "TMBChoreographyExecution.h"
#import "TMBTembooSession.h"
#import "TMBTembooUri.h"


/*!
 * Input object with appropriate setters for specifying arguments to the GetChannel Choreo.
 */
@implementation TMBPusher_Channels_GetChannel_Inputs

	/*!
	 * Set the value of the AppID input for this Choreo.
	*(required, string) The application ID provided by Pusher.
	 */
	-(void)setAppID:(NSString*)AppID {
		[super setInput:@"AppID" toValue:AppID];
	}

	/*!
	 * Set the value of the AppKey input for this Choreo.
	*(required, string) The authenticaion key provided by Pusher.
	 */
	-(void)setAppKey:(NSString*)AppKey {
		[super setInput:@"AppKey" toValue:AppKey];
	}

	/*!
	 * Set the value of the AppSecret input for this Choreo.
	*(required, string) The authentication secret provided by Pusher.
	 */
	-(void)setAppSecret:(NSString*)AppSecret {
		[super setInput:@"AppSecret" toValue:AppSecret];
	}

	/*!
	 * Set the value of the ChannelName input for this Choreo.
	*(required, string) The name of the channel to retrieve.
	 */
	-(void)setChannelName:(NSString*)ChannelName {
		[super setInput:@"ChannelName" toValue:ChannelName];
	}

	/*!
	 * Set the value of the Info input for this Choreo.
	*(optional, string) A comma separated list of attributes which should be returned for the channel (e.g., user_count, subscription_count). Note that the user_count attribute is only applicable for presence channels.
	 */
	-(void)setInfo:(NSString*)Info {
		[super setInput:@"Info" toValue:Info];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetChannel Choreo.
 */
@implementation TMBPusher_Channels_GetChannel_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Pusher.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves information for a specific channel.
 */
@implementation TMBPusher_Channels_GetChannel

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBPusher_Channels_GetChannel Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Pusher/Channels/GetChannel"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBPusher_Channels_GetChannel_ResultSet *results = [[TMBPusher_Channels_GetChannel_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetChannel Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBPusher_Channels_GetChannel_Inputs*)newInputSet {
		return [[TMBPusher_Channels_GetChannel_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListChannels Choreo.
 */
@implementation TMBPusher_Channels_ListChannels_Inputs

	/*!
	 * Set the value of the AppID input for this Choreo.
	*(required, string) The application ID provided by Pusher.
	 */
	-(void)setAppID:(NSString*)AppID {
		[super setInput:@"AppID" toValue:AppID];
	}

	/*!
	 * Set the value of the AppKey input for this Choreo.
	*(required, string) The authenticaion key provided by Pusher.
	 */
	-(void)setAppKey:(NSString*)AppKey {
		[super setInput:@"AppKey" toValue:AppKey];
	}

	/*!
	 * Set the value of the AppSecret input for this Choreo.
	*(required, string) The authentication secret provided by Pusher.
	 */
	-(void)setAppSecret:(NSString*)AppSecret {
		[super setInput:@"AppSecret" toValue:AppSecret];
	}

	/*!
	 * Set the value of the FilterByPrefix input for this Choreo.
	*(optional, string) Filter the returned channels by a specific prefix (e.g. "presence-").
	 */
	-(void)setFilterByPrefix:(NSString*)FilterByPrefix {
		[super setInput:@"FilterByPrefix" toValue:FilterByPrefix];
	}

	/*!
	 * Set the value of the Info input for this Choreo.
	*(optional, string) A comma separated list of attributes which should be returned for each channel (e.g., user_count). Note that the user_count attribute is only applicable for presence channels.
	 */
	-(void)setInfo:(NSString*)Info {
		[super setInput:@"Info" toValue:Info];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListChannels Choreo.
 */
@implementation TMBPusher_Channels_ListChannels_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Pusher.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves information for occupied channels.
 */
@implementation TMBPusher_Channels_ListChannels

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBPusher_Channels_ListChannels Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Pusher/Channels/ListChannels"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBPusher_Channels_ListChannels_ResultSet *results = [[TMBPusher_Channels_ListChannels_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListChannels Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBPusher_Channels_ListChannels_Inputs*)newInputSet {
		return [[TMBPusher_Channels_ListChannels_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the PostEvent Choreo.
 */
@implementation TMBPusher_Events_PostEvent_Inputs

	/*!
	 * Set the value of the AppID input for this Choreo.
	*(required, string) The application ID provided by Pusher.
	 */
	-(void)setAppID:(NSString*)AppID {
		[super setInput:@"AppID" toValue:AppID];
	}

	/*!
	 * Set the value of the AppKey input for this Choreo.
	*(required, string) The authenticaion key provided by Pusher.
	 */
	-(void)setAppKey:(NSString*)AppKey {
		[super setInput:@"AppKey" toValue:AppKey];
	}

	/*!
	 * Set the value of the AppSecret input for this Choreo.
	*(required, string) The authentication secret provided by Pusher.
	 */
	-(void)setAppSecret:(NSString*)AppSecret {
		[super setInput:@"AppSecret" toValue:AppSecret];
	}

	/*!
	 * Set the value of the ChannelName input for this Choreo.
	*(required, string) The name of the channel to publish to. This can also be a list of channels formatted as a JSON array.
	 */
	-(void)setChannelName:(NSString*)ChannelName {
		[super setInput:@"ChannelName" toValue:ChannelName];
	}

	/*!
	 * Set the value of the EventData input for this Choreo.
	*(required, json) A JSON object containing the event data.
	 */
	-(void)setEventData:(NSString*)EventData {
		[super setInput:@"EventData" toValue:EventData];
	}

	/*!
	 * Set the value of the EventName input for this Choreo.
	*(required, string) The name of the event being triggered.
	 */
	-(void)setEventName:(NSString*)EventName {
		[super setInput:@"EventName" toValue:EventName];
	}

	/*!
	 * Set the value of the SocketID input for this Choreo.
	*(optional, string) Excludes the event from being sent to a specific connection.
	 */
	-(void)setSocketID:(NSString*)SocketID {
		[super setInput:@"SocketID" toValue:SocketID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the PostEvent Choreo.
 */
@implementation TMBPusher_Events_PostEvent_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Pusher. A successful request returns an empty JSON hash.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Triggers an event on one or more channels.
 */
@implementation TMBPusher_Events_PostEvent

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBPusher_Events_PostEvent Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Pusher/Events/PostEvent"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBPusher_Events_PostEvent_ResultSet *results = [[TMBPusher_Events_PostEvent_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the PostEvent Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBPusher_Events_PostEvent_Inputs*)newInputSet {
		return [[TMBPusher_Events_PostEvent_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListUsers Choreo.
 */
@implementation TMBPusher_Users_ListUsers_Inputs

	/*!
	 * Set the value of the AppID input for this Choreo.
	*(required, string) The application ID provided by Pusher.
	 */
	-(void)setAppID:(NSString*)AppID {
		[super setInput:@"AppID" toValue:AppID];
	}

	/*!
	 * Set the value of the AppKey input for this Choreo.
	*(required, string) The authenticaion key provided by Pusher.
	 */
	-(void)setAppKey:(NSString*)AppKey {
		[super setInput:@"AppKey" toValue:AppKey];
	}

	/*!
	 * Set the value of the AppSecret input for this Choreo.
	*(required, string) The authentication secret provided by Pusher.
	 */
	-(void)setAppSecret:(NSString*)AppSecret {
		[super setInput:@"AppSecret" toValue:AppSecret];
	}

	/*!
	 * Set the value of the ChannelName input for this Choreo.
	*(required, string) The name of the channel that the users are subscribed to.
	 */
	-(void)setChannelName:(NSString*)ChannelName {
		[super setInput:@"ChannelName" toValue:ChannelName];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListUsers Choreo.
 */
@implementation TMBPusher_Users_ListUsers_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Pusher.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves IDs for users currently subscribed to a presence channel.
 */
@implementation TMBPusher_Users_ListUsers

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBPusher_Users_ListUsers Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Pusher/Users/ListUsers"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBPusher_Users_ListUsers_ResultSet *results = [[TMBPusher_Users_ListUsers_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListUsers Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBPusher_Users_ListUsers_Inputs*)newInputSet {
		return [[TMBPusher_Users_ListUsers_Inputs alloc] init];
	}
@end
	