/*!
 * @TMBInstagram.m
 *
 * Execute Choreographies from the Temboo Instagram bundle.
 *
 * iOS version 10.6.8
 *
 * LICENSE: Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @category   Services
 * @author     Reid Simonton, Temboo, Inc.
 * @copyright  2013 Temboo, Inc.
 * @license    http://www.apache.org/licenses/LICENSE-2.0 Apache License 2.0
 * @version    1.7
 */

#import "TMBInstagram.h"
#import "TMBChoreography_Protected.h"
#import "TMBChoreographyExecution.h"
#import "TMBTembooSession.h"
#import "TMBTembooUri.h"


/*!
 * Input object with appropriate setters for specifying arguments to the CreateMediaComment Choreo.
 */
@implementation TMBInstagram_CreateMediaComment_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token retrieved during the OAuth 2.0 process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the MediaID input for this Choreo.
	*(required, string) The ID of the media object to leave a comment on.
	 */
	-(void)setMediaID:(NSString*)MediaID {
		[super setInput:@"MediaID" toValue:MediaID];
	}

	/*!
	 * Set the value of the Text input for this Choreo.
	*(required, string) The text to post as a comment on the media.
	 */
	-(void)setText:(NSString*)Text {
		[super setInput:@"Text" toValue:Text];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateMediaComment Choreo.
 */
@implementation TMBInstagram_CreateMediaComment_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Instagram.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Creates a comment on a specified media object. 
 */
@implementation TMBInstagram_CreateMediaComment

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBInstagram_CreateMediaComment Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Instagram/CreateMediaComment"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBInstagram_CreateMediaComment_ResultSet *results = [[TMBInstagram_CreateMediaComment_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CreateMediaComment Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBInstagram_CreateMediaComment_Inputs*)newInputSet {
		return [[TMBInstagram_CreateMediaComment_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteMediaComment Choreo.
 */
@implementation TMBInstagram_DeleteMediaComment_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token retrieved during the OAuth 2.0 process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the CommentID input for this Choreo.
	*(required, string) The ID of the comment to delete.
	 */
	-(void)setCommentID:(NSString*)CommentID {
		[super setInput:@"CommentID" toValue:CommentID];
	}

	/*!
	 * Set the value of the MediaID input for this Choreo.
	*(required, string) The ID of the media object that you want to delete comments from.
	 */
	-(void)setMediaID:(NSString*)MediaID {
		[super setInput:@"MediaID" toValue:MediaID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteMediaComment Choreo.
 */
@implementation TMBInstagram_DeleteMediaComment_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Instagram.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Removes a specified comment from a user's media.
 */
@implementation TMBInstagram_DeleteMediaComment

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBInstagram_DeleteMediaComment Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Instagram/DeleteMediaComment"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBInstagram_DeleteMediaComment_ResultSet *results = [[TMBInstagram_DeleteMediaComment_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteMediaComment Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBInstagram_DeleteMediaComment_Inputs*)newInputSet {
		return [[TMBInstagram_DeleteMediaComment_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetFollowerRequests Choreo.
 */
@implementation TMBInstagram_GetFollowerRequests_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token retrieved during the OAuth 2.0 process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetFollowerRequests Choreo.
 */
@implementation TMBInstagram_GetFollowerRequests_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Instagram.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a list of users who have requested permission to follow a given user.
 */
@implementation TMBInstagram_GetFollowerRequests

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBInstagram_GetFollowerRequests Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Instagram/GetFollowerRequests"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBInstagram_GetFollowerRequests_ResultSet *results = [[TMBInstagram_GetFollowerRequests_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetFollowerRequests Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBInstagram_GetFollowerRequests_Inputs*)newInputSet {
		return [[TMBInstagram_GetFollowerRequests_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetLikedMediaForUser Choreo.
 */
@implementation TMBInstagram_GetLikedMediaForUser_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token retrieved during the OAuth 2.0 process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the Count input for this Choreo.
	*(optional, integer) The number of results to return.
	 */
	-(void)setCount:(NSString*)Count {
		[super setInput:@"Count" toValue:Count];
	}

	/*!
	 * Set the value of the MaxID input for this Choreo.
	*(optional, string) Returns media liked before this id.
	 */
	-(void)setMaxID:(NSString*)MaxID {
		[super setInput:@"MaxID" toValue:MaxID];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The ID of the user to retrieve media for. Defaults to 'self' indicating that the authenticating user is assumed.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetLikedMediaForUser Choreo.
 */
@implementation TMBInstagram_GetLikedMediaForUser_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Instagram.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the authenticated user's list of media they've liked.
 */
@implementation TMBInstagram_GetLikedMediaForUser

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBInstagram_GetLikedMediaForUser Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Instagram/GetLikedMediaForUser"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBInstagram_GetLikedMediaForUser_ResultSet *results = [[TMBInstagram_GetLikedMediaForUser_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetLikedMediaForUser Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBInstagram_GetLikedMediaForUser_Inputs*)newInputSet {
		return [[TMBInstagram_GetLikedMediaForUser_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetLocationInformation Choreo.
 */
@implementation TMBInstagram_GetLocationInformation_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(conditional, string) The access token retrieved during the OAuth 2.0 process. Required unless you provide the ClientID.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Instagram after registering your application. Required unless you provide an AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the LocationID input for this Choreo.
	*(required, integer) The ID for the location that you want to retrieve information for.
	 */
	-(void)setLocationID:(NSString*)LocationID {
		[super setInput:@"LocationID" toValue:LocationID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetLocationInformation Choreo.
 */
@implementation TMBInstagram_GetLocationInformation_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Instagram.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves information about a location.
 */
@implementation TMBInstagram_GetLocationInformation

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBInstagram_GetLocationInformation Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Instagram/GetLocationInformation"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBInstagram_GetLocationInformation_ResultSet *results = [[TMBInstagram_GetLocationInformation_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetLocationInformation Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBInstagram_GetLocationInformation_Inputs*)newInputSet {
		return [[TMBInstagram_GetLocationInformation_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetMediaByID Choreo.
 */
@implementation TMBInstagram_GetMediaByID_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(conditional, string) The access token retrieved during the OAuth 2.0 process. Required unless you provide the ClientID.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Instagram after registering your application. Required unless you provide an AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the MediaID input for this Choreo.
	*(required, string) The ID of the media object you want to retrieve.
	 */
	-(void)setMediaID:(NSString*)MediaID {
		[super setInput:@"MediaID" toValue:MediaID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetMediaByID Choreo.
 */
@implementation TMBInstagram_GetMediaByID_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Instagram.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves information about a specified media object.
 */
@implementation TMBInstagram_GetMediaByID

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBInstagram_GetMediaByID Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Instagram/GetMediaByID"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBInstagram_GetMediaByID_ResultSet *results = [[TMBInstagram_GetMediaByID_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetMediaByID Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBInstagram_GetMediaByID_Inputs*)newInputSet {
		return [[TMBInstagram_GetMediaByID_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetMediaByLikes Choreo.
 */
@implementation TMBInstagram_GetMediaByLikes_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(conditional, string) The access token retrieved during the OAuth 2.0 process. Required unless you provide the ClientID.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Instagram after registering your application. Required unless you provide an AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the MediaID input for this Choreo.
	*(required, string) The ID of the media object you want to retrieve.
	 */
	-(void)setMediaID:(NSString*)MediaID {
		[super setInput:@"MediaID" toValue:MediaID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetMediaByLikes Choreo.
 */
@implementation TMBInstagram_GetMediaByLikes_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Instagram.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a list of users who like the specified media.
 */
@implementation TMBInstagram_GetMediaByLikes

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBInstagram_GetMediaByLikes Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Instagram/GetMediaByLikes"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBInstagram_GetMediaByLikes_ResultSet *results = [[TMBInstagram_GetMediaByLikes_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetMediaByLikes Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBInstagram_GetMediaByLikes_Inputs*)newInputSet {
		return [[TMBInstagram_GetMediaByLikes_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetMediaComments Choreo.
 */
@implementation TMBInstagram_GetMediaComments_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(conditional, string) The access token retrieved during the OAuth 2.0 process. Required unless you provide the ClientID.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Instagram after registering your application. Required unless you provide an AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the MediaID input for this Choreo.
	*(required, string) The ID of the media object that you want to get comments for.
	 */
	-(void)setMediaID:(NSString*)MediaID {
		[super setInput:@"MediaID" toValue:MediaID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetMediaComments Choreo.
 */
@implementation TMBInstagram_GetMediaComments_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Instagram.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a full list of comments for a specified media object.
 */
@implementation TMBInstagram_GetMediaComments

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBInstagram_GetMediaComments Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Instagram/GetMediaComments"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBInstagram_GetMediaComments_ResultSet *results = [[TMBInstagram_GetMediaComments_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetMediaComments Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBInstagram_GetMediaComments_Inputs*)newInputSet {
		return [[TMBInstagram_GetMediaComments_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetRecentMediaForLocation Choreo.
 */
@implementation TMBInstagram_GetRecentMediaForLocation_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(conditional, string) The access token retrieved during the OAuth 2.0 process. Required unless you provide the ClientID.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Instagram after registering your application. Required unless you provide an AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the LocationID input for this Choreo.
	*(required, integer) The ID for the location that you want to retrieve information for.
	 */
	-(void)setLocationID:(NSString*)LocationID {
		[super setInput:@"LocationID" toValue:LocationID];
	}

	/*!
	 * Set the value of the MaxID input for this Choreo.
	*(optional, string) Returns media after this max_id.
	 */
	-(void)setMaxID:(NSString*)MaxID {
		[super setInput:@"MaxID" toValue:MaxID];
	}

	/*!
	 * Set the value of the MaxTimestamp input for this Choreo.
	*(optional, date) Returns media before this UNIX timestamp.
	 */
	-(void)setMaxTimestamp:(NSString*)MaxTimestamp {
		[super setInput:@"MaxTimestamp" toValue:MaxTimestamp];
	}

	/*!
	 * Set the value of the MinID input for this Choreo.
	*(optional, string) Returns media before this min_id.
	 */
	-(void)setMinID:(NSString*)MinID {
		[super setInput:@"MinID" toValue:MinID];
	}

	/*!
	 * Set the value of the MinTimestamp input for this Choreo.
	*(optional, date) Returns media after this UNIX timestamp.
	 */
	-(void)setMinTimestamp:(NSString*)MinTimestamp {
		[super setInput:@"MinTimestamp" toValue:MinTimestamp];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetRecentMediaForLocation Choreo.
 */
@implementation TMBInstagram_GetRecentMediaForLocation_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Instagram.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a list of recent media objects from a given location.
 */
@implementation TMBInstagram_GetRecentMediaForLocation

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBInstagram_GetRecentMediaForLocation Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Instagram/GetRecentMediaForLocation"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBInstagram_GetRecentMediaForLocation_ResultSet *results = [[TMBInstagram_GetRecentMediaForLocation_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetRecentMediaForLocation Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBInstagram_GetRecentMediaForLocation_Inputs*)newInputSet {
		return [[TMBInstagram_GetRecentMediaForLocation_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetRecentMediaForUser Choreo.
 */
@implementation TMBInstagram_GetRecentMediaForUser_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token retrieved during the OAuth 2.0 process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the Count input for this Choreo.
	*(optional, integer) The number of results to return.
	 */
	-(void)setCount:(NSString*)Count {
		[super setInput:@"Count" toValue:Count];
	}

	/*!
	 * Set the value of the MaxID input for this Choreo.
	*(optional, string) Return media liked before this id.
	 */
	-(void)setMaxID:(NSString*)MaxID {
		[super setInput:@"MaxID" toValue:MaxID];
	}

	/*!
	 * Set the value of the MinID input for this Choreo.
	*(optional, string) Returns media later than this min_id.
	 */
	-(void)setMinID:(NSString*)MinID {
		[super setInput:@"MinID" toValue:MinID];
	}

	/*!
	 * Set the value of the MinTimestamp input for this Choreo.
	*(optional, date) Returns media after this UNIX timestamp.
	 */
	-(void)setMinTimestamp:(NSString*)MinTimestamp {
		[super setInput:@"MinTimestamp" toValue:MinTimestamp];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The ID of the user whose media to return. Defaults to 'self' indicating that the authenticating user is assumed.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetRecentMediaForUser Choreo.
 */
@implementation TMBInstagram_GetRecentMediaForUser_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Instagram.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the most recent media published by a user.
 */
@implementation TMBInstagram_GetRecentMediaForUser

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBInstagram_GetRecentMediaForUser Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Instagram/GetRecentMediaForUser"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBInstagram_GetRecentMediaForUser_ResultSet *results = [[TMBInstagram_GetRecentMediaForUser_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetRecentMediaForUser Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBInstagram_GetRecentMediaForUser_Inputs*)newInputSet {
		return [[TMBInstagram_GetRecentMediaForUser_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetRelationship Choreo.
 */
@implementation TMBInstagram_GetRelationship_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token retrieved during the OAuth 2.0 process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(required, string) The ID of the target user.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetRelationship Choreo.
 */
@implementation TMBInstagram_GetRelationship_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Instagram.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves information about the relationship between the authenticating user and the specified user.
 */
@implementation TMBInstagram_GetRelationship

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBInstagram_GetRelationship Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Instagram/GetRelationship"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBInstagram_GetRelationship_ResultSet *results = [[TMBInstagram_GetRelationship_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetRelationship Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBInstagram_GetRelationship_Inputs*)newInputSet {
		return [[TMBInstagram_GetRelationship_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetUserInformation Choreo.
 */
@implementation TMBInstagram_GetUserInformation_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token retrieved during the OAuth 2.0 process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) The ID of the user that you want to retrieve. Defaults to 'self' indicating that the authenticating user is assumed.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetUserInformation Choreo.
 */
@implementation TMBInstagram_GetUserInformation_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Instagram.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves basic information about a user.
 */
@implementation TMBInstagram_GetUserInformation

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBInstagram_GetUserInformation Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Instagram/GetUserInformation"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBInstagram_GetUserInformation_ResultSet *results = [[TMBInstagram_GetUserInformation_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetUserInformation Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBInstagram_GetUserInformation_Inputs*)newInputSet {
		return [[TMBInstagram_GetUserInformation_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the LikeMedia Choreo.
 */
@implementation TMBInstagram_LikeMedia_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token retrieved during the OAuth 2.0 process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the MediaID input for this Choreo.
	*(required, string) The ID of the media to like.
	 */
	-(void)setMediaID:(NSString*)MediaID {
		[super setInput:@"MediaID" toValue:MediaID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the LikeMedia Choreo.
 */
@implementation TMBInstagram_LikeMedia_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Instagram.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Sets the specified media as being liked by the authenticating user.
 */
@implementation TMBInstagram_LikeMedia

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBInstagram_LikeMedia Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Instagram/LikeMedia"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBInstagram_LikeMedia_ResultSet *results = [[TMBInstagram_LikeMedia_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the LikeMedia Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBInstagram_LikeMedia_Inputs*)newInputSet {
		return [[TMBInstagram_LikeMedia_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ModifyRelationship Choreo.
 */
@implementation TMBInstagram_ModifyRelationship_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token retrieved during the OAuth 2.0 process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the Action input for this Choreo.
	*(required, string) The type of relationship modification to perform. Valid values are: follow, unfollow, block, unblock, approve, or deny.
	 */
	-(void)setAction:(NSString*)Action {
		[super setInput:@"Action" toValue:Action];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(required, string) The ID of the target user.
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ModifyRelationship Choreo.
 */
@implementation TMBInstagram_ModifyRelationship_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Instagram.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Modifies the relationship between the authenticating user and the target user.
 */
@implementation TMBInstagram_ModifyRelationship

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBInstagram_ModifyRelationship Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Instagram/ModifyRelationship"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBInstagram_ModifyRelationship_ResultSet *results = [[TMBInstagram_ModifyRelationship_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ModifyRelationship Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBInstagram_ModifyRelationship_Inputs*)newInputSet {
		return [[TMBInstagram_ModifyRelationship_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the FinalizeOAuth Choreo.
 */
@implementation TMBInstagram_OAuth_FinalizeOAuth_Inputs

	/*!
	 * Set the value of the AccountName input for this Choreo.
	*(optional, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setAccountName:(NSString*)AccountName {
		[super setInput:@"AccountName" toValue:AccountName];
	}

	/*!
	 * Set the value of the AppKeyName input for this Choreo.
	*(optional, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setAppKeyName:(NSString*)AppKeyName {
		[super setInput:@"AppKeyName" toValue:AppKeyName];
	}

	/*!
	 * Set the value of the AppKeyValue input for this Choreo.
	*(optional, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setAppKeyValue:(NSString*)AppKeyValue {
		[super setInput:@"AppKeyValue" toValue:AppKeyValue];
	}

	/*!
	 * Set the value of the CallbackID input for this Choreo.
	*(required, string) The callback token returned by the InitializeOAuth Choreo. Used to retrieve the authorization code after the user authorizes.
	 */
	-(void)setCallbackID:(NSString*)CallbackID {
		[super setInput:@"CallbackID" toValue:CallbackID];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(required, string) The Client ID provided by Instagram after registering your application.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(required, string) The Client Secret provided by Instagram after registering your application.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the SuppressErrors input for this Choreo.
	*(optional, boolean) When set to true, errors received during the OAuth redirect process will be suppressed and returned in the ErrorMessage output.
	 */
	-(void)setSuppressErrors:(NSString*)SuppressErrors {
		[super setInput:@"SuppressErrors" toValue:SuppressErrors];
	}

	/*!
	 * Set the value of the Timeout input for this Choreo.
	*(optional, integer) The amount of time (in seconds) to poll your Temboo callback URL to see if your app's user has allowed or denied the request for access. Defaults to 20. Max is 60.
	 */
	-(void)setTimeout:(NSString*)Timeout {
		[super setInput:@"Timeout" toValue:Timeout];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the FinalizeOAuth Choreo.
 */
@implementation TMBInstagram_OAuth_FinalizeOAuth_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "AccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) The access token for the user that has granted access to your application.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getAccessToken {
		return [super getOutputByName:@"AccessToken"];
	}

	/*!
	 * Retrieve the value of the "ErrorMessage" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains an error message if an error occurs during the OAuth redirect process and if SuppressErrors is set to true.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getErrorMessage {
		return [super getOutputByName:@"ErrorMessage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Instagram.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Completes the OAuth process by retrieving a Instagram access token for a user, after they have visited the authorization URL returned by the InitializeOAuth Choreo and clicked "allow."
 */
@implementation TMBInstagram_OAuth_FinalizeOAuth

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBInstagram_OAuth_FinalizeOAuth Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Instagram/OAuth/FinalizeOAuth"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBInstagram_OAuth_FinalizeOAuth_ResultSet *results = [[TMBInstagram_OAuth_FinalizeOAuth_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the FinalizeOAuth Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBInstagram_OAuth_FinalizeOAuth_Inputs*)newInputSet {
		return [[TMBInstagram_OAuth_FinalizeOAuth_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the InitializeOAuth Choreo.
 */
@implementation TMBInstagram_OAuth_InitializeOAuth_Inputs

	/*!
	 * Set the value of the AccountName input for this Choreo.
	*(optional, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setAccountName:(NSString*)AccountName {
		[super setInput:@"AccountName" toValue:AccountName];
	}

	/*!
	 * Set the value of the AppKeyName input for this Choreo.
	*(optional, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setAppKeyName:(NSString*)AppKeyName {
		[super setInput:@"AppKeyName" toValue:AppKeyName];
	}

	/*!
	 * Set the value of the AppKeyValue input for this Choreo.
	*(optional, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setAppKeyValue:(NSString*)AppKeyValue {
		[super setInput:@"AppKeyValue" toValue:AppKeyValue];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(required, string) The Client ID provided by Instagram after registering your application.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the CustomCallbackID input for this Choreo.
	*(optional, string) A unique identifier that you can pass to eliminate the need to wait for a Temboo generated CallbackID. Callback identifiers may only contain numbers, letters, periods, and hyphens.
	 */
	-(void)setCustomCallbackID:(NSString*)CustomCallbackID {
		[super setInput:@"CustomCallbackID" toValue:CustomCallbackID];
	}

	/*!
	 * Set the value of the ForwardingURL input for this Choreo.
	*(optional, string) The URL that Temboo will redirect your users to after they grant access to your application. This should include the "https://" or "http://" prefix and be a fully qualified URL.
	 */
	-(void)setForwardingURL:(NSString*)ForwardingURL {
		[super setInput:@"ForwardingURL" toValue:ForwardingURL];
	}

	/*!
	 * Set the value of the Scope input for this Choreo.
	*(conditional, string) The scope of the access you're requesting from the user. Valid values are: basic, public_content, follower_list, comments, relationships, and likes. Multiple scope values should separated by spaces.
	 */
	-(void)setScope:(NSString*)Scope {
		[super setInput:@"Scope" toValue:Scope];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the InitializeOAuth Choreo.
 */
@implementation TMBInstagram_OAuth_InitializeOAuth_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "AuthorizationURL" output from an execution of this Choreo.
	 * @return - NSString* (string) The authorization URL that the application's user needs to go to in order to grant access to your application.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getAuthorizationURL {
		return [super getOutputByName:@"AuthorizationURL"];
	}

	/*!
	 * Retrieve the value of the "CallbackID" output from an execution of this Choreo.
	 * @return - NSString* (string) An ID used to retrieve the callback data that Temboo stores once your application's user authorizes.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getCallbackID {
		return [super getOutputByName:@"CallbackID"];
	}
	
@end

/*!
 * Generates an authorization URL that an application can use to complete the first step in the OAuth process.
 */
@implementation TMBInstagram_OAuth_InitializeOAuth

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBInstagram_OAuth_InitializeOAuth Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Instagram/OAuth/InitializeOAuth"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBInstagram_OAuth_InitializeOAuth_ResultSet *results = [[TMBInstagram_OAuth_InitializeOAuth_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the InitializeOAuth Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBInstagram_OAuth_InitializeOAuth_Inputs*)newInputSet {
		return [[TMBInstagram_OAuth_InitializeOAuth_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the RecentlyTaggedMedia Choreo.
 */
@implementation TMBInstagram_RecentlyTaggedMedia_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(conditional, string) The access token retrieved during the OAuth 2.0 process. Required unless you provide the ClientID.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Instagram after registering your application. Required unless you provide the AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the Count input for this Choreo.
	*(optional, integer) The number of results to return.
	 */
	-(void)setCount:(NSString*)Count {
		[super setInput:@"Count" toValue:Count];
	}

	/*!
	 * Set the value of the MaxID input for this Choreo.
	*(optional, integer) Returns media tagged earlier than this max_tag_id. Used to paginate through results.
	 */
	-(void)setMaxID:(NSString*)MaxID {
		[super setInput:@"MaxID" toValue:MaxID];
	}

	/*!
	 * Set the value of the MinID input for this Choreo.
	*(optional, integer) Returns media tagged later than this max_tag_id. Used to paginate through results.
	 */
	-(void)setMinID:(NSString*)MinID {
		[super setInput:@"MinID" toValue:MinID];
	}

	/*!
	 * Set the value of the TagName input for this Choreo.
	*(required, string) Enter a valid tag identifier, such as: nofliter
	 */
	-(void)setTagName:(NSString*)TagName {
		[super setInput:@"TagName" toValue:TagName];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the RecentlyTaggedMedia Choreo.
 */
@implementation TMBInstagram_RecentlyTaggedMedia_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Instagram.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a list of recently tagged media.
 */
@implementation TMBInstagram_RecentlyTaggedMedia

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBInstagram_RecentlyTaggedMedia Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Instagram/RecentlyTaggedMedia"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBInstagram_RecentlyTaggedMedia_ResultSet *results = [[TMBInstagram_RecentlyTaggedMedia_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the RecentlyTaggedMedia Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBInstagram_RecentlyTaggedMedia_Inputs*)newInputSet {
		return [[TMBInstagram_RecentlyTaggedMedia_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SearchLocations Choreo.
 */
@implementation TMBInstagram_SearchLocations_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(conditional, string) The access token retrieved during the OAuth 2.0 process. Required unless you provide the ClientID.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Instagram after registering your application. Required unless you provide the AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the Distance input for this Choreo.
	*(optional, integer) The distance to search. Default is 1000m (distance=1000), max distance is 5000.
	 */
	-(void)setDistance:(NSString*)Distance {
		[super setInput:@"Distance" toValue:Distance];
	}

	/*!
	 * Set the value of the FoursquareID input for this Choreo.
	*(conditional, string) Returns a location mapped off of a foursquare v2 api location id. If used, you are not required to provide values for Latitude or Longitude.
	 */
	-(void)setFoursquareID:(NSString*)FoursquareID {
		[super setInput:@"FoursquareID" toValue:FoursquareID];
	}

	/*!
	 * Set the value of the Latitude input for this Choreo.
	*(conditional, decimal) Latitude of the center search coordinate. If used, Longitude is required.
	 */
	-(void)setLatitude:(NSString*)Latitude {
		[super setInput:@"Latitude" toValue:Latitude];
	}

	/*!
	 * Set the value of the Longitude input for this Choreo.
	*(conditional, decimal) Longitude of the center search coordinate. If used, Latitude is required.
	 */
	-(void)setLongitude:(NSString*)Longitude {
		[super setInput:@"Longitude" toValue:Longitude];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SearchLocations Choreo.
 */
@implementation TMBInstagram_SearchLocations_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Instagram.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Searches for locations by geographic coordinates. 
 */
@implementation TMBInstagram_SearchLocations

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBInstagram_SearchLocations Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Instagram/SearchLocations"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBInstagram_SearchLocations_ResultSet *results = [[TMBInstagram_SearchLocations_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SearchLocations Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBInstagram_SearchLocations_Inputs*)newInputSet {
		return [[TMBInstagram_SearchLocations_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SearchMedia Choreo.
 */
@implementation TMBInstagram_SearchMedia_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(conditional, string) The access token retrieved during the OAuth 2.0 process. Required unless you provide the ClientID.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Instagram after registering your application. Required unless you provide the AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the Distance input for this Choreo.
	*(optional, integer) The distance in meters. Defaults to 1000. Max is 5000.
	 */
	-(void)setDistance:(NSString*)Distance {
		[super setInput:@"Distance" toValue:Distance];
	}

	/*!
	 * Set the value of the Latitude input for this Choreo.
	*(required, decimal) Latitude of the center search coordinate.
	 */
	-(void)setLatitude:(NSString*)Latitude {
		[super setInput:@"Latitude" toValue:Latitude];
	}

	/*!
	 * Set the value of the Longitude input for this Choreo.
	*(required, decimal) Longitude of the center search coordinate.
	 */
	-(void)setLongitude:(NSString*)Longitude {
		[super setInput:@"Longitude" toValue:Longitude];
	}

	/*!
	 * Set the value of the MaxTimestamp input for this Choreo.
	*(optional, date) A unix timestamp. All media returned will be taken earlier than this timestamp.
	 */
	-(void)setMaxTimestamp:(NSString*)MaxTimestamp {
		[super setInput:@"MaxTimestamp" toValue:MaxTimestamp];
	}

	/*!
	 * Set the value of the MinTimestamp input for this Choreo.
	*(optional, date) A unix timestamp. All media returned will be taken later than this timestamp.
	 */
	-(void)setMinTimestamp:(NSString*)MinTimestamp {
		[super setInput:@"MinTimestamp" toValue:MinTimestamp];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SearchMedia Choreo.
 */
@implementation TMBInstagram_SearchMedia_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Instagram.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Allows you to search for media in a given area.
 */
@implementation TMBInstagram_SearchMedia

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBInstagram_SearchMedia Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Instagram/SearchMedia"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBInstagram_SearchMedia_ResultSet *results = [[TMBInstagram_SearchMedia_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SearchMedia Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBInstagram_SearchMedia_Inputs*)newInputSet {
		return [[TMBInstagram_SearchMedia_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SearchTags Choreo.
 */
@implementation TMBInstagram_SearchTags_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(conditional, string) The access token retrieved during the OAuth 2.0 process. Required unless you provide the ClientID.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Instagram after registering your application. Required unless you provide an AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the Query input for this Choreo.
	*(required, string) A tag name to search for (e.g., snow, beach). Do not include a leading hash symbol in your query.
	 */
	-(void)setQuery:(NSString*)Query {
		[super setInput:@"Query" toValue:Query];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SearchTags Choreo.
 */
@implementation TMBInstagram_SearchTags_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Instagram.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Searches for tags by name.
 */
@implementation TMBInstagram_SearchTags

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBInstagram_SearchTags Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Instagram/SearchTags"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBInstagram_SearchTags_ResultSet *results = [[TMBInstagram_SearchTags_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SearchTags Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBInstagram_SearchTags_Inputs*)newInputSet {
		return [[TMBInstagram_SearchTags_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SearchUsers Choreo.
 */
@implementation TMBInstagram_SearchUsers_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(conditional, string) The access token retrieved during the OAuth 2.0 process. Required unless you provide the ClientID.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Instagram after registering your application. Required unless you provide an AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the Count input for this Choreo.
	*(optional, integer) The number of results to return.
	 */
	-(void)setCount:(NSString*)Count {
		[super setInput:@"Count" toValue:Count];
	}

	/*!
	 * Set the value of the Query input for this Choreo.
	*(required, string) The query string to use for the user search.
	 */
	-(void)setQuery:(NSString*)Query {
		[super setInput:@"Query" toValue:Query];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SearchUsers Choreo.
 */
@implementation TMBInstagram_SearchUsers_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Instagram.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Allows you to search for users by name.
 */
@implementation TMBInstagram_SearchUsers

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBInstagram_SearchUsers Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Instagram/SearchUsers"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBInstagram_SearchUsers_ResultSet *results = [[TMBInstagram_SearchUsers_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SearchUsers Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBInstagram_SearchUsers_Inputs*)newInputSet {
		return [[TMBInstagram_SearchUsers_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the TagName Choreo.
 */
@implementation TMBInstagram_TagName_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(conditional, string) The access token retrieved during the OAuth 2.0 process. Required unless you provide the ClientID.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Instagram after registering your application. Required unless you provide an AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the TagName input for this Choreo.
	*(required, string) Enter a valid tag identifier, such as: nofliter
	 */
	-(void)setTagName:(NSString*)TagName {
		[super setInput:@"TagName" toValue:TagName];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the TagName Choreo.
 */
@implementation TMBInstagram_TagName_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Instagram.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves information about a tag object.
 */
@implementation TMBInstagram_TagName

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBInstagram_TagName Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Instagram/TagName"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBInstagram_TagName_ResultSet *results = [[TMBInstagram_TagName_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the TagName Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBInstagram_TagName_Inputs*)newInputSet {
		return [[TMBInstagram_TagName_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UnlikeMedia Choreo.
 */
@implementation TMBInstagram_UnlikeMedia_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token retrieved during the OAuth 2.0 process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the MediaID input for this Choreo.
	*(required, string) The ID of the media to unlike.
	 */
	-(void)setMediaID:(NSString*)MediaID {
		[super setInput:@"MediaID" toValue:MediaID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UnlikeMedia Choreo.
 */
@implementation TMBInstagram_UnlikeMedia_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Instagram.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Allows a user to unlike a specified media object.
 */
@implementation TMBInstagram_UnlikeMedia

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBInstagram_UnlikeMedia Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Instagram/UnlikeMedia"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBInstagram_UnlikeMedia_ResultSet *results = [[TMBInstagram_UnlikeMedia_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UnlikeMedia Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBInstagram_UnlikeMedia_Inputs*)newInputSet {
		return [[TMBInstagram_UnlikeMedia_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UserFollows Choreo.
 */
@implementation TMBInstagram_UserFollows_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(conditional, string) The access token retrieved during the OAuth 2.0 process. Required unless you provide the ClientID.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Instagram after registering your application. Required unless you provide the AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the Count input for this Choreo.
	*(optional, integer) The number of results to return.
	 */
	-(void)setCount:(NSString*)Count {
		[super setInput:@"Count" toValue:Count];
	}

	/*!
	 * Set the value of the Cursor input for this Choreo.
	*(optional, string) The "next cursor" used to retrieve the next page of results.
	 */
	-(void)setCursor:(NSString*)Cursor {
		[super setInput:@"Cursor" toValue:Cursor];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(required, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UserFollows Choreo.
 */
@implementation TMBInstagram_UserFollows_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Instagram.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieve the list of users that the authenticating user is following.
 */
@implementation TMBInstagram_UserFollows

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBInstagram_UserFollows Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Instagram/UserFollows"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBInstagram_UserFollows_ResultSet *results = [[TMBInstagram_UserFollows_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UserFollows Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBInstagram_UserFollows_Inputs*)newInputSet {
		return [[TMBInstagram_UserFollows_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UserIsFollowedBy Choreo.
 */
@implementation TMBInstagram_UserIsFollowedBy_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(conditional, string) The access token retrieved during the OAuth 2.0 process. Required unless you provide the ClientID.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Instagram after registering your application. Required unless you provide the AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the Count input for this Choreo.
	*(optional, integer) The number of results to return.
	 */
	-(void)setCount:(NSString*)Count {
		[super setInput:@"Count" toValue:Count];
	}

	/*!
	 * Set the value of the Cursor input for this Choreo.
	*(optional, string) The "next cursor" used to retrieve the next page of results.
	 */
	-(void)setCursor:(NSString*)Cursor {
		[super setInput:@"Cursor" toValue:Cursor];
	}

	/*!
	 * Set the value of the UserID input for this Choreo.
	*(optional, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setUserID:(NSString*)UserID {
		[super setInput:@"UserID" toValue:UserID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UserIsFollowedBy Choreo.
 */
@implementation TMBInstagram_UserIsFollowedBy_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Instagram.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieve the list of users following the authenticating user.
 */
@implementation TMBInstagram_UserIsFollowedBy

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBInstagram_UserIsFollowedBy Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Instagram/UserIsFollowedBy"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBInstagram_UserIsFollowedBy_ResultSet *results = [[TMBInstagram_UserIsFollowedBy_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UserIsFollowedBy Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBInstagram_UserIsFollowedBy_Inputs*)newInputSet {
		return [[TMBInstagram_UserIsFollowedBy_Inputs alloc] init];
	}
@end
	