/*!
 * @TMBGovTrack.m
 *
 * Execute Choreographies from the Temboo GovTrack bundle.
 *
 * iOS version 10.6.8
 *
 * LICENSE: Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @category   Services
 * @author     Reid Simonton, Temboo, Inc.
 * @copyright  2013 Temboo, Inc.
 * @license    http://www.apache.org/licenses/LICENSE-2.0 Apache License 2.0
 * @version    1.7
 */

#import "TMBGovTrack.h"
#import "TMBChoreography_Protected.h"
#import "TMBChoreographyExecution.h"
#import "TMBTembooSession.h"
#import "TMBTembooUri.h"


/*!
 * Input object with appropriate setters for specifying arguments to the Bill Choreo.
 */
@implementation TMBGovTrack_Bill_Inputs

	/*!
	 * Set the value of the BillID input for this Choreo.
	*(optional, integer) The ID number of the bill to retrieve. When using this input, all other filter parameters are ignored, and a single record is returned.
	 */
	-(void)setBillID:(NSString*)BillID {
		[super setInput:@"BillID" toValue:BillID];
	}

	/*!
	 * Set the value of the BillType input for this Choreo.
	*(optional, string) The bill's type (e.g. house_resolution, senate_bill, house_bill, etc). Filter operators allowed. Sortable.
	 */
	-(void)setBillType:(NSString*)BillType {
		[super setInput:@"BillType" toValue:BillType];
	}

	/*!
	 * Set the value of the CoSponsors input for this Choreo.
	*(optional, string) The bill's cosponsors. When using this filter, provide the id of the cosponsor which is returned when requesting a single bill object. Filter operators allowed. Sortable.
	 */
	-(void)setCoSponsors:(NSString*)CoSponsors {
		[super setInput:@"CoSponsors" toValue:CoSponsors];
	}

	/*!
	 * Set the value of the Committees input for this Choreo.
	*(optional, string) Committees to which the bill has been referred. When using this filter, provide the id of the committee which is returned when requesting a single bill object. Filter operators allowed. Sortable.
	 */
	-(void)setCommittees:(NSString*)Committees {
		[super setInput:@"Committees" toValue:Committees];
	}

	/*!
	 * Set the value of the Congress input for this Choreo.
	*(optional, string) The number of the congress in which the bill was introduced. The current congress is 113. Filter operators allowed. Sortable.
	 */
	-(void)setCongress:(NSString*)Congress {
		[super setInput:@"Congress" toValue:Congress];
	}

	/*!
	 * Set the value of the CurrentStatus input for this Choreo.
	*(optional, string) The current status of the bill (e.g. passed_bill, prov_kill_veto, introduced, etc). Filter operators allowed. Sortable.
	 */
	-(void)setCurrentStatus:(NSString*)CurrentStatus {
		[super setInput:@"CurrentStatus" toValue:CurrentStatus];
	}

	/*!
	 * Set the value of the CurrentStatusDate input for this Choreo.
	*(optional, string) The date of the last major action on the bill corresponding to the CurrentStatus (in YYYY-MM-DD format). Filter operators allowed. Sortable.
	 */
	-(void)setCurrentStatusDate:(NSString*)CurrentStatusDate {
		[super setInput:@"CurrentStatusDate" toValue:CurrentStatusDate];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) A comma separated list of fields to return in the response. Use double-underscores to span relationships (e.g. person__firstname).
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the IntroducedDate input for this Choreo.
	*(optional, string) The date the bill was introduced (in YYYY-MM-DD format). Filter operators allowed. Sortable.
	 */
	-(void)setIntroducedDate:(NSString*)IntroducedDate {
		[super setInput:@"IntroducedDate" toValue:IntroducedDate];
	}

	/*!
	 * Set the value of the Limit input for this Choreo.
	*(optional, integer) Results are paged 100 per call by default. Set the limit input to a high value to get all of the results at once.
	 */
	-(void)setLimit:(NSString*)Limit {
		[super setInput:@"Limit" toValue:Limit];
	}

	/*!
	 * Set the value of the Number input for this Choreo.
	*(optional, string) The bill's number. This is different from the BillID. Filter operators allowed. Sortable.
	 */
	-(void)setNumber:(NSString*)Number {
		[super setInput:@"Number" toValue:Number];
	}

	/*!
	 * Set the value of the Offset input for this Choreo.
	*(optional, integer) Offset the results by the number given, useful for paging through results.
	 */
	-(void)setOffset:(NSString*)Offset {
		[super setInput:@"Offset" toValue:Offset];
	}

	/*!
	 * Set the value of the Query input for this Choreo.
	*(conditional, string) Filters according to a full-text search on the object.
	 */
	-(void)setQuery:(NSString*)Query {
		[super setInput:@"Query" toValue:Query];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: json (the default) and xml.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Sort input for this Choreo.
	*(optional, string) You can order the results using fieldname (ascending) or -fieldname (descending) where "fieldname" is one of the variables that is listed as 'Sortable' in the description. Ex: '-congress'
	 */
	-(void)setSort:(NSString*)Sort {
		[super setInput:@"Sort" toValue:Sort];
	}

	/*!
	 * Set the value of the Sponsor input for this Choreo.
	*(optional, string) The ID of the sponsor of the bill. Filter operators allowed. Sortable.
	 */
	-(void)setSponsor:(NSString*)Sponsor {
		[super setInput:@"Sponsor" toValue:Sponsor];
	}

	/*!
	 * Set the value of the Terms input for this Choreo.
	*(optional, string) Subject areas associated with the bill. When using this filter, provide the id of the term which is returned when requesting a single bill object. Filter operators allowed. Sortable.
	 */
	-(void)setTerms:(NSString*)Terms {
		[super setInput:@"Terms" toValue:Terms];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Bill Choreo.
 */
@implementation TMBGovTrack_Bill_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from GovTrack.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves bills and resolutions in the U.S. Congress since 1973 (the 93rd Congress).
 */
@implementation TMBGovTrack_Bill

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGovTrack_Bill Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/GovTrack/Bill"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGovTrack_Bill_ResultSet *results = [[TMBGovTrack_Bill_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Bill Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGovTrack_Bill_Inputs*)newInputSet {
		return [[TMBGovTrack_Bill_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Committee Choreo.
 */
@implementation TMBGovTrack_Committee_Inputs

	/*!
	 * Set the value of the Committee input for this Choreo.
	*(optional, string) Indicates if the object is a committee or a subcommittee. To filter for committees, you can pass "null". For subcommittees, pass the ID of the parent. Filter operators allowed. Sortable.
	 */
	-(void)setCommittee:(NSString*)Committee {
		[super setInput:@"Committee" toValue:Committee];
	}

	/*!
	 * Set the value of the CommitteeID input for this Choreo.
	*(optional, integer) The id of the committee resource. When using this input, all other filter parameters are ignored, and a single record is returned.
	 */
	-(void)setCommitteeID:(NSString*)CommitteeID {
		[super setInput:@"CommitteeID" toValue:CommitteeID];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) A comma separated list of fields to return in the response. Use double-underscores to span relationships (e.g. person__firstname).
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Limit input for this Choreo.
	*(optional, integer) Results are paged 100 per call by default. Set the limit input to a high value to get all of the results at once.
	 */
	-(void)setLimit:(NSString*)Limit {
		[super setInput:@"Limit" toValue:Limit];
	}

	/*!
	 * Set the value of the Obsolete input for this Choreo.
	*(optional, string) Whether or not the committee still exists. Set to "true" to return committees that are obsolete. Filter operators allowed. Sortable.
	 */
	-(void)setObsolete:(NSString*)Obsolete {
		[super setInput:@"Obsolete" toValue:Obsolete];
	}

	/*!
	 * Set the value of the Offset input for this Choreo.
	*(optional, integer) Offset the results by the number given, useful for paging through results.
	 */
	-(void)setOffset:(NSString*)Offset {
		[super setInput:@"Offset" toValue:Offset];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: json (the default) and xml.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Sort input for this Choreo.
	*(optional, string) You can order the results using fieldname (ascending) or -fieldname (descending) where "fieldname" is one of the variables that is listed as 'Sortable' in the description. Ex: '-lastname'
	 */
	-(void)setSort:(NSString*)Sort {
		[super setInput:@"Sort" toValue:Sort];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Committee Choreo.
 */
@implementation TMBGovTrack_Committee_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from GovTrack.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Returns committees and subcommittees in the United States Congress, including historical committees.
 */
@implementation TMBGovTrack_Committee

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGovTrack_Committee Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/GovTrack/Committee"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGovTrack_Committee_ResultSet *results = [[TMBGovTrack_Committee_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Committee Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGovTrack_Committee_Inputs*)newInputSet {
		return [[TMBGovTrack_Committee_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CommitteeMember Choreo.
 */
@implementation TMBGovTrack_CommitteeMember_Inputs

	/*!
	 * Set the value of the Committee input for this Choreo.
	*(optional, string) The committee or subcommittee being served on. To filter by this field, you can pass the ID of the committee. Filter operators allowed. Sortable.
	 */
	-(void)setCommittee:(NSString*)Committee {
		[super setInput:@"Committee" toValue:Committee];
	}

	/*!
	 * Set the value of the CommitteeMemberID input for this Choreo.
	*(optional, integer) The ID of the committee member resource. When using this input, all other filter parameters are ignored, and a single record is returned.
	 */
	-(void)setCommitteeMemberID:(NSString*)CommitteeMemberID {
		[super setInput:@"CommitteeMemberID" toValue:CommitteeMemberID];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) A comma separated list of fields to return in the response. Use double-underscores to span relationships (e.g. person__firstname).
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Limit input for this Choreo.
	*(optional, integer) Results are paged 100 per call by default. Set the limit input to a high value to get all of the results at once.
	 */
	-(void)setLimit:(NSString*)Limit {
		[super setInput:@"Limit" toValue:Limit];
	}

	/*!
	 * Set the value of the Offset input for this Choreo.
	*(optional, integer) Offset the results by the number given, useful for paging through results.
	 */
	-(void)setOffset:(NSString*)Offset {
		[super setInput:@"Offset" toValue:Offset];
	}

	/*!
	 * Set the value of the Person input for this Choreo.
	*(optional, string) The ID of the Member of Congress serving on a committee. Filter operators allowed. Sortable.
	 */
	-(void)setPerson:(NSString*)Person {
		[super setInput:@"Person" toValue:Person];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: json (the default) and xml.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Sort input for this Choreo.
	*(optional, string) You can order the results using fieldname (ascending) or -fieldname (descending) where "fieldname" is one of the variables that is listed as 'Sortable' in the description. Ex: '-lastname'
	 */
	-(void)setSort:(NSString*)Sort {
		[super setInput:@"Sort" toValue:Sort];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CommitteeMember Choreo.
 */
@implementation TMBGovTrack_CommitteeMember_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from GovTrack.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Returns records indicating the current membership of a Member of Congress on a committee or subcommittee.
 */
@implementation TMBGovTrack_CommitteeMember

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGovTrack_CommitteeMember Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/GovTrack/CommitteeMember"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGovTrack_CommitteeMember_ResultSet *results = [[TMBGovTrack_CommitteeMember_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CommitteeMember Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGovTrack_CommitteeMember_Inputs*)newInputSet {
		return [[TMBGovTrack_CommitteeMember_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Person Choreo.
 */
@implementation TMBGovTrack_Person_Inputs

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) A comma separated list of fields to return in the response. Use double-underscores to span relationships (e.g. person__firstname).
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Gender input for this Choreo.
	*(optional, string) The person's gender (male or female). For historical data, gender is sometimes not specified. Filter operators allowed. Sortable.
	 */
	-(void)setGender:(NSString*)Gender {
		[super setInput:@"Gender" toValue:Gender];
	}

	/*!
	 * Set the value of the LastName input for this Choreo.
	*(optional, string) The representative's last name. Filter operators allowed. Sortable.
	 */
	-(void)setLastName:(NSString*)LastName {
		[super setInput:@"LastName" toValue:LastName];
	}

	/*!
	 * Set the value of the Limit input for this Choreo.
	*(optional, integer) Results are paged 100 per call by default. Set the limit input to a high value to get all of the results at once.
	 */
	-(void)setLimit:(NSString*)Limit {
		[super setInput:@"Limit" toValue:Limit];
	}

	/*!
	 * Set the value of the Offset input for this Choreo.
	*(optional, integer) Offset the results by the number given, useful for paging through results.
	 */
	-(void)setOffset:(NSString*)Offset {
		[super setInput:@"Offset" toValue:Offset];
	}

	/*!
	 * Set the value of the PersonID input for this Choreo.
	*(optional, integer) The ID number for a person to retrieve. When using this input, all other filter parameters are ignored, and a single record is returned.
	 */
	-(void)setPersonID:(NSString*)PersonID {
		[super setInput:@"PersonID" toValue:PersonID];
	}

	/*!
	 * Set the value of the Query input for this Choreo.
	*(conditional, string) Filters according to a full-text search on the object.
	 */
	-(void)setQuery:(NSString*)Query {
		[super setInput:@"Query" toValue:Query];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: json (the default) and xml.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Sort input for this Choreo.
	*(optional, string) You can order the results using fieldname (ascending) or -fieldname (descending) where "fieldname" is one of the variables that is listed as 'Sortable' in the description. Ex: '-lastname'
	 */
	-(void)setSort:(NSString*)Sort {
		[super setInput:@"Sort" toValue:Sort];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Person Choreo.
 */
@implementation TMBGovTrack_Person_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from GovTrack.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Returns members of Congress and U.S. Presidents since the founding of the nation.
 */
@implementation TMBGovTrack_Person

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGovTrack_Person Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/GovTrack/Person"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGovTrack_Person_ResultSet *results = [[TMBGovTrack_Person_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Person Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGovTrack_Person_Inputs*)newInputSet {
		return [[TMBGovTrack_Person_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Role Choreo.
 */
@implementation TMBGovTrack_Role_Inputs

	/*!
	 * Set the value of the Current input for this Choreo.
	*(optional, string) Whether the role is currently held, or it is archival information. Filter operators allowed. Sortable.
	 */
	-(void)setCurrent:(NSString*)Current {
		[super setInput:@"Current" toValue:Current];
	}

	/*!
	 * Set the value of the District input for this Choreo.
	*(optional, string) For representatives, the number of their congressional district. 0 for at-large districts, -1 in historical data if the district is not known. Filter operators allowed. Sortable.
	 */
	-(void)setDistrict:(NSString*)District {
		[super setInput:@"District" toValue:District];
	}

	/*!
	 * Set the value of the EndDate input for this Choreo.
	*(optional, string) The date the role ended - when the person resigned, died, etc. (in YYYY-MM-DD format). Filter operators allowed. Sortable.
	 */
	-(void)setEndDate:(NSString*)EndDate {
		[super setInput:@"EndDate" toValue:EndDate];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) A comma separated list of fields to return in the response. Use double-underscores to span relationships (e.g. person__firstname).
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Limit input for this Choreo.
	*(optional, integer) Results are paged 100 per call by default. Set the limit input to a high value to get all of the results at once.
	 */
	-(void)setLimit:(NSString*)Limit {
		[super setInput:@"Limit" toValue:Limit];
	}

	/*!
	 * Set the value of the Offset input for this Choreo.
	*(optional, integer) Offset the results by the number given, useful for paging through results.
	 */
	-(void)setOffset:(NSString*)Offset {
		[super setInput:@"Offset" toValue:Offset];
	}

	/*!
	 * Set the value of the Party input for this Choreo.
	*(optional, string) The political party of the person. If the person changes party, it is usually the most recent party during this role. Filter operators allowed. Sortable.
	 */
	-(void)setParty:(NSString*)Party {
		[super setInput:@"Party" toValue:Party];
	}

	/*!
	 * Set the value of the Person input for this Choreo.
	*(optional, string) The person associated with this role. When using this filter, provide the id of the person which is returned when requesting a single role object.
	 */
	-(void)setPerson:(NSString*)Person {
		[super setInput:@"Person" toValue:Person];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: json (the default) and xml.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the RoleID input for this Choreo.
	*(optional, string) Specify the ID number of a role object to retrieve the record for only that role. When using this input, all other filter parameters are ignored, and a single record is returned.
	 */
	-(void)setRoleID:(NSString*)RoleID {
		[super setInput:@"RoleID" toValue:RoleID];
	}

	/*!
	 * Set the value of the RoleType input for this Choreo.
	*(optional, string) The type of role (e.g. senator, representative, or president). Filter operators allowed. Sortable.
	 */
	-(void)setRoleType:(NSString*)RoleType {
		[super setInput:@"RoleType" toValue:RoleType];
	}

	/*!
	 * Set the value of the SenatorClass input for this Choreo.
	*(optional, integer) For senators, their election class, which determines which years they are up for election. Acceptable values: class1, class2, class3. Filter operators allowed. Sortable.
	 */
	-(void)setSenatorClass:(NSString*)SenatorClass {
		[super setInput:@"SenatorClass" toValue:SenatorClass];
	}

	/*!
	 * Set the value of the Sort input for this Choreo.
	*(optional, string) You can order the results by date using fieldname (ascending) or -fieldname (descending), where "fieldname" is either startdate or enddate.
	 */
	-(void)setSort:(NSString*)Sort {
		[super setInput:@"Sort" toValue:Sort];
	}

	/*!
	 * Set the value of the StartDate input for this Choreo.
	*(optional, string) The date the role began  - when the person took office (in YYYY-MM-DD format). Filter operators allowed. Sortable.
	 */
	-(void)setStartDate:(NSString*)StartDate {
		[super setInput:@"StartDate" toValue:StartDate];
	}

	/*!
	 * Set the value of the State input for this Choreo.
	*(optional, string) For senators and representatives, the two-letter USPS abbreviation for the state or territory they are serving. Filter operators allowed. Sortable.
	 */
	-(void)setState:(NSString*)State {
		[super setInput:@"State" toValue:State];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Role Choreo.
 */
@implementation TMBGovTrack_Role_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from GovTrack.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Returns terms held in office by Members of Congress and U.S. Presidents.
 */
@implementation TMBGovTrack_Role

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGovTrack_Role Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/GovTrack/Role"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGovTrack_Role_ResultSet *results = [[TMBGovTrack_Role_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Role Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGovTrack_Role_Inputs*)newInputSet {
		return [[TMBGovTrack_Role_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Vote Choreo.
 */
@implementation TMBGovTrack_Vote_Inputs

	/*!
	 * Set the value of the Chamber input for this Choreo.
	*(optional, string) The chamber in which the vote was held. Valid values are: house or senate. Filter operators allowed but only when filtering by Congress as well. Sortable.
	 */
	-(void)setChamber:(NSString*)Chamber {
		[super setInput:@"Chamber" toValue:Chamber];
	}

	/*!
	 * Set the value of the Congress input for this Choreo.
	*(optional, string) The number of the congress in which the vote took place. The current congress is 113. Filter operators allowed. Sortable.
	 */
	-(void)setCongress:(NSString*)Congress {
		[super setInput:@"Congress" toValue:Congress];
	}

	/*!
	 * Set the value of the Created input for this Choreo.
	*(optional, string) The date (and in recent history also the time) on which the vote was held. Filter operators allowed. Sortable.
	 */
	-(void)setCreated:(NSString*)Created {
		[super setInput:@"Created" toValue:Created];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) A comma separated list of fields to return in the response. Use double-underscores to span relationships (e.g. person__firstname).
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Limit input for this Choreo.
	*(optional, integer) Results are paged 100 per call by default. Set the limit input to a high value to get all of the results at once.
	 */
	-(void)setLimit:(NSString*)Limit {
		[super setInput:@"Limit" toValue:Limit];
	}

	/*!
	 * Set the value of the Number input for this Choreo.
	*(optional, string) The number of the vote, unique to a Congress and session pair. Filter operators allowed. Sortable.
	 */
	-(void)setNumber:(NSString*)Number {
		[super setInput:@"Number" toValue:Number];
	}

	/*!
	 * Set the value of the Offset input for this Choreo.
	*(optional, integer) Offset the results by the number given, useful for paging through results.
	 */
	-(void)setOffset:(NSString*)Offset {
		[super setInput:@"Offset" toValue:Offset];
	}

	/*!
	 * Set the value of the RelatedAmendment input for this Choreo.
	*(optional, string) The ID of a related amendment. Filter operators allowed. Sortable.
	 */
	-(void)setRelatedAmendment:(NSString*)RelatedAmendment {
		[super setInput:@"RelatedAmendment" toValue:RelatedAmendment];
	}

	/*!
	 * Set the value of the RelatedBill input for this Choreo.
	*(optional, string) A bill related to this vote. Filter operators allowed. Sortable.
	 */
	-(void)setRelatedBill:(NSString*)RelatedBill {
		[super setInput:@"RelatedBill" toValue:RelatedBill];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: json (the default) and xml.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Session input for this Choreo.
	*(optional, string) The session of congress. Filter operators allowed. Sortable.
	 */
	-(void)setSession:(NSString*)Session {
		[super setInput:@"Session" toValue:Session];
	}

	/*!
	 * Set the value of the Sort input for this Choreo.
	*(optional, string) You can order the results using created (ascending) or -created (descending) for the dates that each vote was held.
	 */
	-(void)setSort:(NSString*)Sort {
		[super setInput:@"Sort" toValue:Sort];
	}

	/*!
	 * Set the value of the VoteID input for this Choreo.
	*(optional, integer) The ID of a vote object to retrieve. When using this input, all other filter parameters are ignored, and a single record is returned.
	 */
	-(void)setVoteID:(NSString*)VoteID {
		[super setInput:@"VoteID" toValue:VoteID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Vote Choreo.
 */
@implementation TMBGovTrack_Vote_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from GovTrack.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Returns roll call votes in the U.S. Congress since 1789.
 */
@implementation TMBGovTrack_Vote

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGovTrack_Vote Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/GovTrack/Vote"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGovTrack_Vote_ResultSet *results = [[TMBGovTrack_Vote_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Vote Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGovTrack_Vote_Inputs*)newInputSet {
		return [[TMBGovTrack_Vote_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the VoteAndVoter Choreo.
 */
@implementation TMBGovTrack_VoteAndVoter_Inputs

	/*!
	 * Set the value of the Created input for this Choreo.
	*(optional, string) The date (and in recent history also the time) on which the vote was held (in YYYY-MM-DD format). Filter operators allowed. Sortable.
	 */
	-(void)setCreated:(NSString*)Created {
		[super setInput:@"Created" toValue:Created];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) A comma separated list of fields to return in the response. Use double-underscores to span relationships (e.g. person__firstname).
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the Limit input for this Choreo.
	*(optional, integer) Results are paged 100 per call by default. Set the limit input to a high value to get all of the results at once.
	 */
	-(void)setLimit:(NSString*)Limit {
		[super setInput:@"Limit" toValue:Limit];
	}

	/*!
	 * Set the value of the ObjectID input for this Choreo.
	*(optional, integer) The ID of the resource to retrieve. When using this input, all other filter parameters are ignored, and a single record is returned.
	 */
	-(void)setObjectID:(NSString*)ObjectID {
		[super setInput:@"ObjectID" toValue:ObjectID];
	}

	/*!
	 * Set the value of the Offset input for this Choreo.
	*(optional, integer) Offset the results by the number given, useful for paging through results.
	 */
	-(void)setOffset:(NSString*)Offset {
		[super setInput:@"Offset" toValue:Offset];
	}

	/*!
	 * Set the value of the Option input for this Choreo.
	*(optional, string) The way a particular person voted. The value corresponds to the key of an option returned on the Vote Choreo. Filter operators allowed. Sortable.
	 */
	-(void)setOption:(NSString*)Option {
		[super setInput:@"Option" toValue:Option];
	}

	/*!
	 * Set the value of the Person input for this Choreo.
	*(optional, string) The person making this vote. This is an ID number. Filter operators allowed. Sortable.
	 */
	-(void)setPerson:(NSString*)Person {
		[super setInput:@"Person" toValue:Person];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: json (the default) and xml.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Sort input for this Choreo.
	*(optional, string) You can order the results by date using fieldname (ascending) or -fieldname (descending), where "fieldname" is either startdate or enddate.
	 */
	-(void)setSort:(NSString*)Sort {
		[super setInput:@"Sort" toValue:Sort];
	}

	/*!
	 * Set the value of the Vote input for this Choreo.
	*(optional, string) The ID number of the vote that this was part of. This is in the form of an ID number. Filter operators allowed. Sortable.
	 */
	-(void)setVote:(NSString*)Vote {
		[super setInput:@"Vote" toValue:Vote];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the VoteAndVoter Choreo.
 */
@implementation TMBGovTrack_VoteAndVoter_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from GovTrack.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves how people voted on roll call votes in the U.S. Congress since 1789. 
 */
@implementation TMBGovTrack_VoteAndVoter

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBGovTrack_VoteAndVoter Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/GovTrack/VoteAndVoter"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBGovTrack_VoteAndVoter_ResultSet *results = [[TMBGovTrack_VoteAndVoter_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the VoteAndVoter Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBGovTrack_VoteAndVoter_Inputs*)newInputSet {
		return [[TMBGovTrack_VoteAndVoter_Inputs alloc] init];
	}
@end
	