/*!
 * @TMBSalesforce.m
 *
 * Execute Choreographies from the Temboo Salesforce bundle.
 *
 * iOS version 10.6.8
 *
 * LICENSE: Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @category   Services
 * @author     Reid Simonton, Temboo, Inc.
 * @copyright  2013 Temboo, Inc.
 * @license    http://www.apache.org/licenses/LICENSE-2.0 Apache License 2.0
 * @version    1.7
 */

#import "TMBSalesforce.h"
#import "TMBChoreography_Protected.h"
#import "TMBChoreographyExecution.h"
#import "TMBTembooSession.h"
#import "TMBTembooUri.h"


/*!
 * Input object with appropriate setters for specifying arguments to the FinalizeOAuth Choreo.
 */
@implementation TMBSalesforce_OAuth_FinalizeOAuth_Inputs

	/*!
	 * Set the value of the CallbackID input for this Choreo.
	*(required, string) The callback token returned by the InitializeOAuth Choreo. Used to retrieve the authorization code after the user authorizes.
	 */
	-(void)setCallbackID:(NSString*)CallbackID {
		[super setInput:@"CallbackID" toValue:CallbackID];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(required, string) The Client ID provided by Salesforce.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(required, string) The Client Secret provided by Salesforce.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the SuppressErrors input for this Choreo.
	*(optional, boolean) When set to true, errors received during the OAuth redirect process will be suppressed and returned in the ErrorMessage output.
	 */
	-(void)setSuppressErrors:(NSString*)SuppressErrors {
		[super setInput:@"SuppressErrors" toValue:SuppressErrors];
	}

	/*!
	 * Set the value of the Timeout input for this Choreo.
	*(optional, integer) The amount of time (in seconds) to poll your Temboo callback URL to see if your app's user has allowed or denied the request for access. Defaults to 20. Max is 60.
	 */
	-(void)setTimeout:(NSString*)Timeout {
		[super setInput:@"Timeout" toValue:Timeout];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the FinalizeOAuth Choreo.
 */
@implementation TMBSalesforce_OAuth_FinalizeOAuth_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "AccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) The access token for the user that has granted access to your application.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getAccessToken {
		return [super getOutputByName:@"AccessToken"];
	}

	/*!
	 * Retrieve the value of the "ErrorMessage" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains an error message if an error occurs during the OAuth redirect process and if SuppressErrors is set to true.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getErrorMessage {
		return [super getOutputByName:@"ErrorMessage"];
	}

	/*!
	 * Retrieve the value of the "RefreshToken" output from an execution of this Choreo.
	 * @return - NSString* (string) A token that may be used to obtain a new access token when the short-lived access token expires.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getRefreshToken {
		return [super getOutputByName:@"RefreshToken"];
	}
	
@end

/*!
 * Completes the OAuth process by retrieving a Salesforce access token for a user, after they have visited the authorization URL returned by the InitializeOAuth choreo and clicked "allow."
 */
@implementation TMBSalesforce_OAuth_FinalizeOAuth

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBSalesforce_OAuth_FinalizeOAuth Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Salesforce/OAuth/FinalizeOAuth"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBSalesforce_OAuth_FinalizeOAuth_ResultSet *results = [[TMBSalesforce_OAuth_FinalizeOAuth_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the FinalizeOAuth Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBSalesforce_OAuth_FinalizeOAuth_Inputs*)newInputSet {
		return [[TMBSalesforce_OAuth_FinalizeOAuth_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the InitializeOAuth Choreo.
 */
@implementation TMBSalesforce_OAuth_InitializeOAuth_Inputs

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(required, string) The Client ID provided by Salesforce.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the CustomCallbackID input for this Choreo.
	*(optional, string) A unique identifier that you can pass to eliminate the need to wait for a Temboo generated CallbackID. Callback identifiers may only contain numbers, letters, periods, and hyphens.
	 */
	-(void)setCustomCallbackID:(NSString*)CustomCallbackID {
		[super setInput:@"CustomCallbackID" toValue:CustomCallbackID];
	}

	/*!
	 * Set the value of the Display input for this Choreo.
	*(optional, string) Tailors the login page to the user's device type. Currently the only values supported are:  page, popup and touch.
	 */
	-(void)setDisplay:(NSString*)Display {
		[super setInput:@"Display" toValue:Display];
	}

	/*!
	 * Set the value of the ForwardingURL input for this Choreo.
	*(optional, string) The URL that Temboo will redirect your users to after they grant access to your application. This should include the "https://" or "http://" prefix and be a fully qualified URL.
	 */
	-(void)setForwardingURL:(NSString*)ForwardingURL {
		[super setInput:@"ForwardingURL" toValue:ForwardingURL];
	}

	/*!
	 * Set the value of the Immediate input for this Choreo.
	*(optional, string) Avoid interacting with the user.  false - Prompt the user for login and approval (default),  true - If the user is currently logged in and has previously given approval, the approval step is skipped.
	 */
	-(void)setImmediate:(NSString*)Immediate {
		[super setInput:@"Immediate" toValue:Immediate];
	}

	/*!
	 * Set the value of the Scope input for this Choreo.
	*(conditional, string) A space separated list of scope values. Supported values are: api, chatter_api, full, id, refresh_token, visualforce, web. Defaults to: id, api, refresh_token.
	 */
	-(void)setScope:(NSString*)Scope {
		[super setInput:@"Scope" toValue:Scope];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the InitializeOAuth Choreo.
 */
@implementation TMBSalesforce_OAuth_InitializeOAuth_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "AuthorizationURL" output from an execution of this Choreo.
	 * @return - NSString* (string) The authorization URL that the application's user needs to go to in order to grant access to your application.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getAuthorizationURL {
		return [super getOutputByName:@"AuthorizationURL"];
	}

	/*!
	 * Retrieve the value of the "CallbackID" output from an execution of this Choreo.
	 * @return - NSString* (string) An ID used to retrieve the callback data that Temboo stores once your application's user authorizes.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getCallbackID {
		return [super getOutputByName:@"CallbackID"];
	}
	
@end

/*!
 * Generates an authorization URL that an application can use to complete the first step in the OAuth process.
 */
@implementation TMBSalesforce_OAuth_InitializeOAuth

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBSalesforce_OAuth_InitializeOAuth Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Salesforce/OAuth/InitializeOAuth"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBSalesforce_OAuth_InitializeOAuth_ResultSet *results = [[TMBSalesforce_OAuth_InitializeOAuth_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the InitializeOAuth Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBSalesforce_OAuth_InitializeOAuth_Inputs*)newInputSet {
		return [[TMBSalesforce_OAuth_InitializeOAuth_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CreateRecord Choreo.
 */
@implementation TMBSalesforce_Objects_CreateRecord_Inputs

	/*!
	 * Set the value of the SObject input for this Choreo.
	*(required, json) A JSON string containing the SObject properties you wish to set.
	 */
	-(void)setSObject:(NSString*)SObject {
		[super setInput:@"SObject" toValue:SObject];
	}

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the InstanceName input for this Choreo.
	*(required, string) The server url prefix that indicates which instance your Salesforce account is on (e.g. na1, na2, na3, etc).
	 */
	-(void)setInstanceName:(NSString*)InstanceName {
		[super setInput:@"InstanceName" toValue:InstanceName];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: json (the default) and xml.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the SObjectName input for this Choreo.
	*(required, string) The name of the Salesforce object type being created (e.g. Account, Contact, Lead, etc).
	 */
	-(void)setSObjectName:(NSString*)SObjectName {
		[super setInput:@"SObjectName" toValue:SObjectName];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateRecord Choreo.
 */
@implementation TMBSalesforce_Objects_CreateRecord_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "ID" output from an execution of this Choreo.
	 * @return - NSString* (string) The id of the object that was created.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getID {
		return [super getOutputByName:@"ID"];
	}

	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Salesforce.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Creates a new Salesforce Object such as an individual Account, Contact, or Lead record.
 */
@implementation TMBSalesforce_Objects_CreateRecord

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBSalesforce_Objects_CreateRecord Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Salesforce/Objects/CreateRecord"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBSalesforce_Objects_CreateRecord_ResultSet *results = [[TMBSalesforce_Objects_CreateRecord_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CreateRecord Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBSalesforce_Objects_CreateRecord_Inputs*)newInputSet {
		return [[TMBSalesforce_Objects_CreateRecord_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteRecord Choreo.
 */
@implementation TMBSalesforce_Objects_DeleteRecord_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, string) The id of the object to be deleted.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the InstanceName input for this Choreo.
	*(required, string) The server url prefix that indicates which instance your Salesforce account is on (e.g. na1, na2, na3, etc).
	 */
	-(void)setInstanceName:(NSString*)InstanceName {
		[super setInput:@"InstanceName" toValue:InstanceName];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the SObjectName input for this Choreo.
	*(conditional, string) The name of the Salesforce object type being deleted (e.g. Account, Contact, Lead, etc).
	 */
	-(void)setSObjectName:(NSString*)SObjectName {
		[super setInput:@"SObjectName" toValue:SObjectName];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteRecord Choreo.
 */
@implementation TMBSalesforce_Objects_DeleteRecord_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "ResponseStatusCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) The response status code returned from Salesforce. A 204 is expected for a successful request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponseStatusCode {
		return [super getOutputByName:@"ResponseStatusCode"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Salesforce.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Deletes a Salesforce Object such as an individual Account, Contact, or Lead record.
 */
@implementation TMBSalesforce_Objects_DeleteRecord

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBSalesforce_Objects_DeleteRecord Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Salesforce/Objects/DeleteRecord"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBSalesforce_Objects_DeleteRecord_ResultSet *results = [[TMBSalesforce_Objects_DeleteRecord_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteRecord Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBSalesforce_Objects_DeleteRecord_Inputs*)newInputSet {
		return [[TMBSalesforce_Objects_DeleteRecord_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteRecordFields Choreo.
 */
@implementation TMBSalesforce_Objects_DeleteRecordFields_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(required, string) Comma-separated list of up to 10 field names.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, string) The id of the object that that is being modified.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the InstanceName input for this Choreo.
	*(required, string) The server url prefix that indicates which instance your Salesforce account is on (e.g. na1, na2, na3, etc).
	 */
	-(void)setInstanceName:(NSString*)InstanceName {
		[super setInput:@"InstanceName" toValue:InstanceName];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the SObjectName input for this Choreo.
	*(conditional, string) The name of the Salesforce object type being created (e.g. Account, Contact, Lead, etc).
	 */
	-(void)setSObjectName:(NSString*)SObjectName {
		[super setInput:@"SObjectName" toValue:SObjectName];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteRecordFields Choreo.
 */
@implementation TMBSalesforce_Objects_DeleteRecordFields_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "ResponseStatusCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) The response status code returned from Salesforce. A 204 is expected for a successful request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponseStatusCode {
		return [super getOutputByName:@"ResponseStatusCode"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Salesforce.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Deletes specific field values from a specified record.
 */
@implementation TMBSalesforce_Objects_DeleteRecordFields

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBSalesforce_Objects_DeleteRecordFields Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Salesforce/Objects/DeleteRecordFields"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBSalesforce_Objects_DeleteRecordFields_ResultSet *results = [[TMBSalesforce_Objects_DeleteRecordFields_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteRecordFields Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBSalesforce_Objects_DeleteRecordFields_Inputs*)newInputSet {
		return [[TMBSalesforce_Objects_DeleteRecordFields_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DescribeObjectMetadata Choreo.
 */
@implementation TMBSalesforce_Objects_DescribeObjectMetadata_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the InstanceName input for this Choreo.
	*(required, string) The server url prefix that indicates which instance your Salesforce account is on (e.g. na1, na2, na3, etc).
	 */
	-(void)setInstanceName:(NSString*)InstanceName {
		[super setInput:@"InstanceName" toValue:InstanceName];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: json (the default) and xml.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the SObjectName input for this Choreo.
	*(required, string) The name of the SObject to retrieve (e.g. Contact, Lead, Account, etc).
	 */
	-(void)setSObjectName:(NSString*)SObjectName {
		[super setInput:@"SObjectName" toValue:SObjectName];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DescribeObjectMetadata Choreo.
 */
@implementation TMBSalesforce_Objects_DescribeObjectMetadata_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Salesforce.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Describes the individual metadata at all levels for the specified object.
 */
@implementation TMBSalesforce_Objects_DescribeObjectMetadata

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBSalesforce_Objects_DescribeObjectMetadata Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Salesforce/Objects/DescribeObjectMetadata"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBSalesforce_Objects_DescribeObjectMetadata_ResultSet *results = [[TMBSalesforce_Objects_DescribeObjectMetadata_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DescribeObjectMetadata Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBSalesforce_Objects_DescribeObjectMetadata_Inputs*)newInputSet {
		return [[TMBSalesforce_Objects_DescribeObjectMetadata_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetObjectMetadata Choreo.
 */
@implementation TMBSalesforce_Objects_GetObjectMetadata_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the InstanceName input for this Choreo.
	*(required, string) The server url prefix that indicates which instance your Salesforce account is on (e.g. na1, na2, na3, etc).
	 */
	-(void)setInstanceName:(NSString*)InstanceName {
		[super setInput:@"InstanceName" toValue:InstanceName];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: json (the default) and xml.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the SObjectName input for this Choreo.
	*(required, string) The name of the SObject to retrieve (e.g. Contact, Lead, Account, etc).
	 */
	-(void)setSObjectName:(NSString*)SObjectName {
		[super setInput:@"SObjectName" toValue:SObjectName];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetObjectMetadata Choreo.
 */
@implementation TMBSalesforce_Objects_GetObjectMetadata_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Salesforce.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the individual metadata for the specified object type such as a Contact, Lead, or Account object.
 */
@implementation TMBSalesforce_Objects_GetObjectMetadata

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBSalesforce_Objects_GetObjectMetadata Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Salesforce/Objects/GetObjectMetadata"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBSalesforce_Objects_GetObjectMetadata_ResultSet *results = [[TMBSalesforce_Objects_GetObjectMetadata_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetObjectMetadata Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBSalesforce_Objects_GetObjectMetadata_Inputs*)newInputSet {
		return [[TMBSalesforce_Objects_GetObjectMetadata_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetRecord Choreo.
 */
@implementation TMBSalesforce_Objects_GetRecord_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the Fields input for this Choreo.
	*(optional, string) Comma-separated list of up to 10 field names.
	 */
	-(void)setFields:(NSString*)Fields {
		[super setInput:@"Fields" toValue:Fields];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, string) The id of the object that you want to retrieve.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the InstanceName input for this Choreo.
	*(required, string) The server url prefix that indicates which instance your Salesforce account is on (e.g. na1, na2, na3, etc).
	 */
	-(void)setInstanceName:(NSString*)InstanceName {
		[super setInput:@"InstanceName" toValue:InstanceName];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: json (the default) and xml.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the SObjectName input for this Choreo.
	*(conditional, string) The name of the Salesforce object type being retrieved (e.g. Account, Contact, Lead, etc).
	 */
	-(void)setSObjectName:(NSString*)SObjectName {
		[super setInput:@"SObjectName" toValue:SObjectName];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetRecord Choreo.
 */
@implementation TMBSalesforce_Objects_GetRecord_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Salesforce.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a Salesforce Object such as an individual Account, Contact, or Lead record.
 */
@implementation TMBSalesforce_Objects_GetRecord

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBSalesforce_Objects_GetRecord Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Salesforce/Objects/GetRecord"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBSalesforce_Objects_GetRecord_ResultSet *results = [[TMBSalesforce_Objects_GetRecord_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetRecord Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBSalesforce_Objects_GetRecord_Inputs*)newInputSet {
		return [[TMBSalesforce_Objects_GetRecord_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetRecordByExternalID Choreo.
 */
@implementation TMBSalesforce_Objects_GetRecordByExternalID_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the FieldName input for this Choreo.
	*(conditional, string) FieldName for external ID
	 */
	-(void)setFieldName:(NSString*)FieldName {
		[super setInput:@"FieldName" toValue:FieldName];
	}

	/*!
	 * Set the value of the FieldValue input for this Choreo.
	*(conditional, string) Field value for external ID
	 */
	-(void)setFieldValue:(NSString*)FieldValue {
		[super setInput:@"FieldValue" toValue:FieldValue];
	}

	/*!
	 * Set the value of the InstanceName input for this Choreo.
	*(required, string) The server url prefix that indicates which instance your Salesforce account is on (e.g. na1, na2, na3, etc).
	 */
	-(void)setInstanceName:(NSString*)InstanceName {
		[super setInput:@"InstanceName" toValue:InstanceName];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: json (the default) and xml.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the SObjectName input for this Choreo.
	*(conditional, string) The name of the Salesforce object type being created (e.g. Account, Contact, Lead, etc).
	 */
	-(void)setSObjectName:(NSString*)SObjectName {
		[super setInput:@"SObjectName" toValue:SObjectName];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetRecordByExternalID Choreo.
 */
@implementation TMBSalesforce_Objects_GetRecordByExternalID_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Salesforce.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieve records with a specific external ID.
 */
@implementation TMBSalesforce_Objects_GetRecordByExternalID

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBSalesforce_Objects_GetRecordByExternalID Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Salesforce/Objects/GetRecordByExternalID"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBSalesforce_Objects_GetRecordByExternalID_ResultSet *results = [[TMBSalesforce_Objects_GetRecordByExternalID_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetRecordByExternalID Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBSalesforce_Objects_GetRecordByExternalID_Inputs*)newInputSet {
		return [[TMBSalesforce_Objects_GetRecordByExternalID_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListObjectTypes Choreo.
 */
@implementation TMBSalesforce_Objects_ListObjectTypes_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the InstanceName input for this Choreo.
	*(required, string) The server url prefix that indicates which instance your Salesforce account is on (e.g. na1, na2, na3, etc).
	 */
	-(void)setInstanceName:(NSString*)InstanceName {
		[super setInput:@"InstanceName" toValue:InstanceName];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: json (the default) and xml.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListObjectTypes Choreo.
 */
@implementation TMBSalesforce_Objects_ListObjectTypes_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Salesforce.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Lists the available objects and their metadata for your organization's data.
 */
@implementation TMBSalesforce_Objects_ListObjectTypes

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBSalesforce_Objects_ListObjectTypes Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Salesforce/Objects/ListObjectTypes"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBSalesforce_Objects_ListObjectTypes_ResultSet *results = [[TMBSalesforce_Objects_ListObjectTypes_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListObjectTypes Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBSalesforce_Objects_ListObjectTypes_Inputs*)newInputSet {
		return [[TMBSalesforce_Objects_ListObjectTypes_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ChangePassword Choreo.
 */
@implementation TMBSalesforce_Passwords_ChangePassword_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, string) The ID of the user whose password you want to change.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the InstanceName input for this Choreo.
	*(required, string) The server url prefix that indicates which instance your Salesforce account is on (e.g. na1, na2, na3, etc).
	 */
	-(void)setInstanceName:(NSString*)InstanceName {
		[super setInput:@"InstanceName" toValue:InstanceName];
	}

	/*!
	 * Set the value of the NewPassword input for this Choreo.
	*(required, string) The new password.
	 */
	-(void)setNewPassword:(NSString*)NewPassword {
		[super setInput:@"NewPassword" toValue:NewPassword];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ChangePassword Choreo.
 */
@implementation TMBSalesforce_Passwords_ChangePassword_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "ResponseStatusCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) The response status code returned from Salesforce. A 204 is expected for a successful request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponseStatusCode {
		return [super getOutputByName:@"ResponseStatusCode"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Salesforce.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Changes a user's password.
 */
@implementation TMBSalesforce_Passwords_ChangePassword

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBSalesforce_Passwords_ChangePassword Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Salesforce/Passwords/ChangePassword"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBSalesforce_Passwords_ChangePassword_ResultSet *results = [[TMBSalesforce_Passwords_ChangePassword_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ChangePassword Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBSalesforce_Passwords_ChangePassword_Inputs*)newInputSet {
		return [[TMBSalesforce_Passwords_ChangePassword_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetPasswordInfo Choreo.
 */
@implementation TMBSalesforce_Passwords_GetPasswordInfo_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, string) The ID of the user you're getting info for.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the InstanceName input for this Choreo.
	*(required, string) The server url prefix that indicates which instance your Salesforce account is on (e.g. na1, na2, na3, etc).
	 */
	-(void)setInstanceName:(NSString*)InstanceName {
		[super setInput:@"InstanceName" toValue:InstanceName];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: json (the default) and xml.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetPasswordInfo Choreo.
 */
@implementation TMBSalesforce_Passwords_GetPasswordInfo_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Salesforce.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Gets information on a user's password.
 */
@implementation TMBSalesforce_Passwords_GetPasswordInfo

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBSalesforce_Passwords_GetPasswordInfo Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Salesforce/Passwords/GetPasswordInfo"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBSalesforce_Passwords_GetPasswordInfo_ResultSet *results = [[TMBSalesforce_Passwords_GetPasswordInfo_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetPasswordInfo Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBSalesforce_Passwords_GetPasswordInfo_Inputs*)newInputSet {
		return [[TMBSalesforce_Passwords_GetPasswordInfo_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ResetPassword Choreo.
 */
@implementation TMBSalesforce_Passwords_ResetPassword_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the ID input for this Choreo.
	*(required, string) The ID of the user whos password you are resetting.
	 */
	-(void)setID:(NSString*)ID {
		[super setInput:@"ID" toValue:ID];
	}

	/*!
	 * Set the value of the InstanceName input for this Choreo.
	*(required, string) The server url prefix that indicates which instance your Salesforce account is on (e.g. na1, na2, na3, etc).
	 */
	-(void)setInstanceName:(NSString*)InstanceName {
		[super setInput:@"InstanceName" toValue:InstanceName];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: json (the default) and xml.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ResetPassword Choreo.
 */
@implementation TMBSalesforce_Passwords_ResetPassword_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "NewPassword" output from an execution of this Choreo.
	 * @return - NSString* (string) New password returned from Salesforce.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewPassword {
		return [super getOutputByName:@"NewPassword"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Salesforce.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Resets a user's password to new randomized password.
 */
@implementation TMBSalesforce_Passwords_ResetPassword

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBSalesforce_Passwords_ResetPassword Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Salesforce/Passwords/ResetPassword"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBSalesforce_Passwords_ResetPassword_ResultSet *results = [[TMBSalesforce_Passwords_ResetPassword_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ResetPassword Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBSalesforce_Passwords_ResetPassword_Inputs*)newInputSet {
		return [[TMBSalesforce_Passwords_ResetPassword_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetNextResults Choreo.
 */
@implementation TMBSalesforce_Queries_GetNextResults_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the InstanceName input for this Choreo.
	*(required, string) The server url prefix that indicates which instance your Salesforce account is on (e.g. na1, na2, na3, etc).
	 */
	-(void)setInstanceName:(NSString*)InstanceName {
		[super setInput:@"InstanceName" toValue:InstanceName];
	}

	/*!
	 * Set the value of the QueryLocator input for this Choreo.
	*(required, string) The query locator ID to use for retrieving the next batch of records. This ID is returned by the Query Choreo.
	 */
	-(void)setQueryLocator:(NSString*)QueryLocator {
		[super setInput:@"QueryLocator" toValue:QueryLocator];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: json (the default) and xml.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetNextResults Choreo.
 */
@implementation TMBSalesforce_Queries_GetNextResults_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "NextQueryLocator" output from an execution of this Choreo.
	 * @return - NSString* (string) The next query locator to use when there are more results to retrieve.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextQueryLocator {
		return [super getOutputByName:@"NextQueryLocator"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Salesforce.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the remaining query results using a query locator ID.
 */
@implementation TMBSalesforce_Queries_GetNextResults

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBSalesforce_Queries_GetNextResults Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Salesforce/Queries/GetNextResults"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBSalesforce_Queries_GetNextResults_ResultSet *results = [[TMBSalesforce_Queries_GetNextResults_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetNextResults Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBSalesforce_Queries_GetNextResults_Inputs*)newInputSet {
		return [[TMBSalesforce_Queries_GetNextResults_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Query Choreo.
 */
@implementation TMBSalesforce_Queries_Query_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the InstanceName input for this Choreo.
	*(required, string) The server url prefix that indicates which instance your Salesforce account is on (e.g. na1, na2, na3, etc).
	 */
	-(void)setInstanceName:(NSString*)InstanceName {
		[super setInput:@"InstanceName" toValue:InstanceName];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: json (the default) and xml.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the SOQL input for this Choreo.
	*(required, string) The SOQL statement used to retrieve the set of records matching a specific criteria (e.g. select FirstName, LastName from Lead limit 1).
	 */
	-(void)setSOQL:(NSString*)SOQL {
		[super setInput:@"SOQL" toValue:SOQL];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Query Choreo.
 */
@implementation TMBSalesforce_Queries_Query_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "QueryLocator" output from an execution of this Choreo.
	 * @return - NSString* (string) The query locator ID that can be used to retrieve the next batch of records. This will only be present when more than 2000 records meet the criteria of a query causing the response to be truncated.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getQueryLocator {
		return [super getOutputByName:@"QueryLocator"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Salesforce.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Executes the specified SOQL query.
 */
@implementation TMBSalesforce_Queries_Query

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBSalesforce_Queries_Query Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Salesforce/Queries/Query"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBSalesforce_Queries_Query_ResultSet *results = [[TMBSalesforce_Queries_Query_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Query Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBSalesforce_Queries_Query_Inputs*)newInputSet {
		return [[TMBSalesforce_Queries_Query_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetResources Choreo.
 */
@implementation TMBSalesforce_Resources_GetResources_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the InstanceName input for this Choreo.
	*(required, string) The server url prefix that indicates which instance your Salesforce account is on (e.g. na1, na2, na3, etc).
	 */
	-(void)setInstanceName:(NSString*)InstanceName {
		[super setInput:@"InstanceName" toValue:InstanceName];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth Refresh Token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: json (the default) and xml.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Version input for this Choreo.
	*(optional, string) The API version number to retrieve resources for. Defaults to "v28.0".
	 */
	-(void)setVersion:(NSString*)Version {
		[super setInput:@"Version" toValue:Version];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetResources Choreo.
 */
@implementation TMBSalesforce_Resources_GetResources_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Salesforce.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * List available resources for a specified API version.
 */
@implementation TMBSalesforce_Resources_GetResources

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBSalesforce_Resources_GetResources Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Salesforce/Resources/GetResources"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBSalesforce_Resources_GetResources_ResultSet *results = [[TMBSalesforce_Resources_GetResources_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetResources Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBSalesforce_Resources_GetResources_Inputs*)newInputSet {
		return [[TMBSalesforce_Resources_GetResources_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Search Choreo.
 */
@implementation TMBSalesforce_Searching_Search_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the InstanceName input for this Choreo.
	*(required, string) The server url prefix that indicates which instance your Salesforce account is on (e.g. na1, na2, na3, etc).
	 */
	-(void)setInstanceName:(NSString*)InstanceName {
		[super setInput:@"InstanceName" toValue:InstanceName];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: json (the default) and xml.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the SOSL input for this Choreo.
	*(required, string) The SOSL statement used to retrieve the set of records matching a specific criteria (e.g. FIND {joe})
	 */
	-(void)setSOSL:(NSString*)SOSL {
		[super setInput:@"SOSL" toValue:SOSL];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Search Choreo.
 */
@implementation TMBSalesforce_Searching_Search_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Salesforce.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Executes the specified SOSL search.
 */
@implementation TMBSalesforce_Searching_Search

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBSalesforce_Searching_Search Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Salesforce/Searching/Search"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBSalesforce_Searching_Search_ResultSet *results = [[TMBSalesforce_Searching_Search_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Search Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBSalesforce_Searching_Search_Inputs*)newInputSet {
		return [[TMBSalesforce_Searching_Search_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SearchScopeAndOrder Choreo.
 */
@implementation TMBSalesforce_Searching_SearchScopeAndOrder_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(optional, string) A valid access token retrieved during the OAuth process. This is required unless you provide the ClientID, ClientSecret, and RefreshToken to generate a new access token.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(conditional, string) The Client ID provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(conditional, string) The Client Secret provided by Salesforce. Required unless providing a valid AccessToken.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the InstanceName input for this Choreo.
	*(required, string) The server url prefix that indicates which instance your Salesforce account is on (e.g. na1, na2, na3, etc).
	 */
	-(void)setInstanceName:(NSString*)InstanceName {
		[super setInput:@"InstanceName" toValue:InstanceName];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(conditional, string) An OAuth refresh token used to generate a new access token when the original token is expired. Required unless providing a valid AccessToken.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: json (the default) and xml.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SearchScopeAndOrder Choreo.
 */
@implementation TMBSalesforce_Searching_SearchScopeAndOrder_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "NewAccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a new AccessToken when the RefreshToken is provided.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNewAccessToken {
		return [super getOutputByName:@"NewAccessToken"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Salesforce.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Executes the specified SOSL search in the default global search scope of a logged-in user.
 */
@implementation TMBSalesforce_Searching_SearchScopeAndOrder

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBSalesforce_Searching_SearchScopeAndOrder Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Salesforce/Searching/SearchScopeAndOrder"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBSalesforce_Searching_SearchScopeAndOrder_ResultSet *results = [[TMBSalesforce_Searching_SearchScopeAndOrder_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SearchScopeAndOrder Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBSalesforce_Searching_SearchScopeAndOrder_Inputs*)newInputSet {
		return [[TMBSalesforce_Searching_SearchScopeAndOrder_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListVersions Choreo.
 */
@implementation TMBSalesforce_Versions_ListVersions_Inputs

	/*!
	 * Set the value of the InstanceName input for this Choreo.
	*(required, string) The server url prefix that indicates which instance your Salesforce account is on (e.g. na1, na2, na3, etc).
	 */
	-(void)setInstanceName:(NSString*)InstanceName {
		[super setInput:@"InstanceName" toValue:InstanceName];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: json (the default) and xml.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListVersions Choreo.
 */
@implementation TMBSalesforce_Versions_ListVersions_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Salesforce.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Lists summary information about each Salesforce version currently available, including the version, label, and a link to each version's root.
 */
@implementation TMBSalesforce_Versions_ListVersions

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBSalesforce_Versions_ListVersions Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Salesforce/Versions/ListVersions"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBSalesforce_Versions_ListVersions_ResultSet *results = [[TMBSalesforce_Versions_ListVersions_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListVersions Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBSalesforce_Versions_ListVersions_Inputs*)newInputSet {
		return [[TMBSalesforce_Versions_ListVersions_Inputs alloc] init];
	}
@end
	