/*!
 * @file TMBChoreography.m
 *
 * A Choreography in the Temboo Library
 *
 * iOS version 10.6.8
 *
 * LICENSE: Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @category   Services
 * @author     Temboo, Inc.
 * @copyright  2013 Temboo, Inc.
 * @license    http://www.apache.org/licenses/LICENSE-2.0 Apache License 2.0
 * @version    1.7
 */
#import "TMBChoreography.h"
#import "TMBChoreographyExecution.h"
#import "TMBException.h"
#import "TMBTembooResource_Protected.h"
#import "TMBTembooSession.h"
#import "TMBTembooUri.h"

@interface TMBOutputFilter : NSObject
@property (nonatomic, readwrite, copy) NSString *name;
@property (nonatomic, readwrite, copy) NSString *path;
@property (nonatomic, readwrite, copy) NSString *source;
-(id)initWithName:(NSString*)name path:(NSString*)path andVariableSource:(NSString*)source;
@end

/*!
 * Encapsulates choreo output filters
 */

@implementation TMBOutputFilter

//
// Synthesize class properties
@synthesize name;
@synthesize path;
@synthesize source;

-(id)initWithName:(NSString*)n path:(NSString*)p andVariableSource:(NSString*)s {
    
    if(self = [super init]){
        self.name   = n;
        self.path   = p;
        self.source = s;
    }
    
    return self;
}

@end

/******************************************************************************/

/*!
 * Class extension - obscure "protected" members
 */
@interface TMBChoreographyInputSet()
	/*! The underlying collection of choreo inputs this class represents */
	@property (nonatomic, strong) NSMutableArray *inputs;
    /*! Collection of any specified Output Filters */
    @property (nonatomic, strong) NSMutableArray *outputFilters;

-(id)init;
-(id)initWithInputs:(NSDictionary*)inputs;
@end

/*!
 * Encapsulates choreo inputs and formats them for the Temboo REST API
 */
@implementation TMBChoreographyInputSet

//
// Synthesize class properties
//
@synthesize inputs;
@synthesize credential;
@synthesize outputFilters;

/*!
 * Initialize an empty TMBChoreographyInputSet
 *
 * @return An empty TMBChoreographyInputSet instance
 */
-(id)init {
	return [self initWithInputs:nil];
}

/*!
 * Initialize a TMBChoreographyInputSet with the provided inputs
 *
 * @param i
 *		A collection of key/value pairs to be used as choreography inputs
 *
 * @return An initialized TMBChoreographyInputSet instance containing the provided inputs
 */
-(id)initWithInputs:(NSDictionary*)i {
	
	if(self = [super init]){
		// Create our inputs array
		self.inputs = [[NSMutableArray alloc] init];
        // Create Output Filters array
        self.outputFilters = [[NSMutableArray alloc] init];
		
		if(i && [i count]){
			// Add values
			for(id key in i){
				[self setInput:key toValue:[i valueForKey:key]];
			}
		}
	}
	
	return self;
}

/*!
 * Add a named Output Filter, specifying a path and variable source
 *
 * @param name
 *      The name of the new Output Filter
 * @param path
 *      The path to desired data reletive to the source variable
 * @param source
 *      The variable containing the filtered output element(s)
 *
 * @return void
 */
-(void)addOutputFilter:(NSString*)name withPath:(NSString*)path andVariableSource:(NSString*)source {
    [self.outputFilters addObject:[[NSDictionary alloc] initWithObjectsAndKeys:
        name, @"name",
        path, @"path",
        source, @"variable", nil]];
}

/*!
 * Add a profile to be used alone with this input set
 *
 * @return void
 */
-(void)addProfileByName:(NSString *)name {
    self.credential = name;
}

/*!
 * @return A dictionary with a single key "inputs" mapped to this class's underlying array
 *		   of key/value pairs as required by the Temboo REST API
 */
-(NSMutableDictionary*)description {
	// Set inputs under top-level "inputs" key in dict
	NSMutableDictionary *i = [NSMutableDictionary dictionaryWithObject:self.inputs forKey:@"inputs"];
	
    // Set output filters
    [i setObject:self.outputFilters forKey:@"outputFilters"];
    
	if(self.credential){
		// User specified a credential to be used, include it outside the "inputs" dict
		[i setValue:self.credential forKey:@"preset"];
	}
	
	return i;
}

/*!
 * Sets an input key/value pair.
 *
 * Note that this method does not enforce the inputs be limited to a single value for a given
 * key - if the same key is added multiple times, those key/value pairs will *all* be included
 * with any future request utilizing this TMBChoreographyInputSet instance.
 *
 * @param name
 *		The input name to set
 * @param value
 *		The input's value
 *
 * @return void
 */
-(void)setInput:(NSString*)name toValue:(NSString*)value {
	NSDictionary* input = [[NSDictionary alloc] initWithObjectsAndKeys:name, @"name", value, @"value", nil];
	[self.inputs addObject:input];
}

@end

/******************************************************************************/

/*!
 * Class extension - obscure "protected" members
 */
@interface TMBChoreographyResultSet()
	/*! The time at which the choreography execution completed */	
	@property (nonatomic, readwrite, strong) NSDate *endTime;
	/*! The execution completion status */
	@property (nonatomic, readwrite, copy) NSString *status;
	/*! The choreography's execution ID as assigned by the Temboo server */
	@property (nonatomic, readwrite, copy) NSString *executionId;
	/*! The choreography's outputs */
	@property (nonatomic, readwrite, copy) NSDictionary *outputs;
	/*! The time at which the choreography execution started */	
	@property (nonatomic, readwrite, strong) NSDate *startTime;

-(id)initWithResponse:(NSDictionary*)document;
@end

/*!
 * Encapsulates choreography execution output
 */
@implementation TMBChoreographyResultSet

//
// Synthesize class properties
//
@synthesize endTime;
@synthesize executionId;
@synthesize outputs;
@synthesize status;
@synthesize startTime;

/*!
 * Initialize with a Temboo server response
 *
 * @param document
 *		The parsed result from the Temboo server
 *
 * @return An initialized TMBChoreographyResultSet instance
 */
-(id)initWithResponse:(NSDictionary*)document {
		
	if(self = [super init]){
		NSDictionary* execution = [document valueForKey:@"execution"];
		
		// Grab outputs from top-level document
		self.outputs		= [document valueForKey:@"output"];
		
		// Convert milliseconds since epoch to NSDate*
		self.endTime		= [NSDate dateWithTimeIntervalSince1970:[[execution valueForKey:@"endtime"] doubleValue]];
		self.startTime		= [NSDate dateWithTimeIntervalSince1970:[[execution valueForKey:@"starttime"] doubleValue]];
		
		self.executionId	= [execution valueForKey:@"id"];
		self.status			= [execution valueForKey:@"status"];
	}
		
	return self;	
}

/*!
 * Release retained resources
 */

/*!
 * Get the value of a given output
 *
 * @param name
 *		The name of the output to return the value for
 *
 * @return The value for input with the name provided
 *
 * @throws TMBNotFoundException if no output with the given name is found
 */
-(NSString*)getOutputByName:(NSString*)name {
	// Look up the value for the specified output
	NSObject *rawOutput = [self.outputs valueForKey:name];
    NSString *stringOutput;
    
	if(!rawOutput){
		// Output specified is invalid
		[TMBNotFoundException throwWithReason:[NSString stringWithFormat:@"Invalid output specified: '%@'", name]];
	}
    
    if([rawOutput isKindOfClass:[NSArray class]]){
        // output is an array, convert to JSON string
        NSError *jsonError = nil;
        NSData *jsonData = [NSJSONSerialization dataWithJSONObject:rawOutput options:0 error:&jsonError];
        
        if(jsonError)
            [TMBJSONException throwWithReason:@"Failed to serialize user inputs as JSON"];
        
        stringOutput = [[NSString alloc] initWithData:jsonData encoding:NSUTF8StringEncoding];
    }
    else{
        stringOutput = (NSString*)rawOutput;
    }
    
	//NSLog(@"%s - Returning output '%@' (%@): %@", __PRETTY_FUNCTION__, name, [output class], output);
	return stringOutput;
}

/*!
 * Get an array of values for a given output
 *
 * @param name
 *		The name of the output to return the value for
 *
 * @return An array of result values
 *
 * @throws TMBNotFoundException if no output with the given name is found
 */
-(NSArray*)getOutputArrayByName:(NSString*)name {
	// Look up the value for the specified output
	NSObject *rawOutput = [self.outputs valueForKey:name];
	NSArray *outputsArray;
    
	if(!rawOutput){
		// Output specified is invalid
		[TMBNotFoundException throwWithReason:[NSString stringWithFormat:@"Invalid output specified: '%@'", name]];
	}
	
    if([rawOutput isKindOfClass:[NSString class]]){
        outputsArray = [[NSArray alloc] initWithObjects:(NSString*)rawOutput, nil];
    }
    else{
        outputsArray = (NSArray*)rawOutput;
    }
    
	//NSLog(@"%s - Returning output '%@' (%@): %@", __PRETTY_FUNCTION__, name, [output class], output);
	return outputsArray;
}

/*!
 * Get the set of outputs for the choreo execution
 *
 * @return A NSDictionary of key/value pairs represetning the choreo's outputs
 */
-(NSDictionary*)getOutputs {
    return outputs;
}

@end

/******************************************************************************/

/*!
 * Class extension - obscure "protected" members
 */
@interface TMBChoreography()
@property (nonatomic, strong) NSMutableArray *outputFilters;
@property (nonatomic, strong) NSMutableData *responseData;
-(TMBChoreographyExecution*)executeAsyncWithInputs:(TMBChoreographyInputSet*)inputs;
-(TMBChoreographyExecution*)executeAsyncWithInputs:(TMBChoreographyInputSet*)inputs storingResult:(BOOL)store;
-(void)executeWithInputs:(TMBChoreographyInputSet*)inputs andParameters:(NSDictionary*)params delegate:(id<TMBChoreographyDelegate>)d;
// Implemented in superclass
-(id)initWithSession:(TMBTembooSession*)session andUri:(TMBTembooUri*)uri;
-(void)choreographyDidFinishExecuting:(id)result;
@end

/*!
 * A choreography in the Temboo Library
 */
@implementation TMBChoreography

/*! The base path for all choreos in the Library */
static NSString *const BASE_PATH = @"/choreos";

/*! Choreo execution status indicating a choreo error occurred */
static NSString* const STATUS_ERROR = @"ERROR";

@synthesize choreoDelegate;
@synthesize responseData;

/*!
 * Initialize with an existing session and uri
 *
 * @param session
 *		The session under which this choreography will be executed
 * @param uri
 *		A URI fragment corresponding to the method call in the REST API that accesses this choreography
 *
 * @return An initialized TMBChoreography instance
 */
-(id)initWithSession:(TMBTembooSession *)session andUri:(TMBTembooUri *)uri {
	TMBTembooUri *choreoUri = [uri prependPath:BASE_PATH];
	
	if(self = [super initWithSession:session andUri:choreoUri]){
		self.responseData = [NSMutableData data];
	}
	
	return self;
}

/*! NSURLConnectionDelegate Protocol implmementation */
- (void)connection:(NSURLConnection *)connection didReceiveData:(NSData *)data {
    // Append the data chunk to data already received
    [self.responseData appendData:data];
}

/*! NSURLConnectionDelegate Protocol implmementation */
- (void) connection:(NSURLConnection *)connection didFailWithError:(NSError *)error {
	// Notify
	[self.choreoDelegate choreographyDidFailWithError:error];
}

/*! NSURLConnectionDelegate Protocol implmementation */
- (void) connectionDidFinishLoading:(NSURLConnection *) connection {
	NSLog(@"%s - Success! Received %d bytes of data", __PRETTY_FUNCTION__, [responseData length]);
	
	// Allow subclass to wrap the response in the expected result type
	NSError *jsonError = nil;
	NSDictionary *response = [NSJSONSerialization JSONObjectWithData:self.responseData options:0 error:&jsonError];
	
    if(jsonError)
        [TMBJSONException throwWithReason:@"Failed to deserialize response JSON data"];
    
    // Grab the execution result details
    NSDictionary *execution = [response objectForKey:@"execution"];
	
    // Check the execution status
	if([[execution objectForKey:@"status"] isEqualToString:STATUS_ERROR]){
        // Choreo execution error occcurred
        NSError *error = [NSError errorWithDomain:@"TMBChoreographyError" code:0 userInfo:[execution objectForKey:@"lasterror"]];
		[self.choreoDelegate choreographyDidFailWithError:error];
    }
	else {
        NSLog(@"connectionDidFinishLoading: choreographyDidFinishExecuting");
		[self choreographyDidFinishExecuting:response];
    }
}


/*!
 * Asynchronous execution, storing results
 *
 * @param inputs
 *		The inputs to use when executing this choreography
 *
 * @return TMBChoreographyExecution
 *
 * @throws TMBHTTPException If an HTTP error occurs
 * @throws TMBException if an error occurs during the request
 */
-(TMBChoreographyExecution*)executeAsyncWithInputs:(TMBChoreographyInputSet*)inputs {
	return [self executeAsyncWithInputs:inputs storingResult:TRUE];
}

/*!
 * Asynchronous execution, optionally storing results
 *
 * @param inputs
 *		The inputs to use when executing this choreography
 * @param storingResults
 *		TRUE to request the server store the execution results, FALSE otherwise
 *
 * @return TMBChoreographyExecution
 *
 * @throws TMBHTTPException If an HTTP error occurs
 * @throws TMBException if an error occurs during the request
 *
-(TMBChoreographyExecution*)executeAsyncWithInputs:(TMBChoreographyInputSet*)inputs storingResult:(BOOL)store {
	
	// Notify server that we want an asynchronous execution
	NSMutableDictionary *params = [NSMutableDictionary dictionaryWithObject:@"async" forKey:@"mode"];
	
	if(store) {
		// Notify server we want to store results
		[params setObject:@"true" forKey:@"store_results"];
	}
	
	// Initiate the server request
	NSDictionary *response = [self executeWithInputs:inputs andParameters:params];
	
	// Extract the Execution ID from the response
	NSString* executionId = [response objectForKey:@"id"];
	
	// Return a new autorleased TMBChoreographExecution
	return [[[TMBChoreographyExecution alloc] initWithSession:self.session andExecutionId:executionId] autorelease];
}
*/

/*!
 * Synchronous execution with inputs
 *
 * @param inputs
 *		The inputs to use when executing this choreography
 *
 * @return The parsed server result
 *
 * @throws TMBHTTPException If an HTTP error occurs
 * @throws TMBException if an error occurs during the request
 */
-(void)executeWithInputs:(TMBChoreographyInputSet*)inputs delegate:(id<TMBChoreographyDelegate>)d {
	[self executeWithInputs:inputs andParameters:Nil delegate:d];
}

/**
 * Synchronous execution with inputs and parameters
 *
 * @param inputs
 *		The inputs to use when executing this choreography
 * @param params
 *		Query string parameters to include with the request
 *
 * @return The parsed server result
 *
 * @throws TMBHTTPException If an HTTP error occurs
 * @throws TMBException if an error occurs during the request
 */
-(void)executeWithInputs:(TMBChoreographyInputSet*)inputs andParameters:(NSMutableDictionary*)params delegate:(id<TMBChoreographyDelegate>)d {
	
	if(nil == inputs)
		inputs = [self newInputSet];
	
	// Set and retain a reference to the TMBChoregraphyDelegate that initiated the request
	self.choreoDelegate = d;
	
	// Initiate the server request
	[super.session postContentToUri:self.uri withParameters:params andInputs:[inputs description] delegate:self];
}

/*!
 * @return A new, empty TMBChoreographyInputSet
 */
-(TMBChoreographyInputSet*)newInputSet {
	return [self newInputSetWithInputs:nil];
}

/*!
 * Creates a new TMBChoreographyInputSet populated with the key/value inputs provided
 *
 * @param inputs
 *		The inputs to populate the input set with
 *
 * @return A new TMBChoreographyInputSet which includes the provided input arguments
 */
-(TMBChoreographyInputSet*)newInputSetWithInputs:(NSDictionary*)i {
	return [[TMBChoreographyInputSet alloc] initWithInputs:i];
}

@end