/*!
 * @TMBYahoo.m
 *
 * Execute Choreographies from the Temboo Yahoo bundle.
 *
 * iOS version 10.6.8
 *
 * LICENSE: Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @category   Services
 * @author     Reid Simonton, Temboo, Inc.
 * @copyright  2013 Temboo, Inc.
 * @license    http://www.apache.org/licenses/LICENSE-2.0 Apache License 2.0
 * @version    1.7
 */

#import "TMBYahoo.h"
#import "TMBChoreography_Protected.h"
#import "TMBChoreographyExecution.h"
#import "TMBTembooSession.h"
#import "TMBTembooUri.h"


/*!
 * Input object with appropriate setters for specifying arguments to the GetNews Choreo.
 */
@implementation TMBYahoo_Finance_GetNews_Inputs

	/*!
	 * Set the value of the Company input for this Choreo.
	*(required, string) Ticker symbol for one or more companies to search, separated by commas (e.g. YHOO,DIS to include news about Yahoo! and Disney).
	 */
	-(void)setCompany:(NSString*)Company {
		[super setInput:@"Company" toValue:Company];
	}

	/*!
	 * Set the value of the NewsType input for this Choreo.
	*(required, string) Enter the type of news items you want to see in the RSS feed: headline or industry.
	 */
	-(void)setNewsType:(NSString*)NewsType {
		[super setInput:@"NewsType" toValue:NewsType];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: xml (the default) and json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetNews Choreo.
 */
@implementation TMBYahoo_Finance_GetNews_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Yahoo Finance.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the most recent Yahoo Finance Company or Industry news items as an RSS feed.
 */
@implementation TMBYahoo_Finance_GetNews

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBYahoo_Finance_GetNews Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Yahoo/Finance/GetNews"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBYahoo_Finance_GetNews_ResultSet *results = [[TMBYahoo_Finance_GetNews_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetNews Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBYahoo_Finance_GetNews_Inputs*)newInputSet {
		return [[TMBYahoo_Finance_GetNews_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetStockQuote Choreo.
 */
@implementation TMBYahoo_Finance_GetStockQuote_Inputs

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: json (the default) and xml.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the StockSymbol input for this Choreo.
	*(required, string) The stock ticker symbol to search for (e.g., AAPL, GOOG, etc).
	 */
	-(void)setStockSymbol:(NSString*)StockSymbol {
		[super setInput:@"StockSymbol" toValue:StockSymbol];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetStockQuote Choreo.
 */
@implementation TMBYahoo_Finance_GetStockQuote_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Ask" output from an execution of this Choreo.
	 * @return - NSString* (decimal) The asking price.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getAsk {
		return [super getOutputByName:@"Ask"];
	}

	/*!
	 * Retrieve the value of the "Bid" output from an execution of this Choreo.
	 * @return - NSString* (decimal) The bid price.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getBid {
		return [super getOutputByName:@"Bid"];
	}

	/*!
	 * Retrieve the value of the "Change" output from an execution of this Choreo.
	 * @return - NSString* (string) The change in the stock price.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getChange {
		return [super getOutputByName:@"Change"];
	}

	/*!
	 * Retrieve the value of the "DaysHigh" output from an execution of this Choreo.
	 * @return - NSString* (decimal) The high price of the day.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getDaysHigh {
		return [super getOutputByName:@"DaysHigh"];
	}

	/*!
	 * Retrieve the value of the "DaysLow" output from an execution of this Choreo.
	 * @return - NSString* (decimal) The low price of the day.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getDaysLow {
		return [super getOutputByName:@"DaysLow"];
	}

	/*!
	 * Retrieve the value of the "LastTradePriceOnly" output from an execution of this Choreo.
	 * @return - NSString* (decimal) The last trade price.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLastTradePriceOnly {
		return [super getOutputByName:@"LastTradePriceOnly"];
	}

	/*!
	 * Retrieve the value of the "Open" output from an execution of this Choreo.
	 * @return - NSString* (decimal) The price when the market last opened.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getOpen {
		return [super getOutputByName:@"Open"];
	}

	/*!
	 * Retrieve the value of the "PreviousClose" output from an execution of this Choreo.
	 * @return - NSString* (decimal) The previous closing price.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPreviousClose {
		return [super getOutputByName:@"PreviousClose"];
	}

	/*!
	 * Retrieve the value of the "Volume" output from an execution of this Choreo.
	 * @return - NSString* (integer) The volume traded.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getVolume {
		return [super getOutputByName:@"Volume"];
	}

	/*!
	 * Retrieve the value of the "YearHigh" output from an execution of this Choreo.
	 * @return - NSString* (decimal) The price for the year high.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getYearHigh {
		return [super getOutputByName:@"YearHigh"];
	}

	/*!
	 * Retrieve the value of the "YearLow" output from an execution of this Choreo.
	 * @return - NSString* (decimal) The price for the year low.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getYearLow {
		return [super getOutputByName:@"YearLow"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Yahoo Finance.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves information for the specified stock symbol from Yahoo Finance.
 */
@implementation TMBYahoo_Finance_GetStockQuote

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBYahoo_Finance_GetStockQuote Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Yahoo/Finance/GetStockQuote"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBYahoo_Finance_GetStockQuote_ResultSet *results = [[TMBYahoo_Finance_GetStockQuote_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetStockQuote Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBYahoo_Finance_GetStockQuote_Inputs*)newInputSet {
		return [[TMBYahoo_Finance_GetStockQuote_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the FindByAddress Choreo.
 */
@implementation TMBYahoo_PlaceFinder_FindByAddress_Inputs

	/*!
	 * Set the value of the Address input for this Choreo.
	*(required, string) The address to be searched.
	 */
	-(void)setAddress:(NSString*)Address {
		[super setInput:@"Address" toValue:Address];
	}

	/*!
	 * Set the value of the AppID input for this Choreo.
	*(optional, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setAppID:(NSString*)AppID {
		[super setInput:@"AppID" toValue:AppID];
	}

	/*!
	 * Set the value of the GeocodeFlags input for this Choreo.
	*(optional, string) Affects how geocoding is performed for the request. Valid value are: A, C, L, Q, or R. See documentation for more details on this parameter.
	 */
	-(void)setGeocodeFlags:(NSString*)GeocodeFlags {
		[super setInput:@"GeocodeFlags" toValue:GeocodeFlags];
	}

	/*!
	 * Set the value of the ResponseFlags input for this Choreo.
	*(optional, string) Determines which data elements are returned in the response. Valid values are: B, C, D, E, G, I, J, Q, R, T, U, W, X. See documentation for details on this parameter.
	 */
	-(void)setResponseFlags:(NSString*)ResponseFlags {
		[super setInput:@"ResponseFlags" toValue:ResponseFlags];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: xml (the default) and json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the FindByAddress Choreo.
 */
@implementation TMBYahoo_PlaceFinder_FindByAddress_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Latitude" output from an execution of this Choreo.
	 * @return - NSString* (decimal) The latitude coordinate for the location.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLatitude {
		return [super getOutputByName:@"Latitude"];
	}

	/*!
	 * Retrieve the value of the "Longitude" output from an execution of this Choreo.
	 * @return - NSString* (decimal) The longitude coordinate for the location.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLongitude {
		return [super getOutputByName:@"Longitude"];
	}

	/*!
	 * Retrieve the value of the "WOEID" output from an execution of this Choreo.
	 * @return - NSString* (integer) The unique Where On Earth ID of the location.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getWOEID {
		return [super getOutputByName:@"WOEID"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Yahoo PlaceFinder.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves complete geocoding information for a location by specifying an address or partial address.
 */
@implementation TMBYahoo_PlaceFinder_FindByAddress

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBYahoo_PlaceFinder_FindByAddress Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Yahoo/PlaceFinder/FindByAddress"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBYahoo_PlaceFinder_FindByAddress_ResultSet *results = [[TMBYahoo_PlaceFinder_FindByAddress_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the FindByAddress Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBYahoo_PlaceFinder_FindByAddress_Inputs*)newInputSet {
		return [[TMBYahoo_PlaceFinder_FindByAddress_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the FindByCoordinates Choreo.
 */
@implementation TMBYahoo_PlaceFinder_FindByCoordinates_Inputs

	/*!
	 * Set the value of the AppID input for this Choreo.
	*(optional, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setAppID:(NSString*)AppID {
		[super setInput:@"AppID" toValue:AppID];
	}

	/*!
	 * Set the value of the GeocodeFlags input for this Choreo.
	*(optional, string) Affects how geocoding is performed for the request. Valid value are: A, C, L, Q, or R. See documentation for more details on this parameter.
	 */
	-(void)setGeocodeFlags:(NSString*)GeocodeFlags {
		[super setInput:@"GeocodeFlags" toValue:GeocodeFlags];
	}

	/*!
	 * Set the value of the Latitude input for this Choreo.
	*(required, decimal) The latitude coordinate of the location you want to find (e.g., 38.898717).
	 */
	-(void)setLatitude:(NSString*)Latitude {
		[super setInput:@"Latitude" toValue:Latitude];
	}

	/*!
	 * Set the value of the Longitude input for this Choreo.
	*(required, decimal) The longitude coordinate of the location you want to find (e.g., -77.035974).
	 */
	-(void)setLongitude:(NSString*)Longitude {
		[super setInput:@"Longitude" toValue:Longitude];
	}

	/*!
	 * Set the value of the ResponseFlags input for this Choreo.
	*(optional, string) Determines which data elements are returned in the response. Valid values are: B, C, D, E, G, I, J, Q, R, T, U, W, X. See documentation for details on this parameter.
	 */
	-(void)setResponseFlags:(NSString*)ResponseFlags {
		[super setInput:@"ResponseFlags" toValue:ResponseFlags];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: xml (the default) and json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the FindByCoordinates Choreo.
 */
@implementation TMBYahoo_PlaceFinder_FindByCoordinates_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "WOEID" output from an execution of this Choreo.
	 * @return - NSString* (integer) The unique Where On Earth ID of the location.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getWOEID {
		return [super getOutputByName:@"WOEID"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Yahoo PlaceFinder.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves complete location information from a specified pair of latitude and longitude coordinates.
 */
@implementation TMBYahoo_PlaceFinder_FindByCoordinates

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBYahoo_PlaceFinder_FindByCoordinates Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Yahoo/PlaceFinder/FindByCoordinates"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBYahoo_PlaceFinder_FindByCoordinates_ResultSet *results = [[TMBYahoo_PlaceFinder_FindByCoordinates_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the FindByCoordinates Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBYahoo_PlaceFinder_FindByCoordinates_Inputs*)newInputSet {
		return [[TMBYahoo_PlaceFinder_FindByCoordinates_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetTemperature Choreo.
 */
@implementation TMBYahoo_Weather_GetTemperature_Inputs

	/*!
	 * Set the value of the Address input for this Choreo.
	*(required, string) The address to be searched.
	 */
	-(void)setAddress:(NSString*)Address {
		[super setInput:@"Address" toValue:Address];
	}

	/*!
	 * Set the value of the Units input for this Choreo.
	*(optional, string) The unit of temperature in the response. Acceptable inputs: f for Fahrenheit or c for Celsius. Defaults to f. When c is specified, all units measurements returned are changed to metric.
	 */
	-(void)setUnits:(NSString*)Units {
		[super setInput:@"Units" toValue:Units];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetTemperature Choreo.
 */
@implementation TMBYahoo_Weather_GetTemperature_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Temperature" output from an execution of this Choreo.
	 * @return - NSString* (integer) The current temperature (defaults to Fahrenheit).
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getTemperature {
		return [super getOutputByName:@"Temperature"];
	}
	
@end

/*!
 * Retrieves the current temperature from Yahoo Weather for the specified location.
 */
@implementation TMBYahoo_Weather_GetTemperature

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBYahoo_Weather_GetTemperature Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Yahoo/Weather/GetTemperature"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBYahoo_Weather_GetTemperature_ResultSet *results = [[TMBYahoo_Weather_GetTemperature_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetTemperature Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBYahoo_Weather_GetTemperature_Inputs*)newInputSet {
		return [[TMBYahoo_Weather_GetTemperature_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetWeather Choreo.
 */
@implementation TMBYahoo_Weather_GetWeather_Inputs

	/*!
	 * Set the value of the Day input for this Choreo.
	*(optional, integer) An index in the range 1 to 5 that corresponds to the forecast day you want to retrieve. Today corresponds to 1, tomorrow corresponds to 2, and so on. Defaults to 1.
	 */
	-(void)setDay:(NSString*)Day {
		[super setInput:@"Day" toValue:Day];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: xml (the default) and json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Units input for this Choreo.
	*(optional, string) The unit of temperature in the response. Acceptable inputs: f for Fahrenheit or c for Celsius. Defaults to f. When c is specified, all units measurements returned are changed to metric.
	 */
	-(void)setUnits:(NSString*)Units {
		[super setInput:@"Units" toValue:Units];
	}

	/*!
	 * Set the value of the WOEID input for this Choreo.
	*(required, integer) Where On Earth ID for the desired location. This unique integer can be found by first running the GetWeatherByCoordinates Choreo.
	 */
	-(void)setWOEID:(NSString*)WOEID {
		[super setInput:@"WOEID" toValue:WOEID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetWeather Choreo.
 */
@implementation TMBYahoo_Weather_GetWeather_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "ConditionCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) A code representing the current condition.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getConditionCode {
		return [super getOutputByName:@"ConditionCode"];
	}

	/*!
	 * Retrieve the value of the "ConditionText" output from an execution of this Choreo.
	 * @return - NSString* (string) The textual description for the current condition.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getConditionText {
		return [super getOutputByName:@"ConditionText"];
	}

	/*!
	 * Retrieve the value of the "ForecastCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) A code representing the forecast condition.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getForecastCode {
		return [super getOutputByName:@"ForecastCode"];
	}

	/*!
	 * Retrieve the value of the "ForecastText" output from an execution of this Choreo.
	 * @return - NSString* (string) The textual description for the specified day's forecast condition.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getForecastText {
		return [super getOutputByName:@"ForecastText"];
	}

	/*!
	 * Retrieve the value of the "High" output from an execution of this Choreo.
	 * @return - NSString* (integer) The high temperature forecast for the specified day.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getHigh {
		return [super getOutputByName:@"High"];
	}

	/*!
	 * Retrieve the value of the "Humidity" output from an execution of this Choreo.
	 * @return - NSString* (decimal) The current measurement for atmospheric humidity.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getHumidity {
		return [super getOutputByName:@"Humidity"];
	}

	/*!
	 * Retrieve the value of the "Low" output from an execution of this Choreo.
	 * @return - NSString* (integer) The low temperature forecast for the specified day.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLow {
		return [super getOutputByName:@"Low"];
	}

	/*!
	 * Retrieve the value of the "Pressure" output from an execution of this Choreo.
	 * @return - NSString* (decimal) The current measurement for atmospheric pressure.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPressure {
		return [super getOutputByName:@"Pressure"];
	}

	/*!
	 * Retrieve the value of the "Temperature" output from an execution of this Choreo.
	 * @return - NSString* (integer) The current temperature.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getTemperature {
		return [super getOutputByName:@"Temperature"];
	}

	/*!
	 * Retrieve the value of the "Visibility" output from an execution of this Choreo.
	 * @return - NSString* (decimal) The current measurement for visibility.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getVisibility {
		return [super getOutputByName:@"Visibility"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Yahoo Weather.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the Yahoo Weather RSS Feed for any specified location by WOEID.
 */
@implementation TMBYahoo_Weather_GetWeather

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBYahoo_Weather_GetWeather Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Yahoo/Weather/GetWeather"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBYahoo_Weather_GetWeather_ResultSet *results = [[TMBYahoo_Weather_GetWeather_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetWeather Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBYahoo_Weather_GetWeather_Inputs*)newInputSet {
		return [[TMBYahoo_Weather_GetWeather_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetWeatherByAddress Choreo.
 */
@implementation TMBYahoo_Weather_GetWeatherByAddress_Inputs

	/*!
	 * Set the value of the Address input for this Choreo.
	*(required, string) The address to be searched.
	 */
	-(void)setAddress:(NSString*)Address {
		[super setInput:@"Address" toValue:Address];
	}

	/*!
	 * Set the value of the AppID input for this Choreo.
	*(optional, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setAppID:(NSString*)AppID {
		[super setInput:@"AppID" toValue:AppID];
	}

	/*!
	 * Set the value of the Day input for this Choreo.
	*(optional, integer) An index in the range 1 to 5 that corresponds to the forecast day you want to retrieve. Today corresponds to 1, tomorrow corresponds to 2, and so on. Defaults to 1.
	 */
	-(void)setDay:(NSString*)Day {
		[super setInput:@"Day" toValue:Day];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: xml (the default) and json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Units input for this Choreo.
	*(optional, string) The unit of temperature in the response. Acceptable inputs: f for Fahrenheit or c for Celsius. Defaults to f. When c is specified, all units measurements returned are changed to metric.
	 */
	-(void)setUnits:(NSString*)Units {
		[super setInput:@"Units" toValue:Units];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetWeatherByAddress Choreo.
 */
@implementation TMBYahoo_Weather_GetWeatherByAddress_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "ConditionCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) A code representing the current condition.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getConditionCode {
		return [super getOutputByName:@"ConditionCode"];
	}

	/*!
	 * Retrieve the value of the "ConditionText" output from an execution of this Choreo.
	 * @return - NSString* (string) The textual description for the current condition.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getConditionText {
		return [super getOutputByName:@"ConditionText"];
	}

	/*!
	 * Retrieve the value of the "ForecastCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) A code representing the forecast condition.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getForecastCode {
		return [super getOutputByName:@"ForecastCode"];
	}

	/*!
	 * Retrieve the value of the "ForecastText" output from an execution of this Choreo.
	 * @return - NSString* (string) The textual description for the specified day's forecast condition.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getForecastText {
		return [super getOutputByName:@"ForecastText"];
	}

	/*!
	 * Retrieve the value of the "High" output from an execution of this Choreo.
	 * @return - NSString* (integer) The high temperature forecast for the specified day.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getHigh {
		return [super getOutputByName:@"High"];
	}

	/*!
	 * Retrieve the value of the "Humidity" output from an execution of this Choreo.
	 * @return - NSString* (decimal) The current measurement for atmospheric humidity.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getHumidity {
		return [super getOutputByName:@"Humidity"];
	}

	/*!
	 * Retrieve the value of the "Low" output from an execution of this Choreo.
	 * @return - NSString* (integer) The low temperature forecast for the specified day.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLow {
		return [super getOutputByName:@"Low"];
	}

	/*!
	 * Retrieve the value of the "Pressure" output from an execution of this Choreo.
	 * @return - NSString* (decimal) The current measurement for atmospheric pressure.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPressure {
		return [super getOutputByName:@"Pressure"];
	}

	/*!
	 * Retrieve the value of the "Temperature" output from an execution of this Choreo.
	 * @return - NSString* (integer) The current temperature.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getTemperature {
		return [super getOutputByName:@"Temperature"];
	}

	/*!
	 * Retrieve the value of the "Visibility" output from an execution of this Choreo.
	 * @return - NSString* (decimal) The current measurement for visibility.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getVisibility {
		return [super getOutputByName:@"Visibility"];
	}

	/*!
	 * Retrieve the value of the "WOEID" output from an execution of this Choreo.
	 * @return - NSString* (integer) The unique Where On Earth ID of the location.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getWOEID {
		return [super getOutputByName:@"WOEID"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Yahoo Weather.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the Yahoo Weather RSS Feed for any specified location by address.
 */
@implementation TMBYahoo_Weather_GetWeatherByAddress

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBYahoo_Weather_GetWeatherByAddress Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Yahoo/Weather/GetWeatherByAddress"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBYahoo_Weather_GetWeatherByAddress_ResultSet *results = [[TMBYahoo_Weather_GetWeatherByAddress_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetWeatherByAddress Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBYahoo_Weather_GetWeatherByAddress_Inputs*)newInputSet {
		return [[TMBYahoo_Weather_GetWeatherByAddress_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetWeatherByCoordinates Choreo.
 */
@implementation TMBYahoo_Weather_GetWeatherByCoordinates_Inputs

	/*!
	 * Set the value of the AppID input for this Choreo.
	*(optional, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setAppID:(NSString*)AppID {
		[super setInput:@"AppID" toValue:AppID];
	}

	/*!
	 * Set the value of the Day input for this Choreo.
	*(optional, integer) An index in the range 1 to 5 that corresponds to the forecast day you want to retrieve. Today corresponds to 1, tomorrow corresponds to 2, and so on. Defaults to 1.
	 */
	-(void)setDay:(NSString*)Day {
		[super setInput:@"Day" toValue:Day];
	}

	/*!
	 * Set the value of the Latitude input for this Choreo.
	*(required, decimal) The latitude coordinate of the location you want to search (e.g., 38.898717).
	 */
	-(void)setLatitude:(NSString*)Latitude {
		[super setInput:@"Latitude" toValue:Latitude];
	}

	/*!
	 * Set the value of the Longitude input for this Choreo.
	*(required, decimal) The longitude coordinate of the location you want to search (e.g., -77.035974).
	 */
	-(void)setLongitude:(NSString*)Longitude {
		[super setInput:@"Longitude" toValue:Longitude];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: xml (the default) and json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Units input for this Choreo.
	*(optional, string) The unit of temperature in the response. Acceptable inputs: f for Fahrenheit or c for Celsius. Defaults to f. When c is specified, all units measurements returned are changed to metric.
	 */
	-(void)setUnits:(NSString*)Units {
		[super setInput:@"Units" toValue:Units];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetWeatherByCoordinates Choreo.
 */
@implementation TMBYahoo_Weather_GetWeatherByCoordinates_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "ConditionCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) A code representing the current condition.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getConditionCode {
		return [super getOutputByName:@"ConditionCode"];
	}

	/*!
	 * Retrieve the value of the "ConditionText" output from an execution of this Choreo.
	 * @return - NSString* (string) The textual description for the current condition.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getConditionText {
		return [super getOutputByName:@"ConditionText"];
	}

	/*!
	 * Retrieve the value of the "ForecastCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) A code representing the forecast condition.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getForecastCode {
		return [super getOutputByName:@"ForecastCode"];
	}

	/*!
	 * Retrieve the value of the "ForecastText" output from an execution of this Choreo.
	 * @return - NSString* (string) The textual description for the specified day's forecast condition.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getForecastText {
		return [super getOutputByName:@"ForecastText"];
	}

	/*!
	 * Retrieve the value of the "High" output from an execution of this Choreo.
	 * @return - NSString* (integer) The high temperature forecast for the specified day.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getHigh {
		return [super getOutputByName:@"High"];
	}

	/*!
	 * Retrieve the value of the "Humidity" output from an execution of this Choreo.
	 * @return - NSString* (decimal) The current measurement for atmospheric humidity.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getHumidity {
		return [super getOutputByName:@"Humidity"];
	}

	/*!
	 * Retrieve the value of the "Low" output from an execution of this Choreo.
	 * @return - NSString* (integer) The low temperature forecast for the specified day.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLow {
		return [super getOutputByName:@"Low"];
	}

	/*!
	 * Retrieve the value of the "Pressure" output from an execution of this Choreo.
	 * @return - NSString* (decimal) The current measurement for atmospheric pressure.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPressure {
		return [super getOutputByName:@"Pressure"];
	}

	/*!
	 * Retrieve the value of the "Temperature" output from an execution of this Choreo.
	 * @return - NSString* (integer) The current temperature.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getTemperature {
		return [super getOutputByName:@"Temperature"];
	}

	/*!
	 * Retrieve the value of the "Visibility" output from an execution of this Choreo.
	 * @return - NSString* (decimal) The current measurement for visibility.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getVisibility {
		return [super getOutputByName:@"Visibility"];
	}

	/*!
	 * Retrieve the value of the "WOEID" output from an execution of this Choreo.
	 * @return - NSString* (integer) The unique Where On Earth ID of the location.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getWOEID {
		return [super getOutputByName:@"WOEID"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Yahoo Weather.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves the Yahoo Weather RSS Feed for any specified location by geo-coordinates.
 */
@implementation TMBYahoo_Weather_GetWeatherByCoordinates

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBYahoo_Weather_GetWeatherByCoordinates Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Yahoo/Weather/GetWeatherByCoordinates"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBYahoo_Weather_GetWeatherByCoordinates_ResultSet *results = [[TMBYahoo_Weather_GetWeatherByCoordinates_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetWeatherByCoordinates Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBYahoo_Weather_GetWeatherByCoordinates_Inputs*)newInputSet {
		return [[TMBYahoo_Weather_GetWeatherByCoordinates_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the RunQuery Choreo.
 */
@implementation TMBYahoo_YQL_RunQuery_Inputs

	/*!
	 * Set the value of the Diagnostics input for this Choreo.
	*(optional, boolean) When set to "true" (the default), additional debug information about the request is returned.
	 */
	-(void)setDiagnostics:(NSString*)Diagnostics {
		[super setInput:@"Diagnostics" toValue:Diagnostics];
	}

	/*!
	 * Set the value of the Query input for this Choreo.
	*(required, string) A YQL (Yahoo! Query Language) statement to execute.
	 */
	-(void)setQuery:(NSString*)Query {
		[super setInput:@"Query" toValue:Query];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: json (the default) and xml.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the RunQuery Choreo.
 */
@implementation TMBYahoo_YQL_RunQuery_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Yahoo.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Executes a specified YQL (Yahoo Query Language) statement.
 */
@implementation TMBYahoo_YQL_RunQuery

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBYahoo_YQL_RunQuery Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Yahoo/YQL/RunQuery"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBYahoo_YQL_RunQuery_ResultSet *results = [[TMBYahoo_YQL_RunQuery_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the RunQuery Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBYahoo_YQL_RunQuery_Inputs*)newInputSet {
		return [[TMBYahoo_YQL_RunQuery_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ScrapeWebPage Choreo.
 */
@implementation TMBYahoo_YQL_ScrapeWebPage_Inputs

	/*!
	 * Set the value of the Diagnostics input for this Choreo.
	*(optional, boolean) When set to "true" (the default), additional debug information about the request is returned.
	 */
	-(void)setDiagnostics:(NSString*)Diagnostics {
		[super setInput:@"Diagnostics" toValue:Diagnostics];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: json (the default) and xml.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the URL input for this Choreo.
	*(required, string) The URL of the web page to scrape.
	 */
	-(void)setURL:(NSString*)URL {
		[super setInput:@"URL" toValue:URL];
	}

	/*!
	 * Set the value of the XPATH input for this Choreo.
	*(optional, string) An XPATH statement that can be used to extract specific information from the HTML.
	 */
	-(void)setXPATH:(NSString*)XPATH {
		[super setInput:@"XPATH" toValue:XPATH];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ScrapeWebPage Choreo.
 */
@implementation TMBYahoo_YQL_ScrapeWebPage_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Yahoo.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Scrapes HTML from a web page and converts it to JSON or XML so that it can be reused by an application.
 */
@implementation TMBYahoo_YQL_ScrapeWebPage

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBYahoo_YQL_ScrapeWebPage Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Yahoo/YQL/ScrapeWebPage"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBYahoo_YQL_ScrapeWebPage_ResultSet *results = [[TMBYahoo_YQL_ScrapeWebPage_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ScrapeWebPage Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBYahoo_YQL_ScrapeWebPage_Inputs*)newInputSet {
		return [[TMBYahoo_YQL_ScrapeWebPage_Inputs alloc] init];
	}
@end
	