/*!
 * @TMBUtilities.m
 *
 * Execute Choreographies from the Temboo Utilities bundle.
 *
 * iOS version 10.6.8
 *
 * LICENSE: Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @category   Services
 * @author     Reid Simonton, Temboo, Inc.
 * @copyright  2013 Temboo, Inc.
 * @license    http://www.apache.org/licenses/LICENSE-2.0 Apache License 2.0
 * @version    1.7
 */

#import "TMBUtilities.h"
#import "TMBChoreography_Protected.h"
#import "TMBChoreographyExecution.h"
#import "TMBTembooSession.h"
#import "TMBTembooUri.h"


/*!
 * Input object with appropriate setters for specifying arguments to the FinalizeOAuth Choreo.
 */
@implementation TMBUtilities_Authentication_OAuth1_FinalizeOAuth_Inputs

	/*!
	 * Set the value of the AccessTokenURL input for this Choreo.
	*(required, string) The URL for the authorization server that issues access tokens (e.g. https://api.dropbox.com/1/oauth/access_token).
	 */
	-(void)setAccessTokenURL:(NSString*)AccessTokenURL {
		[super setInput:@"AccessTokenURL" toValue:AccessTokenURL];
	}

	/*!
	 * Set the value of the CallbackID input for this Choreo.
	*(required, string) The callback ID returned by the InitializeOAuth Choreo. Used to retrieve the callback data after the user authorizes.
	 */
	-(void)setCallbackID:(NSString*)CallbackID {
		[super setInput:@"CallbackID" toValue:CallbackID];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The Consumer Key provided by the service.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The Consumer Secret provided by the service.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the OAuthTokenSecret input for this Choreo.
	*(required, string) The oauth_token_secret retrieved during the OAuth process. This is returned by the InitializeOAuth Choreo.
	 */
	-(void)setOAuthTokenSecret:(NSString*)OAuthTokenSecret {
		[super setInput:@"OAuthTokenSecret" toValue:OAuthTokenSecret];
	}

	/*!
	 * Set the value of the SuppressErrors input for this Choreo.
	*(optional, boolean) When set to true, errors received during the OAuth redirect process will be suppressed and returned in the ErrorMessage output.
	 */
	-(void)setSuppressErrors:(NSString*)SuppressErrors {
		[super setInput:@"SuppressErrors" toValue:SuppressErrors];
	}

	/*!
	 * Set the value of the Timeout input for this Choreo.
	*(optional, integer) The amount of time (in seconds) to poll your Temboo callback URL to see if your app's user has allowed or denied the request for access. Defaults to 20. Max is 60.
	 */
	-(void)setTimeout:(NSString*)Timeout {
		[super setInput:@"Timeout" toValue:Timeout];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the FinalizeOAuth Choreo.
 */
@implementation TMBUtilities_Authentication_OAuth1_FinalizeOAuth_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "ErrorMessage" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains an error message if an error occurs during the OAuth redirect process and if SuppressErrors is set to true.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getErrorMessage {
		return [super getOutputByName:@"ErrorMessage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (string) The response from the Authorization Server.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Completes the OAuth process by retrieving an access token and access token secret for a user, after they have visited the authorization URL returned by the InitializeOAuth Choreo and clicked "allow."
 */
@implementation TMBUtilities_Authentication_OAuth1_FinalizeOAuth

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Authentication_OAuth1_FinalizeOAuth Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Authentication/OAuth1/FinalizeOAuth"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Authentication_OAuth1_FinalizeOAuth_ResultSet *results = [[TMBUtilities_Authentication_OAuth1_FinalizeOAuth_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the FinalizeOAuth Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Authentication_OAuth1_FinalizeOAuth_Inputs*)newInputSet {
		return [[TMBUtilities_Authentication_OAuth1_FinalizeOAuth_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the InitializeOAuth Choreo.
 */
@implementation TMBUtilities_Authentication_OAuth1_InitializeOAuth_Inputs

	/*!
	 * Set the value of the AuthorizationEndpoint input for this Choreo.
	*(required, string) The endpoint that the user should be redirected to in order to authenticate and grant access (e.g. https://www.dropbox.com/1/oauth/authorize).
	 */
	-(void)setAuthorizationEndpoint:(NSString*)AuthorizationEndpoint {
		[super setInput:@"AuthorizationEndpoint" toValue:AuthorizationEndpoint];
	}

	/*!
	 * Set the value of the ConsumerKey input for this Choreo.
	*(required, string) The Consumer Key provided by the service.
	 */
	-(void)setConsumerKey:(NSString*)ConsumerKey {
		[super setInput:@"ConsumerKey" toValue:ConsumerKey];
	}

	/*!
	 * Set the value of the ConsumerSecret input for this Choreo.
	*(required, string) The Consumer Secret provided by the service.
	 */
	-(void)setConsumerSecret:(NSString*)ConsumerSecret {
		[super setInput:@"ConsumerSecret" toValue:ConsumerSecret];
	}

	/*!
	 * Set the value of the ForwardingURL input for this Choreo.
	*(optional, string) The URL that Temboo will redirect your users to after they grant access to your application. This should include the "https://" or "http://" prefix and be a fully qualified URL.
	 */
	-(void)setForwardingURL:(NSString*)ForwardingURL {
		[super setInput:@"ForwardingURL" toValue:ForwardingURL];
	}

	/*!
	 * Set the value of the RequestTokenEndpoint input for this Choreo.
	*(required, string) The Authorization Server URL where the initial token request occurs (e.g. https://api.dropbox.com/1/oauth/request_token).
	 */
	-(void)setRequestTokenEndpoint:(NSString*)RequestTokenEndpoint {
		[super setInput:@"RequestTokenEndpoint" toValue:RequestTokenEndpoint];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the InitializeOAuth Choreo.
 */
@implementation TMBUtilities_Authentication_OAuth1_InitializeOAuth_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "AuthorizationURL" output from an execution of this Choreo.
	 * @return - NSString* (string) The authorization URL that the application's user needs to go to in order to grant access to your application.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getAuthorizationURL {
		return [super getOutputByName:@"AuthorizationURL"];
	}

	/*!
	 * Retrieve the value of the "CallbackID" output from an execution of this Choreo.
	 * @return - NSString* (string) An ID used to retrieve the callback data that Temboo stores once your application's user authorizes.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getCallbackID {
		return [super getOutputByName:@"CallbackID"];
	}

	/*!
	 * Retrieve the value of the "OAuthTokenSecret" output from an execution of this Choreo.
	 * @return - NSString* (string) The temporary OAuth Token Secret that can be exchanged for permanent tokens using the FinalizeOAuth Choreo.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getOAuthTokenSecret {
		return [super getOutputByName:@"OAuthTokenSecret"];
	}
	
@end

/*!
 * Generates an authorization URL that an application can use to complete the first step in the OAuth 1.0 process.
 */
@implementation TMBUtilities_Authentication_OAuth1_InitializeOAuth

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Authentication_OAuth1_InitializeOAuth Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Authentication/OAuth1/InitializeOAuth"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Authentication_OAuth1_InitializeOAuth_ResultSet *results = [[TMBUtilities_Authentication_OAuth1_InitializeOAuth_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the InitializeOAuth Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Authentication_OAuth1_InitializeOAuth_Inputs*)newInputSet {
		return [[TMBUtilities_Authentication_OAuth1_InitializeOAuth_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the FinalizeOAuth Choreo.
 */
@implementation TMBUtilities_Authentication_OAuth2_FinalizeOAuth_Inputs

	/*!
	 * Set the value of the AccessTokenEndpoint input for this Choreo.
	*(required, string) The URL for the authorization server that issues access tokens (e.g. https://accounts.google.com/o/oauth2/token).
	 */
	-(void)setAccessTokenEndpoint:(NSString*)AccessTokenEndpoint {
		[super setInput:@"AccessTokenEndpoint" toValue:AccessTokenEndpoint];
	}

	/*!
	 * Set the value of the CallbackID input for this Choreo.
	*(required, string) The callback token returned by the InitializeOAuth Choreo. Used to retrieve the authorization code after the user authorizes.
	 */
	-(void)setCallbackID:(NSString*)CallbackID {
		[super setInput:@"CallbackID" toValue:CallbackID];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(required, string) The Client ID provided by the service.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(required, string) The Client Secret provided by the service.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the SuppressErrors input for this Choreo.
	*(optional, boolean) When set to true, errors received during the OAuth redirect process will be suppressed and returned in the ErrorMessage output.
	 */
	-(void)setSuppressErrors:(NSString*)SuppressErrors {
		[super setInput:@"SuppressErrors" toValue:SuppressErrors];
	}

	/*!
	 * Set the value of the Timeout input for this Choreo.
	*(optional, integer) The amount of time (in seconds) to poll your Temboo callback URL to see if your app's user has allowed or denied the request for access. Defaults to 20. Max is 60.
	 */
	-(void)setTimeout:(NSString*)Timeout {
		[super setInput:@"Timeout" toValue:Timeout];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the FinalizeOAuth Choreo.
 */
@implementation TMBUtilities_Authentication_OAuth2_FinalizeOAuth_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "ErrorMessage" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains an error message if an error occurs during the OAuth redirect process and if SuppressErrors is set to true.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getErrorMessage {
		return [super getOutputByName:@"ErrorMessage"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from the server.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Completes the OAuth process by retrieving an access token for a user, after they have visited the authorization URL returned by the InitializeOAuth Choreo and clicked "allow."
 */
@implementation TMBUtilities_Authentication_OAuth2_FinalizeOAuth

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Authentication_OAuth2_FinalizeOAuth Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Authentication/OAuth2/FinalizeOAuth"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Authentication_OAuth2_FinalizeOAuth_ResultSet *results = [[TMBUtilities_Authentication_OAuth2_FinalizeOAuth_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the FinalizeOAuth Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Authentication_OAuth2_FinalizeOAuth_Inputs*)newInputSet {
		return [[TMBUtilities_Authentication_OAuth2_FinalizeOAuth_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the InitializeOAuth Choreo.
 */
@implementation TMBUtilities_Authentication_OAuth2_InitializeOAuth_Inputs

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(required, string) The Client ID provided by the service.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the CustomCallbackID input for this Choreo.
	*(optional, string) A unique identifier that you can pass to eliminate the need to wait for a Temboo-generated CallbackID. Callback identifiers may only contain numbers, letters, periods, and hyphens.
	 */
	-(void)setCustomCallbackID:(NSString*)CustomCallbackID {
		[super setInput:@"CustomCallbackID" toValue:CustomCallbackID];
	}

	/*!
	 * Set the value of the ForwardingURL input for this Choreo.
	*(optional, string) The URL that Temboo will redirect your users to after they grant access to your application. This should include the "https://" or "http://" prefix and be a fully qualified URL.
	 */
	-(void)setForwardingURL:(NSString*)ForwardingURL {
		[super setInput:@"ForwardingURL" toValue:ForwardingURL];
	}

	/*!
	 * Set the value of the RequestTokenEndpoint input for this Choreo.
	*(required, string) The Authorization Server URL where the initial token request occurs (e.g. https://accounts.google.com/o/oauth2/auth).
	 */
	-(void)setRequestTokenEndpoint:(NSString*)RequestTokenEndpoint {
		[super setInput:@"RequestTokenEndpoint" toValue:RequestTokenEndpoint];
	}

	/*!
	 * Set the value of the Scope input for this Choreo.
	*(conditional, string) The OAuth scope that should be associated with the access token being requested. This is not always required. Typically, you can specify multiple scopes separated by spaces, commas, or pipes.
	 */
	-(void)setScope:(NSString*)Scope {
		[super setInput:@"Scope" toValue:Scope];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the InitializeOAuth Choreo.
 */
@implementation TMBUtilities_Authentication_OAuth2_InitializeOAuth_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "AuthorizationURL" output from an execution of this Choreo.
	 * @return - NSString* (string) The authorization URL that the application's user needs to go to in order to grant access to your application.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getAuthorizationURL {
		return [super getOutputByName:@"AuthorizationURL"];
	}

	/*!
	 * Retrieve the value of the "CallbackID" output from an execution of this Choreo.
	 * @return - NSString* (string) An ID used to retrieve the callback data that Temboo stores once your application's user authorizes.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getCallbackID {
		return [super getOutputByName:@"CallbackID"];
	}
	
@end

/*!
 * Generates an authorization URL that an application can use to complete the first step in the OAuth 2.0 process.
 */
@implementation TMBUtilities_Authentication_OAuth2_InitializeOAuth

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Authentication_OAuth2_InitializeOAuth Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Authentication/OAuth2/InitializeOAuth"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Authentication_OAuth2_InitializeOAuth_ResultSet *results = [[TMBUtilities_Authentication_OAuth2_InitializeOAuth_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the InitializeOAuth Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Authentication_OAuth2_InitializeOAuth_Inputs*)newInputSet {
		return [[TMBUtilities_Authentication_OAuth2_InitializeOAuth_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the RefreshToken Choreo.
 */
@implementation TMBUtilities_Authentication_OAuth2_RefreshToken_Inputs

	/*!
	 * Set the value of the AccessTokenEndpoint input for this Choreo.
	*(required, string) The URL for the authorization server that issues access tokens (e.g. https://accounts.google.com/o/oauth2/token).
	 */
	-(void)setAccessTokenEndpoint:(NSString*)AccessTokenEndpoint {
		[super setInput:@"AccessTokenEndpoint" toValue:AccessTokenEndpoint];
	}

	/*!
	 * Set the value of the ClientID input for this Choreo.
	*(required, string) The Client ID provided by the service.
	 */
	-(void)setClientID:(NSString*)ClientID {
		[super setInput:@"ClientID" toValue:ClientID];
	}

	/*!
	 * Set the value of the ClientSecret input for this Choreo.
	*(required, string) The Client Secret provided by the service.
	 */
	-(void)setClientSecret:(NSString*)ClientSecret {
		[super setInput:@"ClientSecret" toValue:ClientSecret];
	}

	/*!
	 * Set the value of the RefreshToken input for this Choreo.
	*(required, string) The refresh token retrieved in the OAuth process to be used when your access token expires.
	 */
	-(void)setRefreshToken:(NSString*)RefreshToken {
		[super setInput:@"RefreshToken" toValue:RefreshToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the RefreshToken Choreo.
 */
@implementation TMBUtilities_Authentication_OAuth2_RefreshToken_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from the server.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Refreshes an expired access token.
 */
@implementation TMBUtilities_Authentication_OAuth2_RefreshToken

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Authentication_OAuth2_RefreshToken Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Authentication/OAuth2/RefreshToken"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Authentication_OAuth2_RefreshToken_ResultSet *results = [[TMBUtilities_Authentication_OAuth2_RefreshToken_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the RefreshToken Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Authentication_OAuth2_RefreshToken_Inputs*)newInputSet {
		return [[TMBUtilities_Authentication_OAuth2_RefreshToken_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetCallbackData Choreo.
 */
@implementation TMBUtilities_Callback_GetCallbackData_Inputs

	/*!
	 * Set the value of the CallbackID input for this Choreo.
	*(required, string) The ID that can used to retrieve request data that the callback URL has captured.
	 */
	-(void)setCallbackID:(NSString*)CallbackID {
		[super setInput:@"CallbackID" toValue:CallbackID];
	}

	/*!
	 * Set the value of the FilterName input for this Choreo.
	*(optional, string) Allows you to filter callback data by a query parameter key-value pair. FilterValue is required when using this input.
	 */
	-(void)setFilterName:(NSString*)FilterName {
		[super setInput:@"FilterName" toValue:FilterName];
	}

	/*!
	 * Set the value of the FilterValue input for this Choreo.
	*(optional, string) Allows you to filter callback data by a query parameter key-value pair. FilterName is required when using this input.
	 */
	-(void)setFilterValue:(NSString*)FilterValue {
		[super setInput:@"FilterValue" toValue:FilterValue];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: json (the default) and xml.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetCallbackData Choreo.
 */
@implementation TMBUtilities_Callback_GetCallbackData_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "CallbackData" output from an execution of this Choreo.
	 * @return - NSString* Contains the request data received at your CalllbackURL.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getCallbackData {
		return [super getOutputByName:@"CallbackData"];
	}
	
@end

/*!
 * Retrieves data captured from a request to your callback URL.
 */
@implementation TMBUtilities_Callback_GetCallbackData

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Callback_GetCallbackData Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Callback/GetCallbackData"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Callback_GetCallbackData_ResultSet *results = [[TMBUtilities_Callback_GetCallbackData_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetCallbackData Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Callback_GetCallbackData_Inputs*)newInputSet {
		return [[TMBUtilities_Callback_GetCallbackData_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the RegisterCallback Choreo.
 */
@implementation TMBUtilities_Callback_RegisterCallback_Inputs

	/*!
	 * Set the value of the CustomCallbackD input for this Choreo.
	*(optional, string) This value is used to register a unique URL associated with your account. If omitted, a random identifier is generated. Using a custom value here is useful when you need the URL to be static.
	 */
	-(void)setCustomCallbackD:(NSString*)CustomCallbackD {
		[super setInput:@"CustomCallbackD" toValue:CustomCallbackD];
	}

	/*!
	 * Set the value of the FilterName input for this Choreo.
	*(optional, string) When using a Custom Callback ID, it can be useful to filter requests using a query parameter. This value is used as a query parameter name, and can be used to lookup request data.
	 */
	-(void)setFilterName:(NSString*)FilterName {
		[super setInput:@"FilterName" toValue:FilterName];
	}

	/*!
	 * Set the value of the FilterValue input for this Choreo.
	*(optional, string) When using a Custom Callback ID, it can be useful to filter requests using a query parameter. This value is used as a query parameter value, and can be used to lookup request data.
	 */
	-(void)setFilterValue:(NSString*)FilterValue {
		[super setInput:@"FilterValue" toValue:FilterValue];
	}

	/*!
	 * Set the value of the ForwardingURL input for this Choreo.
	*(optional, string) The URL that Temboo will redirect a users to after they visit your URL. This should include the "https://" or "http://" prefix and be a fully qualified URL.
	 */
	-(void)setForwardingURL:(NSString*)ForwardingURL {
		[super setInput:@"ForwardingURL" toValue:ForwardingURL];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the RegisterCallback Choreo.
 */
@implementation TMBUtilities_Callback_RegisterCallback_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "CallbackID" output from an execution of this Choreo.
	 * @return - NSString* (string) The ID that can used to retrieve request data that the Callback URL captures.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getCallbackID {
		return [super getOutputByName:@"CallbackID"];
	}

	/*!
	 * Retrieve the value of the "CallbackURL" output from an execution of this Choreo.
	 * @return - NSString* (string) The URL that is listening for an incoming request. Note that this URL will expire in 10 minutes.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getCallbackURL {
		return [super getOutputByName:@"CallbackURL"];
	}
	
@end

/*!
 * Allows you to generate a unique URL that can "listen" for incoming data from web request.
 */
@implementation TMBUtilities_Callback_RegisterCallback

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Callback_RegisterCallback Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Callback/RegisterCallback"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Callback_RegisterCallback_ResultSet *results = [[TMBUtilities_Callback_RegisterCallback_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the RegisterCallback Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Callback_RegisterCallback_Inputs*)newInputSet {
		return [[TMBUtilities_Callback_RegisterCallback_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CSVToJSON Choreo.
 */
@implementation TMBUtilities_DataConversions_CSVToJSON_Inputs

	/*!
	 * Set the value of the CSV input for this Choreo.
	*(required, multiline) The CSV file to convert to JSON. Your CSV data must contain column names.
	 */
	-(void)setCSV:(NSString*)CSV {
		[super setInput:@"CSV" toValue:CSV];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CSVToJSON Choreo.
 */
@implementation TMBUtilities_DataConversions_CSVToJSON_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "JSON" output from an execution of this Choreo.
	 * @return - NSString* (json) The JSON formatted data.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getJSON {
		return [super getOutputByName:@"JSON"];
	}
	
@end

/*!
 * Converts a CSV formatted file to JSON.
 */
@implementation TMBUtilities_DataConversions_CSVToJSON

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_DataConversions_CSVToJSON Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/DataConversions/CSVToJSON"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_DataConversions_CSVToJSON_ResultSet *results = [[TMBUtilities_DataConversions_CSVToJSON_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CSVToJSON Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_DataConversions_CSVToJSON_Inputs*)newInputSet {
		return [[TMBUtilities_DataConversions_CSVToJSON_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CSVToXLS Choreo.
 */
@implementation TMBUtilities_DataConversions_CSVToXLS_Inputs

	/*!
	 * Set the value of the CSV input for this Choreo.
	*(conditional, multiline) The CSV data you want to convert to XLS format. Required unless using the VaultFile input alias (an advanced option used when running Choreos in the Temboo Designer).
	 */
	-(void)setCSV:(NSString*)CSV {
		[super setInput:@"CSV" toValue:CSV];
	}

	/*!
	 * Set the value of the VaultFile input for this Choreo.
	*(optional, vault file) A path to a CSV file you've stored in the Vault. This can be used as an alternative to the CSVFile input variable.
	 */
	-(void)setVaultFile:(NSString*)VaultFile {
		[super setInput:@"VaultFile" toValue:VaultFile];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CSVToXLS Choreo.
 */
@implementation TMBUtilities_DataConversions_CSVToXLS_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "XLS" output from an execution of this Choreo.
	 * @return - NSString* (string) The Base64 encoded Excel data.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getXLS {
		return [super getOutputByName:@"XLS"];
	}
	
@end

/*!
 * Converts a CSV formatted file to Base64 encoded Excel data.
 */
@implementation TMBUtilities_DataConversions_CSVToXLS

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_DataConversions_CSVToXLS Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/DataConversions/CSVToXLS"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_DataConversions_CSVToXLS_ResultSet *results = [[TMBUtilities_DataConversions_CSVToXLS_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CSVToXLS Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_DataConversions_CSVToXLS_Inputs*)newInputSet {
		return [[TMBUtilities_DataConversions_CSVToXLS_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CSVToXML Choreo.
 */
@implementation TMBUtilities_DataConversions_CSVToXML_Inputs

	/*!
	 * Set the value of the CSV input for this Choreo.
	*(required, multiline) The CSV file to convert to XML. Your CSV data must contain column names.
	 */
	-(void)setCSV:(NSString*)CSV {
		[super setInput:@"CSV" toValue:CSV];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CSVToXML Choreo.
 */
@implementation TMBUtilities_DataConversions_CSVToXML_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "XML" output from an execution of this Choreo.
	 * @return - NSString* (xml) The XML formatted data.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getXML {
		return [super getOutputByName:@"XML"];
	}
	
@end

/*!
 * Converts a CSV formatted file to XML.
 */
@implementation TMBUtilities_DataConversions_CSVToXML

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_DataConversions_CSVToXML Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/DataConversions/CSVToXML"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_DataConversions_CSVToXML_ResultSet *results = [[TMBUtilities_DataConversions_CSVToXML_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CSVToXML Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_DataConversions_CSVToXML_Inputs*)newInputSet {
		return [[TMBUtilities_DataConversions_CSVToXML_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the JSONToXML Choreo.
 */
@implementation TMBUtilities_DataConversions_JSONToXML_Inputs

	/*!
	 * Set the value of the JSON input for this Choreo.
	*(required, json) The JSON data that you want to convert to XML.
	 */
	-(void)setJSON:(NSString*)JSON {
		[super setInput:@"JSON" toValue:JSON];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the JSONToXML Choreo.
 */
@implementation TMBUtilities_DataConversions_JSONToXML_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "XML" output from an execution of this Choreo.
	 * @return - NSString* (xml) The converted data in XML format.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getXML {
		return [super getOutputByName:@"XML"];
	}
	
@end

/*!
 * Converts data from JSON format to XML format.
 */
@implementation TMBUtilities_DataConversions_JSONToXML

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_DataConversions_JSONToXML Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/DataConversions/JSONToXML"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_DataConversions_JSONToXML_ResultSet *results = [[TMBUtilities_DataConversions_JSONToXML_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the JSONToXML Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_DataConversions_JSONToXML_Inputs*)newInputSet {
		return [[TMBUtilities_DataConversions_JSONToXML_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the RSSToJSON Choreo.
 */
@implementation TMBUtilities_DataConversions_RSSToJSON_Inputs

	/*!
	 * Set the value of the RSSFeed input for this Choreo.
	*(required, string) The URL for an RSS feed that you wish to convert to JSON.
	 */
	-(void)setRSSFeed:(NSString*)RSSFeed {
		[super setInput:@"RSSFeed" toValue:RSSFeed];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the RSSToJSON Choreo.
 */
@implementation TMBUtilities_DataConversions_RSSToJSON_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The feed data in JSON format.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a specified RSS Feed, and converts it to JSON format.
 */
@implementation TMBUtilities_DataConversions_RSSToJSON

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_DataConversions_RSSToJSON Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/DataConversions/RSSToJSON"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_DataConversions_RSSToJSON_ResultSet *results = [[TMBUtilities_DataConversions_RSSToJSON_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the RSSToJSON Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_DataConversions_RSSToJSON_Inputs*)newInputSet {
		return [[TMBUtilities_DataConversions_RSSToJSON_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the TSVToJSON Choreo.
 */
@implementation TMBUtilities_DataConversions_TSVToJSON_Inputs

	/*!
	 * Set the value of the TSV input for this Choreo.
	*(required, multiline) The TSV file to convert to XML. Your TSV data must contain column names.
	 */
	-(void)setTSV:(NSString*)TSV {
		[super setInput:@"TSV" toValue:TSV];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the TSVToJSON Choreo.
 */
@implementation TMBUtilities_DataConversions_TSVToJSON_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "JSON" output from an execution of this Choreo.
	 * @return - NSString* (json) The JSON formatted data.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getJSON {
		return [super getOutputByName:@"JSON"];
	}
	
@end

/*!
 * Converts a TSV formatted file to JSON.
 */
@implementation TMBUtilities_DataConversions_TSVToJSON

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_DataConversions_TSVToJSON Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/DataConversions/TSVToJSON"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_DataConversions_TSVToJSON_ResultSet *results = [[TMBUtilities_DataConversions_TSVToJSON_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the TSVToJSON Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_DataConversions_TSVToJSON_Inputs*)newInputSet {
		return [[TMBUtilities_DataConversions_TSVToJSON_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the TSVToXML Choreo.
 */
@implementation TMBUtilities_DataConversions_TSVToXML_Inputs

	/*!
	 * Set the value of the TSV input for this Choreo.
	*(required, multiline) The TSV file to convert to XML. Your TSV data must contain column names.
	 */
	-(void)setTSV:(NSString*)TSV {
		[super setInput:@"TSV" toValue:TSV];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the TSVToXML Choreo.
 */
@implementation TMBUtilities_DataConversions_TSVToXML_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "XML" output from an execution of this Choreo.
	 * @return - NSString* (xml) The XML formatted data.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getXML {
		return [super getOutputByName:@"XML"];
	}
	
@end

/*!
 * Converts a TSV formatted file to XML.
 */
@implementation TMBUtilities_DataConversions_TSVToXML

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_DataConversions_TSVToXML Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/DataConversions/TSVToXML"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_DataConversions_TSVToXML_ResultSet *results = [[TMBUtilities_DataConversions_TSVToXML_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the TSVToXML Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_DataConversions_TSVToXML_Inputs*)newInputSet {
		return [[TMBUtilities_DataConversions_TSVToXML_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the XLSToCSV Choreo.
 */
@implementation TMBUtilities_DataConversions_XLSToCSV_Inputs

	/*!
	 * Set the value of the XLS input for this Choreo.
	*(conditional, string) The base64-encoded contents of the Excel file that you want to convert to CSV format. Compatible with Excel 97-2003.
	 */
	-(void)setXLS:(NSString*)XLS {
		[super setInput:@"XLS" toValue:XLS];
	}

	/*!
	 * Set the value of the VaultFile input for this Choreo.
	*(optional, vault file) A path to an Excel file you've stored in the Vault. This can be used as an alternative to the ExcelFile input variable.
	 */
	-(void)setVaultFile:(NSString*)VaultFile {
		[super setInput:@"VaultFile" toValue:VaultFile];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the XLSToCSV Choreo.
 */
@implementation TMBUtilities_DataConversions_XLSToCSV_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "CSV" output from an execution of this Choreo.
	 * @return - NSString* (string) The CSV formatted data.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getCSV {
		return [super getOutputByName:@"CSV"];
	}
	
@end

/*!
 * Converts Excel (.xls) formatted data to CSV.
 */
@implementation TMBUtilities_DataConversions_XLSToCSV

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_DataConversions_XLSToCSV Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/DataConversions/XLSToCSV"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_DataConversions_XLSToCSV_ResultSet *results = [[TMBUtilities_DataConversions_XLSToCSV_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the XLSToCSV Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_DataConversions_XLSToCSV_Inputs*)newInputSet {
		return [[TMBUtilities_DataConversions_XLSToCSV_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the XLSToXML Choreo.
 */
@implementation TMBUtilities_DataConversions_XLSToXML_Inputs

	/*!
	 * Set the value of the XLS input for this Choreo.
	*(conditional, string) The base64-encoded contents of the Excel file that you want to convert to XML. Compatible with Excel 97-2003.
	 */
	-(void)setXLS:(NSString*)XLS {
		[super setInput:@"XLS" toValue:XLS];
	}

	/*!
	 * Set the value of the VaultFile input for this Choreo.
	*(optional, vault file) A path to an Excel file that has been uploaded to the vault. You can use this input alias as an alternative to the ExcelFile input variable.
	 */
	-(void)setVaultFile:(NSString*)VaultFile {
		[super setInput:@"VaultFile" toValue:VaultFile];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the XLSToXML Choreo.
 */
@implementation TMBUtilities_DataConversions_XLSToXML_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "XML" output from an execution of this Choreo.
	 * @return - NSString* (xml) The data in XML format.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getXML {
		return [super getOutputByName:@"XML"];
	}
	
@end

/*!
 * Converts Excel (.xls) formatted data to XML.
 */
@implementation TMBUtilities_DataConversions_XLSToXML

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_DataConversions_XLSToXML Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/DataConversions/XLSToXML"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_DataConversions_XLSToXML_ResultSet *results = [[TMBUtilities_DataConversions_XLSToXML_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the XLSToXML Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_DataConversions_XLSToXML_Inputs*)newInputSet {
		return [[TMBUtilities_DataConversions_XLSToXML_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the XMLToCSV Choreo.
 */
@implementation TMBUtilities_DataConversions_XMLToCSV_Inputs

	/*!
	 * Set the value of the XML input for this Choreo.
	*(required, xml) The XML file to convert to CSV data.
	 */
	-(void)setXML:(NSString*)XML {
		[super setInput:@"XML" toValue:XML];
	}

	/*!
	 * Set the value of the Path input for this Choreo.
	*(optional, string) If your XML is not in "/rowset/row/column_name" format, specify a path to the rows. See documentation for examples.
	 */
	-(void)setPath:(NSString*)Path {
		[super setInput:@"Path" toValue:Path];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the XMLToCSV Choreo.
 */
@implementation TMBUtilities_DataConversions_XMLToCSV_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "CSV" output from an execution of this Choreo.
	 * @return - NSString* (multiline) The CSV data generated from the XML input.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getCSV {
		return [super getOutputByName:@"CSV"];
	}
	
@end

/*!
 * Converts an XML file to CSV format.
 */
@implementation TMBUtilities_DataConversions_XMLToCSV

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_DataConversions_XMLToCSV Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/DataConversions/XMLToCSV"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_DataConversions_XMLToCSV_ResultSet *results = [[TMBUtilities_DataConversions_XMLToCSV_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the XMLToCSV Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_DataConversions_XMLToCSV_Inputs*)newInputSet {
		return [[TMBUtilities_DataConversions_XMLToCSV_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the XMLToJSON Choreo.
 */
@implementation TMBUtilities_DataConversions_XMLToJSON_Inputs

	/*!
	 * Set the value of the XML input for this Choreo.
	*(required, xml) The XML file that you want to convert to JSON format.
	 */
	-(void)setXML:(NSString*)XML {
		[super setInput:@"XML" toValue:XML];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the XMLToJSON Choreo.
 */
@implementation TMBUtilities_DataConversions_XMLToJSON_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "JSON" output from an execution of this Choreo.
	 * @return - NSString* (json) The converted data in JSON format.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getJSON {
		return [super getOutputByName:@"JSON"];
	}
	
@end

/*!
 * Converts data from XML format to JSON format.
 */
@implementation TMBUtilities_DataConversions_XMLToJSON

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_DataConversions_XMLToJSON Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/DataConversions/XMLToJSON"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_DataConversions_XMLToJSON_ResultSet *results = [[TMBUtilities_DataConversions_XMLToJSON_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the XMLToJSON Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_DataConversions_XMLToJSON_Inputs*)newInputSet {
		return [[TMBUtilities_DataConversions_XMLToJSON_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the XMLToTSV Choreo.
 */
@implementation TMBUtilities_DataConversions_XMLToTSV_Inputs

	/*!
	 * Set the value of the XML input for this Choreo.
	*(required, xml) The XML file to convert to TSV data.
	 */
	-(void)setXML:(NSString*)XML {
		[super setInput:@"XML" toValue:XML];
	}

	/*!
	 * Set the value of the Path input for this Choreo.
	*(optional, string) If your XML is not in "/rowset/row/column_name" format, specify a path to the rows. See documentation for examples.
	 */
	-(void)setPath:(NSString*)Path {
		[super setInput:@"Path" toValue:Path];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the XMLToTSV Choreo.
 */
@implementation TMBUtilities_DataConversions_XMLToTSV_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "TSV" output from an execution of this Choreo.
	 * @return - NSString* (multiline) The tab-separated data generated from the XML input.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getTSV {
		return [super getOutputByName:@"TSV"];
	}
	
@end

/*!
 * Converts an XML file to TSV format (TAB-separated values).
 */
@implementation TMBUtilities_DataConversions_XMLToTSV

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_DataConversions_XMLToTSV Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/DataConversions/XMLToTSV"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_DataConversions_XMLToTSV_ResultSet *results = [[TMBUtilities_DataConversions_XMLToTSV_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the XMLToTSV Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_DataConversions_XMLToTSV_Inputs*)newInputSet {
		return [[TMBUtilities_DataConversions_XMLToTSV_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the XMLToXLS Choreo.
 */
@implementation TMBUtilities_DataConversions_XMLToXLS_Inputs

	/*!
	 * Set the value of the XML input for this Choreo.
	*(required, xml) The XML file you want to convert to XLS format. See documentation for information on the required XML schema.
	 */
	-(void)setXML:(NSString*)XML {
		[super setInput:@"XML" toValue:XML];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the XMLToXLS Choreo.
 */
@implementation TMBUtilities_DataConversions_XMLToXLS_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "XLS" output from an execution of this Choreo.
	 * @return - NSString* The Base64 encoded Excel data .
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getXLS {
		return [super getOutputByName:@"XLS"];
	}
	
@end

/*!
 * Converts an XML file to a Base64 encoded Excel file.
 */
@implementation TMBUtilities_DataConversions_XMLToXLS

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_DataConversions_XMLToXLS Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/DataConversions/XMLToXLS"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_DataConversions_XMLToXLS_ResultSet *results = [[TMBUtilities_DataConversions_XMLToXLS_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the XMLToXLS Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_DataConversions_XMLToXLS_Inputs*)newInputSet {
		return [[TMBUtilities_DataConversions_XMLToXLS_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetDate Choreo.
 */
@implementation TMBUtilities_Dates_GetDate_Inputs

	/*!
	 * Set the value of the AddDays input for this Choreo.
	*(optional, integer) Adds the specified number of days to the specified date serial number. A negative number will subtract.
	 */
	-(void)setAddDays:(NSString*)AddDays {
		[super setInput:@"AddDays" toValue:AddDays];
	}

	/*!
	 * Set the value of the AddHours input for this Choreo.
	*(optional, integer) Adds the specified number of hours to the specified date serial number. A negative number will subtract.
	 */
	-(void)setAddHours:(NSString*)AddHours {
		[super setInput:@"AddHours" toValue:AddHours];
	}

	/*!
	 * Set the value of the AddMinutes input for this Choreo.
	*(optional, integer) Adds the specified number of minutes to the specified date serial number. A negative number will subtract.
	 */
	-(void)setAddMinutes:(NSString*)AddMinutes {
		[super setInput:@"AddMinutes" toValue:AddMinutes];
	}

	/*!
	 * Set the value of the AddMonths input for this Choreo.
	*(optional, integer) Adds the specified number of months to the specified date serial number. A negative number will subtract.
	 */
	-(void)setAddMonths:(NSString*)AddMonths {
		[super setInput:@"AddMonths" toValue:AddMonths];
	}

	/*!
	 * Set the value of the AddSeconds input for this Choreo.
	*(optional, integer) Adds the specified number of seconds to the specified date serial number. A negative number will subtract.
	 */
	-(void)setAddSeconds:(NSString*)AddSeconds {
		[super setInput:@"AddSeconds" toValue:AddSeconds];
	}

	/*!
	 * Set the value of the AddYears input for this Choreo.
	*(optional, integer) Adds the specified number of years to the specified date serial number. A negative number will subtract.
	 */
	-(void)setAddYears:(NSString*)AddYears {
		[super setInput:@"AddYears" toValue:AddYears];
	}

	/*!
	 * Set the value of the Format input for this Choreo.
	*(conditional, string) The format that the timestamp should be in. Java SimpleDateFormat conventions are supported. Defaults to "yyyy-MM-dd HH:mm:ss".
	 */
	-(void)setFormat:(NSString*)Format {
		[super setInput:@"Format" toValue:Format];
	}

	/*!
	 * Set the value of the LocaleCountry input for this Choreo.
	*(optional, string) An ISO country code to specify locale.
	 */
	-(void)setLocaleCountry:(NSString*)LocaleCountry {
		[super setInput:@"LocaleCountry" toValue:LocaleCountry];
	}

	/*!
	 * Set the value of the LocaleLanguage input for this Choreo.
	*(optional, string) An ISO language code to specify locale.
	 */
	-(void)setLocaleLanguage:(NSString*)LocaleLanguage {
		[super setInput:@"LocaleLanguage" toValue:LocaleLanguage];
	}

	/*!
	 * Set the value of the LocaleVariant input for this Choreo.
	*(optional, string) A local variant code such as "NY" to add additional context for a locale.
	 */
	-(void)setLocaleVariant:(NSString*)LocaleVariant {
		[super setInput:@"LocaleVariant" toValue:LocaleVariant];
	}

	/*!
	 * Set the value of the SetDay input for this Choreo.
	*(optional, integer) Sets the day of month (1-31) of the specified date serial number.
	 */
	-(void)setSetDay:(NSString*)SetDay {
		[super setInput:@"SetDay" toValue:SetDay];
	}

	/*!
	 * Set the value of the SetHour input for this Choreo.
	*(optional, integer) Sets the hours (0-23) of the specified date serial number.
	 */
	-(void)setSetHour:(NSString*)SetHour {
		[super setInput:@"SetHour" toValue:SetHour];
	}

	/*!
	 * Set the value of the SetMinute input for this Choreo.
	*(optional, integer) Sets the minutes (0-59) of the specified date serial number.
	 */
	-(void)setSetMinute:(NSString*)SetMinute {
		[super setInput:@"SetMinute" toValue:SetMinute];
	}

	/*!
	 * Set the value of the SetMonth input for this Choreo.
	*(optional, integer) Sets the month (1-12) of the specified date serial number.
	 */
	-(void)setSetMonth:(NSString*)SetMonth {
		[super setInput:@"SetMonth" toValue:SetMonth];
	}

	/*!
	 * Set the value of the SetSecond input for this Choreo.
	*(optional, integer) Sets the seconds (0-59) of the specified date serial number.
	 */
	-(void)setSetSecond:(NSString*)SetSecond {
		[super setInput:@"SetSecond" toValue:SetSecond];
	}

	/*!
	 * Set the value of the SetYear input for this Choreo.
	*(optional, integer) Sets the year (such as 1989) of the specified date serial number.
	 */
	-(void)setSetYear:(NSString*)SetYear {
		[super setInput:@"SetYear" toValue:SetYear];
	}

	/*!
	 * Set the value of the TimeZone input for this Choreo.
	*(optional, string) The timezone to use for the date formatting function. Defaults to UTC.
	 */
	-(void)setTimeZone:(NSString*)TimeZone {
		[super setInput:@"TimeZone" toValue:TimeZone];
	}

	/*!
	 * Set the value of the Timestamp input for this Choreo.
	*(conditional, date) A number representing the desired formatted date and time, expressed as the number of milliseconds since January 1, 1970 (epoch time). If not provided, this defaults to NOW().
	 */
	-(void)setTimestamp:(NSString*)Timestamp {
		[super setInput:@"Timestamp" toValue:Timestamp];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetDate Choreo.
 */
@implementation TMBUtilities_Dates_GetDate_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "FormattedDate" output from an execution of this Choreo.
	 * @return - NSString* (date) The formatted version of the timestamp.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getFormattedDate {
		return [super getOutputByName:@"FormattedDate"];
	}
	
@end

/*!
 * Formats a specified timestamp, or generates the current date in a desired format.
 */
@implementation TMBUtilities_Dates_GetDate

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Dates_GetDate Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Dates/GetDate"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Dates_GetDate_ResultSet *results = [[TMBUtilities_Dates_GetDate_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetDate Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Dates_GetDate_Inputs*)newInputSet {
		return [[TMBUtilities_Dates_GetDate_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetDateDifference Choreo.
 */
@implementation TMBUtilities_Dates_GetDateDifference_Inputs

	/*!
	 * Set the value of the EarlierDate input for this Choreo.
	*(required, date) The earlier date to use for the date comparision (e.g., March 2, 2014).
	 */
	-(void)setEarlierDate:(NSString*)EarlierDate {
		[super setInput:@"EarlierDate" toValue:EarlierDate];
	}

	/*!
	 * Set the value of the LaterDate input for this Choreo.
	*(required, date) The later date to use for the date comparision (e.g., March 3, 2014).
	 */
	-(void)setLaterDate:(NSString*)LaterDate {
		[super setInput:@"LaterDate" toValue:LaterDate];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetDateDifference Choreo.
 */
@implementation TMBUtilities_Dates_GetDateDifference_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Difference" output from an execution of this Choreo.
	 * @return - NSString* (integer) The difference between two specified dates, expressed as the number of milliseconds since January 1, 1970 (epoch time). 
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getDifference {
		return [super getOutputByName:@"Difference"];
	}
	
@end

/*!
 * Returns the difference between two specified dates, expressed as the number of milliseconds since January 1, 1970 (epoch time).
 */
@implementation TMBUtilities_Dates_GetDateDifference

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Dates_GetDateDifference Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Dates/GetDateDifference"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Dates_GetDateDifference_ResultSet *results = [[TMBUtilities_Dates_GetDateDifference_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetDateDifference Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Dates_GetDateDifference_Inputs*)newInputSet {
		return [[TMBUtilities_Dates_GetDateDifference_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetTimestamp Choreo.
 */
@implementation TMBUtilities_Dates_GetTimestamp_Inputs

	/*!
	 * Set the value of the AddDays input for this Choreo.
	*(optional, integer) Adds the specified number of days to the specified date serial number. A negative number will subtract.
	 */
	-(void)setAddDays:(NSString*)AddDays {
		[super setInput:@"AddDays" toValue:AddDays];
	}

	/*!
	 * Set the value of the AddHours input for this Choreo.
	*(optional, integer) Adds the specified number of hours to the specified date serial number. A negative number will subtract.
	 */
	-(void)setAddHours:(NSString*)AddHours {
		[super setInput:@"AddHours" toValue:AddHours];
	}

	/*!
	 * Set the value of the AddMinutes input for this Choreo.
	*(optional, integer) Adds the specified number of minutes to the specified date serial number. A negative number will subtract.
	 */
	-(void)setAddMinutes:(NSString*)AddMinutes {
		[super setInput:@"AddMinutes" toValue:AddMinutes];
	}

	/*!
	 * Set the value of the AddMonths input for this Choreo.
	*(optional, integer) Adds the specified number of months to the specified date serial number. A negative number will subtract.
	 */
	-(void)setAddMonths:(NSString*)AddMonths {
		[super setInput:@"AddMonths" toValue:AddMonths];
	}

	/*!
	 * Set the value of the AddSeconds input for this Choreo.
	*(optional, integer) Adds the specified number of seconds to the specified date serial number. A negative number will subtract.
	 */
	-(void)setAddSeconds:(NSString*)AddSeconds {
		[super setInput:@"AddSeconds" toValue:AddSeconds];
	}

	/*!
	 * Set the value of the AddYears input for this Choreo.
	*(optional, integer) Adds the specified number of years to the specified date serial number. A negative number will subtract.
	 */
	-(void)setAddYears:(NSString*)AddYears {
		[super setInput:@"AddYears" toValue:AddYears];
	}

	/*!
	 * Set the value of the Granularity input for this Choreo.
	*(optional, string) Set to "seconds" to return the number of seconds since the epoch. Defaults to "milliseconds".
	 */
	-(void)setGranularity:(NSString*)Granularity {
		[super setInput:@"Granularity" toValue:Granularity];
	}

	/*!
	 * Set the value of the SetDay input for this Choreo.
	*(optional, integer) Sets the day of month (1-31) of the specified date serial number.
	 */
	-(void)setSetDay:(NSString*)SetDay {
		[super setInput:@"SetDay" toValue:SetDay];
	}

	/*!
	 * Set the value of the SetHour input for this Choreo.
	*(optional, integer) Sets the hours (0-23) of the specified date serial number.
	 */
	-(void)setSetHour:(NSString*)SetHour {
		[super setInput:@"SetHour" toValue:SetHour];
	}

	/*!
	 * Set the value of the SetMinute input for this Choreo.
	*(optional, integer) Sets the minutes (0-59) of the specified date serial number.
	 */
	-(void)setSetMinute:(NSString*)SetMinute {
		[super setInput:@"SetMinute" toValue:SetMinute];
	}

	/*!
	 * Set the value of the SetMonth input for this Choreo.
	*(optional, integer) Sets the month (1-12) of the specified date serial number.
	 */
	-(void)setSetMonth:(NSString*)SetMonth {
		[super setInput:@"SetMonth" toValue:SetMonth];
	}

	/*!
	 * Set the value of the SetSecond input for this Choreo.
	*(optional, integer) Sets the seconds (0-59) of the specified date serial number.
	 */
	-(void)setSetSecond:(NSString*)SetSecond {
		[super setInput:@"SetSecond" toValue:SetSecond];
	}

	/*!
	 * Set the value of the SetYear input for this Choreo.
	*(optional, integer) Sets the year (such as 1989) of the specified date serial number.
	 */
	-(void)setSetYear:(NSString*)SetYear {
		[super setInput:@"SetYear" toValue:SetYear];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetTimestamp Choreo.
 */
@implementation TMBUtilities_Dates_GetTimestamp_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Timestamp" output from an execution of this Choreo.
	 * @return - NSString* (date) A the current timestamp, expressed as the number of seconds or milliseconds since January 1, 1970 (epoch time). The Granularity input is used to indicate seconds or milliseconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getTimestamp {
		return [super getOutputByName:@"Timestamp"];
	}
	
@end

/*!
 * Returns the current date and time, expressed as seconds or milliseconds since January 1, 1970 (epoch time).
 */
@implementation TMBUtilities_Dates_GetTimestamp

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Dates_GetTimestamp Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Dates/GetTimestamp"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Dates_GetTimestamp_ResultSet *results = [[TMBUtilities_Dates_GetTimestamp_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetTimestamp Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Dates_GetTimestamp_Inputs*)newInputSet {
		return [[TMBUtilities_Dates_GetTimestamp_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetTimestampFromDateParameters Choreo.
 */
@implementation TMBUtilities_Dates_GetTimestampFromDateParameters_Inputs

	/*!
	 * Set the value of the Day input for this Choreo.
	*(conditional, integer) Sets the day (1-31) of the timestamp.
	 */
	-(void)setDay:(NSString*)Day {
		[super setInput:@"Day" toValue:Day];
	}

	/*!
	 * Set the value of the Granularity input for this Choreo.
	*(optional, string) Set to "seconds" to return the number of seconds since the epoch. Defaults to "milliseconds".
	 */
	-(void)setGranularity:(NSString*)Granularity {
		[super setInput:@"Granularity" toValue:Granularity];
	}

	/*!
	 * Set the value of the Hour input for this Choreo.
	*(optional, integer) Sets the hours (0-23) of the timestamp.
	 */
	-(void)setHour:(NSString*)Hour {
		[super setInput:@"Hour" toValue:Hour];
	}

	/*!
	 * Set the value of the Milliseconds input for this Choreo.
	*(optional, integer) Sets the milliseconds (0-999) of the timestamp.
	 */
	-(void)setMilliseconds:(NSString*)Milliseconds {
		[super setInput:@"Milliseconds" toValue:Milliseconds];
	}

	/*!
	 * Set the value of the Minute input for this Choreo.
	*(optional, integer) Sets the minutes (0-59) of the timestamp.
	 */
	-(void)setMinute:(NSString*)Minute {
		[super setInput:@"Minute" toValue:Minute];
	}

	/*!
	 * Set the value of the Month input for this Choreo.
	*(conditional, integer) Sets the month (1-12) of the timestamp.
	 */
	-(void)setMonth:(NSString*)Month {
		[super setInput:@"Month" toValue:Month];
	}

	/*!
	 * Set the value of the Second input for this Choreo.
	*(optional, integer) Sets the seconds (0-59) of the timestamp.
	 */
	-(void)setSecond:(NSString*)Second {
		[super setInput:@"Second" toValue:Second];
	}

	/*!
	 * Set the value of the Year input for this Choreo.
	*(conditional, integer) Sets the year of the timestamp.
	 */
	-(void)setYear:(NSString*)Year {
		[super setInput:@"Year" toValue:Year];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetTimestampFromDateParameters Choreo.
 */
@implementation TMBUtilities_Dates_GetTimestampFromDateParameters_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Timestamp" output from an execution of this Choreo.
	 * @return - NSString* (date) A number representing the specified date and time, expressed as the number of seconds or milliseconds since January 1, 1970. The Granularity input is used to indicate seconds or milliseconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getTimestamp {
		return [super getOutputByName:@"Timestamp"];
	}
	
@end

/*!
 * Returns the specified date parameters, expressed as the number of seconds or milliseconds since January 1, 1970 (epoch time).
 */
@implementation TMBUtilities_Dates_GetTimestampFromDateParameters

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Dates_GetTimestampFromDateParameters Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Dates/GetTimestampFromDateParameters"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Dates_GetTimestampFromDateParameters_ResultSet *results = [[TMBUtilities_Dates_GetTimestampFromDateParameters_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetTimestampFromDateParameters Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Dates_GetTimestampFromDateParameters_Inputs*)newInputSet {
		return [[TMBUtilities_Dates_GetTimestampFromDateParameters_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetTimestampFromDateString Choreo.
 */
@implementation TMBUtilities_Dates_GetTimestampFromDateString_Inputs

	/*!
	 * Set the value of the AddDays input for this Choreo.
	*(optional, integer) Adds the specified number of days to the specified date serial number. A negative number will subtract.
	 */
	-(void)setAddDays:(NSString*)AddDays {
		[super setInput:@"AddDays" toValue:AddDays];
	}

	/*!
	 * Set the value of the AddHours input for this Choreo.
	*(optional, integer) Adds the specified number of hours to the specified date serial number. A negative number will subtract.
	 */
	-(void)setAddHours:(NSString*)AddHours {
		[super setInput:@"AddHours" toValue:AddHours];
	}

	/*!
	 * Set the value of the AddMinutes input for this Choreo.
	*(optional, integer) Adds the specified number of minutes to the specified date serial number. A negative number will subtract.
	 */
	-(void)setAddMinutes:(NSString*)AddMinutes {
		[super setInput:@"AddMinutes" toValue:AddMinutes];
	}

	/*!
	 * Set the value of the AddMonths input for this Choreo.
	*(optional, integer) Adds the specified number of months to the specified date serial number. A negative number will subtract.
	 */
	-(void)setAddMonths:(NSString*)AddMonths {
		[super setInput:@"AddMonths" toValue:AddMonths];
	}

	/*!
	 * Set the value of the AddSeconds input for this Choreo.
	*(optional, integer) Adds the specified number of seconds to the specified date serial number. A negative number will subtract.
	 */
	-(void)setAddSeconds:(NSString*)AddSeconds {
		[super setInput:@"AddSeconds" toValue:AddSeconds];
	}

	/*!
	 * Set the value of the AddYears input for this Choreo.
	*(optional, integer) Adds the specified number of years to the specified date serial number. A negative number will subtract.
	 */
	-(void)setAddYears:(NSString*)AddYears {
		[super setInput:@"AddYears" toValue:AddYears];
	}

	/*!
	 * Set the value of the Date input for this Choreo.
	*(required, string) A date string to convert to a timestamp (e.g., March 2, 2014).
	 */
	-(void)setDate:(NSString*)Date {
		[super setInput:@"Date" toValue:Date];
	}

	/*!
	 * Set the value of the Granularity input for this Choreo.
	*(optional, string) Set to "seconds" to return the number of seconds since the epoch. Defaults to "milliseconds".
	 */
	-(void)setGranularity:(NSString*)Granularity {
		[super setInput:@"Granularity" toValue:Granularity];
	}

	/*!
	 * Set the value of the SetDay input for this Choreo.
	*(optional, integer) Sets the day of month (1-31) of the specified date serial number.
	 */
	-(void)setSetDay:(NSString*)SetDay {
		[super setInput:@"SetDay" toValue:SetDay];
	}

	/*!
	 * Set the value of the SetHour input for this Choreo.
	*(optional, integer) Sets the hours (0-23) of the specified date serial number.
	 */
	-(void)setSetHour:(NSString*)SetHour {
		[super setInput:@"SetHour" toValue:SetHour];
	}

	/*!
	 * Set the value of the SetMinute input for this Choreo.
	*(optional, integer) Sets the minutes (0-59) of the specified date serial number.
	 */
	-(void)setSetMinute:(NSString*)SetMinute {
		[super setInput:@"SetMinute" toValue:SetMinute];
	}

	/*!
	 * Set the value of the SetMonth input for this Choreo.
	*(optional, integer) Sets the month (1-12) of the specified date serial number.
	 */
	-(void)setSetMonth:(NSString*)SetMonth {
		[super setInput:@"SetMonth" toValue:SetMonth];
	}

	/*!
	 * Set the value of the SetSecond input for this Choreo.
	*(optional, integer) Sets the seconds (0-59) of the specified date serial number.
	 */
	-(void)setSetSecond:(NSString*)SetSecond {
		[super setInput:@"SetSecond" toValue:SetSecond];
	}

	/*!
	 * Set the value of the SetYear input for this Choreo.
	*(optional, integer) Sets the year (such as 1989) of the specified date serial number.
	 */
	-(void)setSetYear:(NSString*)SetYear {
		[super setInput:@"SetYear" toValue:SetYear];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetTimestampFromDateString Choreo.
 */
@implementation TMBUtilities_Dates_GetTimestampFromDateString_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Timestamp" output from an execution of this Choreo.
	 * @return - NSString* (date) A the current timestamp, expressed as the number of seconds or milliseconds since January 1, 1970 (epoch time). The Granularity input is used to indicate seconds or milliseconds.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getTimestamp {
		return [super getOutputByName:@"Timestamp"];
	}
	
@end

/*!
 * Returns the the specified date string, expressed as seconds or milliseconds since January 1, 1970 (epoch time).
 */
@implementation TMBUtilities_Dates_GetTimestampFromDateString

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Dates_GetTimestampFromDateString Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Dates/GetTimestampFromDateString"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Dates_GetTimestampFromDateString_ResultSet *results = [[TMBUtilities_Dates_GetTimestampFromDateString_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetTimestampFromDateString Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Dates_GetTimestampFromDateString_Inputs*)newInputSet {
		return [[TMBUtilities_Dates_GetTimestampFromDateString_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SendEmail Choreo.
 */
@implementation TMBUtilities_Email_SendEmail_Inputs

	/*!
	 * Set the value of the Attachment input for this Choreo.
	*(optional, string) The Base64 encoded contents of the file to attach to the email.  Use this instead of AttachmentURL.
	 */
	-(void)setAttachment:(NSString*)Attachment {
		[super setInput:@"Attachment" toValue:Attachment];
	}

	/*!
	 * Set the value of the AttachmentName input for this Choreo.
	*(optional, string) The name of the file to attach to the email.
	 */
	-(void)setAttachmentName:(NSString*)AttachmentName {
		[super setInput:@"AttachmentName" toValue:AttachmentName];
	}

	/*!
	 * Set the value of the AttachmentURL input for this Choreo.
	*(optional, string) URL of a hosted file that you wish to add as an attachment.  Use this instead of a normal Attachment.
	 */
	-(void)setAttachmentURL:(NSString*)AttachmentURL {
		[super setInput:@"AttachmentURL" toValue:AttachmentURL];
	}

	/*!
	 * Set the value of the BCC input for this Choreo.
	*(optional, string) An email address to BCC on the email you're sending. Can be a comma separated list of email addresses.
	 */
	-(void)setBCC:(NSString*)BCC {
		[super setInput:@"BCC" toValue:BCC];
	}

	/*!
	 * Set the value of the CC input for this Choreo.
	*(optional, string) An email address to CC on the email you're sending. Can be a comma separated list of email addresses.
	 */
	-(void)setCC:(NSString*)CC {
		[super setInput:@"CC" toValue:CC];
	}

	/*!
	 * Set the value of the FromAddress input for this Choreo.
	*(conditional, string) The name and email address that the message is being sent from.
	 */
	-(void)setFromAddress:(NSString*)FromAddress {
		[super setInput:@"FromAddress" toValue:FromAddress];
	}

	/*!
	 * Set the value of the MessageBody input for this Choreo.
	*(required, string) The message body for the email.
	 */
	-(void)setMessageBody:(NSString*)MessageBody {
		[super setInput:@"MessageBody" toValue:MessageBody];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, password) The password for your email account.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the Port input for this Choreo.
	*(required, integer) Specify the port number (i.e. 25 or 465).
	 */
	-(void)setPort:(NSString*)Port {
		[super setInput:@"Port" toValue:Port];
	}

	/*!
	 * Set the value of the Server input for this Choreo.
	*(required, string) The name or IP address of the email server.
	 */
	-(void)setServer:(NSString*)Server {
		[super setInput:@"Server" toValue:Server];
	}

	/*!
	 * Set the value of the Subject input for this Choreo.
	*(required, string) The subject line of the email.
	 */
	-(void)setSubject:(NSString*)Subject {
		[super setInput:@"Subject" toValue:Subject];
	}

	/*!
	 * Set the value of the ToAddress input for this Choreo.
	*(required, string) The email address that you want to send an email to. Can be a comma separated list of email addresses.
	 */
	-(void)setToAddress:(NSString*)ToAddress {
		[super setInput:@"ToAddress" toValue:ToAddress];
	}

	/*!
	 * Set the value of the UseSSL input for this Choreo.
	*(optional, boolean) Set to 1 to connect over SSL. Set to 0 for no SSL. Defaults to 1.
	 */
	-(void)setUseSSL:(NSString*)UseSSL {
		[super setInput:@"UseSSL" toValue:UseSSL];
	}

	/*!
	 * Set the value of the Username input for this Choreo.
	*(required, string) Your username for your email account.
	 */
	-(void)setUsername:(NSString*)Username {
		[super setInput:@"Username" toValue:Username];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SendEmail Choreo.
 */
@implementation TMBUtilities_Email_SendEmail_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Success" output from an execution of this Choreo.
	 * @return - NSString* (boolean) Indicates the result of the SMTP operation. The value will be "true" for a successful request.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getSuccess {
		return [super getOutputByName:@"Success"];
	}
	
@end

/*!
 * Sends an email using a specified email server.
 */
@implementation TMBUtilities_Email_SendEmail

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Email_SendEmail Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Email/SendEmail"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Email_SendEmail_ResultSet *results = [[TMBUtilities_Email_SendEmail_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SendEmail Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Email_SendEmail_Inputs*)newInputSet {
		return [[TMBUtilities_Email_SendEmail_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Base64Decode Choreo.
 */
@implementation TMBUtilities_Encoding_Base64Decode_Inputs

	/*!
	 * Set the value of the Base64EncodedText input for this Choreo.
	*(required, string) The Base64 encoded text to decode. Note that Base64 encoded binary data is not allowed.
	 */
	-(void)setBase64EncodedText:(NSString*)Base64EncodedText {
		[super setInput:@"Base64EncodedText" toValue:Base64EncodedText];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Base64Decode Choreo.
 */
@implementation TMBUtilities_Encoding_Base64Decode_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Text" output from an execution of this Choreo.
	 * @return - NSString* (string) The decoded text.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getText {
		return [super getOutputByName:@"Text"];
	}
	
@end

/*!
 * Returns the specified Base64 encoded string as decoded text.
 */
@implementation TMBUtilities_Encoding_Base64Decode

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Encoding_Base64Decode Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Encoding/Base64Decode"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Encoding_Base64Decode_ResultSet *results = [[TMBUtilities_Encoding_Base64Decode_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Base64Decode Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Encoding_Base64Decode_Inputs*)newInputSet {
		return [[TMBUtilities_Encoding_Base64Decode_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Base64Encode Choreo.
 */
@implementation TMBUtilities_Encoding_Base64Encode_Inputs

	/*!
	 * Set the value of the Text input for this Choreo.
	*(conditional, string) The text that should be Base64 encoded. Required unless providing a value for the URL input.
	 */
	-(void)setText:(NSString*)Text {
		[super setInput:@"Text" toValue:Text];
	}

	/*!
	 * Set the value of the URL input for this Choreo.
	*(conditional, string) A URL to a hosted file that should be Base64 encoded. Required unless providing a value for the Text input.
	 */
	-(void)setURL:(NSString*)URL {
		[super setInput:@"URL" toValue:URL];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Base64Encode Choreo.
 */
@implementation TMBUtilities_Encoding_Base64Encode_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Base64EncodedText" output from an execution of this Choreo.
	 * @return - NSString* (string) The Base64 encoded text.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getBase64EncodedText {
		return [super getOutputByName:@"Base64EncodedText"];
	}
	
@end

/*!
 * Returns the specified text or file as a Base64 encoded string.
 */
@implementation TMBUtilities_Encoding_Base64Encode

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Encoding_Base64Encode Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Encoding/Base64Encode"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Encoding_Base64Encode_ResultSet *results = [[TMBUtilities_Encoding_Base64Encode_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Base64Encode Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Encoding_Base64Encode_Inputs*)newInputSet {
		return [[TMBUtilities_Encoding_Base64Encode_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the HTMLEscape Choreo.
 */
@implementation TMBUtilities_Encoding_HTMLEscape_Inputs

	/*!
	 * Set the value of the UnescapedHTML input for this Choreo.
	*(required, string) The HTML that needs to be escaped.
	 */
	-(void)setUnescapedHTML:(NSString*)UnescapedHTML {
		[super setInput:@"UnescapedHTML" toValue:UnescapedHTML];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the HTMLEscape Choreo.
 */
@implementation TMBUtilities_Encoding_HTMLEscape_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "EscapedHTML" output from an execution of this Choreo.
	 * @return - NSString* (string) The escaped HTML.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getEscapedHTML {
		return [super getOutputByName:@"EscapedHTML"];
	}
	
@end

/*!
 * Replaces HTML markup characters in the specified text with equivalent character entity names. 
 */
@implementation TMBUtilities_Encoding_HTMLEscape

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Encoding_HTMLEscape Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Encoding/HTMLEscape"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Encoding_HTMLEscape_ResultSet *results = [[TMBUtilities_Encoding_HTMLEscape_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the HTMLEscape Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Encoding_HTMLEscape_Inputs*)newInputSet {
		return [[TMBUtilities_Encoding_HTMLEscape_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the HTMLUnescape Choreo.
 */
@implementation TMBUtilities_Encoding_HTMLUnescape_Inputs

	/*!
	 * Set the value of the EscapedHTML input for this Choreo.
	*(required, string) The escaped HTML that should be unescaped.
	 */
	-(void)setEscapedHTML:(NSString*)EscapedHTML {
		[super setInput:@"EscapedHTML" toValue:EscapedHTML];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the HTMLUnescape Choreo.
 */
@implementation TMBUtilities_Encoding_HTMLUnescape_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "UnescapedHTML" output from an execution of this Choreo.
	 * @return - NSString* (string) The unescaped HTML.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getUnescapedHTML {
		return [super getOutputByName:@"UnescapedHTML"];
	}
	
@end

/*!
 * Replaces character entity names in the specified text with equivalent HTML markup characters.
 */
@implementation TMBUtilities_Encoding_HTMLUnescape

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Encoding_HTMLUnescape Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Encoding/HTMLUnescape"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Encoding_HTMLUnescape_ResultSet *results = [[TMBUtilities_Encoding_HTMLUnescape_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the HTMLUnescape Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Encoding_HTMLUnescape_Inputs*)newInputSet {
		return [[TMBUtilities_Encoding_HTMLUnescape_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the URLDecode Choreo.
 */
@implementation TMBUtilities_Encoding_URLDecode_Inputs

	/*!
	 * Set the value of the Text input for this Choreo.
	*(required, string) The text that should be URL decoded.
	 */
	-(void)setText:(NSString*)Text {
		[super setInput:@"Text" toValue:Text];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the URLDecode Choreo.
 */
@implementation TMBUtilities_Encoding_URLDecode_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "URLDecodedText" output from an execution of this Choreo.
	 * @return - NSString* (string) The URL decoded text.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getURLDecodedText {
		return [super getOutputByName:@"URLDecodedText"];
	}
	
@end

/*!
 * Removes URL encoding from the specified text string.
 */
@implementation TMBUtilities_Encoding_URLDecode

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Encoding_URLDecode Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Encoding/URLDecode"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Encoding_URLDecode_ResultSet *results = [[TMBUtilities_Encoding_URLDecode_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the URLDecode Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Encoding_URLDecode_Inputs*)newInputSet {
		return [[TMBUtilities_Encoding_URLDecode_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the URLEncode Choreo.
 */
@implementation TMBUtilities_Encoding_URLEncode_Inputs

	/*!
	 * Set the value of the Text input for this Choreo.
	*(required, string) The text that should be URL encoded.
	 */
	-(void)setText:(NSString*)Text {
		[super setInput:@"Text" toValue:Text];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the URLEncode Choreo.
 */
@implementation TMBUtilities_Encoding_URLEncode_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "URLEncodedText" output from an execution of this Choreo.
	 * @return - NSString* (string) The URL encoded text.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getURLEncodedText {
		return [super getOutputByName:@"URLEncodedText"];
	}
	
@end

/*!
 * Returns the specified text string in the application/x-www-form-urlencoded format.
 */
@implementation TMBUtilities_Encoding_URLEncode

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Encoding_URLEncode Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Encoding/URLEncode"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Encoding_URLEncode_ResultSet *results = [[TMBUtilities_Encoding_URLEncode_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the URLEncode Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Encoding_URLEncode_Inputs*)newInputSet {
		return [[TMBUtilities_Encoding_URLEncode_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the FindSubstring Choreo.
 */
@implementation TMBUtilities_Finding_FindSubstring_Inputs

	/*!
	 * Set the value of the CaseSensitive input for this Choreo.
	*(optional, boolean) When set to true, the search will be case-sensitive. Defaults to false indicating a case-insensitive search.
	 */
	-(void)setCaseSensitive:(NSString*)CaseSensitive {
		[super setInput:@"CaseSensitive" toValue:CaseSensitive];
	}

	/*!
	 * Set the value of the StartNumber input for this Choreo.
	*(optional, string) The character position at which to begin the search. Defaults to 1.
	 */
	-(void)setStartNumber:(NSString*)StartNumber {
		[super setInput:@"StartNumber" toValue:StartNumber];
	}

	/*!
	 * Set the value of the Substring input for this Choreo.
	*(required, string) The sub-string to search within the specified text (searching from left to right).
	 */
	-(void)setSubstring:(NSString*)Substring {
		[super setInput:@"Substring" toValue:Substring];
	}

	/*!
	 * Set the value of the Text input for this Choreo.
	*(required, string) The text to search for a sub-string.
	 */
	-(void)setText:(NSString*)Text {
		[super setInput:@"Text" toValue:Text];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the FindSubstring Choreo.
 */
@implementation TMBUtilities_Finding_FindSubstring_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Count" output from an execution of this Choreo.
	 * @return - NSString* (integer) The count of sub-strings found.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getCount {
		return [super getOutputByName:@"Count"];
	}

	/*!
	 * Retrieve the value of the "Positions" output from an execution of this Choreo.
	 * @return - NSString* (json) The positions of the sub-strings that were found in the search.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getPositions {
		return [super getOutputByName:@"Positions"];
	}
	
@end

/*!
 * Finds all occurrences of a specified substring and returns the substring positions as a JSON array.
 */
@implementation TMBUtilities_Finding_FindSubstring

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Finding_FindSubstring Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Finding/FindSubstring"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Finding_FindSubstring_ResultSet *results = [[TMBUtilities_Finding_FindSubstring_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the FindSubstring Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Finding_FindSubstring_Inputs*)newInputSet {
		return [[TMBUtilities_Finding_FindSubstring_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the RegexMatch Choreo.
 */
@implementation TMBUtilities_Finding_RegexMatch_Inputs

	/*!
	 * Set the value of the Pattern input for this Choreo.
	*(conditional, string) A regular expression.
	 */
	-(void)setPattern:(NSString*)Pattern {
		[super setInput:@"Pattern" toValue:Pattern];
	}

	/*!
	 * Set the value of the Text input for this Choreo.
	*(conditional, string) The text on which to perform a regex match.
	 */
	-(void)setText:(NSString*)Text {
		[super setInput:@"Text" toValue:Text];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the RegexMatch Choreo.
 */
@implementation TMBUtilities_Finding_RegexMatch_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Result" output from an execution of this Choreo.
	 * @return - NSString* (string) The result of the match.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResult {
		return [super getOutputByName:@"Result"];
	}
	
@end

/*!
 * Returns the first substring that matches the specified regular expression pattern in the specified string.
 */
@implementation TMBUtilities_Finding_RegexMatch

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Finding_RegexMatch Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Finding/RegexMatch"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Finding_RegexMatch_ResultSet *results = [[TMBUtilities_Finding_RegexMatch_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the RegexMatch Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Finding_RegexMatch_Inputs*)newInputSet {
		return [[TMBUtilities_Finding_RegexMatch_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the RemoveWhiteSpace Choreo.
 */
@implementation TMBUtilities_Formatting_RemoveWhiteSpace_Inputs

	/*!
	 * Set the value of the FormattedText input for this Choreo.
	*(required, multiline) The formatted text that should have line breaks and tabs removed.
	 */
	-(void)setFormattedText:(NSString*)FormattedText {
		[super setInput:@"FormattedText" toValue:FormattedText];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the RemoveWhiteSpace Choreo.
 */
@implementation TMBUtilities_Formatting_RemoveWhiteSpace_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "CompactText" output from an execution of this Choreo.
	 * @return - NSString* (string) 
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getCompactText {
		return [super getOutputByName:@"CompactText"];
	}
	
@end

/*!
 * Returns the specified formatted text as a compact string with no new lines, tabs, or preceding/trailing white space.
 */
@implementation TMBUtilities_Formatting_RemoveWhiteSpace

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Formatting_RemoveWhiteSpace Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Formatting/RemoveWhiteSpace"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Formatting_RemoveWhiteSpace_ResultSet *results = [[TMBUtilities_Formatting_RemoveWhiteSpace_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the RemoveWhiteSpace Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Formatting_RemoveWhiteSpace_Inputs*)newInputSet {
		return [[TMBUtilities_Formatting_RemoveWhiteSpace_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Delete Choreo.
 */
@implementation TMBUtilities_HTTP_Delete_Inputs

	/*!
	 * Set the value of the Debug input for this Choreo.
	*(optional, boolean) When set to "true", the HTTP debug log will be returned.
	 */
	-(void)setDebug:(NSString*)Debug {
		[super setInput:@"Debug" toValue:Debug];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(optional, password) A valid password. This is used if the request required basic authentication.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the RequestHeaders input for this Choreo.
	*(optional, json) A JSON object containing up to 10 key/value pairs that will be mapped to the HTTP request headers.
	 */
	-(void)setRequestHeaders:(NSString*)RequestHeaders {
		[super setInput:@"RequestHeaders" toValue:RequestHeaders];
	}

	/*!
	 * Set the value of the RequestParameters input for this Choreo.
	*(optional, json) A JSON object containing up to 10 key/value pairs that will be mapped to the url string as HTTP parameters.
	 */
	-(void)setRequestParameters:(NSString*)RequestParameters {
		[super setInput:@"RequestParameters" toValue:RequestParameters];
	}

	/*!
	 * Set the value of the URL input for this Choreo.
	*(required, string) The base URL for the request (including http:// or https://).
	 */
	-(void)setURL:(NSString*)URL {
		[super setInput:@"URL" toValue:URL];
	}

	/*!
	 * Set the value of the Username input for this Choreo.
	*(optional, string) A valid username. This is used if the request required basic authentication.
	 */
	-(void)setUsername:(NSString*)Username {
		[super setInput:@"Username" toValue:Username];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Delete Choreo.
 */
@implementation TMBUtilities_HTTP_Delete_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "HTTPLog" output from an execution of this Choreo.
	 * @return - NSString* (string) A debug log for the http request that was sent. This is only returned when Debug is set to "true".
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getHTTPLog {
		return [super getOutputByName:@"HTTPLog"];
	}

	/*!
	 * Retrieve the value of the "ResponseStatusCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) The response status code.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponseStatusCode {
		return [super getOutputByName:@"ResponseStatusCode"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from the server.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Generates a HTTP DELETE request.
 */
@implementation TMBUtilities_HTTP_Delete

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_HTTP_Delete Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/HTTP/Delete"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_HTTP_Delete_ResultSet *results = [[TMBUtilities_HTTP_Delete_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Delete Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_HTTP_Delete_Inputs*)newInputSet {
		return [[TMBUtilities_HTTP_Delete_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Get Choreo.
 */
@implementation TMBUtilities_HTTP_Get_Inputs

	/*!
	 * Set the value of the Debug input for this Choreo.
	*(optional, boolean) When set to "true", the HTTP debug log will be returned.
	 */
	-(void)setDebug:(NSString*)Debug {
		[super setInput:@"Debug" toValue:Debug];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(optional, password) A valid password. This is used if the request required basic authentication.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the RequestHeaders input for this Choreo.
	*(optional, json) A JSON object containing up to 10 key/value pairs that will be mapped to the HTTP request headers.
	 */
	-(void)setRequestHeaders:(NSString*)RequestHeaders {
		[super setInput:@"RequestHeaders" toValue:RequestHeaders];
	}

	/*!
	 * Set the value of the RequestParameters input for this Choreo.
	*(optional, json) A JSON object containing up to 10 key/value pairs that will be mapped to the url string as http parameters.
	 */
	-(void)setRequestParameters:(NSString*)RequestParameters {
		[super setInput:@"RequestParameters" toValue:RequestParameters];
	}

	/*!
	 * Set the value of the URL input for this Choreo.
	*(required, string) The base URL for the request (including http:// or https://).
	 */
	-(void)setURL:(NSString*)URL {
		[super setInput:@"URL" toValue:URL];
	}

	/*!
	 * Set the value of the Username input for this Choreo.
	*(optional, string) A valid username. This is used if the request required basic authentication.
	 */
	-(void)setUsername:(NSString*)Username {
		[super setInput:@"Username" toValue:Username];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Get Choreo.
 */
@implementation TMBUtilities_HTTP_Get_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "HTTPLog" output from an execution of this Choreo.
	 * @return - NSString* (string) A debug log for the http request that was sent. This is only returned when Debug is set to "true".
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getHTTPLog {
		return [super getOutputByName:@"HTTPLog"];
	}

	/*!
	 * Retrieve the value of the "ResponseStatusCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) The response status code.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponseStatusCode {
		return [super getOutputByName:@"ResponseStatusCode"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from the server.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Generates a HTTP GET request.
 */
@implementation TMBUtilities_HTTP_Get

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_HTTP_Get Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/HTTP/Get"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_HTTP_Get_ResultSet *results = [[TMBUtilities_HTTP_Get_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Get Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_HTTP_Get_Inputs*)newInputSet {
		return [[TMBUtilities_HTTP_Get_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Post Choreo.
 */
@implementation TMBUtilities_HTTP_Post_Inputs

	/*!
	 * Set the value of the RequestBody input for this Choreo.
	*(optional, multiline) The request body for the POST request.
	 */
	-(void)setRequestBody:(NSString*)RequestBody {
		[super setInput:@"RequestBody" toValue:RequestBody];
	}

	/*!
	 * Set the value of the Debug input for this Choreo.
	*(optional, boolean) When set to "true", the HTTP debug log will be returned.
	 */
	-(void)setDebug:(NSString*)Debug {
		[super setInput:@"Debug" toValue:Debug];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(optional, password) A valid password. This is used if the request required basic authentication.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the RequestHeaders input for this Choreo.
	*(optional, json) A JSON object containing up to 10 key/value pairs that will be mapped to the HTTP request headers.
	 */
	-(void)setRequestHeaders:(NSString*)RequestHeaders {
		[super setInput:@"RequestHeaders" toValue:RequestHeaders];
	}

	/*!
	 * Set the value of the RequestParameters input for this Choreo.
	*(optional, json) A JSON object containing up to 10 key/value pairs that will be mapped to the url string as http parameters.
	 */
	-(void)setRequestParameters:(NSString*)RequestParameters {
		[super setInput:@"RequestParameters" toValue:RequestParameters];
	}

	/*!
	 * Set the value of the URL input for this Choreo.
	*(required, string) The base URL for the request (including http:// or https://).
	 */
	-(void)setURL:(NSString*)URL {
		[super setInput:@"URL" toValue:URL];
	}

	/*!
	 * Set the value of the Username input for this Choreo.
	*(optional, string) A valid username. This is used if the request required basic authentication.
	 */
	-(void)setUsername:(NSString*)Username {
		[super setInput:@"Username" toValue:Username];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Post Choreo.
 */
@implementation TMBUtilities_HTTP_Post_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "HTTPLog" output from an execution of this Choreo.
	 * @return - NSString* (string) A debug log for the http request that was sent. This is only returned when Debug is set to "true".
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getHTTPLog {
		return [super getOutputByName:@"HTTPLog"];
	}

	/*!
	 * Retrieve the value of the "ResponseStatusCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) The response status code.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponseStatusCode {
		return [super getOutputByName:@"ResponseStatusCode"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from the server.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Generates a HTTP POST request.
 */
@implementation TMBUtilities_HTTP_Post

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_HTTP_Post Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/HTTP/Post"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_HTTP_Post_ResultSet *results = [[TMBUtilities_HTTP_Post_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Post Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_HTTP_Post_Inputs*)newInputSet {
		return [[TMBUtilities_HTTP_Post_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Put Choreo.
 */
@implementation TMBUtilities_HTTP_Put_Inputs

	/*!
	 * Set the value of the RequestBody input for this Choreo.
	*(optional, multiline) The request body for the PUT request.
	 */
	-(void)setRequestBody:(NSString*)RequestBody {
		[super setInput:@"RequestBody" toValue:RequestBody];
	}

	/*!
	 * Set the value of the Debug input for this Choreo.
	*(optional, boolean) When set to "true", the HTTP debug log will be returned.
	 */
	-(void)setDebug:(NSString*)Debug {
		[super setInput:@"Debug" toValue:Debug];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(optional, password) A valid password. This is used if the request required basic authentication.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}

	/*!
	 * Set the value of the RequestHeaders input for this Choreo.
	*(optional, json) A JSON object containing up to 10 key/value pairs that will be mapped to the HTTP request headers.
	 */
	-(void)setRequestHeaders:(NSString*)RequestHeaders {
		[super setInput:@"RequestHeaders" toValue:RequestHeaders];
	}

	/*!
	 * Set the value of the RequestParameters input for this Choreo.
	*(optional, json) A JSON object containing up to 10 key/value pairs that will be mapped to the url string as HTTP parameters.
	 */
	-(void)setRequestParameters:(NSString*)RequestParameters {
		[super setInput:@"RequestParameters" toValue:RequestParameters];
	}

	/*!
	 * Set the value of the URL input for this Choreo.
	*(required, string) The base URL for the request (including http:// or https://).
	 */
	-(void)setURL:(NSString*)URL {
		[super setInput:@"URL" toValue:URL];
	}

	/*!
	 * Set the value of the Username input for this Choreo.
	*(optional, string) A valid username. This is used if the request required basic authentication.
	 */
	-(void)setUsername:(NSString*)Username {
		[super setInput:@"Username" toValue:Username];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Put Choreo.
 */
@implementation TMBUtilities_HTTP_Put_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "HTTPLog" output from an execution of this Choreo.
	 * @return - NSString* (string) A debug log for the http request that was sent. This is only returned when Debug is set to "true".
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getHTTPLog {
		return [super getOutputByName:@"HTTPLog"];
	}

	/*!
	 * Retrieve the value of the "ResponseStatusCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) The response status code.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponseStatusCode {
		return [super getOutputByName:@"ResponseStatusCode"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from the server.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Generates a HTTP PUT request.
 */
@implementation TMBUtilities_HTTP_Put

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_HTTP_Put Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/HTTP/Put"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_HTTP_Put_ResultSet *results = [[TMBUtilities_HTTP_Put_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Put Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_HTTP_Put_Inputs*)newInputSet {
		return [[TMBUtilities_HTTP_Put_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Checksum Choreo.
 */
@implementation TMBUtilities_Hashing_Checksum_Inputs

	/*!
	 * Set the value of the Algorithm input for this Choreo.
	*(required, string) Algorithm used to calculate the checksum. Valid values are: FIX44,  MD5+BASE64, or MD5+HEX (the default). MD5+BASE64 and MD5+HEX return the result in Base64 and hexadecimal encoding, respectively.
	 */
	-(void)setAlgorithm:(NSString*)Algorithm {
		[super setInput:@"Algorithm" toValue:Algorithm];
	}

	/*!
	 * Set the value of the Base64DecodeValue input for this Choreo.
	*(optional, boolean) Setting this parameter to 1 indicates that the Text is Base64 encoded, and that the choreo should decode the value before calculating the checksum. Defaults to 0.
	 */
	-(void)setBase64DecodeValue:(NSString*)Base64DecodeValue {
		[super setInput:@"Base64DecodeValue" toValue:Base64DecodeValue];
	}

	/*!
	 * Set the value of the Text input for this Choreo.
	*(required, string) The text to return a checksum for.
	 */
	-(void)setText:(NSString*)Text {
		[super setInput:@"Text" toValue:Text];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Checksum Choreo.
 */
@implementation TMBUtilities_Hashing_Checksum_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Checksum" output from an execution of this Choreo.
	 * @return - NSString* (string) The checksum result.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getChecksum {
		return [super getOutputByName:@"Checksum"];
	}
	
@end

/*!
 * Returns a checksum of the specified text calculated using the specified algorithm. 
 */
@implementation TMBUtilities_Hashing_Checksum

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Hashing_Checksum Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Hashing/Checksum"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Hashing_Checksum_ResultSet *results = [[TMBUtilities_Hashing_Checksum_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Checksum Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Hashing_Checksum_Inputs*)newInputSet {
		return [[TMBUtilities_Hashing_Checksum_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the HmacSHA1 Choreo.
 */
@implementation TMBUtilities_Hashing_HmacSHA1_Inputs

	/*!
	 * Set the value of the Key input for this Choreo.
	*(required, string) The secret key used to generate the SHA1-encrypted HMAC signature.
	 */
	-(void)setKey:(NSString*)Key {
		[super setInput:@"Key" toValue:Key];
	}

	/*!
	 * Set the value of the Text input for this Choreo.
	*(required, string) The text that should be SHA1-encrypted.
	 */
	-(void)setText:(NSString*)Text {
		[super setInput:@"Text" toValue:Text];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the HmacSHA1 Choreo.
 */
@implementation TMBUtilities_Hashing_HmacSHA1_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "HmacSignature" output from an execution of this Choreo.
	 * @return - NSString* (string) The HMAC Signature for the specified string.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getHmacSignature {
		return [super getOutputByName:@"HmacSignature"];
	}
	
@end

/*!
 * Generates a SHA1-encrypted HMAC signature for the specified message text using the specified secret key.
 */
@implementation TMBUtilities_Hashing_HmacSHA1

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Hashing_HmacSHA1 Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Hashing/HmacSHA1"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Hashing_HmacSHA1_ResultSet *results = [[TMBUtilities_Hashing_HmacSHA1_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the HmacSHA1 Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Hashing_HmacSHA1_Inputs*)newInputSet {
		return [[TMBUtilities_Hashing_HmacSHA1_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the HmacSHA256 Choreo.
 */
@implementation TMBUtilities_Hashing_HmacSHA256_Inputs

	/*!
	 * Set the value of the Key input for this Choreo.
	*(required, string) The secret key used to generate the SHA256-encrypted HMAC signature.
	 */
	-(void)setKey:(NSString*)Key {
		[super setInput:@"Key" toValue:Key];
	}

	/*!
	 * Set the value of the Text input for this Choreo.
	*(required, string) The text that should be SHA256-encrypted.
	 */
	-(void)setText:(NSString*)Text {
		[super setInput:@"Text" toValue:Text];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the HmacSHA256 Choreo.
 */
@implementation TMBUtilities_Hashing_HmacSHA256_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "HmacSignature" output from an execution of this Choreo.
	 * @return - NSString* (string) The HMAC Signature for the specified string.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getHmacSignature {
		return [super getOutputByName:@"HmacSignature"];
	}
	
@end

/*!
 * Generates a SHA256-encrypted HMAC signature for the specified message text using the specified secret key.
 */
@implementation TMBUtilities_Hashing_HmacSHA256

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Hashing_HmacSHA256 Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Hashing/HmacSHA256"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Hashing_HmacSHA256_ResultSet *results = [[TMBUtilities_Hashing_HmacSHA256_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the HmacSHA256 Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Hashing_HmacSHA256_Inputs*)newInputSet {
		return [[TMBUtilities_Hashing_HmacSHA256_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetValuesFromJSON Choreo.
 */
@implementation TMBUtilities_JSON_GetValuesFromJSON_Inputs

	/*!
	 * Set the value of the JSON input for this Choreo.
	*(required, json) JSON String to search.
	 */
	-(void)setJSON:(NSString*)JSON {
		[super setInput:@"JSON" toValue:JSON];
	}

	/*!
	 * Set the value of the Property input for this Choreo.
	*(required, string) Property to match in the specified JSON string.
	 */
	-(void)setProperty:(NSString*)Property {
		[super setInput:@"Property" toValue:Property];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetValuesFromJSON Choreo.
 */
@implementation TMBUtilities_JSON_GetValuesFromJSON_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) Value(s) that match the property.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Searches for the specified property in the supplied JSON string.
 */
@implementation TMBUtilities_JSON_GetValuesFromJSON

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_JSON_GetValuesFromJSON Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/JSON/GetValuesFromJSON"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_JSON_GetValuesFromJSON_ResultSet *results = [[TMBUtilities_JSON_GetValuesFromJSON_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetValuesFromJSON Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_JSON_GetValuesFromJSON_Inputs*)newInputSet {
		return [[TMBUtilities_JSON_GetValuesFromJSON_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GenerateRandom Choreo.
 */
@implementation TMBUtilities_Numbers_GenerateRandom_Inputs

	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GenerateRandom Choreo.
 */
@implementation TMBUtilities_Numbers_GenerateRandom_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "SignedDecimal" output from an execution of this Choreo.
	 * @return - NSString* (decimal) Signed Decimal in the range of  -0.5 to +0.5.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getSignedDecimal {
		return [super getOutputByName:@"SignedDecimal"];
	}

	/*!
	 * Retrieve the value of the "SignedInteger" output from an execution of this Choreo.
	 * @return - NSString* (integer) SIgned Integer in the range of -2147483648 through 2147483647.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getSignedInteger {
		return [super getOutputByName:@"SignedInteger"];
	}

	/*!
	 * Retrieve the value of the "UnsignedDecimal" output from an execution of this Choreo.
	 * @return - NSString* (decimal) Unsigned Decimal in the range of 0.0 to 1.0.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getUnsignedDecimal {
		return [super getOutputByName:@"UnsignedDecimal"];
	}

	/*!
	 * Retrieve the value of the "UnsignedInteger" output from an execution of this Choreo.
	 * @return - NSString* (integer) Unsigned integer in the range of 0 through 4294967295.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getUnsignedInteger {
		return [super getOutputByName:@"UnsignedInteger"];
	}
	
@end

/*!
 * This choreo generates a random number in a variety of ranges. 
 */
@implementation TMBUtilities_Numbers_GenerateRandom

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Numbers_GenerateRandom Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Numbers/GenerateRandom"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Numbers_GenerateRandom_ResultSet *results = [[TMBUtilities_Numbers_GenerateRandom_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GenerateRandom Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Numbers_GenerateRandom_Inputs*)newInputSet {
		return [[TMBUtilities_Numbers_GenerateRandom_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the HelloWorld Choreo.
 */
@implementation TMBUtilities_Test_HelloWorld_Inputs

	/*!
	 * Set the value of the Value input for this Choreo.
	*(conditional, string) An optional test value to pass into the result message.
	 */
	-(void)setValue:(NSString*)Value {
		[super setInput:@"Value" toValue:Value];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the HelloWorld Choreo.
 */
@implementation TMBUtilities_Test_HelloWorld_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Result" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a sample Choreo output. If no input is provided, the result will be "Hello, world!". When passing an input value, the result will be "Hello, {Value}!".
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResult {
		return [super getOutputByName:@"Result"];
	}
	
@end

/*!
 * Allows you to run a simple test that outputs "Hello, world!" when executed.
 */
@implementation TMBUtilities_Test_HelloWorld

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Test_HelloWorld Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Test/HelloWorld"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Test_HelloWorld_ResultSet *results = [[TMBUtilities_Test_HelloWorld_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the HelloWorld Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Test_HelloWorld_Inputs*)newInputSet {
		return [[TMBUtilities_Test_HelloWorld_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GenerateGUID Choreo.
 */
@implementation TMBUtilities_Text_GenerateGUID_Inputs

	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GenerateGUID Choreo.
 */
@implementation TMBUtilities_Text_GenerateGUID_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "GUID" output from an execution of this Choreo.
	 * @return - NSString* (string) The generated GUID.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getGUID {
		return [super getOutputByName:@"GUID"];
	}
	
@end

/*!
 * Obtain a unique, randomly generated ID or GUID (Globally Unique Identifier).
 */
@implementation TMBUtilities_Text_GenerateGUID

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Text_GenerateGUID Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Text/GenerateGUID"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Text_GenerateGUID_ResultSet *results = [[TMBUtilities_Text_GenerateGUID_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GenerateGUID Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Text_GenerateGUID_Inputs*)newInputSet {
		return [[TMBUtilities_Text_GenerateGUID_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Lowercase Choreo.
 */
@implementation TMBUtilities_Text_Lowercase_Inputs

	/*!
	 * Set the value of the Text input for this Choreo.
	*(required, multiline) The text that should be converted to lowercase.
	 */
	-(void)setText:(NSString*)Text {
		[super setInput:@"Text" toValue:Text];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Lowercase Choreo.
 */
@implementation TMBUtilities_Text_Lowercase_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (string) The converted lowercase text.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Returns the contents of the specified string converted to lowercase.
 */
@implementation TMBUtilities_Text_Lowercase

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Text_Lowercase Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Text/Lowercase"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Text_Lowercase_ResultSet *results = [[TMBUtilities_Text_Lowercase_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Lowercase Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Text_Lowercase_Inputs*)newInputSet {
		return [[TMBUtilities_Text_Lowercase_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Split Choreo.
 */
@implementation TMBUtilities_Text_Split_Inputs

	/*!
	 * Set the value of the Delimiter input for this Choreo.
	*(required, string) The delimiter to search for when splitting the string into sub-strings. See Choreo notes for restrictions with delimiters.
	 */
	-(void)setDelimiter:(NSString*)Delimiter {
		[super setInput:@"Delimiter" toValue:Delimiter];
	}

	/*!
	 * Set the value of the Text input for this Choreo.
	*(required, multiline) The text that should be split into sub-strings.
	 */
	-(void)setText:(NSString*)Text {
		[super setInput:@"Text" toValue:Text];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Split Choreo.
 */
@implementation TMBUtilities_Text_Split_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains the sub-strings formatted as a JSON array.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Splits a string into sub-strings delimited by the specified delmiter pattern.
 */
@implementation TMBUtilities_Text_Split

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Text_Split Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Text/Split"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Text_Split_ResultSet *results = [[TMBUtilities_Text_Split_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Split Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Text_Split_Inputs*)newInputSet {
		return [[TMBUtilities_Text_Split_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Substitute Choreo.
 */
@implementation TMBUtilities_Text_Substitute_Inputs

	/*!
	 * Set the value of the New input for this Choreo.
	*(required, string) New sub-string to replace with.
	 */
	-(void)setNew:(NSString*)New {
		[super setInput:@"New" toValue:New];
	}

	/*!
	 * Set the value of the Old input for this Choreo.
	*(required, string) Old sub-string to replace.
	 */
	-(void)setOld:(NSString*)Old {
		[super setInput:@"Old" toValue:Old];
	}

	/*!
	 * Set the value of the Text input for this Choreo.
	*(required, string) Text to peform substitution.
	 */
	-(void)setText:(NSString*)Text {
		[super setInput:@"Text" toValue:Text];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Substitute Choreo.
 */
@implementation TMBUtilities_Text_Substitute_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (string) The result after the substitution.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Replaces all instances of the specified sub-string within the specified text with a new sub-string. 
 */
@implementation TMBUtilities_Text_Substitute

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Text_Substitute Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Text/Substitute"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Text_Substitute_ResultSet *results = [[TMBUtilities_Text_Substitute_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Substitute Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Text_Substitute_Inputs*)newInputSet {
		return [[TMBUtilities_Text_Substitute_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SubstituteRegex Choreo.
 */
@implementation TMBUtilities_Text_SubstituteRegex_Inputs

	/*!
	 * Set the value of the New input for this Choreo.
	*(required, string) New sub-string to replace with.
	 */
	-(void)setNew:(NSString*)New {
		[super setInput:@"New" toValue:New];
	}

	/*!
	 * Set the value of the Pattern input for this Choreo.
	*(required, string) Regex pattern to use.
	 */
	-(void)setPattern:(NSString*)Pattern {
		[super setInput:@"Pattern" toValue:Pattern];
	}

	/*!
	 * Set the value of the Text input for this Choreo.
	*(required, string) Text to peform substitution.
	 */
	-(void)setText:(NSString*)Text {
		[super setInput:@"Text" toValue:Text];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SubstituteRegex Choreo.
 */
@implementation TMBUtilities_Text_SubstituteRegex_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (string) The result after the substitution.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Finds all instances of the specified regular expression pattern within the given string and passes the specified new sub-string to the result variable. 
 */
@implementation TMBUtilities_Text_SubstituteRegex

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Text_SubstituteRegex Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Text/SubstituteRegex"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Text_SubstituteRegex_ResultSet *results = [[TMBUtilities_Text_SubstituteRegex_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SubstituteRegex Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Text_SubstituteRegex_Inputs*)newInputSet {
		return [[TMBUtilities_Text_SubstituteRegex_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Uppercase Choreo.
 */
@implementation TMBUtilities_Text_Uppercase_Inputs

	/*!
	 * Set the value of the Text input for this Choreo.
	*(required, multiline) The text that should be converted to uppercase.
	 */
	-(void)setText:(NSString*)Text {
		[super setInput:@"Text" toValue:Text];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Uppercase Choreo.
 */
@implementation TMBUtilities_Text_Uppercase_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (string) The converted uppercase text.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Returns the contents of the specified string converted to uppercase.
 */
@implementation TMBUtilities_Text_Uppercase

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Text_Uppercase Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Text/Uppercase"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Text_Uppercase_ResultSet *results = [[TMBUtilities_Text_Uppercase_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Uppercase Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Text_Uppercase_Inputs*)newInputSet {
		return [[TMBUtilities_Text_Uppercase_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteToken Choreo.
 */
@implementation TMBUtilities_TokenStorage_DeleteToken_Inputs

	/*!
	 * Set the value of the Name input for this Choreo.
	*(required, string) The name of the token to delete.
	 */
	-(void)setName:(NSString*)Name {
		[super setInput:@"Name" toValue:Name];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteToken Choreo.
 */
@implementation TMBUtilities_TokenStorage_DeleteToken_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Deleted" output from an execution of this Choreo.
	 * @return - NSString* (boolean) Returns true when token is successfully deleted.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getDeleted {
		return [super getOutputByName:@"Deleted"];
	}
	
@end

/*!
 * Deletes a specified token.
 */
@implementation TMBUtilities_TokenStorage_DeleteToken

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_TokenStorage_DeleteToken Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/TokenStorage/DeleteToken"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_TokenStorage_DeleteToken_ResultSet *results = [[TMBUtilities_TokenStorage_DeleteToken_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteToken Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_TokenStorage_DeleteToken_Inputs*)newInputSet {
		return [[TMBUtilities_TokenStorage_DeleteToken_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetTokenDetails Choreo.
 */
@implementation TMBUtilities_TokenStorage_GetTokenDetails_Inputs

	/*!
	 * Set the value of the Names input for this Choreo.
	*(required, json) A list of tokens to return. This should be formated as a JSON array.
	 */
	-(void)setNames:(NSString*)Names {
		[super setInput:@"Names" toValue:Names];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetTokenDetails Choreo.
 */
@implementation TMBUtilities_TokenStorage_GetTokenDetails_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Tokens" output from an execution of this Choreo.
	 * @return - NSString* (json) The token values.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getTokens {
		return [super getOutputByName:@"Tokens"];
	}
	
@end

/*!
 * Returns one or more tokens represented by a specified list of names.
 */
@implementation TMBUtilities_TokenStorage_GetTokenDetails

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_TokenStorage_GetTokenDetails Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/TokenStorage/GetTokenDetails"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_TokenStorage_GetTokenDetails_ResultSet *results = [[TMBUtilities_TokenStorage_GetTokenDetails_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetTokenDetails Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_TokenStorage_GetTokenDetails_Inputs*)newInputSet {
		return [[TMBUtilities_TokenStorage_GetTokenDetails_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the IsLocked Choreo.
 */
@implementation TMBUtilities_TokenStorage_IsLocked_Inputs

	/*!
	 * Set the value of the Name input for this Choreo.
	*(required, string) The name of the token to check.
	 */
	-(void)setName:(NSString*)Name {
		[super setInput:@"Name" toValue:Name];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the IsLocked Choreo.
 */
@implementation TMBUtilities_TokenStorage_IsLocked_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Locked" output from an execution of this Choreo.
	 * @return - NSString* (boolean) Returns true or false depending on whether the token is locked or not.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLocked {
		return [super getOutputByName:@"Locked"];
	}
	
@end

/*!
 * Returns true or false depending on whether the token is locked or not.
 */
@implementation TMBUtilities_TokenStorage_IsLocked

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_TokenStorage_IsLocked Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/TokenStorage/IsLocked"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_TokenStorage_IsLocked_ResultSet *results = [[TMBUtilities_TokenStorage_IsLocked_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the IsLocked Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_TokenStorage_IsLocked_Inputs*)newInputSet {
		return [[TMBUtilities_TokenStorage_IsLocked_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the IsValid Choreo.
 */
@implementation TMBUtilities_TokenStorage_IsValid_Inputs

	/*!
	 * Set the value of the Name input for this Choreo.
	*(required, string) The name of the token to check.
	 */
	-(void)setName:(NSString*)Name {
		[super setInput:@"Name" toValue:Name];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the IsValid Choreo.
 */
@implementation TMBUtilities_TokenStorage_IsValid_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Valid" output from an execution of this Choreo.
	 * @return - NSString* (boolean) Returns true or false depending on whether the token is valid or not.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getValid {
		return [super getOutputByName:@"Valid"];
	}
	
@end

/*!
 * Returns true or false depending on whether the token is valid or not.
 */
@implementation TMBUtilities_TokenStorage_IsValid

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_TokenStorage_IsValid Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/TokenStorage/IsValid"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_TokenStorage_IsValid_ResultSet *results = [[TMBUtilities_TokenStorage_IsValid_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the IsValid Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_TokenStorage_IsValid_Inputs*)newInputSet {
		return [[TMBUtilities_TokenStorage_IsValid_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the LockToken Choreo.
 */
@implementation TMBUtilities_TokenStorage_LockToken_Inputs

	/*!
	 * Set the value of the Name input for this Choreo.
	*(required, string) The name of the token to lock.
	 */
	-(void)setName:(NSString*)Name {
		[super setInput:@"Name" toValue:Name];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the LockToken Choreo.
 */
@implementation TMBUtilities_TokenStorage_LockToken_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Token" output from an execution of this Choreo.
	 * @return - NSString* (string) When a lock has been obtained, the token value will be returned. If a lock can not be obtained, and empty string is returned.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getToken {
		return [super getOutputByName:@"Token"];
	}
	
@end

/*!
 * Locks a specified token.
 */
@implementation TMBUtilities_TokenStorage_LockToken

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_TokenStorage_LockToken Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/TokenStorage/LockToken"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_TokenStorage_LockToken_ResultSet *results = [[TMBUtilities_TokenStorage_LockToken_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the LockToken Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_TokenStorage_LockToken_Inputs*)newInputSet {
		return [[TMBUtilities_TokenStorage_LockToken_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the RetrieveToken Choreo.
 */
@implementation TMBUtilities_TokenStorage_RetrieveToken_Inputs

	/*!
	 * Set the value of the LockToken input for this Choreo.
	*(optional, boolean) If set to true, the Choreo will attempt to lock the token after retrieving it. If the token is already locked, the Choreo will attempt to get the lock for up-to 1 minute.
	 */
	-(void)setLockToken:(NSString*)LockToken {
		[super setInput:@"LockToken" toValue:LockToken];
	}

	/*!
	 * Set the value of the Name input for this Choreo.
	*(required, string) The name of the token to retrieve.
	 */
	-(void)setName:(NSString*)Name {
		[super setInput:@"Name" toValue:Name];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the RetrieveToken Choreo.
 */
@implementation TMBUtilities_TokenStorage_RetrieveToken_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Token" output from an execution of this Choreo.
	 * @return - NSString* (string) The token value. This will return an empty string if there is no token or if the token has expired.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getToken {
		return [super getOutputByName:@"Token"];
	}

	/*!
	 * Retrieve the value of the "Locked" output from an execution of this Choreo.
	 * @return - NSString* (boolean) Returns true or false depending on whether the token is locked or not.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLocked {
		return [super getOutputByName:@"Locked"];
	}

	/*!
	 * Retrieve the value of the "Valid" output from an execution of this Choreo.
	 * @return - NSString* (boolean) Returns true or false depending on whether the token is valid or not.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getValid {
		return [super getOutputByName:@"Valid"];
	}
	
@end

/*!
 * Retrieves a specified token.
 */
@implementation TMBUtilities_TokenStorage_RetrieveToken

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_TokenStorage_RetrieveToken Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/TokenStorage/RetrieveToken"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_TokenStorage_RetrieveToken_ResultSet *results = [[TMBUtilities_TokenStorage_RetrieveToken_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the RetrieveToken Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_TokenStorage_RetrieveToken_Inputs*)newInputSet {
		return [[TMBUtilities_TokenStorage_RetrieveToken_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SetValid Choreo.
 */
@implementation TMBUtilities_TokenStorage_SetValid_Inputs

	/*!
	 * Set the value of the Name input for this Choreo.
	*(required, string) The name of the token to modify.
	 */
	-(void)setName:(NSString*)Name {
		[super setInput:@"Name" toValue:Name];
	}

	/*!
	 * Set the value of the Valid input for this Choreo.
	*(conditional, boolean) Can be set to true or false depending on whether the token is valid or not.
	 */
	-(void)setValid:(NSString*)Valid {
		[super setInput:@"Valid" toValue:Valid];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SetValid Choreo.
 */
@implementation TMBUtilities_TokenStorage_SetValid_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Updated" output from an execution of this Choreo.
	 * @return - NSString* (boolean) Returns true if the operation was successful.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getUpdated {
		return [super getOutputByName:@"Updated"];
	}
	
@end

/*!
 * Sets a specified token as valid or invalid.
 */
@implementation TMBUtilities_TokenStorage_SetValid

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_TokenStorage_SetValid Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/TokenStorage/SetValid"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_TokenStorage_SetValid_ResultSet *results = [[TMBUtilities_TokenStorage_SetValid_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SetValid Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_TokenStorage_SetValid_Inputs*)newInputSet {
		return [[TMBUtilities_TokenStorage_SetValid_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the StoreToken Choreo.
 */
@implementation TMBUtilities_TokenStorage_StoreToken_Inputs

	/*!
	 * Set the value of the Expires input for this Choreo.
	*(optional, integer) The lifetime of the token (in seconds). Defaults to 0 indicating no expiration.
	 */
	-(void)setExpires:(NSString*)Expires {
		[super setInput:@"Expires" toValue:Expires];
	}

	/*!
	 * Set the value of the Name input for this Choreo.
	*(required, string) The token name. When a token does not exist, it will be inserted. When a token does exist, an update is performed.
	 */
	-(void)setName:(NSString*)Name {
		[super setInput:@"Name" toValue:Name];
	}

	/*!
	 * Set the value of the Value input for this Choreo.
	*(required, string) The token value to store. The maximum number of characters for a token is 4096.
	 */
	-(void)setValue:(NSString*)Value {
		[super setInput:@"Value" toValue:Value];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the StoreToken Choreo.
 */
@implementation TMBUtilities_TokenStorage_StoreToken_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Updated" output from an execution of this Choreo.
	 * @return - NSString* (boolean) Returns true if token is stored successfully.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getUpdated {
		return [super getOutputByName:@"Updated"];
	}
	
@end

/*!
 * Stores a token.
 */
@implementation TMBUtilities_TokenStorage_StoreToken

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_TokenStorage_StoreToken Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/TokenStorage/StoreToken"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_TokenStorage_StoreToken_ResultSet *results = [[TMBUtilities_TokenStorage_StoreToken_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the StoreToken Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_TokenStorage_StoreToken_Inputs*)newInputSet {
		return [[TMBUtilities_TokenStorage_StoreToken_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CanadianPostalCodes Choreo.
 */
@implementation TMBUtilities_Validation_CanadianPostalCodes_Inputs

	/*!
	 * Set the value of the ZipCode input for this Choreo.
	*(required, string) The zip code to validate.
	 */
	-(void)setZipCode:(NSString*)ZipCode {
		[super setInput:@"ZipCode" toValue:ZipCode];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CanadianPostalCodes Choreo.
 */
@implementation TMBUtilities_Validation_CanadianPostalCodes_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Match" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a string indicating the result of the match -- "valid" or "invalid".
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getMatch {
		return [super getOutputByName:@"Match"];
	}
	
@end

/*!
 * Verifies that a given zip code matches the format expected for Canadian addresses.
 */
@implementation TMBUtilities_Validation_CanadianPostalCodes

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Validation_CanadianPostalCodes Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Validation/CanadianPostalCodes"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Validation_CanadianPostalCodes_ResultSet *results = [[TMBUtilities_Validation_CanadianPostalCodes_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CanadianPostalCodes Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Validation_CanadianPostalCodes_Inputs*)newInputSet {
		return [[TMBUtilities_Validation_CanadianPostalCodes_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DutchPostalCodes Choreo.
 */
@implementation TMBUtilities_Validation_DutchPostalCodes_Inputs

	/*!
	 * Set the value of the ZipCode input for this Choreo.
	*(required, string) The zip code to validate.
	 */
	-(void)setZipCode:(NSString*)ZipCode {
		[super setInput:@"ZipCode" toValue:ZipCode];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DutchPostalCodes Choreo.
 */
@implementation TMBUtilities_Validation_DutchPostalCodes_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Match" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a string indicating the result of the match -- "valid" or "invalid".
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getMatch {
		return [super getOutputByName:@"Match"];
	}
	
@end

/*!
 * Verifies that a given zip code matches the format expected for Dutch addresses.
 */
@implementation TMBUtilities_Validation_DutchPostalCodes

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Validation_DutchPostalCodes Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Validation/DutchPostalCodes"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Validation_DutchPostalCodes_ResultSet *results = [[TMBUtilities_Validation_DutchPostalCodes_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DutchPostalCodes Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Validation_DutchPostalCodes_Inputs*)newInputSet {
		return [[TMBUtilities_Validation_DutchPostalCodes_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the EmailAddress Choreo.
 */
@implementation TMBUtilities_Validation_EmailAddress_Inputs

	/*!
	 * Set the value of the EmailAddress input for this Choreo.
	*(required, string) The email address to validate.
	 */
	-(void)setEmailAddress:(NSString*)EmailAddress {
		[super setInput:@"EmailAddress" toValue:EmailAddress];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the EmailAddress Choreo.
 */
@implementation TMBUtilities_Validation_EmailAddress_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Match" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a string indicating the result of the match -- "valid" or "invalid".
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getMatch {
		return [super getOutputByName:@"Match"];
	}
	
@end

/*!
 * Verifies that a given email address matches an expected standard pattern.
 */
@implementation TMBUtilities_Validation_EmailAddress

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Validation_EmailAddress Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Validation/EmailAddress"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Validation_EmailAddress_ResultSet *results = [[TMBUtilities_Validation_EmailAddress_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the EmailAddress Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Validation_EmailAddress_Inputs*)newInputSet {
		return [[TMBUtilities_Validation_EmailAddress_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GermanPostalCodes Choreo.
 */
@implementation TMBUtilities_Validation_GermanPostalCodes_Inputs

	/*!
	 * Set the value of the ZipCode input for this Choreo.
	*(required, string) The zip code to validate.
	 */
	-(void)setZipCode:(NSString*)ZipCode {
		[super setInput:@"ZipCode" toValue:ZipCode];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GermanPostalCodes Choreo.
 */
@implementation TMBUtilities_Validation_GermanPostalCodes_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Match" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a string indicating the result of the match -- "valid" or "invalid".
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getMatch {
		return [super getOutputByName:@"Match"];
	}
	
@end

/*!
 * Verifies that a given zip code matches the format expected for German addresses.
 */
@implementation TMBUtilities_Validation_GermanPostalCodes

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Validation_GermanPostalCodes Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Validation/GermanPostalCodes"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Validation_GermanPostalCodes_ResultSet *results = [[TMBUtilities_Validation_GermanPostalCodes_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GermanPostalCodes Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Validation_GermanPostalCodes_Inputs*)newInputSet {
		return [[TMBUtilities_Validation_GermanPostalCodes_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the JSON Choreo.
 */
@implementation TMBUtilities_Validation_JSON_Inputs

	/*!
	 * Set the value of the JSON input for this Choreo.
	*(required, multiline) The JSON string to validate.
	 */
	-(void)setJSON:(NSString*)JSON {
		[super setInput:@"JSON" toValue:JSON];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the JSON Choreo.
 */
@implementation TMBUtilities_Validation_JSON_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Result" output from an execution of this Choreo.
	 * @return - NSString* (string) The result of the JSON validation. This will return "valid" or "invalid".
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResult {
		return [super getOutputByName:@"Result"];
	}
	
@end

/*!
 * Determines if a specified JSON string is well-formed.
 */
@implementation TMBUtilities_Validation_JSON

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Validation_JSON Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Validation/JSON"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Validation_JSON_ResultSet *results = [[TMBUtilities_Validation_JSON_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the JSON Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Validation_JSON_Inputs*)newInputSet {
		return [[TMBUtilities_Validation_JSON_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the PasswordCriteria Choreo.
 */
@implementation TMBUtilities_Validation_PasswordCriteria_Inputs

	/*!
	 * Set the value of the MaxLength input for this Choreo.
	*(optional, integer) The max length you want to allow for the password. Defaults to 14.
	 */
	-(void)setMaxLength:(NSString*)MaxLength {
		[super setInput:@"MaxLength" toValue:MaxLength];
	}

	/*!
	 * Set the value of the MinLength input for this Choreo.
	*(optional, integer) The minimum length you want to allow for the password. Defaults to 6.
	 */
	-(void)setMinLength:(NSString*)MinLength {
		[super setInput:@"MinLength" toValue:MinLength];
	}

	/*!
	 * Set the value of the Password input for this Choreo.
	*(required, string) The password to validate.
	 */
	-(void)setPassword:(NSString*)Password {
		[super setInput:@"Password" toValue:Password];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the PasswordCriteria Choreo.
 */
@implementation TMBUtilities_Validation_PasswordCriteria_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Match" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a string indicating the result of the match -- "valid" or "invalid".
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getMatch {
		return [super getOutputByName:@"Match"];
	}
	
@end

/*!
 * Verifies that a given password matches a standard pattern for passwords.
 */
@implementation TMBUtilities_Validation_PasswordCriteria

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Validation_PasswordCriteria Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Validation/PasswordCriteria"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Validation_PasswordCriteria_ResultSet *results = [[TMBUtilities_Validation_PasswordCriteria_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the PasswordCriteria Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Validation_PasswordCriteria_Inputs*)newInputSet {
		return [[TMBUtilities_Validation_PasswordCriteria_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the RussianPostalCodes Choreo.
 */
@implementation TMBUtilities_Validation_RussianPostalCodes_Inputs

	/*!
	 * Set the value of the ZipCode input for this Choreo.
	*(required, string) The zip code to validate.
	 */
	-(void)setZipCode:(NSString*)ZipCode {
		[super setInput:@"ZipCode" toValue:ZipCode];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the RussianPostalCodes Choreo.
 */
@implementation TMBUtilities_Validation_RussianPostalCodes_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Match" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a string indicating the result of the match -- "valid" or "invalid".
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getMatch {
		return [super getOutputByName:@"Match"];
	}
	
@end

/*!
 * Verifies that a given zip code matches the format expected for Russian addresses.
 */
@implementation TMBUtilities_Validation_RussianPostalCodes

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Validation_RussianPostalCodes Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Validation/RussianPostalCodes"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Validation_RussianPostalCodes_ResultSet *results = [[TMBUtilities_Validation_RussianPostalCodes_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the RussianPostalCodes Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Validation_RussianPostalCodes_Inputs*)newInputSet {
		return [[TMBUtilities_Validation_RussianPostalCodes_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UKPostalCodes Choreo.
 */
@implementation TMBUtilities_Validation_UKPostalCodes_Inputs

	/*!
	 * Set the value of the ZipCode input for this Choreo.
	*(required, string) The zip code to validate. Letters must be in uppercase to be valid.
	 */
	-(void)setZipCode:(NSString*)ZipCode {
		[super setInput:@"ZipCode" toValue:ZipCode];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UKPostalCodes Choreo.
 */
@implementation TMBUtilities_Validation_UKPostalCodes_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Match" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a string indicating the result of the match -- "valid" or "invalid".
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getMatch {
		return [super getOutputByName:@"Match"];
	}
	
@end

/*!
 * Verifies that a given zip code matches the format expected for UK addresses.
 */
@implementation TMBUtilities_Validation_UKPostalCodes

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Validation_UKPostalCodes Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Validation/UKPostalCodes"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Validation_UKPostalCodes_ResultSet *results = [[TMBUtilities_Validation_UKPostalCodes_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UKPostalCodes Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Validation_UKPostalCodes_Inputs*)newInputSet {
		return [[TMBUtilities_Validation_UKPostalCodes_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the USPostalCodes Choreo.
 */
@implementation TMBUtilities_Validation_USPostalCodes_Inputs

	/*!
	 * Set the value of the ZipCode input for this Choreo.
	*(required, string) The zip code to validate.
	 */
	-(void)setZipCode:(NSString*)ZipCode {
		[super setInput:@"ZipCode" toValue:ZipCode];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the USPostalCodes Choreo.
 */
@implementation TMBUtilities_Validation_USPostalCodes_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Match" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains a string indicating the result of the match --"valid" or "invalid".
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getMatch {
		return [super getOutputByName:@"Match"];
	}
	
@end

/*!
 * Verifies that a given zip code matches the format expected for US addresses.
 */
@implementation TMBUtilities_Validation_USPostalCodes

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Validation_USPostalCodes Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Validation/USPostalCodes"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Validation_USPostalCodes_ResultSet *results = [[TMBUtilities_Validation_USPostalCodes_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the USPostalCodes Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Validation_USPostalCodes_Inputs*)newInputSet {
		return [[TMBUtilities_Validation_USPostalCodes_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the XML Choreo.
 */
@implementation TMBUtilities_Validation_XML_Inputs

	/*!
	 * Set the value of the XMLFile input for this Choreo.
	*(required, xml) The XML file you want to validate.
	 */
	-(void)setXMLFile:(NSString*)XMLFile {
		[super setInput:@"XMLFile" toValue:XMLFile];
	}

	/*!
	 * Set the value of the XSDFile input for this Choreo.
	*(optional, xml) The XSD file to validate an XML file against.
	 */
	-(void)setXSDFile:(NSString*)XSDFile {
		[super setInput:@"XSDFile" toValue:XSDFile];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the XML Choreo.
 */
@implementation TMBUtilities_Validation_XML_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Error" output from an execution of this Choreo.
	 * @return - NSString* (string) The error description that is generated if a validation error occurs.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getError {
		return [super getOutputByName:@"Error"];
	}

	/*!
	 * Retrieve the value of the "Result" output from an execution of this Choreo.
	 * @return - NSString* (string) The result of the validation. Returns the string "valid" or "invalid".
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResult {
		return [super getOutputByName:@"Result"];
	}
	
@end

/*!
 * Validates XML for basic well-formedness and allows you to check XML against a specified XSD schema file.
 */
@implementation TMBUtilities_Validation_XML

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_Validation_XML Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/Validation/XML"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_Validation_XML_ResultSet *results = [[TMBUtilities_Validation_XML_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the XML Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_Validation_XML_Inputs*)newInputSet {
		return [[TMBUtilities_Validation_XML_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetValuesFromXML Choreo.
 */
@implementation TMBUtilities_XML_GetValuesFromXML_Inputs

	/*!
	 * Set the value of the Node input for this Choreo.
	*(required, string) The name of the element or attribute that contains the values you want to return. Note that attribute names should be preceded with an "@" sign (e.g. @name).
	 */
	-(void)setNode:(NSString*)Node {
		[super setInput:@"Node" toValue:Node];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: json or csv.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the XML input for this Choreo.
	*(required, xml) The XML that contains the elements or attributes you want to retrieve.
	 */
	-(void)setXML:(NSString*)XML {
		[super setInput:@"XML" toValue:XML];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetValuesFromXML Choreo.
 */
@implementation TMBUtilities_XML_GetValuesFromXML_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Result" output from an execution of this Choreo.
	 * @return - NSString* The element or attribute values.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResult {
		return [super getOutputByName:@"Result"];
	}
	
@end

/*!
 * Returns all element or attribute values with a specified name.
 */
@implementation TMBUtilities_XML_GetValuesFromXML

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_XML_GetValuesFromXML Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/XML/GetValuesFromXML"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_XML_GetValuesFromXML_ResultSet *results = [[TMBUtilities_XML_GetValuesFromXML_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetValuesFromXML Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_XML_GetValuesFromXML_Inputs*)newInputSet {
		return [[TMBUtilities_XML_GetValuesFromXML_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the RunXPathQuery Choreo.
 */
@implementation TMBUtilities_XML_RunXPathQuery_Inputs

	/*!
	 * Set the value of the Mode input for this Choreo.
	*(conditional, string) Valid values are "select" (the default) or "recursive". Recursive mode will iterate using the provided XPath. Select mode will return the first match if there are multiple rows in the XML provided.
	 */
	-(void)setMode:(NSString*)Mode {
		[super setInput:@"Mode" toValue:Mode];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: json or csv.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the XML input for this Choreo.
	*(required, xml) The XML that contains the elements or attributes you want to retrieve.
	 */
	-(void)setXML:(NSString*)XML {
		[super setInput:@"XML" toValue:XML];
	}

	/*!
	 * Set the value of the XPath input for this Choreo.
	*(required, string) The XPath query to run.
	 */
	-(void)setXPath:(NSString*)XPath {
		[super setInput:@"XPath" toValue:XPath];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the RunXPathQuery Choreo.
 */
@implementation TMBUtilities_XML_RunXPathQuery_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Result" output from an execution of this Choreo.
	 * @return - NSString* The XPath query result.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResult {
		return [super getOutputByName:@"Result"];
	}
	
@end

/*!
 * Executes an XPath query against a specified XML file and returns the result in CSV or JSON format.
 */
@implementation TMBUtilities_XML_RunXPathQuery

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBUtilities_XML_RunXPathQuery Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Utilities/XML/RunXPathQuery"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBUtilities_XML_RunXPathQuery_ResultSet *results = [[TMBUtilities_XML_RunXPathQuery_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the RunXPathQuery Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBUtilities_XML_RunXPathQuery_Inputs*)newInputSet {
		return [[TMBUtilities_XML_RunXPathQuery_Inputs alloc] init];
	}
@end
	