/*!
 * @header Temboo iOS SDK Utilities classes
 *
 * Execute Choreographies from the Temboo Utilities bundle.
 *
 * iOS version 10.6.8
 *
 * LICENSE: Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @category   Services
 * @author     Reid Simonton, Temboo, Inc.
 * @copyright  2013 Temboo, Inc.
 * @license    http://www.apache.org/licenses/LICENSE-2.0 Apache License 2.0
 * @version    1.7
 */

#import <Foundation/Foundation.h>
#import "TMBChoreography.h"

@class TMBTembooSession;

/*! group TMBUtilities.Authentication.OAuth1.FinalizeOAuth Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the FinalizeOAuth Choreo.
 */
@interface TMBUtilities_Authentication_OAuth1_FinalizeOAuth_Inputs : TMBChoreographyInputSet
	-(void)setAccessTokenURL:(NSString*)AccessTokenURL;
	-(void)setCallbackID:(NSString*)CallbackID;
	-(void)setConsumerKey:(NSString*)ConsumerKey;
	-(void)setConsumerSecret:(NSString*)ConsumerSecret;
	-(void)setOAuthTokenSecret:(NSString*)OAuthTokenSecret;
	-(void)setSuppressErrors:(NSString*)SuppressErrors;
	-(void)setTimeout:(NSString*)Timeout;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the FinalizeOAuth Choreo.
 */
@interface TMBUtilities_Authentication_OAuth1_FinalizeOAuth_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getErrorMessage;
	-(NSString*)getResponse;	
@end

/*!
 * Completes the OAuth process by retrieving an access token and access token secret for a user, after they have visited the authorization URL returned by the InitializeOAuth Choreo and clicked "allow."
 */
@interface TMBUtilities_Authentication_OAuth1_FinalizeOAuth : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Authentication_OAuth1_FinalizeOAuth_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Authentication.OAuth1_FinalizeOAuth Choreo */


/*! group TMBUtilities.Authentication.OAuth1.InitializeOAuth Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the InitializeOAuth Choreo.
 */
@interface TMBUtilities_Authentication_OAuth1_InitializeOAuth_Inputs : TMBChoreographyInputSet
	-(void)setAuthorizationEndpoint:(NSString*)AuthorizationEndpoint;
	-(void)setConsumerKey:(NSString*)ConsumerKey;
	-(void)setConsumerSecret:(NSString*)ConsumerSecret;
	-(void)setForwardingURL:(NSString*)ForwardingURL;
	-(void)setRequestTokenEndpoint:(NSString*)RequestTokenEndpoint;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the InitializeOAuth Choreo.
 */
@interface TMBUtilities_Authentication_OAuth1_InitializeOAuth_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getAuthorizationURL;
	-(NSString*)getCallbackID;
	-(NSString*)getOAuthTokenSecret;	
@end

/*!
 * Generates an authorization URL that an application can use to complete the first step in the OAuth 1.0 process.
 */
@interface TMBUtilities_Authentication_OAuth1_InitializeOAuth : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Authentication_OAuth1_InitializeOAuth_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Authentication.OAuth1_InitializeOAuth Choreo */


/*! group TMBUtilities.Authentication.OAuth2.FinalizeOAuth Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the FinalizeOAuth Choreo.
 */
@interface TMBUtilities_Authentication_OAuth2_FinalizeOAuth_Inputs : TMBChoreographyInputSet
	-(void)setAccessTokenEndpoint:(NSString*)AccessTokenEndpoint;
	-(void)setCallbackID:(NSString*)CallbackID;
	-(void)setClientID:(NSString*)ClientID;
	-(void)setClientSecret:(NSString*)ClientSecret;
	-(void)setSuppressErrors:(NSString*)SuppressErrors;
	-(void)setTimeout:(NSString*)Timeout;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the FinalizeOAuth Choreo.
 */
@interface TMBUtilities_Authentication_OAuth2_FinalizeOAuth_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getErrorMessage;
	-(NSString*)getResponse;	
@end

/*!
 * Completes the OAuth process by retrieving an access token for a user, after they have visited the authorization URL returned by the InitializeOAuth Choreo and clicked "allow."
 */
@interface TMBUtilities_Authentication_OAuth2_FinalizeOAuth : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Authentication_OAuth2_FinalizeOAuth_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Authentication.OAuth2_FinalizeOAuth Choreo */


/*! group TMBUtilities.Authentication.OAuth2.InitializeOAuth Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the InitializeOAuth Choreo.
 */
@interface TMBUtilities_Authentication_OAuth2_InitializeOAuth_Inputs : TMBChoreographyInputSet
	-(void)setClientID:(NSString*)ClientID;
	-(void)setCustomCallbackID:(NSString*)CustomCallbackID;
	-(void)setForwardingURL:(NSString*)ForwardingURL;
	-(void)setRequestTokenEndpoint:(NSString*)RequestTokenEndpoint;
	-(void)setScope:(NSString*)Scope;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the InitializeOAuth Choreo.
 */
@interface TMBUtilities_Authentication_OAuth2_InitializeOAuth_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getAuthorizationURL;
	-(NSString*)getCallbackID;	
@end

/*!
 * Generates an authorization URL that an application can use to complete the first step in the OAuth 2.0 process.
 */
@interface TMBUtilities_Authentication_OAuth2_InitializeOAuth : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Authentication_OAuth2_InitializeOAuth_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Authentication.OAuth2_InitializeOAuth Choreo */


/*! group TMBUtilities.Authentication.OAuth2.RefreshToken Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the RefreshToken Choreo.
 */
@interface TMBUtilities_Authentication_OAuth2_RefreshToken_Inputs : TMBChoreographyInputSet
	-(void)setAccessTokenEndpoint:(NSString*)AccessTokenEndpoint;
	-(void)setClientID:(NSString*)ClientID;
	-(void)setClientSecret:(NSString*)ClientSecret;
	-(void)setRefreshToken:(NSString*)RefreshToken;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the RefreshToken Choreo.
 */
@interface TMBUtilities_Authentication_OAuth2_RefreshToken_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Refreshes an expired access token.
 */
@interface TMBUtilities_Authentication_OAuth2_RefreshToken : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Authentication_OAuth2_RefreshToken_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Authentication.OAuth2_RefreshToken Choreo */


/*! group TMBUtilities.Callback.GetCallbackData Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the GetCallbackData Choreo.
 */
@interface TMBUtilities_Callback_GetCallbackData_Inputs : TMBChoreographyInputSet
	-(void)setCallbackID:(NSString*)CallbackID;
	-(void)setFilterName:(NSString*)FilterName;
	-(void)setFilterValue:(NSString*)FilterValue;
	-(void)setResponseFormat:(NSString*)ResponseFormat;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetCallbackData Choreo.
 */
@interface TMBUtilities_Callback_GetCallbackData_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getCallbackData;	
@end

/*!
 * Retrieves data captured from a request to your callback URL.
 */
@interface TMBUtilities_Callback_GetCallbackData : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Callback_GetCallbackData_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Callback_GetCallbackData Choreo */


/*! group TMBUtilities.Callback.RegisterCallback Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the RegisterCallback Choreo.
 */
@interface TMBUtilities_Callback_RegisterCallback_Inputs : TMBChoreographyInputSet
	-(void)setCustomCallbackD:(NSString*)CustomCallbackD;
	-(void)setFilterName:(NSString*)FilterName;
	-(void)setFilterValue:(NSString*)FilterValue;
	-(void)setForwardingURL:(NSString*)ForwardingURL;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the RegisterCallback Choreo.
 */
@interface TMBUtilities_Callback_RegisterCallback_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getCallbackID;
	-(NSString*)getCallbackURL;	
@end

/*!
 * Allows you to generate a unique URL that can "listen" for incoming data from web request.
 */
@interface TMBUtilities_Callback_RegisterCallback : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Callback_RegisterCallback_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Callback_RegisterCallback Choreo */


/*! group TMBUtilities.DataConversions.CSVToJSON Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the CSVToJSON Choreo.
 */
@interface TMBUtilities_DataConversions_CSVToJSON_Inputs : TMBChoreographyInputSet
	-(void)setCSV:(NSString*)CSV;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CSVToJSON Choreo.
 */
@interface TMBUtilities_DataConversions_CSVToJSON_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getJSON;	
@end

/*!
 * Converts a CSV formatted file to JSON.
 */
@interface TMBUtilities_DataConversions_CSVToJSON : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_DataConversions_CSVToJSON_Inputs*)newInputSet;
@end

/*! group TMBUtilities.DataConversions_CSVToJSON Choreo */


/*! group TMBUtilities.DataConversions.CSVToXLS Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the CSVToXLS Choreo.
 */
@interface TMBUtilities_DataConversions_CSVToXLS_Inputs : TMBChoreographyInputSet
	-(void)setCSV:(NSString*)CSV;
	-(void)setVaultFile:(NSString*)VaultFile;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CSVToXLS Choreo.
 */
@interface TMBUtilities_DataConversions_CSVToXLS_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getXLS;	
@end

/*!
 * Converts a CSV formatted file to Base64 encoded Excel data.
 */
@interface TMBUtilities_DataConversions_CSVToXLS : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_DataConversions_CSVToXLS_Inputs*)newInputSet;
@end

/*! group TMBUtilities.DataConversions_CSVToXLS Choreo */


/*! group TMBUtilities.DataConversions.CSVToXML Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the CSVToXML Choreo.
 */
@interface TMBUtilities_DataConversions_CSVToXML_Inputs : TMBChoreographyInputSet
	-(void)setCSV:(NSString*)CSV;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CSVToXML Choreo.
 */
@interface TMBUtilities_DataConversions_CSVToXML_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getXML;	
@end

/*!
 * Converts a CSV formatted file to XML.
 */
@interface TMBUtilities_DataConversions_CSVToXML : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_DataConversions_CSVToXML_Inputs*)newInputSet;
@end

/*! group TMBUtilities.DataConversions_CSVToXML Choreo */


/*! group TMBUtilities.DataConversions.JSONToXML Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the JSONToXML Choreo.
 */
@interface TMBUtilities_DataConversions_JSONToXML_Inputs : TMBChoreographyInputSet
	-(void)setJSON:(NSString*)JSON;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the JSONToXML Choreo.
 */
@interface TMBUtilities_DataConversions_JSONToXML_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getXML;	
@end

/*!
 * Converts data from JSON format to XML format.
 */
@interface TMBUtilities_DataConversions_JSONToXML : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_DataConversions_JSONToXML_Inputs*)newInputSet;
@end

/*! group TMBUtilities.DataConversions_JSONToXML Choreo */


/*! group TMBUtilities.DataConversions.RSSToJSON Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the RSSToJSON Choreo.
 */
@interface TMBUtilities_DataConversions_RSSToJSON_Inputs : TMBChoreographyInputSet
	-(void)setRSSFeed:(NSString*)RSSFeed;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the RSSToJSON Choreo.
 */
@interface TMBUtilities_DataConversions_RSSToJSON_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Retrieves a specified RSS Feed, and converts it to JSON format.
 */
@interface TMBUtilities_DataConversions_RSSToJSON : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_DataConversions_RSSToJSON_Inputs*)newInputSet;
@end

/*! group TMBUtilities.DataConversions_RSSToJSON Choreo */


/*! group TMBUtilities.DataConversions.TSVToJSON Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the TSVToJSON Choreo.
 */
@interface TMBUtilities_DataConversions_TSVToJSON_Inputs : TMBChoreographyInputSet
	-(void)setTSV:(NSString*)TSV;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the TSVToJSON Choreo.
 */
@interface TMBUtilities_DataConversions_TSVToJSON_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getJSON;	
@end

/*!
 * Converts a TSV formatted file to JSON.
 */
@interface TMBUtilities_DataConversions_TSVToJSON : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_DataConversions_TSVToJSON_Inputs*)newInputSet;
@end

/*! group TMBUtilities.DataConversions_TSVToJSON Choreo */


/*! group TMBUtilities.DataConversions.TSVToXML Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the TSVToXML Choreo.
 */
@interface TMBUtilities_DataConversions_TSVToXML_Inputs : TMBChoreographyInputSet
	-(void)setTSV:(NSString*)TSV;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the TSVToXML Choreo.
 */
@interface TMBUtilities_DataConversions_TSVToXML_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getXML;	
@end

/*!
 * Converts a TSV formatted file to XML.
 */
@interface TMBUtilities_DataConversions_TSVToXML : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_DataConversions_TSVToXML_Inputs*)newInputSet;
@end

/*! group TMBUtilities.DataConversions_TSVToXML Choreo */


/*! group TMBUtilities.DataConversions.XLSToCSV Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the XLSToCSV Choreo.
 */
@interface TMBUtilities_DataConversions_XLSToCSV_Inputs : TMBChoreographyInputSet
	-(void)setXLS:(NSString*)XLS;
	-(void)setVaultFile:(NSString*)VaultFile;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the XLSToCSV Choreo.
 */
@interface TMBUtilities_DataConversions_XLSToCSV_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getCSV;	
@end

/*!
 * Converts Excel (.xls) formatted data to CSV.
 */
@interface TMBUtilities_DataConversions_XLSToCSV : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_DataConversions_XLSToCSV_Inputs*)newInputSet;
@end

/*! group TMBUtilities.DataConversions_XLSToCSV Choreo */


/*! group TMBUtilities.DataConversions.XLSToXML Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the XLSToXML Choreo.
 */
@interface TMBUtilities_DataConversions_XLSToXML_Inputs : TMBChoreographyInputSet
	-(void)setXLS:(NSString*)XLS;
	-(void)setVaultFile:(NSString*)VaultFile;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the XLSToXML Choreo.
 */
@interface TMBUtilities_DataConversions_XLSToXML_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getXML;	
@end

/*!
 * Converts Excel (.xls) formatted data to XML.
 */
@interface TMBUtilities_DataConversions_XLSToXML : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_DataConversions_XLSToXML_Inputs*)newInputSet;
@end

/*! group TMBUtilities.DataConversions_XLSToXML Choreo */


/*! group TMBUtilities.DataConversions.XMLToCSV Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the XMLToCSV Choreo.
 */
@interface TMBUtilities_DataConversions_XMLToCSV_Inputs : TMBChoreographyInputSet
	-(void)setXML:(NSString*)XML;
	-(void)setPath:(NSString*)Path;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the XMLToCSV Choreo.
 */
@interface TMBUtilities_DataConversions_XMLToCSV_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getCSV;	
@end

/*!
 * Converts an XML file to CSV format.
 */
@interface TMBUtilities_DataConversions_XMLToCSV : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_DataConversions_XMLToCSV_Inputs*)newInputSet;
@end

/*! group TMBUtilities.DataConversions_XMLToCSV Choreo */


/*! group TMBUtilities.DataConversions.XMLToJSON Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the XMLToJSON Choreo.
 */
@interface TMBUtilities_DataConversions_XMLToJSON_Inputs : TMBChoreographyInputSet
	-(void)setXML:(NSString*)XML;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the XMLToJSON Choreo.
 */
@interface TMBUtilities_DataConversions_XMLToJSON_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getJSON;	
@end

/*!
 * Converts data from XML format to JSON format.
 */
@interface TMBUtilities_DataConversions_XMLToJSON : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_DataConversions_XMLToJSON_Inputs*)newInputSet;
@end

/*! group TMBUtilities.DataConversions_XMLToJSON Choreo */


/*! group TMBUtilities.DataConversions.XMLToTSV Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the XMLToTSV Choreo.
 */
@interface TMBUtilities_DataConversions_XMLToTSV_Inputs : TMBChoreographyInputSet
	-(void)setXML:(NSString*)XML;
	-(void)setPath:(NSString*)Path;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the XMLToTSV Choreo.
 */
@interface TMBUtilities_DataConversions_XMLToTSV_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getTSV;	
@end

/*!
 * Converts an XML file to TSV format (TAB-separated values).
 */
@interface TMBUtilities_DataConversions_XMLToTSV : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_DataConversions_XMLToTSV_Inputs*)newInputSet;
@end

/*! group TMBUtilities.DataConversions_XMLToTSV Choreo */


/*! group TMBUtilities.DataConversions.XMLToXLS Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the XMLToXLS Choreo.
 */
@interface TMBUtilities_DataConversions_XMLToXLS_Inputs : TMBChoreographyInputSet
	-(void)setXML:(NSString*)XML;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the XMLToXLS Choreo.
 */
@interface TMBUtilities_DataConversions_XMLToXLS_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getXLS;	
@end

/*!
 * Converts an XML file to a Base64 encoded Excel file.
 */
@interface TMBUtilities_DataConversions_XMLToXLS : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_DataConversions_XMLToXLS_Inputs*)newInputSet;
@end

/*! group TMBUtilities.DataConversions_XMLToXLS Choreo */


/*! group TMBUtilities.Dates.GetDate Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the GetDate Choreo.
 */
@interface TMBUtilities_Dates_GetDate_Inputs : TMBChoreographyInputSet
	-(void)setAddDays:(NSString*)AddDays;
	-(void)setAddHours:(NSString*)AddHours;
	-(void)setAddMinutes:(NSString*)AddMinutes;
	-(void)setAddMonths:(NSString*)AddMonths;
	-(void)setAddSeconds:(NSString*)AddSeconds;
	-(void)setAddYears:(NSString*)AddYears;
	-(void)setFormat:(NSString*)Format;
	-(void)setLocaleCountry:(NSString*)LocaleCountry;
	-(void)setLocaleLanguage:(NSString*)LocaleLanguage;
	-(void)setLocaleVariant:(NSString*)LocaleVariant;
	-(void)setSetDay:(NSString*)SetDay;
	-(void)setSetHour:(NSString*)SetHour;
	-(void)setSetMinute:(NSString*)SetMinute;
	-(void)setSetMonth:(NSString*)SetMonth;
	-(void)setSetSecond:(NSString*)SetSecond;
	-(void)setSetYear:(NSString*)SetYear;
	-(void)setTimeZone:(NSString*)TimeZone;
	-(void)setTimestamp:(NSString*)Timestamp;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetDate Choreo.
 */
@interface TMBUtilities_Dates_GetDate_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getFormattedDate;	
@end

/*!
 * Formats a specified timestamp, or generates the current date in a desired format.
 */
@interface TMBUtilities_Dates_GetDate : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Dates_GetDate_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Dates_GetDate Choreo */


/*! group TMBUtilities.Dates.GetDateDifference Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the GetDateDifference Choreo.
 */
@interface TMBUtilities_Dates_GetDateDifference_Inputs : TMBChoreographyInputSet
	-(void)setEarlierDate:(NSString*)EarlierDate;
	-(void)setLaterDate:(NSString*)LaterDate;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetDateDifference Choreo.
 */
@interface TMBUtilities_Dates_GetDateDifference_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getDifference;	
@end

/*!
 * Returns the difference between two specified dates, expressed as the number of milliseconds since January 1, 1970 (epoch time).
 */
@interface TMBUtilities_Dates_GetDateDifference : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Dates_GetDateDifference_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Dates_GetDateDifference Choreo */


/*! group TMBUtilities.Dates.GetTimestamp Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the GetTimestamp Choreo.
 */
@interface TMBUtilities_Dates_GetTimestamp_Inputs : TMBChoreographyInputSet
	-(void)setAddDays:(NSString*)AddDays;
	-(void)setAddHours:(NSString*)AddHours;
	-(void)setAddMinutes:(NSString*)AddMinutes;
	-(void)setAddMonths:(NSString*)AddMonths;
	-(void)setAddSeconds:(NSString*)AddSeconds;
	-(void)setAddYears:(NSString*)AddYears;
	-(void)setGranularity:(NSString*)Granularity;
	-(void)setSetDay:(NSString*)SetDay;
	-(void)setSetHour:(NSString*)SetHour;
	-(void)setSetMinute:(NSString*)SetMinute;
	-(void)setSetMonth:(NSString*)SetMonth;
	-(void)setSetSecond:(NSString*)SetSecond;
	-(void)setSetYear:(NSString*)SetYear;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetTimestamp Choreo.
 */
@interface TMBUtilities_Dates_GetTimestamp_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getTimestamp;	
@end

/*!
 * Returns the current date and time, expressed as seconds or milliseconds since January 1, 1970 (epoch time).
 */
@interface TMBUtilities_Dates_GetTimestamp : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Dates_GetTimestamp_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Dates_GetTimestamp Choreo */


/*! group TMBUtilities.Dates.GetTimestampFromDateParameters Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the GetTimestampFromDateParameters Choreo.
 */
@interface TMBUtilities_Dates_GetTimestampFromDateParameters_Inputs : TMBChoreographyInputSet
	-(void)setDay:(NSString*)Day;
	-(void)setGranularity:(NSString*)Granularity;
	-(void)setHour:(NSString*)Hour;
	-(void)setMilliseconds:(NSString*)Milliseconds;
	-(void)setMinute:(NSString*)Minute;
	-(void)setMonth:(NSString*)Month;
	-(void)setSecond:(NSString*)Second;
	-(void)setYear:(NSString*)Year;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetTimestampFromDateParameters Choreo.
 */
@interface TMBUtilities_Dates_GetTimestampFromDateParameters_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getTimestamp;	
@end

/*!
 * Returns the specified date parameters, expressed as the number of seconds or milliseconds since January 1, 1970 (epoch time).
 */
@interface TMBUtilities_Dates_GetTimestampFromDateParameters : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Dates_GetTimestampFromDateParameters_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Dates_GetTimestampFromDateParameters Choreo */


/*! group TMBUtilities.Dates.GetTimestampFromDateString Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the GetTimestampFromDateString Choreo.
 */
@interface TMBUtilities_Dates_GetTimestampFromDateString_Inputs : TMBChoreographyInputSet
	-(void)setAddDays:(NSString*)AddDays;
	-(void)setAddHours:(NSString*)AddHours;
	-(void)setAddMinutes:(NSString*)AddMinutes;
	-(void)setAddMonths:(NSString*)AddMonths;
	-(void)setAddSeconds:(NSString*)AddSeconds;
	-(void)setAddYears:(NSString*)AddYears;
	-(void)setDate:(NSString*)Date;
	-(void)setGranularity:(NSString*)Granularity;
	-(void)setSetDay:(NSString*)SetDay;
	-(void)setSetHour:(NSString*)SetHour;
	-(void)setSetMinute:(NSString*)SetMinute;
	-(void)setSetMonth:(NSString*)SetMonth;
	-(void)setSetSecond:(NSString*)SetSecond;
	-(void)setSetYear:(NSString*)SetYear;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetTimestampFromDateString Choreo.
 */
@interface TMBUtilities_Dates_GetTimestampFromDateString_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getTimestamp;	
@end

/*!
 * Returns the the specified date string, expressed as seconds or milliseconds since January 1, 1970 (epoch time).
 */
@interface TMBUtilities_Dates_GetTimestampFromDateString : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Dates_GetTimestampFromDateString_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Dates_GetTimestampFromDateString Choreo */


/*! group TMBUtilities.Email.SendEmail Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the SendEmail Choreo.
 */
@interface TMBUtilities_Email_SendEmail_Inputs : TMBChoreographyInputSet
	-(void)setAttachment:(NSString*)Attachment;
	-(void)setAttachmentName:(NSString*)AttachmentName;
	-(void)setAttachmentURL:(NSString*)AttachmentURL;
	-(void)setBCC:(NSString*)BCC;
	-(void)setCC:(NSString*)CC;
	-(void)setFromAddress:(NSString*)FromAddress;
	-(void)setMessageBody:(NSString*)MessageBody;
	-(void)setPassword:(NSString*)Password;
	-(void)setPort:(NSString*)Port;
	-(void)setServer:(NSString*)Server;
	-(void)setSubject:(NSString*)Subject;
	-(void)setToAddress:(NSString*)ToAddress;
	-(void)setUseSSL:(NSString*)UseSSL;
	-(void)setUsername:(NSString*)Username;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SendEmail Choreo.
 */
@interface TMBUtilities_Email_SendEmail_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getSuccess;	
@end

/*!
 * Sends an email using a specified email server.
 */
@interface TMBUtilities_Email_SendEmail : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Email_SendEmail_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Email_SendEmail Choreo */


/*! group TMBUtilities.Encoding.Base64Decode Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the Base64Decode Choreo.
 */
@interface TMBUtilities_Encoding_Base64Decode_Inputs : TMBChoreographyInputSet
	-(void)setBase64EncodedText:(NSString*)Base64EncodedText;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Base64Decode Choreo.
 */
@interface TMBUtilities_Encoding_Base64Decode_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getText;	
@end

/*!
 * Returns the specified Base64 encoded string as decoded text.
 */
@interface TMBUtilities_Encoding_Base64Decode : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Encoding_Base64Decode_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Encoding_Base64Decode Choreo */


/*! group TMBUtilities.Encoding.Base64Encode Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the Base64Encode Choreo.
 */
@interface TMBUtilities_Encoding_Base64Encode_Inputs : TMBChoreographyInputSet
	-(void)setText:(NSString*)Text;
	-(void)setURL:(NSString*)URL;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Base64Encode Choreo.
 */
@interface TMBUtilities_Encoding_Base64Encode_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getBase64EncodedText;	
@end

/*!
 * Returns the specified text or file as a Base64 encoded string.
 */
@interface TMBUtilities_Encoding_Base64Encode : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Encoding_Base64Encode_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Encoding_Base64Encode Choreo */


/*! group TMBUtilities.Encoding.HTMLEscape Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the HTMLEscape Choreo.
 */
@interface TMBUtilities_Encoding_HTMLEscape_Inputs : TMBChoreographyInputSet
	-(void)setUnescapedHTML:(NSString*)UnescapedHTML;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the HTMLEscape Choreo.
 */
@interface TMBUtilities_Encoding_HTMLEscape_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getEscapedHTML;	
@end

/*!
 * Replaces HTML markup characters in the specified text with equivalent character entity names. 
 */
@interface TMBUtilities_Encoding_HTMLEscape : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Encoding_HTMLEscape_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Encoding_HTMLEscape Choreo */


/*! group TMBUtilities.Encoding.HTMLUnescape Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the HTMLUnescape Choreo.
 */
@interface TMBUtilities_Encoding_HTMLUnescape_Inputs : TMBChoreographyInputSet
	-(void)setEscapedHTML:(NSString*)EscapedHTML;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the HTMLUnescape Choreo.
 */
@interface TMBUtilities_Encoding_HTMLUnescape_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getUnescapedHTML;	
@end

/*!
 * Replaces character entity names in the specified text with equivalent HTML markup characters.
 */
@interface TMBUtilities_Encoding_HTMLUnescape : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Encoding_HTMLUnescape_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Encoding_HTMLUnescape Choreo */


/*! group TMBUtilities.Encoding.URLDecode Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the URLDecode Choreo.
 */
@interface TMBUtilities_Encoding_URLDecode_Inputs : TMBChoreographyInputSet
	-(void)setText:(NSString*)Text;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the URLDecode Choreo.
 */
@interface TMBUtilities_Encoding_URLDecode_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getURLDecodedText;	
@end

/*!
 * Removes URL encoding from the specified text string.
 */
@interface TMBUtilities_Encoding_URLDecode : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Encoding_URLDecode_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Encoding_URLDecode Choreo */


/*! group TMBUtilities.Encoding.URLEncode Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the URLEncode Choreo.
 */
@interface TMBUtilities_Encoding_URLEncode_Inputs : TMBChoreographyInputSet
	-(void)setText:(NSString*)Text;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the URLEncode Choreo.
 */
@interface TMBUtilities_Encoding_URLEncode_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getURLEncodedText;	
@end

/*!
 * Returns the specified text string in the application/x-www-form-urlencoded format.
 */
@interface TMBUtilities_Encoding_URLEncode : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Encoding_URLEncode_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Encoding_URLEncode Choreo */


/*! group TMBUtilities.Finding.FindSubstring Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the FindSubstring Choreo.
 */
@interface TMBUtilities_Finding_FindSubstring_Inputs : TMBChoreographyInputSet
	-(void)setCaseSensitive:(NSString*)CaseSensitive;
	-(void)setStartNumber:(NSString*)StartNumber;
	-(void)setSubstring:(NSString*)Substring;
	-(void)setText:(NSString*)Text;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the FindSubstring Choreo.
 */
@interface TMBUtilities_Finding_FindSubstring_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getCount;
	-(NSString*)getPositions;	
@end

/*!
 * Finds all occurrences of a specified substring and returns the substring positions as a JSON array.
 */
@interface TMBUtilities_Finding_FindSubstring : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Finding_FindSubstring_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Finding_FindSubstring Choreo */


/*! group TMBUtilities.Finding.RegexMatch Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the RegexMatch Choreo.
 */
@interface TMBUtilities_Finding_RegexMatch_Inputs : TMBChoreographyInputSet
	-(void)setPattern:(NSString*)Pattern;
	-(void)setText:(NSString*)Text;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the RegexMatch Choreo.
 */
@interface TMBUtilities_Finding_RegexMatch_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResult;	
@end

/*!
 * Returns the first substring that matches the specified regular expression pattern in the specified string.
 */
@interface TMBUtilities_Finding_RegexMatch : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Finding_RegexMatch_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Finding_RegexMatch Choreo */


/*! group TMBUtilities.Formatting.RemoveWhiteSpace Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the RemoveWhiteSpace Choreo.
 */
@interface TMBUtilities_Formatting_RemoveWhiteSpace_Inputs : TMBChoreographyInputSet
	-(void)setFormattedText:(NSString*)FormattedText;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the RemoveWhiteSpace Choreo.
 */
@interface TMBUtilities_Formatting_RemoveWhiteSpace_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getCompactText;	
@end

/*!
 * Returns the specified formatted text as a compact string with no new lines, tabs, or preceding/trailing white space.
 */
@interface TMBUtilities_Formatting_RemoveWhiteSpace : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Formatting_RemoveWhiteSpace_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Formatting_RemoveWhiteSpace Choreo */


/*! group TMBUtilities.HTTP.Delete Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the Delete Choreo.
 */
@interface TMBUtilities_HTTP_Delete_Inputs : TMBChoreographyInputSet
	-(void)setDebug:(NSString*)Debug;
	-(void)setPassword:(NSString*)Password;
	-(void)setRequestHeaders:(NSString*)RequestHeaders;
	-(void)setRequestParameters:(NSString*)RequestParameters;
	-(void)setURL:(NSString*)URL;
	-(void)setUsername:(NSString*)Username;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Delete Choreo.
 */
@interface TMBUtilities_HTTP_Delete_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getHTTPLog;
	-(NSString*)getResponseStatusCode;
	-(NSString*)getResponse;	
@end

/*!
 * Generates a HTTP DELETE request.
 */
@interface TMBUtilities_HTTP_Delete : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_HTTP_Delete_Inputs*)newInputSet;
@end

/*! group TMBUtilities.HTTP_Delete Choreo */


/*! group TMBUtilities.HTTP.Get Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the Get Choreo.
 */
@interface TMBUtilities_HTTP_Get_Inputs : TMBChoreographyInputSet
	-(void)setDebug:(NSString*)Debug;
	-(void)setPassword:(NSString*)Password;
	-(void)setRequestHeaders:(NSString*)RequestHeaders;
	-(void)setRequestParameters:(NSString*)RequestParameters;
	-(void)setURL:(NSString*)URL;
	-(void)setUsername:(NSString*)Username;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Get Choreo.
 */
@interface TMBUtilities_HTTP_Get_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getHTTPLog;
	-(NSString*)getResponseStatusCode;
	-(NSString*)getResponse;	
@end

/*!
 * Generates a HTTP GET request.
 */
@interface TMBUtilities_HTTP_Get : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_HTTP_Get_Inputs*)newInputSet;
@end

/*! group TMBUtilities.HTTP_Get Choreo */


/*! group TMBUtilities.HTTP.Post Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the Post Choreo.
 */
@interface TMBUtilities_HTTP_Post_Inputs : TMBChoreographyInputSet
	-(void)setRequestBody:(NSString*)RequestBody;
	-(void)setDebug:(NSString*)Debug;
	-(void)setPassword:(NSString*)Password;
	-(void)setRequestHeaders:(NSString*)RequestHeaders;
	-(void)setRequestParameters:(NSString*)RequestParameters;
	-(void)setURL:(NSString*)URL;
	-(void)setUsername:(NSString*)Username;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Post Choreo.
 */
@interface TMBUtilities_HTTP_Post_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getHTTPLog;
	-(NSString*)getResponseStatusCode;
	-(NSString*)getResponse;	
@end

/*!
 * Generates a HTTP POST request.
 */
@interface TMBUtilities_HTTP_Post : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_HTTP_Post_Inputs*)newInputSet;
@end

/*! group TMBUtilities.HTTP_Post Choreo */


/*! group TMBUtilities.HTTP.Put Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the Put Choreo.
 */
@interface TMBUtilities_HTTP_Put_Inputs : TMBChoreographyInputSet
	-(void)setRequestBody:(NSString*)RequestBody;
	-(void)setDebug:(NSString*)Debug;
	-(void)setPassword:(NSString*)Password;
	-(void)setRequestHeaders:(NSString*)RequestHeaders;
	-(void)setRequestParameters:(NSString*)RequestParameters;
	-(void)setURL:(NSString*)URL;
	-(void)setUsername:(NSString*)Username;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Put Choreo.
 */
@interface TMBUtilities_HTTP_Put_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getHTTPLog;
	-(NSString*)getResponseStatusCode;
	-(NSString*)getResponse;	
@end

/*!
 * Generates a HTTP PUT request.
 */
@interface TMBUtilities_HTTP_Put : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_HTTP_Put_Inputs*)newInputSet;
@end

/*! group TMBUtilities.HTTP_Put Choreo */


/*! group TMBUtilities.Hashing.Checksum Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the Checksum Choreo.
 */
@interface TMBUtilities_Hashing_Checksum_Inputs : TMBChoreographyInputSet
	-(void)setAlgorithm:(NSString*)Algorithm;
	-(void)setBase64DecodeValue:(NSString*)Base64DecodeValue;
	-(void)setText:(NSString*)Text;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Checksum Choreo.
 */
@interface TMBUtilities_Hashing_Checksum_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getChecksum;	
@end

/*!
 * Returns a checksum of the specified text calculated using the specified algorithm. 
 */
@interface TMBUtilities_Hashing_Checksum : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Hashing_Checksum_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Hashing_Checksum Choreo */


/*! group TMBUtilities.Hashing.HmacSHA1 Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the HmacSHA1 Choreo.
 */
@interface TMBUtilities_Hashing_HmacSHA1_Inputs : TMBChoreographyInputSet
	-(void)setKey:(NSString*)Key;
	-(void)setText:(NSString*)Text;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the HmacSHA1 Choreo.
 */
@interface TMBUtilities_Hashing_HmacSHA1_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getHmacSignature;	
@end

/*!
 * Generates a SHA1-encrypted HMAC signature for the specified message text using the specified secret key.
 */
@interface TMBUtilities_Hashing_HmacSHA1 : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Hashing_HmacSHA1_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Hashing_HmacSHA1 Choreo */


/*! group TMBUtilities.Hashing.HmacSHA256 Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the HmacSHA256 Choreo.
 */
@interface TMBUtilities_Hashing_HmacSHA256_Inputs : TMBChoreographyInputSet
	-(void)setKey:(NSString*)Key;
	-(void)setText:(NSString*)Text;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the HmacSHA256 Choreo.
 */
@interface TMBUtilities_Hashing_HmacSHA256_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getHmacSignature;	
@end

/*!
 * Generates a SHA256-encrypted HMAC signature for the specified message text using the specified secret key.
 */
@interface TMBUtilities_Hashing_HmacSHA256 : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Hashing_HmacSHA256_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Hashing_HmacSHA256 Choreo */


/*! group TMBUtilities.JSON.GetValuesFromJSON Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the GetValuesFromJSON Choreo.
 */
@interface TMBUtilities_JSON_GetValuesFromJSON_Inputs : TMBChoreographyInputSet
	-(void)setJSON:(NSString*)JSON;
	-(void)setProperty:(NSString*)Property;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetValuesFromJSON Choreo.
 */
@interface TMBUtilities_JSON_GetValuesFromJSON_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Searches for the specified property in the supplied JSON string.
 */
@interface TMBUtilities_JSON_GetValuesFromJSON : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_JSON_GetValuesFromJSON_Inputs*)newInputSet;
@end

/*! group TMBUtilities.JSON_GetValuesFromJSON Choreo */


/*! group TMBUtilities.Numbers.GenerateRandom Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the GenerateRandom Choreo.
 */
@interface TMBUtilities_Numbers_GenerateRandom_Inputs : TMBChoreographyInputSet
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GenerateRandom Choreo.
 */
@interface TMBUtilities_Numbers_GenerateRandom_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getSignedDecimal;
	-(NSString*)getSignedInteger;
	-(NSString*)getUnsignedDecimal;
	-(NSString*)getUnsignedInteger;	
@end

/*!
 * This choreo generates a random number in a variety of ranges. 
 */
@interface TMBUtilities_Numbers_GenerateRandom : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Numbers_GenerateRandom_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Numbers_GenerateRandom Choreo */


/*! group TMBUtilities.Test.HelloWorld Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the HelloWorld Choreo.
 */
@interface TMBUtilities_Test_HelloWorld_Inputs : TMBChoreographyInputSet
	-(void)setValue:(NSString*)Value;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the HelloWorld Choreo.
 */
@interface TMBUtilities_Test_HelloWorld_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResult;	
@end

/*!
 * Allows you to run a simple test that outputs "Hello, world!" when executed.
 */
@interface TMBUtilities_Test_HelloWorld : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Test_HelloWorld_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Test_HelloWorld Choreo */


/*! group TMBUtilities.Text.GenerateGUID Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the GenerateGUID Choreo.
 */
@interface TMBUtilities_Text_GenerateGUID_Inputs : TMBChoreographyInputSet
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GenerateGUID Choreo.
 */
@interface TMBUtilities_Text_GenerateGUID_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getGUID;	
@end

/*!
 * Obtain a unique, randomly generated ID or GUID (Globally Unique Identifier).
 */
@interface TMBUtilities_Text_GenerateGUID : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Text_GenerateGUID_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Text_GenerateGUID Choreo */


/*! group TMBUtilities.Text.Lowercase Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the Lowercase Choreo.
 */
@interface TMBUtilities_Text_Lowercase_Inputs : TMBChoreographyInputSet
	-(void)setText:(NSString*)Text;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Lowercase Choreo.
 */
@interface TMBUtilities_Text_Lowercase_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Returns the contents of the specified string converted to lowercase.
 */
@interface TMBUtilities_Text_Lowercase : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Text_Lowercase_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Text_Lowercase Choreo */


/*! group TMBUtilities.Text.Split Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the Split Choreo.
 */
@interface TMBUtilities_Text_Split_Inputs : TMBChoreographyInputSet
	-(void)setDelimiter:(NSString*)Delimiter;
	-(void)setText:(NSString*)Text;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Split Choreo.
 */
@interface TMBUtilities_Text_Split_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Splits a string into sub-strings delimited by the specified delmiter pattern.
 */
@interface TMBUtilities_Text_Split : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Text_Split_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Text_Split Choreo */


/*! group TMBUtilities.Text.Substitute Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the Substitute Choreo.
 */
@interface TMBUtilities_Text_Substitute_Inputs : TMBChoreographyInputSet
	-(void)setNew:(NSString*)New;
	-(void)setOld:(NSString*)Old;
	-(void)setText:(NSString*)Text;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Substitute Choreo.
 */
@interface TMBUtilities_Text_Substitute_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Replaces all instances of the specified sub-string within the specified text with a new sub-string. 
 */
@interface TMBUtilities_Text_Substitute : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Text_Substitute_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Text_Substitute Choreo */


/*! group TMBUtilities.Text.SubstituteRegex Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the SubstituteRegex Choreo.
 */
@interface TMBUtilities_Text_SubstituteRegex_Inputs : TMBChoreographyInputSet
	-(void)setNew:(NSString*)New;
	-(void)setPattern:(NSString*)Pattern;
	-(void)setText:(NSString*)Text;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SubstituteRegex Choreo.
 */
@interface TMBUtilities_Text_SubstituteRegex_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Finds all instances of the specified regular expression pattern within the given string and passes the specified new sub-string to the result variable. 
 */
@interface TMBUtilities_Text_SubstituteRegex : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Text_SubstituteRegex_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Text_SubstituteRegex Choreo */


/*! group TMBUtilities.Text.Uppercase Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the Uppercase Choreo.
 */
@interface TMBUtilities_Text_Uppercase_Inputs : TMBChoreographyInputSet
	-(void)setText:(NSString*)Text;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Uppercase Choreo.
 */
@interface TMBUtilities_Text_Uppercase_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Returns the contents of the specified string converted to uppercase.
 */
@interface TMBUtilities_Text_Uppercase : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Text_Uppercase_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Text_Uppercase Choreo */


/*! group TMBUtilities.TokenStorage.DeleteToken Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the DeleteToken Choreo.
 */
@interface TMBUtilities_TokenStorage_DeleteToken_Inputs : TMBChoreographyInputSet
	-(void)setName:(NSString*)Name;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteToken Choreo.
 */
@interface TMBUtilities_TokenStorage_DeleteToken_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getDeleted;	
@end

/*!
 * Deletes a specified token.
 */
@interface TMBUtilities_TokenStorage_DeleteToken : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_TokenStorage_DeleteToken_Inputs*)newInputSet;
@end

/*! group TMBUtilities.TokenStorage_DeleteToken Choreo */


/*! group TMBUtilities.TokenStorage.GetTokenDetails Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the GetTokenDetails Choreo.
 */
@interface TMBUtilities_TokenStorage_GetTokenDetails_Inputs : TMBChoreographyInputSet
	-(void)setNames:(NSString*)Names;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetTokenDetails Choreo.
 */
@interface TMBUtilities_TokenStorage_GetTokenDetails_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getTokens;	
@end

/*!
 * Returns one or more tokens represented by a specified list of names.
 */
@interface TMBUtilities_TokenStorage_GetTokenDetails : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_TokenStorage_GetTokenDetails_Inputs*)newInputSet;
@end

/*! group TMBUtilities.TokenStorage_GetTokenDetails Choreo */


/*! group TMBUtilities.TokenStorage.IsLocked Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the IsLocked Choreo.
 */
@interface TMBUtilities_TokenStorage_IsLocked_Inputs : TMBChoreographyInputSet
	-(void)setName:(NSString*)Name;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the IsLocked Choreo.
 */
@interface TMBUtilities_TokenStorage_IsLocked_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getLocked;	
@end

/*!
 * Returns true or false depending on whether the token is locked or not.
 */
@interface TMBUtilities_TokenStorage_IsLocked : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_TokenStorage_IsLocked_Inputs*)newInputSet;
@end

/*! group TMBUtilities.TokenStorage_IsLocked Choreo */


/*! group TMBUtilities.TokenStorage.IsValid Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the IsValid Choreo.
 */
@interface TMBUtilities_TokenStorage_IsValid_Inputs : TMBChoreographyInputSet
	-(void)setName:(NSString*)Name;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the IsValid Choreo.
 */
@interface TMBUtilities_TokenStorage_IsValid_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getValid;	
@end

/*!
 * Returns true or false depending on whether the token is valid or not.
 */
@interface TMBUtilities_TokenStorage_IsValid : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_TokenStorage_IsValid_Inputs*)newInputSet;
@end

/*! group TMBUtilities.TokenStorage_IsValid Choreo */


/*! group TMBUtilities.TokenStorage.LockToken Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the LockToken Choreo.
 */
@interface TMBUtilities_TokenStorage_LockToken_Inputs : TMBChoreographyInputSet
	-(void)setName:(NSString*)Name;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the LockToken Choreo.
 */
@interface TMBUtilities_TokenStorage_LockToken_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getToken;	
@end

/*!
 * Locks a specified token.
 */
@interface TMBUtilities_TokenStorage_LockToken : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_TokenStorage_LockToken_Inputs*)newInputSet;
@end

/*! group TMBUtilities.TokenStorage_LockToken Choreo */


/*! group TMBUtilities.TokenStorage.RetrieveToken Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the RetrieveToken Choreo.
 */
@interface TMBUtilities_TokenStorage_RetrieveToken_Inputs : TMBChoreographyInputSet
	-(void)setLockToken:(NSString*)LockToken;
	-(void)setName:(NSString*)Name;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the RetrieveToken Choreo.
 */
@interface TMBUtilities_TokenStorage_RetrieveToken_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getToken;
	-(NSString*)getLocked;
	-(NSString*)getValid;	
@end

/*!
 * Retrieves a specified token.
 */
@interface TMBUtilities_TokenStorage_RetrieveToken : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_TokenStorage_RetrieveToken_Inputs*)newInputSet;
@end

/*! group TMBUtilities.TokenStorage_RetrieveToken Choreo */


/*! group TMBUtilities.TokenStorage.SetValid Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the SetValid Choreo.
 */
@interface TMBUtilities_TokenStorage_SetValid_Inputs : TMBChoreographyInputSet
	-(void)setName:(NSString*)Name;
	-(void)setValid:(NSString*)Valid;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SetValid Choreo.
 */
@interface TMBUtilities_TokenStorage_SetValid_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getUpdated;	
@end

/*!
 * Sets a specified token as valid or invalid.
 */
@interface TMBUtilities_TokenStorage_SetValid : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_TokenStorage_SetValid_Inputs*)newInputSet;
@end

/*! group TMBUtilities.TokenStorage_SetValid Choreo */


/*! group TMBUtilities.TokenStorage.StoreToken Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the StoreToken Choreo.
 */
@interface TMBUtilities_TokenStorage_StoreToken_Inputs : TMBChoreographyInputSet
	-(void)setExpires:(NSString*)Expires;
	-(void)setName:(NSString*)Name;
	-(void)setValue:(NSString*)Value;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the StoreToken Choreo.
 */
@interface TMBUtilities_TokenStorage_StoreToken_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getUpdated;	
@end

/*!
 * Stores a token.
 */
@interface TMBUtilities_TokenStorage_StoreToken : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_TokenStorage_StoreToken_Inputs*)newInputSet;
@end

/*! group TMBUtilities.TokenStorage_StoreToken Choreo */


/*! group TMBUtilities.Validation.CanadianPostalCodes Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the CanadianPostalCodes Choreo.
 */
@interface TMBUtilities_Validation_CanadianPostalCodes_Inputs : TMBChoreographyInputSet
	-(void)setZipCode:(NSString*)ZipCode;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CanadianPostalCodes Choreo.
 */
@interface TMBUtilities_Validation_CanadianPostalCodes_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getMatch;	
@end

/*!
 * Verifies that a given zip code matches the format expected for Canadian addresses.
 */
@interface TMBUtilities_Validation_CanadianPostalCodes : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Validation_CanadianPostalCodes_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Validation_CanadianPostalCodes Choreo */


/*! group TMBUtilities.Validation.DutchPostalCodes Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the DutchPostalCodes Choreo.
 */
@interface TMBUtilities_Validation_DutchPostalCodes_Inputs : TMBChoreographyInputSet
	-(void)setZipCode:(NSString*)ZipCode;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DutchPostalCodes Choreo.
 */
@interface TMBUtilities_Validation_DutchPostalCodes_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getMatch;	
@end

/*!
 * Verifies that a given zip code matches the format expected for Dutch addresses.
 */
@interface TMBUtilities_Validation_DutchPostalCodes : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Validation_DutchPostalCodes_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Validation_DutchPostalCodes Choreo */


/*! group TMBUtilities.Validation.EmailAddress Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the EmailAddress Choreo.
 */
@interface TMBUtilities_Validation_EmailAddress_Inputs : TMBChoreographyInputSet
	-(void)setEmailAddress:(NSString*)EmailAddress;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the EmailAddress Choreo.
 */
@interface TMBUtilities_Validation_EmailAddress_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getMatch;	
@end

/*!
 * Verifies that a given email address matches an expected standard pattern.
 */
@interface TMBUtilities_Validation_EmailAddress : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Validation_EmailAddress_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Validation_EmailAddress Choreo */


/*! group TMBUtilities.Validation.GermanPostalCodes Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the GermanPostalCodes Choreo.
 */
@interface TMBUtilities_Validation_GermanPostalCodes_Inputs : TMBChoreographyInputSet
	-(void)setZipCode:(NSString*)ZipCode;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GermanPostalCodes Choreo.
 */
@interface TMBUtilities_Validation_GermanPostalCodes_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getMatch;	
@end

/*!
 * Verifies that a given zip code matches the format expected for German addresses.
 */
@interface TMBUtilities_Validation_GermanPostalCodes : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Validation_GermanPostalCodes_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Validation_GermanPostalCodes Choreo */


/*! group TMBUtilities.Validation.JSON Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the JSON Choreo.
 */
@interface TMBUtilities_Validation_JSON_Inputs : TMBChoreographyInputSet
	-(void)setJSON:(NSString*)JSON;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the JSON Choreo.
 */
@interface TMBUtilities_Validation_JSON_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResult;	
@end

/*!
 * Determines if a specified JSON string is well-formed.
 */
@interface TMBUtilities_Validation_JSON : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Validation_JSON_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Validation_JSON Choreo */


/*! group TMBUtilities.Validation.PasswordCriteria Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the PasswordCriteria Choreo.
 */
@interface TMBUtilities_Validation_PasswordCriteria_Inputs : TMBChoreographyInputSet
	-(void)setMaxLength:(NSString*)MaxLength;
	-(void)setMinLength:(NSString*)MinLength;
	-(void)setPassword:(NSString*)Password;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the PasswordCriteria Choreo.
 */
@interface TMBUtilities_Validation_PasswordCriteria_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getMatch;	
@end

/*!
 * Verifies that a given password matches a standard pattern for passwords.
 */
@interface TMBUtilities_Validation_PasswordCriteria : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Validation_PasswordCriteria_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Validation_PasswordCriteria Choreo */


/*! group TMBUtilities.Validation.RussianPostalCodes Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the RussianPostalCodes Choreo.
 */
@interface TMBUtilities_Validation_RussianPostalCodes_Inputs : TMBChoreographyInputSet
	-(void)setZipCode:(NSString*)ZipCode;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the RussianPostalCodes Choreo.
 */
@interface TMBUtilities_Validation_RussianPostalCodes_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getMatch;	
@end

/*!
 * Verifies that a given zip code matches the format expected for Russian addresses.
 */
@interface TMBUtilities_Validation_RussianPostalCodes : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Validation_RussianPostalCodes_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Validation_RussianPostalCodes Choreo */


/*! group TMBUtilities.Validation.UKPostalCodes Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the UKPostalCodes Choreo.
 */
@interface TMBUtilities_Validation_UKPostalCodes_Inputs : TMBChoreographyInputSet
	-(void)setZipCode:(NSString*)ZipCode;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UKPostalCodes Choreo.
 */
@interface TMBUtilities_Validation_UKPostalCodes_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getMatch;	
@end

/*!
 * Verifies that a given zip code matches the format expected for UK addresses.
 */
@interface TMBUtilities_Validation_UKPostalCodes : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Validation_UKPostalCodes_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Validation_UKPostalCodes Choreo */


/*! group TMBUtilities.Validation.USPostalCodes Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the USPostalCodes Choreo.
 */
@interface TMBUtilities_Validation_USPostalCodes_Inputs : TMBChoreographyInputSet
	-(void)setZipCode:(NSString*)ZipCode;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the USPostalCodes Choreo.
 */
@interface TMBUtilities_Validation_USPostalCodes_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getMatch;	
@end

/*!
 * Verifies that a given zip code matches the format expected for US addresses.
 */
@interface TMBUtilities_Validation_USPostalCodes : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Validation_USPostalCodes_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Validation_USPostalCodes Choreo */


/*! group TMBUtilities.Validation.XML Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the XML Choreo.
 */
@interface TMBUtilities_Validation_XML_Inputs : TMBChoreographyInputSet
	-(void)setXMLFile:(NSString*)XMLFile;
	-(void)setXSDFile:(NSString*)XSDFile;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the XML Choreo.
 */
@interface TMBUtilities_Validation_XML_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getError;
	-(NSString*)getResult;	
@end

/*!
 * Validates XML for basic well-formedness and allows you to check XML against a specified XSD schema file.
 */
@interface TMBUtilities_Validation_XML : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_Validation_XML_Inputs*)newInputSet;
@end

/*! group TMBUtilities.Validation_XML Choreo */


/*! group TMBUtilities.XML.GetValuesFromXML Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the GetValuesFromXML Choreo.
 */
@interface TMBUtilities_XML_GetValuesFromXML_Inputs : TMBChoreographyInputSet
	-(void)setNode:(NSString*)Node;
	-(void)setResponseFormat:(NSString*)ResponseFormat;
	-(void)setXML:(NSString*)XML;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetValuesFromXML Choreo.
 */
@interface TMBUtilities_XML_GetValuesFromXML_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResult;	
@end

/*!
 * Returns all element or attribute values with a specified name.
 */
@interface TMBUtilities_XML_GetValuesFromXML : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_XML_GetValuesFromXML_Inputs*)newInputSet;
@end

/*! group TMBUtilities.XML_GetValuesFromXML Choreo */


/*! group TMBUtilities.XML.RunXPathQuery Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the RunXPathQuery Choreo.
 */
@interface TMBUtilities_XML_RunXPathQuery_Inputs : TMBChoreographyInputSet
	-(void)setMode:(NSString*)Mode;
	-(void)setResponseFormat:(NSString*)ResponseFormat;
	-(void)setXML:(NSString*)XML;
	-(void)setXPath:(NSString*)XPath;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the RunXPathQuery Choreo.
 */
@interface TMBUtilities_XML_RunXPathQuery_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResult;	
@end

/*!
 * Executes an XPath query against a specified XML file and returns the result in CSV or JSON format.
 */
@interface TMBUtilities_XML_RunXPathQuery : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBUtilities_XML_RunXPathQuery_Inputs*)newInputSet;
@end

/*! group TMBUtilities.XML_RunXPathQuery Choreo */
