/*!
 * @TMBMixpanel.m
 *
 * Execute Choreographies from the Temboo Mixpanel bundle.
 *
 * iOS version 10.6.8
 *
 * LICENSE: Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @category   Services
 * @author     Reid Simonton, Temboo, Inc.
 * @copyright  2013 Temboo, Inc.
 * @license    http://www.apache.org/licenses/LICENSE-2.0 Apache License 2.0
 * @version    1.7
 */

#import "TMBMixpanel.h"
#import "TMBChoreography_Protected.h"
#import "TMBChoreographyExecution.h"
#import "TMBTembooSession.h"
#import "TMBTembooUri.h"


/*!
 * Input object with appropriate setters for specifying arguments to the EventData Choreo.
 */
@implementation TMBMixpanel_DataExport_Events_EventData_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) The API Key provided my Mixpanel. You can find your Mixpanel API Key in the project settings dialog in the Mixpanel app.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) The API Secret provided by Mixpanel. You can find your Mixpanel API Secret in the project settings dialog in the Mixpanel app.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the EventNames input for this Choreo.
	*(required, json) A JSON array containing the event or events you wish to get data for.
	 */
	-(void)setEventNames:(NSString*)EventNames {
		[super setInput:@"EventNames" toValue:EventNames];
	}

	/*!
	 * Set the value of the Expire input for this Choreo.
	*(optional, integer) The amount of minutes past NOW() before the request will expire. Defaults to 1.
	 */
	-(void)setExpire:(NSString*)Expire {
		[super setInput:@"Expire" toValue:Expire];
	}

	/*!
	 * Set the value of the Interval input for this Choreo.
	*(required, integer) The time interval to return. This relates to the value provided for Unit.
	 */
	-(void)setInterval:(NSString*)Interval {
		[super setInput:@"Interval" toValue:Interval];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: json (the default) and csv.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Type input for this Choreo.
	*(required, string) The analysis type you would like to get data for. Valid values are: general, unique, or average
	 */
	-(void)setType:(NSString*)Type {
		[super setInput:@"Type" toValue:Type];
	}

	/*!
	 * Set the value of the Unit input for this Choreo.
	*(required, string) The granularity of the data to return. Valid values are: minute, hour, day, week, or month.
	 */
	-(void)setUnit:(NSString*)Unit {
		[super setInput:@"Unit" toValue:Unit];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the EventData Choreo.
 */
@implementation TMBMixpanel_DataExport_Events_EventData_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Mixpanel.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Gets unique, total, or average data for a set of events over the last N days, weeks, or months.
 */
@implementation TMBMixpanel_DataExport_Events_EventData

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBMixpanel_DataExport_Events_EventData Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Mixpanel/DataExport/Events/EventData"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBMixpanel_DataExport_Events_EventData_ResultSet *results = [[TMBMixpanel_DataExport_Events_EventData_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the EventData Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBMixpanel_DataExport_Events_EventData_Inputs*)newInputSet {
		return [[TMBMixpanel_DataExport_Events_EventData_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the TopEvents Choreo.
 */
@implementation TMBMixpanel_DataExport_Events_TopEvents_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) The API Key provided my Mixpanel. You can find your Mixpanel API Key in the project settings dialog in the Mixpanel app.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) The API Secret provided by Mixpanel. You can find your Mixpanel API Secret in the project settings dialog in the Mixpanel app.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the Expire input for this Choreo.
	*(optional, integer) The amount of minutes past NOW() before the request will expire. Defaults to 1.
	 */
	-(void)setExpire:(NSString*)Expire {
		[super setInput:@"Expire" toValue:Expire];
	}

	/*!
	 * Set the value of the Limit input for this Choreo.
	*(optional, integer) The maximum number of events to return. Defaults to 100 (the max the limit).
	 */
	-(void)setLimit:(NSString*)Limit {
		[super setInput:@"Limit" toValue:Limit];
	}

	/*!
	 * Set the value of the Type input for this Choreo.
	*(required, string) The analysis type you would like to get data for. Valid values are: general, unique, or average
	 */
	-(void)setType:(NSString*)Type {
		[super setInput:@"Type" toValue:Type];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the TopEvents Choreo.
 */
@implementation TMBMixpanel_DataExport_Events_TopEvents_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Mixpanel.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Gets the top events for today, with their counts and the normalized percent change from yesterday.
 */
@implementation TMBMixpanel_DataExport_Events_TopEvents

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBMixpanel_DataExport_Events_TopEvents Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Mixpanel/DataExport/Events/TopEvents"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBMixpanel_DataExport_Events_TopEvents_ResultSet *results = [[TMBMixpanel_DataExport_Events_TopEvents_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the TopEvents Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBMixpanel_DataExport_Events_TopEvents_Inputs*)newInputSet {
		return [[TMBMixpanel_DataExport_Events_TopEvents_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the TopNames Choreo.
 */
@implementation TMBMixpanel_DataExport_Events_TopNames_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) The API Key provided my Mixpanel. You can find your Mixpanel API Key in the project settings dialog in the Mixpanel app.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) The API Secret provided by Mixpanel. You can find your Mixpanel API Secret in the project settings dialog in the Mixpanel app.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the Expire input for this Choreo.
	*(optional, integer) The amount of minutes past NOW() before the request will expire. Defaults to 1.
	 */
	-(void)setExpire:(NSString*)Expire {
		[super setInput:@"Expire" toValue:Expire];
	}

	/*!
	 * Set the value of the Limit input for this Choreo.
	*(optional, integer) The maximum number of events to return. Defaults to 100 (the max the limit).
	 */
	-(void)setLimit:(NSString*)Limit {
		[super setInput:@"Limit" toValue:Limit];
	}

	/*!
	 * Set the value of the Type input for this Choreo.
	*(required, string) The analysis type you would like to get data for. Valid values are: general, unique, or average
	 */
	-(void)setType:(NSString*)Type {
		[super setInput:@"Type" toValue:Type];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the TopNames Choreo.
 */
@implementation TMBMixpanel_DataExport_Events_TopNames_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Mixpanel.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Gets a list of the most common events over the last 31 days.
 */
@implementation TMBMixpanel_DataExport_Events_TopNames

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBMixpanel_DataExport_Events_TopNames Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Mixpanel/DataExport/Events/TopNames"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBMixpanel_DataExport_Events_TopNames_ResultSet *results = [[TMBMixpanel_DataExport_Events_TopNames_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the TopNames Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBMixpanel_DataExport_Events_TopNames_Inputs*)newInputSet {
		return [[TMBMixpanel_DataExport_Events_TopNames_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the FunnelData Choreo.
 */
@implementation TMBMixpanel_DataExport_Funnels_FunnelData_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) The API Key provided my Mixpanel. You can find your Mixpanel API Key in the project settings dialog in the Mixpanel app.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) The API Secret provided by Mixpanel. You can find your Mixpanel API Secret in the project settings dialog in the Mixpanel app.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the Expire input for this Choreo.
	*(optional, integer) The amount of minutes past NOW() before the request will expire. Defaults to 1.
	 */
	-(void)setExpire:(NSString*)Expire {
		[super setInput:@"Expire" toValue:Expire];
	}

	/*!
	 * Set the value of the FromDate input for this Choreo.
	*(optional, date) The first date in yyyy-mm-dd format from which a user can begin the first step in the funnel. This date is inclusive.
	 */
	-(void)setFromDate:(NSString*)FromDate {
		[super setInput:@"FromDate" toValue:FromDate];
	}

	/*!
	 * Set the value of the FunnelID input for this Choreo.
	*(required, string) The ID of the funnel to get data for.
	 */
	-(void)setFunnelID:(NSString*)FunnelID {
		[super setInput:@"FunnelID" toValue:FunnelID];
	}

	/*!
	 * Set the value of the Interval input for this Choreo.
	*(optional, integer) The number of days you want your results bucketed into.The default value is 1.
	 */
	-(void)setInterval:(NSString*)Interval {
		[super setInput:@"Interval" toValue:Interval];
	}

	/*!
	 * Set the value of the Length input for this Choreo.
	*(optional, integer) The number of days each user has to complete the funnel, starting from the time they triggered the first step in the funnel. May not be greater than 60 days. Defaults to 14.
	 */
	-(void)setLength:(NSString*)Length {
		[super setInput:@"Length" toValue:Length];
	}

	/*!
	 * Set the value of the Limit input for this Choreo.
	*(optional, integer) Return the top limit property values. This parameter is ignored if the On input is not specified.
	 */
	-(void)setLimit:(NSString*)Limit {
		[super setInput:@"Limit" toValue:Limit];
	}

	/*!
	 * Set the value of the On input for this Choreo.
	*(optional, string) The property expression to segment the event on (e.g., properties["Referred By"] == "Friend"). See Choreo description for examples.
	 */
	-(void)setOn:(NSString*)On {
		[super setInput:@"On" toValue:On];
	}

	/*!
	 * Set the value of the ToDate input for this Choreo.
	*(optional, date) The last date in yyyy-mm-dd format from which a user can begin the first step in the funnel. This date is inclusive. The date range may not be more than 60 days.
	 */
	-(void)setToDate:(NSString*)ToDate {
		[super setInput:@"ToDate" toValue:ToDate];
	}

	/*!
	 * Set the value of the Unit input for this Choreo.
	*(optional, string) This is an alternate way of specifying interval and can set to be 'day' or 'week'.
	 */
	-(void)setUnit:(NSString*)Unit {
		[super setInput:@"Unit" toValue:Unit];
	}

	/*!
	 * Set the value of the Where input for this Choreo.
	*(optional, string) An expression to filter events by  (e.g., properties["Signed Up"]). See Choreo description for examples.
	 */
	-(void)setWhere:(NSString*)Where {
		[super setInput:@"Where" toValue:Where];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the FunnelData Choreo.
 */
@implementation TMBMixpanel_DataExport_Funnels_FunnelData_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Mixpanel.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Gets data for a specified funnel.
 */
@implementation TMBMixpanel_DataExport_Funnels_FunnelData

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBMixpanel_DataExport_Funnels_FunnelData Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Mixpanel/DataExport/Funnels/FunnelData"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBMixpanel_DataExport_Funnels_FunnelData_ResultSet *results = [[TMBMixpanel_DataExport_Funnels_FunnelData_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the FunnelData Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBMixpanel_DataExport_Funnels_FunnelData_Inputs*)newInputSet {
		return [[TMBMixpanel_DataExport_Funnels_FunnelData_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the FunnelList Choreo.
 */
@implementation TMBMixpanel_DataExport_Funnels_FunnelList_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) The API Key provided my Mixpanel. You can find your Mixpanel API Key in the project settings dialog in the Mixpanel app.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) The API Secret provided by Mixpanel. You can find your Mixpanel API Secret in the project settings dialog in the Mixpanel app.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the Expire input for this Choreo.
	*(optional, integer) The amount of minutes past NOW() before the request will expire. Defaults to 1.
	 */
	-(void)setExpire:(NSString*)Expire {
		[super setInput:@"Expire" toValue:Expire];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the FunnelList Choreo.
 */
@implementation TMBMixpanel_DataExport_Funnels_FunnelList_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Mixpanel.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Gets the names and funnel_ids of your funnels.
 */
@implementation TMBMixpanel_DataExport_Funnels_FunnelList

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBMixpanel_DataExport_Funnels_FunnelList Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Mixpanel/DataExport/Funnels/FunnelList"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBMixpanel_DataExport_Funnels_FunnelList_ResultSet *results = [[TMBMixpanel_DataExport_Funnels_FunnelList_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the FunnelList Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBMixpanel_DataExport_Funnels_FunnelList_Inputs*)newInputSet {
		return [[TMBMixpanel_DataExport_Funnels_FunnelList_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Engage Choreo.
 */
@implementation TMBMixpanel_DataExport_People_Engage_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) The API Key provided my Mixpanel. You can find your Mixpanel API Key in the project settings dialog in the Mixpanel app.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) The API Secret provided by Mixpanel. You can find your Mixpanel API Secret in the project settings dialog in the Mixpanel app.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the Expire input for this Choreo.
	*(optional, integer) The amount of minutes past NOW() before the request will expire. Defaults to 1.
	 */
	-(void)setExpire:(NSString*)Expire {
		[super setInput:@"Expire" toValue:Expire];
	}

	/*!
	 * Set the value of the Page input for this Choreo.
	*(optional, integer) Which page of the results to retrieve. Pages start at zero. If the "page" parameter is provided, the session_id parameter must also be provided.
	 */
	-(void)setPage:(NSString*)Page {
		[super setInput:@"Page" toValue:Page];
	}

	/*!
	 * Set the value of the SessionID input for this Choreo.
	*(optional, string) A string id provided in the results of a previous query. Using a session_id speeds up api response, and allows paging through results.
	 */
	-(void)setSessionID:(NSString*)SessionID {
		[super setInput:@"SessionID" toValue:SessionID];
	}

	/*!
	 * Set the value of the Where input for this Choreo.
	*(optional, string) An expression to filter people by (e.g., properties["time"]). See Choreo description for examples.
	 */
	-(void)setWhere:(NSString*)Where {
		[super setInput:@"Where" toValue:Where];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Engage Choreo.
 */
@implementation TMBMixpanel_DataExport_People_Engage_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Mixpanel.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Queries Mixpanel for data about people.
 */
@implementation TMBMixpanel_DataExport_People_Engage

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBMixpanel_DataExport_People_Engage Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Mixpanel/DataExport/People/Engage"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBMixpanel_DataExport_People_Engage_ResultSet *results = [[TMBMixpanel_DataExport_People_Engage_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Engage Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBMixpanel_DataExport_People_Engage_Inputs*)newInputSet {
		return [[TMBMixpanel_DataExport_People_Engage_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the PropertyData Choreo.
 */
@implementation TMBMixpanel_DataExport_Properties_PropertyData_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) The API Key provided my Mixpanel. You can find your Mixpanel API Key in the project settings dialog in the Mixpanel app.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) The API Secret provided by Mixpanel. You can find your Mixpanel API Secret in the project settings dialog in the Mixpanel app.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the EventName input for this Choreo.
	*(required, string) The name of the event that you wish to get data for.
	 */
	-(void)setEventName:(NSString*)EventName {
		[super setInput:@"EventName" toValue:EventName];
	}

	/*!
	 * Set the value of the Expire input for this Choreo.
	*(optional, integer) The amount of minutes past NOW() before the request will expire. Defaults to 1.
	 */
	-(void)setExpire:(NSString*)Expire {
		[super setInput:@"Expire" toValue:Expire];
	}

	/*!
	 * Set the value of the Interval input for this Choreo.
	*(required, integer) The time interval to return. This relates to the value provided for Unit.
	 */
	-(void)setInterval:(NSString*)Interval {
		[super setInput:@"Interval" toValue:Interval];
	}

	/*!
	 * Set the value of the Limit input for this Choreo.
	*(optional, integer) The maximum number of values to return. Defaults to 255.
	 */
	-(void)setLimit:(NSString*)Limit {
		[super setInput:@"Limit" toValue:Limit];
	}

	/*!
	 * Set the value of the PropertyName input for this Choreo.
	*(required, string) The name of the property you would like to get data for.
	 */
	-(void)setPropertyName:(NSString*)PropertyName {
		[super setInput:@"PropertyName" toValue:PropertyName];
	}

	/*!
	 * Set the value of the PropertyValues input for this Choreo.
	*(optional, json) A JSON array containing property values that you wish to retrieve.
	 */
	-(void)setPropertyValues:(NSString*)PropertyValues {
		[super setInput:@"PropertyValues" toValue:PropertyValues];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Valid values are: json (the default) and csv.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Type input for this Choreo.
	*(required, string) The analysis type you would like to get data for. Valid values are: general, unique, or average
	 */
	-(void)setType:(NSString*)Type {
		[super setInput:@"Type" toValue:Type];
	}

	/*!
	 * Set the value of the Unit input for this Choreo.
	*(required, string) The granularity of the data to return. Valid values are: minute, hour, day, week, or month.
	 */
	-(void)setUnit:(NSString*)Unit {
		[super setInput:@"Unit" toValue:Unit];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the PropertyData Choreo.
 */
@implementation TMBMixpanel_DataExport_Properties_PropertyData_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Mixpanel.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Gets unique, total, or average data for of a single event and property over the last N days, weeks, or months.
 */
@implementation TMBMixpanel_DataExport_Properties_PropertyData

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBMixpanel_DataExport_Properties_PropertyData Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Mixpanel/DataExport/Properties/PropertyData"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBMixpanel_DataExport_Properties_PropertyData_ResultSet *results = [[TMBMixpanel_DataExport_Properties_PropertyData_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the PropertyData Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBMixpanel_DataExport_Properties_PropertyData_Inputs*)newInputSet {
		return [[TMBMixpanel_DataExport_Properties_PropertyData_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the TopProperties Choreo.
 */
@implementation TMBMixpanel_DataExport_Properties_TopProperties_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) The API Key provided my Mixpanel. You can find your Mixpanel API Key in the project settings dialog in the Mixpanel app.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) The API Secret provided by Mixpanel. You can find your Mixpanel API Secret in the project settings dialog in the Mixpanel app.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the EventName input for this Choreo.
	*(required, string) The name of the event that you wish to get data for.
	 */
	-(void)setEventName:(NSString*)EventName {
		[super setInput:@"EventName" toValue:EventName];
	}

	/*!
	 * Set the value of the Expire input for this Choreo.
	*(optional, integer) The amount of minutes past NOW() before the request will expire. Defaults to 1.
	 */
	-(void)setExpire:(NSString*)Expire {
		[super setInput:@"Expire" toValue:Expire];
	}

	/*!
	 * Set the value of the Limit input for this Choreo.
	*(optional, integer) The maximum number of properties to return. Defaults to 10.
	 */
	-(void)setLimit:(NSString*)Limit {
		[super setInput:@"Limit" toValue:Limit];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the TopProperties Choreo.
 */
@implementation TMBMixpanel_DataExport_Properties_TopProperties_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Mixpanel.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Gets the top property names for an event.
 */
@implementation TMBMixpanel_DataExport_Properties_TopProperties

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBMixpanel_DataExport_Properties_TopProperties Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Mixpanel/DataExport/Properties/TopProperties"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBMixpanel_DataExport_Properties_TopProperties_ResultSet *results = [[TMBMixpanel_DataExport_Properties_TopProperties_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the TopProperties Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBMixpanel_DataExport_Properties_TopProperties_Inputs*)newInputSet {
		return [[TMBMixpanel_DataExport_Properties_TopProperties_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the TopValues Choreo.
 */
@implementation TMBMixpanel_DataExport_Properties_TopValues_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) The API Key provided my Mixpanel. You can find your Mixpanel API Key in the project settings dialog in the Mixpanel app.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) The API Secret provided by Mixpanel. You can find your Mixpanel API Secret in the project settings dialog in the Mixpanel app.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the EventName input for this Choreo.
	*(required, string) The name of the event that you wish to get data for.
	 */
	-(void)setEventName:(NSString*)EventName {
		[super setInput:@"EventName" toValue:EventName];
	}

	/*!
	 * Set the value of the Expire input for this Choreo.
	*(optional, integer) The amount of minutes past NOW() before the request will expire. Defaults to 1.
	 */
	-(void)setExpire:(NSString*)Expire {
		[super setInput:@"Expire" toValue:Expire];
	}

	/*!
	 * Set the value of the Limit input for this Choreo.
	*(optional, integer) The maximum number of values to return. Defaults to 255.
	 */
	-(void)setLimit:(NSString*)Limit {
		[super setInput:@"Limit" toValue:Limit];
	}

	/*!
	 * Set the value of the PropertyName input for this Choreo.
	*(required, string) The name of the property you would like to get data for.
	 */
	-(void)setPropertyName:(NSString*)PropertyName {
		[super setInput:@"PropertyName" toValue:PropertyName];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the TopValues Choreo.
 */
@implementation TMBMixpanel_DataExport_Properties_TopValues_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Mixpanel.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Gets the top values for a property.
 */
@implementation TMBMixpanel_DataExport_Properties_TopValues

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBMixpanel_DataExport_Properties_TopValues Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Mixpanel/DataExport/Properties/TopValues"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBMixpanel_DataExport_Properties_TopValues_ResultSet *results = [[TMBMixpanel_DataExport_Properties_TopValues_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the TopValues Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBMixpanel_DataExport_Properties_TopValues_Inputs*)newInputSet {
		return [[TMBMixpanel_DataExport_Properties_TopValues_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the RetentionData Choreo.
 */
@implementation TMBMixpanel_DataExport_Retention_RetentionData_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) The API Key provided my Mixpanel. You can find your Mixpanel API Key in the project settings dialog in the Mixpanel app.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) The API Secret provided by Mixpanel. You can find your Mixpanel API Secret in the project settings dialog in the Mixpanel app.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the BornEvent input for this Choreo.
	*(conditional, string) The first event a user must do to be counted in a birth retention cohort. Required when retention_type is 'birth'.
	 */
	-(void)setBornEvent:(NSString*)BornEvent {
		[super setInput:@"BornEvent" toValue:BornEvent];
	}

	/*!
	 * Set the value of the BornWhere input for this Choreo.
	*(optional, string) An expression to filter born_events by. See Choreo description for examples.
	 */
	-(void)setBornWhere:(NSString*)BornWhere {
		[super setInput:@"BornWhere" toValue:BornWhere];
	}

	/*!
	 * Set the value of the EventName input for this Choreo.
	*(optional, string) The event to generate returning counts for.
	 */
	-(void)setEventName:(NSString*)EventName {
		[super setInput:@"EventName" toValue:EventName];
	}

	/*!
	 * Set the value of the Expire input for this Choreo.
	*(optional, integer) The amount of minutes past NOW() before the request will expire. Defaults to 1.
	 */
	-(void)setExpire:(NSString*)Expire {
		[super setInput:@"Expire" toValue:Expire];
	}

	/*!
	 * Set the value of the FromDate input for this Choreo.
	*(required, date) The date in yyyy-mm-dd format from which to begin generating cohorts from. This date is inclusive.
	 */
	-(void)setFromDate:(NSString*)FromDate {
		[super setInput:@"FromDate" toValue:FromDate];
	}

	/*!
	 * Set the value of the Interval input for this Choreo.
	*(optional, integer) The number of days you want your results bucketed into.The default value is 1 or specified by unit.
	 */
	-(void)setInterval:(NSString*)Interval {
		[super setInput:@"Interval" toValue:Interval];
	}

	/*!
	 * Set the value of the IntervalCount input for this Choreo.
	*(optional, integer) The number of intervals you want. Defaults to 1.
	 */
	-(void)setIntervalCount:(NSString*)IntervalCount {
		[super setInput:@"IntervalCount" toValue:IntervalCount];
	}

	/*!
	 * Set the value of the Limit input for this Choreo.
	*(optional, integer) Return the top limit segmentation values. This parameter is ignored if the On input is not specified.
	 */
	-(void)setLimit:(NSString*)Limit {
		[super setInput:@"Limit" toValue:Limit];
	}

	/*!
	 * Set the value of the On input for this Choreo.
	*(optional, string) The property expression to segment the second event on. See Choreo description for examples.
	 */
	-(void)setOn:(NSString*)On {
		[super setInput:@"On" toValue:On];
	}

	/*!
	 * Set the value of the RetentionType input for this Choreo.
	*(conditional, string) The type of retention. Valid values are: birth (the default) or compounded.
	 */
	-(void)setRetentionType:(NSString*)RetentionType {
		[super setInput:@"RetentionType" toValue:RetentionType];
	}

	/*!
	 * Set the value of the ToDate input for this Choreo.
	*(required, date) The date in yyyy-mm-dd format from which to stop generating cohorts from. This date is inclusive.
	 */
	-(void)setToDate:(NSString*)ToDate {
		[super setInput:@"ToDate" toValue:ToDate];
	}

	/*!
	 * Set the value of the Unit input for this Choreo.
	*(optional, string) This is an alternate way of specifying interval. Valid values are: day, week, or month.
	 */
	-(void)setUnit:(NSString*)Unit {
		[super setInput:@"Unit" toValue:Unit];
	}

	/*!
	 * Set the value of the Where input for this Choreo.
	*(optional, string) An expression to filter events by  (e.g., properties["Signed Up"]). See Choreo description for examples.
	 */
	-(void)setWhere:(NSString*)Where {
		[super setInput:@"Where" toValue:Where];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the RetentionData Choreo.
 */
@implementation TMBMixpanel_DataExport_Retention_RetentionData_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Mixpanel.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Gets cohort analysis.
 */
@implementation TMBMixpanel_DataExport_Retention_RetentionData

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBMixpanel_DataExport_Retention_RetentionData Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Mixpanel/DataExport/Retention/RetentionData"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBMixpanel_DataExport_Retention_RetentionData_ResultSet *results = [[TMBMixpanel_DataExport_Retention_RetentionData_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the RetentionData Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBMixpanel_DataExport_Retention_RetentionData_Inputs*)newInputSet {
		return [[TMBMixpanel_DataExport_Retention_RetentionData_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Average Choreo.
 */
@implementation TMBMixpanel_DataExport_Segmentation_Average_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) The API Key provided my Mixpanel. You can find your Mixpanel API Key in the project settings dialog in the Mixpanel app.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) The API Secret provided by Mixpanel. You can find your Mixpanel API Secret in the project settings dialog in the Mixpanel app.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the EventName input for this Choreo.
	*(required, string) The event that you wish to segment on.
	 */
	-(void)setEventName:(NSString*)EventName {
		[super setInput:@"EventName" toValue:EventName];
	}

	/*!
	 * Set the value of the Expire input for this Choreo.
	*(optional, integer) The amount of minutes past NOW() before the request will expire. Defaults to 1.
	 */
	-(void)setExpire:(NSString*)Expire {
		[super setInput:@"Expire" toValue:Expire];
	}

	/*!
	 * Set the value of the FromDate input for this Choreo.
	*(required, date) The date in yyyy-mm-dd format from which to begin querying for the event from. This date is inclusive.
	 */
	-(void)setFromDate:(NSString*)FromDate {
		[super setInput:@"FromDate" toValue:FromDate];
	}

	/*!
	 * Set the value of the On input for this Choreo.
	*(required, string) The expression to average per unit time. The result of the expression should be a numeric value (e.g., properties["time"]. See Choreo description for examples.
	 */
	-(void)setOn:(NSString*)On {
		[super setInput:@"On" toValue:On];
	}

	/*!
	 * Set the value of the ToDate input for this Choreo.
	*(required, date) The date in yyyy-mm-dd format from which to stop querying for the event from. This date is inclusive. The date range may not be more than 30 days.
	 */
	-(void)setToDate:(NSString*)ToDate {
		[super setInput:@"ToDate" toValue:ToDate];
	}

	/*!
	 * Set the value of the Unit input for this Choreo.
	*(optional, string) Determines the buckets into which the property values that you segment on are placed. Valid values are: hour or day.
	 */
	-(void)setUnit:(NSString*)Unit {
		[super setInput:@"Unit" toValue:Unit];
	}

	/*!
	 * Set the value of the Where input for this Choreo.
	*(optional, string) An expression to filter events by  (e.g., properties["time"] >= 2000). See Choreo description for examples.
	 */
	-(void)setWhere:(NSString*)Where {
		[super setInput:@"Where" toValue:Where];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Average Choreo.
 */
@implementation TMBMixpanel_DataExport_Segmentation_Average_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Mixpanel.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Averages an expression for events per unit time.
 */
@implementation TMBMixpanel_DataExport_Segmentation_Average

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBMixpanel_DataExport_Segmentation_Average Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Mixpanel/DataExport/Segmentation/Average"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBMixpanel_DataExport_Segmentation_Average_ResultSet *results = [[TMBMixpanel_DataExport_Segmentation_Average_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Average Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBMixpanel_DataExport_Segmentation_Average_Inputs*)newInputSet {
		return [[TMBMixpanel_DataExport_Segmentation_Average_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Numeric Choreo.
 */
@implementation TMBMixpanel_DataExport_Segmentation_Numeric_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) The API Key provided my Mixpanel. You can find your Mixpanel API Key in the project settings dialog in the Mixpanel app.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) The API Secret provided by Mixpanel. You can find your Mixpanel API Secret in the project settings dialog in the Mixpanel app.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the Buckets input for this Choreo.
	*(required, integer) The number of buckets that you wish to divide the numeric values up into. Bucket ranges are automatically computed based on the maximum and minimum of the 'On' expression.
	 */
	-(void)setBuckets:(NSString*)Buckets {
		[super setInput:@"Buckets" toValue:Buckets];
	}

	/*!
	 * Set the value of the EventName input for this Choreo.
	*(required, string) The event that you wish to segment on.
	 */
	-(void)setEventName:(NSString*)EventName {
		[super setInput:@"EventName" toValue:EventName];
	}

	/*!
	 * Set the value of the Expire input for this Choreo.
	*(optional, integer) The amount of minutes past NOW() before the request will expire. Defaults to 1.
	 */
	-(void)setExpire:(NSString*)Expire {
		[super setInput:@"Expire" toValue:Expire];
	}

	/*!
	 * Set the value of the FromDate input for this Choreo.
	*(required, date) The date in yyyy-mm-dd format from which to begin querying for the event from. This date is inclusive.
	 */
	-(void)setFromDate:(NSString*)FromDate {
		[super setInput:@"FromDate" toValue:FromDate];
	}

	/*!
	 * Set the value of the On input for this Choreo.
	*(required, string) The property expression to segment the event on. Must be a numeric expression (e.g., number(properties["time"]). See Choreo description for examples.
	 */
	-(void)setOn:(NSString*)On {
		[super setInput:@"On" toValue:On];
	}

	/*!
	 * Set the value of the ToDate input for this Choreo.
	*(required, date) The date in yyyy-mm-dd format from which to stop querying for the event from. This date is inclusive.
	 */
	-(void)setToDate:(NSString*)ToDate {
		[super setInput:@"ToDate" toValue:ToDate];
	}

	/*!
	 * Set the value of the Type input for this Choreo.
	*(optional, string) The analysis type you would like to get data for. Valid values are: general, unique, or average
	 */
	-(void)setType:(NSString*)Type {
		[super setInput:@"Type" toValue:Type];
	}

	/*!
	 * Set the value of the Unit input for this Choreo.
	*(optional, string) Determines the buckets into which the property values that you segment on are placed. Valid values are: hour or day.
	 */
	-(void)setUnit:(NSString*)Unit {
		[super setInput:@"Unit" toValue:Unit];
	}

	/*!
	 * Set the value of the Where input for this Choreo.
	*(optional, string) An expression to filter events by  (e.g., number(properties["time"]) >= 2000). See Choreo description for examples.
	 */
	-(void)setWhere:(NSString*)Where {
		[super setInput:@"Where" toValue:Where];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Numeric Choreo.
 */
@implementation TMBMixpanel_DataExport_Segmentation_Numeric_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Mixpanel.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Gets data for an event, segmented and filtered by properties, with values placed into numeric buckets.
 */
@implementation TMBMixpanel_DataExport_Segmentation_Numeric

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBMixpanel_DataExport_Segmentation_Numeric Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Mixpanel/DataExport/Segmentation/Numeric"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBMixpanel_DataExport_Segmentation_Numeric_ResultSet *results = [[TMBMixpanel_DataExport_Segmentation_Numeric_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Numeric Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBMixpanel_DataExport_Segmentation_Numeric_Inputs*)newInputSet {
		return [[TMBMixpanel_DataExport_Segmentation_Numeric_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SegmentationData Choreo.
 */
@implementation TMBMixpanel_DataExport_Segmentation_SegmentationData_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) The API Key provided my Mixpanel. You can find your Mixpanel API Key in the project settings dialog in the Mixpanel app.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) The API Secret provided by Mixpanel. You can find your Mixpanel API Secret in the project settings dialog in the Mixpanel app.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the EventName input for this Choreo.
	*(required, string) The event that you wish to segment on.
	 */
	-(void)setEventName:(NSString*)EventName {
		[super setInput:@"EventName" toValue:EventName];
	}

	/*!
	 * Set the value of the Expire input for this Choreo.
	*(optional, integer) The amount of minutes past NOW() before the request will expire. Defaults to 1.
	 */
	-(void)setExpire:(NSString*)Expire {
		[super setInput:@"Expire" toValue:Expire];
	}

	/*!
	 * Set the value of the FromDate input for this Choreo.
	*(required, date) The date in yyyy-mm-dd format from which to begin querying for the event from. This date is inclusive.
	 */
	-(void)setFromDate:(NSString*)FromDate {
		[super setInput:@"FromDate" toValue:FromDate];
	}

	/*!
	 * Set the value of the Limit input for this Choreo.
	*(optional, integer) Return the top limit property values. This parameter is ignored if the On input is not specified.
	 */
	-(void)setLimit:(NSString*)Limit {
		[super setInput:@"Limit" toValue:Limit];
	}

	/*!
	 * Set the value of the On input for this Choreo.
	*(optional, string) The property expression to segment the event on (e.g., properties["Referred By"] == "Friend"). See Choreo description for examples.
	 */
	-(void)setOn:(NSString*)On {
		[super setInput:@"On" toValue:On];
	}

	/*!
	 * Set the value of the ToDate input for this Choreo.
	*(required, date) The date in yyyy-mm-dd format from which to stop querying for the event from. This date is inclusive.
	 */
	-(void)setToDate:(NSString*)ToDate {
		[super setInput:@"ToDate" toValue:ToDate];
	}

	/*!
	 * Set the value of the Type input for this Choreo.
	*(optional, string) The analysis type you would like to get data for. Valid values are: general, unique, or average
	 */
	-(void)setType:(NSString*)Type {
		[super setInput:@"Type" toValue:Type];
	}

	/*!
	 * Set the value of the Unit input for this Choreo.
	*(optional, string) Determines the buckets into which the property values that you segment on are placed. Valid values are: minute, hour, day (the default), or month.
	 */
	-(void)setUnit:(NSString*)Unit {
		[super setInput:@"Unit" toValue:Unit];
	}

	/*!
	 * Set the value of the Where input for this Choreo.
	*(optional, string) An expression to filter events by  (e.g., properties["Signed Up"]). See Choreo description for examples.
	 */
	-(void)setWhere:(NSString*)Where {
		[super setInput:@"Where" toValue:Where];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SegmentationData Choreo.
 */
@implementation TMBMixpanel_DataExport_Segmentation_SegmentationData_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Mixpanel.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Gets data for an event, segmented and filtered by properties.
 */
@implementation TMBMixpanel_DataExport_Segmentation_SegmentationData

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBMixpanel_DataExport_Segmentation_SegmentationData Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Mixpanel/DataExport/Segmentation/SegmentationData"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBMixpanel_DataExport_Segmentation_SegmentationData_ResultSet *results = [[TMBMixpanel_DataExport_Segmentation_SegmentationData_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SegmentationData Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBMixpanel_DataExport_Segmentation_SegmentationData_Inputs*)newInputSet {
		return [[TMBMixpanel_DataExport_Segmentation_SegmentationData_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Sum Choreo.
 */
@implementation TMBMixpanel_DataExport_Segmentation_Sum_Inputs

	/*!
	 * Set the value of the APIKey input for this Choreo.
	*(required, string) The API Key provided my Mixpanel. You can find your Mixpanel API Key in the project settings dialog in the Mixpanel app.
	 */
	-(void)setAPIKey:(NSString*)APIKey {
		[super setInput:@"APIKey" toValue:APIKey];
	}

	/*!
	 * Set the value of the APISecret input for this Choreo.
	*(required, string) The API Secret provided by Mixpanel. You can find your Mixpanel API Secret in the project settings dialog in the Mixpanel app.
	 */
	-(void)setAPISecret:(NSString*)APISecret {
		[super setInput:@"APISecret" toValue:APISecret];
	}

	/*!
	 * Set the value of the EventName input for this Choreo.
	*(required, string) The event that you wish to segment on.
	 */
	-(void)setEventName:(NSString*)EventName {
		[super setInput:@"EventName" toValue:EventName];
	}

	/*!
	 * Set the value of the Expire input for this Choreo.
	*(optional, integer) The amount of minutes past NOW() before the request will expire. Defaults to 1.
	 */
	-(void)setExpire:(NSString*)Expire {
		[super setInput:@"Expire" toValue:Expire];
	}

	/*!
	 * Set the value of the FromDate input for this Choreo.
	*(required, date) The date in yyyy-mm-dd format from which to begin querying for the event from. This date is inclusive.
	 */
	-(void)setFromDate:(NSString*)FromDate {
		[super setInput:@"FromDate" toValue:FromDate];
	}

	/*!
	 * Set the value of the On input for this Choreo.
	*(required, string) The expression to sum per unit time. Must be a numeric expression (e.g., number(properties["time"]). See Choreo description for examples.
	 */
	-(void)setOn:(NSString*)On {
		[super setInput:@"On" toValue:On];
	}

	/*!
	 * Set the value of the ToDate input for this Choreo.
	*(required, date) The date in yyyy-mm-dd format from which to stop querying for the event from. This date is inclusive. The date range may not be more than 30 days.
	 */
	-(void)setToDate:(NSString*)ToDate {
		[super setInput:@"ToDate" toValue:ToDate];
	}

	/*!
	 * Set the value of the Unit input for this Choreo.
	*(optional, string) Determines the buckets into which the property values that you segment on are placed. Valid values are: hour or day.
	 */
	-(void)setUnit:(NSString*)Unit {
		[super setInput:@"Unit" toValue:Unit];
	}

	/*!
	 * Set the value of the Where input for this Choreo.
	*(optional, string) An expression to filter events by  (e.g., number(properties["time"]) >= 2000). See Choreo description for examples.
	 */
	-(void)setWhere:(NSString*)Where {
		[super setInput:@"Where" toValue:Where];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Sum Choreo.
 */
@implementation TMBMixpanel_DataExport_Segmentation_Sum_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Mixpanel.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Sums an expression for events per unit time.
 */
@implementation TMBMixpanel_DataExport_Segmentation_Sum

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBMixpanel_DataExport_Segmentation_Sum Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Mixpanel/DataExport/Segmentation/Sum"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBMixpanel_DataExport_Segmentation_Sum_ResultSet *results = [[TMBMixpanel_DataExport_Segmentation_Sum_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Sum Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBMixpanel_DataExport_Segmentation_Sum_Inputs*)newInputSet {
		return [[TMBMixpanel_DataExport_Segmentation_Sum_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Track Choreo.
 */
@implementation TMBMixpanel_Events_Track_Inputs

	/*!
	 * Set the value of the DistinctID input for this Choreo.
	*(optional, string) Used to uniquely identify a user associated with your event. When provided, you can track a given user through funnels and distinguish unique users for retention analyses.
	 */
	-(void)setDistinctID:(NSString*)DistinctID {
		[super setInput:@"DistinctID" toValue:DistinctID];
	}

	/*!
	 * Set the value of the EventName input for this Choreo.
	*(required, string) A name for the event (e.g., Signed Up, Uploaded Photo, etc).
	 */
	-(void)setEventName:(NSString*)EventName {
		[super setInput:@"EventName" toValue:EventName];
	}

	/*!
	 * Set the value of the EventProperties input for this Choreo.
	*(optional, json) Additional properties associated with the event formatted as name/value pairs in a JSON object. These properties can be used for segmentation and funnels.
	 */
	-(void)setEventProperties:(NSString*)EventProperties {
		[super setInput:@"EventProperties" toValue:EventProperties];
	}

	/*!
	 * Set the value of the IP input for this Choreo.
	*(optional, string) An IP address string associated with the event (e.g., 127.0.0.1). When set to 0 (the default) Mixpanel will ignore IP information.
	 */
	-(void)setIP:(NSString*)IP {
		[super setInput:@"IP" toValue:IP];
	}

	/*!
	 * Set the value of the Time input for this Choreo.
	*(optional, date) A unix timestamp representing the time the event occurred. If not provided, Mixpanel will use the time the event arrives at the server.
	 */
	-(void)setTime:(NSString*)Time {
		[super setInput:@"Time" toValue:Time];
	}

	/*!
	 * Set the value of the Token input for this Choreo.
	*(required, string) The token provided by Mixpanel. You can find your Mixpanel token in the project settings dialog in the Mixpanel app.
	 */
	-(void)setToken:(NSString*)Token {
		[super setInput:@"Token" toValue:Token];
	}

	/*!
	 * Set the value of the Verbose input for this Choreo.
	*(optional, boolean) When set to 1, the response will contain more information describing the success or failure of the tracking call.
	 */
	-(void)setVerbose:(NSString*)Verbose {
		[super setInput:@"Verbose" toValue:Verbose];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Track Choreo.
 */
@implementation TMBMixpanel_Events_Track_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Mixpanel.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Records an event in Mixpanel.
 */
@implementation TMBMixpanel_Events_Track

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBMixpanel_Events_Track Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Mixpanel/Events/Track"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBMixpanel_Events_Track_ResultSet *results = [[TMBMixpanel_Events_Track_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Track Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBMixpanel_Events_Track_Inputs*)newInputSet {
		return [[TMBMixpanel_Events_Track_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Add Choreo.
 */
@implementation TMBMixpanel_Profiles_Add_Inputs

	/*!
	 * Set the value of the DistinctID input for this Choreo.
	*(required, string) Used to uniquely identify the profile you want to update.
	 */
	-(void)setDistinctID:(NSString*)DistinctID {
		[super setInput:@"DistinctID" toValue:DistinctID];
	}

	/*!
	 * Set the value of the IP input for this Choreo.
	*(optional, string) An IP address string associated with the profile (e.g., 127.0.0.1). When set to 0 (the default) Mixpanel will ignore IP information.
	 */
	-(void)setIP:(NSString*)IP {
		[super setInput:@"IP" toValue:IP];
	}

	/*!
	 * Set the value of the IgnoreTime input for this Choreo.
	*(optional, boolean) When set to true, Mixpanel will not automatically update the "Last Seen" property of the profile. Otherwise, Mixpanel will add a "Last Seen" property associated with any set, append, or add requests.
	 */
	-(void)setIgnoreTime:(NSString*)IgnoreTime {
		[super setInput:@"IgnoreTime" toValue:IgnoreTime];
	}

	/*!
	 * Set the value of the ProfileProperties input for this Choreo.
	*(conditional, json) A JSON object containing names and values of custom profile properties. The current value will be added to any existing property value. If the property doesn't exist, the value will be added to 0.
	 */
	-(void)setProfileProperties:(NSString*)ProfileProperties {
		[super setInput:@"ProfileProperties" toValue:ProfileProperties];
	}

	/*!
	 * Set the value of the Time input for this Choreo.
	*(optional, date) A unix timestamp representing the time of the profile update. If not provided, Mixpanel will use the time the update arrives at the server.
	 */
	-(void)setTime:(NSString*)Time {
		[super setInput:@"Time" toValue:Time];
	}

	/*!
	 * Set the value of the Token input for this Choreo.
	*(required, string) The token provided by Mixpanel. You can find your Mixpanel token in the project settings dialog in the Mixpanel app.
	 */
	-(void)setToken:(NSString*)Token {
		[super setInput:@"Token" toValue:Token];
	}

	/*!
	 * Set the value of the Verbose input for this Choreo.
	*(optional, boolean) When set to 1, the response will contain more information describing the success or failure of the tracking call.
	 */
	-(void)setVerbose:(NSString*)Verbose {
		[super setInput:@"Verbose" toValue:Verbose];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Add Choreo.
 */
@implementation TMBMixpanel_Profiles_Add_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Mixpanel.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Adds or subtracts a value from an existing numeric property value.
 */
@implementation TMBMixpanel_Profiles_Add

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBMixpanel_Profiles_Add Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Mixpanel/Profiles/Add"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBMixpanel_Profiles_Add_ResultSet *results = [[TMBMixpanel_Profiles_Add_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Add Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBMixpanel_Profiles_Add_Inputs*)newInputSet {
		return [[TMBMixpanel_Profiles_Add_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Append Choreo.
 */
@implementation TMBMixpanel_Profiles_Append_Inputs

	/*!
	 * Set the value of the DistinctID input for this Choreo.
	*(required, string) Used to uniquely identify the profile you want to update.
	 */
	-(void)setDistinctID:(NSString*)DistinctID {
		[super setInput:@"DistinctID" toValue:DistinctID];
	}

	/*!
	 * Set the value of the IP input for this Choreo.
	*(optional, string) An IP address string associated with the profile (e.g., 127.0.0.1). When set to 0 (the default) Mixpanel will ignore IP information.
	 */
	-(void)setIP:(NSString*)IP {
		[super setInput:@"IP" toValue:IP];
	}

	/*!
	 * Set the value of the IgnoreTime input for this Choreo.
	*(optional, boolean) When set to true, Mixpanel will not automatically update the "Last Seen" property of the profile. Otherwise, Mixpanel will add a "Last Seen" property associated with any set, append, or add requests.
	 */
	-(void)setIgnoreTime:(NSString*)IgnoreTime {
		[super setInput:@"IgnoreTime" toValue:IgnoreTime];
	}

	/*!
	 * Set the value of the ProfileProperties input for this Choreo.
	*(conditional, json) A JSON object containing a name/value pair representing a list name and value. The current value will be appended to any existing lists. If the list doesn't exist, it will be created.
	 */
	-(void)setProfileProperties:(NSString*)ProfileProperties {
		[super setInput:@"ProfileProperties" toValue:ProfileProperties];
	}

	/*!
	 * Set the value of the Time input for this Choreo.
	*(optional, date) A unix timestamp representing the time of the profile update. If not provided, Mixpanel will use the time the update arrives at the server.
	 */
	-(void)setTime:(NSString*)Time {
		[super setInput:@"Time" toValue:Time];
	}

	/*!
	 * Set the value of the Token input for this Choreo.
	*(required, string) The token provided by Mixpanel. You can find your Mixpanel token in the project settings dialog in the Mixpanel app.
	 */
	-(void)setToken:(NSString*)Token {
		[super setInput:@"Token" toValue:Token];
	}

	/*!
	 * Set the value of the Verbose input for this Choreo.
	*(optional, boolean) When set to 1, the response will contain more information describing the success or failure of the tracking call.
	 */
	-(void)setVerbose:(NSString*)Verbose {
		[super setInput:@"Verbose" toValue:Verbose];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Append Choreo.
 */
@implementation TMBMixpanel_Profiles_Append_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Mixpanel.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Appends a value to a list property on a profile.
 */
@implementation TMBMixpanel_Profiles_Append

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBMixpanel_Profiles_Append Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Mixpanel/Profiles/Append"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBMixpanel_Profiles_Append_ResultSet *results = [[TMBMixpanel_Profiles_Append_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Append Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBMixpanel_Profiles_Append_Inputs*)newInputSet {
		return [[TMBMixpanel_Profiles_Append_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Delete Choreo.
 */
@implementation TMBMixpanel_Profiles_Delete_Inputs

	/*!
	 * Set the value of the DistinctID input for this Choreo.
	*(required, string) Used to uniquely identify the profile you want to update.
	 */
	-(void)setDistinctID:(NSString*)DistinctID {
		[super setInput:@"DistinctID" toValue:DistinctID];
	}

	/*!
	 * Set the value of the Token input for this Choreo.
	*(required, string) The token provided by Mixpanel. You can find your Mixpanel token in the project settings dialog in the Mixpanel app.
	 */
	-(void)setToken:(NSString*)Token {
		[super setInput:@"Token" toValue:Token];
	}

	/*!
	 * Set the value of the Verbose input for this Choreo.
	*(optional, boolean) When set to 1, the response will contain more information describing the success or failure of the tracking call.
	 */
	-(void)setVerbose:(NSString*)Verbose {
		[super setInput:@"Verbose" toValue:Verbose];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Delete Choreo.
 */
@implementation TMBMixpanel_Profiles_Delete_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Mixpanel.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Permanently deletes the profile from Mixpanel, along with all of its properties.
 */
@implementation TMBMixpanel_Profiles_Delete

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBMixpanel_Profiles_Delete Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Mixpanel/Profiles/Delete"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBMixpanel_Profiles_Delete_ResultSet *results = [[TMBMixpanel_Profiles_Delete_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Delete Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBMixpanel_Profiles_Delete_Inputs*)newInputSet {
		return [[TMBMixpanel_Profiles_Delete_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Set Choreo.
 */
@implementation TMBMixpanel_Profiles_Set_Inputs

	/*!
	 * Set the value of the City input for this Choreo.
	*(optional, string) The city associated with the user's location.
	 */
	-(void)setCity:(NSString*)City {
		[super setInput:@"City" toValue:City];
	}

	/*!
	 * Set the value of the Created input for this Choreo.
	*(optional, date) The time when the user created their account. This should be expressed as YYYY-MM-DDThh:mm:ss.
	 */
	-(void)setCreated:(NSString*)Created {
		[super setInput:@"Created" toValue:Created];
	}

	/*!
	 * Set the value of the DistinctID input for this Choreo.
	*(required, string) Used to uniquely identify the profile you want to update.
	 */
	-(void)setDistinctID:(NSString*)DistinctID {
		[super setInput:@"DistinctID" toValue:DistinctID];
	}

	/*!
	 * Set the value of the Email input for this Choreo.
	*(optional, string) The user's email address. Mixpanel can use this property when sending email notifications to your users.
	 */
	-(void)setEmail:(NSString*)Email {
		[super setInput:@"Email" toValue:Email];
	}

	/*!
	 * Set the value of the FirstName input for this Choreo.
	*(optional, string) The first name of the user represented the profile.
	 */
	-(void)setFirstName:(NSString*)FirstName {
		[super setInput:@"FirstName" toValue:FirstName];
	}

	/*!
	 * Set the value of the IP input for this Choreo.
	*(optional, string) An IP address string associated with the profile (e.g., 127.0.0.1). When set to 0 (the default) Mixpanel will ignore IP information.
	 */
	-(void)setIP:(NSString*)IP {
		[super setInput:@"IP" toValue:IP];
	}

	/*!
	 * Set the value of the IgnoreTime input for this Choreo.
	*(optional, boolean) When set to true, Mixpanel will not automatically update the "Last Seen" property of the profile. Otherwise, Mixpanel will add a "Last Seen" property associated with any set, append, or add requests.
	 */
	-(void)setIgnoreTime:(NSString*)IgnoreTime {
		[super setInput:@"IgnoreTime" toValue:IgnoreTime];
	}

	/*!
	 * Set the value of the LastName input for this Choreo.
	*(optional, string) The last name of the user representing the profile.
	 */
	-(void)setLastName:(NSString*)LastName {
		[super setInput:@"LastName" toValue:LastName];
	}

	/*!
	 * Set the value of the Name input for this Choreo.
	*(optional, string) The full name of the user. This can be used as an alternative to FirstName and LastName.
	 */
	-(void)setName:(NSString*)Name {
		[super setInput:@"Name" toValue:Name];
	}

	/*!
	 * Set the value of the Phone input for this Choreo.
	*(optional, string) The user's phone number (e.g., 4805551212). Mixpanel can use this property when sending SMS messages to your users.
	 */
	-(void)setPhone:(NSString*)Phone {
		[super setInput:@"Phone" toValue:Phone];
	}

	/*!
	 * Set the value of the ProfileProperties input for this Choreo.
	*(optional, json) A JSON object containing names and values of custom profile properties. Note that properties that exist already will be overwritten.
	 */
	-(void)setProfileProperties:(NSString*)ProfileProperties {
		[super setInput:@"ProfileProperties" toValue:ProfileProperties];
	}

	/*!
	 * Set the value of the Region input for this Choreo.
	*(optional, string) The region associated with a user's location.
	 */
	-(void)setRegion:(NSString*)Region {
		[super setInput:@"Region" toValue:Region];
	}

	/*!
	 * Set the value of the Time input for this Choreo.
	*(optional, date) A unix timestamp representing the time of the profile update. If not provided, Mixpanel will use the time the update arrives at the server.
	 */
	-(void)setTime:(NSString*)Time {
		[super setInput:@"Time" toValue:Time];
	}

	/*!
	 * Set the value of the Timezone input for this Choreo.
	*(optional, string) The timezone associated with a user's location.
	 */
	-(void)setTimezone:(NSString*)Timezone {
		[super setInput:@"Timezone" toValue:Timezone];
	}

	/*!
	 * Set the value of the Token input for this Choreo.
	*(required, string) The token provided by Mixpanel. You can find your Mixpanel token in the project settings dialog in the Mixpanel app.
	 */
	-(void)setToken:(NSString*)Token {
		[super setInput:@"Token" toValue:Token];
	}

	/*!
	 * Set the value of the Verbose input for this Choreo.
	*(optional, boolean) When set to 1, the response will contain more information describing the success or failure of the tracking call.
	 */
	-(void)setVerbose:(NSString*)Verbose {
		[super setInput:@"Verbose" toValue:Verbose];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Set Choreo.
 */
@implementation TMBMixpanel_Profiles_Set_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Mixpanel.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Sets the properties of a profile.
 */
@implementation TMBMixpanel_Profiles_Set

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBMixpanel_Profiles_Set Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Mixpanel/Profiles/Set"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBMixpanel_Profiles_Set_ResultSet *results = [[TMBMixpanel_Profiles_Set_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Set Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBMixpanel_Profiles_Set_Inputs*)newInputSet {
		return [[TMBMixpanel_Profiles_Set_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SetOnce Choreo.
 */
@implementation TMBMixpanel_Profiles_SetOnce_Inputs

	/*!
	 * Set the value of the DistinctID input for this Choreo.
	*(required, string) Used to uniquely identify the profile you want to update.
	 */
	-(void)setDistinctID:(NSString*)DistinctID {
		[super setInput:@"DistinctID" toValue:DistinctID];
	}

	/*!
	 * Set the value of the IP input for this Choreo.
	*(optional, string) An IP address string associated with the profile (e.g., 127.0.0.1). When set to 0 (the default) Mixpanel will ignore IP information.
	 */
	-(void)setIP:(NSString*)IP {
		[super setInput:@"IP" toValue:IP];
	}

	/*!
	 * Set the value of the IgnoreTime input for this Choreo.
	*(optional, boolean) When set to true, Mixpanel will not automatically update the "Last Seen" property of the profile. Otherwise, Mixpanel will add a "Last Seen" property associated with any set, append, or add requests.
	 */
	-(void)setIgnoreTime:(NSString*)IgnoreTime {
		[super setInput:@"IgnoreTime" toValue:IgnoreTime];
	}

	/*!
	 * Set the value of the ProfileProperties input for this Choreo.
	*(conditional, json) A JSON object containing names and values of custom profile properties. Note that properties existing already will not be overwritten.
	 */
	-(void)setProfileProperties:(NSString*)ProfileProperties {
		[super setInput:@"ProfileProperties" toValue:ProfileProperties];
	}

	/*!
	 * Set the value of the Time input for this Choreo.
	*(optional, date) A unix timestamp representing the time of the profile update. If not provided, Mixpanel will use the time the update arrives at the server.
	 */
	-(void)setTime:(NSString*)Time {
		[super setInput:@"Time" toValue:Time];
	}

	/*!
	 * Set the value of the Token input for this Choreo.
	*(required, string) The token provided by Mixpanel. You can find your Mixpanel token in the project settings dialog in the Mixpanel app.
	 */
	-(void)setToken:(NSString*)Token {
		[super setInput:@"Token" toValue:Token];
	}

	/*!
	 * Set the value of the Verbose input for this Choreo.
	*(optional, boolean) When set to 1, the response will contain more information describing the success or failure of the tracking call.
	 */
	-(void)setVerbose:(NSString*)Verbose {
		[super setInput:@"Verbose" toValue:Verbose];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SetOnce Choreo.
 */
@implementation TMBMixpanel_Profiles_SetOnce_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Mixpanel.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Sets the properties of a profile, but will not overwrite existing properties.
 */
@implementation TMBMixpanel_Profiles_SetOnce

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBMixpanel_Profiles_SetOnce Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Mixpanel/Profiles/SetOnce"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBMixpanel_Profiles_SetOnce_ResultSet *results = [[TMBMixpanel_Profiles_SetOnce_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SetOnce Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBMixpanel_Profiles_SetOnce_Inputs*)newInputSet {
		return [[TMBMixpanel_Profiles_SetOnce_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Transaction Choreo.
 */
@implementation TMBMixpanel_Profiles_Transaction_Inputs

	/*!
	 * Set the value of the DistinctID input for this Choreo.
	*(required, string) Used to uniquely identify the profile you want to update.
	 */
	-(void)setDistinctID:(NSString*)DistinctID {
		[super setInput:@"DistinctID" toValue:DistinctID];
	}

	/*!
	 * Set the value of the IP input for this Choreo.
	*(optional, string) An IP address string associated with the profile (e.g., 127.0.0.1). When set to 0 (the default) Mixpanel will ignore IP information.
	 */
	-(void)setIP:(NSString*)IP {
		[super setInput:@"IP" toValue:IP];
	}

	/*!
	 * Set the value of the IgnoreTime input for this Choreo.
	*(optional, boolean) When set to true, Mixpanel will not automatically update the "Last Seen" property of the profile. Otherwise, Mixpanel will add a "Last Seen" property associated with any set, append, or add requests.
	 */
	-(void)setIgnoreTime:(NSString*)IgnoreTime {
		[super setInput:@"IgnoreTime" toValue:IgnoreTime];
	}

	/*!
	 * Set the value of the Time input for this Choreo.
	*(optional, date) A unix timestamp representing the time of the profile update. If not provided, Mixpanel will use the time the update arrives at the server.
	 */
	-(void)setTime:(NSString*)Time {
		[super setInput:@"Time" toValue:Time];
	}

	/*!
	 * Set the value of the Token input for this Choreo.
	*(required, string) The token provided by Mixpanel. You can find your Mixpanel token in the project settings dialog in the Mixpanel app.
	 */
	-(void)setToken:(NSString*)Token {
		[super setInput:@"Token" toValue:Token];
	}

	/*!
	 * Set the value of the TransactionAmount input for this Choreo.
	*(required, decimal) The amount of the transaction.
	 */
	-(void)setTransactionAmount:(NSString*)TransactionAmount {
		[super setInput:@"TransactionAmount" toValue:TransactionAmount];
	}

	/*!
	 * Set the value of the TransactionProperties input for this Choreo.
	*(optional, json) A JSON object containing any transaction properties you wish to set (e.g. SKU, Product Name, etc).
	 */
	-(void)setTransactionProperties:(NSString*)TransactionProperties {
		[super setInput:@"TransactionProperties" toValue:TransactionProperties];
	}

	/*!
	 * Set the value of the TransactionTime input for this Choreo.
	*(conditional, date) The time of the transaction (e.g., 2013-01-03T09:00:00). If this isn't provided, the current time in UTC is specified.
	 */
	-(void)setTransactionTime:(NSString*)TransactionTime {
		[super setInput:@"TransactionTime" toValue:TransactionTime];
	}

	/*!
	 * Set the value of the Verbose input for this Choreo.
	*(optional, boolean) When set to 1, the response will contain more information describing the success or failure of the tracking call.
	 */
	-(void)setVerbose:(NSString*)Verbose {
		[super setInput:@"Verbose" toValue:Verbose];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Transaction Choreo.
 */
@implementation TMBMixpanel_Profiles_Transaction_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Mixpanel.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Appends transaction data to a profile.
 */
@implementation TMBMixpanel_Profiles_Transaction

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBMixpanel_Profiles_Transaction Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Mixpanel/Profiles/Transaction"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBMixpanel_Profiles_Transaction_ResultSet *results = [[TMBMixpanel_Profiles_Transaction_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Transaction Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBMixpanel_Profiles_Transaction_Inputs*)newInputSet {
		return [[TMBMixpanel_Profiles_Transaction_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Union Choreo.
 */
@implementation TMBMixpanel_Profiles_Union_Inputs

	/*!
	 * Set the value of the DistinctID input for this Choreo.
	*(required, string) Used to uniquely identify the profile you want to update.
	 */
	-(void)setDistinctID:(NSString*)DistinctID {
		[super setInput:@"DistinctID" toValue:DistinctID];
	}

	/*!
	 * Set the value of the IP input for this Choreo.
	*(optional, string) An IP address string associated with the profile (e.g., 127.0.0.1). When set to 0 (the default) Mixpanel will ignore IP information.
	 */
	-(void)setIP:(NSString*)IP {
		[super setInput:@"IP" toValue:IP];
	}

	/*!
	 * Set the value of the IgnoreTime input for this Choreo.
	*(optional, boolean) When set to true, Mixpanel will not automatically update the "Last Seen" property of the profile. Otherwise, Mixpanel will add a "Last Seen" property associated with any set, append, or add requests.
	 */
	-(void)setIgnoreTime:(NSString*)IgnoreTime {
		[super setInput:@"IgnoreTime" toValue:IgnoreTime];
	}

	/*!
	 * Set the value of the ProfileProperties input for this Choreo.
	*(required, json) A JSON object containing a list name and an array of values. The list values in the request are merged with the existing list on the user profile, ignoring duplicate list values.
	 */
	-(void)setProfileProperties:(NSString*)ProfileProperties {
		[super setInput:@"ProfileProperties" toValue:ProfileProperties];
	}

	/*!
	 * Set the value of the Time input for this Choreo.
	*(optional, date) A unix timestamp representing the time of the profile update. If not provided, Mixpanel will use the time the update arrives at the server.
	 */
	-(void)setTime:(NSString*)Time {
		[super setInput:@"Time" toValue:Time];
	}

	/*!
	 * Set the value of the Token input for this Choreo.
	*(required, string) The token provided by Mixpanel. You can find your Mixpanel token in the project settings dialog in the Mixpanel app.
	 */
	-(void)setToken:(NSString*)Token {
		[super setInput:@"Token" toValue:Token];
	}

	/*!
	 * Set the value of the Verbose input for this Choreo.
	*(optional, boolean) When set to 1, the response will contain more information describing the success or failure of the tracking call.
	 */
	-(void)setVerbose:(NSString*)Verbose {
		[super setInput:@"Verbose" toValue:Verbose];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Union Choreo.
 */
@implementation TMBMixpanel_Profiles_Union_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Mixpanel.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Merges list values with an existing list on a profile.
 */
@implementation TMBMixpanel_Profiles_Union

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBMixpanel_Profiles_Union Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Mixpanel/Profiles/Union"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBMixpanel_Profiles_Union_ResultSet *results = [[TMBMixpanel_Profiles_Union_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Union Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBMixpanel_Profiles_Union_Inputs*)newInputSet {
		return [[TMBMixpanel_Profiles_Union_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Unset Choreo.
 */
@implementation TMBMixpanel_Profiles_Unset_Inputs

	/*!
	 * Set the value of the DistinctID input for this Choreo.
	*(required, string) Used to uniquely identify the profile you want to update.
	 */
	-(void)setDistinctID:(NSString*)DistinctID {
		[super setInput:@"DistinctID" toValue:DistinctID];
	}

	/*!
	 * Set the value of the IP input for this Choreo.
	*(optional, string) An IP address string associated with the profile (e.g., 127.0.0.1). When set to 0 (the default) Mixpanel will ignore IP information.
	 */
	-(void)setIP:(NSString*)IP {
		[super setInput:@"IP" toValue:IP];
	}

	/*!
	 * Set the value of the IgnoreTime input for this Choreo.
	*(optional, boolean) When set to true, Mixpanel will not automatically update the "Last Seen" property of the profile. Otherwise, Mixpanel will add a "Last Seen" property associated with any set, append, or add requests.
	 */
	-(void)setIgnoreTime:(NSString*)IgnoreTime {
		[super setInput:@"IgnoreTime" toValue:IgnoreTime];
	}

	/*!
	 * Set the value of the ProfileProperties input for this Choreo.
	*(conditional, json) A JSON array of property names to remove from a profile.
	 */
	-(void)setProfileProperties:(NSString*)ProfileProperties {
		[super setInput:@"ProfileProperties" toValue:ProfileProperties];
	}

	/*!
	 * Set the value of the Time input for this Choreo.
	*(optional, date) A unix timestamp representing the time of the profile update. If not provided, Mixpanel will use the time the update arrives at the server.
	 */
	-(void)setTime:(NSString*)Time {
		[super setInput:@"Time" toValue:Time];
	}

	/*!
	 * Set the value of the Token input for this Choreo.
	*(required, string) The token provided by Mixpanel. You can find your Mixpanel token in the project settings dialog in the Mixpanel app.
	 */
	-(void)setToken:(NSString*)Token {
		[super setInput:@"Token" toValue:Token];
	}

	/*!
	 * Set the value of the Verbose input for this Choreo.
	*(optional, boolean) When set to 1, the response will contain more information describing the success or failure of the tracking call.
	 */
	-(void)setVerbose:(NSString*)Verbose {
		[super setInput:@"Verbose" toValue:Verbose];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Unset Choreo.
 */
@implementation TMBMixpanel_Profiles_Unset_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Mixpanel.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Removes the specified properties and their values from a profile.
 */
@implementation TMBMixpanel_Profiles_Unset

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBMixpanel_Profiles_Unset Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Mixpanel/Profiles/Unset"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBMixpanel_Profiles_Unset_ResultSet *results = [[TMBMixpanel_Profiles_Unset_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Unset Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBMixpanel_Profiles_Unset_Inputs*)newInputSet {
		return [[TMBMixpanel_Profiles_Unset_Inputs alloc] init];
	}
@end
	