/*!
 * @TMBDropbox.m
 *
 * Execute Choreographies from the Temboo Dropbox bundle.
 *
 * iOS version 10.6.8
 *
 * LICENSE: Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @category   Services
 * @author     Reid Simonton, Temboo, Inc.
 * @copyright  2013 Temboo, Inc.
 * @license    http://www.apache.org/licenses/LICENSE-2.0 Apache License 2.0
 * @version    1.7
 */

#import "TMBDropbox.h"
#import "TMBChoreography_Protected.h"
#import "TMBChoreographyExecution.h"
#import "TMBTembooSession.h"
#import "TMBTembooUri.h"


/*!
 * Input object with appropriate setters for specifying arguments to the AccountInfo Choreo.
 */
@implementation TMBDropbox_Account_AccountInfo_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the AppKey input for this Choreo.
	*(required, string) The App Key provided by Dropbox (AKA the OAuth Consumer Key).
	 */
	-(void)setAppKey:(NSString*)AppKey {
		[super setInput:@"AppKey" toValue:AppKey];
	}

	/*!
	 * Set the value of the AppSecret input for this Choreo.
	*(required, string) The App Secret provided by Dropbox (AKA the OAuth Consumer Secret).
	 */
	-(void)setAppSecret:(NSString*)AppSecret {
		[super setInput:@"AppSecret" toValue:AppSecret];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the AccountInfo Choreo.
 */
@implementation TMBDropbox_Account_AccountInfo_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Dropbox. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves information about the user's account.
 */
@implementation TMBDropbox_Account_AccountInfo

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_Account_AccountInfo Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/Account/AccountInfo"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_Account_AccountInfo_ResultSet *results = [[TMBDropbox_Account_AccountInfo_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the AccountInfo Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_Account_AccountInfo_Inputs*)newInputSet {
		return [[TMBDropbox_Account_AccountInfo_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CopyFileOrFolder Choreo.
 */
@implementation TMBDropbox_FileOperations_CopyFileOrFolder_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the AppKey input for this Choreo.
	*(required, string) The App Key provided by Dropbox (AKA the OAuth Consumer Key).
	 */
	-(void)setAppKey:(NSString*)AppKey {
		[super setInput:@"AppKey" toValue:AppKey];
	}

	/*!
	 * Set the value of the AppSecret input for this Choreo.
	*(required, string) The App Secret provided by Dropbox (AKA the OAuth Consumer Secret).
	 */
	-(void)setAppSecret:(NSString*)AppSecret {
		[super setInput:@"AppSecret" toValue:AppSecret];
	}

	/*!
	 * Set the value of the FromCopyRef input for this Choreo.
	*(conditional, string) Specifies a CopyRef previously generated by running the CreateCopyRef choreo. If you specify this value, you must leave FromPath empty.
	 */
	-(void)setFromCopyRef:(NSString*)FromCopyRef {
		[super setInput:@"FromCopyRef" toValue:FromCopyRef];
	}

	/*!
	 * Set the value of the FromPath input for this Choreo.
	*(conditional, string) Specifies the file or folder to be copied. If you specify this value, you must leave the FromCopyRef blank.
	 */
	-(void)setFromPath:(NSString*)FromPath {
		[super setInput:@"FromPath" toValue:FromPath];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Root input for this Choreo.
	*(optional, string) Defaults to "auto" which automatically determines the root folder using your app's permission level. Other options are "sandbox" (App Folder) and "dropbox" (Full Dropbox).
	 */
	-(void)setRoot:(NSString*)Root {
		[super setInput:@"Root" toValue:Root];
	}

	/*!
	 * Set the value of the ToPath input for this Choreo.
	*(required, string) Specifies the destination path, including the new name for the file or folder.
	 */
	-(void)setToPath:(NSString*)ToPath {
		[super setInput:@"ToPath" toValue:ToPath];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CopyFileOrFolder Choreo.
 */
@implementation TMBDropbox_FileOperations_CopyFileOrFolder_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Dropbox. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Copies a file or folder to a new location in the Dropbox tree.
 */
@implementation TMBDropbox_FileOperations_CopyFileOrFolder

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_FileOperations_CopyFileOrFolder Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/FileOperations/CopyFileOrFolder"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_FileOperations_CopyFileOrFolder_ResultSet *results = [[TMBDropbox_FileOperations_CopyFileOrFolder_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CopyFileOrFolder Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_FileOperations_CopyFileOrFolder_Inputs*)newInputSet {
		return [[TMBDropbox_FileOperations_CopyFileOrFolder_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CreateFolder Choreo.
 */
@implementation TMBDropbox_FileOperations_CreateFolder_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the AppKey input for this Choreo.
	*(required, string) The App Key provided by Dropbox (AKA the OAuth Consumer Key).
	 */
	-(void)setAppKey:(NSString*)AppKey {
		[super setInput:@"AppKey" toValue:AppKey];
	}

	/*!
	 * Set the value of the AppSecret input for this Choreo.
	*(required, string) The App Secret provided by Dropbox (AKA the OAuth Consumer Secret).
	 */
	-(void)setAppSecret:(NSString*)AppSecret {
		[super setInput:@"AppSecret" toValue:AppSecret];
	}

	/*!
	 * Set the value of the NewFolderName input for this Choreo.
	*(required, string) The name of the new folder to create. A path with a root folder is also valid (i.e. /RootFolder/NewFolderName).
	 */
	-(void)setNewFolderName:(NSString*)NewFolderName {
		[super setInput:@"NewFolderName" toValue:NewFolderName];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Root input for this Choreo.
	*(optional, string) Defaults to "auto" which automatically determines the root folder using your app's permission level. Other options are "sandbox" (App Folder) and "dropbox" (Full Dropbox).
	 */
	-(void)setRoot:(NSString*)Root {
		[super setInput:@"Root" toValue:Root];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateFolder Choreo.
 */
@implementation TMBDropbox_FileOperations_CreateFolder_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Dropbox. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Creates a Dropbox folder.
 */
@implementation TMBDropbox_FileOperations_CreateFolder

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_FileOperations_CreateFolder Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/FileOperations/CreateFolder"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_FileOperations_CreateFolder_ResultSet *results = [[TMBDropbox_FileOperations_CreateFolder_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CreateFolder Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_FileOperations_CreateFolder_Inputs*)newInputSet {
		return [[TMBDropbox_FileOperations_CreateFolder_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the DeleteFileOrFolder Choreo.
 */
@implementation TMBDropbox_FileOperations_DeleteFileOrFolder_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the AppKey input for this Choreo.
	*(required, string) The App Key provided by Dropbox (AKA the OAuth Consumer Key).
	 */
	-(void)setAppKey:(NSString*)AppKey {
		[super setInput:@"AppKey" toValue:AppKey];
	}

	/*!
	 * Set the value of the AppSecret input for this Choreo.
	*(required, string) The App Secret provided by Dropbox (AKA the OAuth Consumer Secret).
	 */
	-(void)setAppSecret:(NSString*)AppSecret {
		[super setInput:@"AppSecret" toValue:AppSecret];
	}

	/*!
	 * Set the value of the Path input for this Choreo.
	*(required, string) The path to the file or folder you want to delete (i.e. /RootFolder/SubFolder/FileToDelete.txt).
	 */
	-(void)setPath:(NSString*)Path {
		[super setInput:@"Path" toValue:Path];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Root input for this Choreo.
	*(optional, string) Defaults to "auto" which automatically determines the root folder using your app's permission level. Other options are "sandbox" (App Folder) and "dropbox" (Full Dropbox).
	 */
	-(void)setRoot:(NSString*)Root {
		[super setInput:@"Root" toValue:Root];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteFileOrFolder Choreo.
 */
@implementation TMBDropbox_FileOperations_DeleteFileOrFolder_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Dropbox. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Deletes a Dropbox file or folder that you specify.
 */
@implementation TMBDropbox_FileOperations_DeleteFileOrFolder

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_FileOperations_DeleteFileOrFolder Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/FileOperations/DeleteFileOrFolder"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_FileOperations_DeleteFileOrFolder_ResultSet *results = [[TMBDropbox_FileOperations_DeleteFileOrFolder_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the DeleteFileOrFolder Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_FileOperations_DeleteFileOrFolder_Inputs*)newInputSet {
		return [[TMBDropbox_FileOperations_DeleteFileOrFolder_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the MoveFileOrFolder Choreo.
 */
@implementation TMBDropbox_FileOperations_MoveFileOrFolder_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the AppKey input for this Choreo.
	*(required, string) The App Key provided by Dropbox (AKA the OAuth Consumer Key).
	 */
	-(void)setAppKey:(NSString*)AppKey {
		[super setInput:@"AppKey" toValue:AppKey];
	}

	/*!
	 * Set the value of the AppSecret input for this Choreo.
	*(required, string) The App Secret provided by Dropbox (AKA the OAuth Consumer Secret).
	 */
	-(void)setAppSecret:(NSString*)AppSecret {
		[super setInput:@"AppSecret" toValue:AppSecret];
	}

	/*!
	 * Set the value of the FromPath input for this Choreo.
	*(required, string) Specifies the file or folder to be copied.
	 */
	-(void)setFromPath:(NSString*)FromPath {
		[super setInput:@"FromPath" toValue:FromPath];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Root input for this Choreo.
	*(optional, string) Defaults to "auto" which automatically determines the root folder using your app's permission level. Other options are "sandbox" (App Folder) and "dropbox" (Full Dropbox).
	 */
	-(void)setRoot:(NSString*)Root {
		[super setInput:@"Root" toValue:Root];
	}

	/*!
	 * Set the value of the ToPath input for this Choreo.
	*(required, string) Specifies the destination path, including the new name for the file or folder.
	 */
	-(void)setToPath:(NSString*)ToPath {
		[super setInput:@"ToPath" toValue:ToPath];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the MoveFileOrFolder Choreo.
 */
@implementation TMBDropbox_FileOperations_MoveFileOrFolder_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Dropbox. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Moves a file or folder to a new location in the Dropbox tree.
 */
@implementation TMBDropbox_FileOperations_MoveFileOrFolder

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_FileOperations_MoveFileOrFolder Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/FileOperations/MoveFileOrFolder"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_FileOperations_MoveFileOrFolder_ResultSet *results = [[TMBDropbox_FileOperations_MoveFileOrFolder_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the MoveFileOrFolder Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_FileOperations_MoveFileOrFolder_Inputs*)newInputSet {
		return [[TMBDropbox_FileOperations_MoveFileOrFolder_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ZipFile Choreo.
 */
@implementation TMBDropbox_FileOperations_ZipFile_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the AppKey input for this Choreo.
	*(required, string) The App Key provided by Dropbox (AKA the OAuth Consumer Key).
	 */
	-(void)setAppKey:(NSString*)AppKey {
		[super setInput:@"AppKey" toValue:AppKey];
	}

	/*!
	 * Set the value of the AppSecret input for this Choreo.
	*(required, string) The App Secret provided by Dropbox (AKA the OAuth Consumer Secret).
	 */
	-(void)setAppSecret:(NSString*)AppSecret {
		[super setInput:@"AppSecret" toValue:AppSecret];
	}

	/*!
	 * Set the value of the FileName input for this Choreo.
	*(required, string) The name of the Dropbox file you want to zip.
	 */
	-(void)setFileName:(NSString*)FileName {
		[super setInput:@"FileName" toValue:FileName];
	}

	/*!
	 * Set the value of the Folder input for this Choreo.
	*(conditional, string) The name of the folder that contains the file you want to zip. A path to a sub-folder is also valid (i.e. RootFolder/SubFolder). Not required if the file is at the root level.
	 */
	-(void)setFolder:(NSString*)Folder {
		[super setInput:@"Folder" toValue:Folder];
	}

	/*!
	 * Set the value of the Root input for this Choreo.
	*(optional, string) Defaults to "auto" which automatically determines the root folder using your app's permission level. Other options are "sandbox" (App Folder) and "dropbox" (Full Dropbox).
	 */
	-(void)setRoot:(NSString*)Root {
		[super setInput:@"Root" toValue:Root];
	}

	/*!
	 * Set the value of the ZipFileLocation input for this Choreo.
	*(optional, string) The name of the folder to put the new zip file in.A path to a sub-folder is also valid (i.e. RootFolder/SubFolder).  When not specified, the zip file will be put in the root folder.
	 */
	-(void)setZipFileLocation:(NSString*)ZipFileLocation {
		[super setInput:@"ZipFileLocation" toValue:ZipFileLocation];
	}

	/*!
	 * Set the value of the ZipFileName input for this Choreo.
	*(optional, string) The name of the zip file that will be created. If not specified, the original file name will be used with a .zip extension.
	 */
	-(void)setZipFileName:(NSString*)ZipFileName {
		[super setInput:@"ZipFileName" toValue:ZipFileName];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ZipFile Choreo.
 */
@implementation TMBDropbox_FileOperations_ZipFile_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "URL" output from an execution of this Choreo.
	 * @return - NSString* (string) The shared link for the new zip file.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getURL {
		return [super getOutputByName:@"URL"];
	}
	
@end

/*!
 * Creates a zipped version of the specified Dropbox file and returns a shareable link to the new compressed file.
 */
@implementation TMBDropbox_FileOperations_ZipFile

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_FileOperations_ZipFile Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/FileOperations/ZipFile"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_FileOperations_ZipFile_ResultSet *results = [[TMBDropbox_FileOperations_ZipFile_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ZipFile Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_FileOperations_ZipFile_Inputs*)newInputSet {
		return [[TMBDropbox_FileOperations_ZipFile_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ZipFolder Choreo.
 */
@implementation TMBDropbox_FileOperations_ZipFolder_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the AppKey input for this Choreo.
	*(required, string) The App Key provided by Dropbox (AKA the OAuth Consumer Key).
	 */
	-(void)setAppKey:(NSString*)AppKey {
		[super setInput:@"AppKey" toValue:AppKey];
	}

	/*!
	 * Set the value of the AppSecret input for this Choreo.
	*(required, string) The App Secret provided by Dropbox (AKA the OAuth Consumer Secret).
	 */
	-(void)setAppSecret:(NSString*)AppSecret {
		[super setInput:@"AppSecret" toValue:AppSecret];
	}

	/*!
	 * Set the value of the Folder input for this Choreo.
	*(required, string) The name of the folder that contains the files you want to retrieve and zip. A path to a sub-folder is also valid (i.e. RootFolder/SubFolder).
	 */
	-(void)setFolder:(NSString*)Folder {
		[super setInput:@"Folder" toValue:Folder];
	}

	/*!
	 * Set the value of the Root input for this Choreo.
	*(optional, string) Defaults to "auto" which automatically determines the root folder using your app's permission level. Other options are "sandbox" (App Folder) and "dropbox" (Full Dropbox).
	 */
	-(void)setRoot:(NSString*)Root {
		[super setInput:@"Root" toValue:Root];
	}

	/*!
	 * Set the value of the ZipFileLocation input for this Choreo.
	*(optional, string) The name of the folder to put the new zip file in. A path to a sub-folder is also valid (i.e. RootFolder/SubFolder). When not specified, the zip file will be put in the root folder.
	 */
	-(void)setZipFileLocation:(NSString*)ZipFileLocation {
		[super setInput:@"ZipFileLocation" toValue:ZipFileLocation];
	}

	/*!
	 * Set the value of the ZipFileName input for this Choreo.
	*(optional, string) The name of the zip file that will be created. If not specified, the original folder name containing the files to zip will be used with a .zip extension.
	 */
	-(void)setZipFileName:(NSString*)ZipFileName {
		[super setInput:@"ZipFileName" toValue:ZipFileName];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ZipFolder Choreo.
 */
@implementation TMBDropbox_FileOperations_ZipFolder_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "URL" output from an execution of this Choreo.
	 * @return - NSString* 
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getURL {
		return [super getOutputByName:@"URL"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Dropbox. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Creates a zip file containing all files within the specified folder and returns a shareable link to the new compressed file.
 */
@implementation TMBDropbox_FileOperations_ZipFolder

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_FileOperations_ZipFolder Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/FileOperations/ZipFolder"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_FileOperations_ZipFolder_ResultSet *results = [[TMBDropbox_FileOperations_ZipFolder_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ZipFolder Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_FileOperations_ZipFolder_Inputs*)newInputSet {
		return [[TMBDropbox_FileOperations_ZipFolder_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Copy Choreo.
 */
@implementation TMBDropbox_Files_Copy_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token for a specific Dropbox user.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AllowSharedFolder input for this Choreo.
	*(optional, boolean) If true, contents are copied into a shared folder, otherwise an error will be returned if the FromPath contains a shared folder. The default for this field is false.
	 */
	-(void)setAllowSharedFolder:(NSString*)AllowSharedFolder {
		[super setInput:@"AllowSharedFolder" toValue:AllowSharedFolder];
	}

	/*!
	 * Set the value of the AutoRename input for this Choreo.
	*(optional, boolean) If there's a conflict, have the Dropbox server try to autorename the file to avoid the conflict. The default for this field is false.
	 */
	-(void)setAutoRename:(NSString*)AutoRename {
		[super setInput:@"AutoRename" toValue:AutoRename];
	}

	/*!
	 * Set the value of the FromPath input for this Choreo.
	*(required, string) Path in the user's Dropbox to be copied or moved.
	 */
	-(void)setFromPath:(NSString*)FromPath {
		[super setInput:@"FromPath" toValue:FromPath];
	}

	/*!
	 * Set the value of the ToPath input for this Choreo.
	*(required, string) Path in the user's Dropbox that is the destination.
	 */
	-(void)setToPath:(NSString*)ToPath {
		[super setInput:@"ToPath" toValue:ToPath];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Copy Choreo.
 */
@implementation TMBDropbox_Files_Copy_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Dropbox.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Copies a file or folder to a different location in the user's Dropbox.
 */
@implementation TMBDropbox_Files_Copy

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_Files_Copy Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/Files/Copy"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_Files_Copy_ResultSet *results = [[TMBDropbox_Files_Copy_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Copy Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_Files_Copy_Inputs*)newInputSet {
		return [[TMBDropbox_Files_Copy_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CopyReferenceGet Choreo.
 */
@implementation TMBDropbox_Files_CopyReferenceGet_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token for a specific Dropbox user.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the Path input for this Choreo.
	*(required, string) The path to the file or folder you want to get a copy reference to.
	 */
	-(void)setPath:(NSString*)Path {
		[super setInput:@"Path" toValue:Path];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CopyReferenceGet Choreo.
 */
@implementation TMBDropbox_Files_CopyReferenceGet_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Dropbox.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Gets a copy reference to a file or folder.
 */
@implementation TMBDropbox_Files_CopyReferenceGet

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_Files_CopyReferenceGet Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/Files/CopyReferenceGet"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_Files_CopyReferenceGet_ResultSet *results = [[TMBDropbox_Files_CopyReferenceGet_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CopyReferenceGet Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_Files_CopyReferenceGet_Inputs*)newInputSet {
		return [[TMBDropbox_Files_CopyReferenceGet_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CopyReferenceSave Choreo.
 */
@implementation TMBDropbox_Files_CopyReferenceSave_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token for a specific Dropbox user.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the CopyReference input for this Choreo.
	*(required, string) A copy reference returned by CopyReferenceGet.
	 */
	-(void)setCopyReference:(NSString*)CopyReference {
		[super setInput:@"CopyReference" toValue:CopyReference];
	}

	/*!
	 * Set the value of the Path input for this Choreo.
	*(required, string) Path in the user's Dropbox that is the destination.
	 */
	-(void)setPath:(NSString*)Path {
		[super setInput:@"Path" toValue:Path];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CopyReferenceSave Choreo.
 */
@implementation TMBDropbox_Files_CopyReferenceSave_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Dropbox.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Saves a copy reference returned by CopyReferenceGet to the user's Dropbox.
 */
@implementation TMBDropbox_Files_CopyReferenceSave

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_Files_CopyReferenceSave Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/Files/CopyReferenceSave"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_Files_CopyReferenceSave_ResultSet *results = [[TMBDropbox_Files_CopyReferenceSave_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CopyReferenceSave Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_Files_CopyReferenceSave_Inputs*)newInputSet {
		return [[TMBDropbox_Files_CopyReferenceSave_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CreateFolder Choreo.
 */
@implementation TMBDropbox_Files_CreateFolder_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token for a specific Dropbox user.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AutoRename input for this Choreo.
	*(optional, boolean) If there's a conflict, have the Dropbox server try to autorename the file to avoid the conflict. The default for this field is false.
	 */
	-(void)setAutoRename:(NSString*)AutoRename {
		[super setInput:@"AutoRename" toValue:AutoRename];
	}

	/*!
	 * Set the value of the Path input for this Choreo.
	*(required, string) Path in the user's Dropbox to create.
	 */
	-(void)setPath:(NSString*)Path {
		[super setInput:@"Path" toValue:Path];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateFolder Choreo.
 */
@implementation TMBDropbox_Files_CreateFolder_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Dropbox.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Creates a folder at a given path.
 */
@implementation TMBDropbox_Files_CreateFolder

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_Files_CreateFolder Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/Files/CreateFolder"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_Files_CreateFolder_ResultSet *results = [[TMBDropbox_Files_CreateFolder_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CreateFolder Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_Files_CreateFolder_Inputs*)newInputSet {
		return [[TMBDropbox_Files_CreateFolder_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Delete Choreo.
 */
@implementation TMBDropbox_Files_Delete_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token for a specific Dropbox user.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the Path input for this Choreo.
	*(required, string) Path in the user's Dropbox to delete.
	 */
	-(void)setPath:(NSString*)Path {
		[super setInput:@"Path" toValue:Path];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Delete Choreo.
 */
@implementation TMBDropbox_Files_Delete_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Dropbox.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Deletes the file or folder at a given path.
 */
@implementation TMBDropbox_Files_Delete

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_Files_Delete Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/Files/Delete"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_Files_Delete_ResultSet *results = [[TMBDropbox_Files_Delete_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Delete Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_Files_Delete_Inputs*)newInputSet {
		return [[TMBDropbox_Files_Delete_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Download Choreo.
 */
@implementation TMBDropbox_Files_Download_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token for a specific Dropbox user.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the Encode input for this Choreo.
	*(optional, boolean) Binary files must be returned as Base64-encoded data. If the file you are donwloading is plain text, you can set this to false to return the raw data. Defaults to true. 
	 */
	-(void)setEncode:(NSString*)Encode {
		[super setInput:@"Encode" toValue:Encode];
	}

	/*!
	 * Set the value of the Path input for this Choreo.
	*(required, string) The path of the file to download.
	 */
	-(void)setPath:(NSString*)Path {
		[super setInput:@"Path" toValue:Path];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Download Choreo.
 */
@implementation TMBDropbox_Files_Download_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Dropbox.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Downloads a file from a user's Dropbox.
 */
@implementation TMBDropbox_Files_Download

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_Files_Download Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/Files/Download"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_Files_Download_ResultSet *results = [[TMBDropbox_Files_Download_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Download Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_Files_Download_Inputs*)newInputSet {
		return [[TMBDropbox_Files_Download_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetLatestCursor Choreo.
 */
@implementation TMBDropbox_Files_GetLatestCursor_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token for a specific Dropbox user.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the IncludeDeleted input for this Choreo.
	*(optional, boolean) If true, the results will include entries for files and folders that used to exist but were deleted. The default for this field is false.
	 */
	-(void)setIncludeDeleted:(NSString*)IncludeDeleted {
		[super setInput:@"IncludeDeleted" toValue:IncludeDeleted];
	}

	/*!
	 * Set the value of the IncludeHasExplicitSharedMembers input for this Choreo.
	*(optional, boolean) If true, the results will include a flag for each file indicating whether or not that file has any explicit members. The default for this field is false.
	 */
	-(void)setIncludeHasExplicitSharedMembers:(NSString*)IncludeHasExplicitSharedMembers {
		[super setInput:@"IncludeHasExplicitSharedMembers" toValue:IncludeHasExplicitSharedMembers];
	}

	/*!
	 * Set the value of the IncludeMediaInfo input for this Choreo.
	*(optional, boolean) If true, FileMetadata.media_info is set for photo and video. The default for this field is false.
	 */
	-(void)setIncludeMediaInfo:(NSString*)IncludeMediaInfo {
		[super setInput:@"IncludeMediaInfo" toValue:IncludeMediaInfo];
	}

	/*!
	 * Set the value of the Path input for this Choreo.
	*(required, string) The path to the folder you want to see the contents of. This should be empty to list contents at the root level.
	 */
	-(void)setPath:(NSString*)Path {
		[super setInput:@"Path" toValue:Path];
	}

	/*!
	 * Set the value of the Recursive input for this Choreo.
	*(optional, boolean) If true, the list folder operation will be applied recursively to all subfolders and the response will contain contents of all subfolders. The default for this field is false.
	 */
	-(void)setRecursive:(NSString*)Recursive {
		[super setInput:@"Recursive" toValue:Recursive];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetLatestCursor Choreo.
 */
@implementation TMBDropbox_Files_GetLatestCursor_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Cursor" output from an execution of this Choreo.
	 * @return - NSString* (string) A cursor used to retrieve the next set of results with ListFolderContinue.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getCursor {
		return [super getOutputByName:@"Cursor"];
	}
	
@end

/*!
 * Gets a cursor for the folder's state.
 */
@implementation TMBDropbox_Files_GetLatestCursor

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_Files_GetLatestCursor Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/Files/GetLatestCursor"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_Files_GetLatestCursor_ResultSet *results = [[TMBDropbox_Files_GetLatestCursor_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetLatestCursor Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_Files_GetLatestCursor_Inputs*)newInputSet {
		return [[TMBDropbox_Files_GetLatestCursor_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetMetadata Choreo.
 */
@implementation TMBDropbox_Files_GetMetadata_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token for a specific Dropbox user.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the IncludeDeleted input for this Choreo.
	*(optional, boolean) If true, DeletedMetadata will be returned for deleted file or folder, otherwise LookupError.not_found will be returned. The default for this field is false.
	 */
	-(void)setIncludeDeleted:(NSString*)IncludeDeleted {
		[super setInput:@"IncludeDeleted" toValue:IncludeDeleted];
	}

	/*!
	 * Set the value of the IncludeHasExplicitSharedMembers input for this Choreo.
	*(optional, boolean) If true, the results will include a flag for each file indicating whether or not that file has any explicit members. The default for this field is false.
	 */
	-(void)setIncludeHasExplicitSharedMembers:(NSString*)IncludeHasExplicitSharedMembers {
		[super setInput:@"IncludeHasExplicitSharedMembers" toValue:IncludeHasExplicitSharedMembers];
	}

	/*!
	 * Set the value of the IncludeMediaInfo input for this Choreo.
	*(optional, boolean) If true, FileMetadata.media_info is set for photo and video. The default for this field is false.
	 */
	-(void)setIncludeMediaInfo:(NSString*)IncludeMediaInfo {
		[super setInput:@"IncludeMediaInfo" toValue:IncludeMediaInfo];
	}

	/*!
	 * Set the value of the Path input for this Choreo.
	*(required, string) The path of a file or folder on Dropbox.
	 */
	-(void)setPath:(NSString*)Path {
		[super setInput:@"Path" toValue:Path];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetMetadata Choreo.
 */
@implementation TMBDropbox_Files_GetMetadata_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Dropbox.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Returns the metadata for a file or folder.
 */
@implementation TMBDropbox_Files_GetMetadata

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_Files_GetMetadata Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/Files/GetMetadata"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_Files_GetMetadata_ResultSet *results = [[TMBDropbox_Files_GetMetadata_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetMetadata Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_Files_GetMetadata_Inputs*)newInputSet {
		return [[TMBDropbox_Files_GetMetadata_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetThumbnail Choreo.
 */
@implementation TMBDropbox_Files_GetThumbnail_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token for a specific Dropbox user.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the Format input for this Choreo.
	*(optional, string) The format for the thumbnail image, jpeg (default) or png.
	 */
	-(void)setFormat:(NSString*)Format {
		[super setInput:@"Format" toValue:Format];
	}

	/*!
	 * Set the value of the Path input for this Choreo.
	*(required, string) The path of the file to download.
	 */
	-(void)setPath:(NSString*)Path {
		[super setInput:@"Path" toValue:Path];
	}

	/*!
	 * Set the value of the Size input for this Choreo.
	*(optional, string) The size for the thumbnail image. The default for this union is w64h64.
	 */
	-(void)setSize:(NSString*)Size {
		[super setInput:@"Size" toValue:Size];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetThumbnail Choreo.
 */
@implementation TMBDropbox_Files_GetThumbnail_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Dropbox.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Gets a thumbnail for an image.
 */
@implementation TMBDropbox_Files_GetThumbnail

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_Files_GetThumbnail Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/Files/GetThumbnail"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_Files_GetThumbnail_ResultSet *results = [[TMBDropbox_Files_GetThumbnail_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetThumbnail Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_Files_GetThumbnail_Inputs*)newInputSet {
		return [[TMBDropbox_Files_GetThumbnail_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListFolder Choreo.
 */
@implementation TMBDropbox_Files_ListFolder_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token for a specific Dropbox user.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the IncludeDeleted input for this Choreo.
	*(optional, boolean) If true, the results will include entries for files and folders that used to exist but were deleted. The default for this field is false.
	 */
	-(void)setIncludeDeleted:(NSString*)IncludeDeleted {
		[super setInput:@"IncludeDeleted" toValue:IncludeDeleted];
	}

	/*!
	 * Set the value of the IncludeHasExplicitSharedMembers input for this Choreo.
	*(optional, boolean) If true, the results will include a flag for each file indicating whether or not that file has any explicit members. The default for this field is false.
	 */
	-(void)setIncludeHasExplicitSharedMembers:(NSString*)IncludeHasExplicitSharedMembers {
		[super setInput:@"IncludeHasExplicitSharedMembers" toValue:IncludeHasExplicitSharedMembers];
	}

	/*!
	 * Set the value of the IncludeMediaInfo input for this Choreo.
	*(optional, boolean) If true, FileMetadata.media_info is set for photo and video. The default for this field is false.
	 */
	-(void)setIncludeMediaInfo:(NSString*)IncludeMediaInfo {
		[super setInput:@"IncludeMediaInfo" toValue:IncludeMediaInfo];
	}

	/*!
	 * Set the value of the Path input for this Choreo.
	*(required, string) The path to the folder you want to see the contents of. This should be empty to list contents at the root level.
	 */
	-(void)setPath:(NSString*)Path {
		[super setInput:@"Path" toValue:Path];
	}

	/*!
	 * Set the value of the Recursive input for this Choreo.
	*(optional, boolean) If true, the list folder operation will be applied recursively to all subfolders and the response will contain contents of all subfolders. The default for this field is false.
	 */
	-(void)setRecursive:(NSString*)Recursive {
		[super setInput:@"Recursive" toValue:Recursive];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListFolder Choreo.
 */
@implementation TMBDropbox_Files_ListFolder_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Cursor" output from an execution of this Choreo.
	 * @return - NSString* (string) A cursor used to retrieve the next set of results with ListFolderContinue.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getCursor {
		return [super getOutputByName:@"Cursor"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Dropbox.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Starts returning the contents of a folder.
 */
@implementation TMBDropbox_Files_ListFolder

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_Files_ListFolder Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/Files/ListFolder"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_Files_ListFolder_ResultSet *results = [[TMBDropbox_Files_ListFolder_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListFolder Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_Files_ListFolder_Inputs*)newInputSet {
		return [[TMBDropbox_Files_ListFolder_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListFolderContinue Choreo.
 */
@implementation TMBDropbox_Files_ListFolderContinue_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token for a specific Dropbox user.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the Cursor input for this Choreo.
	*(required, string) A cursor used to retrieve the next set of results.
	 */
	-(void)setCursor:(NSString*)Cursor {
		[super setInput:@"Cursor" toValue:Cursor];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListFolderContinue Choreo.
 */
@implementation TMBDropbox_Files_ListFolderContinue_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "LatestCursor" output from an execution of this Choreo.
	 * @return - NSString* (string) The latest cursor which can be used to retrieve the next set of results.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getLatestCursor {
		return [super getOutputByName:@"LatestCursor"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Dropbox.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Once a cursor has been retrieved from ListFolder, use this to paginate through all files and retrieve updates to the folder.
 */
@implementation TMBDropbox_Files_ListFolderContinue

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_Files_ListFolderContinue Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/Files/ListFolderContinue"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_Files_ListFolderContinue_ResultSet *results = [[TMBDropbox_Files_ListFolderContinue_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListFolderContinue Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_Files_ListFolderContinue_Inputs*)newInputSet {
		return [[TMBDropbox_Files_ListFolderContinue_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListRevisions Choreo.
 */
@implementation TMBDropbox_Files_ListRevisions_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token for a specific Dropbox user.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the Limit input for this Choreo.
	*(optional, integer) The number of revisions to return. The default for this field is 10.
	 */
	-(void)setLimit:(NSString*)Limit {
		[super setInput:@"Limit" toValue:Limit];
	}

	/*!
	 * Set the value of the Path input for this Choreo.
	*(conditional, string) The path to the file you want to see the revisions of.
	 */
	-(void)setPath:(NSString*)Path {
		[super setInput:@"Path" toValue:Path];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListRevisions Choreo.
 */
@implementation TMBDropbox_Files_ListRevisions_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Dropbox.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Returns revisions of a file.
 */
@implementation TMBDropbox_Files_ListRevisions

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_Files_ListRevisions Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/Files/ListRevisions"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_Files_ListRevisions_ResultSet *results = [[TMBDropbox_Files_ListRevisions_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListRevisions Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_Files_ListRevisions_Inputs*)newInputSet {
		return [[TMBDropbox_Files_ListRevisions_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Move Choreo.
 */
@implementation TMBDropbox_Files_Move_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token for a specific Dropbox user.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AllowSharedFolder input for this Choreo.
	*(optional, boolean) If true, contents are copied into a shared folder, otherwise an error will be returned if the FromPath contains a shared folder. The default for this field is false.
	 */
	-(void)setAllowSharedFolder:(NSString*)AllowSharedFolder {
		[super setInput:@"AllowSharedFolder" toValue:AllowSharedFolder];
	}

	/*!
	 * Set the value of the AutoRename input for this Choreo.
	*(optional, boolean) If there's a conflict, have the Dropbox server try to autorename the file to avoid the conflict. The default for this field is false.
	 */
	-(void)setAutoRename:(NSString*)AutoRename {
		[super setInput:@"AutoRename" toValue:AutoRename];
	}

	/*!
	 * Set the value of the FromPath input for this Choreo.
	*(required, string) Path in the user's Dropbox to be copied or moved.
	 */
	-(void)setFromPath:(NSString*)FromPath {
		[super setInput:@"FromPath" toValue:FromPath];
	}

	/*!
	 * Set the value of the ToPath input for this Choreo.
	*(required, string) Path in the user's Dropbox that is the destination.
	 */
	-(void)setToPath:(NSString*)ToPath {
		[super setInput:@"ToPath" toValue:ToPath];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Move Choreo.
 */
@implementation TMBDropbox_Files_Move_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Dropbox.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Moves a file or folder to a different location in the user's Dropbox.
 */
@implementation TMBDropbox_Files_Move

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_Files_Move Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/Files/Move"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_Files_Move_ResultSet *results = [[TMBDropbox_Files_Move_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Move Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_Files_Move_Inputs*)newInputSet {
		return [[TMBDropbox_Files_Move_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Restore Choreo.
 */
@implementation TMBDropbox_Files_Restore_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token for a specific Dropbox user.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the Path input for this Choreo.
	*(conditional, string) The path to the file you want to see the revisions of.
	 */
	-(void)setPath:(NSString*)Path {
		[super setInput:@"Path" toValue:Path];
	}

	/*!
	 * Set the value of the Revision input for this Choreo.
	*(required, string) The number of revisions to return. The default for this field is 10.
	 */
	-(void)setRevision:(NSString*)Revision {
		[super setInput:@"Revision" toValue:Revision];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Restore Choreo.
 */
@implementation TMBDropbox_Files_Restore_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Dropbox.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Restores a file to a specific revision.
 */
@implementation TMBDropbox_Files_Restore

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_Files_Restore Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/Files/Restore"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_Files_Restore_ResultSet *results = [[TMBDropbox_Files_Restore_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Restore Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_Files_Restore_Inputs*)newInputSet {
		return [[TMBDropbox_Files_Restore_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Search Choreo.
 */
@implementation TMBDropbox_Files_Search_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token for a specific Dropbox user.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the MaxResults input for this Choreo.
	*(optional, boolean) The maximum number of search results to return. The default for this field is 100.
	 */
	-(void)setMaxResults:(NSString*)MaxResults {
		[super setInput:@"MaxResults" toValue:MaxResults];
	}

	/*!
	 * Set the value of the Mode input for this Choreo.
	*(optional, string) The search mode (filename, filename_and_content, or deleted_filename). Note that searching file content is only available for Dropbox Business accounts. The default for this union is filename.
	 */
	-(void)setMode:(NSString*)Mode {
		[super setInput:@"Mode" toValue:Mode];
	}

	/*!
	 * Set the value of the Path input for this Choreo.
	*(conditional, string) Path in the user's Dropbox to search. This should be empty to search at the root level.
	 */
	-(void)setPath:(NSString*)Path {
		[super setInput:@"Path" toValue:Path];
	}

	/*!
	 * Set the value of the Query input for this Choreo.
	*(optional, boolean) The string to search for. 
	 */
	-(void)setQuery:(NSString*)Query {
		[super setInput:@"Query" toValue:Query];
	}

	/*!
	 * Set the value of the Start input for this Choreo.
	*(optional, integer) The starting index within the search results (used for paging). The default for this field is 0.
	 */
	-(void)setStart:(NSString*)Start {
		[super setInput:@"Start" toValue:Start];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Search Choreo.
 */
@implementation TMBDropbox_Files_Search_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Dropbox.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Searches for files and folders.
 */
@implementation TMBDropbox_Files_Search

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_Files_Search Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/Files/Search"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_Files_Search_ResultSet *results = [[TMBDropbox_Files_Search_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Search Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_Files_Search_Inputs*)newInputSet {
		return [[TMBDropbox_Files_Search_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Upload Choreo.
 */
@implementation TMBDropbox_Files_Upload_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token for a specific Dropbox user.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AutoRename input for this Choreo.
	*(optional, boolean) If there's a conflict, as determined by mode, have the Dropbox server try to autorename the file to avoid conflict. The default for this field is false.
	 */
	-(void)setAutoRename:(NSString*)AutoRename {
		[super setInput:@"AutoRename" toValue:AutoRename];
	}

	/*!
	 * Set the value of the ContentType input for this Choreo.
	*(optional, string) The content type of the file being uploaded. Defaults to application/octet-stream.
	 */
	-(void)setContentType:(NSString*)ContentType {
		[super setInput:@"ContentType" toValue:ContentType];
	}

	/*!
	 * Set the value of the FileContent input for this Choreo.
	*(conditional, string) The Base64-encoded file content of the file you want to upload. Required unless uploading a file from a URL using the FileURL input. Encoding is not required when ContentType is set to "text/plain".
	 */
	-(void)setFileContent:(NSString*)FileContent {
		[super setInput:@"FileContent" toValue:FileContent];
	}

	/*!
	 * Set the value of the FileURL input for this Choreo.
	*(optional, string) The public URL for file you want to upload. This can be used as an alternative to uploading Base64 encoded data.
	 */
	-(void)setFileURL:(NSString*)FileURL {
		[super setInput:@"FileURL" toValue:FileURL];
	}

	/*!
	 * Set the value of the Mode input for this Choreo.
	*(optional, string) Selects what to do if the file already exists. Valid values are: add (default), overwrite, and update.
	 */
	-(void)setMode:(NSString*)Mode {
		[super setInput:@"Mode" toValue:Mode];
	}

	/*!
	 * Set the value of the Mute input for this Choreo.
	*(optional, boolean) If true, this tells the clients that this modification shouldn't result in a user notification. The default for this field is false.
	 */
	-(void)setMute:(NSString*)Mute {
		[super setInput:@"Mute" toValue:Mute];
	}

	/*!
	 * Set the value of the Path input for this Choreo.
	*(required, string) Path in the user's Dropbox to save the file, including the filename.
	 */
	-(void)setPath:(NSString*)Path {
		[super setInput:@"Path" toValue:Path];
	}

	/*!
	 * Set the value of the Revision input for this Choreo.
	*(optional, string) The revision identifier. Used only when Mode is set to "update".
	 */
	-(void)setRevision:(NSString*)Revision {
		[super setInput:@"Revision" toValue:Revision];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Upload Choreo.
 */
@implementation TMBDropbox_Files_Upload_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	
@end

/*!
 * Creates a new file with the contents provided in the request.
 */
@implementation TMBDropbox_Files_Upload

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_Files_Upload Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/Files/Upload"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_Files_Upload_ResultSet *results = [[TMBDropbox_Files_Upload_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Upload Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_Files_Upload_Inputs*)newInputSet {
		return [[TMBDropbox_Files_Upload_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UploadSessionAppend Choreo.
 */
@implementation TMBDropbox_Files_UploadSessionAppend_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token for a specific Dropbox user.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the Close input for this Choreo.
	*(optional, boolean) If true, the current session will be closed, at which point you won't be able to call UploadSessionAppend anymore with the current session. The default for this field is false.
	 */
	-(void)setClose:(NSString*)Close {
		[super setInput:@"Close" toValue:Close];
	}

	/*!
	 * Set the value of the ContentType input for this Choreo.
	*(optional, string) The content type of the file being uploaded. Defaults to application/octet-stream.
	 */
	-(void)setContentType:(NSString*)ContentType {
		[super setInput:@"ContentType" toValue:ContentType];
	}

	/*!
	 * Set the value of the FileContent input for this Choreo.
	*(conditional, string) The next file segment to upload. Binary files should be Base64-encoded. Encoding is not required when ContentType is set to "text/plain".
	 */
	-(void)setFileContent:(NSString*)FileContent {
		[super setInput:@"FileContent" toValue:FileContent];
	}

	/*!
	 * Set the value of the Offset input for this Choreo.
	*(required, integer) The amount of data that has been uploaded so far.
	 */
	-(void)setOffset:(NSString*)Offset {
		[super setInput:@"Offset" toValue:Offset];
	}

	/*!
	 * Set the value of the SessionID input for this Choreo.
	*(conditional, string) The upload session ID returned from UploadSessionStart.
	 */
	-(void)setSessionID:(NSString*)SessionID {
		[super setInput:@"SessionID" toValue:SessionID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UploadSessionAppend Choreo.
 */
@implementation TMBDropbox_Files_UploadSessionAppend_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	
@end

/*!
 * Appends more data to an upload session.
 */
@implementation TMBDropbox_Files_UploadSessionAppend

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_Files_UploadSessionAppend Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/Files/UploadSessionAppend"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_Files_UploadSessionAppend_ResultSet *results = [[TMBDropbox_Files_UploadSessionAppend_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UploadSessionAppend Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_Files_UploadSessionAppend_Inputs*)newInputSet {
		return [[TMBDropbox_Files_UploadSessionAppend_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UploadSessionFinish Choreo.
 */
@implementation TMBDropbox_Files_UploadSessionFinish_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token for a specific Dropbox user.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AutoRename input for this Choreo.
	*(optional, boolean) If there's a conflict, as determined by mode, have the Dropbox server try to autorename the file to avoid conflict. The default for this field is false.
	 */
	-(void)setAutoRename:(NSString*)AutoRename {
		[super setInput:@"AutoRename" toValue:AutoRename];
	}

	/*!
	 * Set the value of the ContentType input for this Choreo.
	*(optional, string) The content type of the file being uploaded. Defaults to application/octet-stream.
	 */
	-(void)setContentType:(NSString*)ContentType {
		[super setInput:@"ContentType" toValue:ContentType];
	}

	/*!
	 * Set the value of the FileContent input for this Choreo.
	*(conditional, string) The remaining file content. Encoding is not required when ContentType is set to "text/plain". This can be left empty if the last file chunk was sent using UploadSessionAppend.
	 */
	-(void)setFileContent:(NSString*)FileContent {
		[super setInput:@"FileContent" toValue:FileContent];
	}

	/*!
	 * Set the value of the Mode input for this Choreo.
	*(optional, string) Selects what to do if the file already exists. Valid values are: add (default), overwrite, and update.
	 */
	-(void)setMode:(NSString*)Mode {
		[super setInput:@"Mode" toValue:Mode];
	}

	/*!
	 * Set the value of the Mute input for this Choreo.
	*(optional, boolean) If true, this tells the clients that this modification shouldn't result in a user notification. The default for this field is false.
	 */
	-(void)setMute:(NSString*)Mute {
		[super setInput:@"Mute" toValue:Mute];
	}

	/*!
	 * Set the value of the Offset input for this Choreo.
	*(required, integer) The amount of data that has been uploaded so far.
	 */
	-(void)setOffset:(NSString*)Offset {
		[super setInput:@"Offset" toValue:Offset];
	}

	/*!
	 * Set the value of the Path input for this Choreo.
	*(required, string) Path in the user's Dropbox to save the file.
	 */
	-(void)setPath:(NSString*)Path {
		[super setInput:@"Path" toValue:Path];
	}

	/*!
	 * Set the value of the Revision input for this Choreo.
	*(optional, string) The revision identifier. Used only when Mode is set to "update".
	 */
	-(void)setRevision:(NSString*)Revision {
		[super setInput:@"Revision" toValue:Revision];
	}

	/*!
	 * Set the value of the SessionID input for this Choreo.
	*(required, string) The upload session ID returned from UploadSessionStart.
	 */
	-(void)setSessionID:(NSString*)SessionID {
		[super setInput:@"SessionID" toValue:SessionID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UploadSessionFinish Choreo.
 */
@implementation TMBDropbox_Files_UploadSessionFinish_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	
@end

/*!
 * Finishes an upload session and save the uploaded data to the given file path.
 */
@implementation TMBDropbox_Files_UploadSessionFinish

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_Files_UploadSessionFinish Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/Files/UploadSessionFinish"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_Files_UploadSessionFinish_ResultSet *results = [[TMBDropbox_Files_UploadSessionFinish_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UploadSessionFinish Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_Files_UploadSessionFinish_Inputs*)newInputSet {
		return [[TMBDropbox_Files_UploadSessionFinish_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UploadSessionStart Choreo.
 */
@implementation TMBDropbox_Files_UploadSessionStart_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token for a specific Dropbox user.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the Close input for this Choreo.
	*(optional, boolean) If true, the current session will be closed, at which point you won't be able to call UploadSessionAppend anymore with the current session. The default for this field is false.
	 */
	-(void)setClose:(NSString*)Close {
		[super setInput:@"Close" toValue:Close];
	}

	/*!
	 * Set the value of the ContentType input for this Choreo.
	*(optional, string) The content type of the file being uploaded. Defaults to application/octet-stream.
	 */
	-(void)setContentType:(NSString*)ContentType {
		[super setInput:@"ContentType" toValue:ContentType];
	}

	/*!
	 * Set the value of the FileContent input for this Choreo.
	*(conditional, string) The first file segment to upload. Binary files should be Base64-encoded. Encoding is not required when ContentType is set to "text/plain".
	 */
	-(void)setFileContent:(NSString*)FileContent {
		[super setInput:@"FileContent" toValue:FileContent];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UploadSessionStart Choreo.
 */
@implementation TMBDropbox_Files_UploadSessionStart_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "SessionID" output from an execution of this Choreo.
	 * @return - NSString* (string) The upload session ID that can be used to append to the upload or finish the upload.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getSessionID {
		return [super getOutputByName:@"SessionID"];
	}
	
@end

/*!
 * Allows you to upload a single file in one or more requests. This call starts a new upload session with the first chunk of data.
 */
@implementation TMBDropbox_Files_UploadSessionStart

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_Files_UploadSessionStart Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/Files/UploadSessionStart"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_Files_UploadSessionStart_ResultSet *results = [[TMBDropbox_Files_UploadSessionStart_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UploadSessionStart Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_Files_UploadSessionStart_Inputs*)newInputSet {
		return [[TMBDropbox_Files_UploadSessionStart_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ChunkedUpload Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_ChunkedUpload_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the AppKey input for this Choreo.
	*(required, string) The App Key provided by Dropbox (AKA the OAuth Consumer Key).
	 */
	-(void)setAppKey:(NSString*)AppKey {
		[super setInput:@"AppKey" toValue:AppKey];
	}

	/*!
	 * Set the value of the AppSecret input for this Choreo.
	*(required, string) The App Secret provided by Dropbox (AKA the OAuth Consumer Secret).
	 */
	-(void)setAppSecret:(NSString*)AppSecret {
		[super setInput:@"AppSecret" toValue:AppSecret];
	}

	/*!
	 * Set the value of the Chunk input for this Choreo.
	*(conditional, string) A Base64 encoded chunk of data from the file being uploaded. If resuming and upload, the chunk should begin at the number of bytes into the file that equals the NextOffset.
	 */
	-(void)setChunk:(NSString*)Chunk {
		[super setInput:@"Chunk" toValue:Chunk];
	}

	/*!
	 * Set the value of the Offset input for this Choreo.
	*(conditional, string) The byte offset of this chunk, relative to the beginning of the full file. This is not required when uploading the first chunk of a file.
	 */
	-(void)setOffset:(NSString*)Offset {
		[super setInput:@"Offset" toValue:Offset];
	}

	/*!
	 * Set the value of the UploadID input for this Choreo.
	*(conditional, string) The ID of the upload session returned after uploading the initial file chunk. This is not required when uploading the first chunk of a file. This value is returned in the UploadSessionID output.
	 */
	-(void)setUploadID:(NSString*)UploadID {
		[super setInput:@"UploadID" toValue:UploadID];
	}

	/*!
	 * Set the value of the VaultFile input for this Choreo.
	*(optional, vault file) The path to a vault file you want to upload. Required unless using the FileContents input.
	 */
	-(void)setVaultFile:(NSString*)VaultFile {
		[super setInput:@"VaultFile" toValue:VaultFile];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ChunkedUpload Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_ChunkedUpload_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Expires" output from an execution of this Choreo.
	 * @return - NSString* (string) The expiration time of the upload.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getExpires {
		return [super getOutputByName:@"Expires"];
	}

	/*!
	 * Retrieve the value of the "NextOffset" output from an execution of this Choreo.
	 * @return - NSString* (string) The current byte offset that the server will expect. This value can be passed to the Offset input on subsequent requests when uploading chunks repeatedly.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getNextOffset {
		return [super getOutputByName:@"NextOffset"];
	}

	/*!
	 * Retrieve the value of the "UploadSessionID" output from an execution of this Choreo.
	 * @return - NSString* (string) The upload ID returned after uploading an initial file chunk. This can be passed to the UploadID input for uploading subsequent chunks, and finally to the CommitChunkedUpload Choreo.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getUploadSessionID {
		return [super getOutputByName:@"UploadSessionID"];
	}
	
@end

/*!
 * Uploads larger files to Dropbox in multiple chunks, and offers a way to resume if an upload gets interrupted.
 */
@implementation TMBDropbox_FilesAndMetadata_ChunkedUpload

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_FilesAndMetadata_ChunkedUpload Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/FilesAndMetadata/ChunkedUpload"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_FilesAndMetadata_ChunkedUpload_ResultSet *results = [[TMBDropbox_FilesAndMetadata_ChunkedUpload_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ChunkedUpload Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_FilesAndMetadata_ChunkedUpload_Inputs*)newInputSet {
		return [[TMBDropbox_FilesAndMetadata_ChunkedUpload_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CommitChunkedUpload Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_CommitChunkedUpload_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the AppKey input for this Choreo.
	*(required, string) The App Key provided by Dropbox (AKA the OAuth Consumer Key).
	 */
	-(void)setAppKey:(NSString*)AppKey {
		[super setInput:@"AppKey" toValue:AppKey];
	}

	/*!
	 * Set the value of the AppSecret input for this Choreo.
	*(required, string) The App Secret provided by Dropbox (AKA the OAuth Consumer Secret).
	 */
	-(void)setAppSecret:(NSString*)AppSecret {
		[super setInput:@"AppSecret" toValue:AppSecret];
	}

	/*!
	 * Set the value of the Locale input for this Choreo.
	*(optional, string) The metadata returned on successful upload will have its size field translated based on the given locale.
	 */
	-(void)setLocale:(NSString*)Locale {
		[super setInput:@"Locale" toValue:Locale];
	}

	/*!
	 * Set the value of the Overwrite input for this Choreo.
	*(optional, boolean) Indicates what happens when a file already exists at the specified path. If true (the default), the existing file will be overwritten. If false, the new file will be automatically renamed.
	 */
	-(void)setOverwrite:(NSString*)Overwrite {
		[super setInput:@"Overwrite" toValue:Overwrite];
	}

	/*!
	 * Set the value of the ParentRevision input for this Choreo.
	*(optional, string) The revision of the file you're editing. If this value matches the latest version of the file on the user's Dropbox, that file will be replaced, otherwise a new file will be created.
	 */
	-(void)setParentRevision:(NSString*)ParentRevision {
		[super setInput:@"ParentRevision" toValue:ParentRevision];
	}

	/*!
	 * Set the value of the Path input for this Choreo.
	*(required, string) The path to the file you want to write to (i.e. /RootFolder/SubFolder/MyFile.txt).
	 */
	-(void)setPath:(NSString*)Path {
		[super setInput:@"Path" toValue:Path];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Root input for this Choreo.
	*(optional, string) Defaults to "auto" which automatically determines the root folder using your app's permission level. Other options are "sandbox" (App Folder) and "dropbox" (Full Dropbox).
	 */
	-(void)setRoot:(NSString*)Root {
		[super setInput:@"Root" toValue:Root];
	}

	/*!
	 * Set the value of the UploadID input for this Choreo.
	*(required, string) Used to identify the chunked upload session you'd like to commit. This is returned from the ChunkedUpload Choreo.
	 */
	-(void)setUploadID:(NSString*)UploadID {
		[super setInput:@"UploadID" toValue:UploadID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CommitChunkedUpload Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_CommitChunkedUpload_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Dropbox. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Completes an upload initiated by the ChunkedUpload Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_CommitChunkedUpload

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_FilesAndMetadata_CommitChunkedUpload Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/FilesAndMetadata/CommitChunkedUpload"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_FilesAndMetadata_CommitChunkedUpload_ResultSet *results = [[TMBDropbox_FilesAndMetadata_CommitChunkedUpload_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CommitChunkedUpload Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_FilesAndMetadata_CommitChunkedUpload_Inputs*)newInputSet {
		return [[TMBDropbox_FilesAndMetadata_CommitChunkedUpload_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CreateCopyRef Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_CreateCopyRef_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the AppKey input for this Choreo.
	*(required, string) The App Key provided by Dropbox (AKA the OAuth Consumer Key).
	 */
	-(void)setAppKey:(NSString*)AppKey {
		[super setInput:@"AppKey" toValue:AppKey];
	}

	/*!
	 * Set the value of the AppSecret input for this Choreo.
	*(required, string) The App Secret provided by Dropbox (AKA the OAuth Consumer Secret).
	 */
	-(void)setAppSecret:(NSString*)AppSecret {
		[super setInput:@"AppSecret" toValue:AppSecret];
	}

	/*!
	 * Set the value of the Path input for this Choreo.
	*(required, string) The path to the file you want to make available as a copy reference, i.e.  "testfile.txt" or "/Folder/testfile.txt".
	 */
	-(void)setPath:(NSString*)Path {
		[super setInput:@"Path" toValue:Path];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Root input for this Choreo.
	*(optional, string) Defaults to "auto" which automatically determines the root folder using your app's permission level. Other options are "sandbox" (App Folder) and "dropbox" (Full Dropbox).
	 */
	-(void)setRoot:(NSString*)Root {
		[super setInput:@"Root" toValue:Root];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateCopyRef Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_CreateCopyRef_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Dropbox. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Allows you to create a copy reference (copy_ref) that you can later reference to copy files or folders to another user's Dropbox using the CopyFileOrFolder choreo. 
 */
@implementation TMBDropbox_FilesAndMetadata_CreateCopyRef

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_FilesAndMetadata_CreateCopyRef Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/FilesAndMetadata/CreateCopyRef"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_FilesAndMetadata_CreateCopyRef_ResultSet *results = [[TMBDropbox_FilesAndMetadata_CreateCopyRef_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CreateCopyRef Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_FilesAndMetadata_CreateCopyRef_Inputs*)newInputSet {
		return [[TMBDropbox_FilesAndMetadata_CreateCopyRef_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Delta Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_Delta_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the AppKey input for this Choreo.
	*(required, string) The App Key provided by Dropbox (AKA the OAuth Consumer Key).
	 */
	-(void)setAppKey:(NSString*)AppKey {
		[super setInput:@"AppKey" toValue:AppKey];
	}

	/*!
	 * Set the value of the AppSecret input for this Choreo.
	*(required, string) The App Secret provided by Dropbox (AKA the OAuth Consumer Secret).
	 */
	-(void)setAppSecret:(NSString*)AppSecret {
		[super setInput:@"AppSecret" toValue:AppSecret];
	}

	/*!
	 * Set the value of the Cursor input for this Choreo.
	*(optional, string) A string that is used to keep track of your current state. On the next call pass in this value to return delta entries that have been recorded since the cursor was returned.
	 */
	-(void)setCursor:(NSString*)Cursor {
		[super setInput:@"Cursor" toValue:Cursor];
	}

	/*!
	 * Set the value of the IncludeMediaInfo input for this Choreo.
	*(optional, boolean) If set to true, each file will include a photo_info dictionary for photos and a video_info dictionary for videos with additional media info.
	 */
	-(void)setIncludeMediaInfo:(NSString*)IncludeMediaInfo {
		[super setInput:@"IncludeMediaInfo" toValue:IncludeMediaInfo];
	}

	/*!
	 * Set the value of the Locale input for this Choreo.
	*(optional, string) The metadata returned will have its size field translated based on the given locale.
	 */
	-(void)setLocale:(NSString*)Locale {
		[super setInput:@"Locale" toValue:Locale];
	}

	/*!
	 * Set the value of the PathPrefix input for this Choreo.
	*(optional, string) Filters the response to only include entries at or under the specified path.
	 */
	-(void)setPathPrefix:(NSString*)PathPrefix {
		[super setInput:@"PathPrefix" toValue:PathPrefix];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Delta Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_Delta_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Dropbox. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Allows you keep up with changes to files and folders in a user's Dropbox.
 */
@implementation TMBDropbox_FilesAndMetadata_Delta

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_FilesAndMetadata_Delta Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/FilesAndMetadata/Delta"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_FilesAndMetadata_Delta_ResultSet *results = [[TMBDropbox_FilesAndMetadata_Delta_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Delta Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_FilesAndMetadata_Delta_Inputs*)newInputSet {
		return [[TMBDropbox_FilesAndMetadata_Delta_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetDownloadLink Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_GetDownloadLink_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the AppKey input for this Choreo.
	*(required, string) The App Key provided by Dropbox (AKA the OAuth Consumer Key).
	 */
	-(void)setAppKey:(NSString*)AppKey {
		[super setInput:@"AppKey" toValue:AppKey];
	}

	/*!
	 * Set the value of the AppSecret input for this Choreo.
	*(required, string) The App Secret provided by Dropbox (AKA the OAuth Consumer Secret).
	 */
	-(void)setAppSecret:(NSString*)AppSecret {
		[super setInput:@"AppSecret" toValue:AppSecret];
	}

	/*!
	 * Set the value of the Path input for this Choreo.
	*(required, string) The path to the file or folder you want to return a download link for (i.e. /RootFolder/SubFolder/MyFile.txt).
	 */
	-(void)setPath:(NSString*)Path {
		[super setInput:@"Path" toValue:Path];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Root input for this Choreo.
	*(optional, string) Defaults to "auto" which automatically determines the root folder using your app's permission level. Other options are "sandbox" (App Folder) and "dropbox" (Full Dropbox).
	 */
	-(void)setRoot:(NSString*)Root {
		[super setInput:@"Root" toValue:Root];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetDownloadLink Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_GetDownloadLink_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "URL" output from an execution of this Choreo.
	 * @return - NSString* (string) The file's download URL.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getURL {
		return [super getOutputByName:@"URL"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Dropbox. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a direct download link allowing an application to stream the contents of the media.
 */
@implementation TMBDropbox_FilesAndMetadata_GetDownloadLink

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_FilesAndMetadata_GetDownloadLink Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/FilesAndMetadata/GetDownloadLink"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_FilesAndMetadata_GetDownloadLink_ResultSet *results = [[TMBDropbox_FilesAndMetadata_GetDownloadLink_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetDownloadLink Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_FilesAndMetadata_GetDownloadLink_Inputs*)newInputSet {
		return [[TMBDropbox_FilesAndMetadata_GetDownloadLink_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetFile Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_GetFile_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the AppKey input for this Choreo.
	*(required, string) The App Key provided by Dropbox (AKA the OAuth Consumer Key).
	 */
	-(void)setAppKey:(NSString*)AppKey {
		[super setInput:@"AppKey" toValue:AppKey];
	}

	/*!
	 * Set the value of the AppSecret input for this Choreo.
	*(required, string) The App Secret provided by Dropbox (AKA the OAuth Consumer Secret).
	 */
	-(void)setAppSecret:(NSString*)AppSecret {
		[super setInput:@"AppSecret" toValue:AppSecret];
	}

	/*!
	 * Set the value of the EncodeFileContent input for this Choreo.
	*(optional, boolean) File content is returned as Base64 encoded data by default. Text files can be returned as Base64 decoded by setting this input to "false". Note that binary files should always be Base64 encoded.
	 */
	-(void)setEncodeFileContent:(NSString*)EncodeFileContent {
		[super setInput:@"EncodeFileContent" toValue:EncodeFileContent];
	}

	/*!
	 * Set the value of the IncludeMetadata input for this Choreo.
	*(optional, boolean) If set to true, metadata about the file is returned. Defaults to false, indicating that only the file content is returned.
	 */
	-(void)setIncludeMetadata:(NSString*)IncludeMetadata {
		[super setInput:@"IncludeMetadata" toValue:IncludeMetadata];
	}

	/*!
	 * Set the value of the Path input for this Choreo.
	*(required, string) The path to file you want to retrieve (i.e. RootFolder/SubFolder/MyFile.txt). Only the file name is necessary when the file is at the root level.
	 */
	-(void)setPath:(NSString*)Path {
		[super setInput:@"Path" toValue:Path];
	}

	/*!
	 * Set the value of the Root input for this Choreo.
	*(optional, string) Defaults to "auto" which automatically determines the root folder using your app's permission level. Other options are "sandbox" (App Folder) and "dropbox" (Full Dropbox).
	 */
	-(void)setRoot:(NSString*)Root {
		[super setInput:@"Root" toValue:Root];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetFile Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_GetFile_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "FileMetadata" output from an execution of this Choreo.
	 * @return - NSString* (json) The metadata for the file. This only returned when IncludeMetadata is set to true.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getFileMetadata {
		return [super getOutputByName:@"FileMetadata"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (string) The response from Dropbox. The response will contain the contents of the file you are retrieving as Base64 encoded data.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Gets the content and metadata for a specified file.
 */
@implementation TMBDropbox_FilesAndMetadata_GetFile

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_FilesAndMetadata_GetFile Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/FilesAndMetadata/GetFile"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_FilesAndMetadata_GetFile_ResultSet *results = [[TMBDropbox_FilesAndMetadata_GetFile_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetFile Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_FilesAndMetadata_GetFile_Inputs*)newInputSet {
		return [[TMBDropbox_FilesAndMetadata_GetFile_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetFileRevisions Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_GetFileRevisions_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the AppKey input for this Choreo.
	*(required, string) The App Key provided by Dropbox (AKA the OAuth Consumer Key).
	 */
	-(void)setAppKey:(NSString*)AppKey {
		[super setInput:@"AppKey" toValue:AppKey];
	}

	/*!
	 * Set the value of the AppSecret input for this Choreo.
	*(required, string) The App Secret provided by Dropbox (AKA the OAuth Consumer Secret).
	 */
	-(void)setAppSecret:(NSString*)AppSecret {
		[super setInput:@"AppSecret" toValue:AppSecret];
	}

	/*!
	 * Set the value of the Path input for this Choreo.
	*(required, string) The path to the file that you want to return revisions for (i.e. /RootFolder/SubFolder/MyFile.txt).
	 */
	-(void)setPath:(NSString*)Path {
		[super setInput:@"Path" toValue:Path];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the RevisionLimit input for this Choreo.
	*(optional, integer) Default is 10. Max is 1,000. When listing a file, the service will not report listings containing more than the amount specified here.
	 */
	-(void)setRevisionLimit:(NSString*)RevisionLimit {
		[super setInput:@"RevisionLimit" toValue:RevisionLimit];
	}

	/*!
	 * Set the value of the Root input for this Choreo.
	*(optional, string) Defaults to "auto" which automatically determines the root folder using your app's permission level. Other options are "sandbox" (App Folder) and "dropbox" (Full Dropbox).
	 */
	-(void)setRoot:(NSString*)Root {
		[super setInput:@"Root" toValue:Root];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetFileRevisions Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_GetFileRevisions_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Dropbox. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves metadata for the previous revisions of a file
 */
@implementation TMBDropbox_FilesAndMetadata_GetFileRevisions

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_FilesAndMetadata_GetFileRevisions Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/FilesAndMetadata/GetFileRevisions"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_FilesAndMetadata_GetFileRevisions_ResultSet *results = [[TMBDropbox_FilesAndMetadata_GetFileRevisions_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetFileRevisions Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_FilesAndMetadata_GetFileRevisions_Inputs*)newInputSet {
		return [[TMBDropbox_FilesAndMetadata_GetFileRevisions_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetShareableLink Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_GetShareableLink_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the AppKey input for this Choreo.
	*(required, string) The App Key provided by Dropbox (AKA the OAuth Consumer Key).
	 */
	-(void)setAppKey:(NSString*)AppKey {
		[super setInput:@"AppKey" toValue:AppKey];
	}

	/*!
	 * Set the value of the AppSecret input for this Choreo.
	*(required, string) The App Secret provided by Dropbox (AKA the OAuth Consumer Secret).
	 */
	-(void)setAppSecret:(NSString*)AppSecret {
		[super setInput:@"AppSecret" toValue:AppSecret];
	}

	/*!
	 * Set the value of the Path input for this Choreo.
	*(required, string) The path to the file or folder you want to return a shareable link for (i.e. /RootFolder/SubFolder/MyFile.txt).
	 */
	-(void)setPath:(NSString*)Path {
		[super setInput:@"Path" toValue:Path];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Root input for this Choreo.
	*(optional, string) Defaults to "auto" which automatically determines the root folder using your app's permission level. Other options are "sandbox" (App Folder) and "dropbox" (Full Dropbox).
	 */
	-(void)setRoot:(NSString*)Root {
		[super setInput:@"Root" toValue:Root];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetShareableLink Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_GetShareableLink_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Dropbox. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a shareable link to files or folders.
 */
@implementation TMBDropbox_FilesAndMetadata_GetShareableLink

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_FilesAndMetadata_GetShareableLink Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/FilesAndMetadata/GetShareableLink"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_FilesAndMetadata_GetShareableLink_ResultSet *results = [[TMBDropbox_FilesAndMetadata_GetShareableLink_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetShareableLink Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_FilesAndMetadata_GetShareableLink_Inputs*)newInputSet {
		return [[TMBDropbox_FilesAndMetadata_GetShareableLink_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetThumbnail Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_GetThumbnail_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the AppKey input for this Choreo.
	*(required, string) The App Key provided by Dropbox (AKA the OAuth Consumer Key).
	 */
	-(void)setAppKey:(NSString*)AppKey {
		[super setInput:@"AppKey" toValue:AppKey];
	}

	/*!
	 * Set the value of the AppSecret input for this Choreo.
	*(required, string) The App Secret provided by Dropbox (AKA the OAuth Consumer Secret).
	 */
	-(void)setAppSecret:(NSString*)AppSecret {
		[super setInput:@"AppSecret" toValue:AppSecret];
	}

	/*!
	 * Set the value of the ImageFormat input for this Choreo.
	*(optional, string) The thumbnail format to return for the specified image. Accepted values are: jpeg (default) or png.
	 */
	-(void)setImageFormat:(NSString*)ImageFormat {
		[super setInput:@"ImageFormat" toValue:ImageFormat];
	}

	/*!
	 * Set the value of the Path input for this Choreo.
	*(required, string) The path to the file that you want to generate a thumbnail for (i.e. /RootFolder/SubFolder/MyFile.txt).
	 */
	-(void)setPath:(NSString*)Path {
		[super setInput:@"Path" toValue:Path];
	}

	/*!
	 * Set the value of the Root input for this Choreo.
	*(optional, string) Defaults to "auto" which automatically determines the root folder using your app's permission level. Other options are "sandbox" (App Folder) and "dropbox" (Full Dropbox).
	 */
	-(void)setRoot:(NSString*)Root {
		[super setInput:@"Root" toValue:Root];
	}

	/*!
	 * Set the value of the Size input for this Choreo.
	*(optional, string) The size of the thumbnail to generate. Accepted values are: small, medium, s, m, l, xl. See Choreo documentation for exact dimensions. Defaults to "small".
	 */
	-(void)setSize:(NSString*)Size {
		[super setInput:@"Size" toValue:Size];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetThumbnail Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_GetThumbnail_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (string) The base64 encoded image content of the thumbnail.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves a thumbnail for a specified image.
 */
@implementation TMBDropbox_FilesAndMetadata_GetThumbnail

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_FilesAndMetadata_GetThumbnail Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/FilesAndMetadata/GetThumbnail"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_FilesAndMetadata_GetThumbnail_ResultSet *results = [[TMBDropbox_FilesAndMetadata_GetThumbnail_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetThumbnail Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_FilesAndMetadata_GetThumbnail_Inputs*)newInputSet {
		return [[TMBDropbox_FilesAndMetadata_GetThumbnail_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the LatestCursor Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_LatestCursor_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the AppKey input for this Choreo.
	*(required, string) The App Key provided by Dropbox (AKA the OAuth Consumer Key).
	 */
	-(void)setAppKey:(NSString*)AppKey {
		[super setInput:@"AppKey" toValue:AppKey];
	}

	/*!
	 * Set the value of the AppSecret input for this Choreo.
	*(required, string) The App Secret provided by Dropbox (AKA the OAuth Consumer Secret).
	 */
	-(void)setAppSecret:(NSString*)AppSecret {
		[super setInput:@"AppSecret" toValue:AppSecret];
	}

	/*!
	 * Set the value of the IncludeMediaInfo input for this Choreo.
	*(optional, boolean) If true, the returned cursor will be encoded with include_media_info set to true for use with the Delta Choreo.
	 */
	-(void)setIncludeMediaInfo:(NSString*)IncludeMediaInfo {
		[super setInput:@"IncludeMediaInfo" toValue:IncludeMediaInfo];
	}

	/*!
	 * Set the value of the PathPrefix input for this Choreo.
	*(optional, string) If present, the returned cursor will be encoded with a path_prefix for the specified path for use with the Delta Choreo.
	 */
	-(void)setPathPrefix:(NSString*)PathPrefix {
		[super setInput:@"PathPrefix" toValue:PathPrefix];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the LatestCursor Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_LatestCursor_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Dropbox. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Allows you to quickly get a cursor for the server's state, for use with the Delta Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_LatestCursor

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_FilesAndMetadata_LatestCursor Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/FilesAndMetadata/LatestCursor"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_FilesAndMetadata_LatestCursor_ResultSet *results = [[TMBDropbox_FilesAndMetadata_LatestCursor_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the LatestCursor Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_FilesAndMetadata_LatestCursor_Inputs*)newInputSet {
		return [[TMBDropbox_FilesAndMetadata_LatestCursor_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the ListFolderContents Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_ListFolderContents_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the AppKey input for this Choreo.
	*(required, string) The App Key provided by Dropbox (AKA the OAuth Consumer Key).
	 */
	-(void)setAppKey:(NSString*)AppKey {
		[super setInput:@"AppKey" toValue:AppKey];
	}

	/*!
	 * Set the value of the AppSecret input for this Choreo.
	*(required, string) The App Secret provided by Dropbox (AKA the OAuth Consumer Secret).
	 */
	-(void)setAppSecret:(NSString*)AppSecret {
		[super setInput:@"AppSecret" toValue:AppSecret];
	}

	/*!
	 * Set the value of the FileLimit input for this Choreo.
	*(optional, integer) Dropbox will not return a list that exceeds this specified limit. Defaults to 10,000.
	 */
	-(void)setFileLimit:(NSString*)FileLimit {
		[super setInput:@"FileLimit" toValue:FileLimit];
	}

	/*!
	 * Set the value of the Folder input for this Choreo.
	*(optional, string) The path to a folder for which to retrieve metadata (i.e. /RootFolder/SubFolder). Note that a path to file can also be passed.
	 */
	-(void)setFolder:(NSString*)Folder {
		[super setInput:@"Folder" toValue:Folder];
	}

	/*!
	 * Set the value of the Hash input for this Choreo.
	*(optional, string) The value of a hash field from a previous request to get metadata on a folder. When provided, a 304 (not Modified) status code is returned instead of a folder listing if no changes have been made.
	 */
	-(void)setHash:(NSString*)Hash {
		[super setInput:@"Hash" toValue:Hash];
	}

	/*!
	 * Set the value of the IncludeDeleted input for this Choreo.
	*(optional, boolean) Only applicable when List is set. If this parameter is set to true, contents will include the metadata of deleted children.
	 */
	-(void)setIncludeDeleted:(NSString*)IncludeDeleted {
		[super setInput:@"IncludeDeleted" toValue:IncludeDeleted];
	}

	/*!
	 * Set the value of the List input for this Choreo.
	*(optional, boolean) If true (the default), the folder's metadata will include a contents field with a list of metadata entries for the contents of the folder.
	 */
	-(void)setList:(NSString*)List {
		[super setInput:@"List" toValue:List];
	}

	/*!
	 * Set the value of the Locale input for this Choreo.
	*(optional, string) If your app supports any language other than English, insert the appropriate IETF language tag, and the metadata returned will have its size field translated based on the given locale (e.g., pt-BR).
	 */
	-(void)setLocale:(NSString*)Locale {
		[super setInput:@"Locale" toValue:Locale];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Revision input for this Choreo.
	*(optional, string) When including a particular revision number, only the metadata for that revision will be returned.
	 */
	-(void)setRevision:(NSString*)Revision {
		[super setInput:@"Revision" toValue:Revision];
	}

	/*!
	 * Set the value of the Root input for this Choreo.
	*(optional, string) Defaults to "auto" which automatically determines the root folder using your app's permission level. Other options are "sandbox" (App Folder) and "dropbox" (Full Dropbox).
	 */
	-(void)setRoot:(NSString*)Root {
		[super setInput:@"Root" toValue:Root];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListFolderContents Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_ListFolderContents_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "ResponseStatusCode" output from an execution of this Choreo.
	 * @return - NSString* (integer) The response status code returned from Dropbox.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponseStatusCode {
		return [super getOutputByName:@"ResponseStatusCode"];
	}

	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Dropbox. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Retrieves metadata (including folder contents) for a folder or file in Dropbox.
 */
@implementation TMBDropbox_FilesAndMetadata_ListFolderContents

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_FilesAndMetadata_ListFolderContents Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/FilesAndMetadata/ListFolderContents"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_FilesAndMetadata_ListFolderContents_ResultSet *results = [[TMBDropbox_FilesAndMetadata_ListFolderContents_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the ListFolderContents Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_FilesAndMetadata_ListFolderContents_Inputs*)newInputSet {
		return [[TMBDropbox_FilesAndMetadata_ListFolderContents_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the LongPollDelta Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_LongPollDelta_Inputs

	/*!
	 * Set the value of the Cursor input for this Choreo.
	*(required, string) A delta cursor as returned from a call to the Delta Choreo.
	 */
	-(void)setCursor:(NSString*)Cursor {
		[super setInput:@"Cursor" toValue:Cursor];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Timeout input for this Choreo.
	*(optional, string) An integer indicating the amount of time, in seconds, to wait for a response. The default value is 30 seconds, which is also the minimum allowed value. The maximum is 480 seconds.
	 */
	-(void)setTimeout:(NSString*)Timeout {
		[super setInput:@"Timeout" toValue:Timeout];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the LongPollDelta Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_LongPollDelta_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Dropbox. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Used in conjunction with the Delta Choreo, this allows you to perform a long-poll request to wait for changes on an account.
 */
@implementation TMBDropbox_FilesAndMetadata_LongPollDelta

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_FilesAndMetadata_LongPollDelta Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/FilesAndMetadata/LongPollDelta"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_FilesAndMetadata_LongPollDelta_ResultSet *results = [[TMBDropbox_FilesAndMetadata_LongPollDelta_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the LongPollDelta Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_FilesAndMetadata_LongPollDelta_Inputs*)newInputSet {
		return [[TMBDropbox_FilesAndMetadata_LongPollDelta_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the RestoreFileToRevision Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_RestoreFileToRevision_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the AppKey input for this Choreo.
	*(required, string) The App Key provided by Dropbox (AKA the OAuth Consumer Key).
	 */
	-(void)setAppKey:(NSString*)AppKey {
		[super setInput:@"AppKey" toValue:AppKey];
	}

	/*!
	 * Set the value of the AppSecret input for this Choreo.
	*(required, string) The App Secret provided by Dropbox (AKA the OAuth Consumer Secret).
	 */
	-(void)setAppSecret:(NSString*)AppSecret {
		[super setInput:@"AppSecret" toValue:AppSecret];
	}

	/*!
	 * Set the value of the Path input for this Choreo.
	*(required, string) The path to the file that you want to return revisions for (i.e. /RootFolder/SubFolder/MyFile.txt).
	 */
	-(void)setPath:(NSString*)Path {
		[super setInput:@"Path" toValue:Path];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Revision input for this Choreo.
	*(required, string) The revision of the file to restore. The revision ID can be retrieved by running the GetFileRevision Choreo.
	 */
	-(void)setRevision:(NSString*)Revision {
		[super setInput:@"Revision" toValue:Revision];
	}

	/*!
	 * Set the value of the Root input for this Choreo.
	*(optional, string) Defaults to "auto" which automatically determines the root folder using your app's permission level. Other options are "sandbox" (App Folder) and "dropbox" (Full Dropbox).
	 */
	-(void)setRoot:(NSString*)Root {
		[super setInput:@"Root" toValue:Root];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the RestoreFileToRevision Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_RestoreFileToRevision_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Dropbox. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Restores a specified file to a previous revision.
 */
@implementation TMBDropbox_FilesAndMetadata_RestoreFileToRevision

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_FilesAndMetadata_RestoreFileToRevision Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/FilesAndMetadata/RestoreFileToRevision"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_FilesAndMetadata_RestoreFileToRevision_ResultSet *results = [[TMBDropbox_FilesAndMetadata_RestoreFileToRevision_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the RestoreFileToRevision Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_FilesAndMetadata_RestoreFileToRevision_Inputs*)newInputSet {
		return [[TMBDropbox_FilesAndMetadata_RestoreFileToRevision_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the SearchFilesAndFolders Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_SearchFilesAndFolders_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the AppKey input for this Choreo.
	*(required, string) The App Key provided by Dropbox (AKA the OAuth Consumer Key).
	 */
	-(void)setAppKey:(NSString*)AppKey {
		[super setInput:@"AppKey" toValue:AppKey];
	}

	/*!
	 * Set the value of the AppSecret input for this Choreo.
	*(required, string) The App Secret provided by Dropbox (AKA the OAuth Consumer Secret).
	 */
	-(void)setAppSecret:(NSString*)AppSecret {
		[super setInput:@"AppSecret" toValue:AppSecret];
	}

	/*!
	 * Set the value of the FileLimit input for this Choreo.
	*(optional, integer) Dropbox will not return a list that exceeds this specified limit. Defaults to 10,000.
	 */
	-(void)setFileLimit:(NSString*)FileLimit {
		[super setInput:@"FileLimit" toValue:FileLimit];
	}

	/*!
	 * Set the value of the Path input for this Choreo.
	*(optional, string) The path to the folder you want to search from (i.e. /RootFolder/SubFolder). Leave blank to search ALL.
	 */
	-(void)setPath:(NSString*)Path {
		[super setInput:@"Path" toValue:Path];
	}

	/*!
	 * Set the value of the Query input for this Choreo.
	*(required, string) The search string. Must be at least three characters long.
	 */
	-(void)setQuery:(NSString*)Query {
		[super setInput:@"Query" toValue:Query];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Root input for this Choreo.
	*(optional, string) Defaults to "auto" which automatically determines the root folder using your app's permission level. Other options are "sandbox" (App Folder) and "dropbox" (Full Dropbox).
	 */
	-(void)setRoot:(NSString*)Root {
		[super setInput:@"Root" toValue:Root];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SearchFilesAndFolders Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_SearchFilesAndFolders_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Dropbox. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Allows you to search Dropbox for files or folders by a keyword search.
 */
@implementation TMBDropbox_FilesAndMetadata_SearchFilesAndFolders

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_FilesAndMetadata_SearchFilesAndFolders Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/FilesAndMetadata/SearchFilesAndFolders"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_FilesAndMetadata_SearchFilesAndFolders_ResultSet *results = [[TMBDropbox_FilesAndMetadata_SearchFilesAndFolders_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the SearchFilesAndFolders Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_FilesAndMetadata_SearchFilesAndFolders_Inputs*)newInputSet {
		return [[TMBDropbox_FilesAndMetadata_SearchFilesAndFolders_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the UploadFile Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_UploadFile_Inputs

	/*!
	 * Set the value of the FileContents input for this Choreo.
	*(conditional, string) The Base64-encoded contents of the file you want to upload. Required UNLESS uploading a file from a URL using the FileContentsFromURL input variable.
	 */
	-(void)setFileContents:(NSString*)FileContents {
		[super setInput:@"FileContents" toValue:FileContents];
	}

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The Access Token retrieved during the OAuth process.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) The Access Token Secret retrieved during the OAuth process.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the AppKey input for this Choreo.
	*(required, string) The App Key provided by Dropbox (AKA the OAuth Consumer Key).
	 */
	-(void)setAppKey:(NSString*)AppKey {
		[super setInput:@"AppKey" toValue:AppKey];
	}

	/*!
	 * Set the value of the AppSecret input for this Choreo.
	*(required, string) The App Secret provided by Dropbox (AKA the OAuth Consumer Secret).
	 */
	-(void)setAppSecret:(NSString*)AppSecret {
		[super setInput:@"AppSecret" toValue:AppSecret];
	}

	/*!
	 * Set the value of the FileContentsFromURL input for this Choreo.
	*(conditional, string) URL for file you want to upload. Alternative to uploading Base64Encoded data. If specifiying URL, leave FileContents blank. Valid URLs: http(s) requests only.
	 */
	-(void)setFileContentsFromURL:(NSString*)FileContentsFromURL {
		[super setInput:@"FileContentsFromURL" toValue:FileContentsFromURL];
	}

	/*!
	 * Set the value of the FileName input for this Choreo.
	*(required, string) The name of the file you are uploading to Dropbox.
	 */
	-(void)setFileName:(NSString*)FileName {
		[super setInput:@"FileName" toValue:FileName];
	}

	/*!
	 * Set the value of the Folder input for this Choreo.
	*(optional, string) The name of the folder that that you want to upload a file to. A path to a sub-folder is also valid (i.e. /RootFolder/SubFolder).
	 */
	-(void)setFolder:(NSString*)Folder {
		[super setInput:@"Folder" toValue:Folder];
	}

	/*!
	 * Set the value of the ResponseFormat input for this Choreo.
	*(optional, string) The format that the response should be in. Can be set to xml or json. Defaults to json.
	 */
	-(void)setResponseFormat:(NSString*)ResponseFormat {
		[super setInput:@"ResponseFormat" toValue:ResponseFormat];
	}

	/*!
	 * Set the value of the Root input for this Choreo.
	*(optional, string) Defaults to "auto" which automatically determines the root folder using your app's permission level. Other options are "sandbox" (App Folder) and "dropbox" (Full Dropbox).
	 */
	-(void)setRoot:(NSString*)Root {
		[super setInput:@"Root" toValue:Root];
	}

	/*!
	 * Set the value of the VaultFile input for this Choreo.
	*(optional, vault file) The path to a vault file you want to upload. Required unless using the FileContents input.
	 */
	-(void)setVaultFile:(NSString*)VaultFile {
		[super setInput:@"VaultFile" toValue:VaultFile];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UploadFile Choreo.
 */
@implementation TMBDropbox_FilesAndMetadata_UploadFile_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* The response from Dropbox. Corresponds to the ResponseFormat input. Defaults to json.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Uploads a file to your Dropbox account.
 */
@implementation TMBDropbox_FilesAndMetadata_UploadFile

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_FilesAndMetadata_UploadFile Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/FilesAndMetadata/UploadFile"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_FilesAndMetadata_UploadFile_ResultSet *results = [[TMBDropbox_FilesAndMetadata_UploadFile_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the UploadFile Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_FilesAndMetadata_UploadFile_Inputs*)newInputSet {
		return [[TMBDropbox_FilesAndMetadata_UploadFile_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the FinalizeOAuth Choreo.
 */
@implementation TMBDropbox_OAuth_FinalizeOAuth_Inputs

	/*!
	 * Set the value of the AccountName input for this Choreo.
	*(optional, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setAccountName:(NSString*)AccountName {
		[super setInput:@"AccountName" toValue:AccountName];
	}

	/*!
	 * Set the value of the AppKeyName input for this Choreo.
	*(optional, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setAppKeyName:(NSString*)AppKeyName {
		[super setInput:@"AppKeyName" toValue:AppKeyName];
	}

	/*!
	 * Set the value of the AppKeyValue input for this Choreo.
	*(optional, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setAppKeyValue:(NSString*)AppKeyValue {
		[super setInput:@"AppKeyValue" toValue:AppKeyValue];
	}

	/*!
	 * Set the value of the CallbackID input for this Choreo.
	*(required, string) The callback token returned by the InitializeOAuth Choreo. Used to retrieve the callback data after the user authorizes.
	 */
	-(void)setCallbackID:(NSString*)CallbackID {
		[super setInput:@"CallbackID" toValue:CallbackID];
	}

	/*!
	 * Set the value of the DropboxAppKey input for this Choreo.
	*(required, string) The APP Key provided by Dropbox (AKA the OAuth Consumer Key).
	 */
	-(void)setDropboxAppKey:(NSString*)DropboxAppKey {
		[super setInput:@"DropboxAppKey" toValue:DropboxAppKey];
	}

	/*!
	 * Set the value of the DropboxAppSecret input for this Choreo.
	*(required, string) The App Secret provided by Dropbox (AKA the OAuth Consumer Secret).
	 */
	-(void)setDropboxAppSecret:(NSString*)DropboxAppSecret {
		[super setInput:@"DropboxAppSecret" toValue:DropboxAppSecret];
	}

	/*!
	 * Set the value of the OAuthTokenSecret input for this Choreo.
	*(required, string) The OAuthTokenSecret returned by the InitializeOAuth Choreo.
	 */
	-(void)setOAuthTokenSecret:(NSString*)OAuthTokenSecret {
		[super setInput:@"OAuthTokenSecret" toValue:OAuthTokenSecret];
	}

	/*!
	 * Set the value of the SuppressErrors input for this Choreo.
	*(optional, boolean) When set to true, errors received during the OAuth redirect process will be suppressed and returned in the ErrorMessage output.
	 */
	-(void)setSuppressErrors:(NSString*)SuppressErrors {
		[super setInput:@"SuppressErrors" toValue:SuppressErrors];
	}

	/*!
	 * Set the value of the Timeout input for this Choreo.
	*(optional, integer) The amount of time (in seconds) to poll your Temboo callback URL to see if your app's user has allowed or denied the request for access. Defaults to 20. Max is 60.
	 */
	-(void)setTimeout:(NSString*)Timeout {
		[super setInput:@"Timeout" toValue:Timeout];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the FinalizeOAuth Choreo.
 */
@implementation TMBDropbox_OAuth_FinalizeOAuth_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "AccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) The Access Token retrieved during the OAuth process.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getAccessToken {
		return [super getOutputByName:@"AccessToken"];
	}

	/*!
	 * Retrieve the value of the "AccessTokenSecret" output from an execution of this Choreo.
	 * @return - NSString* (string) The Access Token Secret retrieved during the OAuth process.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getAccessTokenSecret {
		return [super getOutputByName:@"AccessTokenSecret"];
	}

	/*!
	 * Retrieve the value of the "ErrorMessage" output from an execution of this Choreo.
	 * @return - NSString* (string) Contains an error message if an error occurs during the OAuth redirect process and if SuppressErrors is set to true.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getErrorMessage {
		return [super getOutputByName:@"ErrorMessage"];
	}

	/*!
	 * Retrieve the value of the "UserID" output from an execution of this Choreo.
	 * @return - NSString* (integer) The Dropbox user ID associated with the access token and secret returned.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getUserID {
		return [super getOutputByName:@"UserID"];
	}
	
@end

/*!
 * Completes the OAuth process by retrieving a Dropbox access token and access token secret for a user, after they have visited the authorization URL returned by the InitializeOAuth choreo and clicked "allow."
 */
@implementation TMBDropbox_OAuth_FinalizeOAuth

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_OAuth_FinalizeOAuth Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/OAuth/FinalizeOAuth"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_OAuth_FinalizeOAuth_ResultSet *results = [[TMBDropbox_OAuth_FinalizeOAuth_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the FinalizeOAuth Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_OAuth_FinalizeOAuth_Inputs*)newInputSet {
		return [[TMBDropbox_OAuth_FinalizeOAuth_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the FromOAuth1 Choreo.
 */
@implementation TMBDropbox_OAuth_FromOAuth1_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) An OAuth 1.0 access token for a specific Dropbox user.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccessTokenSecret input for this Choreo.
	*(required, string) An OAuth 1.0 access token secret for a specific Dropbox user.
	 */
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret {
		[super setInput:@"AccessTokenSecret" toValue:AccessTokenSecret];
	}

	/*!
	 * Set the value of the AppKey input for this Choreo.
	*(required, string) The App Key provided by Dropbox after registering your application.
	 */
	-(void)setAppKey:(NSString*)AppKey {
		[super setInput:@"AppKey" toValue:AppKey];
	}

	/*!
	 * Set the value of the AppSecret input for this Choreo.
	*(required, string) The App Secret provided by Dropbox after registering your application.
	 */
	-(void)setAppSecret:(NSString*)AppSecret {
		[super setInput:@"AppSecret" toValue:AppSecret];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the FromOAuth1 Choreo.
 */
@implementation TMBDropbox_OAuth_FromOAuth1_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "OAuth2AccessToken" output from an execution of this Choreo.
	 * @return - NSString* (string) The OAuth 2.0 access for a specific Dropbox user.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getOAuth2AccessToken {
		return [super getOutputByName:@"OAuth2AccessToken"];
	}
	
@end

/*!
 * Creates an OAuth 2.0 access token from the supplied OAuth 1.0 access token.
 */
@implementation TMBDropbox_OAuth_FromOAuth1

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_OAuth_FromOAuth1 Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/OAuth/FromOAuth1"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_OAuth_FromOAuth1_ResultSet *results = [[TMBDropbox_OAuth_FromOAuth1_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the FromOAuth1 Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_OAuth_FromOAuth1_Inputs*)newInputSet {
		return [[TMBDropbox_OAuth_FromOAuth1_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the InitializeOAuth Choreo.
 */
@implementation TMBDropbox_OAuth_InitializeOAuth_Inputs

	/*!
	 * Set the value of the AccountName input for this Choreo.
	*(optional, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setAccountName:(NSString*)AccountName {
		[super setInput:@"AccountName" toValue:AccountName];
	}

	/*!
	 * Set the value of the AppKeyName input for this Choreo.
	*(optional, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setAppKeyName:(NSString*)AppKeyName {
		[super setInput:@"AppKeyName" toValue:AppKeyName];
	}

	/*!
	 * Set the value of the AppKeyValue input for this Choreo.
	*(optional, string) Deprecated (retained for backward compatibility only).
	 */
	-(void)setAppKeyValue:(NSString*)AppKeyValue {
		[super setInput:@"AppKeyValue" toValue:AppKeyValue];
	}

	/*!
	 * Set the value of the DropboxAppKey input for this Choreo.
	*(required, string) The App Key provided by Dropbox (AKA the OAuth Consumer Key).
	 */
	-(void)setDropboxAppKey:(NSString*)DropboxAppKey {
		[super setInput:@"DropboxAppKey" toValue:DropboxAppKey];
	}

	/*!
	 * Set the value of the DropboxAppSecret input for this Choreo.
	*(required, string) The AppSecret provided by Dropbox (AKA the OAuth Consumer Secret).
	 */
	-(void)setDropboxAppSecret:(NSString*)DropboxAppSecret {
		[super setInput:@"DropboxAppSecret" toValue:DropboxAppSecret];
	}

	/*!
	 * Set the value of the ForwardingURL input for this Choreo.
	*(optional, string) The URL that Temboo will redirect your users to after they grant access to your application. This should include the "https://" or "http://" prefix and be a fully qualified URL.
	 */
	-(void)setForwardingURL:(NSString*)ForwardingURL {
		[super setInput:@"ForwardingURL" toValue:ForwardingURL];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the InitializeOAuth Choreo.
 */
@implementation TMBDropbox_OAuth_InitializeOAuth_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "AuthorizationURL" output from an execution of this Choreo.
	 * @return - NSString* (string) The authorization URL that the application's user needs to go to in order to grant access to your application.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getAuthorizationURL {
		return [super getOutputByName:@"AuthorizationURL"];
	}

	/*!
	 * Retrieve the value of the "CallbackID" output from an execution of this Choreo.
	 * @return - NSString* (string) An ID used to retrieve the callback data that Temboo stores once your application's user authorizes.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getCallbackID {
		return [super getOutputByName:@"CallbackID"];
	}

	/*!
	 * Retrieve the value of the "OAuthTokenSecret" output from an execution of this Choreo.
	 * @return - NSString* (string) The temporary OAuth Token Secret that can be exchanged for a final token secret using the FinalizeOAuth Choreo.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getOAuthTokenSecret {
		return [super getOutputByName:@"OAuthTokenSecret"];
	}
	
@end

/*!
 * Generates an authorization URL that an application can use to complete the first step in the OAuth process.
 */
@implementation TMBDropbox_OAuth_InitializeOAuth

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_OAuth_InitializeOAuth Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/OAuth/InitializeOAuth"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_OAuth_InitializeOAuth_ResultSet *results = [[TMBDropbox_OAuth_InitializeOAuth_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the InitializeOAuth Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_OAuth_InitializeOAuth_Inputs*)newInputSet {
		return [[TMBDropbox_OAuth_InitializeOAuth_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the Revoke Choreo.
 */
@implementation TMBDropbox_OAuth_Revoke_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token for a specific Dropbox user.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Revoke Choreo.
 */
@implementation TMBDropbox_OAuth_Revoke_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Dropbox.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Disables the access token used to authenticate the call.
 */
@implementation TMBDropbox_OAuth_Revoke

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_OAuth_Revoke Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/OAuth/Revoke"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_OAuth_Revoke_ResultSet *results = [[TMBDropbox_OAuth_Revoke_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the Revoke Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_OAuth_Revoke_Inputs*)newInputSet {
		return [[TMBDropbox_OAuth_Revoke_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the CreateSharedLink Choreo.
 */
@implementation TMBDropbox_Sharing_CreateSharedLink_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token for a specific Dropbox user.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the Path input for this Choreo.
	*(required, string) The path to share.
	 */
	-(void)setPath:(NSString*)Path {
		[super setInput:@"Path" toValue:Path];
	}

	/*!
	 * Set the value of the ShortURL input for this Choreo.
	*(optional, boolean) Whether to return a shortened URL. The default for this field is false.
	 */
	-(void)setShortURL:(NSString*)ShortURL {
		[super setInput:@"ShortURL" toValue:ShortURL];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateSharedLink Choreo.
 */
@implementation TMBDropbox_Sharing_CreateSharedLink_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Dropbox.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Creates a shared link.
 */
@implementation TMBDropbox_Sharing_CreateSharedLink

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_Sharing_CreateSharedLink Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/Sharing/CreateSharedLink"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_Sharing_CreateSharedLink_ResultSet *results = [[TMBDropbox_Sharing_CreateSharedLink_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the CreateSharedLink Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_Sharing_CreateSharedLink_Inputs*)newInputSet {
		return [[TMBDropbox_Sharing_CreateSharedLink_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the RevokeSharedLink Choreo.
 */
@implementation TMBDropbox_Sharing_RevokeSharedLink_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token for a specific Dropbox user.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the URL input for this Choreo.
	*(required, string) The URL of the shared link.
	 */
	-(void)setURL:(NSString*)URL {
		[super setInput:@"URL" toValue:URL];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the RevokeSharedLink Choreo.
 */
@implementation TMBDropbox_Sharing_RevokeSharedLink_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	/*!
	 * Retrieve the value of the "Response" output from an execution of this Choreo.
	 * @return - NSString* (json) The response from Dropbox.
	 * @throws TMBExceptionNotFound if output does not exist. (Note an empty response is considered valid)
	 */
	-(NSString*)getResponse {
		return [super getOutputByName:@"Response"];
	}
	
@end

/*!
 * Revokes a shared link.
 */
@implementation TMBDropbox_Sharing_RevokeSharedLink

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_Sharing_RevokeSharedLink Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/Sharing/RevokeSharedLink"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_Sharing_RevokeSharedLink_ResultSet *results = [[TMBDropbox_Sharing_RevokeSharedLink_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the RevokeSharedLink Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_Sharing_RevokeSharedLink_Inputs*)newInputSet {
		return [[TMBDropbox_Sharing_RevokeSharedLink_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetAccount Choreo.
 */
@implementation TMBDropbox_Users_GetAccount_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token for a specific Dropbox user.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}

	/*!
	 * Set the value of the AccountID input for this Choreo.
	*(required, string) A user's account identifier.
	 */
	-(void)setAccountID:(NSString*)AccountID {
		[super setInput:@"AccountID" toValue:AccountID];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetAccount Choreo.
 */
@implementation TMBDropbox_Users_GetAccount_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	
@end

/*!
 * Retrieves information about a user's account.
 */
@implementation TMBDropbox_Users_GetAccount

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_Users_GetAccount Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/Users/GetAccount"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_Users_GetAccount_ResultSet *results = [[TMBDropbox_Users_GetAccount_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetAccount Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_Users_GetAccount_Inputs*)newInputSet {
		return [[TMBDropbox_Users_GetAccount_Inputs alloc] init];
	}
@end



/*!
 * Input object with appropriate setters for specifying arguments to the GetCurrentAccount Choreo.
 */
@implementation TMBDropbox_Users_GetCurrentAccount_Inputs

	/*!
	 * Set the value of the AccessToken input for this Choreo.
	*(required, string) The access token for a specific Dropbox user.
	 */
	-(void)setAccessToken:(NSString*)AccessToken {
		[super setInput:@"AccessToken" toValue:AccessToken];
	}	

@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetCurrentAccount Choreo.
 */
@implementation TMBDropbox_Users_GetCurrentAccount_ResultSet

	-(id)initWithResponse:(NSDictionary*)document {
		return [super initWithResponse:document];
	}	
	
	
@end

/*!
 * Retrieves information about the current user's account.
 */
@implementation TMBDropbox_Users_GetCurrentAccount

	/*!
	 * Initialize the Choreo with a valid session
	 *
	 * @param session - A TMBTembooSession containing valid Temboo Application Key credentials
	 *
	 * @return - The initialized TMBDropbox_Users_GetCurrentAccount Choreo
	 */
	-(id)initWithSession:(TMBTembooSession *)session {
		TMBTembooUri *uri = [[TMBTembooUri alloc] initWithString:@"/Library/Dropbox/Users/GetCurrentAccount"];
		return [super initWithSession:session andUri:uri];
	}	
	
	/**
	 * TMBChoreographyDelegate protocol method - called when this Choreo's execution
	 * completes successfully. This class in turn calls the same function on the object
	 * (delegate) which was specified as when the choreo execution was initiated.
	 *
	 * @param response The execution response from the Temboo server
	 *
	 * @return void
	 */
	-(void)choreographyDidFinishExecuting:(NSDictionary*)response {
		TMBDropbox_Users_GetCurrentAccount_ResultSet *results = [[TMBDropbox_Users_GetCurrentAccount_ResultSet alloc] initWithResponse:response];
		[self.choreoDelegate choreographyDidFinishExecuting:results];
	}
	
	/*!
	 * Obtain an inputs object for the GetCurrentAccount Choreo.
	 *
	 * @return - New inputs object
	 */
	-(TMBDropbox_Users_GetCurrentAccount_Inputs*)newInputSet {
		return [[TMBDropbox_Users_GetCurrentAccount_Inputs alloc] init];
	}
@end
	