/*!
 * @header Temboo iOS SDK Dropbox classes
 *
 * Execute Choreographies from the Temboo Dropbox bundle.
 *
 * iOS version 10.6.8
 *
 * LICENSE: Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @category   Services
 * @author     Reid Simonton, Temboo, Inc.
 * @copyright  2013 Temboo, Inc.
 * @license    http://www.apache.org/licenses/LICENSE-2.0 Apache License 2.0
 * @version    1.7
 */

#import <Foundation/Foundation.h>
#import "TMBChoreography.h"

@class TMBTembooSession;

/*! group TMBDropbox.Account.AccountInfo Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the AccountInfo Choreo.
 */
@interface TMBDropbox_Account_AccountInfo_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret;
	-(void)setAppKey:(NSString*)AppKey;
	-(void)setAppSecret:(NSString*)AppSecret;
	-(void)setResponseFormat:(NSString*)ResponseFormat;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the AccountInfo Choreo.
 */
@interface TMBDropbox_Account_AccountInfo_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Retrieves information about the user's account.
 */
@interface TMBDropbox_Account_AccountInfo : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_Account_AccountInfo_Inputs*)newInputSet;
@end

/*! group TMBDropbox.Account_AccountInfo Choreo */


/*! group TMBDropbox.FileOperations.CopyFileOrFolder Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the CopyFileOrFolder Choreo.
 */
@interface TMBDropbox_FileOperations_CopyFileOrFolder_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret;
	-(void)setAppKey:(NSString*)AppKey;
	-(void)setAppSecret:(NSString*)AppSecret;
	-(void)setFromCopyRef:(NSString*)FromCopyRef;
	-(void)setFromPath:(NSString*)FromPath;
	-(void)setResponseFormat:(NSString*)ResponseFormat;
	-(void)setRoot:(NSString*)Root;
	-(void)setToPath:(NSString*)ToPath;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CopyFileOrFolder Choreo.
 */
@interface TMBDropbox_FileOperations_CopyFileOrFolder_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Copies a file or folder to a new location in the Dropbox tree.
 */
@interface TMBDropbox_FileOperations_CopyFileOrFolder : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_FileOperations_CopyFileOrFolder_Inputs*)newInputSet;
@end

/*! group TMBDropbox.FileOperations_CopyFileOrFolder Choreo */


/*! group TMBDropbox.FileOperations.CreateFolder Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the CreateFolder Choreo.
 */
@interface TMBDropbox_FileOperations_CreateFolder_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret;
	-(void)setAppKey:(NSString*)AppKey;
	-(void)setAppSecret:(NSString*)AppSecret;
	-(void)setNewFolderName:(NSString*)NewFolderName;
	-(void)setResponseFormat:(NSString*)ResponseFormat;
	-(void)setRoot:(NSString*)Root;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateFolder Choreo.
 */
@interface TMBDropbox_FileOperations_CreateFolder_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Creates a Dropbox folder.
 */
@interface TMBDropbox_FileOperations_CreateFolder : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_FileOperations_CreateFolder_Inputs*)newInputSet;
@end

/*! group TMBDropbox.FileOperations_CreateFolder Choreo */


/*! group TMBDropbox.FileOperations.DeleteFileOrFolder Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the DeleteFileOrFolder Choreo.
 */
@interface TMBDropbox_FileOperations_DeleteFileOrFolder_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret;
	-(void)setAppKey:(NSString*)AppKey;
	-(void)setAppSecret:(NSString*)AppSecret;
	-(void)setPath:(NSString*)Path;
	-(void)setResponseFormat:(NSString*)ResponseFormat;
	-(void)setRoot:(NSString*)Root;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the DeleteFileOrFolder Choreo.
 */
@interface TMBDropbox_FileOperations_DeleteFileOrFolder_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Deletes a Dropbox file or folder that you specify.
 */
@interface TMBDropbox_FileOperations_DeleteFileOrFolder : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_FileOperations_DeleteFileOrFolder_Inputs*)newInputSet;
@end

/*! group TMBDropbox.FileOperations_DeleteFileOrFolder Choreo */


/*! group TMBDropbox.FileOperations.MoveFileOrFolder Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the MoveFileOrFolder Choreo.
 */
@interface TMBDropbox_FileOperations_MoveFileOrFolder_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret;
	-(void)setAppKey:(NSString*)AppKey;
	-(void)setAppSecret:(NSString*)AppSecret;
	-(void)setFromPath:(NSString*)FromPath;
	-(void)setResponseFormat:(NSString*)ResponseFormat;
	-(void)setRoot:(NSString*)Root;
	-(void)setToPath:(NSString*)ToPath;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the MoveFileOrFolder Choreo.
 */
@interface TMBDropbox_FileOperations_MoveFileOrFolder_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Moves a file or folder to a new location in the Dropbox tree.
 */
@interface TMBDropbox_FileOperations_MoveFileOrFolder : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_FileOperations_MoveFileOrFolder_Inputs*)newInputSet;
@end

/*! group TMBDropbox.FileOperations_MoveFileOrFolder Choreo */


/*! group TMBDropbox.FileOperations.ZipFile Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the ZipFile Choreo.
 */
@interface TMBDropbox_FileOperations_ZipFile_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret;
	-(void)setAppKey:(NSString*)AppKey;
	-(void)setAppSecret:(NSString*)AppSecret;
	-(void)setFileName:(NSString*)FileName;
	-(void)setFolder:(NSString*)Folder;
	-(void)setRoot:(NSString*)Root;
	-(void)setZipFileLocation:(NSString*)ZipFileLocation;
	-(void)setZipFileName:(NSString*)ZipFileName;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ZipFile Choreo.
 */
@interface TMBDropbox_FileOperations_ZipFile_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getURL;	
@end

/*!
 * Creates a zipped version of the specified Dropbox file and returns a shareable link to the new compressed file.
 */
@interface TMBDropbox_FileOperations_ZipFile : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_FileOperations_ZipFile_Inputs*)newInputSet;
@end

/*! group TMBDropbox.FileOperations_ZipFile Choreo */


/*! group TMBDropbox.FileOperations.ZipFolder Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the ZipFolder Choreo.
 */
@interface TMBDropbox_FileOperations_ZipFolder_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret;
	-(void)setAppKey:(NSString*)AppKey;
	-(void)setAppSecret:(NSString*)AppSecret;
	-(void)setFolder:(NSString*)Folder;
	-(void)setRoot:(NSString*)Root;
	-(void)setZipFileLocation:(NSString*)ZipFileLocation;
	-(void)setZipFileName:(NSString*)ZipFileName;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ZipFolder Choreo.
 */
@interface TMBDropbox_FileOperations_ZipFolder_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getURL;
	-(NSString*)getResponse;	
@end

/*!
 * Creates a zip file containing all files within the specified folder and returns a shareable link to the new compressed file.
 */
@interface TMBDropbox_FileOperations_ZipFolder : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_FileOperations_ZipFolder_Inputs*)newInputSet;
@end

/*! group TMBDropbox.FileOperations_ZipFolder Choreo */


/*! group TMBDropbox.Files.Copy Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the Copy Choreo.
 */
@interface TMBDropbox_Files_Copy_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setAllowSharedFolder:(NSString*)AllowSharedFolder;
	-(void)setAutoRename:(NSString*)AutoRename;
	-(void)setFromPath:(NSString*)FromPath;
	-(void)setToPath:(NSString*)ToPath;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Copy Choreo.
 */
@interface TMBDropbox_Files_Copy_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Copies a file or folder to a different location in the user's Dropbox.
 */
@interface TMBDropbox_Files_Copy : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_Files_Copy_Inputs*)newInputSet;
@end

/*! group TMBDropbox.Files_Copy Choreo */


/*! group TMBDropbox.Files.CopyReferenceGet Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the CopyReferenceGet Choreo.
 */
@interface TMBDropbox_Files_CopyReferenceGet_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setPath:(NSString*)Path;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CopyReferenceGet Choreo.
 */
@interface TMBDropbox_Files_CopyReferenceGet_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Gets a copy reference to a file or folder.
 */
@interface TMBDropbox_Files_CopyReferenceGet : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_Files_CopyReferenceGet_Inputs*)newInputSet;
@end

/*! group TMBDropbox.Files_CopyReferenceGet Choreo */


/*! group TMBDropbox.Files.CopyReferenceSave Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the CopyReferenceSave Choreo.
 */
@interface TMBDropbox_Files_CopyReferenceSave_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setCopyReference:(NSString*)CopyReference;
	-(void)setPath:(NSString*)Path;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CopyReferenceSave Choreo.
 */
@interface TMBDropbox_Files_CopyReferenceSave_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Saves a copy reference returned by CopyReferenceGet to the user's Dropbox.
 */
@interface TMBDropbox_Files_CopyReferenceSave : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_Files_CopyReferenceSave_Inputs*)newInputSet;
@end

/*! group TMBDropbox.Files_CopyReferenceSave Choreo */


/*! group TMBDropbox.Files.CreateFolder Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the CreateFolder Choreo.
 */
@interface TMBDropbox_Files_CreateFolder_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setAutoRename:(NSString*)AutoRename;
	-(void)setPath:(NSString*)Path;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateFolder Choreo.
 */
@interface TMBDropbox_Files_CreateFolder_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Creates a folder at a given path.
 */
@interface TMBDropbox_Files_CreateFolder : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_Files_CreateFolder_Inputs*)newInputSet;
@end

/*! group TMBDropbox.Files_CreateFolder Choreo */


/*! group TMBDropbox.Files.Delete Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the Delete Choreo.
 */
@interface TMBDropbox_Files_Delete_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setPath:(NSString*)Path;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Delete Choreo.
 */
@interface TMBDropbox_Files_Delete_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Deletes the file or folder at a given path.
 */
@interface TMBDropbox_Files_Delete : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_Files_Delete_Inputs*)newInputSet;
@end

/*! group TMBDropbox.Files_Delete Choreo */


/*! group TMBDropbox.Files.Download Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the Download Choreo.
 */
@interface TMBDropbox_Files_Download_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setEncode:(NSString*)Encode;
	-(void)setPath:(NSString*)Path;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Download Choreo.
 */
@interface TMBDropbox_Files_Download_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Downloads a file from a user's Dropbox.
 */
@interface TMBDropbox_Files_Download : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_Files_Download_Inputs*)newInputSet;
@end

/*! group TMBDropbox.Files_Download Choreo */


/*! group TMBDropbox.Files.GetLatestCursor Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the GetLatestCursor Choreo.
 */
@interface TMBDropbox_Files_GetLatestCursor_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setIncludeDeleted:(NSString*)IncludeDeleted;
	-(void)setIncludeHasExplicitSharedMembers:(NSString*)IncludeHasExplicitSharedMembers;
	-(void)setIncludeMediaInfo:(NSString*)IncludeMediaInfo;
	-(void)setPath:(NSString*)Path;
	-(void)setRecursive:(NSString*)Recursive;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetLatestCursor Choreo.
 */
@interface TMBDropbox_Files_GetLatestCursor_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getCursor;	
@end

/*!
 * Gets a cursor for the folder's state.
 */
@interface TMBDropbox_Files_GetLatestCursor : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_Files_GetLatestCursor_Inputs*)newInputSet;
@end

/*! group TMBDropbox.Files_GetLatestCursor Choreo */


/*! group TMBDropbox.Files.GetMetadata Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the GetMetadata Choreo.
 */
@interface TMBDropbox_Files_GetMetadata_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setIncludeDeleted:(NSString*)IncludeDeleted;
	-(void)setIncludeHasExplicitSharedMembers:(NSString*)IncludeHasExplicitSharedMembers;
	-(void)setIncludeMediaInfo:(NSString*)IncludeMediaInfo;
	-(void)setPath:(NSString*)Path;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetMetadata Choreo.
 */
@interface TMBDropbox_Files_GetMetadata_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Returns the metadata for a file or folder.
 */
@interface TMBDropbox_Files_GetMetadata : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_Files_GetMetadata_Inputs*)newInputSet;
@end

/*! group TMBDropbox.Files_GetMetadata Choreo */


/*! group TMBDropbox.Files.GetThumbnail Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the GetThumbnail Choreo.
 */
@interface TMBDropbox_Files_GetThumbnail_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setFormat:(NSString*)Format;
	-(void)setPath:(NSString*)Path;
	-(void)setSize:(NSString*)Size;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetThumbnail Choreo.
 */
@interface TMBDropbox_Files_GetThumbnail_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Gets a thumbnail for an image.
 */
@interface TMBDropbox_Files_GetThumbnail : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_Files_GetThumbnail_Inputs*)newInputSet;
@end

/*! group TMBDropbox.Files_GetThumbnail Choreo */


/*! group TMBDropbox.Files.ListFolder Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the ListFolder Choreo.
 */
@interface TMBDropbox_Files_ListFolder_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setIncludeDeleted:(NSString*)IncludeDeleted;
	-(void)setIncludeHasExplicitSharedMembers:(NSString*)IncludeHasExplicitSharedMembers;
	-(void)setIncludeMediaInfo:(NSString*)IncludeMediaInfo;
	-(void)setPath:(NSString*)Path;
	-(void)setRecursive:(NSString*)Recursive;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListFolder Choreo.
 */
@interface TMBDropbox_Files_ListFolder_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getCursor;
	-(NSString*)getResponse;	
@end

/*!
 * Starts returning the contents of a folder.
 */
@interface TMBDropbox_Files_ListFolder : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_Files_ListFolder_Inputs*)newInputSet;
@end

/*! group TMBDropbox.Files_ListFolder Choreo */


/*! group TMBDropbox.Files.ListFolderContinue Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the ListFolderContinue Choreo.
 */
@interface TMBDropbox_Files_ListFolderContinue_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setCursor:(NSString*)Cursor;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListFolderContinue Choreo.
 */
@interface TMBDropbox_Files_ListFolderContinue_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getLatestCursor;
	-(NSString*)getResponse;	
@end

/*!
 * Once a cursor has been retrieved from ListFolder, use this to paginate through all files and retrieve updates to the folder.
 */
@interface TMBDropbox_Files_ListFolderContinue : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_Files_ListFolderContinue_Inputs*)newInputSet;
@end

/*! group TMBDropbox.Files_ListFolderContinue Choreo */


/*! group TMBDropbox.Files.ListRevisions Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the ListRevisions Choreo.
 */
@interface TMBDropbox_Files_ListRevisions_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setLimit:(NSString*)Limit;
	-(void)setPath:(NSString*)Path;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListRevisions Choreo.
 */
@interface TMBDropbox_Files_ListRevisions_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Returns revisions of a file.
 */
@interface TMBDropbox_Files_ListRevisions : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_Files_ListRevisions_Inputs*)newInputSet;
@end

/*! group TMBDropbox.Files_ListRevisions Choreo */


/*! group TMBDropbox.Files.Move Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the Move Choreo.
 */
@interface TMBDropbox_Files_Move_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setAllowSharedFolder:(NSString*)AllowSharedFolder;
	-(void)setAutoRename:(NSString*)AutoRename;
	-(void)setFromPath:(NSString*)FromPath;
	-(void)setToPath:(NSString*)ToPath;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Move Choreo.
 */
@interface TMBDropbox_Files_Move_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Moves a file or folder to a different location in the user's Dropbox.
 */
@interface TMBDropbox_Files_Move : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_Files_Move_Inputs*)newInputSet;
@end

/*! group TMBDropbox.Files_Move Choreo */


/*! group TMBDropbox.Files.Restore Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the Restore Choreo.
 */
@interface TMBDropbox_Files_Restore_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setPath:(NSString*)Path;
	-(void)setRevision:(NSString*)Revision;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Restore Choreo.
 */
@interface TMBDropbox_Files_Restore_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Restores a file to a specific revision.
 */
@interface TMBDropbox_Files_Restore : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_Files_Restore_Inputs*)newInputSet;
@end

/*! group TMBDropbox.Files_Restore Choreo */


/*! group TMBDropbox.Files.Search Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the Search Choreo.
 */
@interface TMBDropbox_Files_Search_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setMaxResults:(NSString*)MaxResults;
	-(void)setMode:(NSString*)Mode;
	-(void)setPath:(NSString*)Path;
	-(void)setQuery:(NSString*)Query;
	-(void)setStart:(NSString*)Start;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Search Choreo.
 */
@interface TMBDropbox_Files_Search_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Searches for files and folders.
 */
@interface TMBDropbox_Files_Search : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_Files_Search_Inputs*)newInputSet;
@end

/*! group TMBDropbox.Files_Search Choreo */


/*! group TMBDropbox.Files.Upload Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the Upload Choreo.
 */
@interface TMBDropbox_Files_Upload_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setAutoRename:(NSString*)AutoRename;
	-(void)setContentType:(NSString*)ContentType;
	-(void)setFileContent:(NSString*)FileContent;
	-(void)setFileURL:(NSString*)FileURL;
	-(void)setMode:(NSString*)Mode;
	-(void)setMute:(NSString*)Mute;
	-(void)setPath:(NSString*)Path;
	-(void)setRevision:(NSString*)Revision;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Upload Choreo.
 */
@interface TMBDropbox_Files_Upload_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	
@end

/*!
 * Creates a new file with the contents provided in the request.
 */
@interface TMBDropbox_Files_Upload : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_Files_Upload_Inputs*)newInputSet;
@end

/*! group TMBDropbox.Files_Upload Choreo */


/*! group TMBDropbox.Files.UploadSessionAppend Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the UploadSessionAppend Choreo.
 */
@interface TMBDropbox_Files_UploadSessionAppend_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setClose:(NSString*)Close;
	-(void)setContentType:(NSString*)ContentType;
	-(void)setFileContent:(NSString*)FileContent;
	-(void)setOffset:(NSString*)Offset;
	-(void)setSessionID:(NSString*)SessionID;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UploadSessionAppend Choreo.
 */
@interface TMBDropbox_Files_UploadSessionAppend_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	
@end

/*!
 * Appends more data to an upload session.
 */
@interface TMBDropbox_Files_UploadSessionAppend : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_Files_UploadSessionAppend_Inputs*)newInputSet;
@end

/*! group TMBDropbox.Files_UploadSessionAppend Choreo */


/*! group TMBDropbox.Files.UploadSessionFinish Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the UploadSessionFinish Choreo.
 */
@interface TMBDropbox_Files_UploadSessionFinish_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setAutoRename:(NSString*)AutoRename;
	-(void)setContentType:(NSString*)ContentType;
	-(void)setFileContent:(NSString*)FileContent;
	-(void)setMode:(NSString*)Mode;
	-(void)setMute:(NSString*)Mute;
	-(void)setOffset:(NSString*)Offset;
	-(void)setPath:(NSString*)Path;
	-(void)setRevision:(NSString*)Revision;
	-(void)setSessionID:(NSString*)SessionID;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UploadSessionFinish Choreo.
 */
@interface TMBDropbox_Files_UploadSessionFinish_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	
@end

/*!
 * Finishes an upload session and save the uploaded data to the given file path.
 */
@interface TMBDropbox_Files_UploadSessionFinish : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_Files_UploadSessionFinish_Inputs*)newInputSet;
@end

/*! group TMBDropbox.Files_UploadSessionFinish Choreo */


/*! group TMBDropbox.Files.UploadSessionStart Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the UploadSessionStart Choreo.
 */
@interface TMBDropbox_Files_UploadSessionStart_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setClose:(NSString*)Close;
	-(void)setContentType:(NSString*)ContentType;
	-(void)setFileContent:(NSString*)FileContent;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UploadSessionStart Choreo.
 */
@interface TMBDropbox_Files_UploadSessionStart_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getSessionID;	
@end

/*!
 * Allows you to upload a single file in one or more requests. This call starts a new upload session with the first chunk of data.
 */
@interface TMBDropbox_Files_UploadSessionStart : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_Files_UploadSessionStart_Inputs*)newInputSet;
@end

/*! group TMBDropbox.Files_UploadSessionStart Choreo */


/*! group TMBDropbox.FilesAndMetadata.ChunkedUpload Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the ChunkedUpload Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_ChunkedUpload_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret;
	-(void)setAppKey:(NSString*)AppKey;
	-(void)setAppSecret:(NSString*)AppSecret;
	-(void)setChunk:(NSString*)Chunk;
	-(void)setOffset:(NSString*)Offset;
	-(void)setUploadID:(NSString*)UploadID;
	-(void)setVaultFile:(NSString*)VaultFile;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ChunkedUpload Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_ChunkedUpload_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getExpires;
	-(NSString*)getNextOffset;
	-(NSString*)getUploadSessionID;	
@end

/*!
 * Uploads larger files to Dropbox in multiple chunks, and offers a way to resume if an upload gets interrupted.
 */
@interface TMBDropbox_FilesAndMetadata_ChunkedUpload : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_FilesAndMetadata_ChunkedUpload_Inputs*)newInputSet;
@end

/*! group TMBDropbox.FilesAndMetadata_ChunkedUpload Choreo */


/*! group TMBDropbox.FilesAndMetadata.CommitChunkedUpload Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the CommitChunkedUpload Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_CommitChunkedUpload_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret;
	-(void)setAppKey:(NSString*)AppKey;
	-(void)setAppSecret:(NSString*)AppSecret;
	-(void)setLocale:(NSString*)Locale;
	-(void)setOverwrite:(NSString*)Overwrite;
	-(void)setParentRevision:(NSString*)ParentRevision;
	-(void)setPath:(NSString*)Path;
	-(void)setResponseFormat:(NSString*)ResponseFormat;
	-(void)setRoot:(NSString*)Root;
	-(void)setUploadID:(NSString*)UploadID;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CommitChunkedUpload Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_CommitChunkedUpload_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Completes an upload initiated by the ChunkedUpload Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_CommitChunkedUpload : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_FilesAndMetadata_CommitChunkedUpload_Inputs*)newInputSet;
@end

/*! group TMBDropbox.FilesAndMetadata_CommitChunkedUpload Choreo */


/*! group TMBDropbox.FilesAndMetadata.CreateCopyRef Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the CreateCopyRef Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_CreateCopyRef_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret;
	-(void)setAppKey:(NSString*)AppKey;
	-(void)setAppSecret:(NSString*)AppSecret;
	-(void)setPath:(NSString*)Path;
	-(void)setResponseFormat:(NSString*)ResponseFormat;
	-(void)setRoot:(NSString*)Root;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateCopyRef Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_CreateCopyRef_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Allows you to create a copy reference (copy_ref) that you can later reference to copy files or folders to another user's Dropbox using the CopyFileOrFolder choreo. 
 */
@interface TMBDropbox_FilesAndMetadata_CreateCopyRef : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_FilesAndMetadata_CreateCopyRef_Inputs*)newInputSet;
@end

/*! group TMBDropbox.FilesAndMetadata_CreateCopyRef Choreo */


/*! group TMBDropbox.FilesAndMetadata.Delta Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the Delta Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_Delta_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret;
	-(void)setAppKey:(NSString*)AppKey;
	-(void)setAppSecret:(NSString*)AppSecret;
	-(void)setCursor:(NSString*)Cursor;
	-(void)setIncludeMediaInfo:(NSString*)IncludeMediaInfo;
	-(void)setLocale:(NSString*)Locale;
	-(void)setPathPrefix:(NSString*)PathPrefix;
	-(void)setResponseFormat:(NSString*)ResponseFormat;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Delta Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_Delta_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Allows you keep up with changes to files and folders in a user's Dropbox.
 */
@interface TMBDropbox_FilesAndMetadata_Delta : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_FilesAndMetadata_Delta_Inputs*)newInputSet;
@end

/*! group TMBDropbox.FilesAndMetadata_Delta Choreo */


/*! group TMBDropbox.FilesAndMetadata.GetDownloadLink Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the GetDownloadLink Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_GetDownloadLink_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret;
	-(void)setAppKey:(NSString*)AppKey;
	-(void)setAppSecret:(NSString*)AppSecret;
	-(void)setPath:(NSString*)Path;
	-(void)setResponseFormat:(NSString*)ResponseFormat;
	-(void)setRoot:(NSString*)Root;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetDownloadLink Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_GetDownloadLink_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getURL;
	-(NSString*)getResponse;	
@end

/*!
 * Retrieves a direct download link allowing an application to stream the contents of the media.
 */
@interface TMBDropbox_FilesAndMetadata_GetDownloadLink : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_FilesAndMetadata_GetDownloadLink_Inputs*)newInputSet;
@end

/*! group TMBDropbox.FilesAndMetadata_GetDownloadLink Choreo */


/*! group TMBDropbox.FilesAndMetadata.GetFile Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the GetFile Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_GetFile_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret;
	-(void)setAppKey:(NSString*)AppKey;
	-(void)setAppSecret:(NSString*)AppSecret;
	-(void)setEncodeFileContent:(NSString*)EncodeFileContent;
	-(void)setIncludeMetadata:(NSString*)IncludeMetadata;
	-(void)setPath:(NSString*)Path;
	-(void)setRoot:(NSString*)Root;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetFile Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_GetFile_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getFileMetadata;
	-(NSString*)getResponse;	
@end

/*!
 * Gets the content and metadata for a specified file.
 */
@interface TMBDropbox_FilesAndMetadata_GetFile : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_FilesAndMetadata_GetFile_Inputs*)newInputSet;
@end

/*! group TMBDropbox.FilesAndMetadata_GetFile Choreo */


/*! group TMBDropbox.FilesAndMetadata.GetFileRevisions Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the GetFileRevisions Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_GetFileRevisions_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret;
	-(void)setAppKey:(NSString*)AppKey;
	-(void)setAppSecret:(NSString*)AppSecret;
	-(void)setPath:(NSString*)Path;
	-(void)setResponseFormat:(NSString*)ResponseFormat;
	-(void)setRevisionLimit:(NSString*)RevisionLimit;
	-(void)setRoot:(NSString*)Root;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetFileRevisions Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_GetFileRevisions_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Retrieves metadata for the previous revisions of a file
 */
@interface TMBDropbox_FilesAndMetadata_GetFileRevisions : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_FilesAndMetadata_GetFileRevisions_Inputs*)newInputSet;
@end

/*! group TMBDropbox.FilesAndMetadata_GetFileRevisions Choreo */


/*! group TMBDropbox.FilesAndMetadata.GetShareableLink Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the GetShareableLink Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_GetShareableLink_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret;
	-(void)setAppKey:(NSString*)AppKey;
	-(void)setAppSecret:(NSString*)AppSecret;
	-(void)setPath:(NSString*)Path;
	-(void)setResponseFormat:(NSString*)ResponseFormat;
	-(void)setRoot:(NSString*)Root;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetShareableLink Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_GetShareableLink_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Retrieves a shareable link to files or folders.
 */
@interface TMBDropbox_FilesAndMetadata_GetShareableLink : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_FilesAndMetadata_GetShareableLink_Inputs*)newInputSet;
@end

/*! group TMBDropbox.FilesAndMetadata_GetShareableLink Choreo */


/*! group TMBDropbox.FilesAndMetadata.GetThumbnail Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the GetThumbnail Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_GetThumbnail_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret;
	-(void)setAppKey:(NSString*)AppKey;
	-(void)setAppSecret:(NSString*)AppSecret;
	-(void)setImageFormat:(NSString*)ImageFormat;
	-(void)setPath:(NSString*)Path;
	-(void)setRoot:(NSString*)Root;
	-(void)setSize:(NSString*)Size;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetThumbnail Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_GetThumbnail_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Retrieves a thumbnail for a specified image.
 */
@interface TMBDropbox_FilesAndMetadata_GetThumbnail : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_FilesAndMetadata_GetThumbnail_Inputs*)newInputSet;
@end

/*! group TMBDropbox.FilesAndMetadata_GetThumbnail Choreo */


/*! group TMBDropbox.FilesAndMetadata.LatestCursor Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the LatestCursor Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_LatestCursor_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret;
	-(void)setAppKey:(NSString*)AppKey;
	-(void)setAppSecret:(NSString*)AppSecret;
	-(void)setIncludeMediaInfo:(NSString*)IncludeMediaInfo;
	-(void)setPathPrefix:(NSString*)PathPrefix;
	-(void)setResponseFormat:(NSString*)ResponseFormat;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the LatestCursor Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_LatestCursor_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Allows you to quickly get a cursor for the server's state, for use with the Delta Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_LatestCursor : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_FilesAndMetadata_LatestCursor_Inputs*)newInputSet;
@end

/*! group TMBDropbox.FilesAndMetadata_LatestCursor Choreo */


/*! group TMBDropbox.FilesAndMetadata.ListFolderContents Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the ListFolderContents Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_ListFolderContents_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret;
	-(void)setAppKey:(NSString*)AppKey;
	-(void)setAppSecret:(NSString*)AppSecret;
	-(void)setFileLimit:(NSString*)FileLimit;
	-(void)setFolder:(NSString*)Folder;
	-(void)setHash:(NSString*)Hash;
	-(void)setIncludeDeleted:(NSString*)IncludeDeleted;
	-(void)setList:(NSString*)List;
	-(void)setLocale:(NSString*)Locale;
	-(void)setResponseFormat:(NSString*)ResponseFormat;
	-(void)setRevision:(NSString*)Revision;
	-(void)setRoot:(NSString*)Root;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the ListFolderContents Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_ListFolderContents_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponseStatusCode;
	-(NSString*)getResponse;	
@end

/*!
 * Retrieves metadata (including folder contents) for a folder or file in Dropbox.
 */
@interface TMBDropbox_FilesAndMetadata_ListFolderContents : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_FilesAndMetadata_ListFolderContents_Inputs*)newInputSet;
@end

/*! group TMBDropbox.FilesAndMetadata_ListFolderContents Choreo */


/*! group TMBDropbox.FilesAndMetadata.LongPollDelta Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the LongPollDelta Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_LongPollDelta_Inputs : TMBChoreographyInputSet
	-(void)setCursor:(NSString*)Cursor;
	-(void)setResponseFormat:(NSString*)ResponseFormat;
	-(void)setTimeout:(NSString*)Timeout;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the LongPollDelta Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_LongPollDelta_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Used in conjunction with the Delta Choreo, this allows you to perform a long-poll request to wait for changes on an account.
 */
@interface TMBDropbox_FilesAndMetadata_LongPollDelta : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_FilesAndMetadata_LongPollDelta_Inputs*)newInputSet;
@end

/*! group TMBDropbox.FilesAndMetadata_LongPollDelta Choreo */


/*! group TMBDropbox.FilesAndMetadata.RestoreFileToRevision Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the RestoreFileToRevision Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_RestoreFileToRevision_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret;
	-(void)setAppKey:(NSString*)AppKey;
	-(void)setAppSecret:(NSString*)AppSecret;
	-(void)setPath:(NSString*)Path;
	-(void)setResponseFormat:(NSString*)ResponseFormat;
	-(void)setRevision:(NSString*)Revision;
	-(void)setRoot:(NSString*)Root;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the RestoreFileToRevision Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_RestoreFileToRevision_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Restores a specified file to a previous revision.
 */
@interface TMBDropbox_FilesAndMetadata_RestoreFileToRevision : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_FilesAndMetadata_RestoreFileToRevision_Inputs*)newInputSet;
@end

/*! group TMBDropbox.FilesAndMetadata_RestoreFileToRevision Choreo */


/*! group TMBDropbox.FilesAndMetadata.SearchFilesAndFolders Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the SearchFilesAndFolders Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_SearchFilesAndFolders_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret;
	-(void)setAppKey:(NSString*)AppKey;
	-(void)setAppSecret:(NSString*)AppSecret;
	-(void)setFileLimit:(NSString*)FileLimit;
	-(void)setPath:(NSString*)Path;
	-(void)setQuery:(NSString*)Query;
	-(void)setResponseFormat:(NSString*)ResponseFormat;
	-(void)setRoot:(NSString*)Root;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the SearchFilesAndFolders Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_SearchFilesAndFolders_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Allows you to search Dropbox for files or folders by a keyword search.
 */
@interface TMBDropbox_FilesAndMetadata_SearchFilesAndFolders : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_FilesAndMetadata_SearchFilesAndFolders_Inputs*)newInputSet;
@end

/*! group TMBDropbox.FilesAndMetadata_SearchFilesAndFolders Choreo */


/*! group TMBDropbox.FilesAndMetadata.UploadFile Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the UploadFile Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_UploadFile_Inputs : TMBChoreographyInputSet
	-(void)setFileContents:(NSString*)FileContents;
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret;
	-(void)setAppKey:(NSString*)AppKey;
	-(void)setAppSecret:(NSString*)AppSecret;
	-(void)setFileContentsFromURL:(NSString*)FileContentsFromURL;
	-(void)setFileName:(NSString*)FileName;
	-(void)setFolder:(NSString*)Folder;
	-(void)setResponseFormat:(NSString*)ResponseFormat;
	-(void)setRoot:(NSString*)Root;
	-(void)setVaultFile:(NSString*)VaultFile;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the UploadFile Choreo.
 */
@interface TMBDropbox_FilesAndMetadata_UploadFile_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Uploads a file to your Dropbox account.
 */
@interface TMBDropbox_FilesAndMetadata_UploadFile : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_FilesAndMetadata_UploadFile_Inputs*)newInputSet;
@end

/*! group TMBDropbox.FilesAndMetadata_UploadFile Choreo */


/*! group TMBDropbox.OAuth.FinalizeOAuth Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the FinalizeOAuth Choreo.
 */
@interface TMBDropbox_OAuth_FinalizeOAuth_Inputs : TMBChoreographyInputSet
	-(void)setAccountName:(NSString*)AccountName;
	-(void)setAppKeyName:(NSString*)AppKeyName;
	-(void)setAppKeyValue:(NSString*)AppKeyValue;
	-(void)setCallbackID:(NSString*)CallbackID;
	-(void)setDropboxAppKey:(NSString*)DropboxAppKey;
	-(void)setDropboxAppSecret:(NSString*)DropboxAppSecret;
	-(void)setOAuthTokenSecret:(NSString*)OAuthTokenSecret;
	-(void)setSuppressErrors:(NSString*)SuppressErrors;
	-(void)setTimeout:(NSString*)Timeout;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the FinalizeOAuth Choreo.
 */
@interface TMBDropbox_OAuth_FinalizeOAuth_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getAccessToken;
	-(NSString*)getAccessTokenSecret;
	-(NSString*)getErrorMessage;
	-(NSString*)getUserID;	
@end

/*!
 * Completes the OAuth process by retrieving a Dropbox access token and access token secret for a user, after they have visited the authorization URL returned by the InitializeOAuth choreo and clicked "allow."
 */
@interface TMBDropbox_OAuth_FinalizeOAuth : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_OAuth_FinalizeOAuth_Inputs*)newInputSet;
@end

/*! group TMBDropbox.OAuth_FinalizeOAuth Choreo */


/*! group TMBDropbox.OAuth.FromOAuth1 Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the FromOAuth1 Choreo.
 */
@interface TMBDropbox_OAuth_FromOAuth1_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setAccessTokenSecret:(NSString*)AccessTokenSecret;
	-(void)setAppKey:(NSString*)AppKey;
	-(void)setAppSecret:(NSString*)AppSecret;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the FromOAuth1 Choreo.
 */
@interface TMBDropbox_OAuth_FromOAuth1_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getOAuth2AccessToken;	
@end

/*!
 * Creates an OAuth 2.0 access token from the supplied OAuth 1.0 access token.
 */
@interface TMBDropbox_OAuth_FromOAuth1 : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_OAuth_FromOAuth1_Inputs*)newInputSet;
@end

/*! group TMBDropbox.OAuth_FromOAuth1 Choreo */


/*! group TMBDropbox.OAuth.InitializeOAuth Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the InitializeOAuth Choreo.
 */
@interface TMBDropbox_OAuth_InitializeOAuth_Inputs : TMBChoreographyInputSet
	-(void)setAccountName:(NSString*)AccountName;
	-(void)setAppKeyName:(NSString*)AppKeyName;
	-(void)setAppKeyValue:(NSString*)AppKeyValue;
	-(void)setDropboxAppKey:(NSString*)DropboxAppKey;
	-(void)setDropboxAppSecret:(NSString*)DropboxAppSecret;
	-(void)setForwardingURL:(NSString*)ForwardingURL;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the InitializeOAuth Choreo.
 */
@interface TMBDropbox_OAuth_InitializeOAuth_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getAuthorizationURL;
	-(NSString*)getCallbackID;
	-(NSString*)getOAuthTokenSecret;	
@end

/*!
 * Generates an authorization URL that an application can use to complete the first step in the OAuth process.
 */
@interface TMBDropbox_OAuth_InitializeOAuth : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_OAuth_InitializeOAuth_Inputs*)newInputSet;
@end

/*! group TMBDropbox.OAuth_InitializeOAuth Choreo */


/*! group TMBDropbox.OAuth.Revoke Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the Revoke Choreo.
 */
@interface TMBDropbox_OAuth_Revoke_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the Revoke Choreo.
 */
@interface TMBDropbox_OAuth_Revoke_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Disables the access token used to authenticate the call.
 */
@interface TMBDropbox_OAuth_Revoke : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_OAuth_Revoke_Inputs*)newInputSet;
@end

/*! group TMBDropbox.OAuth_Revoke Choreo */


/*! group TMBDropbox.Sharing.CreateSharedLink Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the CreateSharedLink Choreo.
 */
@interface TMBDropbox_Sharing_CreateSharedLink_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setPath:(NSString*)Path;
	-(void)setShortURL:(NSString*)ShortURL;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the CreateSharedLink Choreo.
 */
@interface TMBDropbox_Sharing_CreateSharedLink_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Creates a shared link.
 */
@interface TMBDropbox_Sharing_CreateSharedLink : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_Sharing_CreateSharedLink_Inputs*)newInputSet;
@end

/*! group TMBDropbox.Sharing_CreateSharedLink Choreo */


/*! group TMBDropbox.Sharing.RevokeSharedLink Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the RevokeSharedLink Choreo.
 */
@interface TMBDropbox_Sharing_RevokeSharedLink_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setURL:(NSString*)URL;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the RevokeSharedLink Choreo.
 */
@interface TMBDropbox_Sharing_RevokeSharedLink_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	-(NSString*)getResponse;	
@end

/*!
 * Revokes a shared link.
 */
@interface TMBDropbox_Sharing_RevokeSharedLink : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_Sharing_RevokeSharedLink_Inputs*)newInputSet;
@end

/*! group TMBDropbox.Sharing_RevokeSharedLink Choreo */


/*! group TMBDropbox.Users.GetAccount Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the GetAccount Choreo.
 */
@interface TMBDropbox_Users_GetAccount_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
	-(void)setAccountID:(NSString*)AccountID;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetAccount Choreo.
 */
@interface TMBDropbox_Users_GetAccount_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	
@end

/*!
 * Retrieves information about a user's account.
 */
@interface TMBDropbox_Users_GetAccount : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_Users_GetAccount_Inputs*)newInputSet;
@end

/*! group TMBDropbox.Users_GetAccount Choreo */


/*! group TMBDropbox.Users.GetCurrentAccount Choreo */

/*!
 * Input object with appropriate setters for specifying arguments to the GetCurrentAccount Choreo.
 */
@interface TMBDropbox_Users_GetCurrentAccount_Inputs : TMBChoreographyInputSet
	-(void)setAccessToken:(NSString*)AccessToken;
@end

/*!
 * Results object with appropriate getters for retrieving outputs from the GetCurrentAccount Choreo.
 */
@interface TMBDropbox_Users_GetCurrentAccount_ResultSet : TMBChoreographyResultSet
	-(id)initWithResponse:(NSDictionary*)document;
	
@end

/*!
 * Retrieves information about the current user's account.
 */
@interface TMBDropbox_Users_GetCurrentAccount : TMBChoreography <TMBChoreographyDelegate>
	-(id)initWithSession:(TMBTembooSession*)session;
	-(TMBDropbox_Users_GetCurrentAccount_Inputs*)newInputSet;
@end

/*! group TMBDropbox.Users_GetCurrentAccount Choreo */
