package com.temboo.Library.Dropbox.FilesAndMetadata;

/*
Copyright 2014 Temboo, Inc.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

import org.json.JSONArray;
import org.json.JSONObject;
import org.json.JSONException;
import java.math.BigDecimal;
import com.temboo.core.Choreography;
import com.temboo.core.Choreography.ResultSet;
import com.temboo.core.Choreography.InputSet;
import com.temboo.core.TembooException;
import com.temboo.core.TembooPath;
import com.temboo.core.TembooSession;

/** 
ChunkedUpload

Uploads larger files to Dropbox in multiple chunks, and offers a way to resume if an upload gets interrupted.
*/
public class ChunkedUpload extends Choreography {

	/**
	Create a new instance of the ChunkedUpload Choreo. A TembooSession object, containing a valid
	set of Temboo credentials, must be supplied.
	*/
	public ChunkedUpload(TembooSession session) {
		super(session, TembooPath.pathFromStringNoException("/Library/Dropbox/FilesAndMetadata/ChunkedUpload"));
	}


	/** 
	Obtain an InputSet object, used to define inputs for an execution of this Choreo.
	
	@return ChunkedUploadInputSet
	*/
	public ChunkedUploadInputSet newInputSet() {
		return new ChunkedUploadInputSet();
	}
	
	
	/**
	 * Execute the Choreo using the specified InputSet as parameters, wait for the Choreo to complete 
	 * and return a ResultSet containing the execution results.
	 * @param choreoInputs
	 * @return
	 * @throws TembooException
	 */
	 @Override
	public ChunkedUploadResultSet execute(InputSet choreoInputs) throws TembooException {
		JSONObject result = super.executeWithResults(choreoInputs);
		return new ChunkedUploadResultSet(result);
	}
	
	
	/**
		An InputSet with methods appropriate for specifying the inputs to the ChunkedUpload
        Choreo. The InputSet object is used to specify input parameters when executing this choreo.
	*/
	public static class ChunkedUploadInputSet extends InputSet {
		/** 
		Set the value of the AccessToken input for this Choreo. 

		@param String - (required, string) The Access Token retrieved during the OAuth process.
		*/
		public void set_AccessToken(String value) {
			this.setInput("AccessToken", value);
		}


		/** 
		Set the value of the AccessTokenSecret input for this Choreo. 

		@param String - (required, string) The Access Token Secret retrieved during the OAuth process.
		*/
		public void set_AccessTokenSecret(String value) {
			this.setInput("AccessTokenSecret", value);
		}


		/** 
		Set the value of the AppKey input for this Choreo. 

		@param String - (required, string) The App Key provided by Dropbox (AKA the OAuth Consumer Key).
		*/
		public void set_AppKey(String value) {
			this.setInput("AppKey", value);
		}


		/** 
		Set the value of the AppSecret input for this Choreo. 

		@param String - (required, string) The App Secret provided by Dropbox (AKA the OAuth Consumer Secret).
		*/
		public void set_AppSecret(String value) {
			this.setInput("AppSecret", value);
		}


		/** 
		Set the value of the Chunk input for this Choreo. 

		@param String - (conditional, string) A Base64 encoded chunk of data from the file being uploaded. If resuming and upload, the chunk should begin at the number of bytes into the file that equals the NextOffset.
		*/
		public void set_Chunk(String value) {
			this.setInput("Chunk", value);
		}


		/** 
		Set the value of the Offset input for this Choreo. 

		@param String - (conditional, string) The byte offset of this chunk, relative to the beginning of the full file. This is not required when uploading the first chunk of a file.
		*/
		public void set_Offset(String value) {
			this.setInput("Offset", value);
		}


		/** 
		Set the value of the UploadID input for this Choreo. 

		@param String - (conditional, string) The ID of the upload session returned after uploading the initial file chunk. This is not required when uploading the first chunk of a file. This value is returned in the UploadSessionID output.
		*/
		public void set_UploadID(String value) {
			this.setInput("UploadID", value);
		}


		/** 
		Set the value of the VaultFile input for this Choreo. 

		@param TembooPath - (optional, vault file) The path to a vault file you want to upload. Required unless using the FileContents input.
		*/

	}
	
	
	/**
		A ResultSet with methods tailored to the values returned by the ChunkedUpload Choreo.
        The ResultSet object is used to retrieve the results of a Choreo execution.
	*/
	public static class ChunkedUploadResultSet extends ResultSet {
		public ChunkedUploadResultSet(JSONObject doc) throws TembooException {
			super(doc);
		}

		/** 
		Retrieve the value for the "Expires" output from this Choreo execution

		@return String - (string) The expiration time of the upload.
		*/
		public String get_Expires() {
			return this.getResultString("Expires");
		}
		/** 
		Retrieve the value for the "NextOffset" output from this Choreo execution

		@return String - (string) The current byte offset that the server will expect. This value can be passed to the Offset input on subsequent requests when uploading chunks repeatedly.
		*/
		public String get_NextOffset() {
			return this.getResultString("NextOffset");
		}
		/** 
		Retrieve the value for the "UploadSessionID" output from this Choreo execution

		@return String - (string) The upload ID returned after uploading an initial file chunk. This can be passed to the UploadID input for uploading subsequent chunks, and finally to the CommitChunkedUpload Choreo.
		*/
		public String get_UploadSessionID() {
			return this.getResultString("UploadSessionID");
		}
	}
	
}
