package com.temboo.core;

/*
Copyright 2014 Temboo, Inc.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 */

import java.math.BigDecimal;
import java.net.URISyntaxException;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

public class Choreography extends TembooResource {
	public Choreography(TembooSession session, TembooPath uri) {
		super(session, uri);
	}

	
	/**
	 * Execute the choreography using the specified InputSet as parameters, wait for the choreo to complete 
	 * and return a ResultSet containing the execution results.
	 * @param choreoInputs
	 * @return
	 * @throws TembooException
	 */
	protected ResultSet execute(InputSet choreoInputs) throws TembooException {
		JSONObject document = executeWithResults(choreoInputs);
		return new ResultSet(document);
	}

	/**
	 * Execute the choreography, using the specified InputSet as parameters, and return immediately, discarding the execution results.
	 * @param choreoInputs
	 * @return
	 * @throws TembooException
	 */
	public String executeNoResults(InputSet choreoInputs)
			throws TembooException {

		return executeNoResults(choreoInputs, false);
	}
	

	/**
	 * Execute the choreography asynchronously, using the specified InputSet as parameters, and return a ChoreographyExecution 
	 * object which may be used to monitor the status of the running execution, or to obtain choreography results.
	 * @param choreoInputs
	 * @return
	 * @throws TembooException
	 */
	public ChoreographyExecution executeAsync(InputSet choreoInputs)
			throws TembooException {
		
		String exec_id = executeNoResults(choreoInputs, true);
		try {
			return new ChoreographyExecution(session, exec_id);
		} catch (URISyntaxException e) {
			throw new TembooException(e);
		}
	}
	
	
	/**
	 * Execute the choreography synchronously, using the specified InputSet as parameters, and return a 
	 * Document object representing the result XML.
	 * 
	 * @param choreoInputs
	 * @return
	 * @throws TembooException
	 */
	protected JSONObject executeWithResults(InputSet choreoInputs)
			throws TembooException {
		if (null == choreoInputs) {
			choreoInputs = new InputSet();
		}
		byte[] json = choreoInputs.formatInputs();
		
		HashMap<String, String> paramMap = new HashMap<String, String>();
		paramMap.put("source_id", TembooSession.CLIENT_IDENTIFIER);
		return session.postContent(getPath(), json, paramMap);		
	}
	
	
	/**
	 * Internal convenience method used to perform asynchronous choreography executions
	 * @param choreoInputs
	 * @param storeChoreoResults
	 * @return
	 * @throws TembooException
	 */
	private String executeNoResults(InputSet choreoInputs, boolean storeChoreoResults) throws TembooException {
		String id = null;
		if (null == choreoInputs) {
			choreoInputs = new InputSet();
		}
		try {
			byte[] xml = choreoInputs.formatInputs();
			HashMap<String, String> paramMap = new HashMap<String, String>();
			paramMap.put("mode", "async");
			paramMap.put("source_id", TembooSession.CLIENT_IDENTIFIER);
			if(storeChoreoResults)
				paramMap.put("store_results", "true");
			
			JSONObject result = session.postContent(getPath(), xml, paramMap);
			id = (String) result.get("id");

		} catch (Exception e) {
			throw new TembooException(e);
		}
		return id;
}


	static String getStaticBasePath() {
		return TembooSession.BASE_PATH + "/choreos";
	}

	protected String getBasePath() {
		return getStaticBasePath();
	}


	public static class InputSet {
		private final Map<String, String> inputs = new HashMap<String, String>();
		private final Map<String, TembooPath> aliases = new HashMap<String, TembooPath>();
        private final List<OutputFilter> outputFilters = new ArrayList<OutputFilter>();
		private String credential = null;
		static final DecimalFormat decimalFormat = new DecimalFormat("########################.########################");

		
		/**
		 * Specify the Credential object to be used when running this choreo.
		 * @param credentialName
		 */
		public void setCredential(String credentialName) {
			credential = credentialName;
		}
		
		/**
		 * Add a Profile object to be used when running this choreo.
		 * @param profileName
		 */
		public void addProfile(String profileName) {
			credential = profileName;
		}
		
		/**
		 * Set the named alias to the specified value
		 * @param name
		 * @param value
		 */
		public void setAlias(String name, TembooPath value) {
			aliases.put(name, value);
		}
		

		/**
		 * Set the named input to the specified value
		 * @param name
		 * @param value
		 */
		public void setInput(String name, String value) {
			inputs.put(name, value);
		}

		/**
		 * Set the named input to the specified value
		 * @param name
		 * @param value
		 */
		public void setInput(String name, BigDecimal value) {
			if (value == null) {
				inputs.put(name, "0");
			}
			String stringValue = decimalFormat.format(value.stripTrailingZeros());
			inputs.put(name, stringValue);
		}

		/**
		 * Set the named input to the specified value
		 * @param name
		 * @param value
		 */
		public void setInput(String name, Integer value) {
			if (value == null) {
				inputs.put(name, "0");
			}
			String stringValue = value.toString();
			inputs.put(name, stringValue);
		}

		/**
		 * Set the named input to the specified value
		 * @param name
		 * @param value
		 */
		public void setInput(String name, Boolean value) {
			if (value == null) {
				inputs.put(name, Boolean.toString(false));
			}
			String stringValue = value.toString();
			inputs.put(name, stringValue);
		}
		
        /**
         * Add an output filter to the results.
         * @param filterName
         * @param path
         * @param outputVariableSource
         */
        public void addOutputFilter(String filterName, String path, String outputVariableSource) {
            outputFilters.add(new OutputFilter(filterName, path, outputVariableSource));
        }

		private byte[] formatInputs() throws TembooException {
			try {

				JSONObject allInputs = new JSONObject();
		
				if (null != credential) 
					allInputs.put("preset", credential);					
				
				JSONArray inputList = new JSONArray();
				for (Entry<String, String> entry : inputs.entrySet()) {
					String name = entry.getKey();
					JSONObject input = new JSONObject();
					input.put("name", name);
					input.put("value", entry.getValue());
					inputList.put(input);
				}
				
				allInputs.put("inputs", inputList);
				
				JSONArray outputList = new JSONArray();

                for (OutputFilter filter : outputFilters) {
					JSONObject output = new JSONObject();
                    output.put("name", filter.getFilterName());
                    output.put("variable", filter.getOutputVariableSource());
                    output.put("path", filter.getPath());
                    outputList.put(output);
                }

                allInputs.put("outputFilters", outputList);
                String str = allInputs.toString();
				return str.getBytes();
			
			} catch (Exception e) {
				throw new TembooException(e);
			}			
		}
	}

	public static class ResultSet {

		/**
		 * Represents the possible completion statuses of a choreo		 *
		 */
		public enum Status {
			SUCCESS, ERROR, TERMINATED_MANUALLY, TERMINATED_LIMIT, RUNNING
		}

		private String id = "";
		private Status status = Status.ERROR;
		private Date startTime = null, endTime = null;
		private Map<String, Object> outputs = new HashMap<String, Object>();

		public static JSONObject getJSONObject(Object json, String key) {
			JSONObject toReturn = null;

			try {
				if(json != null && (json instanceof JSONObject))
					toReturn = ((JSONObject) json).optJSONObject(key);					
			} catch(Exception e) {
			}
			if(toReturn == null)
				toReturn = new JSONObject();
			
			return toReturn;
		}
		
		protected ResultSet(JSONObject doc) throws TembooException {
			try {
				if (doc != null) {
					JSONObject execution = (JSONObject) doc.get("execution");
					id = (String) execution.get("id");
					String statusString = (String) execution.get("status");
	
					status = Status.valueOf(statusString);
					String timeStr = (String) execution.get("starttime");
					startTime = new Date(Long.parseLong(timeStr));
					timeStr = (String) execution.get("endtime");
						endTime = new Date(Long.parseLong(timeStr));
	
					JSONObject outputList = (JSONObject) doc.get("output");
					Iterator<String> keys = outputList.keys();
					while(keys.hasNext()) {
						String key = keys.next();
						outputs.put(key, outputList.opt(key));
					}
				}
			} catch(JSONException e) {
				throw new TembooException(e);
			}
		}

		public static Object getJSONFromString(String str) {
			try {
				if (str.trim().startsWith("{")) {
					return new JSONObject(str);
				} else {
					return new JSONArray(str);
				}
			} catch (JSONException e) {
				return null;
			}
		}

		
		// get the completion status of the choreo (success, error, manually
		// terminated, etc.)
		public Status getCompletionStatus() {
			return status;
		}

		// get the start time of the choreo (in UTC)
		public Date getStartTime() {
			return startTime;
		}

		// get the completion time of the choreo (in UTC)
		public Date getCompletionTime() {
			return endTime;
		}

		// get the execution id
		public String getId() {
			return id;
		}

		// get the exception that was thrown during execution, if any; returns
		// null if execution was successful
		public TembooException getException() {
			return null;
		}

		// get the value of the named output
		public String getResultString(String key) {
			return outputs.get(key).toString();
		}

        // get the value of the named output as list
        public JSONArray getResultList(String key) {
            final Object o = outputs.get(key);
            if (o instanceof JSONArray) {
                return (JSONArray) o;
            } else {
                JSONArray list = new JSONArray();
                list.put(o);
                return list;
            }
        }

		// get the set of output names for this execution
		public Iterator<String> keySet() {
			return outputs.keySet().iterator();
		}

		public Map<String, Object> getOutputs() {
			return outputs;
		}
	}
	
    private static class OutputFilter {
        final String filterName;
        final String path;
        final String outputVariableSource;
        protected OutputFilter(String theFilterName, String thePath, String theOutputVariableSource) {
            filterName = theFilterName;
            path = thePath;
            outputVariableSource = theOutputVariableSource;
        }

        public String getFilterName() {
            return filterName;
        }

        public String getPath() {
            return path;
        }

        public String getOutputVariableSource() {
            return outputVariableSource;
        }
    }
}
